/**********************************************************
 **
 ** bsc_ctrl.c : parser for BSC branching functions
 **
 ** Copyright (c) 2004, Kyle A. York
 ** All rights reserved
 **
 ***********************************************************/
#include "pfile.h"
#include "pftoken.h"
#include "pf_expr.h"
#include "pf_cmd.h"
#include "expr.h"

#include "label.h"
#include "bsc_file.h"
#include "bsc_parm.h"
#include "bsc_tokn.h"
#include "bsc_expr.h"
#include "bsc_blck.h"
#include "bsc_ctrl.h"
#include "bsc_tokn.h"

/*
 * NAME
 *   bsc_parse_for
 *
 * DESCRIPTION
 *   implements the for command
 *
 * PARAMETERS
 *   bsc      : 
 *   param_ct : # of parameters parsed
 *   params   : parsed parameters
 *
 * RETURN
 *   none
 *
 * NOTES
 *   FOR var = start TO end
 *   ...
 *   NEXT
 * this becomes:
 *   label_top:
 *      block
 *      var == end? --> yes, goto label_end
 *      var = var + 1
 *      goto label_top
 *   label_end:
 *
 */
void bsc_parse_for(pfile_t *pf, unsigned sub,
  size_t param_ct, const param_t *params)
{
  value_t  var_control;

  var_control = params[0].u.var;
  if (pf_token_is(pf, pf_token_current, pfile_log_err, 
    "to")) {
    static const char *endwords[] = {"next"};
    label_t   label_top;  /* top of loop        */
    value_t   var_done;

    pf_token_get(pf, pf_token_next);

    var_done = 0;
    bsc_expr_parse(pf, 0, 0, &var_done);
    label_top = 0;
    pfile_label_alloc(pf, 0, &label_top);

    if (var_done && label_top) {
      pfile_cmd_label_add(pf, label_top);
    }
    bsc_block_process(pf, BSC_BLOCK_PROCESS_FLAG_SKIP_STMT, 
        COUNT(endwords), endwords, 0);
    if (var_done) {
      label_t  label_end;  /* end of statement   */
      value_t  var_tmp;    /* control != end     */

      label_end = 0;
      pfile_label_alloc(pf, 0, &label_end);
      var_tmp = pfile_value_temp_get(pf, 1, 0);

      if (var_tmp && label_end) {
        pfile_cmd_op_add(pf, operator_eq, &var_tmp, var_control, var_done);
        pfile_cmd_branch_add(pf, cmd_branchtype_goto, cmd_branchcond_true,
            label_end, var_tmp, 0, 0);
        pfile_cmd_op_add(pf, operator_incr, &var_control, 0, 0);
        pfile_cmd_branch_add(pf, cmd_branchtype_goto, cmd_branchcond_none,
            label_top, 0, 0, 0);
        pfile_cmd_label_add(pf, label_end);
      }
      value_release(var_tmp);
      label_release(label_end);
    }
    label_release(label_top);
    value_release(var_done);
  }
}

/*
 * NAME
 *   bsc_parse_if
 *
 * DESCRIPTION
 *   process the if statement
 *
 * PARAMETERS
 *   bsc       :
 *   param_ct :
 *   params   :
 *
 * RETURN
 *
 * NOTES
 *   IF expr THEN
 *     ...
 *   [ELSE
 *     ...]
 *   [ELIF
 *     ...]
 *   END IF
 */
void bsc_parse_if(pfile_t *pf, unsigned sub, size_t param_ct, 
  const param_t *params)
{
  if (pf_token_is(pf, pf_token_current, pfile_log_err, 
      "then")) {
    static const char *endwords[] = {"elif", "else", "endif" };
    enum { EW_ELIF, EW_ELSE, EW_ENDIF };
    size_t    ew;
    label_t   label_end;
    label_t   label_next;
    boolean_t first;

    if (result_ok != pfile_label_alloc(pf, 0, &label_end)) {
      label_end = 0;
    }
    label_next = 0;
    pf_token_get(pf, pf_token_next);
    first = boolean_true;
    do {
      value_t cond;

      if (first) {
        /* first time through */
        cond = params[0].u.var;
        value_lock(cond);
        first = boolean_false;
      } else {
        if (label_next) {
          pfile_cmd_label_add(pf, label_next);
          label_release(label_next);
        }
        if (result_ok != bsc_expr_parse(pf, EXPR_FLAG_IF, 0, &cond)) {
          cond = 0;
        }
        if (pf_token_is(pf, pf_token_current, pfile_log_err, 
          "then")) {
          pf_token_get(pf, pf_token_next);
        }
      }
      if (result_ok != pfile_label_alloc(pf, 0, &label_next)) {
        label_next = 0;
      }
      if (label_next) {
        pfile_cmd_branch_add(pf, cmd_branchtype_goto, 
          cmd_branchcond_false, label_next, cond, 0, 0);
      }
      value_release(cond);
      bsc_block_process(pf, BSC_BLOCK_PROCESS_FLAG_SKIP_STMT, 
          COUNT(endwords), endwords, &ew);
      if (label_end) {
        pfile_cmd_branch_add(pf, cmd_branchtype_goto,
          cmd_branchcond_none, label_end, 0, 0, 0);
      }
    } while (EW_ELIF == ew);
    if (label_next) {
      pfile_cmd_label_add(pf, label_next);
      label_release(label_next);
    }
    if (EW_ELSE == ew) {
      bsc_block_process(pf, BSC_BLOCK_PROCESS_FLAG_SKIP_STMT, 
          COUNT(endwords) - EW_ENDIF, endwords + EW_ENDIF, &ew);
    }
    if (label_end) {
      pfile_cmd_label_add(pf, label_end);
      label_release(label_end);
    }
  }
}

/*
 * NAME
 *   bsc_parse_branch
 *
 * DESCRIPTION
 *   parse GOTO label, GOSUB label, and RETURN
 *
 * PARAMETERS
 *
 * RETURN
 *
 * NOTES
 */
void bsc_parse_branch(pfile_t *pf, unsigned sub, size_t param_ct, 
  const param_t *params)
{
  label_t    lbl;
  boolean_t  err;

  err = boolean_false;
  if (cmd_branchtype_return == sub) {
    lbl = 0;
  } else if (!bsc_token_is_identifier(pf)) {
    err = boolean_true;
  } else if (result_ok != pf_token_to_label(pf, pfile_log_err, 
      boolean_true, &lbl)) {
    err = boolean_true;
  } else {
    pf_token_get(pf, pf_token_next);
    if ((1 != bsc_file_pass_get(pf))
      && !label_flag_test(lbl, LABEL_FLAG_DEFINED)) {
      pfile_log(pf, pfile_log_err, BSC_MSG_NEVER_DEFINED, 
        label_name_get(lbl));
      label_release(lbl);
      err = boolean_true;
    }
  }
  if (!err) {
    pfile_cmd_branch_add(pf, sub, cmd_branchcond_none, lbl, 0, 0, 0);
    label_release(lbl);
  }
}

/*
 * NAME
 *   bsc_parse_while
 *
 * DESCRIPTION
 *   parse the while block
 *
 * PARAMETERS
 *
 * RETURN
 *
 * NOTES
 *   format:
 *     WHILE expr
 *       block
 *     ENDWHILE  
 *     
 */
void bsc_parse_while(pfile_t *pf, unsigned sub, size_t param_ct, 
  const param_t *params)
{
  static const char *endwords[] = {"endwhile"};
  label_t   label_top;  /* top of loop        */
  label_t   label_skip;
  value_t   var_done;

  label_top = 0;
  pfile_label_alloc(pf, 0, &label_top);
  label_skip = 0;
  pfile_label_alloc(pf, 0, &label_skip);

  if (label_top) {
    pfile_cmd_label_add(pf, label_top);
  }

  var_done = 0;
  bsc_expr_parse(pf, 0, 0, &var_done);
  if (var_done && label_skip) {
    pfile_cmd_branch_add(pf, cmd_branchtype_goto, cmd_branchcond_false,
        label_skip, var_done, 0, 0);
  }

  bsc_block_process(pf, BSC_BLOCK_PROCESS_FLAG_SKIP_STMT,
      COUNT(endwords), endwords, 0);

  if (label_top) {
    pfile_cmd_branch_add(pf, cmd_branchtype_goto, cmd_branchcond_none,
        label_top, 0, 0, 0);
  }
  if (label_skip) {
    pfile_cmd_label_add(pf, label_skip);
  }

  value_release(var_done);
  label_release(label_skip);
  label_release(label_top);
}

/*
 * NAME
 *   bsc_parse_do
 *
 * DESCRIPTION
 *   parse the do command
 *
 * PARAMETERS
 *
 * RETURN
 *
 * NOTES
 *   format:
 *     DO
 *       block
 *     WHILE expr
 */
void bsc_parse_do(pfile_t *pf, unsigned sub, size_t param_ct, 
  const param_t *params)
{
  static const char *endwords[] = {"while"};
  label_t   label_top;  /* top of loop        */
  value_t   var_done;

  label_top = 0;
  pfile_label_alloc(pf, 0, &label_top);

  if (label_top) {
    pfile_cmd_label_add(pf, label_top);
  }

  bsc_block_process(pf, BSC_BLOCK_PROCESS_FLAG_SKIP_STMT, 
      COUNT(endwords), endwords, 0);

  var_done = 0;
  bsc_expr_parse(pf, 0, 0, &var_done);
  if (var_done && label_top) {
    pfile_cmd_branch_add(pf, cmd_branchtype_goto, cmd_branchcond_true,
        label_top, var_done, 0, 0);
  }

  value_release(var_done);
  label_release(label_top);
}


