/**********************************************************
 **
 ** bsc_intr.c : BSC intrinsic function parser
 **
 ** Copyright (c) 2004, Kyle A. York
 ** All rights reserved
 **
 ***********************************************************/
#include "pfile.h"
#include "pf_expr.h"
#include "pf_cmd.h"
#include "pftoken.h"
#include "expr.h"

#include "label.h"
#include "bsc_file.h"
#include "bsc_parm.h"
#include "bsc_cmd.h"
#include "bsc_blck.h"
#include "bsc_ctrl.h"
#include "bsc_tokn.h"
#include "bsc_expr.h"
#include "bsc_intr.h"

/* misc. notes:
 *    Tosc = freq / 4
 *    freq is in KHz
 *
 * all code is generated assuming that the operators are variables.
 * if it turns out that they are constants, the unneeded code will be
 * removed later.
 */    

/* the following naming is used:
 *    bsc_expr_add_reg */
static void bsc_expr_add_var(pfile_t *pf, expr_t **stk, value_t val,
    operator_t op)
{
  pfile_expr_push(pf, stk, val, op);
}

static void bsc_expr_add_reg(pfile_t *pf, expr_t **stk, const char *reg,
    operator_t op)
{
  value_t val;

  val = pfile_value_find(pf, pfile_log_err, PFILE_LENGTH_DEF, reg);
  if (val) {
    bsc_expr_add_var(pf, stk, val, op);
    value_release(val);
  }
}

static void bsc_expr_add_const(pfile_t *pf, expr_t **stk, 
    variable_const_t n, operator_t op)
{
  value_t val;

  if (result_ok == pfile_constant_get(pf, n, &val)) {
    bsc_expr_add_var(pf, stk, val, op);
    value_release(val);
  }
}

static void bsc_assign_reg_val(pfile_t *pf, operator_t op,
    const char *dst, value_t src)
{
  value_t dval;

  dval = pfile_value_find(pf, pfile_log_err, PFILE_LENGTH_DEF, dst);
  if (dval) {
    pfile_cmd_op_add(pf, op, &dval, src, 0);
    value_release(dval);
  }
}

static void bsc_assign_reg_reg(pfile_t *pf, operator_t op,
    const char *dst, const char *src)
{
  value_t sval;

  sval = pfile_value_find(pf, pfile_log_err, 
    PFILE_LENGTH_DEF, src);
  if (sval) {
    bsc_assign_reg_val(pf, op, dst, sval);
    value_release(sval);
  }
}

static void bsc_assign_reg_const(pfile_t *pf, operator_t op,
    const char *dst, variable_const_t n)
{
  value_t sval;

  if (result_ok == pfile_constant_get(pf, n, &sval)) {
    bsc_assign_reg_val(pf, op, dst, sval);
    value_release(sval);
  }
}

static void bsc_assign_val_reg(pfile_t *pf, operator_t op,
    value_t dst, const char *src)
{
  value_t sval;
  
  sval = pfile_value_find(pf, pfile_log_err, PFILE_LENGTH_DEF, src);
  if (sval) {
    pfile_cmd_op_add(pf, op, &dst, sval, 0);
    value_release(sval);
  }
}

static void bsc_reg_bit_wait(pfile_t *pf, const char *reg,
  const char *bit, cmd_branchcond_t cond)
{
  label_t  lbl;
  result_t rc;

  rc = pfile_label_alloc(pf, 0, &lbl);
  if (result_ok == rc) {
    expr_t  *exprstk;

    exprstk = 0;
    pfile_cmd_label_add(pf, lbl);
    bsc_expr_add_reg(pf, &exprstk, reg, operator_bittst);
    bsc_expr_add_reg(pf, &exprstk, bit, operator_null);
    if (exprstk) {
      pfile_cmd_branch_add(pf, cmd_branchtype_goto,
        cond, lbl, expr_val_get(exprstk), 0, 0);
    }
    label_release(lbl);
    expr_list_free(&exprstk);
  }
}

static void bsc_parse_pwm_duty(pfile_t *pf, value_t duty, 
  value_t channel, value_t pr2)
{
  /* ccpr?l:ccp?con<5:4> = pr2 * duty / 100 */
  boolean_t  pr2_release;
  value_t    c_1;     /* constant 1 */

  if (!pr2) {
    pr2 = pfile_value_find(pf, pfile_log_err, PFILE_LENGTH_DEF, "_pr2");
    pr2_release = boolean_true;
  } else {
    pr2_release = boolean_false;
  }

  /* need a big one for this! */
  if (pr2 && (result_ok == pfile_constant_get(pf, 1, &c_1))) {
    expr_t  *exprstk;
    value_t  tmp;
    label_t  skip;

    value_sz_set(c_1, 2);
    exprstk = 0;
    /* tmp = 1 * duty * pr2 / 100 */
    /* note: the 1 forces the expression to 16 bits */
    bsc_expr_add_var(pf, &exprstk, c_1, operator_mul);
    bsc_expr_add_var(pf, &exprstk, duty, operator_mul);
    bsc_expr_add_var(pf, &exprstk, pr2, operator_div);
    bsc_expr_add_const(pf, &exprstk, 100, operator_null);
    tmp = expr_val_get(exprstk);
    value_flag_set(tmp, VARIABLE_FLAG_NO_OPT);
    value_lock(tmp);
    expr_list_free(&exprstk);
    if (result_ok == pfile_label_alloc(pf, 0, &skip)) {
      variable_t vtmp;

      /* if (channel & 1)
       *   _ccpr1l = tmp
       *   _ccp1con = 15 */
      bsc_expr_add_var(pf, &exprstk, channel, operator_bittst);
      bsc_expr_add_const(pf, &exprstk, 0, operator_null);
      pfile_cmd_branch_add(pf, cmd_branchtype_goto,
          cmd_branchcond_false, skip, expr_val_get(exprstk), 0, 0);
      expr_list_free(&exprstk);
      /* ccpr1l = tmp */
      bsc_assign_reg_val(pf, operator_assign, "_ccpr1l", tmp);
      bsc_assign_reg_const(pf, operator_assign, "_ccp1con", 15);
      pfile_cmd_label_add(pf, skip);
      label_release(skip);
      vtmp = pfile_variable_find(pf, pfile_log_none, PFILE_LENGTH_DEF,
          "_ccp2con");
      if (!vtmp) {
        /* i should probably log a warning if channel is constant
           & bit 1 is set */
      } else {
        variable_release(vtmp);
        if (result_ok == pfile_label_alloc(pf, 0, &skip)) {
          /* if (channel & 2)
           *   _ccpr2l = tmp
           *   _ccp2con = 15 */
          bsc_expr_add_var(pf, &exprstk, channel, operator_bittst);
          bsc_expr_add_const(pf, &exprstk, 1, operator_null);
          pfile_cmd_branch_add(pf, cmd_branchtype_goto,
              cmd_branchcond_false, skip, expr_val_get(exprstk), 0, 0);
          expr_list_free(&exprstk);
          /* ccpr2l = tmp */
          bsc_assign_reg_val(pf, operator_assign, "_ccpr2l", tmp);
          bsc_assign_reg_const(pf, operator_assign, "_ccp2con", 15);
          pfile_cmd_label_add(pf, skip);
          label_release(skip);
        }
      }
    }
    value_flag_clr(tmp, VARIABLE_FLAG_NO_OPT);
    value_release(tmp);
    value_release(c_1);
  }
  if (pr2_release) {
    value_release(pr2);
  }
}

/*
 * NAME
 *   bsc_parse_pwm_config
 *
 * DESCRIPTION
 *   parse the pwm_config command
 *
 * PARAMETERS
 *   bsc:
 *   period:
 *   pr2:    if non-null, on exit this will hold the value
 *           assigned to pr2
 *
 * RETURN
 *
 * NOTES
 */
static void bsc_parse_pwm_period(pfile_t *pf, value_t period,
    value_t *pr2)
{
  /* pr2 = period (in ms) / freq (in KHz) - 1 */
  /* if (pr2 > 1023) 
   *    pr2 /= 16
   *    prescale = 1:0 (16) 
   * else if (pr2 > 255)
   *    pr2 /= 4
   *    prescale = 0:1 ( 4)
   * else
   *    prescale = 0:0 ( 1)
   */
  value_t          tmp;
  expr_t          *exprstk;
  label_t          done;
  value_t          c_100;

  if (pr2) {
    *pr2 = 0;
  }
  exprstk = 0;
  if (result_ok == pfile_constant_get(pf, 100, &c_100)) {
    value_sz_set(c_100, 4); /* this needs to be 32 bits */
  
    bsc_expr_add_var(pf, &exprstk, period, operator_mul);
    bsc_expr_add_var(pf, &exprstk, c_100, operator_div);
    bsc_expr_add_reg(pf, &exprstk, "_freq", operator_sub);
    bsc_expr_add_const(pf, &exprstk, 1, operator_null);
    tmp = expr_val_get(exprstk);
    value_lock(tmp);
    expr_list_free(&exprstk);
    /* this is probably sloppy, but the code generator will eliminate
     * any unncessary code, so instead of special casing const/non-const
     * I'll just output as if it was non-const & let the code generator
     * work its magic */
      
    /* if tmp > 1023 goto skip */
    bsc_expr_add_var(pf, &exprstk, tmp, operator_gt);
    bsc_expr_add_const(pf, &exprstk, 1023, operator_null);
    if (result_ok == pfile_label_alloc(pf, 0, &done)) {
      label_t skip;

      if (result_ok == pfile_label_alloc(pf, 0, &skip)) {
        pfile_cmd_branch_add(pf, cmd_branchtype_goto, 
            cmd_branchcond_false, skip, expr_val_get(exprstk), 0, 0);
        expr_list_free(&exprstk);
        /* pr2 = tmp /= 4; prescale = 1:0 */
        bsc_expr_add_reg(pf, &exprstk, "_pr2", operator_assign);
        bsc_expr_add_var(pf, &exprstk, tmp, operator_div);
        bsc_expr_add_const(pf, &exprstk, 4, operator_null);
        expr_list_free(&exprstk);
        bsc_assign_reg_reg(pf, operator_bitset, "_t2con", "_t2ckps1");
        bsc_assign_reg_reg(pf, operator_bitclr, "_t2con", "_t2ckps0");
        pfile_cmd_branch_add(pf, cmd_branchtype_goto, 
            cmd_branchcond_none, done, 0, 0, 0);
        /* skip: if tmp > 255 goto skip1 */
        pfile_cmd_label_add(pf, skip);
        label_release(skip);
        skip = 0;
        if (result_ok == pfile_label_alloc(pf, 0, &skip)) {
          bsc_expr_add_var(pf, &exprstk, tmp, operator_gt);
          bsc_expr_add_const(pf, &exprstk, 255, operator_null);
          pfile_cmd_branch_add(pf, cmd_branchtype_goto, 
              cmd_branchcond_false, skip, expr_val_get(exprstk), 0, 0);
          expr_list_free(&exprstk);
          /* pr2  = tmp / 2; prescale = 0:1 */
          bsc_expr_add_reg(pf, &exprstk, "_pr2", operator_assign);
          bsc_expr_add_var(pf, &exprstk, tmp, operator_div);
          bsc_expr_add_const(pf, &exprstk, 2, operator_null);
          expr_list_free(&exprstk);
          bsc_assign_reg_reg(pf, operator_bitclr, "_t2con", "_t2ckps1");
          bsc_assign_reg_reg(pf, operator_bitset, "_t2con", "_t2ckps0");
          pfile_cmd_branch_add(pf, cmd_branchtype_goto, 
              cmd_branchcond_none, done, 0, 0, 0);
          /* skip1: pr2 = tmp; prescale = 0:0 */
          pfile_cmd_label_add(pf, skip);
          bsc_assign_reg_val(pf, operator_assign, "_pr2", tmp);
          bsc_assign_reg_reg(pf, operator_bitclr, "_t2con", "_t2ckps1");
          bsc_assign_reg_reg(pf, operator_bitclr, "_t2con", "_t2ckps0");
          pfile_cmd_label_add(pf, done);
          label_release(skip);
        }
      }
      label_release(done);
    }
    if (pr2) {
      *pr2 = tmp;
    } else {
      value_release(tmp);
    }
    value_release(c_100);
  }
}

/*
 * NAME
 *   bsc_parse_intr_config
 *
 * DESCRIPTION
 *   enable an interrupt if desired
 *
 * PARAMETERS
 *   pf : 
 *   intr : interrupt enable value
 *   reg  : register that holds IE bit
 *   bit  : IE bit
 *
 * RETURN
 *
 * NOTES
 */
static void bsc_parse_intr_config(pfile_t *pf, value_t intr,
  bsc_isr_type_t type, const char *errstr, const char *reg, const char *bit)
{
  result_t  rc;
  boolean_t need;
  label_t   lbl;

  /* check that the ISR exists */

  need = !value_is_const(intr) || value_const_get(intr);
  rc = bsc_file_isr_get(pf, type, need, &lbl);
  if (result_ok == rc) {
    if (bsc_file_pass_get(pf) > 1) {
      if (!label_flag_test(lbl, LABEL_FLAG_DEFINED)) {
        char ir_name[BSC_IR_NAME_SZ_MAX];

        sprintf(ir_name, "IR_%s", errstr);
        pfile_log(pf, pfile_log_err, BSC_MSG_NEVER_DEFINED, errstr);
      }
    }
    if (need) {
      label_usage_bump(lbl, ctr_bump_incr);
    }
    label_release(lbl);
  }

  if ((result_ok == rc) && need) {
    /* programatically enable this */
    if (result_ok == pfile_label_alloc(pf, 0, &lbl)) {
      pfile_cmd_branch_add(pf, cmd_branchtype_goto, cmd_branchcond_false,
        lbl, intr, 0, 0);
      bsc_assign_reg_reg(pf, operator_bitset, reg, bit);
      pfile_cmd_label_add(pf, lbl);
      label_release(lbl);
    }
  }
}

static void bsc_parse_adc_config(pfile_t *pf, value_t channel,
    value_t power, value_t format, value_t control,
    value_t intr)
{
  /* result:
   *   ADCON0<7:6>
   *            00 -- max. freq =  1.25MHz
   *            01 -- max. freq =  5.00MHz
   *            10 -- max. freq = 20.00MHz
   *            11 -- use RC
   *   ADCON0 = 128 | ((channel & 0x07) << 3) | !!power
   *   ADCON1 = (!!format << 7) | (control & 0x0f)
   */
  /* tmp = channel & 7 */
  value_t tmp;
  expr_t  *exprstk;
  /* disable the adc interrupt */
  bsc_assign_reg_reg(pf, operator_bitclr, "_pie1", "_adie");
  bsc_assign_reg_reg(pf, operator_bitclr, "_pir1", "_adif");

  exprstk = 0;

  bsc_expr_add_var(pf, &exprstk, channel, operator_andb);
  bsc_expr_add_const(pf, &exprstk, 7, operator_null);
  tmp = expr_val_get(exprstk);
  value_lock(tmp);
  expr_list_free(&exprstk);

  /* adcon0 = tmp << 3 | 128 | power */
  bsc_expr_add_reg(pf, &exprstk, "_adcon0", operator_assign);
  bsc_expr_add_var(pf, &exprstk, tmp, operator_shift_left);
  bsc_expr_add_const(pf, &exprstk, 3, operator_orb);
  bsc_expr_add_const(pf, &exprstk, 
      (oscillator_crystal == pfile_oscillator_get(pf)) ? 128 : 192, 
      operator_orb);
  bsc_expr_add_var(pf, &exprstk, power, operator_null);
  expr_list_free(&exprstk);
  value_release(tmp);

  /* tmp = control & 0x0f */
  bsc_expr_add_var(pf, &exprstk, control, operator_shift_left);
  bsc_expr_add_const(pf, &exprstk, 0x0f, operator_null);
  tmp = expr_val_get(exprstk);
  value_lock(tmp);
  expr_list_free(&exprstk);

  /* adcon1 = format << 7 | tmp */
  bsc_expr_add_reg(pf, &exprstk, "_adcon1", operator_assign);
  bsc_expr_add_var(pf, &exprstk, format, operator_shift_left);
  bsc_expr_add_const(pf, &exprstk, 7, operator_orb);
  bsc_expr_add_var(pf, &exprstk, tmp, operator_null);
  expr_list_free(&exprstk);
  bsc_parse_intr_config(pf, intr, bsc_isr_type_adc, "ADC", "_pie1", "_adie");
  value_release(tmp);
}

/*
 * x = freq * 1000 / 16 / baud
 * if (x > 256)
 *   x /= 16
 */
typedef enum { usart_config_rx, usart_config_tx } usart_config_t;
static void bsc_parse_usart_config(pfile_t *pf, usart_config_t dir, 
    value_t baud, value_t intr)
{
  expr_t          *exprstk;
  value_t          tmp;
  const char      *bit;
  label_t          done;
  bsc_file_flag_t  flag;

  /* disable the interrupt */
  bit  = "{unknown}";
  flag = 0;
  switch (dir) {
    case usart_config_tx:
      /* tx */
      bsc_assign_reg_reg(pf, operator_bitclr, "_pie1", "_txie");
      /*bsc_assign_reg_reg(pf, operator_bitclr, "_pir1", "_txif");*/
      bit = "_txie";
      flag = bsc_file_flag_usart_config_tx;
      break;
    case usart_config_rx:
      /* rx */
      bsc_assign_reg_reg(pf, operator_bitclr, "_pie1", "_rcie");
      flag = bsc_file_flag_usart_config_rx;
      /*bsc_assign_reg_reg(pf, operator_bitclr, "_pir1", "_rcif");*/
      bit = "_rcie";
      break;
  }
  if (!flag) {
    pfile_log(pf, pfile_log_err, BSC_MSG_INTERNAL_ERROR);
  } else {
    bsc_file_flag_set(pf, flag);
    exprstk = 0;
    /* tmp = freq * 1000 / 16 / baud - 1 */
    bsc_expr_add_reg(pf, &exprstk, "_freq", operator_mul);
    bsc_expr_add_const(pf, &exprstk, 1000, operator_div);
    bsc_expr_add_const(pf, &exprstk, 16, operator_div);
    bsc_expr_add_var(pf, &exprstk, baud, operator_sub);
    bsc_expr_add_const(pf, &exprstk, 1, operator_null);
    tmp = expr_val_get(exprstk);
    value_lock(tmp);
    expr_list_free(&exprstk);

    if (result_ok == pfile_label_alloc(pf, 0, &done)) {
      label_t skip;

      if (result_ok == pfile_label_alloc(pf, 0, &skip)) {
        bsc_expr_add_var(pf, &exprstk, tmp, operator_gt);
        bsc_expr_add_const(pf, &exprstk, 256, operator_null);
        pfile_cmd_branch_add(pf, cmd_branchtype_goto, 
            cmd_branchcond_false, skip, expr_val_get(exprstk), 0, 0);
        /* tmp > 256, spbrg = tmp / 4 - 1, clr brgh */
        expr_list_free(&exprstk);
        bsc_expr_add_reg(pf, &exprstk, "_spbrg", operator_assign);
        bsc_expr_add_var(pf, &exprstk, tmp, operator_div);
        bsc_expr_add_const(pf, &exprstk, 4, operator_null);
        expr_list_free(&exprstk);
        bsc_assign_reg_reg(pf, operator_bitclr, "_txsta", "_brgh");
        pfile_cmd_branch_add(pf, cmd_branchtype_goto,
            cmd_branchcond_none, done, 0, 0, 0);
        /* tmp <= 256, spbrg = tmp - 1, set brgh */
        pfile_cmd_label_add(pf, skip);
        bsc_assign_reg_val(pf, operator_assign, "_spbrg", tmp);
        bsc_assign_reg_reg(pf, operator_bitset, "_txsta", "_brgh");
        pfile_cmd_label_add(pf, done);
        label_release(skip);
      }
      label_release(done);
    }
    /* some more common setup */
    bsc_assign_reg_reg(pf, operator_bitclr, "_txsta", "_sync"); /* async. operation */
    bsc_assign_reg_reg(pf, operator_bitset, "_rcsta", "_spen"); /* enable the usart */
    bsc_assign_reg_const(pf, operator_bitset, "_trisc", 7);
    bsc_assign_reg_const(pf, operator_bitset, "_trisc", 6);


    switch (dir) {
      case usart_config_tx:
        bsc_assign_reg_reg(pf, operator_bitclr, "_txsta", "_tx9"); /* 8-bit op */
        bsc_assign_reg_reg(pf, operator_bitset, "_txsta", "_txen"); /* TX enable */
        bsc_parse_intr_config(pf, intr, bsc_isr_type_tx, "TX", "_pie1", bit);
        break;
      case usart_config_rx:
        bsc_assign_reg_reg(pf, operator_bitclr, "_rcsta", "_rx9");  /* 8-bit op */
        bsc_assign_reg_reg(pf, operator_bitset, "_rcsta", "_cren"); /* continuous RX */
        bsc_parse_intr_config(pf, intr, bsc_isr_type_rx, "RX", "_pie1", bit);
        break;
    }
    value_release(tmp);
  }
}

static void bsc_parse_timer_config(pfile_t *pf, value_t timeout, 
  value_t intr)
{
  expr_t   *exprstk;
  value_t   tmp;
  value_t   dst;
  boolean_t is_reset;
  value_t   prescale_lookup;

  prescale_lookup = pfile_value_find(pf, pfile_log_none, PFILE_LENGTH_DEF,
      "_prescale_lookup");
  if (!prescale_lookup) {
    static const variable_const_t lookup[] = { 0, 1, 2, 2, 3, 3, 3, 3};
    size_t     ii;
    variable_t var;

    pfile_value_alloc(pf, "_prescale_lookup", 
      VARIABLE_FLAG_CONST | VARIABLE_FLAG_GLOBAL,
        1, COUNT(lookup), 0, &prescale_lookup);
    var = value_variable_get(prescale_lookup);

    for (ii = 0; ii < COUNT(lookup); ii++) {
      variable_const_set(var, ii, lookup[ii]);
    }
  }

  if ((value_is_const(intr)
      && value_const_get(intr))
    || !value_is_const(intr)) {
    static const char *str = "_timerreset";

    dst = pfile_value_find(pf, pfile_log_none, PFILE_LENGTH_DEF, str);
    if (!dst) {
      pfile_value_alloc(pf, str, 
        VARIABLE_FLAG_VOLATILE | VARIABLE_FLAG_GLOBAL, 2, 1, 0, &dst);
    }
    is_reset = boolean_true;
  } else {
    dst = pfile_value_find(pf, pfile_log_err, PFILE_LENGTH_DEF, "_tmr1");
    is_reset = boolean_false;
  }
  if (dst) {
    value_t tmp_lookup;
    value_t tmp_lo;
    /* tmp = freq * timeout / 4000   */

    /* disable the timer interrupt */
    bsc_assign_reg_reg(pf, operator_bitclr, "_pie1", "_tmr1ie");
    bsc_assign_reg_reg(pf, operator_bitclr, "_pir1", "_tmr1if");
    /* setup the timeout */
    exprstk = 0;
    bsc_expr_add_reg(pf, &exprstk, "_freq", operator_mul);
    bsc_expr_add_var(pf, &exprstk, timeout, operator_div);
    bsc_expr_add_const(pf, &exprstk, 4000, operator_null);
    tmp = expr_val_get(exprstk);
    value_lock(tmp);
    expr_list_free(&exprstk);

    /* tmp = full value, let's find the prescale. the prescale value
     * is prescale_lookup(tmp >> 16) */
    bsc_expr_add_var(pf, &exprstk, tmp, operator_shift_right);
    bsc_expr_add_const(pf, &exprstk, 16, operator_null);

    tmp_lookup = expr_val_get(exprstk);
    value_lock(tmp_lookup);
    expr_list_free(&exprstk);

    /* tmp >>= lookup[tmp_lookup] */
    abort(); /* the baseofs_set below is not correct! */
    value_baseofs_set(prescale_lookup, value_variable_get(tmp_lookup));

    bsc_expr_add_const(pf, &exprstk, 0, operator_sub);
    bsc_expr_add_var(pf, &exprstk, tmp, operator_shift_right);
    bsc_expr_add_var(pf, &exprstk, prescale_lookup, operator_null);

    tmp_lo = expr_val_get(exprstk);
    value_lock(tmp_lo);
    expr_list_free(&exprstk);

    if (is_reset) {
      bsc_assign_reg_val(pf, operator_assign, "_timerreset", tmp_lo);
    }
    bsc_assign_reg_val(pf, operator_assign, "_tmr1", tmp_lo);

    /* finally, _t1con = _t1con & ~48 | (tmp_lookup << 4) */
    bsc_expr_add_reg(pf, &exprstk, "_t1con", operator_assign);
    bsc_expr_add_reg(pf, &exprstk, "_t1con", operator_andb);
    bsc_expr_add_const(pf, &exprstk, ~48, operator_orb);
    bsc_expr_add_var(pf, &exprstk, tmp_lookup, operator_shift_left);
    bsc_expr_add_const(pf, &exprstk, 4, operator_null);
    expr_list_free(&exprstk);

    value_release(tmp_lo);
    value_release(tmp_lookup);
    /* if intr is const 0, the timer isr is not used (here)
     * if intr is const non-0, or non-const timer isr is used
     */
    bsc_parse_intr_config(pf, intr, bsc_isr_type_timer, "Timer", "_pie1", 
      "_tmr1ie");
    value_release(tmp);
    value_release(dst);
  }
  value_release(prescale_lookup);
}

static void bsc_parse_pwm_config(pfile_t *pf, value_t period, 
  value_t duty, value_t channel, value_t intr)
{
  value_t tmp;
  value_t pr2;

  tmp     = 0;

  /* disable the interrupt */
  bsc_assign_reg_reg(pf, operator_bitclr, "_pie1", "_tmr2ie");
  bsc_assign_reg_reg(pf, operator_bitclr, "_pir1", "_tmr2if");

  bsc_parse_pwm_period(pf, period, &pr2);
  bsc_parse_pwm_duty(pf, duty, channel, pr2);
  if (pr2) {
    value_release(pr2);
  }
  /* enable the interrupt */
  bsc_parse_intr_config(pf, intr, bsc_isr_type_pwm, "PWM", "_pie1", 
      "_tmr2ie");

  /* _ccp1_mplex_reg<_ccp1_mplex_bit> must be cleared when using the
     CCP module */
  bsc_assign_reg_reg(pf, operator_bitclr, "_ccp1_mplex_reg", 
    "_ccp1_mplex_bit");
  /* turn on timer 2 */
  bsc_assign_reg_reg(pf, operator_bitset, "_t2con", "_tmr2on");
  if (tmp) {
    value_release(tmp);
  }
}

static void bsc_parse_interrupt_portb(pfile_t *pf, value_t enable,
    value_t pullup)
{
  label_t  lbl_enable;
  label_t  lbl_end;

  /* disable the interrupt */
  bsc_assign_reg_reg(pf, operator_bitclr, "_intcon", "_rbie");
  bsc_assign_reg_reg(pf, operator_bitclr, "_intcon", "_rbif");

  /* enable the interrupt */
  bsc_parse_intr_config(pf, enable, bsc_isr_type_rb, "RB", "_intcon", 
      "_rbie");
  lbl_enable = 0;
  lbl_end    = 0;
  if ((result_ok == pfile_label_alloc(pf, 0, &lbl_enable))
      && (result_ok == pfile_label_alloc(pf, 0, &lbl_end))) {
    pfile_cmd_branch_add(pf, cmd_branchtype_goto, cmd_branchcond_true, 
        lbl_enable, pullup, 0, 0);
    bsc_assign_reg_reg(pf, operator_bitclr, "_option_reg", "_not_rbpu");
    pfile_cmd_branch_add(pf, cmd_branchtype_goto, cmd_branchcond_none,
        lbl_end, 0, 0, 0);
    pfile_cmd_label_add(pf, lbl_enable);
    bsc_assign_reg_reg(pf, operator_bitset, "_option_reg", "_not_rbpu");
    pfile_cmd_label_add(pf, lbl_end);
  }
  label_release(lbl_end);
  label_release(lbl_enable);
}

static void bsc_parse_eeprom_write(pfile_t *pf, value_t addr,
    value_t val)
{
  value_t  cx55; /* 0x55 */

  if (result_ok == pfile_constant_get(pf, 0x55, &cx55)) {
    value_t cxaa; /* 0xaa */
    if (result_ok == pfile_constant_get(pf, 0xaa, &cxaa)) {
      value_t c1;

      if (result_ok == pfile_constant_get(pf, 1, &c1)) {
        expr_t *exprstk;

        exprstk = 0;
        /* tmp = _intcon & (1 << _gie) */
        bsc_expr_add_reg(pf, &exprstk, "_intcon", operator_andb);
        bsc_expr_add_var(pf, &exprstk, c1, operator_shift_left);
        bsc_expr_add_reg(pf, &exprstk, "_gie", operator_null);

        bsc_reg_bit_wait(pf, "_eecon1", "_wr", cmd_branchcond_true);
        bsc_assign_reg_val(pf, operator_assign, "_eeadr", addr);
        bsc_assign_reg_val(pf, operator_assign, "_eedata", val);
        bsc_assign_reg_reg(pf, operator_bitclr, "_eecon1", "_eepgd");
        bsc_assign_reg_reg(pf, operator_bitset, "_eecon1", "_wren");
        bsc_assign_reg_reg(pf, operator_bitclr, "_intcon", "_gie");
        bsc_assign_reg_val(pf, operator_assign, "_eecon2", cx55);
        bsc_assign_reg_val(pf, operator_assign, "_eecon2", cxaa);
        bsc_assign_reg_reg(pf, operator_bitset, "_eecon1", "_wr");

        bsc_assign_reg_val(pf, operator_orb, "_intcon", expr_val_get(exprstk));
        bsc_assign_reg_reg(pf, operator_bitclr, "_eecon1", "_wren");
        expr_list_free(&exprstk);
        value_release(c1);
      }
      value_release(cxaa);
    }
    value_release(cx55);
  }
}

static void bsc_parse_eeprom_read(pfile_t *pf, value_t addr,
    value_t dst)
{
  bsc_assign_reg_val(pf, operator_assign, "_eeadr", addr);
  bsc_assign_reg_reg(pf, operator_bitclr, "_eecon1", "_eepgd");
  bsc_assign_reg_reg(pf, operator_bitset, "_eecon1", "_rd");
  bsc_assign_val_reg(pf, operator_assign, dst, "_eedata");
}


static void bsc_intrin_warn(pfile_t *pf, 
    bsc_file_flag_t flag, const char *msg)
{
  if ((bsc_file_pass_get(pf) == 2)
    && (!bsc_file_flag_test(pf, flag))) {
    pfile_log(pf, pfile_log_warn, "%s", msg);
  }
}
        
void bsc_parse_intrinsic(pfile_t *pf, unsigned sub, size_t param_ct,
  const param_t *params)
{
  static const char *port_data[] = { "_porta", "_portb", "_portc", 
                                "_portd", "_porte"};
  static const char *port_ctl[] = { "_trisa", "_trisb", "_trisc",
                                "_trisd", "_trise" };

  operator_t  op;
  expr_t     *exprstk;
  cmd_t       cmd;

  exprstk = 0;
  cmd = bsc_cmd_alloc(sub, param_ct, params);
  if (cmd) {
    pfile_cmd_add(pf, cmd);
  }
  /* pfile_cmd_intrinsic_add(pf, 0, sub, param_ct, params); */
  op = operator_null;
  switch ((bsc_intrinsic_type_t) sub) {
    case bsc_intrinsic_type_null:
      break;
    /**************************************************
     * misc. commands
     **************************************************/
    case bsc_intrinsic_type_clear:
      /* call the CLEAR function */
      pfile_cmd_special_add(pf, cmd_type_clear, 0);
      break;
    case bsc_intrinsic_type_end:
      pfile_cmd_special_add(pf, cmd_type_end, 0);
      /* jump to the END label */
      break;
    case bsc_intrinsic_type_delay:
      /* create a delay (this is complex!) */
      break;
    case bsc_intrinsic_type_interrupt_disable:
      op = operator_bitclr;
      /* fall through */
    case bsc_intrinsic_type_interrupt_enable:
      bsc_file_flag_set(pf, bsc_file_flag_interrupt_enable);
      if (operator_null == op) {
        op = operator_bitset;
      }
      bsc_expr_add_reg(pf, &exprstk, "_intcon", op);
      bsc_expr_add_reg(pf, &exprstk, "_gie", operator_null);
      expr_list_free(&exprstk);
      /* bsc_expr_assign_reg(pf, &exprstk, "intcon"); */
      bsc_expr_add_reg(pf, &exprstk, "_intcon", op);
      bsc_expr_add_reg(pf, &exprstk, "_peie", operator_null);
      /* bsc_expr_assign_reg(pf, &exprstk, "intcon"); */
      break;
    /************************************************** 
     * ADC commands 
     **************************************************/
    case bsc_intrinsic_type_adc_config:
      bsc_file_flag_set(pf, bsc_file_flag_adc_config);
      bsc_intrin_warn(pf, bsc_file_flag_adc_start,
          "ADC is never started");
      bsc_parse_adc_config(pf, params[0].u.var, params[1].u.var,
          params[2].u.var, params[3].u.var, params[4].u.var);
      break;
    case bsc_intrinsic_type_adc_on:
      bsc_file_flag_set(pf, bsc_file_flag_adc_start);
      bsc_intrin_warn(pf, bsc_file_flag_adc_config,
          "ADC is never configured");
      op = operator_bitset;
      /* fall through */
    case bsc_intrinsic_type_adc_off:
      if (operator_null == op) {
        op = operator_bitclr;
      }
      bsc_expr_add_reg(pf, &exprstk, "_adcon0", operator_bitset);
      bsc_expr_add_reg(pf, &exprstk, "_adon", operator_null);
      break;
    case bsc_intrinsic_type_adc_start:
      bsc_file_flag_set(pf, bsc_file_flag_adc_start);
      bsc_expr_add_reg(pf, &exprstk, "_adcon0", operator_bitset);
      bsc_expr_add_reg(pf, &exprstk, "_go", operator_null);
      break;
    case bsc_intrinsic_type_adc_store:
      bsc_intrin_warn(pf, bsc_file_flag_adc_start,
          "ADC is never started");
      {
        value_t val;

        val = params[0].u.var;
        if (1 == value_sz_get(val)) {
          bsc_assign_val_reg(pf, operator_assign, val, "_adresl");
        } else {
          value_t val_hi;
          value_t vofs;

          value_clone(&val_hi, val);
          value_sz_set(val, 1);
          value_sz_set(val_hi, value_sz_get(val_hi) - 1);
          pfile_constant_get(pf, 
              value_const_get(value_baseofs_get(val_hi)) + 1, &vofs);
          value_baseofs_set(val_hi, vofs);
          value_release(vofs);
          bsc_assign_val_reg(pf, operator_assign, val, "_adresl");
          bsc_assign_val_reg(pf, operator_assign, val_hi, "_adresh");
          value_release(val_hi);
        }
      }
      break;
    case bsc_intrinsic_type_adc_wait:
      /* loop while adcon0:go is 0 */
      bsc_intrin_warn(pf, bsc_file_flag_adc_start,
          "ADC is never started");
      bsc_reg_bit_wait(pf, "_adcon0", "_go", cmd_branchcond_true);
      break;
    /************************************************** 
     * PWM commands 
     **************************************************/
    case bsc_intrinsic_type_pwm_config:
      /* param[0] = period                    */
      /* param[1] = duty     (0..100)         */
      /* param[2] = channel (1, 2, 3 = both)  */
      /* param[3] = 0 (poll) 1 (interrupt)    */
      bsc_parse_pwm_config(pf, params[0].u.var, params[1].u.var,
        params[2].u.var, params[3].u.var);
      break;
    case bsc_intrinsic_type_pwm_duty:
      /* param[0] = duty cycle (0..100)      */
      /* param[1] = channel (1, 2, 3 = both) */
      bsc_parse_pwm_duty(pf, params[0].u.var, params[1].u.var, 0);
      break;
    case bsc_intrinsic_type_pwm_off:
      if ((0 == params[0].u.sval)
        || (2 == params[0].u.sval)) {
        bsc_expr_add_reg(pf, &exprstk, "_ccp1con", operator_assign);
        bsc_expr_add_const(pf, &exprstk, 0, operator_null);
        expr_list_free(&exprstk);
      }

      if ((1 == params[0].u.sval)
        || (2 == params[0].u.sval)) {
        value_t ccp2con;

        ccp2con = pfile_value_find(pf, pfile_log_none, PFILE_LENGTH_DEF,
          "_ccp2con");
        if (ccp2con) {
          bsc_expr_add_var(pf, &exprstk, ccp2con, operator_assign);
          bsc_expr_add_const(pf, &exprstk, 0, operator_null);
          expr_list_free(&exprstk);
          value_release(ccp2con);
        }
      }
      break;
    case bsc_intrinsic_type_pwm_period:
      /* long & complex */
      bsc_parse_pwm_period(pf, params[0].u.var, 0);
      break;
    case bsc_intrinsic_type_pwm_reg:
      /* long & complex */
      break;
    /************************************************** 
     * port commands 
     **************************************************/
    case bsc_intrinsic_type_pinhigh:
      op = operator_bitset;
      /* fall throught */
    case bsc_intrinsic_type_pinlow:
      if (operator_null == op) {
        op = operator_bitclr;
      }
      if (!pfile_variable_exists(pf, PFILE_LENGTH_DEF, 
            port_data[params[0].u.sval])) {
        bsc_file_log_unavail(pf, port_data[params[0].u.sval]);
      } else {
        bsc_expr_add_reg(pf, &exprstk, port_data[params[0].u.sval], op);
        bsc_expr_add_var(pf, &exprstk, params[1].u.var, operator_null);
      }
      break;
    case bsc_intrinsic_type_setport:
      if (!pfile_variable_exists(pf, PFILE_LENGTH_DEF,
            port_data[params[0].u.sval])) {
        bsc_file_log_unavail(pf, port_data[params[0].u.sval]);
      } else {
        bsc_expr_add_reg(pf, &exprstk, port_ctl[params[0].u.sval],
          operator_assign);
        bsc_expr_add_var(pf, &exprstk, params[1].u.var, operator_null);
      }
      break;
    case bsc_intrinsic_type_pinrd:
      if (!pfile_variable_exists(pf, PFILE_LENGTH_DEF,
            port_data[params[0].u.sval])) {
        bsc_file_log_unavail(pf, port_data[params[0].u.sval]);
      } else {
        bsc_expr_add_var(pf, &exprstk, params[2].u.var, operator_assign);
        bsc_expr_add_reg(pf, &exprstk, port_data[params[0].u.sval], 
            operator_bittst);
        bsc_expr_add_var(pf, &exprstk, params[1].u.var, operator_null);
      }
      break;
    case bsc_intrinsic_type_portout:
      if (!pfile_variable_exists(pf, PFILE_LENGTH_DEF,
            port_data[params[0].u.sval])) {
        bsc_file_log_unavail(pf, port_data[params[0].u.sval]);
      } else {
        bsc_assign_reg_val(pf, operator_assign, port_data[params[0].u.sval],
            params[1].u.var);
      }
      break;
    case bsc_intrinsic_type_portrd:
      if (!pfile_variable_exists(pf, PFILE_LENGTH_DEF,
            port_data[params[0].u.sval])) {
        bsc_file_log_unavail(pf, port_data[params[0].u.sval]);
      } else {
        bsc_assign_val_reg(pf, operator_assign, params[1].u.var,
            port_data[params[0].u.sval]);
      }
      break;
    /************************************************** 
     * usart commands 
     **************************************************/
    case bsc_intrinsic_type_rx_err_check:
      /* dst = (RCSTA & 0x06) >> 1 */
      /*     --> RCSTA/2 & 0x03 */
      bsc_intrin_warn(pf, bsc_file_flag_usart_config_rx,
          "USART RX is never configured");
      bsc_expr_add_var(pf, &exprstk, params[0].u.var, operator_assign);
      bsc_expr_add_reg(pf, &exprstk, "_rcsta", operator_shift_right);
      bsc_expr_add_const(pf, &exprstk, 1, operator_andb);
      bsc_expr_add_const(pf, &exprstk, 3, operator_null);
      break;
    case bsc_intrinsic_type_rx_store:
      /* dst = rcreg */
      bsc_intrin_warn(pf, bsc_file_flag_usart_config_rx,
          "USART RX is never configured");
      bsc_expr_add_var(pf, &exprstk, params[0].u.var, operator_assign);
      bsc_expr_add_reg(pf, &exprstk, "_rcreg", operator_null);
      break;
    case bsc_intrinsic_type_tx_load:
      /* txreg = src */
      bsc_intrin_warn(pf, bsc_file_flag_usart_config_tx,
          "USART TX is never configured");
      bsc_expr_add_reg(pf, &exprstk, "_txreg", operator_assign);
      bsc_expr_add_var(pf, &exprstk, params[0].u.var, operator_null);
      break;
    case bsc_intrinsic_type_usart_config:
      /* big, long, ugly */
      bsc_parse_usart_config(pf, params[0].u.sval, params[2].u.var, 
          params[3].u.var);
      break;
    case bsc_intrinsic_type_wait_rx:
      bsc_intrin_warn(pf, bsc_file_flag_usart_config_rx,
          "USART RX is never configured");
      bsc_reg_bit_wait(pf, "_pir1", "_rcif", cmd_branchcond_false);
      break;
    case bsc_intrinsic_type_wait_tx:
      /* we need 2 NOPs to protect against someone immediately
       * following tx_load */
      bsc_intrin_warn(pf, bsc_file_flag_usart_config_rx,
          "USART TX is never configured");
      pfile_cmd_special_add(pf, cmd_type_nop, 0);
      pfile_cmd_special_add(pf, cmd_type_nop, 0);
      bsc_reg_bit_wait(pf, "_pir1", "_txif", cmd_branchcond_false);
      break;
    /************************************************** 
     * usart commands 
     **************************************************/
    case bsc_intrinsic_type_timer_config:
      /* long & complex */
      bsc_intrin_warn(pf, bsc_file_flag_timer_start,
          "timer is never started");
      bsc_file_flag_set(pf, bsc_file_flag_timer_config);
      bsc_parse_timer_config(pf, params[0].u.var, params[1].u.var);
      break;
    case bsc_intrinsic_type_timer_start:
      bsc_file_flag_set(pf, bsc_file_flag_timer_start);
      bsc_intrin_warn(pf, bsc_file_flag_timer_config,
          "timer is never configured");
      bsc_expr_add_reg(pf, &exprstk, "_t1con", operator_bitset);
      bsc_expr_add_reg(pf, &exprstk, "_tmr1on", operator_null);
      break;
    case bsc_intrinsic_type_timer_stop:
      bsc_expr_add_reg(pf, &exprstk, "_t1con", operator_bitclr);
      bsc_expr_add_reg(pf, &exprstk, "_tmr1on", operator_null);
      expr_list_free(&exprstk);
      bsc_expr_add_reg(pf, &exprstk, "_pie1", operator_bitclr);
      bsc_expr_add_reg(pf, &exprstk, "_tmr1ie", operator_bitclr);
      break;
    case bsc_intrinsic_type_timer_countdown:
      bsc_intrin_warn(pf, bsc_file_flag_timer_start,
          "timer is never started");
      bsc_reg_bit_wait(pf, "_pir1", "_tmr1if", cmd_branchcond_false);
      bsc_expr_add_reg(pf, &exprstk, "_pie1", operator_bitclr);
      bsc_expr_add_reg(pf, &exprstk, "_tmr1ie", operator_bitclr);
      break;
    case bsc_intrinsic_type_timer_read:
      bsc_intrin_warn(pf, bsc_file_flag_timer_start,
          "timer is never started");
      bsc_assign_val_reg(pf, operator_assign, params[0].u.var, "_tmr1");
      break;
    case bsc_intrinsic_type_timer_write:
      bsc_intrin_warn(pf, bsc_file_flag_timer_start,
          "timer is never started");
      bsc_assign_reg_val(pf, operator_assign, "_tmr1", params[0].u.var);
      break;
    case bsc_intrinsic_type_sleep:
      pfile_cmd_special_add(pf, cmd_type_sleep, 0);
      break;
    case bsc_intrinsic_type_interrupt_portb:
      bsc_parse_interrupt_portb(pf, params[0].u.var, params[1].u.var);
      break;
    /************************************************** 
     * EEPROM commands 
     **************************************************/
    case bsc_intrinsic_type_eeprom_write:
      bsc_parse_eeprom_write(pf, params[0].u.var, params[1].u.var);
      break;
    case bsc_intrinsic_type_eeprom_read:
      bsc_parse_eeprom_read(pf, params[0].u.var, params[1].u.var);
      break;
    case bsc_intrinsic_type_eeprom_wait:
      bsc_reg_bit_wait(pf, "_eecon1", "_wr", cmd_branchcond_true);
      break;
  }
  expr_list_free(&exprstk);
}

void bsc_parse_ir(pfile_t *pf, unsigned sub, size_t param_ct,
  const param_t *params)
{
  label_t         skip;
  label_t         isr;
  const char     *eword;
  bsc_isr_type_t  isrtype;
  const char     *reg;
  const char     *bit;

  isrtype  = bsc_isr_type_first;
  reg      = 0;
  bit      = 0;
  eword    = 0;
  switch ((bsc_ir_type_t) sub) {
    case bsc_ir_type_null: 
      break;
    case bsc_ir_type_adc:  
      eword    = "ir_adc_end"; 
      reg      = "_pir1";
      bit      = "_adif";
      isrtype  = bsc_isr_type_adc;
      break;
    case bsc_ir_type_pwm:  
      eword    = "ir_pwm_end"; 
      isrtype  = bsc_isr_type_pwm;
      reg      = "_pir2";
      bit      = "_tmr2if";
      break;
    case bsc_ir_type_rx:   
      eword    = "ir_rx_end";  
      isrtype  = bsc_isr_type_rx;
      reg      = "_pir1";
      bit      = "_rcif";
      break;
    case bsc_ir_type_tx:   
      eword    = "ir_tx_end"; 
      isrtype  = bsc_isr_type_tx;
      reg      = "_pir1";
      bit      = "_txif";
      break;
    case bsc_ir_type_timer:
      eword    = "ir_timer_end"; 
      isrtype  = bsc_isr_type_timer;
      reg      = "_pir1";
      bit      = "_tmr1if";
      break;
    case bsc_ir_type_rb:
      eword    = "ir_portb_end";
      isrtype  = bsc_isr_type_rb;
      reg      = "_intcon";
      bit      = "_rbif";
      break;
    case bsc_ir_type_user:
      eword    = "ir_user_end";
      isrtype  = bsc_isr_type_user;
      reg      = 0;
      bit      = 0;
  }
    
  if ((bsc_file_pass_get(pf) == 2)
    && !bsc_file_flag_test(pf, bsc_file_flag_interrupt_enable)) {
    pfile_log(pf, pfile_log_warn, BSC_MSG_INTERRUPTS_NOT_ENABLED);
  }
  if (result_ok == pfile_label_alloc(pf, 0, &skip)) {
    value_t  tmp;
    expr_t  *exprstk;

    exprstk = 0;
    if (result_ok == bsc_file_isr_get(pf, isrtype, boolean_true, &isr)) {
      if (bsc_isr_type_user == isrtype) {
        label_usage_bump(isr, ctr_bump_incr);
      }
      if ((bsc_file_pass_get(pf) == 1)
        && (label_flag_test(isr, LABEL_FLAG_DEFINED))) {
          pfile_log(pf, pfile_log_err, BSC_MSG_MULTI_DEF,
            label_name_get(isr));
      } else {
        label_flag_set(isr, LABEL_FLAG_DEFINED);
        if ((bsc_file_pass_get(pf) > 1) 
            && !label_flag_test(isr, LABEL_FLAG_USED)) {
          pfile_log(pf, pfile_log_warn, BSC_MSG_NEVER_USED,
              label_name_get(isr));
        }
      }
    } else {
      isr = 0;
    }

    pfile_cmd_branch_add(pf, cmd_branchtype_goto,
      cmd_branchcond_none, skip, 0, 0, 0);
  /* put this command on the previous line (makes
     the listing look a bit better */
    {
      cmd_t cmd;

      cmd = pfile_cmdlist_get(pf);
      if (cmd) {
        cmd_line_set(cmd, cmd_line_get(cmd) - 1);
      }
    }
    if (isr) {
      pfile_cmd_label_add(pf, isr);
    }
    /* if (reg & bit) == 0, goto nobit */
    if (reg) {
      if (result_ok == pfile_value_temp_get(pf, 1, 0, &tmp)) {
        bsc_expr_add_var(pf, &exprstk, tmp, operator_assign);
        bsc_expr_add_reg(pf, &exprstk, reg, operator_bittst);
        bsc_expr_add_reg(pf, &exprstk, bit, operator_null);
        pfile_cmd_branch_add(pf, cmd_branchtype_goto, cmd_branchcond_false,
          bsc_file_isr_next_get(pf, isrtype), expr_val_get(exprstk), 0, 0);
        expr_list_free(&exprstk);
        value_release(tmp);
      }
    }
    if (bsc_isr_type_timer == isrtype) {
      value_t r1;
      /* and reset the timer */
      r1 = pfile_value_find(pf, pfile_log_err, PFILE_LENGTH_DEF, "_tmr1");
      if (r1) {
        value_t b1;

        b1 = pfile_value_find(pf, pfile_log_none, PFILE_LENGTH_DEF, 
              "_timerreset");
        if (b1) {
          pfile_cmd_op_add(pf, operator_assign, &r1, b1, 0);
          value_release(b1);
        }
        value_release(r1);
      }
    }
    bsc_block_process(pf, BSC_BLOCK_PROCESS_FLAG_SKIP_STMT, 1, &eword, 0);
    if ((bsc_ir_type_rx != sub) 
        && (bsc_ir_type_tx != sub)
        && (bsc_ir_type_user != sub)) {
      /* clear the bit */
      value_t r1;

      r1 = pfile_value_find(pf, pfile_log_err, PFILE_LENGTH_DEF, reg);
      if (r1) {
        value_t b1;

        b1 = pfile_value_find(pf, pfile_log_err, PFILE_LENGTH_DEF, bit);
        if (b1) {
          pfile_cmd_op_add(pf, operator_bitclr, &r1, b1, 0);
          value_release(b1);
        }
        value_release(r1);
      }
    }
    pfile_cmd_branch_add(pf, cmd_branchtype_goto, 
      cmd_branchcond_none, bsc_file_isr_next_get(pf, isrtype), 0, 0, 0);
    pfile_cmd_label_add(pf, skip);
    label_release(skip);
    label_release(isr);
  }
}

/*
 * NAME
 *   bsc_parse_print
 *
 * DESCRIPTION
 *   parse the print command
 *
 * PARAMETERS
 *
 * RETURN
 *
 * NOTES
 *   format:
 *      print [str|expr[,...]]
 */

void bsc_parse_print(pfile_t *pf, unsigned sub, size_t param_ct, 
  const param_t *params)
{
  pf_token_get_t  which;
  value_t         txreg;

  which = pf_token_current;
  txreg = pfile_value_find(pf, pfile_log_none, PFILE_LENGTH_DEF, "_txreg");
  bsc_intrin_warn(pf, bsc_file_flag_usart_config_tx, "USART TX never configured");
  do {
    const char *ptr;
    value_t     val;

    ptr = pf_token_get(pf, which);
    which = pf_token_next;
    if ('"' == *ptr) {
      if (result_ok == bsc_token_to_str(pf, pfile_log_err, &val)) {
        /* dump the val */
        /* value_use_ct_bump(val, ctr_incr); */
        pf_token_get(pf, pf_token_next);
        /* the instruction count is 6 + ct for the loop, and 3 * ct for
           inline, so we'll inline anything <= 3 and loop > 3 */
        if (txreg) {
          if (value_ct_get(val) <= 3) {
            variable_ct_t ii;

            for (ii = value_ct_get(val); ii; ii--) {
              value_t vdup;

              bsc_reg_bit_wait(pf, "_pir1", "_txif", cmd_branchcond_false);
              if (result_ok == value_clone(&vdup, val)) {
                value_t vii;

                pfile_constant_get(pf, ii - 1, &vii);
                value_baseofs_set(vdup, vii);
                value_ct_set(vdup, 1);
                value_release(vii);

                pfile_cmd_op_add(pf, operator_assign, &txreg, vdup, 0);
                value_release(vdup);
              }
              if (ii - 1) {
                /* we need to add a couple of NOPs here because
                   it takes a bit for _pir1:_txif to go high */
                pfile_cmd_special_add(pf, cmd_type_nop, 0);
                pfile_cmd_special_add(pf, cmd_type_nop, 0);
              }
            }
          } else {
            label_t loop;

            if (result_ok == pfile_label_alloc(pf, 0, &loop)) {
              value_t loop_ctr;

              if (result_ok == pfile_value_temp_get(pf, 1, 0, &loop_ctr)) {
                value_t loop_end;

                value_baseofs_set(val, loop_ctr);
                if (result_ok == pfile_constant_get(pf, value_ct_get(val), 
                      &loop_end)) {
                  value_ct_set(val, 1);
                  pfile_cmd_op_add(pf, operator_assign, &loop_ctr, 
                    loop_end, 0);
                  pfile_cmd_label_add(pf, loop);
                  bsc_reg_bit_wait(pf, "_pir1", "_txif", cmd_branchcond_false);
                  pfile_cmd_op_add(pf, operator_decr, &loop_ctr, 0, 0);
                  pfile_cmd_op_add(pf, operator_assign, &txreg, val, 0);
                  pfile_cmd_branch_add(pf, cmd_branchtype_goto, 
                    cmd_branchcond_true, loop, loop_ctr, 0, 0);
                  value_release(loop_end);
                }
                value_release(loop_ctr);
              }
            }
            label_release(loop);
          }
        }
        value_release(val);
      }
    } else {
      value_t       dst;
      label_t       print_fn;
      const char   *print_fn_name;
      variable_sz_t sz;

      if (result_ok == bsc_expr_parse(pf, 0, 0, &val)) {
        if (txreg) {
          dst = pfile_value_find(pf, pfile_log_none, PFILE_LENGTH_DEF, 
              "_printval");
          /* the size of printval will be the size of the largest value
           * printed */
          if (!dst) {
            pfile_value_alloc(pf, "_printval", 
              VARIABLE_FLAG_GLOBAL 
              | VARIABLE_FLAG_AUTO 
              | VARIABLE_FLAG_ASSIGNED
              | VARIABLE_FLAG_USED, 
              value_sz_get(val), 1, 0, &dst);
          } else if (value_sz_get(dst) < value_sz_get(val)) {
            value_sz_set(dst, value_sz_get(val));
            variable_sz_set(value_variable_get(dst), value_sz_get(val));
          }
          if (dst) {
            sz = value_sz_get(dst);
            if (sz < value_sz_get(val)) {
              variable_sz_set(value_variable_get(dst), value_sz_get(val));
            }
            pfile_cmd_op_add(pf, operator_assign, &dst, val, 0);
            print_fn_name = (value_vflag_test(val, VARIABLE_FLAG_SIGNED)) 
              ? "_sprint" : "_print";
            print_fn = pfile_label_find(pf, pfile_log_none, print_fn_name);
            if (!print_fn) {
              pfile_label_alloc(pf, print_fn_name, &print_fn);
            }
            if (print_fn) {
              pfile_cmd_branch_add(pf, cmd_branchtype_call, 
                  cmd_branchcond_none, print_fn, 0, 0, 0);
              label_release(print_fn);
            }
            value_release(dst);
          }
        }
        value_release(val);
      }
    }
  } while (pf_token_is(pf, pf_token_current, pfile_log_none, ","));
  if (txreg) {
    value_release(txreg);
  } else {
    bsc_file_log_unavail(pf, "print");
  }
}

