/**********************************************************
 **
 ** bsc_proc.c : BSC PROC/CALL/IR_* parsing
 **
 ** Copyright (c) 2004, Kyle A. York
 ** All rights reserved
 **
 ***********************************************************/
#include <stdlib.h>
#include <string.h>

#include "pfile.h"
#include "pf_cmd.h"
#include "pftoken.h"
#include "pfproc.h"
#include "label.h"
#include "value.h"
#include "bsc_file.h"
#include "bsc_tokn.h"
#include "bsc_blck.h"
#include "bsc_expr.h"
#include "bsc_proc.h"

#define PROC_PARAMS_MAX 100

/*
 * NAME
 *   bsc_parse_proc
 *
 * DESCRIPTION
 *   define a procedure
 *
 * PARAMETERS
 *   pf       : pfile
 *   sub      : sub fn (== 0)
 *   param_ct : # of parameters (1)
 *   params   : parameter (label)
 *
 * RETURN
 *   none
 *
 * NOTES
 *      parse:
 *      label (need not exist)
 *      '(' [var1 [, var2...]] ')'
 *      block
 *      endproc
 */
void bsc_parse_proc(pfile_t *pf, unsigned sub, size_t param_ct, 
  const param_t *params)
{
  static const char *ewords[] = { "endproc" };
  label_t       proc_end;
  label_t       lbl;
  pfile_proc_t *proc;

  lbl = 0;
  if (1 == bsc_file_pass_get(pf)) {
    /* all this is only needed on the first pass,
       after that, skip it! */
    result_t    rc;
    boolean_t   is_new;

    rc = pf_token_to_label(pf, pfile_log_none, boolean_false, &lbl);
    if (result_ok == rc) {
      if (label_flag_test(lbl, LABEL_FLAG_DEFINED)) {
        pfile_log(pf, pfile_log_err, BSC_MSG_MULTI_DEF, pf_token_ptr_get(pf));
      }
      proc = pfile_proc_find(pf, pfile_log_err, pf_token_ptr_get(pf));
    } else if (result_not_found == rc) {
      rc = pfile_label_alloc(pf, pf_token_ptr_get(pf), &lbl);
      if (result_ok == rc) {
        pfile_proc_create(pf, lbl, &proc);
      }
    }
    if (result_ok == rc) {
      pfile_proc_enter(pf, pfile_proc_tag_get(proc));
      is_new = !label_flag_test(lbl, LABEL_FLAG_DEFINED);
      if (is_new) {
        label_flag_set(lbl, LABEL_FLAG_DEFINED);
      }
      if (pf_token_is(pf, pf_token_next, pfile_log_err, "(")) {
        size_t ii;

        pfile_block_enter(pf);
        for (ii = 0;
            (ii < PROC_PARAMS_MAX)
            && (!ii
              || pf_token_is(pf, pf_token_current, pfile_log_none, ","));
            ii++) {
          value_t        var;
          variable_def_t vdef;

          /* skip the ',' or '(' */
          pf_token_get(pf, pf_token_next);
          /* is this a type? */
          vdef = pfile_variable_def_find(pf, pfile_log_none, 
              pf_token_ptr_get(pf));
          if (vdef) {
            const char *ptr;
            flag_t      flags;

            if (pf_token_is(pf, pf_token_next, pfile_log_none,
                "volatile")) {
              flags = VARIABLE_FLAG_REFERENCE | VARIABLE_FLAG_POINTER;
              ptr = pf_token_get(pf, pf_token_next);
            } else {
              ptr = pf_token_get(pf, pf_token_current);
            }
            if (!bsc_token_is_identifier(pf)) {
              pfile_log(pf, pfile_log_err, BSC_MSG_VARIABLE_EXPECTED);
            } else {
              rc = pfile_value_alloc(pf, ptr, flags, 0, 1, vdef, &var);
              pf_token_get(pf, pf_token_next);
              if (result_ok != rc) {
                pfile_log_syserr(pf, rc);
              } else {
                if (value_vflag_test(var, VARIABLE_FLAG_REFERENCE)) {
                  value_sz_set(var, 2);
                }
                if (label_flag_test(lbl, LABEL_FLAG_USED)) {
                  value_assign_ct_bump(var, ctr_bump_incr);
                  value_use_ct_bump(var, ctr_bump_incr);
                }
                pfile_proc_param_add(proc, var);
              }
            }
          } else if (result_ok == pf_token_to_value(pf, 
                (ii) ? pfile_log_err : pfile_log_none, &var)) {
            pf_token_get(pf, pf_token_next);
            if (value_is_const(var)) {
              pfile_log(pf, pfile_log_err, BSC_MSG_VARIABLE_EXPECTED);
              value_release(var);
              var = 0;
            } else if (is_new) {
              /* this is not optimal as a parameter may never be
               * assigned or used, but I've no choice */
              value_flag_set(var, VARIABLE_FLAG_USED);
              value_flag_set(var, VARIABLE_FLAG_ASSIGNED);
              pfile_proc_param_add(proc, var);
              value_assign_ct_bump(var, ctr_bump_incr);
            } else {
              value_release(var);
            }
          } else /* if (0 == ii) */ {
            break; /* null parameter list, bail! */
          }
          if (!var) {
            break;
          }
        }
        if (pf_token_is(pf, pf_token_current, pfile_log_err, ")")) {
          pf_token_get(pf, pf_token_next);
        }
        pfile_block_leave(pf);
      }
      pfile_proc_leave(pf);
    }
  } else {
    proc = pfile_proc_find(pf, pfile_log_err, pf_token_ptr_get(pf));
    pf_token_to_label(pf, pfile_log_err, boolean_false, &lbl);
    bsc_statement_skip(pf);
    if (!label_flag_test(lbl, LABEL_FLAG_USED)) {
      pfile_log(pf, pfile_log_warn, BSC_MSG_NEVER_USED,
          label_name_get(lbl));
    }
  }
  /* process the block */
  if (result_ok != pfile_label_alloc(pf, 0, &proc_end)) {
    proc_end = 0;
  }
  if (proc_end) {
    pfile_cmd_branch_add(pf, cmd_branchtype_goto, 
      cmd_branchcond_none, proc_end, 0, 0, 0);
  }
  if (lbl) {
    pfile_cmd_label_add(pf, lbl);
    pfile_proc_enter(pf, label_name_get(lbl));
  }
  bsc_block_process(pf, BSC_BLOCK_PROCESS_FLAG_SKIP_STMT, 
      COUNT(ewords), ewords, 0);
  if (lbl) {
    pfile_proc_leave(pf);
  }
  if (proc_end) {
    pfile_cmd_label_add(pf, proc_end);
    label_release(proc_end);
  }
  label_release(lbl);
}

/*
 * NAME
 *   bsc_parse_call
 *
 * DESCRIPTION
 *   call a procedure
 *
 * PARAMETERS
 *   pf       : pfile
 :   sub      : sub fn ( == 0)
 *   param_ct : # of parameters (0)
 *   params   : parameters (0)
 *
 * RETURN
 *   none
 *
 * NOTES
 *   format: call proc( [expr1 [,expr2...]] )
 *   code:
 *     proc.param[0] = expr1
 *     proc.param[1] = expr2
 *     ...
 *     call procname
 *     for each parameter that was a variable, set it back
 */
void bsc_parse_call(pfile_t *pf, unsigned sub, size_t param_ct, 
  const param_t *params)
{
  label_t       lbl;
  pfile_proc_t *proc;
  size_t        parms_ct;
  value_t      *parms;

  /* find the procedure. it needn't exist on the first pass, but must
   * on the second */
  proc = pfile_proc_find(pf, 
    (1 == bsc_file_pass_get(pf)) ? pfile_log_none : pfile_log_err,
    pf_token_ptr_get(pf));
  if (proc) {
    lbl = pfile_proc_label_get(proc);
    label_lock(lbl);
    pfile_proc_call_add(pfile_proc_active_get(pf), proc);
  } else {
    /* create this so we'll know two things
     *   (1) it's used
     *   (2) its variables are used */
    if (result_ok == pfile_label_alloc(pf, pf_token_ptr_get(pf), &lbl)) {
      pfile_proc_create(pf, lbl, &proc);
      label_flag_set(lbl, LABEL_FLAG_USED);
      proc = 0;
    }
  }
  if (lbl && !label_flag_test(lbl, LABEL_FLAG_DEFINED)) {
    label_release(lbl);
    lbl = 0;
    proc = 0;
  }

  parms_ct = (proc) ? pfile_proc_param_ct_get(proc) : PROC_PARAMS_MAX;
  parms    = (proc && parms_ct) ? CALLOC(sizeof(*parms), parms_ct) : 0;
  if (param_ct && proc && !parms) {
    pfile_log_syserr(pf, result_memory);
  } else if (!proc && (1 != bsc_file_pass_get(pf))) {
    pfile_log(pf, pfile_log_err, BSC_MSG_NEVER_DEFINED, 
      pf_token_ptr_get(pf));
  }
  if (pf_token_is(pf, pf_token_next, pfile_log_err, "(")) {
    size_t   ct;

    if (!parms_ct) {
      pf_token_get(pf, pf_token_next);
    } else {
      for (ct = 0; 
           (ct < parms_ct)
           && (!ct || pf_token_is(pf, pf_token_current, 
             pfile_log_none, ","));
           ct++) {
        /* skip either '(' or ',' */
        pf_token_get(pf, pf_token_next);

        if (!ct && pf_token_is(pf, pf_token_current, pfile_log_none, ")")) {
          break;
        }

        if (result_ok == bsc_expr_parse(pf, 0, 0, parms ? &parms[ct] : 0)) {
        } else if ((0 == ct) && !parms) {
          break; /* null parameter list, bail */
        } else if (parms) {
          parms[ct] = 0;
        }
      }
    }
    if (pf_token_is(pf, pf_token_current, pfile_log_err, ")")) {
      pf_token_get(pf, pf_token_next);
    }
    if (lbl) {
      pfile_cmd_branch_add(pf, cmd_branchtype_call, cmd_branchcond_none,
        lbl, 0, proc, parms);
      label_release(lbl);
    } else {
      if (parms) {
        for (ct = 0; ct < parms_ct; ct++) {
          if (parms[ct]) {
            value_release(parms[ct]);
          }
        }
        FREE(parms);
      }
    }
  }
}

