/**********************************************************
 **
 ** bsc_tokn.c : BSC token parsing bits
 **
 ** Copyright (c) 2004, Kyle A. York
 ** All rights reserved
 **
 ***********************************************************/
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <math.h>
#include <errno.h>

#include "pfile.h"
#include "pftoken.h"
#include "bsc_file.h"
#include "bsc_tokn.h"


/*
 * NAME
 *   allow_*
 *
 * DESCRIPTION
 *   used by the parsing functions
 *
 * PARAMETERS
 *   ch : character to test
 *
 * RETURN
 *   TRUE if the character is allowed, FALSE if not
 *
 * NOTES
 *   none
 */
static boolean_t allow_identifier(int ch)
{
  return ISALPHA(ch) || ISDIGIT(ch) || ('_' == ch);
}

static boolean_t allow_decimal(int ch)
{
  return ISDIGIT(ch);
}

static boolean_t allow_hex(int ch)
{
  return ISXDIGIT(ch);
}

static boolean_t allow_binary(int ch)
{
  return ('0' == ch) || ('1' == ch);
}

static boolean_t allow_space(int ch)
{
  return ('\n' != ch) && ISSPACE(ch);
}

static boolean_t allow_comment(int ch)
{
  return ('\n' != ch);
}

/*
 * NAME
 *   bsc_token_get
 *
 * DESCRIPTION
 *   retrieve a token
 *
 * PARAMETERS
 *   pf    : returned by pfile_open()
 *   which : which token to get (first, current, next)
 *   dst   : [out] points to retrieved token
 *
 * RETURN
 *   none
 *
 * NOTES
 */
const char *bsc_token_get(pfile_t *pf, pf_token_get_t which)
{
  static const char xdig[] = "0123456789abcdef";
  int      ch;

  switch (which) {
    case pf_token_first:
      pfile_rewind(pf);
      /* fall through */
    case pf_token_next:
      /* skip any preceding spaces */
      ch = pf_token_parse_allow(pf, allow_space, 0);
      pf_token_reset(pf);
      if (ISALPHA(ch)) {
        ch = TOLOWER(ch);
      }
      pf_token_ch_append(pf, ch);
      if (('_' == ch) || ISALPHA(ch)) {
        /* identifier */
        ch = pf_token_parse_allow(pf, allow_identifier, 
          PF_TOKEN_PARSE_ALLOW_STORE
          | PF_TOKEN_PARSE_ALLOW_UNGET);
      } else if (ISDIGIT(ch)) {
        /* decimal constant */
        ch = pf_token_parse_allow(pf, allow_decimal, 
          PF_TOKEN_PARSE_ALLOW_STORE);
        if ('.' == ch) {
          pf_token_ch_append(pf, ch);
          /*pfile_token_get(pf, pfile_token_next, 0); */
          pf_token_parse_allow(pf, allow_decimal,
               PF_TOKEN_PARSE_ALLOW_STORE | PF_TOKEN_PARSE_ALLOW_UNGET);
        } else {
          pfile_ch_unget(pf, ch);
        }
      } else {
        switch (ch) {
          case '%': /* binary constant        */
            pf_token_parse_allow(pf, allow_binary, 
              PF_TOKEN_PARSE_ALLOW_STORE | PF_TOKEN_PARSE_ALLOW_UNGET);
            break;
          case '$': /* hex constant           */
            pf_token_parse_allow(pf, allow_hex, 
              PF_TOKEN_PARSE_ALLOW_STORE | PF_TOKEN_PARSE_ALLOW_UNGET);
            break;
          case '!': /* might be !! */
            ch = pfile_ch_get(pf);
            if ('!' == ch) {
              pf_token_ch_append(pf, ch);
            } else {
              pfile_ch_unget(pf, ch);
            }
            break;
          case '&': /* might be && */
            ch = pfile_ch_get(pf);
            if ('&' == ch) {
              pf_token_ch_append(pf, ch);
            } else {
              pfile_ch_unget(pf, ch);
            }
            break;
          case '|': /* might be || */
            ch = pfile_ch_get(pf);
            if ('|' == ch) {
              pf_token_ch_append(pf, ch);
            } else {
              pfile_ch_unget(pf, ch);
            }
            break;
          case '<': /* might be < or <= or <> */
            ch = pfile_ch_get(pf);
            if (('=' == ch) || ('>' == ch) || ('<' == ch)) {
              pf_token_ch_append(pf, ch);
            } else {
              pfile_ch_unget(pf, ch);
            }
            break;
          case '>': /* might be > or >=       */
            ch = pfile_ch_get(pf);
            if (('=' == ch) || ('>' == ch)) {
              pf_token_ch_append(pf, ch);
            } else {
              pfile_ch_unget(pf, ch);
            }
            break;
          case '\'': /* comment, skip to EOL */
            ch = pf_token_parse_allow(pf, allow_comment, 0);
            /* the next character must either be EOL or EOF */
            pf_token_reset(pf);
            pf_token_ch_append(pf, ch);
            break;
          case '\"': /* string constant, ends at EOL or " */
            {
              enum {
                ch_type_normal = 0, /* normal processing */
                ch_type_esc,        /* escape character */
                ch_type_hex,        /* hex processing   */
                ch_type_oct         /* oct processing */
              } mode;
              char val;     /* value if processing hex or oct */

              val = 0;
              mode = ch_type_normal;

              do {
                ch = pfile_ch_get(pf);
                switch (mode) {
                  case ch_type_normal:
                    if ('\\' == ch) {
                      mode = ch_type_esc;
                    } else {
                      if ((EOF != ch) && ('\n' != ch)) {
                        pf_token_ch_append(pf, ch);
                      }
                    }
                    break;
                  case ch_type_esc:
                    mode = ch_type_normal;
                    if (ISOCTDIGIT(ch)) {
                      mode = ch_type_oct;
                      val = (ch - '0');
                    } else {
                      switch (ch) {
                        case 'a': 
                          pf_token_ch_append(pf, '\a'); /* bell */
                          break; 
                        case 'b': 
                          pf_token_ch_append(pf, '\b'); /* bkpc */
                          break;
                        case 'f': 
                          pf_token_ch_append(pf, '\f'); /* ff */
                          break;
                        case 'n': 
                          pf_token_ch_append(pf, '\n'); /* lf */
                          break;
                        case 'r': 
                          pf_token_ch_append(pf, '\r'); /* cr */
                          break;
                        case 't': 
                          pf_token_ch_append(pf, '\t'); /* h.tab */
                          break;
                        case 'v': 
                          pf_token_ch_append(pf, '\v'); /* v.tab */
                          break;
                        case 'x': 
                          mode = ch_type_hex; /* 1 or 2 hex digits */
                          val  = 0;
                          break;
                        default:
                          if ((EOF != ch) && ('\n' != ch)) {
                            pf_token_ch_append(pf, ch);
                            ch = 0;
                          }
                      }
                    }
                    break;
                    
                  case ch_type_oct:
                    if (ISOCTDIGIT(ch)) {
                      val = val * 8 + (ch - '0');
                    } else {
                      pf_token_ch_append(pf, val);
                      mode = ch_type_normal;
                      pfile_ch_unget(pf, ch);
                      ch = 0;
                    }
                    break;
                  case ch_type_hex:
                    if (ISXDIGIT(ch)) {
                      val = val * 16 + (strchr(xdig, TOLOWER(ch)) - xdig);
                    } else {
                      pf_token_ch_append(pf, ch);
                      mode = ch_type_normal;
                      pfile_ch_unget(pf, ch);
                      ch = 0;
                    }
                    break;
                }
              } while ((EOF != ch) && ('\n' != ch) && ('"' != ch));
              if ('"' != ch) {
                pfile_ch_unget(pf, ch);
              }
            }
            break;
          default: /* don't know what this is, so it must be its own token */
            break;
        }
      }
      /* fall through */
    case pf_token_current:
      break;
  }
  return pf_token_ptr_get(pf);
}

/*
 * NAME
 *   bsc_token_to_str
 *
 * DESCRIPTION
 *   convert the current token to a string variable
 *
 * PARAMETERS
 *   bsc  :
 *   plog :
 *   dst  :
 *
 * RETURN
 *
 * NOTES
 */
result_t bsc_token_to_str(pfile_t *pf, pfile_log_t plog, value_t *dst)
{
  const char *ptr;
  result_t    rc;

  ptr = pf_token_get(pf, pf_token_current);
  if ('"' == *ptr) {
    value_t val;

    val = pfile_string_find(pf, strlen(ptr) - 2, ptr + 1);
    if (val) {
      rc = result_ok;
    } else {
      size_t   n;
      char     lblname[64];
      char    *strname;
      label_t  lbl;
      size_t   strsz;

      n = bsc_file_strid_next(pf);
      sprintf(lblname, "_lookup__str%u_0", n);
      strname = lblname + 8;
      strsz   = strlen(strname) - 2;
      lbl = pfile_label_find(pf, pfile_log_none, lblname);
      if (lbl) {
        /* we're already been here, nothing more to do */
        val = pfile_value_find(pf, pfile_log_err, strsz, strname);
        label_release(lbl);
        rc = (val) ? result_ok : result_not_found;
      } else {
        rc = pfile_label_alloc(pf, lblname, &lbl);
        if (result_ok == rc) {
          strname[strsz] = 0;

          strsz = strlen(ptr) - 2;
          rc = pfile_value_alloc(pf, strname, VARIABLE_FLAG_CONST, 1, strsz,
            0, &val);
          if (result_ok == rc) {
            size_t     ii;
            variable_t var;

            var = value_variable_get(val);
            for (ii = strsz; ii; ii--) {
              /* strings are read backwards when printed, 
               * so record it backwards */
              variable_const_set(var, strsz - ii, ptr[ii]);
            }
            label_release(lbl);
          }
        }
      }
    }
    if (result_ok == rc) {
      *dst = val;
    }
  } else {
    pfile_log(pf, plog, BSC_MSG_STRING_EXPECTED);
    rc = result_invalid;
  }
  return rc;
}

/*
 * NAME
 *   bsc_token_to_constant
 *
 * DESCRIPTION
 *   convert the current token to an unnamed constant
 *
 * PARAMETERS
 *   pf   : returned by pfile_open()
 *   plog : how to log an invalid result
 *   dst  : [out] holds result on success
 *
 * RETURN
 *   0      : no error
 *   EINVAL : current token is not a constant
 *   ERANGE : constant is too large
 *
 * NOTES
 *   [dst] will be *locked* on successful return
 */
result_t bsc_token_to_constant(pfile_t *pf, pfile_log_t plog,
    value_t *dst)
{
  int            base;
  const char    *ptr;
  unsigned long  n;
  result_t       rc;
  value_t        val;

  ptr = pf_token_get(pf, pf_token_current);
  if (ISDIGIT(*ptr)) {
    base = 10;
  } else if ('$' == *ptr) {
    ptr++;
    base = 16;
  } else if ('%' == *ptr) {
    ptr++;
    base = 2;
  } else {
    base = 0;
  }
  if (0 == base) {
    rc = result_invalid;
  } else {
    char *eptr;

    rc = result_ok;
    n = strtoul(ptr, &eptr, base);
    if ((ULONG_MAX == n) && (ERANGE == errno)) {
      rc = result_range;
    } else if (*eptr) {
      if (10 != base) {
        rc = result_invalid;
      } else {
        if ('.' == *eptr) {
          n = ceil(1000.0 * strtod(ptr, 0));
        }
      }
    }
    if (result_ok == rc) {
      rc = pfile_constant_get(pf, n, &val);
    }
  }
  if (result_invalid == rc) {
    pfile_log(pf, plog, BSC_MSG_CONSTANT_EXPECTED);
  } else if (result_range == rc) {
    pfile_log(pf, pfile_log_err, BSC_MSG_CONSTANT_RANGE);
  } else if (result_ok == rc) {
    *dst = val;
  }
  return rc;
}

const char *bsc_token_hack(pfile_t *pf)
{
  int      ch;
  size_t   ct;

  /* skip any spaces */
  do {
    ch = pfile_ch_get(pf);
  } while (('\n' != ch) && ISSPACE(ch));

  /* read the next word */
  pf_token_reset(pf);
  pf_token_ch_append(pf, ch);
  for (ct = 1; ; ct++) {
    ch = pfile_ch_get(pf);
    if ((EOF == ch) || ISSPACE(ch)) {
      pfile_ch_unget(pf, ch);
      break; /* <--- */
    }
    pf_token_ch_append(pf, TOLOWER(ch));
  }
  return pf_token_ptr_get(pf);
}

boolean_t bsc_token_is_eos(pfile_t *pf)
{
  return pf_token_is_eof(pf)
    || pf_token_is(pf, pf_token_current, pfile_log_none, "\n");
}

boolean_t bsc_token_is_identifier(pfile_t *pf)
{
  const char *ptr;

  ptr = pf_token_get(pf, pf_token_current);
  return ('_' == *ptr) || ISALPHA(*ptr);
}

