/**********************************************************
 **
 ** cmd_asm.c : manipulators for cmd_asm_t
 **
 ** Copyright (c) 2005, Kyle A. York
 ** All rights reserved
 **
 ***********************************************************/
#include <stdio.h>
#include <assert.h>
#include "../libutils/mem.h"
#include "../libpic12/pic_inst.h"
#include "cmdd.h"
#include "cmd_asm.h"

static void cmd_asm_dump_value(value_t val, const char *str, FILE *dst)
{
  if (val) {
    if (value_name_get(val)) {
      fprintf(dst, "%s%s", str, value_name_get(val));
    } else {
      fprintf(dst, "%s%u", str, (unsigned) value_const_get(val));
    }
  }
}

static void cmd_asm_dump(cmd_t cmd, FILE *dst)
{
  pic_optype_t type;

  fprintf(dst, "%s", pic_opcode_str(cmd_asm_op_get(cmd)));
  type = pic_optype_get(cmd_asm_op_get(cmd));
  switch (type) {
    case pic_optype_none:
      break;
    case pic_optype_f:
    case pic_optype_f_d:
    case pic_optype_f_b:
      cmd_asm_dump_value(cmd_asm_val_get(cmd), " ", dst);
      if (pic_optype_f_d == type) {
        fprintf(dst, ", %s", pic_opdst_str(cmd_asm_opdst_get(cmd)));
      } else if (pic_optype_f_b == type) {
        cmd_asm_dump_value(cmd_asm_n_get(cmd), ", ", dst);
      }
      break;
    case pic_optype_n:
    case pic_optype_tris:
      cmd_asm_dump_value(cmd_asm_n_get(cmd), " ", dst);
      break;
    case pic_optype_k:
      if (cmd_asm_lbl_get(cmd)) {
        fprintf(dst, " %s", label_name_get(cmd_asm_lbl_get(cmd)));
      } else {
        cmd_asm_dump_value(cmd_asm_n_get(cmd), " ", dst);
      }
      break;
    case pic_optype_db:
      {
        uchar *data;
        size_t data_sz;
        size_t ii;

        data_sz = cmd_asm_data_sz_get(cmd);
        data    = cmd_asm_data_get(cmd);
        for (ii = 0; ii < data_sz; ii++) {
          fprintf(dst, "%u%s", data[ii], (ii + 1 < data_sz) ? "," : "");
        }
      }
      break;
  }
}

static void cmd_asm_free(cmd_t cmd)
{
  cmd_asm_val_set(cmd, VALUE_NONE);
  cmd_asm_n_set(cmd, VALUE_NONE);
  cmd_asm_lbl_set(cmd, LABEL_NONE);
  cmd_asm_data_set(cmd, 0, 0);
}

static cmd_t cmd_asm_dup(const cmd_t cmd)
{
  return cmd_asm_alloc(cmd_asm_op_get(cmd), cmd_asm_val_get(cmd),
    cmd_asm_valofs_get(cmd),
    cmd_asm_opdst_get(cmd), cmd_asm_n_get(cmd), cmd_asm_lbl_get(cmd),
    cmd_asm_flag_get_all(cmd),
    cmd_asm_data_sz_get(cmd),
    cmd_asm_data_get(cmd));
}

static void cmd_asm_label_remap(cmd_t cmd, const label_map_t *map)
{
  cmd_label_remap2(cmd, map, cmd_asm_lbl_get, cmd_asm_lbl_set);
}

static void cmd_asm_variable_remap(cmd_t cmd, const variable_map_t *map)
{
  cmd_variable_remap2(cmd, map, cmd_asm_val_get, cmd_asm_val_set);
  cmd_variable_remap2(cmd, map, cmd_asm_n_get, cmd_asm_n_set);
}

static void cmd_asm_value_remap(cmd_t cmd, const value_map_t *map)
{
  cmd_value_remap2(cmd, map, cmd_asm_val_get, cmd_asm_val_set);
  cmd_value_remap2(cmd, map, cmd_asm_n_get, cmd_asm_n_set);
}

static flag_t cmd_asm_variable_accessed(const cmd_t cmd, variable_t var)
{
  return ((value_variable_get(cmd_asm_val_get(cmd)) == var)
        || (value_variable_get(cmd_asm_n_get(cmd)) == var))
    ? CMD_VARIABLE_ACCESS_FLAG_READ | CMD_VARIABLE_ACCESS_FLAG_WRITTEN
    : CMD_VARIABLE_ACCESS_FLAG_NONE;
}

static flag_t cmd_asm_value_accessed(const cmd_t cmd, value_t val)
{
  return (cmd_asm_val_get(cmd) == val)
        || (cmd_asm_n_get(cmd) == val)
    ? CMD_VARIABLE_ACCESS_FLAG_READ | CMD_VARIABLE_ACCESS_FLAG_WRITTEN
    : CMD_VARIABLE_ACCESS_FLAG_NONE;
}

static const cmd_vtbl_t cmd_asm_vtbl = {
  cmd_asm_free,
  cmd_asm_dump,
  cmd_asm_dup,
  cmd_asm_label_remap,
  cmd_asm_variable_remap,
  cmd_asm_value_remap,
  cmd_asm_variable_accessed,
  cmd_asm_value_accessed
};

cmd_t cmd_asm_alloc(pic_opcode_t op, value_t val, size_t valofs,
    pic_opdst_t opdst, value_t n, label_t lbl, flag_t flags,
    size_t data_sz, uchar *data)
{
  cmd_t cmd;

  cmd = cmd_alloc(cmd_type_asm, &cmd_asm_vtbl);
  if (cmd) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      ptr->u.op_asm.op      = op;
      ptr->u.op_asm.val     = VALUE_NONE;
      ptr->u.op_asm.valofs  = valofs;
      ptr->u.op_asm.opdst   = opdst;
      ptr->u.op_asm.n       = VALUE_NONE;
      ptr->u.op_asm.lbl     = LABEL_NONE;
      ptr->u.op_asm.flags   = flags;
      ptr->u.op_asm.data_sz = 0;
      ptr->u.op_asm.data    = 0;
      
      cmd_asm_val_set(cmd, val);
      cmd_asm_n_set(cmd, n);
      cmd_asm_lbl_set(cmd, lbl);
      cmd_asm_data_set(cmd, data_sz, data);
    }
  }
  return cmd;
}

pic_opcode_t cmd_asm_op_get(const cmd_t cmd)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr) ? ptr->u.op_asm.op : pic_opcode_none;
}

void cmd_asm_op_set(cmd_t cmd, pic_opcode_t op)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr) {
    ptr->u.op_asm.op = op;
  }
}

value_t cmd_asm_val_get(const cmd_t cmd)
{
  const struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr) ? ptr->u.op_asm.val : VALUE_NONE;
}

void cmd_asm_val_set(cmd_t cmd, value_t val)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr) {
    /* if a varaible is used in an assembly statement, mark it both
     * assigned *and* used */
    value_lock(val);
    value_assign_ct_bump(ptr->u.op_asm.val, ctr_bump_decr);
    value_use_ct_bump(ptr->u.op_asm.val, ctr_bump_decr);
    value_release(ptr->u.op_asm.val);
    ptr->u.op_asm.val = val;
    value_assign_ct_bump(val, ctr_bump_incr);
    value_use_ct_bump(val, ctr_bump_incr);
  }
}

size_t cmd_asm_valofs_get(const cmd_t cmd)
{
  const struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr) ? ptr->u.op_asm.valofs : 0;
}

void cmd_asm_valofs_set(cmd_t cmd, size_t valofs)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr) {
    /* if a varaible is used in an assembly statement, mark it both
     * assigned *and* used */
    ptr->u.op_asm.valofs = valofs;
  }
}

pic_opdst_t  cmd_asm_opdst_get(const cmd_t cmd)
{
  const struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr) ? ptr->u.op_asm.opdst : pic_opdst_none;
}

void cmd_asm_opdst_set(cmd_t cmd, pic_opdst_t dst)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr) {
    ptr->u.op_asm.opdst = dst;
  }
}

value_t cmd_asm_n_get(const cmd_t cmd)
{
  const struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr) ? ptr->u.op_asm.n : VALUE_NONE;
}

void         cmd_asm_n_set(cmd_t cmd, value_t n)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr) {
    value_lock(n);
    value_use_ct_bump(ptr->u.op_asm.n, ctr_bump_decr);
    value_release(ptr->u.op_asm.n);
    ptr->u.op_asm.n = n;
    value_use_ct_bump(n, ctr_bump_incr);
  }
}

label_t cmd_asm_lbl_get(const cmd_t cmd)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr) ? ptr->u.op_asm.lbl : LABEL_NONE;
}

void cmd_asm_lbl_set(cmd_t cmd, label_t lbl)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr) {
    label_lock(lbl);
    label_usage_bump(ptr->u.op_asm.lbl, ctr_bump_decr);
    label_release(ptr->u.op_asm.lbl);
    ptr->u.op_asm.lbl = lbl;
    label_usage_bump(lbl, ctr_bump_incr);
  }
}

boolean_t cmd_asm_flag_test(const cmd_t cmd, flag_t flag)
{
  const struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr) ? (ptr->u.op_asm.flags & flag) == flag : boolean_false;
}

flag_t cmd_asm_flag_get_all(const cmd_t cmd)
{
  const struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr) ? ptr->u.op_asm.flags : 0;
}

size_t cmd_asm_data_sz_get(const cmd_t cmd)
{
  const struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr) ? ptr->u.op_asm.data_sz : 0;
}

uchar *cmd_asm_data_get(const cmd_t cmd)
{
  const struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr) ? ptr->u.op_asm.data : 0;
}

void   cmd_asm_data_set(cmd_t cmd, size_t sz, uchar *data)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr) {
    FREE(ptr->u.op_asm.data);
    ptr->u.op_asm.data_sz = sz;
    ptr->u.op_asm.data    = data;
  }
}



