/************************************************************
 **
 ** pic.c : pic code generation definitions
 **
 ** Copyright (c) 2004-2005, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/
#include <string.h>
#include <errno.h>
#include <assert.h>

#include "../libutils/mem.h"
#include "../libcore/cmd_asm.h"
#include "../libcore/cmd_brch.h"
#include "../libcore/cmd_op.h"
#include "../libcore/cmd_usec.h"
#include "../libcore/pf_proc.h"
#include "../libcore/pf_msg.h"
#include "piccolst.h"
#include "pic_brop.h"
#include "pic_daop.h"
#include "pic_inst.h"
#include "pic_msg.h"
#include "pic_op.h"
#include "pic_opfn.h"
#include "pic_stvar.h"
#include "pic_var.h"
#include "pic_stk.h"
#include "picdelay.h"
#include "pic_wopt.h"
#include "pic_emu.h"
#include "pic.h"

#define PIC_LABEL_COLUMN_SZ 32
#define PIC_OPCODE_COLUMN_SZ 8
#define STRINGIZE2(x) #x
#define STRINGIZE(x) STRINGIZE2(x)

static unsigned pic_stack_depth;

const char *pic_opcode_str(pic_opcode_t op)
{
  const char *ostr;

  ostr = "{unknown}";
  switch (op) {
    case pic_opcode_none:   ostr = "{none}"; break;
    case pic_opcode_addwf:  ostr = "addwf";  break;
    case pic_opcode_andwf:  ostr = "andwf";  break;
    case pic_opcode_xorwf:  ostr = "xorwf";  break;
    case pic_opcode_iorwf:  ostr = "iorwf";  break;
    case pic_opcode_subwf:  ostr = "subwf";  break;
    case pic_opcode_comf:   ostr = "comf";   break;
    case pic_opcode_decf:   ostr = "decf";   break;
    case pic_opcode_decfsz: ostr = "decfsz"; break;
    case pic_opcode_incf:   ostr = "incf";   break;
    case pic_opcode_incfsz: ostr = "incfsz"; break;
    case pic_opcode_rlf:    ostr = "rlf";    break;
    case pic_opcode_rrf:    ostr = "rrf";    break;
    case pic_opcode_movf:   ostr = "movf";   break;
    case pic_opcode_swapf:  ostr = "swapf";  break;
    case pic_opcode_clrf:   ostr = "clrf";   break;
    case pic_opcode_clrw:   ostr = "clrw";   break;
    case pic_opcode_movwf:  ostr = "movwf";  break;
    case pic_opcode_nop:    ostr = "nop";    break;
    case pic_opcode_retfie: ostr = "retfie"; break;
    case pic_opcode_ret:    ostr = "return"; break;
    case pic_opcode_sleep:  ostr = "sleep";  break;
    case pic_opcode_clrwdt: ostr = "clrwdt"; break;
    case pic_opcode_bcf:    ostr = "bcf";    break;
    case pic_opcode_bsf:    ostr = "bsf";    break;
    case pic_opcode_btfsc:  ostr = "btfsc";  break;
    case pic_opcode_btfss:  ostr = "btfss";  break;
    case pic_opcode_addlw:  ostr = "addlw";  break;
    case pic_opcode_andlw:  ostr = "andlw";  break;
    case pic_opcode_iorlw:  ostr = "iorlw";  break;
    case pic_opcode_movlw:  ostr = "movlw";  break;
    case pic_opcode_sublw:  ostr = "sublw";  break;
    case pic_opcode_xorlw:  ostr = "xorlw";  break;
    case pic_opcode_call:   ostr = "call";   break;
    case pic_opcode_goto:   ostr = "goto";   break;
    case pic_opcode_org:    ostr = "org";    break;
    case pic_opcode_end:    ostr = "end";    break;
    case pic_opcode_retlw:  ostr = "retlw";  break;
    case pic_opcode_datalo_set: ostr = "datalo_set"; break;
    case pic_opcode_datalo_clr: ostr = "datalo_clr"; break;
    case pic_opcode_datahi_set: ostr = "datahi_set"; break;
    case pic_opcode_datahi_clr: ostr = "datahi_clr"; break;
    case pic_opcode_irp_set:      ostr = "irp_set"; break;
    case pic_opcode_irp_clr:      ostr = "irp_clr"; break;
    case pic_opcode_branchlo_set: ostr = "branchlo_set"; break;
    case pic_opcode_branchlo_clr: ostr = "branchlo_clr"; break;
    case pic_opcode_branchlo_nop: ostr = "branchlo_nop"; break;
    case pic_opcode_branchhi_set: ostr = "branchhi_set"; break;
    case pic_opcode_branchhi_clr: ostr = "branchhi_clr"; break;
    case pic_opcode_branchhi_nop: ostr = "branchhi_nop"; break;
    case pic_opcode_option:       ostr = "option"; break;
    case pic_opcode_tris:         ostr = "tris"; break;
    case pic_opcode_db:           ostr = "db"; break;
  }
  return ostr;
}

pic_optype_t pic_optype_get(pic_opcode_t op)
{
  pic_optype_t type;

  type = pic_optype_none;
  switch (op) {
    case pic_opcode_none:
    case pic_opcode_org:
    case pic_opcode_end:
    case pic_opcode_clrw:
    case pic_opcode_nop:
    case pic_opcode_retfie:
    case pic_opcode_ret:
    case pic_opcode_sleep:
    case pic_opcode_clrwdt:
    case pic_opcode_option:
    case pic_opcode_db:
      type = pic_optype_none;
      break;
    case pic_opcode_clrf:
    case pic_opcode_movwf:
    case pic_opcode_datalo_set:
    case pic_opcode_datalo_clr:
    case pic_opcode_datahi_set:
    case pic_opcode_datahi_clr:
    case pic_opcode_irp_set:
    case pic_opcode_irp_clr:
      type = pic_optype_f;
      break;
    case pic_opcode_addwf:
    case pic_opcode_andwf:
    case pic_opcode_xorwf:
    case pic_opcode_iorwf:
    case pic_opcode_subwf:
    case pic_opcode_comf:
    case pic_opcode_decf:
    case pic_opcode_decfsz:
    case pic_opcode_incf:
    case pic_opcode_incfsz:
    case pic_opcode_rlf:
    case pic_opcode_rrf:
    case pic_opcode_movf:
    case pic_opcode_swapf:
      type = pic_optype_f_d;
      break;
    case pic_opcode_bcf:
    case pic_opcode_bsf:
    case pic_opcode_btfsc:
    case pic_opcode_btfss:
      type = pic_optype_f_b;
      break;
    case pic_opcode_addlw:
    case pic_opcode_andlw:
    case pic_opcode_iorlw:
    case pic_opcode_movlw:
    case pic_opcode_sublw:
    case pic_opcode_xorlw:
    case pic_opcode_retlw:
    case pic_opcode_tris:
      type = pic_optype_n;
      break;
    case pic_opcode_call:
    case pic_opcode_goto:
    case pic_opcode_branchlo_set:
    case pic_opcode_branchlo_clr:
    case pic_opcode_branchlo_nop:
    case pic_opcode_branchhi_set:
    case pic_opcode_branchhi_clr:
    case pic_opcode_branchhi_nop:
      type = pic_optype_k;
      break;
  }
  return type;
}

const char *pic_opdst_str(pic_opdst_t type)
{
  const char *str;

  str = "{none}";
  switch (type) {
    case pic_opdst_none: break;
    case pic_opdst_w:    str = "w"; break;
    case pic_opdst_f:    str = "f"; break;
  }
  return str;
}


/************************************************************
 **
 ** utility functions
 **
 ************************************************************/

/*
 * NAME
 *   pic_branch_proc
 *
 * DESCRIPTION
 *   execute a procedure call
 *
 * PARAMETERS
 *   pf          : pfile handle
 *   brcond      : condition
 *   brval       : value for condition
 *   proc        :
 *   proc_params :
 *   
 * RETURN
 *   none
 *
 * NOTES
 *   There are three types of procedures
 *     1. normal -- variable block is discretely allocated local variables
 *        1. copy the IN parameters
 *        2. call proc
 *        3. copy the OUT parameters
 *     2. recursive -- all local variables are allocated out of a single block
 *        1. call the pre-fn
 *           (push the local variables onto the stack)
 *        2. copy the IN parameters
 *        3. call proc
 *        4. copy the OUT parameter
 *        5. call the post-fn
 *           (pop the local variables off of the stack)
 *     3. indirect -- parameters are allocated in a single block,
 *                    remaining locals are allocated discretetly
 *        1. copy the IN parameters to the pic_temp block
 *        2. call the indirect fn
 *           1. copy the pic_temp block to the parameter block
 *           2. execute the function
 *              1. copy the function's address into _indirect_call_addr
 *              2. call indirect_call
 *                 1. load PCLATH
 *                 2. load PCL
 *           3. copy the parameter block to the pic_temp block
 *        3. copy the OUT parameters from the pic_temp block   
 *
 * the first 1-byte OUT parameter (including the return value)
 * will be returned in W; the first 1-byte IN parameter
 * will be passed in W *except* for indirect functions
 */
label_t pic_label_find(pfile_t *pf, const char *name, boolean_t alloc)
{
  label_t lbl;

  lbl = pfile_label_find(pf, pfile_log_none, name);
  if (!lbl && alloc) {
    lbl = pfile_label_alloc(pf, name);
  }
  return lbl;
}

value_t pic_value_find_or_alloc(pfile_t *pf, const char *name,
  variable_def_type_t type, variable_sz_t sz)
{
  value_t val;

  val = pfile_value_find(pf, pfile_log_none, name);
  if (!val) {
    variable_def_t def;

    def = variable_def_alloc(0, type, VARIABLE_DEF_FLAG_NONE, sz);
    pfile_value_alloc(pf, PFILE_VARIABLE_ALLOC_GLOBAL,
      name, def, &val);
  }
  return val;
}

/* return the parameter number that will be passed in W
 * flag = VARIABLE_FLAG_IN or VARIABLE_FLAG_OUT
 */
size_t pic_proc_w_param_get(const pfile_proc_t *proc,
    flag_t flag)
{
  size_t ii;

  if (((VARIABLE_DEF_FLAG_IN == flag)
    && (pfile_proc_flag_test(proc, PFILE_PROC_FLAG_INDIRECT)
      || pfile_proc_flag_test(proc, PFILE_PROC_FLAG_REENTRANT)))
      || ((VARIABLE_DEF_FLAG_OUT == flag)
        && pfile_proc_flag_test(proc, PFILE_PROC_FLAG_REENTRANT))
      || pfile_proc_flag_test(proc, PFILE_PROC_FLAG_TASK)) {
    /* a function that is called indirectly *never* gets a parameter
     * passed in W, likewise a recursive function cannot ever return
     * a value in W */
    ii = -1;
  } else {
    for (ii = 0; ii < pfile_proc_param_ct_get(proc); ii++) {
      value_t val;

      val = pfile_proc_param_get(proc, ii);
      if (value_dflag_test(val, flag) 
          && (1 == value_sz_get(val))
          && !value_dflag_test(val, VARIABLE_DEF_FLAG_BIT)
          && !variable_is_const(value_variable_get(val))
          /*&& !value_is_const(val)
          && value_variable_get(val)*/) {
        break;
      }
    }
  }
  if (ii == pfile_proc_param_ct_get(proc)) {
    ii = -1;
  }

  return ii;
}

/* move a one-byte value in val to W */
void pic_value_move_to_w(pfile_t *pf, value_t val)
{
  pic_op(pf, operator_assign, VALUE_NONE, val, VALUE_NONE);
}

/*
 * start a new task
 *    FOR task_ct USING task_idx loop
 *      IF !task_list[task_idx] THEN
 *        task_list[task_idx] = val
 *        RETURN
 *      END IF
 *    END LOOP
 */
static void pic_task_start(pfile_t *pf, value_t prval)
{
  label_t       fn;
  value_t       fnval;
  pic_code_t    code;
  pfile_proc_t *proc;
  label_t       lbl;

  proc = value_proc_get(prval);
  lbl  = pfile_proc_label_get(proc);

  /* this is probably a hack, but I don't know how else to do it!
     is might be best ot create a pic_instr_append_n_d() */
  fnval = pic_var_sign_get(pf);
  code = pic_instr_append_f_d(pf, pic_opcode_movlw, prval, 1, pic_opdst_w);
  pic_code_brdst_set(code, lbl);
  pic_code_value_set(code, VALUE_NONE);
  pic_instr_append_f(pf, pic_opcode_movwf, fnval, 0);
  code = pic_instr_append_f_d(pf, pic_opcode_movlw, prval, 0, pic_opdst_w);
  pic_code_brdst_set(code, lbl);
  pic_code_value_set(code, VALUE_NONE);
  fn = pic_label_find(pf, PIC_LABEL_TASK_START, boolean_true);
  pic_instr_append_n(pf, pic_opcode_call, fn);
  label_release(fn);
  pic_var_sign_release(pf, fnval);
}

static void pic_branch_proc(pfile_t *pf, cmd_branchcond_t brcond,
    value_t brval, value_t prval, value_t *proc_params)
{
  label_t brskip;

  /* I don't know that this will ever be possible, but it's probably
   * worth coding up just in case */
  assert(cmd_branchcond_none == brcond);
  if (cmd_branchcond_none == brcond) {
    brskip = LABEL_NONE;
  } else {
    pic_opcode_t op;

    brskip = pfile_label_alloc(pf, 0);
    if (value_is_single_bit(brval)) {
      op = pic_opcode_nop;
      switch (brcond) {
        case cmd_branchcond_true:  op = pic_opcode_btfsc; break;
        case cmd_branchcond_false: op = pic_opcode_btfss; break;
        case cmd_branchcond_none:  break;
      }
      pic_instr_append_f(pf, op, brval, 0);
    } else {
      op = pic_opcode_nop;
      switch (brcond) {
        case cmd_branchcond_true:  op = pic_opcode_btfss; break;
        case cmd_branchcond_false: op = pic_opcode_btfsc; break;
        case cmd_branchcond_none:  break;
      }
      pic_op(pf, operator_logical, VALUE_NONE, brval, VALUE_NONE);
      /* pic_instr_append_w_kn(pf, pic_opcode_andlw, 1); */
      pic_instr_append_reg_flag(pf, op, "_status", "_z");
    }
    pic_instr_append_n(pf, pic_opcode_goto, brskip);
  }
  if (variable_def_type_function == value_type_get(prval)) {
    pfile_proc_t *proc;

    proc = value_proc_get(prval);
    if (!proc
      || pfile_proc_flag_test(proc, PFILE_PROC_FLAG_REENTRANT)
      || pfile_proc_flag_test(proc, PFILE_PROC_FLAG_INDIRECT)) {
      /* an re-entrant, indirect, or call to a function that is called
       * indirectly elsewhere. all parameters go into the global parameter
       * block. */
      variable_def_member_t mbr_head;
      variable_def_member_t mbr_ptr;
      value_t              *tvals;
      size_t                ct;
      size_t                return_in_w;

      mbr_head = variable_def_member_get(value_def_get(prval));
      for (ct = 0, mbr_ptr = mbr_head; 
           mbr_ptr; 
           ct++, mbr_ptr = variable_def_member_link_get(mbr_ptr))
        ; /* null body */
      tvals = MALLOC(sizeof(*tvals) * ct);
      if (!tvals) {
        pfile_log_syserr(pf, ENOMEM);
      } else {
        size_t  ii;
        label_t lbl;
        value_t lsb;
        value_t val2;

        /* setup all of the parameters into the temporary area
           proc_params[0] = where to put the return value */
        for (ii = 0, mbr_ptr = mbr_head, 
              return_in_w = -1;
             ii < ct; 
             ii++, mbr_ptr = variable_def_member_link_get(mbr_ptr)) {
          variable_def_t def_ptr;

          def_ptr = variable_def_member_def_get(mbr_ptr);
          if ((-1 == return_in_w)
              && variable_def_flag_test(def_ptr, VARIABLE_DEF_FLAG_OUT)
              && (variable_def_sz_get(def_ptr) == 1)
              && !variable_def_flag_test(def_ptr, VARIABLE_DEF_FLAG_BIT)) {
            return_in_w = ii;
          }
          if (variable_def_flag_test(def_ptr, VARIABLE_DEF_FLAG_IN)) {
            /* if this is a re-entrant procedure and the parameter is not
             * AUTO, go ahead and make the assignment */
            value_t pval;

            pval = pfile_proc_param_get(proc, ii);
            if (!pval || value_is_auto(pval)) {
              tvals[ii] = pic_var_temp_get_def(pf, def_ptr);
              pic_op(pf, operator_assign, tvals[ii], proc_params[ii], 
                VALUE_NONE);
            } else if (!value_is_const(pval) && !value_is_lookup(pval)) {
              pic_op(pf, operator_assign, pval, proc_params[ii],
                  VALUE_NONE);
              tvals[ii] = VALUE_NONE;
            }
          } else {
            tvals[ii] = VALUE_NONE;
          }
        }
        for (ii = ct; ii > 1; ii--) {
          if (tvals[ii-1]) {
            pic_var_temp_release(pf, tvals[ii-1]);
          }
        }
        /* execute the function */
        if (!proc) {
          /* an indirect call */
          lsb = pic_var_sign_get(pf);
          val2 = value_clone(prval);
          value_indirect_clear(val2);
          pic_instr_append_f_d(pf, pic_opcode_movf, val2, 0, pic_opdst_w);
          pic_instr_append_f(pf, pic_opcode_movwf, lsb, 0);
          pic_instr_append_f_d(pf, pic_opcode_movf, val2, 1, pic_opdst_w);
          value_release(val2);
          lbl = pic_label_find(pf, PIC_LABEL_INDIRECT, boolean_true);
        } else {
          lbl = pfile_proc_label_get(proc);
          label_lock(lbl);
          lsb = VALUE_NONE;
        }
        pic_instr_append_n(pf, pic_opcode_call, lbl);
        label_release(lbl);
        if (VALUE_NONE != lsb) {
          pic_var_sign_release(pf, lsb);
        }
        /* pickup any return values */
        if ((-1 != return_in_w) && proc_params[return_in_w]) {
          if (value_is_indirect(proc_params[return_in_w])) {
            value_t rtmp;

            rtmp = pic_var_sign_get(pf);
            pic_op(pf, operator_assign, rtmp, VALUE_NONE, VALUE_NONE);
            pic_op(pf, operator_assign, proc_params[return_in_w], rtmp,
                VALUE_NONE);
            pic_var_sign_release(pf, rtmp);
          } else {
            pic_op(pf, operator_assign, proc_params[return_in_w], VALUE_NONE,
                VALUE_NONE);
          }
        }
        for (ii = 0, mbr_ptr = mbr_head;
             ii < ct;
             ii++, mbr_ptr = variable_def_member_link_get(mbr_ptr)) {
          variable_def_t def_ptr;

          def_ptr = variable_def_member_def_get(mbr_ptr);
          if ((return_in_w != ii)
              && variable_def_flag_test(def_ptr, VARIABLE_DEF_FLAG_OUT)) {
            tvals[ii] = pic_var_temp_get_def(pf, def_ptr);
            if (value_use_ct_get(proc_params[ii])) {
              pic_op(pf, operator_assign, proc_params[ii], tvals[ii], 
                VALUE_NONE);
            }
          } else {
            tvals[ii] = 0;
          }
        }
        for (ii = ct; ii; ii--) {
          if (tvals[ii-1]) {
            pic_var_temp_release(pf, tvals[ii-1]);
          }
        }
      }
    } else {
      /* a direct call */
      value_t val;
      size_t  ii;
      size_t  param_in_w;
      size_t  return_in_w;

      param_in_w = pic_proc_w_param_get(proc, VARIABLE_DEF_FLAG_IN);
      for (ii = 0; ii < pfile_proc_param_ct_get(proc); ii++) {
        /* if the proc parameter has gone const, it will be reflected in the
         * attached variable, *not* the value itself */
        val = pfile_proc_param_get(proc, ii);
        if (value_dflag_test(val, VARIABLE_DEF_FLAG_IN)) {
          if (!value_is_const(val) 
            && !variable_is_const(value_variable_get(val))) {
            if (ii != param_in_w) {
              pic_op(pf, operator_assign, val, proc_params[ii], VALUE_NONE);
            }
          }
        }
      }
      if (-1 != param_in_w) {
        /* nb: the assign/use ct bits are done in the loop above */
        pic_value_move_to_w(pf, proc_params[param_in_w]);
      }
      if (pfile_proc_flag_test(proc, PFILE_PROC_FLAG_TASK)) {
        pic_task_start(pf, prval);
      } else {
        pic_instr_append_n(pf, pic_opcode_call, pfile_proc_label_get(proc));
      }

      return_in_w = pic_proc_w_param_get(proc, VARIABLE_DEF_FLAG_OUT);
      if (-1 != return_in_w) {
        /* nb: the assign/use ct bits are done in the loop above */
        if (value_is_indirect(proc_params[return_in_w])) {
          value_t rtmp;

          if (proc_params[return_in_w]) {
            rtmp = pic_var_sign_get(pf);
            pic_op(pf, operator_assign, rtmp, VALUE_NONE, VALUE_NONE);
            pic_op(pf, operator_assign, proc_params[return_in_w], rtmp,
                VALUE_NONE);
            pic_var_sign_release(pf, rtmp);
          }
        } else if (value_is_function(proc_params[return_in_w])) {
          pfile_proc_t *rproc;

          rproc = value_proc_get(proc_params[return_in_w]);
          pic_instr_append_n(pf, pic_opcode_call, pfile_proc_label_get(rproc));
        } else {
          if (proc_params[return_in_w] 
              && !value_is_const(proc_params[return_in_w])) {
            pic_op(pf, operator_assign, proc_params[return_in_w], VALUE_NONE,
                VALUE_NONE);
          }
        }
      }
      for (ii = 0; ii < pfile_proc_param_ct_get(proc); ii++) {
        val = pfile_proc_param_get(proc, ii);

        if (value_dflag_test(val, VARIABLE_DEF_FLAG_OUT)) {
          if (proc_params[ii]
              && !value_is_const(proc_params[ii])
              && !variable_is_const(value_variable_get(proc_params[ii]))) {
            if (ii != return_in_w) {
              pic_op(pf, operator_assign, proc_params[ii], val, VALUE_NONE);
            }
          }
        }
      }
    }
  }
  if (brskip) {
    pic_instr_append_label(pf, brskip);
    label_release(brskip);
  }
}

static void pic_suspend(pfile_t *pf)
{
  label_t    ret;
  value_t    return_lsb;
  pic_code_t code;
  label_t    suspend;

  suspend = pic_label_find(pf, PIC_LABEL_TASK_SUSPEND, boolean_true);
  ret = pfile_label_alloc(pf, 0);
  return_lsb = pic_var_sign_get(pf);
  code = pic_instr_append(pf, pic_opcode_movlw);
  pic_code_brdst_set(code, ret);
  pic_instr_append_f(pf, pic_opcode_movwf, return_lsb, 0);
  pic_code_ofs_set(code, 1);
  code = pic_instr_append(pf, pic_opcode_movlw);
  pic_code_brdst_set(code, ret);
  pic_instr_append_n(pf, pic_opcode_goto, suspend);
  pic_instr_append_label(pf, ret);

  label_release(suspend);
  label_release(ret);
  pic_var_sign_release(pf, return_lsb);
}

/*
 * NAME
 *   pic_branch
 *
 * DESCRIPTION
 *   handle command type cmd_type_branch
 *
 * PARAMETERS
 *   pf  : pfile
 *   cmd : a branch command
 *
 * RETURN
 *
 * NOTES
 */
static void pic_branch(pfile_t *pf, cmd_branchtype_t brtype,
    cmd_branchcond_t brcond, label_t dst, value_t val,
    value_t proc, value_t *proc_params)
{
  if (proc) {
    pic_branch_proc(pf, brcond, val, proc, proc_params);
  } else {
    pic_opcode_t op;

    if (cmd_branchcond_none != brcond) {
      /* val might be NULL if Z is already set elsewhere */
      if (val) {
        /* pic_instr_append_w_kn(pf, pic_opcode_andlw, 1); */
        if (value_is_single_bit(val)) {
          op = pic_opcode_nop;
          switch (brcond) {
            case cmd_branchcond_true:  op = pic_opcode_btfsc; break;
            case cmd_branchcond_false: op = pic_opcode_btfss; break;
            case cmd_branchcond_none:  break;
          }
          pic_instr_append_f(pf, op, val, 0);
        } else {
          op = pic_opcode_nop;
          switch (brcond) {
            case cmd_branchcond_true:  op = pic_opcode_btfss; break;
            case cmd_branchcond_false: op = pic_opcode_btfsc; break;
            case cmd_branchcond_none:  break;
          }
          pic_op(pf, operator_logical, 0, val, 0);
          pic_instr_append_reg_flag(pf, op, "_status", "_z");
        }
      } else {
        op = pic_opcode_nop;
        switch (brcond) {
          case cmd_branchcond_true:  op = pic_opcode_btfss; break;
          case cmd_branchcond_false: op = pic_opcode_btfsc; break;
          case cmd_branchcond_none:  break;
        }
        pic_instr_append_reg_flag(pf, op, "_status", "_z");
      }
    }
    op = pic_opcode_none;
    switch (brtype) {
      case cmd_branchtype_task_start:
      case cmd_branchtype_task_end:
      case cmd_branchtype_none:         break;
      case cmd_branchtype_task_suspend: pic_suspend(pf); break;
      case cmd_branchtype_goto:         op = pic_opcode_goto; break;
      case cmd_branchtype_call:         op = pic_opcode_call; break;
      case cmd_branchtype_return:       op = pic_opcode_ret;  break;
    }
    if (pic_opcode_ret == op) {
      pic_instr_append(pf, op);
    } else if (pic_opcode_none != op) {
      pic_instr_append_n(pf, op, dst);
    }
  }
  if (cmd_branchtype_call == brtype) {
    /* we might have called someone so reset the known values */
    pic_last_values_reset();
  }
}

/* this is yet another special condition in the form:
   _temp = expression
   if (_temp) then goto dst; end if
   in this case, depending on the expression, things can be simplified.
   unfortunately this *could* result in more temporary variable space being
   allocated than necessary but I'm not quite sure how to solve that yet;
   
   the only operations that are allowed are:
     relationals
     equality
     logical
     notl
     cmpb 
   If this changed, pic_cmd_optimize() must also be changed!

   only one of tval or flag may exist!
*/
static void pic_branch_cond_append(pfile_t *pf, pic_opcode_t pop,
    label_t brdst, pic_opcode_t brop, value_t tval, const char *flag)
{
  if (tval) {
    pic_instr_append_f(pf, brop, tval, 0);
  } else {
    pic_instr_append_reg_flag(pf, brop, "_status", flag);
  }
  if (pic_opcode_ret == pop) {
    pic_instr_append(pf, pop);
  } else {
    pic_instr_append_n(pf, pop, brdst);
  }
}

static pic_opcode_t pic_branch_cond_brop_invert(pic_opcode_t op)
{
  return (pic_opcode_btfss == op) ? pic_opcode_btfsc : pic_opcode_btfss;
}

static void pic_branch_cond(pfile_t *pf, cmd_t cmd)
{
  operator_t   op;
  pic_opcode_t pop;
  label_t      lbl;
  value_t      dst;
  value_t      val1;
  value_t      val2;

  op   = cmd_optype_get(cmd);
  dst  = cmd_opdst_get(cmd);
  val1 = cmd_opval1_get(cmd);
  val2 = cmd_opval2_get(cmd);

  cmd = cmd_link_get(cmd);
  pop = pic_opcode_nop;
  switch (cmd_brtype_get(cmd)) {
    case cmd_branchtype_task_start:
    case cmd_branchtype_task_suspend:
    case cmd_branchtype_task_end:
    case cmd_branchtype_none:   assert(0); break;
    case cmd_branchtype_goto:   pop = pic_opcode_goto; break;
    case cmd_branchtype_call:   pop = pic_opcode_call; break;
    case cmd_branchtype_return: pop = pic_opcode_ret; break;
  }

  lbl = cmd_brdst_get(cmd);
  if (operator_is_relation(op)) {
    label_t skip;

    op = pic_relational(pf, op, dst, val1, val2);

    if (cmd_branchcond_false == cmd_brcond_get(cmd)) {
      /* invert the operator */
      switch (op) {
        case operator_lt: op = operator_ge; break;
        case operator_le: op = operator_gt; break;
        case operator_eq: op = operator_ne; break;
        case operator_ne: op = operator_eq; break;
        case operator_ge: op = operator_lt; break;
        case operator_gt: op = operator_le; break;
        default: assert(0); break;
      }
    }

    skip = ((operator_lt == op)
        || (operator_le == op)
        || (operator_ge == op)
        || (operator_gt == op))
      ? pfile_label_alloc(pf, 0)
      : LABEL_NONE;
    switch (op) {
      /*
       * nb : carry is *inverted* because subtract is done via
       *      2's complement add
       */
      case operator_lt: /* c and !z */
        pic_branch_cond_append(pf, pic_opcode_goto, skip, pic_opcode_btfsc, 
          VALUE_NONE, "_z");
        pic_branch_cond_append(pf, pop, lbl, pic_opcode_btfsc, VALUE_NONE, 
          "_c");
        break;
      case operator_le: /* c or z */
        pic_branch_cond_append(pf, pic_opcode_goto, lbl, pic_opcode_btfsc, 
          VALUE_NONE, "_z");
        pic_branch_cond_append(pf, pop, lbl, pic_opcode_btfsc, VALUE_NONE, 
          "_c");
        break;
      case operator_eq: /* z */
        pic_branch_cond_append(pf, pop, lbl, pic_opcode_btfsc, VALUE_NONE, 
          "_z");
        break;
      case operator_ne: /* !z */
        pic_branch_cond_append(pf, pop, lbl, pic_opcode_btfss, VALUE_NONE, 
          "_z");
        break;
      case operator_ge: /* !c or z */
        pic_branch_cond_append(pf, pic_opcode_goto, lbl, pic_opcode_btfsc, 
          VALUE_NONE, "_z");
        pic_branch_cond_append(pf, pop, lbl, pic_opcode_btfss, VALUE_NONE, 
          "_c");
        break;
      case operator_gt: /* !c and !z */
        /* the complex case :( */
        pic_branch_cond_append(pf, pic_opcode_goto, skip, pic_opcode_btfsc, 
          VALUE_NONE, "_z");
        pic_branch_cond_append(pf, pop, lbl, pic_opcode_btfss, VALUE_NONE, 
          "_c");
        break;
      default:
        assert(0);
    }
    if (skip) {
      pic_instr_append_label(pf, skip);
      label_release(skip);
    }
  } else {
    /* 
     * op is one of:
     *    operator_logical
     *    operator_notl
     *    operator_cmpb
     * these are all very much the same
     */
    value_t      tmp;
    const char  *flag;
    pic_opcode_t brop;

    flag = 0;
    if (dst) {
      tmp = dst;
      pic_op(pf, op, dst, val1, VALUE_NONE);
    } else if (value_is_single_bit(val1) 
        || (!value_is_lookup(val1) 
          && !value_is_indirect(val1)
          && (1 == value_sz_get(val1)))) {
      tmp = val1;
    } else {
      if (operator_cmpb == op) {
        tmp = pic_var_temp_get(pf, VARIABLE_FLAG_NONE, 
            value_byte_sz_get(val1));
      } else {
        tmp = VALUE_NONE;
      }
      pic_op(pf, (operator_notl == op) ? operator_logical : op, 
          tmp, val1, VALUE_NONE);
    }
    if (value_is_single_bit(tmp)) {
      /* notl is the same as cmpb; logical is the reverse */
      brop = (operator_logical == op)
        ? pic_opcode_btfsc : pic_opcode_btfss;
    } else if (1 == value_sz_get(tmp)) {
      flag = "_z";
      if ((operator_logical == op) || (operator_notl == op)) {
        /* z is set if tmp is FALSE, clear if TRUE */
        pic_instr_append_f_d(pf, pic_opcode_movf, tmp, 0, pic_opdst_w);
        brop = (operator_logical == op)
          ? pic_opcode_btfss : pic_opcode_btfsc;
      } else if (operator_cmpb == op) {
        pic_instr_append_f_d(pf, pic_opcode_comf, tmp, 0, pic_opdst_w);
        brop = pic_opcode_btfss;
      } else {
        assert(0);
      }
    } else {
      if (operator_cmpb == op) {
        pic_op(pf, operator_logical, VALUE_NONE, tmp, VALUE_NONE);
        op = operator_logical;
      }
      brop = (operator_logical == op)
        ? pic_opcode_btfss
        : pic_opcode_btfsc;
      flag = "_z";
    }
    if (cmd_branchcond_false == cmd_brcond_get(cmd)) {
      brop = pic_branch_cond_brop_invert(brop);
    }
    pic_branch_cond_append(pf, pop, lbl, brop, (flag) ? VALUE_NONE : tmp, 
        flag);
    if ((tmp != dst) && (tmp != val1)) {
      pic_var_temp_release(pf, tmp);
    }
  }
}


/*
 * NAME
 *   pic_isr_cleanup
 *
 * DESCRIPTION
 *   handle command type cmd_type_isr_cleanup
 *
 * PARAMETERS
 *   pf  : pfile
 *   cmd : command
 *
 * RETURN
 *   none
 *
 * NOTES
 */
void pic_isr_cleanup(pfile_t *pf)
{
  label_t lbl;

  lbl = pfile_isr_entry_get(pf);
  if (lbl) {
    pic_var_isr_t isr_vars;

    label_release(lbl);

    pic_var_isr_get(pf, boolean_true, &isr_vars);
    if (!pfile_flag_test(pf, PFILE_FLAG_MISC_INTERRUPT_FAST)
      && isr_vars.isr_state) {
      value_t tmp;

      tmp = value_clone(isr_vars.pic_state);
      value_def_set(tmp, value_def_get(isr_vars.isr_state));
      pic_op(pf, operator_assign, tmp, isr_vars.isr_state, VALUE_NONE);
      value_release(tmp);
    }
    pic_instr_default_flag_set(pf, PIC_CODE_FLAG_NO_OPTIMIZE);
    pic_instr_append_f_d(pf, pic_opcode_movf, isr_vars.pclath, 0, pic_opdst_w);
    pic_instr_append_reg(pf, pic_opcode_movwf, "_pclath");
    pic_instr_append_f_d(pf, pic_opcode_swapf, isr_vars.status, 0, pic_opdst_w);
    pic_instr_append_reg(pf, pic_opcode_movwf, "_status");
    pic_instr_append_f_d(pf, pic_opcode_swapf, isr_vars.w, 0, pic_opdst_f);
    pic_instr_append_f_d(pf, pic_opcode_swapf, isr_vars.w, 0, pic_opdst_w);
    pic_instr_append(pf, pic_opcode_retfie);
    pic_instr_default_flag_clr(pf, PIC_CODE_FLAG_NO_OPTIMIZE);
    pic_var_isr_release(pf, &isr_vars);
  }
}

/*
 * NAME
 *   pic_end
 *
 * DESCRIPTION
 *   handle commands of type cmd_type_end
 *
 * PARAMETERS
 *
 * RETURN
 *
 * NOTES
 *   this simply loops to a sleep instruction
 */
void pic_end(pfile_t *pf)
{
  if (!pfile_flag_test(pf, PFILE_FLAG_DEBUG_EMULATOR)) {
    label_t lbl;

    lbl = pfile_label_alloc(pf, 0);
    if (lbl) {
      pic_instr_append_label(pf, lbl);
      pic_instr_append(pf, pic_opcode_sleep);
      pic_instr_append_n(pf, pic_opcode_goto, lbl);
      label_release(lbl);
    }
  }
}


/*
 * NAME
 *   pic_clear
 *
 * DESCRIPTION
 *   set all data to 0
 *
 * PARAMETERS
 *   none
 *
 * RETURN
 *
 * NOTES
 */
void pic_clear(pfile_t *pf)
{
  pic_bank_info_t *bnk;
  size_t           bnk_no;
  value_t          val;
  label_t          lbl;

  val = VALUE_NONE;
  lbl = pic_label_find(pf, PIC_LABEL_MEMSET, boolean_true);
  for (bnk = pic_bank_info_get(pf), bnk_no = 0; bnk; bnk = bnk->link, 
      bnk_no++) {
    /* if there's nothing allocated in this bank, do nothing */
    size_t  sz;

    sz = bnk->hi - bnk->lo + 1;
    if (!lbl) {
      val = pic_var_sign_get(pf);
      pic_instr_append_f(pf, pic_opcode_clrf, val, 0);
    }
    pic_instr_append_w_kn(pf, pic_opcode_movlw, bnk->lo & 0xff);
    pic_instr_append_reg(pf, pic_opcode_movwf, "_fsr");
    pic_instr_append_reg_flag(pf, 
      (bnk->lo >= 256) ? pic_opcode_bsf : pic_opcode_bcf,
      "_status", "_irp");
    pic_instr_append_w_kn(pf, pic_opcode_movlw, sz);
    pic_instr_append_n(pf, pic_opcode_call, lbl);
  }
  pic_var_sign_release(pf, val);
  label_release(lbl);
}

/*
 * NAME
 *   pic_branch_bittest_out
 *
 * DESCRIPTION
 *   output a bittest for a branch
 *
 * PARAMETERS
 *   pf   :
 *   val1 : value to test
 *   val2 : bit #
 *
 * RETURN
 *   none
 *
 * NOTES
 *   this is similar to pic_bittest() but that cannot be used because
 *   it always assumes btfsc, whereas we need either btfsc or btfss
 *   here
 */
void pic_brtest_out(pfile_t *pf, value_t val1, value_t val2,
    cmd_branchcond_t cond)
{
  variable_const_t n;
  pic_opcode_t     pop;

  pop = (cmd_branchcond_true == cond) ? pic_opcode_btfsc : pic_opcode_btfss;

  n = value_const_get(val2);
  if (n >= 8U * value_sz_get(val1)) {
    if (pfile_flag_test(pf, PFILE_FLAG_WARN_RANGE)
        && pfile_flag_test(pf, PFILE_FLAG_WARN_BACKEND)) {
      pfile_log(pf, pfile_log_warn, PFILE_MSG_CONSTANT_RANGE);
    }
    pic_instr_append(pf, pic_opcode_clrw);
    pic_instr_append_reg_flag(pf, pop, "_status", "_z");
  } else {
    if ((n / 8) || !value_name_get(val1)) {
      pic_instr_append_f_bn(pf, pop, val1, (size_t) n / 8, n & 7);
    } else {
      pic_instr_append_f_b(pf, pop, val1, (size_t) n / 8, val2);
    }
  }
}

unsigned pic_proc_frame_sz_get(pfile_block_t *blk)
{
  unsigned   frame_sz;
  unsigned   subframe_sz_max;
  variable_t var;

  for (frame_sz = 0, var = pfile_block_variable_list_head(blk);
       var;
       var = variable_link_get(var)) {
    if (variable_is_auto(var)) {
      frame_sz += variable_sz_get(var);
    }
  }
  subframe_sz_max = 0;
  for (blk = pfile_block_child_get(blk); 
       blk; 
       blk = pfile_block_sibbling_get(blk)) {
    unsigned subframe_sz;

    subframe_sz = pic_proc_frame_sz_get(blk);
    if (subframe_sz > subframe_sz_max) {
      subframe_sz_max = subframe_sz;
    }
  }
  return frame_sz + subframe_sz_max;
}

static value_t pic_proc_rval_get(pfile_t *pf, pfile_proc_t *proc)
{
  value_t rval;

  rval = VALUE_NONE;
  if (pfile_proc_flag_test(proc, PFILE_PROC_FLAG_REENTRANT)) {
    const char *proc_name;
    size_t      sz;
    char       *rname;

    proc_name = label_name_get(pfile_proc_label_get(proc));
    sz        = 1 + strlen(proc_name);
    rname = MALLOC(2 + sz);
    if (!rname) {
      pfile_log_syserr(pf, ENOMEM);
    } else {
      rname[0] = '_';
      rname[1] = 'r';
      memcpy(rname + 2, proc_name, sz);
      rval = pic_value_find_or_alloc(pf, rname, variable_def_type_integer, 1);
      variable_flag_set(value_variable_get(rval), VARIABLE_FLAG_AUTO);
      FREE(rname);
    }
  }
  return rval;
}


/* pic_proc_enter processing: 
 * for re-entrant functions:
 *    increment _r{procname}
 *    if _r{procname} > 1
 *       push all local variables onto the stack
 * for re-entrant or indirect functions
 *    copy parameters from the system parameter block into
 *    the local paramter area
 */
void pic_proc_enter(pfile_t *pf, pfile_proc_t *proc)
{
  size_t  param_in_w;
  size_t  return_in_w;

  param_in_w = pic_proc_w_param_get(proc, VARIABLE_DEF_FLAG_IN);
  return_in_w = pic_proc_w_param_get(proc, VARIABLE_DEF_FLAG_OUT);

  if (0 != return_in_w) {
    /* create the temporary variable for the return value */
    value_t        tmp;
    variable_def_t def;
    value_t        val;

    val = pfile_proc_param_get(proc, 0);
    if (!value_variable_get(val)) {
      assert(!value_variable_get(val));

      def = value_def_get(val);
      tmp = pic_var_temp_get_def(pf, value_def_get(val));
      value_variable_set(val, value_variable_get(tmp));
      pic_var_temp_release(pf, tmp);
      value_def_set(val, def);
    }
  }

  pic_instr_append_label(pf, pfile_proc_label_get(proc));

  if (pfile_proc_flag_test(proc, PFILE_PROC_FLAG_REENTRANT)
    || pfile_proc_flag_test(proc, PFILE_PROC_FLAG_INDIRECT)) {
    value_t   *tvals;
    size_t     ct;

    /* re-entrant and indirect functions pass all parameters into
     * the global parameter area */
    if (pfile_proc_flag_test(proc, PFILE_PROC_FLAG_REENTRANT)
        && pfile_proc_frame_sz_get(proc)) {
      value_t         memcpy_params[2]; 
      value_t         rval;
      variable_t      var;
      label_t         lbl_skip;
      label_t         lbl_exec;
      label_t         lbl_stkpush;
      variable_base_t base;
      value_t         val;

      rval = pic_proc_rval_get(pf, proc);
      var = pfile_proc_variable_get_first(proc);
      while (variable_is_volatile(var)) {
        var = variable_link_get(var);
      }
      val = value_alloc(var);
      pic_memcpy_params_get(pf, memcpy_params);
      /* if no one else is in this proc, skip the push */
      pic_instr_append_f_d(pf, pic_opcode_incf, rval, 0, pic_opdst_f);
      pic_instr_append_f_d(pf, pic_opcode_decfsz, rval, 0, pic_opdst_w);
      lbl_exec = pfile_label_alloc(pf, 0);
      lbl_skip = pfile_label_alloc(pf, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_exec);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_skip);
      pic_instr_append_label(pf, lbl_exec);
      label_release(lbl_exec);
      /* push the local variables onto the stack */
      base = variable_base_get(var, 0);
      pic_instr_append_w_k(pf, pic_opcode_movlw, val);
      pic_instr_append_f(pf, pic_opcode_movwf, memcpy_params[0], 0);
      if (base >> 8) {
        pic_instr_append_w_kn(pf, pic_opcode_movlw, base >> 8);
        pic_instr_append_f(pf, pic_opcode_movwf, memcpy_params[0], 1);
      } else {
        pic_instr_append_f(pf, pic_opcode_clrf, memcpy_params[0], 1);
      }
      pic_instr_append_w_kn(pf, pic_opcode_movlw, 
          pfile_proc_frame_sz_get(proc));
      lbl_stkpush = pic_label_find(pf, PIC_LABEL_STKPUSH, boolean_true);
      pic_instr_append_n(pf, pic_opcode_call, lbl_stkpush);
      label_release(lbl_stkpush);
      pic_instr_append_label(pf, lbl_skip);
      label_release(lbl_skip);
      pic_memcpy_params_release(pf, memcpy_params);
      value_release(val);
      value_release(rval);
    }
    /* copy the values from temporary to local */
    ct = pfile_proc_param_ct_get(proc);
    tvals = MALLOC(sizeof(*tvals) * ct);
    if (!tvals) {
      pfile_log_syserr(pf, ENOMEM);
    } else {
      size_t ii;

      for (ii = 0; ii < ct; ii++) {
        value_t val;

        val = pfile_proc_param_get(proc, ii);
        if (!value_is_const(val)
          && value_dflag_test(val, VARIABLE_DEF_FLAG_IN)
          && value_is_auto(val)) {
          tvals[ii] = pic_var_temp_get_def(pf, value_def_get(val));
          pic_op(pf, operator_assign, val, tvals[ii], VALUE_NONE);
        } else {
          tvals[ii] = VALUE_NONE;
        }
      }
      if (-1 != param_in_w) {
        pic_value_move_to_w(pf, tvals[param_in_w]);
      }
      for (ii = ct; ii; ii--) {
        if (tvals[ii-1]) {
          pic_var_temp_release(pf, tvals[ii-1]);
        }
      }
      FREE(tvals);
    }
  }
  if (-1 != param_in_w) {
    value_t val;

    val = pfile_proc_param_get(proc, param_in_w);
    if (value_is_auto(val) || value_is_volatile(val)) {
      pic_op(pf, operator_assign, val, VALUE_NONE, VALUE_NONE);
    } else {
      /*
       * if the parameter shares space with an auto or volatile variable
       * go ahead & make the assignment
       */
      variable_t var;

      var = value_variable_get(val);
#if 0
      while (var && !variable_is_auto(var) && !variable_is_volatile(var)) {
        var = variable_master_get(var);
      }
#endif
      if (var) {
        /* do the assignment */
        value_t tmp;

        tmp = value_alloc(var);
        pic_op(pf, operator_assign, tmp, VALUE_NONE, VALUE_NONE);
        value_release(tmp);
      }
    }
  }
}

/* pic_proc_leave processing:
 * for re-entrant or indirect functions
 *    copy OUT parameters from the local parameter block into
 *    the system paramter block
 * for re-entrant functions:
 *    decrement _r{procname}
 *    if _r{procname} > 0
 *       pop all local variables from the stack
 */
void pic_proc_leave(pfile_t *pf, pfile_proc_t *proc)
{
  if (pfile_proc_flag_test(proc, PFILE_PROC_FLAG_TASK)) {
    label_t lbl;

    lbl = pic_label_find(pf, PIC_LABEL_TASK_SUICIDE, boolean_true);
    pic_instr_append_n(pf, pic_opcode_goto, lbl);
    label_release(lbl);
  } else {
    size_t return_in_w;

    return_in_w = pic_proc_w_param_get(proc, VARIABLE_DEF_FLAG_OUT);

    if (pfile_proc_flag_test(proc, PFILE_PROC_FLAG_INDIRECT)
      || pfile_proc_flag_test(proc, PFILE_PROC_FLAG_REENTRANT)) {
      /* copy OUT parameters to the system parameter block */
      value_t *tval;
      size_t   ct;

      ct = pfile_proc_param_ct_get(proc);
      tval = MALLOC(sizeof(*tval) * ct);
      if (!tval) {
        pfile_log_syserr(pf, ENOMEM);
      } else {
        size_t ii;

        for (ii = 0; ii < ct; ii++) {
          value_t val;

          val = pfile_proc_param_get(proc, ii);
          if (value_dflag_test(val, VARIABLE_DEF_FLAG_OUT) 
            && (ii != return_in_w)
            && (value_is_volatile(val)
              || value_is_auto(val))) {
            tval[ii] = pic_var_temp_get_def(pf, value_def_get(val));
            pic_op(pf, operator_assign, tval[ii], val, VALUE_NONE);
          } else {
            tval[ii] = VALUE_NONE;
          }
        }
        for (ii = ct; ii; ii--) {
          if (tval[ii-1]) {
            pic_var_temp_release(pf, tval[ii-1]);
          }
        }
        FREE(tval);
      }
    }
    if (pfile_proc_flag_test(proc, PFILE_PROC_FLAG_REENTRANT)
        && pfile_proc_frame_sz_get(proc)) {
      value_t         memcpy_params[2];
      value_t         rval;
      variable_t      var;
      value_t         val;
      label_t         lbl_exec;
      label_t         lbl_skip;
      label_t         lbl_stkpop;
      variable_base_t vbase_hi;

      rval = pic_proc_rval_get(pf, proc);
      lbl_skip = pfile_label_alloc(pf, 0);
      lbl_exec   = pfile_label_alloc(pf, 0);
      pic_instr_append_f_d(pf, pic_opcode_decfsz, rval, 0, pic_opdst_f);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_exec);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_skip);
      pic_instr_append_label(pf, lbl_exec);
      label_release(lbl_exec);
      /* do stuff */
      var = pfile_proc_variable_get_first(proc);
      while (variable_is_volatile(var)) {
        var = variable_link_get(var);
      }
      val = value_alloc(var);
      pic_memcpy_params_get(pf, memcpy_params);
      pic_instr_append_w_k(pf, pic_opcode_movlw, val);
      pic_instr_append_f(pf, pic_opcode_movwf, memcpy_params[1], 0);
      vbase_hi = variable_base_get(var, 0) >> 8;
      if (vbase_hi) {
        pic_instr_append_w_kn(pf, pic_opcode_movlw, vbase_hi);
        pic_instr_append_f(pf, pic_opcode_movwf, memcpy_params[1], 1);
      } else {
        pic_instr_append_f(pf, pic_opcode_clrf, memcpy_params[1], 1);
      }
      lbl_stkpop = pic_label_find(pf, PIC_LABEL_STKPOP, boolean_true);
      pic_instr_append_w_kn(pf, pic_opcode_movlw, 
          pfile_proc_frame_sz_get(proc));
      pic_instr_append_n(pf, pic_opcode_call, lbl_stkpop);
      label_release(lbl_stkpop);
      pic_instr_append_label(pf, lbl_skip);
      label_release(lbl_skip);
      value_release(val);
      pic_memcpy_params_release(pf, memcpy_params);
      value_release(rval);
    }
    if ((-1 != return_in_w) 
      && (0 != return_in_w)) {
      value_t retval;

      retval = pfile_proc_param_get(proc, return_in_w); 
      if (value_is_const(retval)) {
        pic_instr_append_w_k(pf, pic_opcode_movlw, 
            pfile_proc_param_get(proc, return_in_w));
      } else if (retval) {
        pic_instr_append_f_d(pf, pic_opcode_movf, 
            pfile_proc_param_get(proc, return_in_w), 0, pic_opdst_w);
      }
    }
    pic_instr_append(pf, pic_opcode_ret);
  }
}

static void pic_cmd_to_asm(pfile_t *pf, cmd_t cmd)
{
  pic_code_t   code;
  pic_opcode_t op;
  label_t      lbl;
  value_t      val;

  op = cmd_asm_op_get(cmd);
  lbl = cmd_asm_lbl_get(cmd);
  val = cmd_asm_val_get(cmd);

  switch (op) {
    case pic_opcode_datalo_clr:
    case pic_opcode_datalo_set:
      op = pic_value_datalo_get(val);
      break;
    case pic_opcode_datahi_clr:
    case pic_opcode_datahi_set:
      op = pic_value_datahi_get(val);
      break;
    case pic_opcode_branchlo_clr:
    case pic_opcode_branchlo_set:
      op = (label_pc_get(lbl) & 0x0800)
        ? pic_opcode_branchlo_set
        : pic_opcode_branchlo_clr;
      break;
    case pic_opcode_branchhi_clr:
    case pic_opcode_branchhi_set:
      op = (label_pc_get(lbl) & 0x1000)
        ? pic_opcode_branchhi_set
        : pic_opcode_branchhi_clr;
      break;
    default:
      break;
  }
  if (cmd_asm_flag_test(cmd, PIC_CODE_FLAG_NO_OPTIMIZE)) {
    pic_instr_default_flag_set(pf, PIC_CODE_FLAG_NO_OPTIMIZE);
  }
  code = pic_instr_alloc(pf, LABEL_NONE, op);
  pic_code_cmd_set(code, cmd);
  pic_code_flag_set_all(code, cmd_asm_flag_get_all(cmd));/*PIC_CODE_FLAG_NO_OPTIMIZE);*/
  pic_code_brdst_set(code, lbl);
  pic_code_dst_set(code, cmd_asm_opdst_get(cmd));
  pic_code_literal_set(code, cmd_asm_n_get(cmd));
  pic_code_value_set(code, val);
  pic_code_ofs_set(code, cmd_asm_valofs_get(cmd));
  if (cmd_asm_flag_test(cmd, PIC_CODE_FLAG_NO_OPTIMIZE)) {
    pic_instr_default_flag_clr(pf, PIC_CODE_FLAG_NO_OPTIMIZE);
  }
}


/*
 * NAME
 *   pic_cmd_out
 *
 * DESCRIPTION
 *   convert a cmd_t to one or more PIC instructions
 *
 * PARAMETERS
 *   pf  : pfile
 *   cmd : cmd to convert
 *
 * RETURN
 *   none
 *
 * NOTES
 */
void pic_cmd_out(pfile_t *pf, cmd_t *cmd_ptr)
{
  pic_code_t plst;
  cmd_t      cmd_nxt;
  cmd_t      cmd;
  value_t    val;
  pic_code_t code_first;

  code_first = pic_code_list_tail_get(pf);

  cmd = *cmd_ptr;
  plst = pic_code_list_head_get(pf);
  if ((cmd_type_proc_enter != cmd_type_get(cmd)) && cmd_label_get(cmd)) {
    pic_instr_append_label(pf, cmd_label_get(cmd));
  }
  cmd_nxt = cmd_link_get(cmd);
  val = cmd_opval2_get(cmd);
  if (!val) {
    val = cmd_opval1_get(cmd);
  }
  if (value_is_const(val)
      && (((operator_andb == cmd_optype_get(cmd))
          && !(value_const_get(val) & (value_const_get(val) - 1))))
      && (cmd_type_branch == cmd_type_get(cmd_nxt))
      && (0 == memcmp(value_name_get(cmd_opdst_get(cmd)), "_t", 2))
      && (value_is_same(cmd_opdst_get(cmd), cmd_brval_get(cmd_nxt)))) {
    /* this is a special case:
     *   bittst [tmp], [x], [constant bit]
     *   branch [true|false] [tmp] [dst]
     *   ..or..
     *   andb [tmp], [x], [constant bit]
     *   branch [true|false] [tmp] [dst]
     *
     * since andb x,0x80  --> bittst x,7
     */
    variable_const_t n;
    variable_const_t bit;

    n = value_const_get(val);
    for (bit = 0;
         0 != (n >>= 1);
         bit++)
      ; /* null body */
    val = pfile_constant_get(pf, bit, VARIABLE_DEF_NONE);
    if (val) {
      pic_brtest_out(pf, cmd_opval1_get(cmd), val,
          cmd_brcond_get(cmd_nxt));
      pic_branch(pf, cmd_brtype_get(cmd_nxt), cmd_branchcond_none,
          cmd_brdst_get(cmd_nxt), 0, 0, 0);
      value_release(val);
    }
    *cmd_ptr = cmd_nxt;
    cmd = cmd_nxt;
  } else if ((cmd_type_operator == cmd_type_get(cmd))
    && (operator_is_relation(cmd_optype_get(cmd))
      || (operator_logical == cmd_optype_get(cmd))
      || (operator_notl == cmd_optype_get(cmd))
      || (operator_cmpb == cmd_optype_get(cmd)))
    && (cmd_type_branch == cmd_type_get(cmd_link_get(cmd)))
    && value_is_same(cmd_opdst_get(cmd), cmd_brval_get(cmd_link_get(cmd)))) {
    pic_branch_cond(pf, cmd);
    *cmd_ptr = cmd_nxt;
    cmd = cmd_nxt;
  } else {
    switch (cmd_type_get(cmd)) {
      case cmd_type_asm:
        pic_cmd_to_asm(pf, cmd);
        break;
      case cmd_type_operator:      
        pic_op(pf, cmd_optype_get(cmd), cmd_opdst_get(cmd), 
            cmd_opval1_get(cmd), cmd_opval2_get(cmd)); 
        break;
      case cmd_type_branch:    
        pic_branch(pf, cmd_brtype_get(cmd), cmd_brcond_get(cmd),
            cmd_brdst_get(cmd), cmd_brval_get(cmd), cmd_brproc_get(cmd),
            cmd_brproc_params_get(cmd)); 
        break;
      case cmd_type_nop:       
        pic_instr_append(pf, pic_opcode_nop);
        break;
      case cmd_type_sleep:
        pic_instr_append(pf, pic_opcode_sleep);
        break;
      case cmd_type_end:
        pic_end(pf);
        break;
      case cmd_type_isr_cleanup:
        pic_isr_cleanup(pf);
        break;
      case cmd_type_proc_enter:
        pic_proc_enter(pf, cmd_proc_get(cmd));
        break;
      case cmd_type_proc_leave:
        pic_proc_leave(pf, cmd_proc_get(cmd));
        break;
      case cmd_type_block_start:
      case cmd_type_block_end:
      case cmd_type_statement_end:
      case cmd_type_label:
        break;
      case cmd_type_usec_delay:
        pic_delay_create(pf, cmd_usec_delay_get(cmd));
        break;
      case cmd_type_assert:
        /* put in a dummy opcode */
        pic_instr_append(pf, pic_opcode_nop);
        break;
    }
  }
  for (code_first = (code_first) 
         ? pic_code_next_get(code_first) : pic_code_list_head_get(pf);
      code_first;
      code_first = pic_code_next_get(code_first)) {
    pic_code_cmd_set(code_first, cmd);
  }
}

/*
 * NAME
 *   pic_bitstate_changed
 *
 * DESCRIPTION
 *   determine if a bitstate has changed
 *
 * PARAMETERS
 *   old : original state
 *   new : new state
 *
 * RETURN
 *   1 : changed
 *   0 : no change
 *
 * NOTES
 *   a bit has changed if it's not the same as new and it's not
 *   indeterminte
 */

boolean_t pic_bitstate_changed(pic_bitstate_t old, pic_bitstate_t new)
{
  return ((old != new) 
      && (pic_bitstate_indeterminate != old));
}


/*
 * NAME
 *   pic_code_label_find
 *
 * DESCRIPTION
 *   find a label in the code
 *
 * PARAMETERS
 *   pf   : 
 *   lbl  : label to find
 *   ppcl : [out] holds result on success
 *
 * RETURN
 *   0      : no error
 *   ENOENT : label not found
 *
 * NOTES
 */
pic_code_t pic_code_label_find(pfile_t *pf, const label_t lbl)
{
  return (pic_code_t) label_code_get(lbl);
}

/*
 * NAME
 *   pic_cmd_dump
 *
 * DESCRIPTION
 *
 * PARAMETERS
 *
 * RETURN
 *
 * NOTES
 */
static int pic_tag_dump(pfile_t *pf, pfile_write_t where,
  const char *pre, const char *name, unsigned tag_n)
{
  const char *fmt;

  fmt = tag_n ? "%s__%s%u" : "%s%s";
  return pfile_write(pf, where, fmt, pre, name, tag_n);
}

static int pic_variable_tag_dump(pfile_t *pf, pfile_write_t where,
  variable_t var)
{
  return pic_tag_dump(pf, where, "v_", variable_name_get(var),
    variable_tag_n_get(var));
}

static int pic_label_tag_dump(pfile_t *pf, pfile_write_t where,
  label_t lbl)
{
  return pic_tag_dump(pf, where, "l_", label_name_get(lbl),
    label_tag_n_get(lbl));
}

static void pic_bitstate_dump(pfile_t *pf, pfile_write_t where,
    pic_bitstate_t st, char set, char clr)
{
  char ch;

  ch = '!';
  switch (st) {
    case pic_bitstate_unknown:       ch = '-'; break;
    case pic_bitstate_set:           ch = set; break;
    case pic_bitstate_clr:           ch = clr; break;
    case pic_bitstate_indeterminate: ch = '?'; break;
  }
  pfile_write(pf, where, "%c", ch);
}

static void pic_variable_name_write(pfile_t *pf, pfile_write_t where,
    variable_t var)
{
  const char *fmt;

  fmt = (variable_tag_n_get(var)) ? "%s%u" : "%s";
  pfile_write(pf, where, fmt, variable_name_get(var), 
      variable_tag_n_get(var));
}


void pic_code_dump(pfile_t *pf, const pic_code_t code)
{
  const char *comma;
  int         sz;
  unsigned    pcode;
  label_t     lbl;
  boolean_t   need_cr;

  if (pic_opcode_db == pic_code_op_get(code)) {
    uchar   *data;
    size_t   data_sz;
    size_t   ii;
    cmd_t    cmd;
    pic_pc_t pc;

    cmd     = pic_code_cmd_get(code);
    data_sz = cmd_asm_data_sz_get(cmd);
    data    = cmd_asm_data_get(cmd);
    for (ii = 0, pc = 2 * pic_code_pc_get(code); 
         ii < data_sz; 
         ii += 2, pc += 2) {
      pfile_write_hex(pf, pc, data[ii+1]);
      pfile_write_hex(pf, pc + 1, data[ii]);
    }
    pcode = 0xffff;
  } else {
    pcode = pic_code_to_pcode(pf, code);
    if (0xffff != pcode) {
      pfile_write_hex(pf, 2 * pic_code_pc_get(code), (uchar) (pcode & 0xff));
      pfile_write_hex(pf, 2 * pic_code_pc_get(code) + 1, 
        (uchar) ((pcode >> 8) & 0xff));
    }
  }

#if 0
  {
    pfile_source_t *src;
    cmd_t           cmd;

    cmd = pic_code_cmd_get(code);
    src = cmd_source_get(cmd);

    pfile_write(pf, pfile_write_lst, "; code=0x%x at %s:%u (next=%x, (flags=%u)\n", code,
      pfile_source_name_get(src), pfile_source_line_get(src),
      (unsigned) pic_code_next_get(code), pic_code_flag_get_all(code));
  }
#endif

  lbl = pic_code_label_get(code);
  if (label_usage_get(lbl)) {
    sz = pic_label_tag_dump(pf, pfile_write_asm, lbl);
    pic_last_values_reset();
    need_cr = boolean_true;
  } else {
    sz = 0;
    need_cr = boolean_false;
  }
  if ((pic_opcode_none != pic_code_op_get(code))
    && (pfile_flag_test(pf, PFILE_FLAG_DEBUG_COMPILER)
      || ((pic_opcode_branchlo_nop != pic_code_op_get(code))
        && (pic_opcode_branchhi_nop != pic_code_op_get(code))))) {
    need_cr = boolean_true;
    sz += pfile_write(pf, pfile_write_asm, 
        "%*s%-" STRINGIZE(PIC_OPCODE_COLUMN_SZ) "s ", 
        (sz < (PIC_LABEL_COLUMN_SZ - 1)) 
        ? (PIC_LABEL_COLUMN_SZ - 1)- sz : 1, " ",
        pic_opcode_str(pic_code_op_get(code)));
    if (pic_opcode_db == pic_code_op_get(code)) {
      uchar *data;
      size_t data_sz;
      cmd_t  cmd;

      cmd     = pic_code_cmd_get(code);
      data_sz = cmd_asm_data_sz_get(cmd);
      data    = cmd_asm_data_get(cmd);

      while (data_sz--) {
        pfile_write(pf, pfile_write_asm, "0x%02x%s",
          *data, (data_sz) ? "," : "");
        data++;
      }
        

    } else {
      comma = "";
      if (pic_code_value_get(code)) {
        size_t     ofs;
        variable_t master;
        variable_t var;
        variable_t show;
        value_t    val;
        boolean_t  is_indirect;

        val = pic_code_value_get(code);
        if (value_is_indirect(val)) {
          /* this is an array, so use the _ind register */
          val = pfile_value_find(pf, pfile_log_err, "_ind");
          is_indirect = boolean_true;
          assert(val);
        } else {
          value_lock(val);
          is_indirect = boolean_false;
        }
        var    = value_variable_get(val);
        master = variable_master_get(var);

        ofs = value_const_get(value_baseofs_get(val))
             + pic_code_ofs_get(code);
#if 0
        if (master) {
          ofs += variable_base_get(var) - variable_base_get(master);
        }
#endif

        if (variable_dflag_test(var, VARIABLE_DEF_FLAG_BIT) && master) {
          show = master;
          /* ofs += variable_base_get(var, 0) - variable_base_get(master, 0); */
          ofs += variable_bit_offset_get(var) / 8;
        } else {
          show = var;
        }
        if (!value_is_const(val) && variable_name_get(show)) {
          sz += pic_variable_tag_dump(pf, pfile_write_asm, show);
        } else {
          sz += pfile_write(pf, pfile_write_asm, "%u", value_const_get(val));
        }
        if (ofs && !is_indirect) {
          sz += pfile_write(pf, pfile_write_asm, "+%u", ofs);
        }
        if (value_dflag_test(val, VARIABLE_DEF_FLAG_BIT)) {
          if ((pic_opcode_datalo_set != pic_code_op_get(code))
            && (pic_opcode_datalo_clr != pic_code_op_get(code))
            && (pic_opcode_datahi_set != pic_code_op_get(code))
            && (pic_opcode_datahi_clr != pic_code_op_get(code))) {
            sz += pfile_write(pf, pfile_write_asm, ", %u",
                value_bit_offset_get(val) & 0x07);
          }
          sz += pfile_write(pf, pfile_write_asm, " ; ");
          pic_variable_name_write(pf, pfile_write_asm, var);
        }
        if (pic_opdst_none != pic_code_dst_get(code)) {
          sz += pfile_write(pf, pfile_write_asm, ",%c", 
              (pic_opdst_w == pic_code_dst_get(code)) ? 'w' : 'f');
        }
        comma = ", ";
        value_release(val);
      }
      if (pic_code_literal_get(code)) {
        value_t lit;

        lit = pic_code_literal_get(code);
        if (value_name_get(lit) && !variable_is_array(value_variable_get(lit))) {
          sz += pfile_write(pf, pfile_write_asm, "%s", comma);
          sz += pic_variable_tag_dump(pf, pfile_write_asm,
            value_variable_get(lit));
        } else if (pic_opcode_org != pic_code_op_get(code)) {
          sz += pfile_write(pf, pfile_write_asm, "%s%lu", comma, 
              value_const_get(lit) & 0xff);
        } else {
          sz += pfile_write(pf, pfile_write_asm, "%s%lu", comma, 
              value_const_get(lit));
        }
      }
      if (pic_code_brdst_get(code)) {
        sz += pfile_write(pf, pfile_write_asm, "%s", 
            pic_code_ofs_get(code) ? "HIGH " : "");
        sz += pic_label_tag_dump(pf, pfile_write_asm,
          pic_code_brdst_get(code));
      }
    }
    if (pfile_flag_test(pf, PFILE_FLAG_DEBUG_COMPILER)) {
      pic_databits_state_t   dbits;
      pic_branchbits_state_t brbits;
      if (sz < 60) {
        pfile_write(pf, pfile_write_asm, "%*s", 60 - sz, "");
      }
      /*pfile_write(pf, pfile_write_asm, " %p", code);*/
      pfile_write(pf, pfile_write_asm, "; %2u %c%c ",
          pic_code_depth_get(code),
          pic_code_flag_test(code, PIC_CODE_FLAG_NO_OPTIMIZE)
            ? '-' : 'O',
          pic_code_flag_test(code, PIC_CODE_FLAG_VISITED)
            ? 'V' : '-');
      pic_code_databits_get(code, &dbits);
      pic_code_branchbits_get(code, &brbits);
      pic_bitstate_dump(pf, pfile_write_asm, dbits.before.rp1, 'R', 'r');
      pic_bitstate_dump(pf, pfile_write_asm, dbits.before.rp0, 'S', 's');
      pfile_write(pf, pfile_write_asm, " ");
      pic_bitstate_dump(pf, pfile_write_asm, dbits.action.rp1, 'R', 'r');
      pic_bitstate_dump(pf, pfile_write_asm, dbits.action.rp0, 'S', 's');
      pfile_write(pf, pfile_write_asm, " [");
      pic_bitstate_dump(pf, pfile_write_asm, brbits.before.pclath4, 'H', 'h');
      pic_bitstate_dump(pf, pfile_write_asm, brbits.before.pclath3, 'L', 'l');
      pfile_write(pf, pfile_write_asm, " ");
      pic_bitstate_dump(pf, pfile_write_asm, brbits.action.pclath4, 'H', 'h');
      pic_bitstate_dump(pf, pfile_write_asm, brbits.action.pclath3, 'L', 'l');
      pfile_write(pf, pfile_write_asm, "]");
      pfile_write(pf, pfile_write_asm, " %04x %02x%02x", 
          pic_code_pc_get(code), (unsigned char) (pcode >> 8),
          (unsigned char) (pcode & 0xff));
      if (pic_code_w_value_get(code)) {
        pfile_write(pf, pfile_write_asm, "\n%60s; W = ", "");
        pic_variable_tag_dump(pf, pfile_write_asm, 
          value_variable_get(pic_code_w_value_get(code)));
      }
    }
  }
  if (need_cr) {
    pfile_write(pf, pfile_write_asm, "\n");
  }
}

static void pic_eeprom_dump(pfile_t *pf, value_t eeprom)
{
  value_t sz;

  sz = pfile_value_find(pf, pfile_log_none, "_eeprom_used");
  if (sz && (value_const_get(sz))) {
    /* dump the eeprom bits */
    unsigned ii;
    value_t  v_base;
    variable_const_t base;

    v_base = pfile_value_find(pf, pfile_log_err, "_eeprom_base");
    base = value_const_get(v_base);
    value_release(v_base);

    pfile_write(pf, pfile_write_asm, "\n; EEPROM data\n\n");
    pfile_write(pf, pfile_write_asm, 
      "%*s%-" STRINGIZE(PIC_OPCODE_COLUMN_SZ) "s0x%04x\n", 
      PIC_LABEL_COLUMN_SZ - 1, "", "org", value_const_get(eeprom));
    pfile_write(pf, pfile_write_asm, 
      "%*s%-" STRINGIZE(PIC_OPCODE_COLUMN_SZ) "s", 
      PIC_LABEL_COLUMN_SZ - 1, "", "dw");
    value_release(eeprom);

    eeprom =pfile_value_find(pf, pfile_log_none, "_eeprom");
    value_dereference(eeprom);
    for (ii = 0; ii < value_const_get(sz); ii++) {
      value_t c;
      variable_const_t n;

      c = pfile_constant_get(pf, ii, VARIABLE_DEF_NONE);
      value_baseofs_set(eeprom, c);
      value_release(c);
      n = value_const_get(eeprom);
      pfile_write(pf, pfile_write_lst, "%s%u",
          (ii) ? "," : "", n);
      pfile_write_hex(pf, (base + ii) * 2, (uchar) (n & 0xff));
      pfile_write_hex(pf, (base + ii) * 2 + 1, (uchar) ((n >> 8) & 0xff));
    }
    pfile_write(pf, pfile_write_lst, "\n");
  }
  value_release(sz);
}

static void pic_config_dump(pfile_t *pf, value_t config)
{
  if (pfile_flag_test(pf, PFILE_FLAG_BOOT_FUSES)) {
    value_t          fuses;

    fuses = pfile_value_find(pf, pfile_log_err, "_fuses");
    if (value_is_array(fuses)) {
      variable_ct_t ii;

      for (ii = 0; ii < value_ct_get(fuses); ii++) {
        variable_const_t n;
        variable_const_t base;
        value_t          tfuses;
        value_t          tbase;

        tfuses = value_subscript_set(fuses, ii);
        tbase  = value_subscript_set(config, ii);

        n    = value_const_get(tfuses);
        base = value_const_get(tbase);

        pfile_write_hex(pf, 2 * base,     (uchar) (n & 0xff));
        pfile_write_hex(pf, 2 * base + 1, (uchar) ((n >> 8) & 0xff));
        value_release(tbase);
        value_release(tfuses);
      }
    } else {
      variable_const_t n;
      variable_const_t base;

      n    = value_const_get(fuses);
      base = value_const_get(config);

      pfile_write_hex(pf, 2 * base,     (uchar) (n & 0xff));
      pfile_write_hex(pf, 2 * base + 1, (uchar) ((n >> 8) & 0xff));
    }
    value_release(fuses);
  }
}

static pic_code_t pic_code_out;

void pic_cmd_dump(pfile_t *pf, const cmd_t cmd, boolean_t first)
{
  /* if first is TRUE, we need to dump the preamble */
  if (first) {
    pic_stack_depth = pic_stack_depth_get(pf);
    for (pic_code_out = pic_code_list_head_get(pf); 
         pic_code_out && (pic_code_cmd_get(pic_code_out) != cmd); 
         pic_code_out = pic_code_next_get(pic_code_out)) {
      pic_code_dump(pf, pic_code_out);
    }
  }
  while (pic_code_out && (pic_code_cmd_get(pic_code_out) == cmd)) {
    pic_code_dump(pf, pic_code_out);
    pic_code_out = pic_code_next_get(pic_code_out);
  }
  if (CMD_NONE == cmd) {
    value_t eeprom;
    value_t config;

    eeprom = pfile_value_find(pf, pfile_log_none, "_eeprom_base");
    config = pfile_value_find(pf, pfile_log_none, "_fuse_base");
    if (value_const_get(config) < value_const_get(eeprom)) {
      pic_config_dump(pf, config);
      pic_eeprom_dump(pf, eeprom);
    } else {
      pic_eeprom_dump(pf, eeprom);
      pic_config_dump(pf, config);
    }
    value_release(config);
    value_release(eeprom);
    pfile_write(pf, pfile_write_asm, "%*s\n", PIC_LABEL_COLUMN_SZ + 2, "end");
    if (!pfile_errct_get(pf)) {
      value_t sz;
      value_t stack_sz;

      sz = pfile_value_find(pf, pfile_log_none, "_code_size");
      if (!sz) {
        if (pfile_flag_test(pf, PFILE_FLAG_WARN_BACKEND)) {
          pfile_log(pf, pfile_log_warn, "Maximum code size is not set!");
        }
      } else {
        pic_code_t ptr;

        ptr = pic_code_list_tail_get(pf); 
        pic_blist_info_log(pf);
        if (ptr) {
          unsigned code_sz;

          code_sz = (ptr) ? pic_code_pc_get(ptr) + 1 : 0;
          pfile_log(pf, pfile_log_info, PIC_MSG_CODE_USED,
              code_sz, (unsigned) value_const_get(sz));
          if (code_sz >= value_const_get(sz)) {
            pfile_log(pf, pfile_log_err, PIC_MSG_CODE_TOO_BIG);
          }
        }
        value_release(sz);
      }
      pfile_log(pf, pfile_log_info, PIC_MSG_DATA_USED, pic_blist_max,
          pic_blist_total);
      pfile_log(pf, pfile_log_info, PIC_MSG_STACK_AVAIL, (ulong) pic_stk_sz);

      stack_sz = pfile_value_find(pf, pfile_log_none, "_stack_size");
      if (pic_stack_depth >= 15) {
        pfile_log(pf, pfile_log_info, "Hardware stack depth INFINITE");
      } else {
        pfile_log(pf, pfile_log_info, "Hardware stack depth %u", pic_stack_depth);
      }
      if (!stack_sz) {
        if (pfile_flag_test(pf, PFILE_FLAG_WARN_BACKEND)) {
          pfile_log(pf, pfile_log_warn, "stack size not set");
        }
      } else if (pic_stack_depth >= value_const_get(stack_sz)) {
        pfile_log(pf,
            pfile_flag_test(pf, PFILE_FLAG_WARN_STACK_OVERFLOW)
            ? pfile_log_warn
            : pfile_log_err,
            "Hardware stack overflow!");
      }
      value_release(stack_sz);
    }
  }
}

/* create a map of all used data areas! the map will show 'v'
 * for volatile, '*' for used and '-' for unused */
static void pic_data_area_map_create(pfile_t *pf)
{
  char *map;

  map     = calloc(1024, 1); /* nothing should be out of this range */
  if (map) {
    pfile_proc_t   *proc;
    variable_base_t highest;
    variable_base_t lowest;
    size_t          ii;

    lowest  = -1;
    highest = 0;
    for (proc = pfile_proc_root_get(pf);
         proc;
         proc = pfile_proc_next(proc)) {
      pfile_block_t *blk;

      for (blk = pfile_proc_block_root_get(proc);
           blk;
           blk = pfile_block_next(blk)) {
        variable_t var;

        for (var = pfile_block_variable_get_first(blk);
             var;
             var = variable_link_get(var)) {
          variable_base_t base;

          base = variable_base_get(var, 0);
          if (!variable_master_get(var) && (VARIABLE_BASE_UNKNOWN != base)) {
            if (base < 1024) {
              size_t sz;

              sz = variable_sz_get(var);
              if (0 == highest) {
                highest = base;
                lowest = base;
              } else if (base < lowest) {
                lowest = base;
              } else if (base > highest) {
                highest = base;
              }
              for (ii = 0; ii < sz; ii++, base++) {
                if (variable_is_volatile(var)) {
                  map[base] = 'v';
                } else if (!variable_is_auto(var)) {
                  map[base] = 'u';
                } else if (0 == map[base]) {
                  map[base] = '*';
                }
              }
            }
          }
        }
      }
    }
    lowest  = (lowest / 32) * 32;
    highest = ((highest + 31) / 32) * 32;
    if (pfile_flag_test(pf, PFILE_FLAG_DEBUG_COMPILER)) {
      for (ii = lowest; ii < highest; ii+= 32) {
        size_t jj;

        printf("%04lx: ", (ulong) ii);
        for (jj = 0; jj < 32; jj++) {
          if (!(jj & 0x03)) {
            fputc(' ', stdout);
          }
          fputc(map[ii+jj] ? map[ii+jj] : '-', stdout);
        }
        fputc('\n', stdout);
      }
    }
    if (pfile_flag_test(pf, PFILE_FLAG_MISC_CLEAR_BSS)) {
      boolean_t is_first;
      label_t   lbl_memset;
      value_t   pic_loop;
      size_t    base;

      lbl_memset = pic_label_find(pf, PIC_LABEL_MEMSET, boolean_true);
      pic_loop   = pic_var_loop_get(pf);

      for (base = lowest, is_first = boolean_true; base < highest; base++) {
        if ('*' == map[base]) {
          size_t len;

          for (len = 0; (base + len < highest) && ('*' == map[base + len]); len++)
            ;
          if (len <= 7) {
            value_t        val;
            variable_def_t def;
            char           bss_name[32];

            def = variable_def_alloc(0, variable_def_type_integer,
                VARIABLE_DEF_FLAG_VOLATILE, len);
            sprintf(bss_name, "_pic_bss_%lx", (ulong) base);
            pfile_value_alloc(pf, PFILE_VARIABLE_ALLOC_GLOBAL,
                bss_name, def, &val);
            variable_base_set(value_variable_get(val), base, 0);
            value_assign_ct_bump(val, ctr_bump_incr);
            for (ii = 0; ii < len; ii++) {
              pic_instr_append_f(pf, pic_opcode_clrf, val, ii);
            }
            value_release(val);
          } else {
            pic_instr_append_reg_flag(pf,
                (base & 0x0100) ? pic_opcode_bsf : pic_opcode_bcf,
                "_status", "_irp");
            pic_instr_append_w_kn(pf, pic_opcode_movlw, base & 0xff);
            pic_instr_append_reg(pf, pic_opcode_movwf, "_fsr");
            pic_instr_append_w_kn(pf, pic_opcode_movlw, len);
            pic_instr_append_f(pf, pic_opcode_movwf, pic_loop, 0);
            pic_instr_append(pf, pic_opcode_clrw);
            pic_instr_append_n(pf, pic_opcode_call, lbl_memset);
          }
          /* bad form, I know, but the clearest I can think to do! */
          base += len - 1;
        }
      }
      pic_var_loop_release(pf, pic_loop);
      label_release(lbl_memset);
    }
    free(map);
  }
}


/*
 * NAME
 *   pic_code_isr_preamble
 *
 * DESCRIPTION
 *   create the isr preamble code
 *
 * PARAMETERS
 *   pf : pfile
 *
 * RETURN
 *
 * NOTES
 *   from the PIC reference:
 *      movwf w_temp     ; w --> w_temp
 *      swapf status, w  ; status --> w (nybble swapped but not flags
 *                                      effected)
 *      clrf  status     ; 0 --> status (irp/rp1/rp0 = clr)
 *      movwf status_tmp ; w --> status_temp
 *      movf  pclath, w  ; pclath --> w
 *      movwf pclath_tmp ; w --> pclath_temp
 *      clrf  pclath     ; 0 --> pclath
 *      ...
 *      {isr}
 *      ...
 *      movf  pclath_tmp,w ; pclath_tmp --> w
 *      movwf pclath       ; w --> pclath
 *      swapf status_tmp,w ; status_tmp --> w
 *      movwf status       ; w --> status
 *      swapf w_temp,f     ; w_temp->h <--> w_temp->l --> w_temp
 *      swapf w_temp,w     ; w_temp->h <--> w_temp->l --> w
 *      retfie
 *
 *  also, the lookup tables may not cross 256 byte boundaries (giving a maximum of
 *    254 entries/table). Since I might need to futz with the ORGs here, I don't
 *    want to have to deal with moving them around if branchbits get inserted, so
 *    I'm going to waste a bit of space here & force the branchbits. it doesn't matter
 *    what I use as long as the order is:  HI, LO, goto. The analyzer will make
 *    any necessary corrections later.
 */
static void pic_preuser_init(pfile_t *pf, label_t lbl_user)
{
  pfile_proc_t *proc;
  value_t       stkptr;
  value_t       task_list;
  value_t       task_active;
  pic_code_t    code_start;
  label_t       lbl;

  code_start = pic_code_list_tail_get(pf);
  
  lbl = pic_label_find(pf, PIC_LABEL_PREUSER, boolean_false);
  if (lbl) {
    pic_instr_append_label(pf, lbl);
    label_release(lbl);
  }

  /* any recursive functions need their recursive count cleared */
  for (proc = pfile_proc_root_get(pf);
       proc;
       proc = pfile_proc_next(proc)) {
    value_t rval;

    rval = pic_proc_rval_get(pf, proc);
    if (pfile_proc_frame_sz_get(proc) && rval) {
      pic_instr_append_f(pf, pic_opcode_clrf, rval, 0);
    }
    value_release(rval);
  }
  /* if the software stack is used, the stack pointer must be set */
  stkptr = pfile_value_find(pf, pfile_log_none, "_pic_stkptr");
  if (stkptr) {
    pic_instr_append_w_kn(pf, pic_opcode_movlw, pic_stk_base & 0xff);
    pic_instr_append_f(pf, pic_opcode_movwf, stkptr, 0);
    pic_instr_append_w_kn(pf, pic_opcode_movlw, pic_stk_base >> 8);
    pic_instr_append_f(pf, pic_opcode_movwf, stkptr, 1);
    value_release(stkptr);
  }
  /* if the task list is used, the task array 
   * & active pointers must be cleared */
  task_list = pfile_value_find(pf, pfile_log_none, "_task_list");
  if (task_list) {
    unsigned task_ct;

    task_ct = 2 * pfile_task_ct_get(pf);
    while (task_ct--) {
      pic_instr_append_f(pf, pic_opcode_clrf, task_list, task_ct);
    }
    value_release(task_list);
  }

  task_active = pfile_value_find(pf, pfile_log_none, "_task_active");
  if (task_active) {
    pic_instr_append_f(pf, pic_opcode_clrf, task_active, 0);
    value_release(task_active);
  }
  pic_data_area_map_create(pf);
  /* if PFILE_FLAG_CLEAR_BSS is set we're going to assume it produced
   * some code, otherwise the program is too trivial */
  if ((pic_code_list_tail_get(pf) != code_start)
      || pfile_flag_test(pf, PFILE_FLAG_MISC_CLEAR_BSS)) {
    /* code was generated, so create the label */
    label_release(pic_label_find(pf, PIC_LABEL_PREUSER, boolean_true));
    pic_instr_append_n(pf, pic_opcode_goto, lbl_user);
  }
}

label_t pic_lookup_label_find(pfile_t *pf,
  variable_t var, unsigned ofs, boolean_t alloc)
{
  size_t  sz;
  label_t lbl;
  char   *name;

  sz  = 19; /* '_lookup_\0' */
  sz += strlen(variable_name_get(var));
  name = MALLOC(sz);
  lbl = LABEL_NONE;
  if (name) {
    sprintf(name, "_lookup_%s%u_%u",
      variable_name_get(var), variable_tag_n_get(var), ofs);
    lbl = pic_label_find(pf, name, alloc);
    FREE(name);
  }
  return lbl;
}

/* look to see if the byte at ofs in lookup variable var
 * is a constant, if so return it; otherwise return -1
 */ 
size_t pic_lookup_is_const(variable_t var, size_t ofs)
{
  variable_def_t        def;
  variable_def_member_t mbr;
  size_t                ct;
  size_t                sz;
  size_t                ii;
  size_t                key;

  def = variable_def_get(var);
  mbr = variable_def_member_get(def);
  ct  = variable_def_member_ct_get(mbr);
  sz  = variable_def_sz_get(variable_def_member_def_get(mbr));
  def = variable_def_alloc(0, variable_def_type_integer,
    VARIABLE_DEF_FLAG_NONE, 1);
  key = -1;
  for (ii = 0; ii < ct; ii++) {
    size_t ch;

    ch = (size_t) variable_const_get(var, def, ii * sz + sz - 1 - ofs);
    if (!ii) {
      key = ch;
    } else if (key != ch) {
      break;
    }
  }
  if (ii < ct) {
    key = -1;
  }
  variable_def_free(def);
  return key;
}

/*
 * create all of the lookup tables. this is done simply:
 * 1. create an array with all of the lookup tables & sizes
 * 2. sort by size
 * 3. create the tables, largest to smallest adjusting the PC
 *    as necessary when crossing a 256 byte boundary
 */

typedef struct pic_lookup_inf_ {
  variable_t var;
  size_t     ofs;
  size_t     ct;
} pic_lookup_inf_t;

static int lookup_inf_cmp(const void *A, const void *B)
{
  const pic_lookup_inf_t *a = A;
  const pic_lookup_inf_t *b = B;
  int   cmp;

  if (a->ct < b->ct) {
    cmp = 1;
  } else if (a->ct == b->ct) {
    cmp = 0;
  } else {
    cmp = -1;
  }
  return cmp;
}

#if 0
static boolean_t pic_lookup_needed(const pfile_t *pf)
{
  pfile_proc_t *proc;
  boolean_t     needed;

  needed = boolean_false;
  for (proc = pfile_proc_root_get(pf); 
       proc; 
       proc = pfile_proc_next(proc)) {
    pfile_block_t *blk;

    for (blk = pfile_proc_block_root_get(proc); 
         blk;
         blk = pfile_block_next(blk)) {
      variable_t var;

      for (var = pfile_block_variable_list_head(blk);
           var;
           var = variable_link_get(var)) {
        if (variable_is_lookup(var)) {
          /* create the lookup table */
          variable_def_t        def;
          variable_def_member_t mbr;
          size_t                ct;
          size_t                sz;
          size_t                ofs;

          def = variable_def_get(var);
          mbr = variable_def_member_get(def);
          ct  = variable_def_member_ct_get(mbr);
          sz  = variable_def_sz_get(variable_def_member_def_get(mbr));
          /* a seperate table is created for each byte */
          for (ofs = 0; ofs < sz; ofs++) {
            size_t  ii;
            char    key;

            /* if all of the values at this position are the same,
               don't bother with a lookup table! */
            for (ii = 0, key = 0; ii < ct; ii++) {
              char ch;

              ch = (char) variable_const_get(var, def, 
                  ii * sz + sz - 1 - ofs);
              if (!ii) {
                key = ch;
              } else if (key != ch) {
                break;
              }
            }
            needed = (ii < ct);
            if (needed) {
              goto fini;
            }
          }
        }
      }
    }
  }
fini:
  return needed;
}
#endif

static void pic_lookup_init(pfile_t *pf)
{
  pfile_proc_t *proc;
  struct {
    size_t            alloc;
    size_t            used;
    pic_lookup_inf_t *data;
  } lookup_inf = {0};

  for (proc = pfile_proc_root_get(pf); 
       proc; 
       proc = pfile_proc_next(proc)) {
    pfile_block_t *blk;

    for (blk = pfile_proc_block_root_get(proc); 
         blk;
         blk = pfile_block_next(blk)) {
      variable_t var;

      for (var = pfile_block_variable_list_head(blk);
           var;
           var = variable_link_get(var)) {
        if (variable_is_lookup(var)) {
          /* create the lookup table */
          variable_def_t        def;
          variable_def_member_t mbr;
          size_t                ct;
          size_t                sz;
          size_t                ofs;

          def = variable_def_get(var);
          mbr = variable_def_member_get(def);
          ct  = variable_def_member_ct_get(mbr);
          sz  = variable_def_sz_get(variable_def_member_def_get(mbr));
          /* a seperate table is created for each byte */
          def = variable_def_alloc(0, variable_def_type_integer,
            VARIABLE_DEF_FLAG_NONE, 1);
          for (ofs = 0; ofs < sz; ofs++) {
            size_t  ii;
            char    key;

            /* if all of the values at this position are the same,
               don't bother with a lookup table! */
            for (ii = 0, key = 0; ii < ct; ii++) {
              char ch;

              ch = (char) variable_const_get(var, def, 
                  ii * sz + sz - 1 - ofs);
              if (!ii) {
                key = ch;
              } else if (key != ch) {
                break;
              }
            }
            if (ii < ct) {
              if (lookup_inf.used == lookup_inf.alloc) {
                void  *tmp;
                size_t tmp_sz;
                size_t tmp_ct;

                tmp_ct = (lookup_inf.alloc) ? 2 * lookup_inf.alloc : 16;
                tmp_sz = tmp_ct * sizeof(*lookup_inf.data);
                tmp = REALLOC(lookup_inf.data, tmp_sz);
                if (!tmp) {
                  pfile_log_syserr(pf, ENOMEM);
                } else {
                  lookup_inf.alloc = tmp_ct;
                  lookup_inf.data  = tmp;
                }
              }
              if (lookup_inf.used < lookup_inf.alloc) {
                lookup_inf.data[lookup_inf.used].var = var;
                lookup_inf.data[lookup_inf.used].ofs = ofs;
                lookup_inf.data[lookup_inf.used].ct  = ct;
                lookup_inf.used++;
              }
            }
          }
          variable_def_free(def);
        }
      }
    }
  }
  if (lookup_inf.used) {
    size_t pc;

    label_release(pic_label_find(pf, PIC_LABEL_LOOKUP, boolean_true));

    pic_instr_default_flag_set(pf, PIC_CODE_FLAG_VISITED);
    qsort(lookup_inf.data, lookup_inf.used, sizeof(*lookup_inf.data),
      lookup_inf_cmp);

    pic_branchbits_pc_set(pf);
    pc = pic_code_list_pc_next_get(pf);

    while (lookup_inf.used) {
      size_t    ii;
      boolean_t used;

      for (ii = 0, used = boolean_false; ii < lookup_inf.used; ) {
        pic_code_t            code;
        label_t               lbl;
        size_t                sz;
        variable_def_t        def;
        variable_def_member_t mbr;
        boolean_t             writeit;

        def = variable_def_get(lookup_inf.data[ii].var);
        mbr = variable_def_member_get(def);
        sz  = variable_def_sz_get(variable_def_member_def_get(mbr));
        def = variable_def_alloc(0, variable_def_type_integer,
          VARIABLE_DEF_FLAG_NONE, 1);

        lbl = pic_lookup_label_find(pf, lookup_inf.data[ii].var, 
          lookup_inf.data[ii].ofs, boolean_true);
        writeit = boolean_false;
        if (lookup_inf.data[ii].ct > 255) {
          /* the offset is passed in W:tmp */
          value_t tmp;

          tmp = pic_var_loop_get(pf);
          pc += 6;
          pic_instr_append_label(pf, lbl);
          if (pc >> 8) {
            pc++;
            pic_instr_append_w_kn(pf, pic_opcode_addlw, pc >> 8);
          }
          pic_instr_append_reg(pf, pic_opcode_movwf, "_pclath");
          pic_instr_append_f_d(pf, pic_opcode_movf, tmp, 0, pic_opdst_w);
          pic_instr_append_w_kn(pf, pic_opcode_addlw, pc & 0xff);
          pic_instr_append_reg_flag(pf, pic_opcode_btfsc, "_status", "_c");
          pic_instr_append_reg_d(pf, pic_opcode_incf, "_pclath", pic_opdst_f);
          pic_instr_append_reg(pf, pic_opcode_movwf, "_pcl");
          writeit = boolean_true;
          pic_var_loop_release(pf, tmp);
        } else {
          size_t pc_hi;

          pc_hi = pc + lookup_inf.data[ii].ct;
          if (pc / 256 == pc_hi / 256) {
            /* put in the table */
            pic_instr_append_label(pf, lbl);
            code = pic_instr_append_reg_d(pf, pic_opcode_addwf, "_pcl",
              pic_opdst_f);
            writeit = boolean_true;
            pc++;
          }
        }
        if (writeit) {
          /* remove this from the list by bumping all other elements down */
          size_t jj;

          for (jj = 0; jj < lookup_inf.data[ii].ct; jj++) {
            code = pic_instr_append_w_kn(pf, pic_opcode_retlw,
              variable_const_get(lookup_inf.data[ii].var, def, 
                jj * sz + sz - lookup_inf.data[ii].ofs - 1));
          }
          pc += lookup_inf.data[ii].ct;
          for (jj = ii + 1; jj < lookup_inf.used; jj++) {
            lookup_inf.data[jj - 1] = lookup_inf.data[jj];
          }
          lookup_inf.used--;
          used = boolean_true;
        } else {
          ii++;
        }
        label_release(lbl);
      }
      if (!used) {
        /* we need to waste some space here. put in a new origin */
        pic_pc_t pc_new;

        pc_new = 256 * ((pc + 255) / 256);
        pic_instr_append_w_kn(pf, pic_opcode_org, pc_new);
        pfile_log(pf, pfile_log_debug, 
            "Wasting %lu bytes for lookup table alignment", 
            (ulong) (pc_new - pc));
        pc = pc_new;
      }
    }
    FREE(lookup_inf.data);
    pic_instr_default_flag_clr(pf, PIC_CODE_FLAG_VISITED);
  }
}

void pic_code_preamble(pfile_t *pf)
{
  label_t lbl_user;
  label_t lbl_pic_reset;   /* the reset vector */
  label_t lbl_pic_isr;     /* the ISR vector */
  label_t lbl_pic_preuser; /* the preuser vector */
  label_t lbl_pic_lookup;  /* dummy label present if there are any
                              lookup tables */
  label_t lbl_entry;       /* actual entry point, either lbl_user,
                              lbl_preuser, or none */

  lbl_pic_reset   = pic_label_find(pf, PIC_LABEL_RESET, boolean_true);
  lbl_pic_isr     = pic_label_find(pf, PIC_LABEL_ISR, boolean_true);
  lbl_pic_preuser = pic_label_find(pf, PIC_LABEL_PREUSER, 
    pfile_flag_test(pf, PFILE_FLAG_MISC_CLEAR_BSS));
  lbl_pic_lookup  = pic_label_find(pf, PIC_LABEL_LOOKUP, boolean_false);

  lbl_user = pfile_user_entry_get(pf);

  if (!lbl_user) {
    pfile_log(pf, pfile_log_crit, PIC_MSG_NO_USER_ENTRY);
  } else {
    label_t          lbl_loader; /* used by the boot loader */
    label_t          lbl_isr;
    unsigned         pc;
    value_t          code_sz;
    variable_const_t n_code_sz;

    code_sz    = pfile_value_find(pf, pfile_log_err, "_code_size");
    n_code_sz  = value_const_get(code_sz);
    lbl_isr    = pfile_isr_entry_get(pf);
    lbl_loader = LABEL_NONE;
    
    pic_instr_default_flag_set(pf, PIC_CODE_FLAG_NO_OPTIMIZE);
    lbl_entry = (lbl_pic_preuser)
      ? lbl_pic_preuser
      : lbl_user;
    if (pfile_flag_test(pf, PFILE_FLAG_BOOT_RICK)) {
      pc = 3;
      pic_instr_append_w_kn(pf, pic_opcode_org, pc);
      pic_instr_append_label(pf, lbl_pic_reset);
      if (lbl_pic_lookup) {
        /* we need to skip the lookup table, so jump to a dummy
         * location that will jump to the real address
         */
        lbl_loader = pfile_label_alloc(pf, 0);
        pic_instr_append_n(pf, pic_opcode_goto, lbl_loader);
      } else {
        pic_instr_append_n(pf, pic_opcode_goto, lbl_entry);
      }
    } else if (pfile_flag_test(pf, PFILE_FLAG_BOOT_BLOADER)) {
      pic_instr_append_w_kn(pf, pic_opcode_org, 0);
      lbl_loader = pfile_label_alloc(pf, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_loader);
      pc = 4;
      pic_instr_append_w_kn(pf, pic_opcode_org, pc);
    } else {
      pc = 0;
      pic_instr_append_w_kn(pf, pic_opcode_org, pc);
      pic_instr_append_label(pf, lbl_pic_reset);
      if (lbl_isr 
        || lbl_pic_lookup 
        || pfile_flag_test(pf, PFILE_FLAG_BOOT_LONG_START)
        || pic_intrinsics_exist(pf)) {
        if (pfile_flag_test(pf, PFILE_FLAG_BOOT_LONG_START)
          || (n_code_sz >= 4096)) {
          pic_instr_append_n(pf, pic_opcode_branchhi_clr, lbl_entry);
        }
        if (pfile_flag_test(pf, PFILE_FLAG_BOOT_LONG_START)
          || (n_code_sz >= 2048)) {
          pic_instr_append_n(pf, pic_opcode_branchlo_clr, lbl_entry);
        }
        pic_instr_append_n(pf, pic_opcode_goto, lbl_entry);
        if (pfile_flag_test(pf, PFILE_FLAG_BOOT_LONG_START)) {
          pic_instr_append(pf, pic_opcode_nop);
        }
      }
    }

    if (lbl_isr) {
      /* nb: for the RF PIC loader, the ISR preamble code cannot exceed 2K */
      pic_var_isr_t isr_vars;

      /* the interrupt vector is at 4, jump to the real ISR */
      pic_instr_append_w_kn(pf, pic_opcode_org, 4);
      pic_instr_append_label(pf, lbl_pic_isr);
      /* finally, the ISR preamble */
      pic_var_isr_get(pf, boolean_true, &isr_vars);
      pic_instr_append_f(pf, pic_opcode_movwf, isr_vars.w, 0);
      pic_instr_append_reg_d(pf, pic_opcode_swapf, "_status", pic_opdst_w);
      pic_instr_append_reg(pf, pic_opcode_clrf, "_status");
      pic_instr_append_f(pf, pic_opcode_movwf, isr_vars.status, 0);
      pic_instr_append_reg_d(pf, pic_opcode_movf, "_pclath", pic_opdst_w);
      pic_instr_append_f(pf, pic_opcode_movwf, isr_vars.pclath, 0);
      pic_instr_append_reg(pf, pic_opcode_clrf, "_pclath");
      if (!pfile_flag_test(pf, PFILE_FLAG_MISC_INTERRUPT_FAST)
        && isr_vars.isr_state) {
        pic_op(pf, operator_assign, isr_vars.isr_state, isr_vars.pic_state, 
            VALUE_NONE);
      }
      if (n_code_sz > 4096) {
        pic_instr_append_n(pf, pic_opcode_branchhi_set, lbl_isr);
      }
      if (n_code_sz > 2048) {
        pic_instr_append_n(pf, pic_opcode_branchlo_set, lbl_isr);
      }
      pic_instr_append_n(pf, pic_opcode_goto, lbl_isr);
      pic_var_isr_release(pf, &isr_vars);
      label_release(lbl_isr);
    }
    if (lbl_loader) {
      pic_instr_append_label(pf, lbl_loader);
      if (n_code_sz >= 4096) {
        pic_instr_append_n(pf, pic_opcode_branchhi_clr, lbl_entry);
      }
      if (n_code_sz >= 2048) {
        pic_instr_append_n(pf, pic_opcode_branchlo_clr, lbl_entry);
      }
      pic_instr_append_n(pf, pic_opcode_goto, lbl_entry);
      label_release(lbl_loader);
    }
    /* the lookup table *must* follow the ISR entry */
    pic_lookup_init(pf);
    /* from here on we can start optimizing again */
    pic_instr_default_flag_clr(pf, PIC_CODE_FLAG_NO_OPTIMIZE);
    if (pfile_flag_test(pf, PFILE_FLAG_MISC_CLEAR_BSS)) {
      /* assume memset is used */
      label_release(pic_label_find(pf, PIC_LABEL_MEMSET, boolean_true));
    }
    pic_intrinsics_create(pf);
    pic_preuser_init(pf, lbl_user);
    
    label_release(lbl_user);
    value_release(code_sz);
  }
  label_release(lbl_pic_lookup);
  label_release(lbl_pic_preuser);
  label_release(lbl_pic_reset);
  label_release(lbl_pic_isr);
}

void pic_code_cleanup(pfile_t *pf)
{
  pic_code_list_reset(pf);
  pic_blist_free(pf);
}

void pic_code_emu(pfile_t *pf)
{
  pic_emu_state_t state;

  state = pic_emu_state_alloc(pf);
  pic_emu(pf, state);
  pic_emu_state_free(state);
}

boolean_t pic_code_is_suspend(pfile_t *pf, pic_code_t code)
{
  label_t   lbl;
  boolean_t is_suspend;

  lbl = pfile_label_find(pf, pfile_log_none, PIC_LABEL_TASK_SUSPEND);
  is_suspend = lbl && (lbl == pic_code_brdst_get(code));
  label_release(lbl);
  return is_suspend;
}

static void pic_code_mark_used(pfile_t *pf, pic_code_t code)
{
  pic_code_t next;
  boolean_t  code_is_cond;

  code_is_cond = boolean_false;
  while (code && !pic_code_flag_test(code, PIC_CODE_FLAG_VISITED)) {
    boolean_t next_is_cond;

    next_is_cond = boolean_false;
    next = pic_code_next_get(code);
    pic_code_flag_set(code, PIC_CODE_FLAG_VISITED);
    switch (pic_code_op_get(code)) {
      case pic_opcode_btfsc:
      case pic_opcode_btfss:
      case pic_opcode_incfsz:
      case pic_opcode_decfsz:
        next_is_cond = boolean_true;
        break;
      case pic_opcode_retfie:
      case pic_opcode_retlw:
      case pic_opcode_ret:
        if (!code_is_cond) {
          next = 0;
        }
        break;
      case pic_opcode_goto:
        if (!code_is_cond && !pic_code_is_suspend(pf, code)) {
          next = pic_code_label_find(pf, pic_code_brdst_get(code));
          break;
        }
        /* a conditional goto will act like a call */
      case pic_opcode_call:
        pic_code_mark_used(pf, 
          pic_code_label_find(pf, pic_code_brdst_get(code)));
        break;
      default:
        break;
    }
    code_is_cond = next_is_cond;
    code = next;
  }
}

void pic_code_free_unused(pfile_t *pf)
{
  pic_code_t    code;
  unsigned      ct = 0;
  label_t       lbl;
  pfile_proc_t *proc;

  lbl = pfile_label_find(pf, pfile_log_none, PIC_LABEL_ISR);
  if (lbl) {
    pic_code_mark_used(pf, pic_code_label_find(pf, lbl));
    label_release(lbl);
  }
  lbl = pfile_label_find(pf, pfile_log_err, PIC_LABEL_RESET);
  if (lbl) {
    pic_code_mark_used(pf, pic_code_label_find(pf, lbl));
    label_release(lbl);
  }
  for (proc = pfile_proc_root_get(pf);
       proc;
       proc = pfile_proc_next(proc)) {
    pfile_block_t *blk;

    if (pfile_proc_flag_test(proc, PFILE_PROC_FLAG_INDIRECT)
        || pfile_proc_flag_test(proc, PFILE_PROC_FLAG_TASK))
        /*&& !pfile_proc_flag_test(proc, PFILE_PROC_FLAG_DIRECT)) */{
      code = pic_code_label_find(pf, pfile_proc_label_get(proc));
      pic_code_mark_used(pf, code);
    }
    for (blk = pfile_proc_block_root_get(proc);
         blk;
         blk = pfile_block_next(blk)) {
      for (lbl = pfile_block_label_list_head(blk);
           lbl;
           lbl = label_link_get(lbl)) {
        if (label_usage_get(lbl)) {
          code = pic_code_label_find(pf, lbl);
          pic_code_mark_used(pf, code);
        }
      }
    }
  }

  code = pic_code_list_head_get(pf);
  while (code) {
    pic_code_t next;

    next = pic_code_next_get(code);
    if (!pic_code_flag_test(code, PIC_CODE_FLAG_VISITED)
      && !pic_code_flag_test(code, PIC_CODE_FLAG_NO_OPTIMIZE)
      && (pic_opcode_org != pic_code_op_get(code))
      && (pic_opcode_end != pic_code_op_get(code))
      && !pic_code_label_get(code)) {
      pic_code_list_remove(pf, code);
      pic_code_free(code);
      ct++;
    }
    code = next;
  }
  pfile_log(pf, pfile_log_debug, "%u unused codes removed", ct);
}


/*
 * NAME
 *   pic_variable_counters_clear
 *
 * DESCRIPTION
 *   set the assign & use counters on all variables to 0
 *
 * PARAMETERS
 *   pf : pfile handle
 *
 * RETURN
 *   none
 *
 * NOTES
 *   I only want to keep the *real* counters -- those used
 *   by the assembly instructions -- so clear all counters
 *   here, produce the code, then set the counter based
 *   on the generated code
 */
static void pic_variable_counters_clear(pfile_t *pf)
{
  pfile_proc_t *proc;

  for (proc = pfile_proc_root_get(pf);
       proc;
       proc = pfile_proc_next(proc)) {
    pfile_block_t *blk;

    for (blk = pfile_proc_block_root_get(proc);
         blk;
         blk = pfile_block_next(blk)) {
      variable_t var;

      for (var = pfile_block_variable_list_head(blk);
           var;
           var = variable_link_get(var)) {
        if (!variable_is_const(var)) {
          variable_assign_ct_set(var, 0);
          variable_use_ct_set(var, 0);
        }
      }
    }
  }
}

/*
 * NAME
 *   pic_variable_counters_set
 *
 * DESCRIPTION
 *   set the assign & use counters based on the generated code
 *
 * PARAMETERS
 *   pf : pfile handle
 *
 * RETURN
 *   none
 *
 * NOTES
 */
void pic_variable_counters_set(pfile_t *pf)
{
  pic_code_t code;

  pic_variable_counters_clear(pf);
  for (code = pic_code_list_head_get(pf);
       code;
       code = pic_code_next_get(code)) {
    value_t      val;
    value_t      literal;
    pic_opdst_t  dst;
    pic_opcode_t op;

    val = pic_code_value_get(code);
    dst = pic_code_dst_get(code);
    op  = pic_code_op_get(code);
    literal = pic_code_literal_get(code);

    switch (pic_code_op_get(code)) {
      case pic_opcode_end:
      case pic_opcode_none: 
      case pic_opcode_nop:
      case pic_opcode_clrw:
      case pic_opcode_retfie:
      case pic_opcode_ret:
      case pic_opcode_sleep:
      case pic_opcode_clrwdt:
      case pic_opcode_option:
      case pic_opcode_call:
      case pic_opcode_goto:
      case pic_opcode_datalo_set:
      case pic_opcode_datalo_clr:
      case pic_opcode_datahi_set:
      case pic_opcode_datahi_clr:
      case pic_opcode_irp_set:
      case pic_opcode_irp_clr:
      case pic_opcode_branchlo_set:
      case pic_opcode_branchlo_clr:
      case pic_opcode_branchlo_nop:
      case pic_opcode_branchhi_set:
      case pic_opcode_branchhi_clr:
      case pic_opcode_branchhi_nop:
      case pic_opcode_db:
        break; /* no action */
      case pic_opcode_addwf:
      case pic_opcode_andwf:
      case pic_opcode_xorwf:
      case pic_opcode_iorwf:
      case pic_opcode_subwf:
      case pic_opcode_comf:
      case pic_opcode_decf:
      case pic_opcode_decfsz:
      case pic_opcode_incf:
      case pic_opcode_incfsz:
      case pic_opcode_rlf:
      case pic_opcode_rrf:
      case pic_opcode_movf:
      case pic_opcode_swapf:
        if (pic_opdst_f == op) {
          value_assign_ct_bump(val, ctr_bump_incr);
        } else {
          value_use_ct_bump(val, ctr_bump_incr);
        }
        break;
      case pic_opcode_movwf:
      case pic_opcode_clrf:
      case pic_opcode_bcf:
      case pic_opcode_bsf:
        value_assign_ct_bump(val, ctr_bump_incr);
        break;
      case pic_opcode_btfsc:
      case pic_opcode_btfss:
        value_use_ct_bump(val, ctr_bump_incr);
        break;
      case pic_opcode_addlw:
      case pic_opcode_andlw:
      case pic_opcode_iorlw:
      case pic_opcode_movlw:
      case pic_opcode_sublw:
      case pic_opcode_xorlw:
      case pic_opcode_retlw:
      case pic_opcode_tris:
      case pic_opcode_org:
        if (!value_is_const(val)) {
          value_assign_ct_bump(val, ctr_bump_incr);
        }
        value_use_ct_bump(val, ctr_bump_incr);
        value_use_ct_bump(literal, ctr_bump_incr);
        break;
    }
  }
}

char *pic_chip_name_get(pfile_t *pf)
{
  variable_t chip_var;
  char      *chip_name;

  chip_name = 0;
  chip_var = pfile_variable_find(pf, pfile_log_err, "target_chip_name", 0);
  if (!variable_is_array(chip_var)) {
    pfile_log(pf, pfile_log_err, "\"target_chip_name\" is invalid");
  } else {
    char          *tmp;
    variable_ct_t  ct;
    variable_def_t def;

    def = variable_def_get(chip_var);

    ct = variable_def_member_ct_get(
           variable_def_member_get(def));

    def = variable_def_member_def_get(variable_def_member_get(def));

    tmp = MALLOC(1 + ct);
    if (!tmp) {
      pfile_log_syserr(pf, ENOMEM);
    } else {
      size_t ii;

      for (ii = 0; ii < ct; ii++) {
        tmp[ii] = (char) variable_const_get(chip_var, def, ii);
      }
      tmp[ii] = 0;
      chip_name = tmp;
    }
  }
  variable_release(chip_var);
  return chip_name;
}

void pic_asm_header_write(pfile_t *pf, const char *chip_name)
{
  value_t          code_sz;
  variable_const_t n_code_sz;

  pfile_write(pf, pfile_write_asm, 
      "%32slist p=%s, r=dec\n"
      "%32serrorlevel -306 ; no page boundary warnings\n"
      "%32serrorlevel -302 ; no bank 0 warnings\n"
      "%32serrorlevel -202 ; no 'argument out of range' warnings\n"
      "\n", "", chip_name ? chip_name : "", "", "", "");

  code_sz   = pfile_value_find(pf, pfile_log_err, "_code_size");
  n_code_sz = value_const_get(code_sz);
  value_release(code_sz);

  if (pfile_flag_test(pf, PFILE_FLAG_BOOT_FUSES)) {
    value_t fuses;

    fuses = pfile_value_find(pf, pfile_log_err, "_fuses");
    if (value_is_array(fuses)) {
      value_t       base;
      variable_ct_t ii;

      base = pfile_value_find(pf, pfile_log_err, "_fuse_base");
      if (value_ct_get(base) != value_ct_get(fuses)) {
        pfile_log(pf, pfile_log_err, 
            "_fuses and _fuses_base must contain the same number of entries");
      }
      for (ii = 0; ii < value_ct_get(fuses); ii++) {
        value_t tfuses;
        value_t tbase;

        tfuses = value_subscript_set(fuses, ii);
        tbase  = value_subscript_set(base, ii);
        pfile_write(pf, pfile_write_asm, "%32s__config 0x%04x, 0x%04x\n",
            "", value_const_get(tbase), value_const_get(tfuses));
        value_release(tbase);
        value_release(tfuses);
      }
      value_release(base);
    } else {
      pfile_write(pf, pfile_write_asm, "%32s__config 0x%04x\n", "",
        value_const_get(fuses));
    }
    value_release(fuses);
  }
  /* set some useful macros; these are done as absolute address because
   * it's easier than trying to determine what the true variable names
   * will be */
  pfile_write(pf, pfile_write_asm, 
      "datahi_set macro val\n"
      "  bsf 3, 6 ; STATUS<rp1>\n"
      "  endm\n");
  pfile_write(pf, pfile_write_asm,
      "datahi_clr macro val\n"
      "  bcf 3, 6 ; STATUS<rp1>\n"
      "  endm\n");
  pfile_write(pf, pfile_write_asm,
      "datalo_set macro val\n"
      "  bsf 3, 5 ; STATUS<rp0>\n"
      "  endm\n");
  pfile_write(pf, pfile_write_asm,
      "datalo_clr macro val\n"
      "  bcf 3, 5 ; STATUS<rp0>\n"
      "  endm\n");
  pfile_write(pf, pfile_write_asm,
      "irp_clr macro\n"
      "  bcf 3, 7 ; STATUS<irp>\n"
      "  endm\n");
  pfile_write(pf, pfile_write_asm,
      "irp_set macro\n"
      "  bsf 3, 7 ; STATUS<irp>\n"
      "  endm\n");
  pfile_write(pf, pfile_write_asm,
      "branchhi_set macro lbl\n"
      "  %s\n"
      "  endm\n", 
      (n_code_sz < 4096) ? "nop" : "  bsf 10, 4 ; PCLATH<4>");
  pfile_write(pf, pfile_write_asm,
      "branchhi_clr macro lbl\n"
      "  %s\n"
      "  endm\n",
      (n_code_sz < 4096) ? "nop" : "  bcf 10, 4 ; PCLATH<4>");
  if (pfile_flag_test(pf, PFILE_FLAG_DEBUG_COMPILER)) {
    pfile_write(pf, pfile_write_asm,
        "branchhi_nop macro lbl\n"
        "  endm\n");
  }
  pfile_write(pf, pfile_write_asm,
      "branchlo_set macro lbl\n"
      "  %s\n"
      "  endm\n",
      (n_code_sz  < 2048) ? "nop" : "  bsf 10, 3 ; PCLATH<3>");
  pfile_write(pf, pfile_write_asm,
      "branchlo_clr macro lbl\n"
      "  %s\n"
      "  endm\n",
      (n_code_sz  < 2048) ? "nop" : "  bcf 10, 3 ; PCLATH<3>");
  if (pfile_flag_test(pf, PFILE_FLAG_DEBUG_COMPILER)) {
    pfile_write(pf, pfile_write_asm,
        "branchlo_nop macro lbl\n"
        "  endm\n");
  }
  {
    label_t       lptr;
    pfile_proc_t *proc;

    for (proc = pfile_proc_root_get(pf);
         proc;
         proc = pfile_proc_next(proc)) {
      pfile_block_t *blk;

      for (blk = pfile_proc_block_root_get(proc);
           blk;
           blk = pfile_block_next(blk)) {
        variable_t vptr;

        for (vptr = pfile_block_variable_list_head(blk);
             vptr;
             vptr = variable_link_get(vptr)) {
          variable_const_t vdef;

          if (variable_assign_ct_get(vptr) || variable_use_ct_get(vptr)) {
            if (variable_is_const(vptr)) {
              size_t sz;
              if (variable_use_ct_get(vptr) && !variable_is_array(vptr)) {
                vdef = variable_const_get(vptr, variable_def_get(vptr), 0);
                sz = pic_variable_tag_dump(pf, pfile_write_asm, vptr);
                pfile_write(pf, pfile_write_asm, 
                    "%*sEQU %u\n", 
                    (sz < (PIC_LABEL_COLUMN_SZ - 1)) 
                    ? (PIC_LABEL_COLUMN_SZ - 1 - sz) : 1, "", vdef);
              }
            } else if (variable_def_type_function != variable_type_get(vptr)) {
              size_t     sz;
              variable_t master;

              vdef = variable_base_get(vptr, 0);

              sz = pic_variable_tag_dump(pf, pfile_write_asm, vptr);
              pfile_write(pf, pfile_write_asm, 
                  "%*sEQU 0x%04x", 
                  (sz < (PIC_LABEL_COLUMN_SZ - 1)) 
                  ? (PIC_LABEL_COLUMN_SZ - 1 - sz) : 1, "", vdef);
              master = variable_master_get(vptr);
              if (master) {
                variable_base_t offset;

                offset = variable_base_get(vptr, 0) 
                  - variable_base_get(master, 0);
                pfile_write(pf, pfile_write_asm, "    ; ");
                pic_variable_name_write(pf, pfile_write_asm, master);
                if (offset) {
                  pfile_write(pf, pfile_write_asm, " + %u", offset);
                }
                if (variable_dflag_test(vptr, VARIABLE_DEF_FLAG_BIT)) {
                  pfile_write(pf, pfile_write_asm,
                      " : %u", variable_bit_offset_get(vptr));
                }
              }
              pfile_write(pf, pfile_write_asm, "\n");
            }
          }
        }
      }
    }
    for (lptr = pfile_label_temp_head_get(pf);
         lptr;
         lptr = label_link_get(lptr)) {
      if (label_flag_test(lptr, LABEL_FLAG_USER)) {
        size_t sz;

        sz = pic_label_tag_dump(pf, pfile_write_asm, lptr);
        pfile_write(pf, pfile_write_asm,
            "%*sEQU 0x%04x\n",
            (sz < PIC_LABEL_COLUMN_SZ - 1)
          ? (PIC_LABEL_COLUMN_SZ - 1 -sz) : 1, "", label_pc_get(lptr));
      }
    }
  }
}


/*
 * NAME
 *   pic_cmd_generate
 *
 * DESCRIPTION
 *   populate the pic_codes[] array
 *
 * PARAMETERS
 *   pf  : 
 *   cmd : first command
 *
 * RETURN
 *   none
 *
 * NOTES
 */
void pic_cmd_generate(pfile_t *pf, const cmd_t cmd)
{
  label_t          lbl_tmp;
  cmd_t            cmd_ptr;
  unsigned         pass;
  pic_code_t       code;
  pfile_proc_t    *proc;

  lbl_tmp = pfile_lblptr_get(pf);
  /* define the processor (this needs to be changed) */
  pic_code_list_init(pf);
  for (proc = pfile_proc_root_get(pf);
       proc;
       proc = pfile_proc_next(proc)) {
    pfile_proc_frame_sz_calc(proc);
  }
  for (pass = 0; (pass < 2) && !pfile_errct_get(pf); pass++) {
    pfile_log(pf, pfile_log_info, PIC_MSG_GENERATING_CODE, 1 + pass);
    if (1 == pass) {
      pic_code_list_reset(pf);
      pfile_lblptr_set(pf, lbl_tmp);
      pic_code_preamble(pf);
    }

    for (code = pic_code_list_head_get(pf); 
         code; 
         code = pic_code_next_get(code)) {
      pic_code_cmd_set(code, cmd);
    }
    for (cmd_ptr = cmd; cmd_ptr; cmd_ptr = cmd_link_get(cmd_ptr)) {
      if (cmd_is_reachable(cmd_ptr)) {
        pic_cmd_out(pf, &cmd_ptr);
      }
    }
    if (0 == pass) {
      /* we need to create the intrinsic functions here. that guarentees
       * any needed internal variables are also created */
      pic_code_preamble(pf);
      pic_variable_counters_set(pf);
      pic_variable_alloc(pf);
    }
  }

#if 1
#if 1
  pic_w_value_optimize(pf);
  pic_code_branch_optimize(pf);
  pic_code_return_literal_optimimze(pf);
  pic_code_branch_optimize(pf);
  pic_code_skip_cond_optimize(pf);
#if 1
  pic_code_free_unused(pf);
#endif
#if 1
  pic_code_databits_optimize(pf);
  pic_code_databits_remove(pf);
#endif
  /* each time something is removed, we must again go through the branchbits
     optimization. otherwise the code could be larger than necessary.
     For example, say the code is 2K + 1 byte. The first branchbits_optimize
     will have to address both PCLATH bits. brnachbits_remove() will notice
     the smaller size and remove all PCLATH<4> instructions. When these
     instructions are removed, the code size could fall below 2K in which
     case there is no reason to have *any* PCLATH instructions */
#if 1
  pic_code_branchbits_remove(pf);
  do {
    pic_code_branchbits_optimize(pf);
  } while (pic_code_branchbits_remove(pf));
#endif
#endif
#endif
  /* printf("PIC analyze max depth: %u\n", max_depth); */
  /* pic_cmd_dump(pf); */
  /* note: i'll also need to analyze starting at the interrupt vector (4) */
  {
    char *chip_name;

    chip_name = pic_chip_name_get(pf);
    pic_asm_header_write(pf, chip_name);
    FREE(chip_name);
  }
  pic_variable_counters_set(pf);
}

boolean_t pic_opcode_is_conditional(pic_opcode_t op)
{
  return (pic_opcode_incfsz == op)
    || (pic_opcode_decfsz == op)
    || (pic_opcode_btfsc == op)
    || (pic_opcode_btfss == op);
}

boolean_t pic_opcode_is_pseudo_op(pic_opcode_t op)
{
  return (pic_opcode_org == op)
      || (pic_opcode_end == op)
      || (pic_opcode_none == op)
      || (pic_opcode_branchlo_nop == op)
      || (pic_opcode_branchhi_nop == op);
}

boolean_t pic_code_modifies_pcl(pfile_t *pf, pic_code_t code)
{
  pic_opcode_t op;
  boolean_t    rc;

  op = pic_code_op_get(code);
  rc = boolean_false;
  switch (op) {
    case pic_opcode_movwf:
    case pic_opcode_addwf:
    case pic_opcode_andwf:
    case pic_opcode_xorwf:
    case pic_opcode_iorwf:
    case pic_opcode_subwf:
    case pic_opcode_comf:
    case pic_opcode_decf:
    case pic_opcode_decfsz:
    case pic_opcode_incf:
    case pic_opcode_incfsz:
    case pic_opcode_rlf:
    case pic_opcode_rrf:
    case pic_opcode_swapf:
    case pic_opcode_clrf:
    case pic_opcode_bcf:
    case pic_opcode_bsf:
      /* if this is an operation on which the dest is _pcl, treat it as a return */
      if ((pic_opcode_movwf == op) 
          || (pic_opdst_f == pic_code_dst_get(code))) {
        value_t    val;
        variable_t var;
        
        val = pic_code_value_get(code);
        var = pfile_variable_find(pf, pfile_log_err, "_pcl", 0);
        if (value_variable_get(val) == var) {
          rc = boolean_true;
        }
        variable_release(var);
      }
      break;

    default: 
      break;
  }
  return rc;
}


void pic_init(pfile_t *pf)
{
}

