/************************************************************
 **
 ** pic_brop.c : PIC branch optimization definitions
 **
 ** Copyright (c) 2004-2005, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/
#include <string.h>
#include <assert.h>

#include "../libcore/pf_proc.h"
#include "pic.h"
#include "piccolst.h"
#include "pic_inst.h"
#include "pic_msg.h"
#include "pic_opfn.h"
#include "pic_brop.h"

/* note: this is very similar to databit optimization, but must
         be iterative because each time an operator is added or
         removed all of the other locations change which could
         cause yet another iteration
         the iterations work as follows:
           pic_code_pc_set() : sets the PC on all instructions & labels
           repeat
             analyze user path
             analyze data path
             remove redundant instructions
           while (label bank changed) */

/* run through & make sure all of the PC values are set correctly */
void pic_branchbits_pc_set(pfile_t *pf)
{
  pic_code_t code;
  pic_pc_t   pc;

  for (pc = 0, code = pic_code_list_head_get(pf);
       code;
       code = pic_code_next_get(code)) {
    if (pic_opcode_org == pic_code_op_get(code)) {
      /* unlike the other opcodes, I want to set the PC of an ORG
         to the ORG value */
      value_t org;

      org = pic_code_literal_get(code);
      pc = (pic_pc_t) value_const_get(org);
    }
    pic_code_pc_set(code, pc);
    if (pic_code_label_get(code)) {
      label_pc_set(pic_code_label_get(code), pc);
    }
    /* don't increment the PC for pseudo-ops */
    pc += pic_code_sz_get(code);
#if 0
    if (!pic_opcode_is_pseudo_op(pic_code_op_get(code))) {
      pc++;
    }
#endif
  }
}

/*
 * NAME
 *   pic_code_branchbits_analyze
 *
 * DESCRIPTION
 *   branch analyzing & fixing
 *
 * PARAMETERS
 *   pf : pfile
 *
 * RETURN
 *   TRUE if a branchbit transitions from NOP to SET/CLR
 *   FALSE if not
 *
 * NOTES
 */
static boolean_t pic_code_branchbits_analyze(pfile_t *pf, pic_code_t code,
    const pic_branchbits_t *pbranchbits, unsigned depth)
{
  pic_branchbits_t bbits;
  boolean_t        pv_code_is_cond;
  boolean_t        code_is_cond;
  boolean_t        ret;

  bbits = *pbranchbits;

  code_is_cond = boolean_false;
  ret          = boolean_false;

  do {
    pic_branchbits_state_t cbitstate;
    pic_code_t             next;
    pic_opcode_t           op;
    pic_pc_t               brlbl_pc;

    pv_code_is_cond = code_is_cond;
    code_is_cond    = boolean_false;
    next            = pic_code_next_get(code);
    op              = pic_code_op_get(code);

    pic_code_branchbits_get(code, &cbitstate);
    if ((cbitstate.before.pclath3 == bbits.pclath3)
        && (cbitstate.before.pclath4 == bbits.pclath4)) {
      /* this is what we expect, no change here */
      return ret;
    }

    if (pic_bitstate_unknown != cbitstate.before.pclath3) {
      if (cbitstate.before.pclath3 != bbits.pclath3) {
        bbits.pclath3 = pic_bitstate_indeterminate;
      }
      if (cbitstate.before.pclath4 != bbits.pclath4) {
        bbits.pclath4 = pic_bitstate_indeterminate;
      }
    }
    cbitstate.before = bbits;
    code_is_cond = boolean_false;

    if (pic_opcode_is_branch_bit(op)) {
      pic_code_t   brcode;
      pic_opcode_t brop;
      label_t      lbl;

      brcode = code;
      lbl = pic_code_brdst_get(brcode);
      brlbl_pc = label_pc_get(lbl);
      if ((pic_opcode_branchlo_set == op)
        || (pic_opcode_branchlo_clr == op)
        || (pic_opcode_branchlo_nop == op)) {
        brop = (brlbl_pc & 0x800) 
             ? pic_opcode_branchlo_set : pic_opcode_branchlo_clr;
        if (pic_opcode_branchlo_nop == op) {
          /* is this unnecessary? */
          if ((pic_opcode_branchlo_clr == brop)
            && (pic_bitstate_clr == bbits.pclath3)) {
            brop = op;
          } else if ((pic_opcode_branchlo_set == brop)
            && (pic_bitstate_set == bbits.pclath3)) {
            brop = op;
          } else {
            pic_code_flag_set(code, PIC_CODE_FLAG_BRANCHLO_LOCK);
          }
        }
      } else {
        /* op must be pic_opcode_branchhi_[set|clr] */
        brop = (brlbl_pc & 0x1000) 
             ? pic_opcode_branchhi_set : pic_opcode_branchhi_clr;
        if (pic_opcode_branchhi_nop == op) {
          if ((pic_opcode_branchhi_clr == brop)
            && (pic_bitstate_clr == bbits.pclath4)) {
            brop = op;
          } else if ((pic_opcode_branchhi_set == brop)
            && (pic_bitstate_set == bbits.pclath4)) {
            brop = op;
          } else {
            pic_code_flag_set(code, PIC_CODE_FLAG_BRANCHHI_LOCK);
          }
        }
      }
      if (brop != op) {
        op = brop;
        pic_code_op_set(code, brop);
        ret = boolean_true;
      }
    }

    switch (op) {
      case pic_opcode_branchlo_set:
        bbits.pclath3 = pic_bitstate_set;
        break;
      case pic_opcode_branchlo_clr:
        bbits.pclath3 = pic_bitstate_clr;
        break;
      case pic_opcode_branchhi_set:
        bbits.pclath4 = pic_bitstate_set;
        break;
      case pic_opcode_branchhi_clr:
        bbits.pclath4 = pic_bitstate_clr;
        break;
      case pic_opcode_ret:
      case pic_opcode_retfie:
      case pic_opcode_retlw:
        /* if this is unconditional we're done otherwise simply continue */
        if (!pv_code_is_cond) {
          next = 0;
        }
        break;
      case pic_opcode_goto:
      case pic_opcode_call:
        {
          pic_code_t brcode;
          boolean_t  is_suspend;

          brcode = pic_code_label_find(pf, pic_code_brdst_get(code));
          is_suspend = pic_code_is_suspend(pf, code);

          if ((pic_opcode_goto == pic_code_op_get(code)) 
              && !pv_code_is_cond
              && !is_suspend) {
            /* an absolute goto so next = brcode */
            next = brcode;
          } else {
            /* this is either a conditional goto or a call. either
               way we first process then branch, then continue
               processing from where we were. for the time being
               a call is going to invalidate the known bits */
            ret = pic_code_branchbits_analyze(pf, brcode, &bbits, depth + 1) 
              || ret;
            if (is_suspend || (pic_opcode_call == pic_code_op_get(code))) {
              bbits.pclath3 = pic_bitstate_indeterminate;
              bbits.pclath4 = pic_bitstate_indeterminate;
            }
          }
        }
        break;
      case pic_opcode_incfsz:
      case pic_opcode_decfsz:
      case pic_opcode_btfsc:
      case pic_opcode_btfss:
        code_is_cond = boolean_true;
        break;
      default:
        break;
    }
    if (pic_code_modifies_pcl(pf, code)) {
      next = 0;
    }
    cbitstate.action = bbits;
    pic_code_branchbits_set(code, &cbitstate);
    code = next;
    if (code) {
      pic_code_branchbits_get(code, &cbitstate);
    }
  } while (code);
  return ret;
}

/* this returns TRUE if a label has crossed as code boundary. in that
   case the entire analyze phase must be done again! 
   
   pre_add is TRUE is one of the branch-bits transitioned from NOP to SET/CLR
   once this becomes TRUE it remains TRUE and no other branchbits will be
   removed
*/
boolean_t pic_code_branchbits_post_analyze(pfile_t *pf)
{
  pic_code_t pv;
  pic_code_t code;
  unsigned   ins_ct; /* instruction ct */
  unsigned   rem_ct;
  boolean_t  lbl_fix;

  lbl_fix = boolean_false;
  pic_branchbits_pc_set(pf);
  for (ins_ct = 0, rem_ct = 0, pv = 0, code = pic_code_list_head_get(pf);
       code;
       ins_ct++) {
    pic_code_t next;

    next = pic_code_next_get(code);
    if (!pic_code_flag_test(code, PIC_CODE_FLAG_NO_OPTIMIZE)) {
      pic_branchbits_state_t cbitstate;

      pic_code_branchbits_get(code, &cbitstate);

      switch (pic_code_op_get(code)) {
        case pic_opcode_branchlo_set:
        case pic_opcode_branchlo_clr:
          if ((cbitstate.before.pclath3 == cbitstate.action.pclath3)
            && !pic_code_flag_test(code, PIC_CODE_FLAG_BRANCHLO_LOCK)) {
            pic_code_op_set(code, pic_opcode_branchlo_nop);
            rem_ct++;
          }
          break;
        case pic_opcode_branchhi_set:
        case pic_opcode_branchhi_clr:
          if ((cbitstate.before.pclath4 == cbitstate.action.pclath4) 
            && !pic_code_flag_test(code, PIC_CODE_FLAG_BRANCHHI_LOCK)) {
            pic_code_op_set(code, pic_opcode_branchhi_nop);
            rem_ct++;
          }
          break;
        default:
          break;
      }
      /* adjust the PC */
      pic_code_pc_set(code, (pic_pc_t) (pic_code_pc_get(code) - rem_ct));
      if (pic_code_label_get(code)) {
        pic_pc_t pc_old;
        pic_pc_t pc_new;
        label_t  lbl;

        lbl = pic_code_label_get(code);
        pc_old = label_pc_get(lbl);
        pc_new = pc_old - rem_ct;
        label_pc_set(lbl, pc_new);
        if ((pc_old & 0x1800) != (pc_new & 0x1800)) {
          pfile_log(pf, pfile_log_debug, "...%s moved from %04x to %04x",
              label_name_get(lbl), pc_old, pc_new);
          lbl_fix = boolean_true;
        }
      }
      pv = code;
    }
    code = next;
  }
  pfile_log(pf, pfile_log_debug, "...removed %u of %u instructions", 
      rem_ct, ins_ct);
  pic_branchbits_pc_set(pf);
  return lbl_fix;
}

/* i must set all branchbits to unknown before calling analyze */
static void pic_code_branchbits_pre_analyze(pfile_t *pf)
{
  pic_code_t             code;
  pic_branchbits_state_t brstate;
  pic_branchbits_t       brbits;

  brbits.pclath3 = pic_bitstate_unknown;
  brbits.pclath4 = pic_bitstate_unknown;
  brstate.before = brbits;
  brstate.action = brbits;
  for (code = pic_code_list_head_get(pf);
       code;
       code = pic_code_next_get(code)) {
    pic_code_branchbits_set(code, &brstate);
  }
}

void pic_code_branchbits_optimize(pfile_t *pf)
{
  unsigned  pass;
  boolean_t changed;

  pass = 0;

  pic_branchbits_pc_set(pf);
  do {
    label_t       lbl;
    pfile_proc_t *proc;

    pfile_log(pf, pfile_log_debug, PIC_MSG_FIXING_CODE_BITS, ++pass);
    changed = boolean_false;
    pic_code_branchbits_pre_analyze(pf);
    lbl = pfile_label_find(pf, pfile_log_none, PIC_LABEL_ISR);
    if (lbl) {
      pic_code_t code;

      code = pic_code_label_find(pf, lbl);
      if (code) {
        pic_branchbits_t branchbits;

        branchbits.pclath4 = pic_bitstate_clr;
        branchbits.pclath3 = pic_bitstate_clr;

        changed = pic_code_branchbits_analyze(pf, code, &branchbits, 1) 
          || changed;
      }
      label_release(lbl);
    }

    lbl = pfile_label_find(pf, pfile_log_err, PIC_LABEL_RESET);
    if (lbl) {
      pic_code_t code;

      code = pic_code_label_find(pf, lbl);
      if (code) {
        pic_branchbits_t branchbits;

        branchbits.pclath4 = pic_bitstate_clr;
        branchbits.pclath3 = pic_bitstate_clr;

        changed = pic_code_branchbits_analyze(pf, code, &branchbits, 1)
          || changed;
      }
      label_release(lbl);
    }
    /* find all procedures that are called indirectly */
    for (proc = pfile_proc_root_get(pf);
         proc;
         proc = pfile_proc_next(proc)) {
      if (pfile_proc_flag_test(proc, PFILE_PROC_FLAG_INDIRECT)
          || pfile_proc_flag_test(proc, PFILE_PROC_FLAG_TASK)) {
        lbl = pfile_proc_label_get(proc);
        if (lbl) {
          pic_branchbits_t branchbits;
          pic_code_t       code;

          code = (pic_code_t) label_code_get(lbl);
          if (code) {
            branchbits.pclath4 = pic_bitstate_indeterminate;
            branchbits.pclath3 = pic_bitstate_indeterminate;

            changed = pic_code_branchbits_analyze(pf, code, &branchbits, 1)
              || changed;
          }
        }
      }
    }
  } while (pic_code_branchbits_post_analyze(pf) || changed);
}

/*
 * NAME
 *   pic_code_branch_optimize
 *
 * DESCRIPTION
 *   optimize branches
 *
 * PARAMETERS
 *   pf : pfile handle
 *
 * RETURN
 *   none
 *
 * NOTES
 *   optimizations performed:
 *     branch to goto
 *     ---> resolve & branch
 *     goto to return
 *     ---> replace with return
 *     call to return
 *     ---> remove call
 *     goto following instruction
 *     ---> remove goto
 *     call followed by return
 *     ---> call changed to goto 
 */
/* if the code becomes less than 6K, remove all branchhi_* ops,
 * if it becomes less than 4K, remove all branchlo_* ops
 */ 
pic_code_t pic_code_brnext_get(pfile_t *pf, pic_code_t code, label_t *dst_lbl)
{
  label_t    brlbl;
  pic_code_t brdst;

  brlbl = pic_code_brdst_get(code);
  brdst = pic_code_label_find(pf, brlbl);
  /* let's figure out what the destination *really* is */
  while (brdst
      && (pic_opcode_is_branch_bit(pic_code_op_get(brdst))
        || (pic_opcode_none == pic_code_op_get(brdst)))) {
    brdst = pic_code_next_get(brdst);
  }
  if (dst_lbl) {
    *dst_lbl = brlbl;
  }
  return brdst;
}

void pic_code_branch_optimize(pfile_t *pf)
{
  pic_code_t code;
  unsigned   goto_return         = 0;
  unsigned   call_dst_return     = 0;
  unsigned   return_follows_call = 0;
  unsigned   branch_goto         = 0;
  unsigned   total_freed         = 0;
  boolean_t  prev_cond; /* true if the previous instruction is conditional */

  prev_cond = boolean_false;

  pfile_log(pf, pfile_log_debug, "optimizing branches...");
  code = pic_code_list_head_get(pf);
  while (code) {
    pic_code_t next;

    next = pic_code_next_get(code);
    if (!pic_opcode_is_pseudo_op(pic_code_op_get(code))) {
      boolean_t code_cond;

      code_cond = pic_opcode_is_conditional(pic_code_op_get(code));

      if (!pic_code_flag_test(code, PIC_CODE_FLAG_NO_OPTIMIZE)
        && !prev_cond
        && ((pic_opcode_goto == pic_code_op_get(code))
          || (pic_opcode_call == pic_code_op_get(code)))) {
        pic_code_t brdst;
        label_t    brlbl;

        for (brdst = next;
             brdst
             && (pic_opcode_none == pic_code_op_get(brdst))
             && (pic_code_label_get(brdst) != pic_code_brdst_get(code));
             brdst = pic_code_next_get(brdst))
          ; /* empty */
        if ((pic_opcode_goto == pic_code_op_get(code))
          && (pic_code_label_get(brdst) == pic_code_brdst_get(code))) {
          /* GOTO label 
             label: */
          pic_code_list_remove(pf, code);
          pic_code_free(code);
        } else {
          pic_code_t brnext;

          /* brnext is required to detect infinite loops */
          brdst  = pic_code_brnext_get(pf, code, &brlbl);
          brnext = pic_code_brnext_get(pf, brdst, 0);
          while (brdst 
            && (pic_opcode_goto == pic_code_op_get(brdst))
            && (brdst != brnext)) {
            brdst  = pic_code_brnext_get(pf, brdst, &brlbl);
            brnext = pic_code_brnext_get(pf, brnext, 0);
          }
          if (brnext == brdst) {
            pfile_log(pf, pfile_log_debug, "loop detected: %s",
              label_name_get(brlbl));
          }
#if 0
          do {
            brlbl = pic_code_brdst_get(brdst);
            brdst = pic_code_label_find(pf, brlbl);
            /* let's figure out what the destination *really* is */
            while (brdst
                && (pic_opcode_is_branch_bit(pic_code_op_get(brdst))
                  || (pic_opcode_none == pic_code_op_get(brdst)))) {
              brdst = pic_code_next_get(brdst);
            }
          } while (brdst && (pic_opcode_goto == pic_code_op_get(brdst)));
#endif
          /* brlbl is the last label we've found and brdst is the first
           * instruction past the last label */
          if (brlbl != pic_code_brdst_get(code)) {
            /* change any branch bit destinations */
            pic_code_t brbits;

            brbits = pic_code_prev_get(code);
            while ((pic_code_brdst_get(brbits) == pic_code_brdst_get(code))
                && pic_opcode_is_branch_bit(pic_code_op_get(brbits))) {
              pic_code_brdst_set(brbits, brlbl);
              brbits = pic_code_prev_get(brbits);
            }
          }
          if ((pic_opcode_ret == pic_code_op_get(brdst))
            || (pic_opcode_retfie == pic_code_op_get(brdst))
            || (pic_opcode_retlw == pic_code_op_get(brdst))) {
            pic_code_t code_tmp;

            if (pic_opcode_goto == pic_code_op_get(code)) {
              /* GOTO a RETURN ; replace the GOTO */
              pic_code_op_set(code, pic_code_op_get(brdst));
              pic_code_literal_set(code, pic_code_literal_get(brdst));
              pic_code_brdst_set(code, LABEL_NONE);
              goto_return++;
              code = pic_code_prev_get(code);
            } else {
              /* CALL a RETURN ; remove the CALL */
              call_dst_return++;
              code_tmp = pic_code_prev_get(code);
              /* if this is call --> retlw, replace with movlw
                 otherwise remove this instruction */
              if (pic_opcode_retlw == pic_code_op_get(brdst)) {
                pic_code_t rcode; /* replacement code */

                rcode = pic_code_alloc(
                  pic_code_label_get(code),
                  pic_opcode_movlw,
                  pic_code_flag_get_all(code));
                pic_code_literal_set(rcode, pic_code_literal_get(brdst));
                pic_code_cmd_set(rcode, pic_code_cmd_get(code));
                pic_code_list_insert(pf, code_tmp, rcode);
              } else {
                total_freed++;
              }
              pic_code_list_remove(pf, code);
              pic_code_free(code);
              code = code_tmp;
            }
            while (code && pic_opcode_is_branch_bit(pic_code_op_get(code))) {
              code_tmp = pic_code_prev_get(code);
              pic_code_list_remove(pf, code);
              pic_code_free(code);
              total_freed++;
              code = code_tmp;
            }
          } else if (pic_code_brdst_get(code) != brlbl) {
            pic_code_brdst_set(code, brlbl);
            branch_goto++;
          }
          if (pic_opcode_call == pic_code_op_get(code)) {
            /* is this call followed by return? if yes, make it a goto */
            pic_code_t code_tmp;

            code_tmp = pic_code_next_get(code);
            while (code_tmp
                && (pic_opcode_none == pic_code_op_get(code_tmp))) {
              code_tmp = pic_code_next_get(code_tmp);
            }
            if (pic_opcode_ret == pic_code_op_get(code_tmp)) {
              return_follows_call++;
              pic_code_op_set(code, pic_opcode_goto);
              next = pic_code_next_get(code_tmp);
            }
          }
        }
      }
      prev_cond = code_cond;
    }
    code = next;
  }
  pfile_log(pf, pfile_log_debug, 
      "...goto/return(%u) call/return(%u) branch/goto(%u)",
      goto_return, call_dst_return, branch_goto);
  pfile_log(pf, pfile_log_debug,
      "...return follows call(%u) total freed(%u)",
      return_follows_call, total_freed);
}

/*
 *     movlw or clrw followed by return
 *     ---> retlw
 */
void pic_code_return_literal_optimimze(pfile_t *pf)
{
  pic_code_t code;
  unsigned   return_lit = 0;

  pfile_log(pf, pfile_log_debug, "optimizing return literal...");
  for (code = pic_code_list_head_get(pf);
       code;
       code = pic_code_next_get(code)) {
    if (!pic_code_flag_test(code, PIC_CODE_FLAG_NO_OPTIMIZE)) {
      pic_code_t next;

      next = pic_code_next_get(code);
      while (next && (pic_opcode_none == pic_code_op_get(next))) {
        next = pic_code_next_get(next);
      }

      if (((pic_opcode_movlw == pic_code_op_get(code))
          || (pic_opcode_clrw == pic_code_op_get(code)))
          && (pic_opcode_ret == pic_code_op_get(next))) {
        if (pic_opcode_clrw == pic_code_op_get(code)) {
          value_t val;

          val = pfile_constant_get(pf, 0, VARIABLE_DEF_NONE);
          pic_code_literal_set(code, val);
          value_release(val);
        }
        pic_code_op_set(code, pic_opcode_retlw);
        return_lit++;
      }
    }
  }
  pfile_log(pf, pfile_log_debug, "...return literal(%u)", return_lit);
}

/* this looks for:
      btfsc/btfss blah
      goto label
      one instruction
   label:

   and changes is into
      (invert condition)
      one instruction
   label:

   it's done on it's own so it can be called by pic_code_branchbits_remove
*/
boolean_t pic_code_skip_cond_optimize(pfile_t *pf)
{
  pic_code_t code;
  unsigned   ct;

  ct = 0;

  for (code = pic_code_list_head_get(pf);
       code;
       code = pic_code_next_get(code)) {
    if ((pic_opcode_btfsc == pic_code_op_get(code))
      || (pic_opcode_btfss == pic_code_op_get(code))) {
      pic_code_t next;

      next = pic_code_next_get(code);
      if (pic_opcode_goto == pic_code_op_get(next)) {
        pic_code_t next2;

        next2 = pic_code_next_get(next);
        next2 = pic_code_next_get(next2);
        if (pic_code_label_get(next2) == pic_code_brdst_get(next)) {
          pic_code_op_set(code,
            (pic_opcode_btfsc == pic_code_op_get(code))
            ? pic_opcode_btfss : pic_opcode_btfsc);
          pic_code_list_remove(pf, next);
          pic_code_free(next);
          ct++;
        }
      }
    }
  }
  pfile_log(pf, pfile_log_debug,
      "%u instructions removed by skip_cond optimizer", ct);
  return ct != 0;
}

/* nb: since a branchlo_nop could turn into a branchlo_[set|clr] it needs
       to be accounted for */
boolean_t pic_code_branchbits_remove(pfile_t *pf)
{
  /*do {*/
    unsigned   branchhi_ct     = 0;
    unsigned   branchlo_ct     = 0;
    unsigned   branchlo_nop_ct = 0;
    unsigned   pc              = 0;
    unsigned   total;
    pic_code_t code;
    enum {
      remove_none = 0,
      remove_hi,
      remove_all
    } rem;

    pfile_log(pf, pfile_log_debug, "optimizing branch bits(2)...");
    pic_branchbits_pc_set(pf);
    for (code = pic_code_list_head_get(pf);
         code;
         code = pic_code_next_get(code)) {
      if (pic_code_pc_get(code) > pc) {
        pc = pic_code_pc_get(code);
      }
      switch (pic_code_op_get(code)) {
        case pic_opcode_branchlo_nop:
          branchlo_nop_ct++;
          break;
        case pic_opcode_branchlo_clr:
        case pic_opcode_branchlo_set:
          if (!pic_code_flag_test(code, PIC_CODE_FLAG_NO_OPTIMIZE)) {
            branchlo_ct++;
          }
          break;
        case pic_opcode_branchhi_clr:
        case pic_opcode_branchhi_set:
          if (!pic_code_flag_test(code, PIC_CODE_FLAG_NO_OPTIMIZE)) {
            branchhi_ct++;
          }
          break;
        default:
          break;
      }
    }
    if (pc - branchlo_ct - branchhi_ct < 2048) {
      /* remove all branch bits */
      rem = remove_all;
    } else if (pc - branchhi_ct + branchlo_nop_ct < 4096) {
      /* remove all branchhi bits */
      rem = remove_hi;
    } else {
      rem = remove_none;
    }
    total = 0;
    if (remove_none != rem) {
      pic_code_t next;

      for (code = pic_code_list_head_get(pf);
           code;
           code = next) {
        next = pic_code_next_get(code);
        if (!pic_code_flag_test(code, PIC_CODE_FLAG_NO_OPTIMIZE)) {
          switch (pic_code_op_get(code)) {
            case pic_opcode_branchlo_clr:
            case pic_opcode_branchlo_set:
            case pic_opcode_branchlo_nop:
              if (remove_all != rem) {
                break;
              }
              /* fall through */
            case pic_opcode_branchhi_clr:
            case pic_opcode_branchhi_set:
            case pic_opcode_branchhi_nop:
              pic_code_list_remove(pf, code);
              pic_code_free(code);
              total++;
              break;
            default:
              break;
          }
        }
      }
    }
    pfile_log(pf, pfile_log_debug, "...unneeded branch bits(%u)", 
        total);
  return pic_code_skip_cond_optimize(pf) || (0 != total);  
  /*} while (pic_code_skip_cond_optimize(pf));*/
  /*return 0;*/
}

