/************************************************************
 **
 ** pic_emu.c : PIC instruction emulator
 **
 ** Copyright (c) 2007, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/
#include <string.h>
#include <assert.h>

#include "../libcore/cmd.h"
#include "piccolst.h"
#include "pic_inst.h"

#include "pic_emu.h"

#define PIC_DATA_MEM_SIZE 512
#define PIC_CALL_STACK_SIZE 8

#define PIC_SFR_IND    0x0000
#define PIC_SFR_PCL    0x0002
#define PIC_SFR_STATUS 0x0003
#define   PIC_SFR_STATUS_C   0
#define   PIC_SFR_STATUS_DC  1
#define   PIC_SFR_STATUS_Z   2
#define   PIC_SFR_STATUS_RP1 6
#define   PIC_SFR_STATUS_RP0 5
#define   PIC_SFR_STATUS_IRP 7
#define PIC_SFR_FSR    0x0004
#define PIC_SFR_PCLATH 0x000a

#define BIT(n) (1U << n)

unsigned pic_emu_instr_ct;

struct pic_emu_state_ {
  uchar     w;
  uchar    *mem;
  uchar    *imem;    /* bits 0..3 : # of times read
                             4..7 : # of times written */
  unsigned *and_map; /* before reading or writing, the position is
                        AND'd with this value so all reads & writes
                        will occur at the lowest allowed value */
  pic_code_t call_stack[PIC_CALL_STACK_SIZE];
  unsigned   call_stack_ptr;

  /*label_t    lbl_entry; *//* required */
};

/* validate pos based on the and_map, status:RP1 and status:RP0;
 * return the translated position (lowest position for those mapped
 * in more than one bank) */

/* pos = literal position (0..511)
 * f   = lower 7 bits
 */
static unsigned pic_emu_data_mem_xlate(
    const pic_emu_state_t state, unsigned pos, unsigned f,
    boolean_t no_xlate, const pfile_pos_t *fpos)
{
  if (pos >= PIC_DATA_MEM_SIZE) {
    printf("%s:%u access out of range (%u >= %u)\n",
        (fpos) ? pfile_source_name_get(fpos->src) : "internal", 
        (fpos) ? fpos->line : 0,
        pos, PIC_DATA_MEM_SIZE);
    abort();
  } else if ((pos & 0x7f) != f) {
    printf("%s:%u access doesn't match registry (%u != %u)\n", 
        (fpos) ? pfile_source_name_get(fpos->src) : "internal", 
        (fpos) ? fpos->line : 0,
        pos & 0x7f, f);
    abort();
  } else {
    if (no_xlate || (0 == pos)) {
      f = pos;
    } else {
      /* get bits 7 & 8 from STATUS */
      f |= (state->mem[PIC_SFR_STATUS]
          & (BIT(PIC_SFR_STATUS_RP1) | BIT(PIC_SFR_STATUS_RP0))) << 2;
    }
    /* mask off bits from pos & pos_hi */
    pos    &= state->and_map[pos];
    f      &= state->and_map[pos];
    /* check that the high two bits are correct */

    if (pos != f) {
      printf("%s:%u RP1:RP0 don't match: (got=%u expect=%u)\n",
        (fpos) ? pfile_source_name_get(fpos->src) : "internal", 
        (fpos) ? fpos->line : 0,
        f, pos);
      abort();
    } else {
      if (PIC_SFR_IND == pos) {
        /* translate indirect */
        pos = state->mem[PIC_SFR_FSR] +
          ((state->mem[PIC_SFR_STATUS] & BIT(PIC_SFR_STATUS_IRP))
          ? 0x0100 : 0x0000);
      }
    }
  }
  return pos;
}

/* read a byte from data memory */
static uchar pic_emu_data_mem_read(
    const pic_emu_state_t state, unsigned pos, unsigned f,
    boolean_t check_uninit, boolean_t no_xlate, const pfile_pos_t *fpos)
{
  pos = pic_emu_data_mem_xlate(state, pos, f, no_xlate, fpos);
  if (check_uninit && (0 == ((state->imem[pos] & 0xf0)))) {
    printf("%s:%u uninitialized memory read (%u)!\n", 
        (fpos) ? pfile_source_name_get(fpos->src) : "internal", 
        (fpos) ? fpos->line : 0,
        pos);
  }
  if ((state->imem[pos] & 0x0f) != 0x0f) {
    state->imem[pos]++;
  }
  return state->mem[pos];
}

/* write a byte to data memory; d = 0(W) 1(F) */
static void pic_emu_data_mem_write(
    pic_emu_state_t state, unsigned pos, unsigned f, uchar ch,
    boolean_t no_xlate, const pfile_pos_t *fpos)
{
  pos = pic_emu_data_mem_xlate(state, pos, f, no_xlate, fpos);
  if ((state->imem[pos] & 0xf0) != 0xf0) {
    state->imem[pos] += 0x10;
  }
  state->mem[pos] = ch;
}

static void pic_emu_status_bit_change(
    pic_emu_state_t state, unsigned mask, unsigned val,
    const pfile_pos_t *fpos)
{
  pic_emu_data_mem_write(state, PIC_SFR_STATUS, PIC_SFR_STATUS, 
      (pic_emu_data_mem_read(state, PIC_SFR_STATUS, PIC_SFR_STATUS,
                             boolean_false, boolean_false, fpos)
      & ~mask) | val, boolean_false, fpos);
}

static void pic_emu_arithmetic_write(
    pic_emu_state_t state, unsigned pos, unsigned f, unsigned d,
    unsigned val, unsigned status_bits, const pfile_pos_t *fpos)
{
  if (status_bits & BIT(PIC_SFR_STATUS_C)) {
    pic_emu_status_bit_change(state, BIT(PIC_SFR_STATUS_C),
        (val > 255) ? BIT(PIC_SFR_STATUS_C) : 0, fpos);
  }
  val &= 0xff;
  if (status_bits & BIT(PIC_SFR_STATUS_Z)) {
    pic_emu_status_bit_change(state, BIT(PIC_SFR_STATUS_Z),
        (0 == val) ? BIT(PIC_SFR_STATUS_Z) : 0, fpos);
  }
  if (0 == d) {
    state->w = (uchar) val;
  } else {
    pic_emu_data_mem_write(state, pos, f, (uchar) val, boolean_false, fpos);
  }
}

void pic_emu_var_add(pfile_t *pf, const pic_emu_var_def_t *vdef)
{
  variable_def_t def;
  variable_t     var;

  def = variable_def_alloc(0, variable_def_type_integer,
      vdef->dflags, vdef->sz);
  pfile_variable_alloc(pf, PFILE_VARIABLE_ALLOC_GLOBAL,
    vdef->name, def, VARIABLE_NONE, &var);
  if (vdef->dflags & VARIABLE_DEF_FLAG_CONST) {
    variable_const_set(var, def, 0, vdef->base[0]);
  } else {
    size_t jj;

    for (jj = 0; 
        (jj < VARIABLE_MIRROR_CT)
        && (VARIABLE_BASE_UNKNOWN != vdef->base[jj]); 
          jj++) {
      variable_base_set(var, vdef->base[jj], jj);
    }
  }
  variable_release(var);
}

pic_emu_state_t pic_emu_state_alloc(pfile_t *pf)
{
  pic_emu_state_t  state;

  state = malloc(sizeof(*state));
  state->w       = 0;
  state->mem     = calloc(PIC_DATA_MEM_SIZE, sizeof(*state->mem));
  state->imem    = calloc(PIC_DATA_MEM_SIZE, sizeof(*state->imem));
  state->and_map = calloc(PIC_DATA_MEM_SIZE, sizeof(*state->and_map));
  state->call_stack_ptr = 0;

  pic_emu_data_mem_init(state);
  return state;
}

void pic_emu_state_free(pic_emu_state_t state)
{
  /*label_release(state->lbl_entry);*/
  free(state->and_map);
  free(state->imem);
  free(state->mem);
  free(state);
}

variable_const_t pic_emu_value_read(const pic_emu_state_t state,
                   const value_t val, const pfile_pos_t *fpos)
{
  variable_const_t cn;

  if (VARIABLE_NONE == value_variable_get(val)) {
    cn = pic_emu_w_get(state);
  } else {
    variable_sz_t    ii;
    variable_base_t  base;
    uchar            ch;

    base = value_base_get(val);
#if 0
    pic_emu_status_bit_change(state,
      BIT(PIC_SFR_STATUS_RP1) | BIT(PIC_SFR_STATUS_RP0),
      (base >> 2) & 0x60);
#endif
    cn = 0;
    ch = 0;
    for (ii = 0, cn = 0; ii < value_byte_sz_get(val); ii++, base++) {
      ch  = pic_emu_data_mem_read(state, base, base & 0x7f,
          boolean_true, boolean_true, fpos);
      cn |= ch << (8UL * ii);
    }
    if (value_is_signed(val) && (ch & 0x80)) {
      while (ii < 4) {
        cn |= 0xff << (8UL * ii);
        ii++;
      }
    }
    if (value_is_bit(val)) {
      cn >>= value_bit_offset_get(val);
      cn &= (1 << value_sz_get(val)) - 1;
      if (value_is_signed(val) 
          && (cn & (1 << (value_sz_get(val) - 1)))) {
        cn = -cn;
      }
    }
  }
  return cn;
}

void pic_emu_value_write(pic_emu_state_t state,
                   const value_t val, variable_const_t cn,
                   const pfile_pos_t *fpos)
{
  if (VARIABLE_NONE == value_variable_get(val)) {
    pic_emu_w_set(state, cn);
  } else {
    variable_sz_t ii;
    variable_base_t base;

    base = value_base_get(val);
#if 0
    pic_emu_status_bit_change(state,
      BIT(PIC_SFR_STATUS_RP1) | BIT(PIC_SFR_STATUS_RP0),
      (base >> 2) & 0x60);
#endif
    if (value_is_bit(val)) {
      if (value_is_boolean(val)) {
        cn = (cn) ? 1 : 0;
      } else {
        cn &= ((1UL << value_sz_get(val)) - 1);
      }
      cn <<= value_bit_offset_get(val);
    }
    for (ii = 0; ii < value_sz_get(val); ii++, cn >>= 8) {
      pic_emu_data_mem_write(state, base + ii, (base + ii) & 0x7f, cn,
          boolean_true, fpos);
    }
  }
}

/* determine if a volatile variable was accessed more than once */
void pic_emu_volatile_check(const pic_emu_state_t state, 
                   const value_t val)
{
  if (value_is_volatile(val)) {
    variable_base_t base;
    variable_sz_t   ii;

    base = value_base_get(val);
    for (ii = 0; ii < value_sz_get(val); ii++, base++) {
      uchar ch;
      
      assert(base < PIC_DATA_MEM_SIZE);
      ch = state->imem[base & state->and_map[base]];
      ch = (ch & 0x0f) + (ch >> 4);
      assert(ch < 2);
    }
  }
}

uchar pic_emu_w_get(const pic_emu_state_t state)
{
  return (state) ? state->w : 0;
}

void pic_emu_w_set(pic_emu_state_t state, uchar cn)
{
  if (state) {
    state->w = cn;
  }
}

#if 0
label_t pic_emu_entry_get(const pic_emu_state_t state)
{
  return (state) ? state->lbl_entry : LABEL_NONE;
}
#endif

void pic_emu_data_mem_init(pic_emu_state_t state)
{
  size_t ii;
  static const struct {
    unsigned lo;
    unsigned hi;
  } SFR_regions[] = {
    {0x000, 0x01f},
    {0x080, 0x09f},
    {0x100, 0x10f},
    {0x180, 0x18f}
  };
  /* anything not listed is only available in one bank */
  static const struct {
    unsigned pos;
    unsigned and;
  } and_map[] = {
    { 0x0000, 0x007f }, /* IND                    */
    { 0x0001, 0x00ff }, /* TMR0 {0x0001, 0x0101)  */
    { 0x0002, 0x007f }, /* PCL                    */
    { 0x0003, 0x007f }, /* STATUS                 */
    { 0x0004, 0x007f }, /* FSR                    */
    { 0x0006, 0x00ff }, /* PORTB {0x0006, 0x0106} */
    { 0x000a, 0x007f }, /* PCLATH                 */
    { 0x000b, 0x007f }, /* INTCON                 */
    { 0x0070, 0x007f }, /* shared region...       */
    { 0x0071, 0x007f },
    { 0x0072, 0x007f },
    { 0x0073, 0x007f },
    { 0x0074, 0x007f },
    { 0x0075, 0x007f },
    { 0x0076, 0x007f },
    { 0x0077, 0x007f },
    { 0x0078, 0x007f },
    { 0x0079, 0x007f },
    { 0x007a, 0x007f },
    { 0x007b, 0x007f },
    { 0x007c, 0x007f },
    { 0x007d, 0x007f },
    { 0x007e, 0x007f },
    { 0x007f, 0x007f },
    { 0x0081, 0x00ff }, /* OPTION_REG */
    { 0x0086, 0x00ff }  /* TRISB */
  };
  static const uchar deadbeaf[] = {0xde, 0xad, 0xbe, 0xef};

  /* setup the SFR regions */
  memset(state->imem, 0, sizeof(state->imem));

  for (ii = 0; ii < COUNT(SFR_regions); ii++) {
    size_t jj;

    for (jj = SFR_regions[ii].lo; jj <= SFR_regions[ii].hi; jj++) {
      state->imem[jj] = 0xff;
    }
  }
  for (ii = 0; ii < PIC_DATA_MEM_SIZE; ii++) {
    state->and_map[ii] = 0xffff; /* assume all bits needed */
  }
  for (ii = 0; ii < COUNT(and_map); ii++) {
    size_t jj;

    for (jj = and_map[ii].pos; jj < PIC_DATA_MEM_SIZE; jj += 80) {
      if ((jj & and_map[ii].and) == and_map[ii].pos) {
        state->and_map[jj] = and_map[ii].and;
      }
    }
  }
  for (ii = 0; ii < PIC_DATA_MEM_SIZE; ii++) {
    if (!state->imem[ii]) {
      state->mem[ii] = deadbeaf[ii % sizeof(deadbeaf)];
    }
  }
}

static boolean_t pic_code_is_exec(pic_code_t code)
{
  return (pic_opcode_org          != pic_code_op_get(code))
      && (pic_opcode_end          != pic_code_op_get(code)) 
      && (pic_opcode_none         != pic_code_op_get(code))
      && (pic_opcode_branchlo_nop != pic_code_op_get(code))
      && (pic_opcode_branchhi_nop != pic_code_op_get(code));
}

static pic_code_t pic_code_next_exec_get(pic_code_t code)
{
  do {
    code = pic_code_next_get(code);
  } while ((PIC_CODE_NONE != code) && !pic_code_is_exec(code));
  return code;
}

/* execute a single instruction */
pic_code_t pic_emu_instr(pfile_t *pf,
    pic_emu_state_t state, const pic_code_t code)
{
  FILE      *old_asm;
  FILE      *old_hex;
  pic_code_t next;
  pfile_pos_t fpos;
  unsigned   pcode;
  cmd_t      cmd;
  static const struct {
    unsigned     mask;
    unsigned     value;
    pic_opcode_t op;
  } decode[] = {
    /* byte oriented */
    {0x3f80, 0x0080, pic_opcode_movwf  }, /* movwf   f   */
    {0x3f9f, 0x0000, pic_opcode_nop    }, /* nop     -   */
    {0x3f00, 0x0100, pic_opcode_clrf   }, /* clr[wf] f,d */
    {0x3f00, 0x0200, pic_opcode_subwf  }, /* subwf   f,d */
    {0x3f00, 0x0300, pic_opcode_decf   }, /* decf    f,d */
    {0x3f00, 0x0400, pic_opcode_iorwf  }, /* iorwf   f,d */
    {0x3f00, 0x0500, pic_opcode_andwf  }, /* andwf   f,d */
    {0x3f00, 0x0600, pic_opcode_xorwf  }, /* xorwf   f,d */
    {0x3f00, 0x0700, pic_opcode_addwf  }, /* addwf   f,d */
    {0x3f00, 0x0800, pic_opcode_movf   }, /* movf    f,d */
    {0x3f00, 0x0900, pic_opcode_comf   }, /* comf    f,d */
    {0x3f00, 0x0a00, pic_opcode_incf   }, /* incf    f,d */
    {0x3f00, 0x0b00, pic_opcode_decfsz }, /* decfsz  f,d */
    {0x3f00, 0x0c00, pic_opcode_rrf    }, /* rrf     f,d */
    {0x3f00, 0x0d00, pic_opcode_rlf    }, /* rlf     f,d */
    {0x3f00, 0x0e00, pic_opcode_swapf  }, /* swapf   f,d */
    {0x3f00, 0x0f00, pic_opcode_incfsz }, /* incfsz  f,d */
    /* bit oriented */
    {0x3c00, 0x1000, pic_opcode_bcf    }, /* bcf     f,b */
    {0x3c00, 0x1400, pic_opcode_bsf    }, /* bsf     f,b */
    {0x3c00, 0x1800, pic_opcode_btfsc  }, /* btfsc   f,b */
    {0x3c00, 0x1c00, pic_opcode_btfss  }, /* btfss   f,b */
    /* literal & control */
    {0x3fff, 0x0008, pic_opcode_ret    }, /* return  -   */
    {0x3fff, 0x0009, pic_opcode_retfie }, /* retfie  -   */
    {0x3fff, 0x0063, pic_opcode_sleep  }, /* sleep   -   */
    {0x3fff, 0x0064, pic_opcode_clrwdt }, /* clrwdt  -   */
    {0x3800, 0x2000, pic_opcode_call   }, /* call    n   */
    {0x3800, 0x2800, pic_opcode_goto   }, /* goto    n   */
    {0x3c00, 0x3000, pic_opcode_movlw  }, /* movlw   k   */
    {0x3c00, 0x3400, pic_opcode_retlw  }, /* retlw   k   */
    {0x3f00, 0x3800, pic_opcode_iorlw  }, /* iorlw   k   */
    {0x3f00, 0x3900, pic_opcode_andlw  }, /* andlw   k   */
    {0x3f00, 0x3a00, pic_opcode_xorlw  }, /* xorlw   k   */
    {0x3e00, 0x3c00, pic_opcode_sublw  }, /* sublw   k   */
    {0x3e00, 0x3e00, pic_opcode_addlw  }  /* addlw   k   */
  };

  /* probably a really cheezy way to show each line, but hey! */
  if (!pfile_flag_test(pf, PFILE_FLAG_MISC_QUIET)) {
    old_hex = pfile_hex_file_set(pf, 0);
    old_asm = pfile_asm_file_set(pf, stdout);
    printf("%4u: ", pic_code_pc_get(code));
    pic_code_dump(pf, code);
    pfile_asm_file_set(pf, old_asm);
    pfile_hex_file_set(pf, old_hex);
  }
  pic_emu_instr_ct++;
  next  = pic_code_next_exec_get(code);
  cmd   = pic_code_cmd_get(code);
  fpos.src  = cmd_source_get(cmd);
  fpos.line = cmd_line_get(cmd);
  /*fpos  = cmd_file_pos_get(cmd);*/
  if (cmd_type_assert == cmd_type_get(cmd)) {
    value_t val;

    val = cmd_assert_value_get(cmd);
    if ((!value_is_const(val) && !pic_emu_value_read(state, val, &fpos)) 
        || (value_is_const(val) && !value_const_get(val))) {
      fprintf(stderr, "%s:%u assertion failed\n",
          pfile_source_name_get(fpos.src), fpos.line);
      /*abort();*/
    }
  } else {
    pcode = pic_code_to_pcode(pf, code);
    if (0xffff != pcode) {
      unsigned        k; /* 0 <= k <= 255  */
      unsigned        n; /* 0 <= n <= 1023 */
      unsigned        b; /* 0 <= b <= 7    */
      unsigned        f; /* 0 <= f <= 127  */
      unsigned        d; /* 0 <= d <= 1    */
      value_t         val;
      variable_base_t valpos;
      pic_opcode_t    op;
      size_t          ii;

      k      = pcode & 0x00ff;        /* literal                */
      n      = pcode & 0x03ff;        /* call/goto              */
      b      = (pcode >> 7) & 0x0007; /* bit ops                */
      f      = pcode & 0x007f;        /* file register          */
      d      = (pcode >> 7) & 0x0001; /* destination (0=w, 1=f) */
      val    = pic_code_value_get(code);
      valpos = 0xffff;

      if (VALUE_NONE != val) {
        if (value_is_const(val)) {
          valpos = value_const_get(val);
        } else if (value_base_get(val) == 0) {
          valpos = 0;
        } else {
          valpos = (value_base_get(val) 
            + value_const_get(value_baseofs_get(val))
            + value_bit_offset_get(val) / 8
            + pic_code_ofs_get(code));
        }
      }

      for (ii = 0; 
          (ii < COUNT(decode))
          && ((pcode & decode[ii].mask) != decode[ii].value);
           ii++)
        ;
      if (ii >= COUNT(decode)) {
        printf("cannot find: %s\n", pic_opcode_str(pic_code_op_get(code)));
      } else {
        op = decode[ii].op;
        switch (op) {
          case pic_opcode_org:
          case pic_opcode_end:
          case pic_opcode_none:
          case pic_opcode_datalo_set:
          case pic_opcode_datalo_clr:
          case pic_opcode_datahi_set:
          case pic_opcode_datahi_clr:
          case pic_opcode_irp_set:
          case pic_opcode_irp_clr:
          case pic_opcode_branchlo_set:
          case pic_opcode_branchlo_clr:
          case pic_opcode_branchlo_nop:
          case pic_opcode_branchhi_set:
          case pic_opcode_branchhi_clr:
          case pic_opcode_branchhi_nop:
          case pic_opcode_db:
          case pic_opcode_option:
          case pic_opcode_tris:
            assert(0);
            break;
          case pic_opcode_subwf:
          case pic_opcode_addwf:
            {
              unsigned n0;
              unsigned n1;
              unsigned n2;

              n1 = pic_emu_data_mem_read(state, valpos, f,
                boolean_true, boolean_false, &fpos) & 0xff;
              n2 = ((pic_opcode_addwf) == op ? state->w : -state->w) & 0xff;

              n0 = n1 + n2;
              /* it appears that negating 0 sets the C flag */
              if ((pic_opcode_subwf == op) && (0 == state->w)) {
                n0 |= 0x100;
              }

              pic_emu_arithmetic_write(state, valpos, f, d, n0,
                BIT(PIC_SFR_STATUS_C) | BIT(PIC_SFR_STATUS_Z), &fpos);
            }
            break;
          case pic_opcode_andwf:
            pic_emu_arithmetic_write(state, valpos, f, d,
              pic_emu_data_mem_read(state, valpos, f,
                boolean_true, boolean_false, &fpos) & state->w,
              BIT(PIC_SFR_STATUS_Z), &fpos);
            break;
          case pic_opcode_xorwf:
            pic_emu_arithmetic_write(state, valpos, f, d,
              pic_emu_data_mem_read(state, valpos, f,
                boolean_true, boolean_false, &fpos) ^ state->w,
              BIT(PIC_SFR_STATUS_Z), &fpos);
            break;
          case pic_opcode_iorwf:
            pic_emu_arithmetic_write(state, valpos, f, d,
              pic_emu_data_mem_read(state, valpos, f,
                boolean_true, boolean_false, &fpos) | state->w,
              BIT(PIC_SFR_STATUS_Z), &fpos);
            break;
          case pic_opcode_comf:
            pic_emu_arithmetic_write(state, valpos, f, d,
              ~pic_emu_data_mem_read(state, valpos, f,
                boolean_true, boolean_false, &fpos),
              BIT(PIC_SFR_STATUS_Z), &fpos);
            break;
          case pic_opcode_decf:
          case pic_opcode_decfsz:
            pic_emu_arithmetic_write(state, valpos, f, d,
              pic_emu_data_mem_read(state, valpos, f,
                boolean_true, boolean_false, &fpos) - 1,
              BIT(PIC_SFR_STATUS_Z), &fpos);
            if ((pic_opcode_decfsz == op)
              && (state->mem[PIC_SFR_STATUS] & BIT(PIC_SFR_STATUS_Z))) {
              next = pic_code_next_exec_get(next);
            }
            break;
          case pic_opcode_incf:
          case pic_opcode_incfsz:
            pic_emu_arithmetic_write(state, valpos, f, d,
              pic_emu_data_mem_read(state, valpos, f,
                boolean_true, boolean_false, &fpos) + 1,
              BIT(PIC_SFR_STATUS_Z), &fpos);
            if ((pic_opcode_incfsz == op)
              && (state->mem[PIC_SFR_STATUS] & BIT(PIC_SFR_STATUS_Z))) {
              next = pic_code_next_exec_get(next);
            }
            break;
          case pic_opcode_rlf:
            {
              unsigned mem;

              mem = (pic_emu_data_mem_read(state, valpos, f,
                  boolean_true, boolean_false, &fpos) << 1)
                | ((state->mem[PIC_SFR_STATUS] & BIT(PIC_SFR_STATUS_C))
                  ? 1 : 0);
              pic_emu_arithmetic_write(state, valpos, f, d, mem,
                  BIT(PIC_SFR_STATUS_C), &fpos);
            }
            break;
          case pic_opcode_rrf:
            {
              unsigned  mem;
              boolean_t c_flag;

              mem = pic_emu_data_mem_read(state, valpos, f, boolean_true, 
                  boolean_false, &fpos);
              c_flag = !!(mem & 1);

              mem = (mem >> 1) 
                | ((state->mem[PIC_SFR_STATUS] & BIT(PIC_SFR_STATUS_C)) 
                ? 0x80 : 0x00);
              pic_emu_arithmetic_write(state, valpos, f, d, mem, 0, &fpos);
              pic_emu_status_bit_change(state, BIT(PIC_SFR_STATUS_C),
                  (c_flag) ? BIT(PIC_SFR_STATUS_C) : 0, &fpos);
            }
            break;
          case pic_opcode_movf:
            pic_emu_arithmetic_write(state, valpos, f, d,
              pic_emu_data_mem_read(state, valpos, f, boolean_true,
                boolean_false, &fpos),
              BIT(PIC_SFR_STATUS_Z), &fpos);
            break;
          case pic_opcode_swapf:
            {
              unsigned ch;

              ch = pic_emu_data_mem_read(state, valpos, f, boolean_true,
                  boolean_false, &fpos);
              ch = ((ch << 4) & 0xf0) | (ch >> 4);
              pic_emu_arithmetic_write(state, valpos, f, d, ch, 0, &fpos);
            }
            break;
          case pic_opcode_clrf:
          case pic_opcode_clrw:
            pic_emu_arithmetic_write(state, valpos, f, d, 0,
                BIT(PIC_SFR_STATUS_Z), &fpos);
            break;
          case pic_opcode_movwf:
            pic_emu_data_mem_write(state, valpos, f, state->w,
                boolean_false, &fpos);
            break;
          case pic_opcode_nop:
            break; /* duh! */
          case pic_opcode_retlw:
            state->w = k;
            /* fall through */
          case pic_opcode_retfie:
          case pic_opcode_ret:
            state->call_stack_ptr = ((state->call_stack_ptr)
              ? state->call_stack_ptr : PIC_CALL_STACK_SIZE) - 1;
            next = state->call_stack[state->call_stack_ptr];
            break;
          case pic_opcode_sleep:
            break;
          case pic_opcode_clrwdt:
            break;
          case pic_opcode_bcf:
            /* special case : if f is STATUS and b is RP0 or RP1
                              then ignore valpos */
            if ((PIC_SFR_STATUS == f)
              && ((PIC_SFR_STATUS_RP1 == b)
                || (PIC_SFR_STATUS_RP0 == b)
                || (PIC_SFR_STATUS_IRP == b))) {
              valpos = f;
            } else if (PIC_SFR_PCLATH == f) {
              valpos = f;
            }
            pic_emu_data_mem_write(state, valpos, f,
                pic_emu_data_mem_read(state, valpos, f,
                  boolean_false,
                  boolean_false, &fpos)
                & ~(1 << b), boolean_false, &fpos);
            break;
          case pic_opcode_bsf:
            /* special case : if f is STATUS and b is RP0 or RP1
                              then ignore valpos */
            if ((PIC_SFR_STATUS == f)
              && ((PIC_SFR_STATUS_RP1 == b)
                || (PIC_SFR_STATUS_RP0 == b))) {
              valpos = f;
            } else if (PIC_SFR_PCLATH == f) {
              valpos = f;
            }
            pic_emu_data_mem_write(state, valpos, f,
                pic_emu_data_mem_read(state, valpos, f, boolean_false,
                  boolean_false, &fpos)
                | (1 << b), boolean_false, &fpos);
            break;
          case pic_opcode_btfsc:
            if (!(pic_emu_data_mem_read(state, valpos, f, boolean_true,
                    boolean_false, &fpos)
                & (1 << b))) {
              next = pic_code_next_exec_get(next);
            }
            break;
          case pic_opcode_btfss:
            if (pic_emu_data_mem_read(state, valpos, f, boolean_true,
                  boolean_false, &fpos)
                & (1 << b)) {
              next = pic_code_next_exec_get(next);
            }
            break;
          case pic_opcode_addlw:
          case pic_opcode_sublw:
            pic_emu_arithmetic_write(state, 0, 0, 0,
                state->w + ((pic_opcode_addlw == op) ? k : (-k)),
                BIT(PIC_SFR_STATUS_C) | BIT(PIC_SFR_STATUS_Z), &fpos);
            break;
          case pic_opcode_andlw:
            pic_emu_arithmetic_write(state, 0, 0, 0,
                state->w & k, BIT(PIC_SFR_STATUS_Z), &fpos);
            break;
          case pic_opcode_iorlw:
            pic_emu_arithmetic_write(state, 0, 0, 0,
                state->w | k, BIT(PIC_SFR_STATUS_Z), &fpos);
            break;
          case pic_opcode_movlw:
            pic_emu_arithmetic_write(state, 0, 0, 0,
                k, 0, &fpos);
            break;
          case pic_opcode_xorlw:
            pic_emu_arithmetic_write(state, 0, 0, 0,
                state->w ^ k, BIT(PIC_SFR_STATUS_Z), &fpos);
            break;
          case pic_opcode_call:
            state->call_stack[state->call_stack_ptr] = next;
            if (PIC_CALL_STACK_SIZE == ++state->call_stack_ptr) {
              state->call_stack_ptr = 0;
            }
            /* fall through */
          case pic_opcode_goto:
            /* almost all jumps are forward; calls are a crap shoot */
            next = pic_code_next_get(code);
            while ((next != code)
              && (pic_code_label_get(next) != pic_code_brdst_get(code))) {
              next = pic_code_next_get(next);
              if (PIC_CODE_NONE == next) {
                next = pic_code_list_head_get(pf);
              }
            }
            if (code == next) {
              printf("%s:%u jump destination not found: %s\n",
                pfile_source_name_get(fpos.src), fpos.line,
                label_name_get(pic_code_brdst_get(code)));
              abort();
            }
            break;
        }
      }
    }
  }
  return next;
}

void pic_emu(pfile_t *pf, pic_emu_state_t state)
{
  pic_code_t code;

  state->mem[PIC_SFR_STATUS] = 0;
  state->mem[PIC_SFR_IND]    = 0;
  state->mem[PIC_SFR_PCLATH] = 0;
  /* state->w                   = 0; */

  /* find the first executable instruction */
  for (code = pic_code_list_head_get(pf);
       (PIC_CODE_NONE != code)
       && !pic_code_is_exec(code);
       code = pic_code_next_get(code))
    ; /* empty body */
  while (PIC_CODE_NONE != code) {
     code = pic_emu_instr(pf, state, code);
  }
  if (!pfile_flag_test(pf, PFILE_FLAG_MISC_QUIET)) {
    printf("%u instructions executed\n", pic_emu_instr_ct);
  }
}


