/************************************************************
 **
 ** pic_opfn.c : PIC built-in function generation
 **
 ** Copyright (c) 2004-2005, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/
#include "pic_inst.h"
#include "pic_stvar.h"
#include "pic_op.h"
#include "pic_opfn.h"

/*
 * NAME
 *   pic_multiply_create
 *
 * DESCRIPTION
 *   create an n-bit multiply
 *
 * PARAMETERS
 *   pf : pfile
 *
 * RETURN
 *
 * NOTES
 *   multiplier   = val 1
 *   multiplicand = val 2
 *   algorithm
 *      for ii = 1 to bits
 *         mresult <<= 1
 *         multiplier <<= 1
 *         if (no carry)
 *           mresult += multiplicand
 *         end if
 *      next
 *   on exit, mresult holds the result
 *   nb: since mresult is shifted bits times, it needn't be cleared
 */
static void pic_multiply_create(pfile_t *pf)
{
  pic_var_mul_t mvars;
  label_t       top;
  label_t       skip;
  value_t       c_1;
  value_t       loop;

  pic_var_mul_get(pf, -1, &mvars);
  c_1    = pfile_constant_get(pf, 1, VARIABLE_DEF_NONE);
  loop   = pic_var_loop_get(pf);
  top    = pfile_label_alloc(pf, 0);
  skip   = pfile_label_alloc(pf, 0);

  pic_instr_append_w_kn(pf, pic_opcode_movlw, 8 * value_sz_get(mvars.mresult));
  pic_instr_append_f(pf, pic_opcode_movwf, loop, 0);
  pic_instr_append_label(pf, top);
  pic_op(pf, operator_shift_left, mvars.mresult, c_1, 0);
  pic_op(pf, operator_shift_left, mvars.multiplier, c_1, 0);
  pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_c");
  pic_instr_append_n(pf, pic_opcode_goto, skip);
  pic_op(pf, operator_add, mvars.mresult, mvars.multiplicand, 0);
  pic_instr_append_label(pf, skip);
  pic_instr_append_f_d(pf, pic_opcode_decfsz, loop, 0, pic_opdst_f);
  pic_instr_append_n(pf, pic_opcode_goto, top);
  pic_instr_append(pf, pic_opcode_ret);

  label_release(skip);
  label_release(top);
  pic_var_loop_release(pf, loop);
  value_release(c_1);
  pic_var_mul_release(pf, &mvars);
}

/*
 * NAME
 *    pic_divide_create
 *
 * DESCRIPTION
 *    create an n-bit divide
 *
 * PARAMETERS
 *    pf : pfile
 *
 * RETURN
 *
 * NOTES
 *   entry:
 *     dividend, divisor are set on entry
 *   algorithm:
 *     for ii = 1 to bits
 *       quotient <<= 1
 *       dividend <<= 1
 *       remainder = remainder << 1 | carry from prev. op
 *       if (remainder >= divisor)
 *         remainder -= divisor
 *         quotient |= 1
 *       end if
 *     next 
 */
static void pic_sdivide_create(pfile_t *pf)
{
  pic_var_div_t dvars;
  value_t       sign;
  value_t       loop;
  value_t       c_1;
  value_t       c_0;
  label_t       top;
  label_t       skip;
  label_t       sdivide;
  label_t       divide;
  label_t       lbl;

  divide = pfile_label_find(pf, pfile_log_none, PIC_LABEL_DIVIDE);
  sdivide = pfile_label_find(pf, pfile_log_none, PIC_LABEL_SDIVIDE);

  sign   = pic_var_sign_get(pf);
  pic_var_div_get(pf, -1, &dvars);
  c_1 = pfile_constant_get(pf, 1, VARIABLE_DEF_NONE);
  c_0 = pfile_constant_get(pf, 0, VARIABLE_DEF_NONE);

  loop   = pic_var_loop_get(pf);
  top    = pfile_label_alloc(pf, 0);
  skip   = pfile_label_alloc(pf, 0);

  if (sdivide) {
    /* create the signed divide preamble */
    lbl = pfile_label_alloc(pf, 0);
    pic_instr_append(pf, pic_opcode_clrw);
    pic_instr_append_f_bn(pf, pic_opcode_btfss, dvars.dividend, 
      value_sz_get(dvars.dividend) - 1, 7);
    pic_instr_append_n(pf, pic_opcode_goto, lbl);
    /* dividend = -dividend */
    pic_op(pf, operator_neg, dvars.dividend, dvars.dividend, 0);
    pic_instr_append_w_kn(pf, pic_opcode_movlw, 1);
    pic_instr_append_label(pf, lbl);
    label_release(lbl);
    lbl = 0;
    /* sign = 0 (unsigned) 1 (signed) */
    pic_instr_append_f(pf, pic_opcode_movwf, sign, 0);

    lbl = pfile_label_alloc(pf, 0);

    pic_instr_append(pf, pic_opcode_clrw);
    pic_instr_append_f_bn(pf, pic_opcode_btfss, dvars.divisor, 
      value_sz_get(dvars.divisor) - 1, 7);
    pic_instr_append_n(pf, pic_opcode_goto, lbl);
    /* divisor = -divisor */
    pic_op(pf, operator_neg, dvars.divisor, dvars.divisor, 0);
    pic_instr_append_w_kn(pf, pic_opcode_movlw, 1);
    pic_instr_append_label(pf, lbl);
    /* sign = 0 (unsigned) 1 (signed) */
    pic_instr_append_f_d(pf, pic_opcode_xorwf, sign, 0, pic_opdst_f);

    label_release(lbl);
  }

  if (sdivide && divide) {
    /* sdivide needs to jump over a statement here */
    lbl = pfile_label_alloc(pf, 0);
    pic_instr_append_n(pf, pic_opcode_goto, lbl);
    pic_instr_append_label(pf, divide);
    pic_instr_append_f(pf, pic_opcode_clrf, sign, 0);
    pic_instr_append_label(pf, lbl);
    label_release(lbl);
  } else {
    pic_instr_append_label(pf, divide);
  }
  /* # bits --> loop */
  pic_instr_append_w_kn(pf, pic_opcode_movlw, 8 * value_sz_get(dvars.quotient));
  pic_instr_append_f(pf, pic_opcode_movwf, loop, 0);
  pic_op(pf, operator_assign, dvars.remainder, c_0, 0);
  pic_instr_append_label(pf, top);
  /* quotient <<= 1 */
  pic_op(pf, operator_shift_left, dvars.quotient, c_1, 0);
  /* {dividend:remainder} <<= 1 */
  pic_op(pf, operator_shift_left, dvars.divaccum, c_1, 0);
  /* remainder -= divisor */
  pic_op(pf, operator_sub, dvars.remainder, dvars.divisor, 0);
  /* if divisor > remainder, add it back */
  pic_instr_append_reg_flag(pf, pic_opcode_btfsc, "_status", "_c");
  pic_instr_append_n(pf, pic_opcode_goto, skip);
  pic_op(pf, operator_add, dvars.remainder, dvars.divisor, 0);
  /* status<c> is guarenteed to be set here, so this simply skips the next
    instruction */
  pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_c");
  pic_instr_append_label(pf, skip);
  pic_instr_append_f_bn(pf, pic_opcode_bsf, dvars.quotient, 0, 0);
  pic_instr_append_f_d(pf, pic_opcode_decfsz, loop, 0, pic_opdst_f);
  pic_instr_append_n(pf, pic_opcode_goto, top);

  if (sdivide) {
    /* what to do with the result */
    pic_instr_append_f_d(pf, pic_opcode_movf, sign, 0, pic_opdst_w);
    pic_instr_append_reg_flag(pf, pic_opcode_btfsc, "_status", "_z");
    pic_instr_append(pf, pic_opcode_ret);
    /* invert the result */
    pic_op(pf, operator_neg, dvars.quotient, 0, 0);
    pic_op(pf, operator_neg, dvars.remainder, 0, 0);
  }
  pic_instr_append(pf, pic_opcode_ret);

  pic_var_sign_release(pf, sign);
  label_release(sdivide);
  label_release(divide);
  label_release(skip);
  label_release(top);
  pic_var_loop_release(pf, loop);
  value_release(c_1);
  value_release(c_0);
  pic_var_div_release(pf, &dvars);
}

/*
 * NAME
 *    pic_divide_create
 *
 * DESCRIPTION
 *    create a signed divide
 *
 * PARAMETERS
 *    pf
 *
 * RETURN
 *
 * NOTES
 *   on entry:
 *   algorithm:
 *     if dividend & divisor are not the same sign, the result is (-)
 *     dividend = abs(dividend)
 *     divisor = abs(divisor)
 *     call divide (above)
 *     if (-) then quotient = -quotient
 */
static void pic_divide_create(pfile_t *pf)
{
  label_t lbl;

  lbl = pfile_label_find(pf, pfile_log_none, PIC_LABEL_SDIVIDE);
  if (lbl) {
    label_release(lbl);
  } else {
    pic_sdivide_create(pf);
  }
}

/*
 * NAME
 *   pic_memcmp_create
 *
 * DESCRIPTION
 *   create memory compare
 *
 * PARAMETERS
 *
 * RETURN
 *
 * NOTES
 *   _pic_temp[0,1] = &a
 *   _pic_temp[2,3] = &b
 *   w holds the length; 0 = 256
 *   uses _sign and _loop
 *   Z set if equal
 *   C set if a < b
 */
static void pic_memcmp_create(pfile_t *pf)
{
  label_t top;
  value_t val1;
  value_t val2;
  value_t loop;
  value_t sign; /* temporary holder */

  top  = pfile_label_alloc(pf, 0);
  val1 = pic_var_temp_get(pf, VARIABLE_DEF_FLAG_NONE, 2);
  val2 = pic_var_temp_get(pf, VARIABLE_DEF_FLAG_NONE, 2);
  sign = pic_var_sign_get(pf);
  loop = pic_var_loop_get(pf);

  pic_instr_append_f(pf, pic_opcode_movwf, loop, 0);
  pic_instr_append_label(pf, top);
  /* w = *pic_memcmp_a */
  pic_instr_append_f_d(pf, pic_opcode_movf, val1, 0, pic_opdst_w);
  pic_instr_append_reg(pf, pic_opcode_movwf, "_fsr");
  pic_instr_append_reg_flag(pf, pic_opcode_bcf, "_status", "_irp");
  pic_instr_append_f_d(pf, pic_opcode_movf, val1, 1, pic_opdst_f);
  pic_instr_append_reg_flag(pf, pic_opcode_btfsc, "_status", "_z");
  pic_instr_append_reg_flag(pf, pic_opcode_bsf, "_status", "_irp");
  pic_instr_append_reg_d(pf, pic_opcode_movf, "_ind", pic_opdst_w);
  pic_instr_append_f(pf, pic_opcode_movwf, sign, 0);
  /* w = *pic_memcmp_b  - w */
  pic_instr_append_f_d(pf, pic_opcode_movf, val2, 0, pic_opdst_w);
  pic_instr_append_reg(pf, pic_opcode_movwf, "_fsr");
  pic_instr_append_reg_flag(pf, pic_opcode_bcf, "_status", "_irp");
  pic_instr_append_f_d(pf, pic_opcode_movf, val2, 1, pic_opdst_f);
  pic_instr_append_reg_flag(pf, pic_opcode_btfsc, "_status", "_z");
  pic_instr_append_reg_flag(pf, pic_opcode_bsf, "_status", "_irp");
  pic_instr_append_reg_d(pf, pic_opcode_movf, "_ind", pic_opdst_w);
  pic_instr_append_f_d(pf, pic_opcode_subwf, sign, 0, pic_opdst_w);
  /* test _sign */
  pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_z");
  pic_instr_append(pf, pic_opcode_ret);
  pic_instr_append_f_d(pf, pic_opcode_incf, val1, 0, pic_opdst_f);
  pic_instr_append_f_d(pf, pic_opcode_incf, val2, 0, pic_opdst_f);
  pic_instr_append_f(pf, pic_opcode_decfsz, loop, 0);
  pic_instr_append_n(pf, pic_opcode_goto, top);
  pic_instr_append(pf, pic_opcode_ret);

  pic_var_loop_release(pf, loop);
  pic_var_sign_release(pf, sign);
  pic_var_temp_release(pf, val2);
  pic_var_temp_release(pf, val1);
  label_release(top);
}

/*
 * NAME
 *   pic_memset_create
 *
 * DESCRIPTION
 *   create the memset function
 *
 * PARAMETERS
 *   
 * RETURN
 *
 * NOTES
 *   _fsr:_irp = &dst
 *   _pic_loop = length
 *   w         = value
 */
static void pic_memset_create(pfile_t *pf)
{
  label_t top;
  value_t loop;

  top  = pfile_label_alloc(pf, 0);
  loop = pic_var_loop_get(pf);

  pic_instr_append_label(pf, top);
  pic_instr_append_reg(pf, pic_opcode_movwf, "_ind");
  pic_instr_append_reg_d(pf, pic_opcode_incf, "_fsr", pic_opdst_f);
  pic_instr_append_f_d(pf, pic_opcode_decfsz, loop, 0, pic_opdst_f);
  pic_instr_append_n(pf, pic_opcode_goto, top);
  pic_instr_append(pf, pic_opcode_ret);

  pic_var_loop_release(pf, loop);
  label_release(top);
}

/*
 * NAME
 *   pic_memcpy_create
 *
 * DESCRIPTION
 *   create the mem copy
 *
 * PARAMETERS
 *
 * RETURN
 *
 * NOTES
 *   _pic_memcpy_src = src
 *   _pic_memcpy_dst = dst
 *   w = length (0 = 256)
 */
static void pic_memcpy_create(pfile_t *pf)
{
  label_t top;
  value_t params[2];
  value_t loop;
  value_t sign; /* temporary holder */

  pic_memcpy_params_get(pf, params);
  sign = pic_var_sign_get(pf);
  loop = pic_var_loop_get(pf);
  top  = pfile_label_alloc(pf, 0);

  pic_instr_append_f(pf, pic_opcode_movwf, loop, 0);
  pic_instr_append_label(pf, top);
  /* sign = *src */
  pic_instr_append_f_d(pf, pic_opcode_movf, params[0], 0, pic_opdst_w);
  pic_instr_append_reg(pf, pic_opcode_movwf, "_fsr");
  pic_instr_append_reg_flag(pf, pic_opcode_bcf, "_status", "_irp");
  pic_instr_append_f_d(pf, pic_opcode_movf, params[0], 1, pic_opdst_f);
  pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_z");
  pic_instr_append_reg_flag(pf, pic_opcode_bsf, "_status", "_irp");
  pic_instr_append_reg_d(pf, pic_opcode_movf, "_ind", pic_opdst_w);
  pic_instr_append_f(pf, pic_opcode_movwf, sign, 0);
  pic_instr_append_f_d(pf, pic_opcode_incf, params[0], 0, pic_opdst_f);
  /* *dst = sign */
  pic_instr_append_f_d(pf, pic_opcode_movf, params[1], 0, pic_opdst_w);
  pic_instr_append_reg(pf, pic_opcode_movwf, "_fsr");
  pic_instr_append_reg_flag(pf, pic_opcode_bcf, "_status", "_irp");
  pic_instr_append_f_d(pf, pic_opcode_movf, params[1], 1, pic_opdst_f);
  pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_z");
  pic_instr_append_reg_flag(pf, pic_opcode_bsf, "_status", "_irp");
  pic_instr_append_f_d(pf, pic_opcode_movf, sign, 0, pic_opdst_w);
  pic_instr_append_reg(pf, pic_opcode_movwf, "_ind");
  /* test _sign */
  pic_instr_append_f_d(pf, pic_opcode_incf, params[1], 0, pic_opdst_f);
  pic_instr_append_f_d(pf, pic_opcode_decfsz, loop, 0, pic_opdst_f);
  pic_instr_append_n(pf, pic_opcode_goto, top);
  pic_instr_append(pf, pic_opcode_ret);

  pic_var_loop_release(pf, loop);
  pic_var_sign_release(pf, sign);
  pic_memcpy_params_release(pf, params);
  label_release(top);
}

/*
 * NAME
 *   pic_stkpush_create
 *
 * DESCRIPTION
 *   create the stack push function
 *
 * PARAMETERS
 *
 * RETURN
 *
 * NOTES
 *   entry
 *     _pic_memcpy_src = src
 *     w               = size
 *   function
 *     stkptr -= w
 *     _pic_memcpy_dst = stkptr
 *     goto memcpy
 */
static void pic_stkpush_create(pfile_t *pf)
{
  value_t memcpy_params[2];

  value_t stkptr; /* stack pointer      */
  value_t loop;   /* temp storage for w */
  label_t pic_memcpy;

  pic_memcpy_params_get(pf, memcpy_params);
  stkptr     = pic_var_stkptr_get(pf);
  loop       = pic_var_loop_get(pf);
  pic_memcpy = pic_label_find(pf, PIC_LABEL_MEMCPY, boolean_true);

  pic_instr_append_f(pf, pic_opcode_movwf, loop, 0);
  pic_instr_append_f_d(pf, pic_opcode_subwf, stkptr, 0, pic_opdst_f);
  pic_op(pf, operator_assign, memcpy_params[1], stkptr, VALUE_NONE);
  pic_instr_append_f_d(pf, pic_opcode_movf, loop, 0, pic_opdst_w);
  pic_instr_append_n(pf, pic_opcode_goto, pic_memcpy);
  label_release(pic_memcpy);
  pic_var_loop_release(pf, loop);
  pic_var_stkptr_release(pf, stkptr);
  pic_memcpy_params_release(pf, memcpy_params);
}

/*
 * NAME
 *   pic_stkpop_create
 *
 * DESCRIPTION
 *   create the stack pop function
 *
 * PARAMETERS
 *
 * RETURN
 *
 * NOTES
 *   entry
 *     pic_memcpy_dst = dst
 *     w              = size
 *   function
 *     pic_memcpy_src = stkptr
 *     call memcpy
 *     stkptr += w
 *   nb: stkptr must be incremented *after* the memcpy to protect against
 *       interruption
 */
static void pic_stkpop_create(pfile_t *pf)
{
  value_t memcpy_params[2];
  value_t loop_tmp;
  value_t stkptr; /* stack pointer      */
  label_t pic_memcpy;

  pic_memcpy_params_get(pf, memcpy_params);
  loop_tmp   = pic_var_loop_tmp_get(pf);
  stkptr     = pic_var_stkptr_get(pf);
  pic_memcpy = pic_label_find(pf, PIC_LABEL_MEMCPY, boolean_true);

  pic_instr_append_f(pf, pic_opcode_movwf, loop_tmp, 0);
  pic_op(pf, operator_assign, memcpy_params[0], stkptr, VALUE_NONE);
  pic_instr_append_f_d(pf, pic_opcode_movf, loop_tmp, 0, pic_opdst_w);
  pic_instr_append_n(pf, pic_opcode_call, pic_memcpy);
  pic_instr_append_f_d(pf, pic_opcode_movf, loop_tmp, 0, pic_opdst_w);
  pic_instr_append_f_d(pf, pic_opcode_addwf, stkptr, 0, pic_opdst_f);
  pic_instr_append(pf, pic_opcode_ret);
  label_release(pic_memcpy);
  pic_var_stkptr_release(pf, stkptr);
  pic_var_loop_tmp_release(pf, loop_tmp);
  pic_memcpy_params_release(pf, memcpy_params);
}

/*
 * NAME
 *   pic_indirect_create
 *
 * DESCRIPTION
 *   create the indirect call routine  
 *
 * PARAMETERS
 *
 * RETURN
 *
 * NOTES
 *   entry:
 *     _pic_sign holds the LSB
 *     W         holds the MSB
 *   This simply puts W into _pcl which executes a jump to _pclath:_pcl
 *   It needs to be its own function because the return value needs
 *   to be put onto the stack.
 */
static void pic_indirect_create(pfile_t *pf)
{
  value_t lsb;

  lsb = pic_var_sign_get(pf);

  pic_instr_append_reg(pf, pic_opcode_movwf, "_pclath");
  pic_instr_append_f_d(pf, pic_opcode_movf, lsb, 0, pic_opdst_w);
  pic_instr_append_reg(pf, pic_opcode_movwf, "_pcl");
  pic_var_sign_release(pf, lsb);
}

/*
 * NAME
 *   pic_task_start_create
 *
 * DESCRIPTION
 *   create the task start procedure
 *
 * PARAMETERS
 *   _pic_sign holds the MSB
 *   W         holds the LSB
 *
 * RETURN
 *   none
 *
 * NOTES
 *   basically:
 *   LOOP task_ct USING task_idx LOOP
 *     IF 0 == task_list[task_idx] THEN
 *       task_list[task_idx] = entry
 *       RETURN
 *     END IF
 *   END_LOOP
 */
static void pic_task_start_create(pfile_t *pf)
{
  value_t       idx;
  value_t       task_list;
  value_t       task_lsb;
  value_t       task_msb;
  label_t       lbl_top;
  label_t       lbl_found;
  variable_ct_t task_ct;


  task_ct   = pfile_task_ct_get(pf);
  if (task_ct < 2) {
    pfile_log(pf, pfile_log_err, "task count must be >= 2!");
  }

  task_list = pfile_value_find(pf, pfile_log_none, "_task_list");
  if (VALUE_NONE == task_list) {
    variable_def_t def;
    variable_def_t vdef;
    value_t        task_active;
    /* task_list is an array of 2-byte integers! */
    vdef = variable_def_alloc(0, variable_def_type_integer,
        VARIABLE_DEF_FLAG_NONE, 2);
    def  = variable_def_alloc(0, variable_def_type_array,
        VARIABLE_DEF_FLAG_NONE, 0);
    variable_def_member_add(def, 0, vdef, task_ct);

    task_list = VALUE_NONE;
    pfile_value_alloc(pf, PFILE_VARIABLE_ALLOC_GLOBAL, "_task_list", 
        def, &task_list);
    variable_flag_set(value_variable_get(task_list), VARIABLE_FLAG_AUTO);
#if 0 /* KY: only bump with actual instructions */
    value_use_ct_bump(task_list, ctr_bump_incr);
#endif
    /* also, allocate _task_active */

    def = variable_def_alloc(0, variable_def_type_integer,
        VARIABLE_DEF_FLAG_NONE, 1);
    task_active = VALUE_NONE;
    pfile_value_alloc(pf, PFILE_VARIABLE_ALLOC_GLOBAL, "_task_active",
        def, &task_active);
    variable_flag_set(value_variable_get(task_active), VARIABLE_FLAG_AUTO);
    value_release(task_active);
  }


  idx       = pic_var_loop_get(pf);
  task_lsb  = pic_var_loop_tmp_get(pf);
  task_msb  = pic_var_sign_get(pf);

  pic_instr_append_f(pf, pic_opcode_movwf, task_lsb, 0);
  pic_indirect_setup(pf, task_list, 2 * task_ct - 1);
  pic_instr_append_w_kn(pf, pic_opcode_movlw, task_ct);
  pic_instr_append_f(pf, pic_opcode_movwf, idx, 0);
  lbl_top = pfile_label_alloc(pf, 0);
  lbl_found = pfile_label_alloc(pf, 0);
  pic_instr_append_label(pf, lbl_top);
  pic_instr_append_reg_d(pf, pic_opcode_movf, "_ind", pic_opdst_w);
  pic_instr_append_reg_d(pf, pic_opcode_decf, "_fsr", pic_opdst_f);
  pic_instr_append_reg_d(pf, pic_opcode_iorwf, "_ind", pic_opdst_w);
  pic_instr_append_reg_flag(pf, pic_opcode_btfsc, "_status", "_z");
  pic_instr_append_n(pf, pic_opcode_goto, lbl_found);
  pic_instr_append_reg_d(pf, pic_opcode_decf, "_fsr", pic_opdst_f);
  pic_instr_append_f_d(pf, pic_opcode_decfsz, idx, 0, pic_opdst_f);
  pic_instr_append_n(pf, pic_opcode_goto, lbl_top);
  pic_instr_append(pf, pic_opcode_ret);
  pic_instr_append_label(pf, lbl_found);
  pic_instr_append_f_d(pf, pic_opcode_movf, task_lsb, 0, pic_opdst_w);
  pic_instr_append_reg(pf, pic_opcode_movwf, "_ind");
  pic_instr_append_reg_d(pf, pic_opcode_incf, "_fsr", pic_opdst_f);
  pic_instr_append_f_d(pf, pic_opcode_movf, task_msb, 0, pic_opdst_w);
  pic_instr_append_reg(pf, pic_opcode_movwf, "_ind");
  pic_instr_append(pf, pic_opcode_ret);

  label_release(lbl_found);
  label_release(lbl_top);

  pic_var_sign_release(pf, task_msb);
  pic_var_loop_tmp_release(pf, task_lsb);
  pic_var_loop_release(pf, idx);
  value_release(task_list);
}


/*
 * NAME
 *   pic_task_kill_create
 *
 * DESCRIPTION
 *   create the pic_task_kill function
 *
 * PARAMETERS
 *   pf : pfile handle
 *
 * RETURN
 *   none
 *
 * NOTES
 *   on entry, W holds the task ID. 
 *   basically:
 *     _task_active = W
 *     _task_list[_task_active] = 0
 *     suspend
 */
static void pic_task_kill_create(pfile_t *pf)
{
  value_t task_active;

  task_active = pfile_value_find(pf, pfile_log_err, "_task_active");
  pic_op(pf, operator_assign, task_active, VALUE_NONE, VALUE_NONE);
  value_release(task_active);
  label_release(pic_label_find(pf, PIC_LABEL_TASK_SUICIDE, boolean_true));
}

static void pic_task_suicide_create(pfile_t *pf)
{
  value_t task_msb;

  task_msb = pic_var_sign_get(pf);
  pic_instr_append_f(pf, pic_opcode_clrf, task_msb, 0);
  pic_instr_append(pf, pic_opcode_clrw);
  /* fall through to suspend! */
  value_release(task_msb);
  label_release(pic_label_find(pf, PIC_LABEL_TASK_SUSPEND, boolean_true));
}

/*
 * NAME
 *   pic_task_suspend_create
 *
 * DESCRIPTION
 *   suspend the current task
 *
 * PARAMETERS
 *   _pic_sign holds the MSB of the next instruction for this task
 *   W         holds the LSB of the next instruction for this task
 *
 * RETURN
 *   none
 *
 * NOTES
 *   basically:
 *      task_list[task_idx].lsb = lsb
 *      task_list[task_idx].msb = msb
 *      FOREVER LOOP
 *        IF (0 == task_idx) THEN
 *          task_idx = task_ct
 *        END IF
 *        task_idx--
 *        IF (task_list[task_idx].lsb | task_list[task_idx].msb) THEN
 *          GOTO task_list[task_idx]
 *        END IF
 *      END LOOP

 *   LOOP task_ct USING task_idx LOOP
 *     IF 0 == task_list[task_idx] THEN
 *       task_list[task_idx] = entry
 *       RETURN
 *     END IF
 *   END_LOOP
 *
 * Note2: to kill a task, suspend with the lsb/msb of 0
 */
static void pic_task_suspend_create(pfile_t *pf)
{
  value_t  idx;
  value_t  task_list;
  value_t  task_lsb;
  value_t  task_msb;
  unsigned task_ct;
  label_t lbl_top;
  label_t lbl_skip;
  value_t task_idx;

  task_ct   = pfile_task_ct_get(pf);
  task_list = pfile_value_find(pf, pfile_log_err, "_task_list");
  task_idx  = pfile_value_find(pf, pfile_log_err, "_task_active");
  idx       = pic_var_loop_get(pf);
  task_lsb  = pic_var_loop_tmp_get(pf);
  task_msb  = pic_var_sign_get(pf);
  lbl_top   = pfile_label_alloc(pf, 0);
  lbl_skip  = pfile_label_alloc(pf, 0);

  pic_instr_append_f(pf, pic_opcode_movwf, task_lsb, 0);
  pic_indirect_setup(pf, task_list, 0);

  /* store the return address */
  pic_instr_append_f_d(pf, pic_opcode_movf, task_idx, 0, pic_opdst_w);
  pic_instr_append_reg_d(pf, pic_opcode_addwf, "_fsr", pic_opdst_f);
  pic_instr_append_reg_d(pf, pic_opcode_addwf, "_fsr", pic_opdst_f);
  pic_instr_append_f_d(pf, pic_opcode_movf, task_lsb, 0, pic_opdst_w);
  pic_instr_append_reg(pf, pic_opcode_movwf, "_ind");
  pic_instr_append_reg_d(pf, pic_opcode_incf, "_fsr", pic_opdst_f);
  pic_instr_append_f_d(pf, pic_opcode_movf, task_msb, 0, pic_opdst_w);
  pic_instr_append_reg(pf, pic_opcode_movwf, "_ind");
  pic_instr_append_reg_d(pf, pic_opcode_decf, "_fsr", pic_opdst_f);
  /* find an active task */
  pic_instr_append_label(pf, lbl_top);
  pic_instr_append_f_d(pf, pic_opcode_movf, task_idx, 0, pic_opdst_f);
  pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_z");
  pic_instr_append_n(pf, pic_opcode_goto, lbl_skip);
  pic_instr_append_w_kn(pf, pic_opcode_movlw, task_ct);
  pic_instr_append_f(pf, pic_opcode_movwf, task_idx, 0);
  pic_instr_append_reg_d(pf, pic_opcode_addwf, "_fsr", pic_opdst_f);
  pic_instr_append_reg_d(pf, pic_opcode_addwf, "_fsr", pic_opdst_f);
  pic_instr_append_label(pf, lbl_skip);
  pic_instr_append_f_d(pf, pic_opcode_decf, task_idx, 0, pic_opdst_f);
  pic_instr_append_reg_d(pf, pic_opcode_decf, "_fsr", pic_opdst_f);
  pic_instr_append_reg_d(pf, pic_opcode_movf, "_ind", pic_opdst_w);
  pic_instr_append_reg_d(pf, pic_opcode_decf, "_fsr", pic_opdst_f);
  pic_instr_append_reg_d(pf, pic_opcode_iorwf, "_ind", pic_opdst_w);
  pic_instr_append_reg_flag(pf, pic_opcode_btfsc, "_status", "_z");
  pic_instr_append_n(pf, pic_opcode_goto, lbl_top);
  /* jump to the active task */
  pic_instr_append_reg_d(pf, pic_opcode_incf, "_fsr", pic_opdst_f);
  pic_instr_append_reg_d(pf, pic_opcode_movf, "_ind", pic_opdst_w);
  pic_instr_append_reg(pf, pic_opcode_movwf, "_pclath");
  pic_instr_append_reg_d(pf, pic_opcode_decf, "_fsr", pic_opdst_f);
  pic_instr_append_reg_d(pf, pic_opcode_movf, "_ind", pic_opdst_w);
  pic_instr_append_reg(pf, pic_opcode_movwf, "_pcl");



  label_release(lbl_skip);
  label_release(lbl_top);

  pic_var_sign_release(pf, task_msb);
  pic_var_loop_tmp_release(pf, task_lsb);
  pic_var_loop_release(pf, idx);
  value_release(task_idx);
  value_release(task_list);
}

/*
 * NAME
 *   pic_intrinsics_create 
 *
 * DESCRIPTION
 *   create the intrinsic
 *
 * PARAMETERS
 *   pf : pfile handle
 *
 * RETURN
 *   none
 *
 * NOTES
 *   the functions are created if any references exist
 *   make sure to update the static table for any other intrinsice
 */
/* nb: this table is somewhat order dependent. if an intrinsic
       relies on another one, it should come *before* the other
       one to guarentee both are generated */
static const struct {
  const char *tag;
  void (*fn)(pfile_t *);
} intrinsics[] = {
  {PIC_LABEL_STKPUSH,      pic_stkpush_create},
  {PIC_LABEL_STKPOP,       pic_stkpop_create},
  {PIC_LABEL_MULTIPLY,     pic_multiply_create},
  {PIC_LABEL_SDIVIDE,      pic_sdivide_create},
  {PIC_LABEL_DIVIDE,       pic_divide_create},
  {PIC_LABEL_MEMSET,       pic_memset_create},
  {PIC_LABEL_MEMCPY,       pic_memcpy_create},
  {PIC_LABEL_MEMCMP,       pic_memcmp_create},
  {PIC_LABEL_INDIRECT,     pic_indirect_create},
  {PIC_LABEL_TASK_START,   pic_task_start_create},
  /* KILL/SUICIDE/SUSPEND must come in this order! */
  {PIC_LABEL_TASK_KILL,    pic_task_kill_create},
  {PIC_LABEL_TASK_SUICIDE, pic_task_suicide_create},
  {PIC_LABEL_TASK_SUSPEND, pic_task_suspend_create}
};

void pic_intrinsics_create(pfile_t *pf)
{
  size_t  ii;

  for (ii = 0; ii < COUNT(intrinsics); ii++) {
    label_t lbl;

    lbl = pfile_label_find(pf, pfile_log_none, intrinsics[ii].tag);
    if (lbl) {
      if (pic_divide_create != intrinsics[ii].fn) {
        pic_instr_append_label(pf, lbl);
      }
      label_release(lbl);
      intrinsics[ii].fn(pf);
    }
  }
}

boolean_t pic_intrinsics_exist(pfile_t *pf)
{
  label_t lbl;
  size_t  ii;

  for (ii = 0, lbl = LABEL_NONE; !lbl && (ii < COUNT(intrinsics)); ii++) {
    lbl = pfile_label_find(pf, pfile_log_none, intrinsics[ii].tag);
    if (lbl) {
      label_release(lbl);
    }
  }
  return (LABEL_NONE != lbl);
}


