/************************************************************
 **
 ** pic_stvar.c : pic state variable maintainence
 **
 ** Copyright (c) 2005, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/

/*
 * The PIC state variables are as follows:
 *   _pic_state -- this contains the values needed for temporary
 *                 storage, multiplication, division, intrinsic
 *                 parameters and indirect parameters
 *                   struct _pic_state_ {
 *                     loop      : used for internal looping (shift, mul, div)
 *                     sign      : used for sign extension
 *                     stack_ptr : holds the stack pointer for recursion
 *                     union {
 *                       temp : used when an operation occurs over two
 *                              arrays, or to hold the parameters for
 *                              intrinsics (memset, memcpy, memcmp, ...)
 *                              or for indirect parameter passing
 *                       struct {
 *                         multiplier
 *                         multiplicand
 *                         mresult
 *                       } mul;
 *                       struct {
 *                         dividend
 *                         divisor
 *                         quotient
 *                         remainder
 *                       } div;
 *  _pic_istate -- holds a copy of pic_state while in an interrupt
 *                 currently holds a *complete* copy, but in the future
 *                 should only holds as much as is needed
 *  _pic_isr_w      -- must be shared or allocated in all banks
 *  _pic_isr_status -- must be allocated in the first bank
 *  _pic_isr_pclath -- must be allocated in the first bank
 */
#include <assert.h>
#include "../libcore/pf_proc.h"
#include "pic_var.h"
#include "pic_stvar.h"

#define PIC_STVAR_FLAG_ACCUM    0x0001
#define PIC_STVAR_FLAG_LOOP     0x0002
#define PIC_STVAR_FLAG_LOOP_TMP 0x0004
#define PIC_STVAR_FLAG_SIGN     0x0008

static unsigned stvar_flags; /* make sure two routines don't both
                                grab the same variable! */

static variable_t pic_var_state_var_get(pfile_t *pf)
{
  variable_t var;

  var = pfile_variable_find(pf, pfile_log_none, "_pic_state", 0);
  if (!var) {
    pfile_variable_alloc(pf, PFILE_VARIABLE_ALLOC_GLOBAL,
      "_pic_state", VARIABLE_DEF_NONE, VARIABLE_NONE, &var);
    variable_flag_set(var, VARIABLE_FLAG_AUTO);
  }
  return var;
}

/*
 * NAME
 *   pic_var_value_get
 *
 * DESCRIPTION
 *   get the named value; allocate if necessary
 *
 * PARAMETERS
 *   pf   : pfile handle
 *   name :
 *   type :
 *   sz   :
 *
 * RETURN
 *   value
 *
 * NOTES
 *   the value's size can be reset here; the act of getting a value
 *   bumps the assign & usage count so it will be allocated later!
 */
static value_t pic_var_value_get(pfile_t *pf, const char *name,
    variable_def_type_t type, variable_sz_t sz, boolean_t set_master)
{
  value_t val;

  val = pfile_value_find(pf, pfile_log_none, name);
  if (!val && ((variable_sz_t) -1 != sz)) {
    variable_t     var;
    variable_t     master;
    variable_def_t def;

    master = (set_master) ? pic_var_state_var_get(pf) : VARIABLE_NONE;
    def    = variable_def_alloc(0, type, VARIABLE_DEF_FLAG_NONE, sz);
    pfile_variable_alloc(pf, PFILE_VARIABLE_ALLOC_GLOBAL, name, def, master, 
        &var);
    val = value_alloc(var);
    variable_release(master);
    variable_release(var);
  }
  if (val) {
    /*assert(value_type_get(val) == type);*/
    if ((value_type_get(val) != type)
      || ((((variable_sz_t) -1) != sz) && (value_sz_get(val) != sz))) {
      variable_def_t def;

      def = variable_def_alloc(0, type, VARIABLE_DEF_FLAG_NONE, sz);
      variable_def_set(value_variable_get(val), def);
      value_def_set(val, def);
    }
  }
  return val;
}

value_t pic_var_loop_get(pfile_t *pf)
{
  assert(!(stvar_flags & PIC_STVAR_FLAG_LOOP));
  stvar_flags |= PIC_STVAR_FLAG_LOOP;
  return pic_var_value_get(pf, "_pic_loop", variable_def_type_integer, 1, 
    boolean_true);
}

void pic_var_loop_release(pfile_t *pf, value_t val)
{
  assert(stvar_flags & PIC_STVAR_FLAG_LOOP);
  stvar_flags &= ~PIC_STVAR_FLAG_LOOP;
  value_release(val);
}

value_t pic_var_loop_tmp_get(pfile_t *pf)
{
  assert(!(stvar_flags & PIC_STVAR_FLAG_LOOP_TMP));
  stvar_flags |= PIC_STVAR_FLAG_LOOP_TMP;
  return pic_var_value_get(pf, "_pic_loop_tmp", variable_def_type_integer, 1, 
      boolean_true);
}

void pic_var_loop_tmp_release(pfile_t *pf, value_t val)
{
  assert(stvar_flags & PIC_STVAR_FLAG_LOOP_TMP);
  stvar_flags &= ~PIC_STVAR_FLAG_LOOP_TMP;
  value_release(val);
}

value_t pic_var_sign_get(pfile_t *pf)
{
  assert(!(stvar_flags & PIC_STVAR_FLAG_SIGN));
  stvar_flags |= PIC_STVAR_FLAG_SIGN;
  return pic_var_value_get(pf, "_pic_sign", variable_def_type_integer, 1,
    boolean_true);
}

void pic_var_sign_release(pfile_t *pf, value_t val)
{
  assert(stvar_flags & PIC_STVAR_FLAG_SIGN);
  stvar_flags &= ~PIC_STVAR_FLAG_SIGN;
  value_release(val);
}

value_t pic_var_accum_get(pfile_t *pf)
{
  assert(!(stvar_flags & PIC_STVAR_FLAG_ACCUM));
  stvar_flags |= PIC_STVAR_FLAG_ACCUM;
  return pfile_value_find(pf, pfile_log_err, "_pic_accum");
}

void pic_var_accum_release(pfile_t *pf, value_t val)
{
  assert(stvar_flags & PIC_STVAR_FLAG_ACCUM);
  stvar_flags &= ~PIC_STVAR_FLAG_ACCUM;
  value_release(val);
}

boolean_t pic_memcpy_params_get(pfile_t *pf, value_t *params)
{
  params[0] = pic_var_value_get(pf, "_pic_memcpy_src", 
      variable_def_type_integer, 2, boolean_true);
  params[1] = pic_var_value_get(pf, "_pic_memcpy_dst",
      variable_def_type_integer, 2, boolean_true);
  return boolean_true;
}

void pic_memcpy_params_release(pfile_t *pf, value_t *params)
{
  value_release(params[1]);
  value_release(params[0]);
}

value_t pic_var_stkptr_get(pfile_t *pf)
{
  value_t val;

  val = pfile_value_find(pf, pfile_log_none, "_pic_stkptr");
  if (!val) {
    variable_def_t def;

    def = variable_def_alloc(0, variable_def_type_integer, 
        VARIABLE_DEF_FLAG_NONE, 2);
    pfile_value_alloc(pf, 0, "_pic_stkptr", def, &val);
    variable_flag_set(value_variable_get(val), VARIABLE_FLAG_AUTO);
  }
  return val;
}

void pic_var_stkptr_release(pfile_t *pf, value_t stkptr)
{
  value_release(stkptr);
}

variable_sz_t pic_temp_sz_in_use;
variable_sz_t pic_temp_sz_in_use;
value_t pic_var_temp_get(pfile_t *pf, flag_t flags, variable_sz_t sz)
{
  value_t        val;

  if (0 == sz) {
    val = VALUE_NONE;
  } else {
    variable_t     var;
    variable_def_t def;

    val = pfile_value_find(pf, pfile_log_none, "_pic_temp");
    if (!val) {
      val = pic_var_value_get(pf, "_pic_temp", variable_def_type_integer, sz, 
          boolean_true);
    }
    if (pic_temp_sz_in_use) {
      value_t tmp;

      tmp = pfile_constant_get(pf, pic_temp_sz_in_use, VARIABLE_DEF_NONE);
      value_baseofs_set(val, tmp);
      value_release(tmp);
    }
    pic_temp_sz_in_use += sz;
    var = value_variable_get(val);
    if (variable_sz_get(var) < pic_temp_sz_in_use) {
      def = variable_def_alloc(0, variable_def_type_integer, 
          VARIABLE_DEF_FLAG_NONE, pic_temp_sz_in_use);
      variable_def_set(var, def);
    }
    def = variable_def_alloc(0, variable_def_type_integer, 
        flags & VARIABLE_DEF_FLAG_SIGNED, sz);
    value_def_set(val, def);
  }
  return val;
}

value_t pic_var_temp_get_def(pfile_t *pf, variable_def_t def)
{
  return pic_var_temp_get(pf, variable_def_flags_get_all(def), 
      variable_def_byte_sz_get(def));
}

void pic_var_temp_release(pfile_t *pf, value_t tmp)
{
  variable_const_t n;
  value_t          baseofs;

  pic_temp_sz_in_use -= value_sz_get(tmp);
  baseofs = value_baseofs_get(tmp);
  n       = (baseofs) ? value_const_get(baseofs) : 0;
  assert(pic_temp_sz_in_use == n);
  value_release(tmp);
}

variable_sz_t pic_var_mul_div_sz_get(pfile_t *pf, variable_sz_t req, 
    const char *name)
{
  if ((variable_sz_t) -1 != req) {
    variable_t var;

    var = pfile_variable_find(pf, pfile_log_none, name, 0);
    if (var) {
      if (variable_sz_get(var) != req) {
        if (req < variable_sz_get(var)) {
          printf("req=%u result=%u\n",
              req, variable_sz_get(var));
          req = variable_sz_get(var);
        } else {
          variable_def_t def;

          def = variable_def_alloc(0, variable_def_type_integer, 
              VARIABLE_DEF_FLAG_NONE, req);
          variable_def_set(var, def);
          printf("req=%u result=%u\n",
              req, variable_sz_get(var));
        }
      }
      variable_release(var);
    }
  }
  return req;
}

boolean_t pic_var_mul_get(pfile_t *pf, variable_sz_t sz, pic_var_mul_t *dst)
{
  sz = pic_var_mul_div_sz_get(pf, sz, "_pic_multiplier");
  pic_var_loop_release(pf, pic_var_loop_get(pf));
  dst->multiplier = pic_var_value_get(pf, "_pic_multiplier",
      variable_def_type_integer, sz, boolean_true);
  dst->multiplicand = pic_var_value_get(pf, "_pic_multiplicand",
      variable_def_type_integer, sz, boolean_true);
  dst->mresult = pic_var_value_get(pf, "_pic_mresult",
      variable_def_type_integer, sz, boolean_true);
  return (dst->multiplier != VALUE_NONE);
}

void pic_var_mul_release(pfile_t *pf, pic_var_mul_t *dst)
{
  value_release(dst->mresult);
  value_release(dst->multiplicand);
  value_release(dst->multiplier);
}

boolean_t pic_var_div_get(pfile_t *pf, variable_sz_t sz, pic_var_div_t *dst)
{
  sz = pic_var_mul_div_sz_get(pf, sz, "_pic_divisor");
  pic_var_loop_release(pf, pic_var_loop_get(pf));
  dst->divisor = pic_var_value_get(pf, "_pic_divisor",
      variable_def_type_integer, sz, boolean_true);
  dst->dividend = pic_var_value_get(pf, "_pic_dividend",
      variable_def_type_integer, sz, boolean_true);
  dst->quotient = pic_var_value_get(pf, "_pic_quotient",
      variable_def_type_integer, sz, boolean_true);
  dst->remainder = pic_var_value_get(pf, "_pic_remainder",
      variable_def_type_integer, sz, boolean_true);
  dst->divaccum = pic_var_value_get(pf, "_pic_divaccum",
      variable_def_type_integer, (((variable_sz_t) -1) == sz) ? sz : 2 * sz, 
      boolean_true);
  return (dst->divisor != VALUE_NONE);
}

void pic_var_div_release(pfile_t *pf, pic_var_div_t *dst)
{
  value_release(dst->divaccum);
  value_release(dst->remainder);
  value_release(dst->quotient);
  value_release(dst->dividend);
  value_release(dst->divisor);
}

boolean_t pic_var_isr_get(pfile_t *pf, boolean_t alloc, pic_var_isr_t *dst)
{
  variable_t isr_status;

  isr_status = pfile_variable_find(pf, pfile_log_none, "_pic_isr_status", 0);
  if (isr_status || alloc) {
    variable_t picstate;

    picstate = pic_var_state_var_get(pf);
    if (variable_sz_get(picstate)) {
      dst->isr_state = pic_var_value_get(pf, "_pic_isr_state",
        variable_def_type_integer, variable_sz_get(picstate), boolean_false);
    } else {
      dst->isr_state = VALUE_NONE;
    }
    variable_release(picstate);
    dst->w = pfile_value_find(pf, pfile_log_err, "_pic_isr_w");
    dst->status = pic_var_value_get(pf, "_pic_isr_status",
        variable_def_type_integer, 1, boolean_false);
    dst->pclath = pic_var_value_get(pf, "_pic_isr_pclath",
        variable_def_type_integer, 1, boolean_false);
    dst->pic_state = value_alloc(pic_var_state_var_get(pf));
  }
  variable_release(isr_status);
  return isr_status != VARIABLE_NONE;
}

void pic_var_isr_release(pfile_t *pf, pic_var_isr_t *dst)
{
  variable_release(value_variable_get(dst->pic_state));
  value_release(dst->pic_state);
  value_release(dst->isr_state);
  value_release(dst->pclath);
  value_release(dst->status);
  value_release(dst->w);
}


/*
 * NAME
 *   pic_stvar_fixup
 *
 * DESCRIPTION
 *   once all needed state variables have been accessed
 *   position each in its place in the _pic_state union
 *
 * PARAMETERS
 *   pf : pfile
 *
 * RETURN
 *   none
 *
 * NOTES
 */
void pic_stvar_fixup(pfile_t *pf)
{
  variable_sz_t pic_state_sz;
  variable_sz_t pic_state_pos[7];
  variable_t    var;
  /* this creates a union in the expected way. */
  static struct {
    unsigned    pos;
    const char *tag;
  } varinfo[] = {
    {0, "_pic_loop"},
    {1, "_pic_loop_tmp"},    /* used for stkpop */
    {2, "_pic_sign"},
    {3, "_pic_memcpy_src"},
    {4, "_pic_memcpy_dst"},
    {5, "_pic_temp"},
    {3, "_pic_divaccum"},
    {3, "_pic_dividend"},
    {4, "_pic_remainder"},
    {5, "_pic_divisor"},
    {6, "_pic_quotient"},
    {2, "_pic_multiplier"},
    {3, "_pic_multiplicand"},
    {4, "_pic_mresult"}
    /* {2, "_pic_stkptr"} */
  };
  size_t ii;

  pic_state_sz     = 0;
  pic_state_pos[0] = 0;
  for (ii = 0; ii < COUNT(varinfo); ii++) {
    variable_sz_t pos;

    var = pfile_variable_find(pf, pfile_log_none, varinfo[ii].tag, 0);
    pos = varinfo[ii].pos;
    if (pos) {
      pic_state_pos[pos] = pic_state_pos[pos-1];
    }
    if (var) {
      variable_base_set(var, pic_state_pos[pos], 0);
#if 0
      printf("%s:%u (%u)\n", variable_name_get(var),
          variable_sz_get(var), pic_state_pos[pos]);
#endif
      pic_state_pos[pos] += variable_sz_get(var);
      if (pic_state_pos[pos] > pic_state_sz) {
        pic_state_sz = pic_state_pos[pos];
      }
      variable_release(var);
    }
  }
  var = pfile_variable_find(pf, pfile_log_none, "_pic_state", 0);
  if (var) {
    variable_def_t def;
    pic_var_isr_t  isr_vars;

    if (0 == pic_state_sz) {
      pic_state_sz = 1;
    }
    def = variable_def_alloc(0, variable_def_type_integer,
          VARIABLE_DEF_FLAG_NONE, pic_state_sz);
    variable_def_set(var, def);
    pic_variable_alloc_one(pf, PFILE_PROC_NONE, var);
    variable_flag_clr(var, VARIABLE_FLAG_AUTO);
    variable_release(var);
    if (pic_var_isr_get(pf, boolean_false, &isr_vars)) {
      variable_t isr_state;

      isr_state = value_variable_get(isr_vars.isr_state);
      variable_def_set(isr_state, def);
      pic_variable_alloc_one(pf, PFILE_PROC_NONE, 
          value_variable_get(isr_vars.w));
      pic_variable_alloc_one(pf, PFILE_PROC_NONE, 
          value_variable_get(isr_vars.status));
      pic_variable_alloc_one(pf, PFILE_PROC_NONE, 
          value_variable_get(isr_vars.pclath));
      if (isr_vars.isr_state) {
        pic_variable_alloc_one(pf, PFILE_PROC_NONE, 
            value_variable_get(isr_vars.isr_state));
      }
      pic_var_isr_release(pf, &isr_vars);
    }
  }
}

