/************************************************************
 **
 ** testgen.c : relational test JAL code generator
 **
 ** Copyright (c) 2005, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

static void var_name(char *dst, size_t sz, size_t sign, size_t which)
{
  const char *type; 

  type = "";
  switch (sz) {
    case 0: type = "bit"; break;
    case 1: type = "byte"; break;
    case 2: type = "word"; break;
    case 3: type = "dword"; break;
  }
  sprintf(dst, "%s_%s%u",
      (sign) ? "u" : "s",
      type,
      which);
}

typedef enum test_type_ {
  TEST_TYPE_NONE,
  TEST_TYPE_VAR,
  TEST_TYPE_CONST,
  TEST_TYPE_VOLATILE
} test_type_t;

static FILE *output_open(test_type_t type, unsigned n)
{
  char buf[16];
  FILE *out;

  sprintf(buf, "test%04u.jal", n);
  out = fopen(buf, "w");
  if (out) {
    const char *vstr;
    const char *cstr;

    vstr = (TEST_TYPE_VOLATILE == type)
      ? "volatile " : "";
    cstr = (TEST_TYPE_CONST == type)
      ? "const" : "var";

    fprintf(out, 
        "include c16f877\n"
        "\n"
        "var %sbit    u_bit0,   u_bit1  =    0\n"
        "var %sbyte   u_byte0,  u_byte1 =   10\n"
        "var %ssbyte  s_byte0,  s_byte1 =  -10\n"
        "var %sword   u_word0,  u_word1 =   10\n"
        "var %ssword  s_word0,  s_word1 =  -10\n"
        "var %sdword  u_dword0, u_dword1 =  10\n"
        "var %ssdword s_dword0, s_dword1 = -10\n",
        vstr, vstr, vstr, vstr, vstr, vstr, vstr
        );
    fprintf(out,
        "%s %sbit    u_bit2   =  1\n"
        "%s %sbyte   u_byte2  = 20\n"
        "%s %ssbyte  s_byte2  = 10\n"
        "%s %sword   u_word2  = 20\n"
        "%s %ssword  s_word2  = 10\n"
        "%s %sdword  u_dword2 = 20\n"
        "%s %ssdword s_dword2 = 10\n",
        cstr, vstr,
        cstr, vstr,
        cstr, vstr,
        cstr, vstr,
        cstr, vstr,
        cstr, vstr,
        cstr, vstr);

    fprintf(out,
        "procedure showit(word in x) is\n"
        "  assert x != 65535\n"
        "end procedure\n"
        "\n");
    /* make sure all variables are used & accessed! */
#if 0
    fprintf(out,
        "u_bit0 = u_bit0 + u_byte0 + s_byte0 + u_word0 + s_word0 + "
        "u_dword0 + s_dword0\n"
        "\n"
        );
#else
    fprintf(out,
        "u_bit0    = 1\n"
        "u_byte0   = u_bit0\n"
        "s_byte0   = u_byte0\n"
        "u_word0   = s_byte0\n"
        "s_word0   = u_word0\n"
        "u_dword0  = s_word0\n"
        "s_dword0  = u_dword0\n"
        "u_bit0    = s_dword0\n");
#endif
  }
  return out;
}

int main(int argc, char **argv)
{
  size_t        sz1;
  unsigned      test_no;
  long          var1_sval[7] = { 0, 10, -10, 10, -10, 10, -10 };
  unsigned long var1_uval[7] = { 0, 10, -10, 10, -10, 10, -10 };
  long          var2_sval[7] = { 1, 20,  10, 20,  10, 20,  10 };
  unsigned long var2_uval[7] = { 1, 20,  10, 20,  10, 20,  10 };
  FILE         *out;
  test_type_t   test_type;
  int           direct;
  const char   *prog;

  test_type = TEST_TYPE_NONE;
  prog = *argv;
  argv++;
  argc--;
  if (argc && (0 == strcmp(*argv, "direct"))) {
    direct = 1;
    argv++;
    argc--;
  } else {
    direct = 0;
  }

  if (argc) {
    if (0 == strcmp(*argv, "var")) {
      test_type = TEST_TYPE_VAR;
      argc--;
      argv++;
    } else if (0 == strcmp(*argv, "const")) {
      test_type = TEST_TYPE_CONST;
      argc--;
      argv++;
    } else if (0 == strcmp(*argv, "volatile")) {
      test_type = TEST_TYPE_VOLATILE;
      argc--;
      argv++;
    }
  }
  if (argc || (TEST_TYPE_NONE == test_type)) {
    printf("Format: %s [direct] var|const|volatile\n"
           "  direct   : execute a direct if\n"
           "                if (expr) then ...\n"
           "             otherwise execute an indirect if\n"
           "                var = expr\n"
           "                if (var) then ...\n"
           "  var      : both arguments are variables\n"
           "  const    : one argument is constant\n"
           "  volatile : all variables are volatile\n",
           argv[0]);
    exit(0);
  }


  test_no = 0;
  out = output_open(test_type, test_no);
  for (sz1 = 0; sz1 < 4; sz1++) {
    size_t sign1;

    for (sign1 = (sz1) ? 0 : 1; sign1 < 2; sign1++) {
      size_t sz2;
      char   var1[10];

      var_name(var1, sz1, sign1, 1);

      for (sz2 = 0; sz2 < 4; sz2++) {
        size_t sign2;

        for (sign2 = (sz2) ? 0 : 1; sign2 < 2; sign2++) {
          size_t sz0;
          char   var2[10];
          size_t rel;

          var_name(var2, sz2, sign2, 2);
          for (sz0 = (direct) ? 3 : 0; sz0 < 4; sz0++) {
            size_t sign0;

            for (sign0 = (sz0 && !direct) ? 0 : 1; sign0 < 2; sign0++) {
              char   var0[10];

              var_name(var0, sz0, sign0, 0);

              for (rel = 0; rel < 6; rel++) {
                const char *rstr;
                int         is_signed;
                int         is_true;
                size_t      pos1;
                size_t      pos2;
                size_t      bsz1;
                size_t      bsz2;

                bsz1 = (sz1 < 2) ? 0 : sz1;
                bsz2 = (sz2 < 2) ? 0 : sz2;

                if ((0 == sign1) && (0 == sign2))  {
                  is_signed = 1;
                } else if (0 == sign1) {
                  is_signed = bsz1 > bsz2;
                } else {
                  is_signed = bsz2 > bsz1;
                }

                if (0 == sz1) {
                  pos1 = 0;
                } else {
                  pos1 = 1 + (sz1 - 1) * 2 + (1 - sign1);
                }

                if (0 == sz2) {
                  pos2 = 0;
                } else {
                  pos2 = 1 + (sz2 - 1) * 2 + (1 - sign2);
                }

                rstr = "";
                is_true = 0;
                switch (rel) {
                  case 0: 
                    rstr = "<"; 
                    is_true = (is_signed)
                      ? var1_sval[pos1] < var2_sval[pos2]
                      : var1_uval[pos1] < var2_uval[pos2];
                    break;
                  case 1: 
                    rstr = "<=";
                    is_true = (is_signed)
                      ? var1_sval[pos1] <= var2_sval[pos2]
                      : var1_uval[pos1] <= var2_uval[pos2];
                    break;
                  case 2: 
                    rstr = "==";
                    is_true = (is_signed)
                      ? var1_sval[pos1] == var2_sval[pos2]
                      : var1_uval[pos1] == var2_uval[pos2];
                    break;
                  case 3: 
                    rstr = "!=";
                    is_true = (is_signed)
                      ? var1_sval[pos1] != var2_sval[pos2]
                      : var1_uval[pos1] != var2_uval[pos2];
                    break;
                  case 4: 
                    rstr = ">=";
                    is_true = (is_signed)
                      ? var1_sval[pos1] >= var2_sval[pos2]
                      : var1_uval[pos1] >= var2_uval[pos2];
                    break;
                  case 5: 
                    rstr = ">"; 
                    is_true = (is_signed)
                      ? var1_sval[pos1] > var2_sval[pos2]
                      : var1_uval[pos1] > var2_uval[pos2];
                    break;
                }
                if (!direct) {
                  fprintf(out, "%s = %s %s %s ; %ssigned", 
                      var0, var1, rstr, var2, 
                      (is_signed) ? "" : "un");
                  if (is_signed) {
                    fprintf(out, " %ld %s %ld\n", 
                        var1_sval[pos1], rstr, var2_sval[pos2]);
                  } else {
                    fprintf(out, " %lu %s %lu\n", 
                        var1_uval[pos1], rstr, var2_uval[pos2]);
                  }
                  fprintf(out, 
                      "if %s%s then showit(%u) else assert(0) end if\n",
                      is_true ? "" : "!",
                      var0, test_no);
                } else {
                  fprintf(out, 
                      "if %s %s %s then ; %ssigned",
                      var1, rstr, var2, (is_signed) ? "" : "un");
                  if (is_signed) {
                    fprintf(out, " %ld %s %ld\n", 
                        var1_sval[pos1], rstr, var2_sval[pos2]);
                  } else {
                    fprintf(out, " %lu %s %lu\n", 
                        var1_uval[pos1], rstr, var2_uval[pos2]);
                  }
                  if (is_true) {
                    fprintf(out,
                        "  showit(%u)\n"
                        "else\n"
                        "  assert 0 \n"
                        "end if\n", test_no);
                  } else {
                    fprintf(out,
                        "  assert 0\n"
                        "else\n"
                        "  showit(%u)\n"
                        "end if\n", test_no);
                  }
#if 0
                  fprintf(out,
                      "  showit(%u)\n"
                      "else\n"
                      "  showit(%u)\n"
                      "end if\n",
                      is_true ? test_no : 65535,
                      is_true ? 65535 : test_no);
#endif
                }
                test_no++;

                if (!(test_no % 128)) {
                  fclose(out);
                  out = output_open(test_type, test_no);
                }
              }
            }
          }
        }
      }
    }
  }
  fclose(out);
  return 0;
}
