/************************************************************
 **
 ** bsc_blck.c : BSC block processing definitions
 **
 ** Copyright (c) 2004, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/

#include <string.h>

#include "pfile.h"
#include "pf_expr.h"
#include "pf_cmd.h"
#include "pftoken.h"
#include "vardef.h"
#include "pic.h"
#include "bsc_file.h"
#include "bsc_parm.h"
#include "bsc_tokn.h"
#include "bsc_vdef.h"
#include "bsc_ctrl.h"
#include "bsc_proc.h"
#include "bsc_intr.h"
#include "bsc_expr.h"
#include "bsc_cmd.h"
#include "pic_bsc.h"
#include "bsc_blck.h"
#ifdef MSDOS
#include <dir.h>
#define FILENAME_MAX MAXPATH
#endif
static void bsc_parse_pragma(pfile_t *pf, unsigned sub, size_t param_ct,
    const param_t *params);
static void bsc_parse_include(pfile_t *pf, unsigned sub, size_t param_ct,
    const param_t *params);

static const char *port_ids[] = { "a", "b", "c", "d", "e", 0 };
static const char *pwm_ids[]  = { "ccp1", "ccp2", "all", 0 };
static const char *usart_dir_ids[] = { "rx", "tx", 0 };
static const char *usart_bit_ids[] = { "8", 0 };

static param_dcr_t param_usart[] = {
  { param_type_cstr, usart_dir_ids },
  { param_type_cstr, usart_bit_ids },
  { param_type_expr, 0             },
  { param_type_expr, 0             }
};

static param_dcr_t param_pwm[] = {
  { param_type_cstr, pwm_ids }
};

static param_dcr_t param_c2[] = { 
  { param_type_cexpr, 0 },
  { param_type_expr,  0 }
};

static param_dcr_t param_e1[] = { 
  { param_type_expr, 0 }
};

static param_dcr_t param_e2[] = { 
  { param_type_expr, 0 },
  { param_type_expr, 0 }
};

static param_dcr_t param_e4[] = { 
  { param_type_expr, 0 },
  { param_type_expr, 0 },
  { param_type_expr, 0 },
  { param_type_expr, 0 }
};

static param_dcr_t param_p1_e1[] = { 
  { param_type_cstr, port_ids },
  { param_type_expr, 0 }
};

static param_dcr_t param_p1_e1_v1[] = { 
  { param_type_cstr, port_ids },
  { param_type_expr, 0 },
  { param_type_var,  0 }
};

static param_dcr_t param_p1_v1[] = { 
  { param_type_cstr, port_ids },
  { param_type_var,  0}
};

static param_dcr_t param_a1[] = { 
  { param_type_assign, 0 }
};

static param_dcr_t param_v1[] = { 
  { param_type_var, 0 }
};

static param_dcr_t param_e1_l2_e2[] = { 
  { param_type_expr,  0 },
  { param_type_lexpr, 0 },
  { param_type_lexpr, 0 },
  { param_type_expr,  0 },
  { param_type_expr,  0 }
};

static param_dcr_t param_e1_v1[] = {
  { param_type_expr, 0 },
  { param_type_var,  0 }
};

#define PARAM_PAIR(x) COUNT(x), x
#define PARAM_NULL    0, 0

typedef struct command_ {
  const char     *token;
  unsigned        misc;        /* used for misc. stuff */
  parse_fn_t      fn;
  size_t          param_ct;
  param_dcr_t    *param_dcr;
  const char     *reg_reqd;    /* this register must exist to continue */
} command_t;

static const command_t language[] = {
  /* variable definition */
  {"record",       0,                               bsc_parse_record,
    PARAM_NULL},
  {"constant",     0,                               bsc_parse_constdef,
   PARAM_NULL},
  /* flow control */
  {"for",          bsc_intrinsic_type_null,  bsc_parse_for,
   PARAM_PAIR(param_a1)},
  {"if",           bsc_intrinsic_type_null,  bsc_parse_if,
   PARAM_PAIR(param_e1)},
  {"gosub",        cmd_branchtype_call,         bsc_parse_branch,
   PARAM_NULL},
  {"goto",         cmd_branchtype_goto,         bsc_parse_branch,
   PARAM_NULL},
  {"return",       cmd_branchtype_return,       bsc_parse_branch,
   PARAM_NULL},
  {"call",         0,                           bsc_parse_call,
   PARAM_NULL},
  {"proc",         0,                           bsc_parse_proc,
   PARAM_NULL},
  /* misc. */
  {"clear",        bsc_intrinsic_type_clear,         bsc_parse_intrinsic,
   PARAM_NULL},
  {"delay",        bsc_intrinsic_type_delay,         bsc_parse_intrinsic,
   PARAM_PAIR(param_c2)},
  {"interrupt_enable", bsc_intrinsic_type_interrupt_enable,   
    bsc_parse_intrinsic, PARAM_NULL},
  {"interrupt_disable", bsc_intrinsic_type_interrupt_disable, 
    bsc_parse_intrinsic, PARAM_NULL},
  /* IR */
  {"ir_adc_begin", bsc_ir_type_adc,                  bsc_parse_ir,
   PARAM_NULL, "_adif"},
  {"ir_pwm_begin", bsc_ir_type_pwm,                  bsc_parse_ir,
   PARAM_NULL, "_tmr2if"},
  {"ir_rx_begin",  bsc_ir_type_rx,                   bsc_parse_ir,
   PARAM_NULL, "_rcif"},
  {"ir_tx_begin",  bsc_ir_type_tx,                   bsc_parse_ir,
   PARAM_NULL, "_txif"},
  {"ir_timer_begin", bsc_ir_type_timer,              bsc_parse_ir,
   PARAM_NULL, "_tmr1if"},
  {"ir_portb_begin", bsc_ir_type_rb,                 bsc_parse_ir,
    PARAM_NULL, "_rbie"},
  {"ir_user_begin", bsc_ir_type_user,                bsc_parse_ir,
    PARAM_NULL, 0},
  /* ADC */
  {"adc_config",   bsc_intrinsic_type_adc_config,    bsc_parse_intrinsic,
   PARAM_PAIR(param_e1_l2_e2), "_adcon0"},
  {"adc_on",       bsc_intrinsic_type_adc_on,        bsc_parse_intrinsic,
   PARAM_NULL, "_adcon0"},
  {"adc_off",      bsc_intrinsic_type_adc_off,       bsc_parse_intrinsic,
   PARAM_NULL, "_adcon0"},
  {"adc_start",    bsc_intrinsic_type_adc_start,    bsc_parse_intrinsic,
   PARAM_NULL, "_adcon0"},
  {"adc_store",    bsc_intrinsic_type_adc_store,    bsc_parse_intrinsic,
   PARAM_PAIR(param_v1), "_adresl"},
  {"adc_wait",     bsc_intrinsic_type_adc_wait,     bsc_parse_intrinsic,
   PARAM_NULL, "_adcon0"},
  /* PWM */
  {"pwm_config",   bsc_intrinsic_type_pwm_config,   bsc_parse_intrinsic,
   PARAM_PAIR(param_e4)},
  {"pwm_duty",     bsc_intrinsic_type_pwm_duty,     bsc_parse_intrinsic,
   PARAM_PAIR(param_e2), "_ccp1con"},
  {"pwm_off",      bsc_intrinsic_type_pwm_off,      bsc_parse_intrinsic,
   PARAM_PAIR(param_pwm), "_ccp1con"},
  {"pwm_period",   bsc_intrinsic_type_pwm_period,   bsc_parse_intrinsic,
   PARAM_PAIR(param_e1), "_ccp1con"},
  {"pwm_reg",      bsc_intrinsic_type_pwm_reg,      bsc_parse_intrinsic,
   PARAM_PAIR(param_e1), "_ccp1con"},
  /* PORT control */
  {"interrupt_portb", bsc_intrinsic_type_interrupt_portb,  bsc_parse_intrinsic,
   PARAM_PAIR(param_e2), "_rbie"},
  {"pinhigh",      bsc_intrinsic_type_pinhigh,      bsc_parse_intrinsic,
   PARAM_PAIR(param_p1_e1)},
  {"pinlow",       bsc_intrinsic_type_pinlow,       bsc_parse_intrinsic,
   PARAM_PAIR(param_p1_e1)},
  {"pinrd",        bsc_intrinsic_type_pinrd,        bsc_parse_intrinsic,
   PARAM_PAIR(param_p1_e1_v1)},
  {"portout",      bsc_intrinsic_type_portout,      bsc_parse_intrinsic,
   PARAM_PAIR(param_p1_e1)},
  {"portrd",       bsc_intrinsic_type_portrd,       bsc_parse_intrinsic,
   PARAM_PAIR(param_p1_v1)},
  {"setport",      bsc_intrinsic_type_setport,      bsc_parse_intrinsic,
   PARAM_PAIR(param_p1_e1)},
  /* USART */
  {"rx_err_check", bsc_intrinsic_type_rx_err_check, bsc_parse_intrinsic,
   PARAM_PAIR(param_v1), "_rcsta"},
  {"rx_store",     bsc_intrinsic_type_rx_store,     bsc_parse_intrinsic,
   PARAM_PAIR(param_v1), "_rcreg"},
  {"tx_load",      bsc_intrinsic_type_tx_load,      bsc_parse_intrinsic,
   PARAM_PAIR(param_e1), "_txreg"},
  {"usart_config", bsc_intrinsic_type_usart_config, bsc_parse_intrinsic,
   PARAM_PAIR(param_usart), "_spbrg"},
  {"wait_rx",      bsc_intrinsic_type_wait_rx,      bsc_parse_intrinsic,
   PARAM_NULL, "_rcif"},
  {"wait_tx",      bsc_intrinsic_type_wait_tx,      bsc_parse_intrinsic,
   PARAM_NULL, "_txif"},
  /* timer */
  {"timer_config", bsc_intrinsic_type_timer_config, bsc_parse_intrinsic,
   PARAM_PAIR(param_e2), "_t1con"},
  {"timer_start",  bsc_intrinsic_type_timer_start, bsc_parse_intrinsic,
   PARAM_NULL, "_t1con"},
  {"timer_stop", bsc_intrinsic_type_timer_stop, bsc_parse_intrinsic,
   PARAM_NULL, "_t1con"},
  {"timer_countdown", bsc_intrinsic_type_timer_countdown, bsc_parse_intrinsic,
   PARAM_NULL, "_tmr1if"},
  {"timer_read", bsc_intrinsic_type_timer_read, bsc_parse_intrinsic,
    PARAM_PAIR(param_v1), "_tmr1"},
  {"timer_write", bsc_intrinsic_type_timer_write, bsc_parse_intrinsic,
    PARAM_PAIR(param_e1), "_tmr1"},
  {"print",    0, bsc_parse_print, PARAM_NULL},
  {"while",    0, bsc_parse_while, PARAM_NULL},
  {"do",       0, bsc_parse_do,    PARAM_NULL},
  {"sleep",    bsc_intrinsic_type_sleep, bsc_parse_intrinsic, PARAM_NULL},
  {"clear",    bsc_intrinsic_type_clear, bsc_parse_intrinsic, PARAM_NULL},
  {"pragma",   0, bsc_parse_pragma, PARAM_NULL},
  {"include",  0, bsc_parse_include, PARAM_NULL},
  {"eeprom_write", bsc_intrinsic_type_eeprom_write, bsc_parse_intrinsic,
    PARAM_PAIR(param_e2), "_eedata"},
  {"eeprom_read", bsc_intrinsic_type_eeprom_read, bsc_parse_intrinsic,
    PARAM_PAIR(param_e1_v1), "_eedata"},
  {"eeprom_wait", bsc_intrinsic_type_eeprom_wait, bsc_parse_intrinsic,
    PARAM_NULL, "_eedata"}
};

/*
 * NAME
 *   statement_skip
 *
 * DESCRIPTION
 *   skip to the end of this statement
 *
 * PARAMETERS
 *   pf : pfile
 *
 * RETURN
 *   none
 *
 * NOTES
 *   used by variable & procedure definition after the first
 *   pass
 */
void bsc_statement_skip(pfile_t *pf)
{
  while (!pf_token_is_eos(pf)) {
    pf_token_get(pf, pf_token_next);
  }
}

void bsc_statement_next(pfile_t *pf)
{
  do {
    if (!bsc_token_is_eos(pf)) {
      if (!pf_token_is(pf, pf_token_current, pfile_log_none,
        "'")) {
        pfile_log(pf, pfile_log_warn, BSC_MSG_EXTRA_CHAR_ON_LINE);
      }
      bsc_statement_skip(pf);
    }
    pf_token_get(pf, pf_token_next);
  } while (!pf_token_is_eof(pf) && bsc_token_is_eos(pf));
}



/*
 * NAME
 *   command_process
 *
 * DESCRIPTION
 *   process a command
 *
 * PARAMETERS
 *   cmd : command to process
 *
 * RETURN
 *   none
 *
 * NOTES
 *   this parses any necessary parameters, executing the command on success
 */
static void command_process(pfile_t *pf, const command_t *cmd)
{
  param_t    *param;
  size_t      param_ct;
  boolean_t   err;
  cmd_t       cmd_init;

  err = boolean_false;
  cmd_init = pfile_cmdlist_get(pf);
  if (!cmd->param_ct) {
    param_ct = 0;
    param    = 0;
  } else {
    param_ct = 0;
    param    = CALLOC(sizeof(*param), cmd->param_ct);
    if (0 == param) {
      pfile_log_syserr(pf, result_memory);
    } else {
      param_t *param_ptr;

      for (param_ct = 0, param_ptr = param; 
           (param_ct < cmd->param_ct)
           && (!param_ct
             || pf_token_is(pf, pf_token_current, pfile_log_err, ","));
           param_ct++, param_ptr++) {
        if (param_ct) {
          pf_token_get(pf, pf_token_next);
        }
        param_ptr->type = cmd->param_dcr[param_ct].type;
        switch (cmd->param_dcr[param_ct].type) {
          case param_type_null:
            break;
          case param_type_expr:
          case param_type_lexpr:
          case param_type_cexpr:
            if (result_ok == bsc_expr_parse(pf, 0,0,  &param_ptr->u.var)) {
              if (param_type_lexpr == cmd->param_dcr[param_ct].type) {
                /* need one more -- !! var */
                pfile_cmd_op_add(pf, operator_logical, 
                    &param_ptr->u.var, 0, 0);
              } else if ((param_type_cexpr == cmd->param_dcr[param_ct].type)
                && !value_is_const(param_ptr->u.var)) {
                pfile_log(pf, pfile_log_err, BSC_MSG_CONSTANT_EXPECTED);
                err = boolean_true;
              }
            } else {
              err = boolean_true;
            }
            break;
          case param_type_var:
            if (result_ok == pf_token_to_value(pf, pfile_log_err, 
                  &param_ptr->u.var)) {
              pf_token_get(pf, pf_token_next);
            } else {
              err = boolean_true;
            }
            break;
          case param_type_cstr:
            {
              size_t       ct;
              const char **strs;

              strs = cmd->param_dcr[param_ct].param;

              for (ct = 0; 
                   strs[ct] 
                   && !pf_token_is(pf, pf_token_current, 
                     pfile_log_none, strs[ct]);
                   ct++)
                ;
              if (strs[ct]) {
                param_ptr->u.sval = ct;
                pf_token_get(pf, pf_token_next);
              } else {
                pfile_log(pf, pfile_log_err, BSC_MSG_SYNTAX_ERROR);
                err = boolean_true;
              }
            }
            break;
          case param_type_assign:
            {
              value_t var;

              if (result_ok == pf_token_to_value(pf, pfile_log_err, &var)) {
                if (pf_token_is(pf, pf_token_next, pfile_log_err,
                   "=")) {
                  value_t dst;

                  pf_token_get(pf, pf_token_next);
                  if (result_ok == bsc_expr_parse(pf, 0, 0, &dst)) {
                    pfile_cmd_op_add(pf, operator_assign, &var, dst, 0);
                    value_lock(var);
                    value_release(dst);
                    param_ptr->u.var = var;
                  } else {
                    err = boolean_true;
                  }
                } else {
                  err = boolean_true;
                }
                value_release(var);
              } else {
                err = boolean_true;
              }
            }
            break;
          case param_type_label:
            if (result_ok == pf_token_to_label(pf, pfile_log_err, 
              boolean_false, &param_ptr->u.lbl)) {
              pf_token_get(pf, pf_token_next);
            }
            break;
        }
      }
    }
  }
  if (!err && (param_ct == cmd->param_ct)) {
    if (cmd->reg_reqd && !pfile_variable_exists(pf, 
              PFILE_LENGTH_DEF, cmd->reg_reqd)) {
      bsc_file_log_unavail(pf, cmd->token);
    } else {
      cmd->fn(pf, cmd->misc, param_ct, param);
    }
  }
  while (param_ct--) {
    switch (param[param_ct].type) {
      case param_type_null:
      case param_type_cstr:
        break;
      case param_type_expr:
      case param_type_cexpr:
      case param_type_lexpr:
      case param_type_assign:
      case param_type_var:
        value_release(param[param_ct].u.var);
        break;
      case param_type_label:
        label_release(param[param_ct].u.lbl);
        break;
    }
  }
  if (param) {
    FREE(param);
  }
}


/*
 * NAME
 *   bsc_block_process
 *
 * DESCRIPTION
 *   process until a block end occurs
 *
 * PARAMETERS
 *   pf       : returned by pfile_open()
 *   endct    : # of endwords
 *   endwords : words that end the  block
 *   ew       : [out] word # that ended the block, or EOF
 *
 * RETURN
 *   none
 *
 * NOTES
 *   return on EOF or when cb returns TRUE
 */
static int token_is_end(pfile_t *pf, size_t endct, const char **endwords,
  size_t *ew)
{
  int      rc;

  if (pf_token_is_eof(pf)) {
    if (ew) {
      *ew = EOF;
    }
    rc = 1;
  } else {
    size_t ii;

    for (ii = 0;
         (ii < endct)
         && !pf_token_is(pf, pf_token_current, pfile_log_none,
           endwords[ii]);
         ii++)
      ;
    rc = (ii < endct);
    if (rc) {
      if (ew) {
        *ew = ii;
      }
      pf_token_get(pf, pf_token_next);
    }
  }
  return rc;
}

void bsc_block_process(pfile_t *pf, flag_t flags, size_t endct, 
    const char **endwords, size_t *ew)
{
  boolean_t  allow_label;
  unsigned   line_start;

  if (flags & BSC_BLOCK_PROCESS_FLAG_SKIP_STMT) {
    bsc_statement_next(pf);
  }

  if (!(flags & BSC_BLOCK_PROCESS_FLAG_NO_BLOCK)) {
    pfile_cmd_special_add(pf, cmd_type_block_start, 0);
    pfile_block_enter(pf);
  }
  line_start = pfile_line_get(pf);
  allow_label = boolean_true;
  while (!token_is_end(pf, endct, endwords, ew)) {
    char        keyword[256];
    const char *ptr;
    size_t      ii;
    cmd_t       cmd_init;

    cmd_init = pfile_cmdlist_tail_get(pf);
    ptr = pf_token_get(pf, pf_token_current);
    strncpy(keyword, ptr, sizeof(keyword) - 1);
    keyword[sizeof(keyword) - 1] = 0;
    pf_token_get(pf, pf_token_next);

    if (allow_label
      && pf_token_is(pf, pf_token_current, pfile_log_none,
        ":")) {
      /* keyword is a label */
      label_t label;

      label = pfile_label_find(pf, pfile_log_none, keyword);
      if (label) {
        if ((1 == bsc_file_pass_get(pf))
          && label_flag_test(label, LABEL_FLAG_DEFINED)) {
          label_release(label);
          label = 0;
          pfile_log(pf, pfile_log_err, BSC_MSG_MULTI_DEF, keyword);
        } else if ((2 == bsc_file_pass_get(pf))
            && !label_flag_test(label, LABEL_FLAG_USED)) {
          pfile_log(pf, pfile_log_warn, BSC_MSG_NEVER_USED, 
              label_name_get(label));
        }
      } else {
        pfile_label_alloc(pf, keyword, &label);
      }
      if (label) {
        label_flag_set(label, LABEL_FLAG_DEFINED);
        pfile_cmd_label_add(pf, label);
        label_release(label);
      }
      ptr = pf_token_get(pf, pf_token_next);
      if (bsc_token_is_eos(pf)) {
        bsc_statement_next(pf);
      } else {
        allow_label = boolean_false;
      }
    } else {
      /* end must be special cased becuase it can be used
         as a two word keyword */
      if (0 == strcmp(keyword, "end")) {
        for (ii = 0;
             (ii < endct)
             && !((0 == memcmp(endwords[ii], "end", 3))
               && (0 == strcmp(endwords[ii]+3, ptr)));
             ii++)
          ;
        if (ii < endct) {
          if (ew) {
            *ew = ii;
          }
          pf_token_get(pf, pf_token_next);
          break; /* <--- */
        }
        bsc_parse_intrinsic(pf, bsc_intrinsic_type_end, 0, 0);
        bsc_statement_next(pf);
      } else {
        variable_def_t def;

        def = pfile_variable_def_find(pf, pfile_log_none, keyword);
        if (def) {
          bsc_parse_def(pf, variable_def_flags_get_all(def), def);
        } else {
          for (ii = 0;
              (ii < COUNT(language))
              && strcmp(keyword, language[ii].token);
              ii++)
            ;
          if (COUNT(language) == ii) {
            if (strcmp(keyword, "\n")) {
              /* this is an assignment */
              value_t var;

              var = 0;
              pf_identifier_to_value(pf, pfile_log_err, keyword, &var);
              if (var) {
                bsc_expr_parse(pf, EXPR_FLAG_ASSIGN, var, 0);
                value_release(var);
              }
            }
          } else {
            command_process(pf, language + ii);
          }
        }
        bsc_statement_next(pf);
      }
      allow_label = boolean_true;
    }
    if (pfile_cmdlist_tail_get(pf) != cmd_init) {
      pfile_cmd_special_add(pf, cmd_type_statement_end, 0);
    }
  }
  if (!(flags & BSC_BLOCK_PROCESS_FLAG_NO_BLOCK)) {
    pfile_cmd_special_add(pf, cmd_type_block_end, 0);
    pfile_block_leave(pf);
  }
  if (endct && pf_token_is_eof(pf)) {
    pfile_log(pf, pfile_log_err, BSC_MSG_END_EXPECTED,
      endwords[0], line_start);
  }
}

void bsc_include_process(pfile_t *pf)
{
  pf_token_get(pf, pf_token_first);
  bsc_block_process(pf, BSC_BLOCK_PROCESS_FLAG_NO_BLOCK, 0, 0, 0);
}

/*
 * NAME
 *   source_process
 *
 * DESCRIPTION
 *   process the source file
 *
 * PARAMETERS
 *   pf : file
 *
 * RETURN
 *   none
 *
 * NOTES
 *   this parses out the PIC, FREQ and OSCILLATOR keywords
 *   before handing things off to block_process
 */
void bsc_source_process(pfile_t *pf)
{
  label_t  lbl;
  char     pic_include[32];

  strcpy(pic_include, "p16f877.inc"); /* the default */

  bsc_file_pass_reset(pf); /* bump the pass counter */
  /* get the PIC if present, otherwise assume 16f877 */
  pfile_log(pf, pfile_log_info, BSC_MSG_COMPILING_PASS, 
      bsc_file_pass_get(pf));
  if (pf_token_is(pf, pf_token_current, pfile_log_none, 
        "pic")) {
    const char *ptr;

    ptr = bsc_token_hack(pf);
    if (strlen(ptr) > sizeof(pic_include) - 6) {
      pfile_log(pf, pfile_log_err, BSC_MSG_SYNTAX_ERROR);
    } else {
      sprintf(pic_include, "p%s.inc", ptr);
    }
    pf_token_get(pf, pf_token_next);
    bsc_statement_next(pf);
  }
  /* get the frequency if present, otherwise assume 20000 */
  if (pf_token_is(pf, pf_token_current, pfile_log_none, 
        "freq")) {
    variable_t freq;

    freq = pfile_variable_find(pf, pfile_log_err, PFILE_LENGTH_DEF, 
      "_freq");
    if (freq) {
      value_t tmp;

      pf_token_get(pf, pf_token_next);
      if (result_ok == bsc_expr_parse(pf, 0, 0, &tmp)) {
        if (!(value_vflag_test(tmp, VARIABLE_FLAG_CONST))) {
          pfile_log(pf, pfile_log_err, BSC_MSG_CONSTANT_EXPECTED);
        } else {
          variable_const_set(freq, 0, value_const_get(tmp));
        }
        value_release(tmp);
      }
      variable_release(freq);
      bsc_statement_next(pf);
    }
  }
  if (pf_token_is(pf, pf_token_current, pfile_log_none, 
        "oscillator")) {
    if (bsc_file_pass_get(pf) == 1) {
      if (pf_token_is(pf, pf_token_next, pfile_log_none, 
            "crystal")) {
        pf_token_get(pf, pf_token_next);
      } else if (pf_token_is(pf, pf_token_current, pfile_log_none,
            "rc")) {
        pf_token_get(pf, pf_token_next);
      } else {
        pfile_log(pf, pfile_log_err, BSC_MSG_OSCILLATOR_BAD);
      }
    } else {
      bsc_statement_skip(pf);
    }
    bsc_statement_next(pf);
  }

  /* the entry point is named, ``_main'' */
  lbl = 0;
  if (1 == bsc_file_pass_get(pf)) {
    if (result_ok == pfile_label_alloc(pf, "_main", &lbl)) {
      pfile_user_entry_set(pf, lbl);
    }
  } else {
    lbl = pfile_label_find(pf, pfile_log_err, "_main");
  }
  if (lbl) {
    pfile_cmd_label_add(pf, lbl);
    label_release(lbl);
  }
  pfile_include_process(pf, 0, pic_include);
  bsc_block_process(pf, BSC_BLOCK_PROCESS_FLAG_NONE, 0, 0, 0);
  pfile_cmd_special_add(pf, cmd_type_end, 0);
}

void bsc_file_process(pfile_t *pf)
{
  size_t   ii;
  static const struct {
    const char   *tag;
    variable_sz_t sz;
    flag_t        flags;
  } basetypes[] = {
    {"boolean",0, VARIABLE_FLAG_BIT | VARIABLE_FLAG_BOOLEAN},
    {"bit",    0, VARIABLE_FLAG_BIT},
    {"sbit",   0, VARIABLE_FLAG_BIT | VARIABLE_FLAG_SIGNED},
    {"byte",   1, 0},
    {"sbyte",  1, VARIABLE_FLAG_SIGNED},
    {"word",   2, 0},
    {"sword",  2, VARIABLE_FLAG_SIGNED},
    {"dword",  4, 0},
    {"sdword", 4, VARIABLE_FLAG_SIGNED}
  };

  for (ii = 0; ii < COUNT(basetypes); ii++) {
    variable_def_t vdef;

    vdef = variable_def_alloc(basetypes[ii].tag, basetypes[ii].sz,
      basetypes[ii].flags);
    if (vdef) {
      pfile_variable_def_add(pf, vdef);
    }
  }

  pic_init(pf); /* intialize the pic-specific bits */
  bsc_source_process(pf);
  if (!pfile_errct_get(pf)) {
    bsc_isr_type_t isr_type;

    pfile_label_fixup(pf);
    pfile_variable_fixup(pf);
    for (isr_type = bsc_isr_type_first;
         isr_type < bsc_isr_type_ct;
         isr_type++) {
      label_t isr;

      if (result_ok == bsc_file_isr_get(pf, isr_type, boolean_false, &isr)) {
        if (label_flag_test(isr, LABEL_FLAG_USED)) {
          pfile_isr_entry_set(pf, isr);
          label_release(isr);
          break;
        } else {
          label_release(isr);
        }
      }
    }
  }
  if (!pfile_errct_get(pf)) {
    bsc_source_process(pf);
    if (!pfile_errct_get(pf)) {
      pfile_cmd_dump(pf);
    }
  }
  if (pfile_flag_test(pf, PFILE_FLAG_DEBUG)) {
    pfile_label_dump(pf);
    pfile_variable_dump(pf);
  }
}

static void bsc_parse_pragma(pfile_t *pf, unsigned sub, size_t param_ct,
    const param_t *params)
{
  if (pf_token_is(pf, pf_token_current, pfile_log_none, "warn")) {
    if (pf_token_is(pf, pf_token_next, pfile_log_none, "on")) {
      pfile_flag_clr(pf, PFILE_FLAG_NOWARN);
      pf_token_get(pf, pf_token_next);
    } else if (pf_token_is(pf, pf_token_current, pfile_log_none, "off")) {
      pfile_flag_set(pf, PFILE_FLAG_NOWARN);
      pf_token_get(pf, pf_token_next);
    } else {
      pfile_log(pf, pfile_log_err, BSC_MSG_ON_OR_OFF);
    }
  } else {
    /* pass it off to the PIC for further processing */
    pic_parse_pragma(pf);
  }
}

static void bsc_parse_include(pfile_t *pf, unsigned sub, size_t param_ct,
    const param_t *params)
{
  const char *ptr;

  if (bsc_token_is_eos(pf)) {
    pfile_log(pf, pfile_log_err, BSC_MSG_FILENAME_BAD);
  } else {
    size_t sz;

    ptr = pf_token_get(pf, pf_token_current);
    if (*ptr != '"') {
      pfile_log(pf, pfile_log_err, BSC_MSG_EXPECTED, "\"");
    } else {
      ptr++;
      sz = strlen(ptr);

      if (sz && ('\"' == ptr[sz-1])) {
        sz--;
      }
      if (!sz) {
        pfile_log(pf, pfile_log_err, BSC_MSG_FILENAME_BAD);
      } else {
        pfile_include_process(pf, sz, ptr);
      }
    }
    pf_token_get(pf, pf_token_next);
  }
}

