/**********************************************************
 **
 ** bsc_cmd.c : manipulators for bsc_cmd_t
 **
 ** Copyright (c) 2004, Kyle A. York
 ** All rights reserved
 **
 ***********************************************************/
#include <stdlib.h>
#include <string.h>

#include "variable.h"
#include "label.h"
#include "cmd.h"
#include "bsc_cmd.h"
#include "cmd_dcl.h"

static void bsc_cmd_dump(cmd_t cmd, FILE *dst)
{
  const char *istr;
  size_t      ii;

  istr = "{unknown}";
  switch (bsc_cmd_type_get(cmd)) {
    case bsc_intrinsic_type_null:        istr = "nop";          break;
    case bsc_intrinsic_type_clear:       istr = "clear";        break;
    case bsc_intrinsic_type_end:         istr = "end";          break;
    case bsc_intrinsic_type_delay:       istr = "delay";        break;
    case bsc_intrinsic_type_adc_config:  istr = "adc_config";   break;
    case bsc_intrinsic_type_adc_on:      istr = "adc_on";       break;
    case bsc_intrinsic_type_adc_off:     istr = "adc_off";      break;
    case bsc_intrinsic_type_adc_start:   istr = "adc_start";    break;
    case bsc_intrinsic_type_adc_store:   istr = "adc_store";    break;
    case bsc_intrinsic_type_adc_wait:    istr = "adc_wait";     break;
    case bsc_intrinsic_type_pwm_config:  istr = "pwm_config";   break;
    case bsc_intrinsic_type_pwm_duty:    istr = "pwm_duty";     break;
    case bsc_intrinsic_type_pwm_off:     istr = "pwm_off";      break;
    case bsc_intrinsic_type_pwm_period:  istr = "pwm_period";   break;
    case bsc_intrinsic_type_pwm_reg:     istr = "pwm_reg";      break;
    case bsc_intrinsic_type_pinhigh:     istr = "pinhigh";      break;
    case bsc_intrinsic_type_pinlow:      istr = "pinlow";       break;
    case bsc_intrinsic_type_pinrd:       istr = "pinrd";        break;
    case bsc_intrinsic_type_portout:     istr = "portout";      break;
    case bsc_intrinsic_type_portrd:      istr = "portrd";       break;
    case bsc_intrinsic_type_rx_err_check:istr = "rx_err_check"; break;
    case bsc_intrinsic_type_rx_store:    istr = "rx_store";     break;
    case bsc_intrinsic_type_setport:     istr = "setport";      break;
    case bsc_intrinsic_type_tx_load:     istr = "tx_load";      break;
    case bsc_intrinsic_type_usart_config:istr = "usart_config"; break;
    case bsc_intrinsic_type_wait_rx:     istr = "wait_rx";      break;
    case bsc_intrinsic_type_wait_tx:     istr = "wait_tx";      break;
    case bsc_intrinsic_type_timer_config:istr = "timer_config"; break;
    case bsc_intrinsic_type_timer_start: istr = "timer_start";  break;
    case bsc_intrinsic_type_timer_stop:  istr = "timer_stop";   break;
    case bsc_intrinsic_type_timer_countdown: istr = "timer_countdown"; break;
    case bsc_intrinsic_type_interrupt_enable:
      istr = "interrupt_enable";
      break;
    case bsc_intrinsic_type_interrupt_disable:
      istr = "interrupt_disable";
      break;
    case bsc_intrinsic_type_timer_read:  istr = "timer_read"; break;
    case bsc_intrinsic_type_timer_write: istr = "timer_write"; break;
    case bsc_intrinsic_type_sleep: istr = "sleep"; break;
    case bsc_intrinsic_type_interrupt_portb: istr = "interrupt_portb"; break;
    case bsc_intrinsic_type_eeprom_write: istr = "eeprom_write"; break;
    case bsc_intrinsic_type_eeprom_read: istr = "eeprom_read"; break;
    case bsc_intrinsic_type_eeprom_wait: istr = "eeprom_wait"; break;
  }
  fprintf(dst, "%s ", istr);
  for (ii = 0; ii < bsc_cmd_paramct_get(cmd); ii++) {
    if (ii) {
      fprintf(dst, ", ");
    }
    switch (bsc_cmd_param_get(cmd, ii)->type) {
      case param_type_null:
        break;
      case param_type_expr:
      case param_type_cexpr:
      case param_type_lexpr:
      case param_type_assign:
      case param_type_var:
        value_dump(bsc_cmd_param_get(cmd, ii)->u.var, dst);
        break;
      case param_type_cstr:
        fprintf(dst, "str{%u}", bsc_cmd_param_get(cmd, ii)->u.sval);
        break;
      case param_type_label:
        fprintf(dst, "%s", label_name_get(bsc_cmd_param_get(cmd, ii)->u.lbl));
        break;
    }
  }
}

static void bsc_cmd_free(cmd_t cmd)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr) {
    size_t     ii;

    for (ii = 0; ii < ptr->u.bsc.param_ct; ii++) {
      switch (ptr->u.bsc.params[ii].type) {
        case param_type_null:
          break;
        case param_type_expr:
        case param_type_cexpr:
        case param_type_lexpr:
        case param_type_assign:
        case param_type_var:
          value_release(ptr->u.bsc.params[ii].u.var);
          break;
        case param_type_cstr:
          break;
        case param_type_label:
          label_release(ptr->u.bsc.params[ii].u.lbl);
          break;
      }
    }
  }
}

static const cmd_vtbl_t bsc_cmd_vtbl = {
  bsc_cmd_free,
  bsc_cmd_dump
};

cmd_t bsc_cmd_alloc(
  bsc_intrinsic_type_t op, size_t param_ct, const param_t *params)
{
  cmd_t cmd;

  cmd = cmd_alloc(cmd_type_intrinsic, &bsc_cmd_vtbl);
  if (cmd) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      size_t ii;

      ptr->u.bsc.type     = op;
      ptr->u.bsc.param_ct = param_ct;
      ptr->u.bsc.params   = (param_ct) 
        ? MALLOC(param_ct * sizeof(*ptr->u.bsc.params)) 
        : 0;
      if (ptr->u.bsc.params) {
        memcpy(ptr->u.bsc.params, params, sizeof(*params) * param_ct);
      }
      for (ii = 0; ii < ptr->u.bsc.param_ct; ii++) {
        switch (ptr->u.bsc.params[ii].type) {
          case param_type_null:
            break;
          case param_type_expr:
          case param_type_cexpr:
          case param_type_lexpr:
          case param_type_assign:
          case param_type_var:
            value_lock(ptr->u.bsc.params[ii].u.var);
            break;
          case param_type_cstr:
            break;
          case param_type_label:
            label_lock(ptr->u.bsc.params[ii].u.lbl);
            break;
        }
      }
    }
  }
  return cmd;
}

bsc_intrinsic_type_t bsc_cmd_type_get(const cmd_t cmd)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr && (cmd_type_intrinsic == cmd_type_get(cmd)))
    ? ptr->u.bsc.type
    : bsc_intrinsic_type_null;
}

void bsc_cmd_type_set(cmd_t cmd, bsc_intrinsic_type_t type)
{
  if (cmd_type_intrinsic == cmd_type_get(cmd)) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      ptr->u.bsc.type = type;
    }
  }
}

size_t bsc_cmd_paramct_get(const cmd_t cmd)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr && (cmd_type_intrinsic == cmd_type_get(cmd)))
    ? ptr->u.bsc.param_ct
    : 0;
}

const param_t *bsc_cmd_param_get(const cmd_t cmd, size_t ii)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr
    && (cmd_type_intrinsic == cmd_type_get(cmd))
    && (ii < bsc_cmd_paramct_get(cmd)))
    ? ptr->u.bsc.params + ii
    : 0;
}

