/**********************************************************
 **
 ** bsc_expr.c : BSC expression parsing
 **
 ** Copyright (c) 2004, Kyle A. York
 ** All rights reserved
 **
 ***********************************************************/
#include "ptype.h"
#include "pfile.h"
#include "pf_expr.h"
#include "pftoken.h"
#include "vardef.h"
#include "expr.h"
#include "bsc_tokn.h"
#include "bsc_expr.h"

/*
 * NAME
 *   bsc_subscript_parse
 *
 * DESCRIPTION
 *   get the subscript for an array variable
 *
 * PARAMETERS
 *   pf  : pfile
 *   val : value
 *
 * RETURN
 *   EINVAL : syntax error
 *   
 * NOTES
 *   the pfile token should be '(' on entry
 */
static result_t bsc_subscript_parse(pfile_t *pf, value_t val)
{
  result_t rc;
  value_t  subscript;

  rc = bsc_expr_parse(pf, 0, 0, &subscript);
  if (result_ok == rc) {
    if (value_sz_get(val) > 1) {
      value_t tmp;
      value_t tmp_sz;

      tmp_sz = pfile_constant_get(pf, value_sz_get(val));
      tmp    = pfile_value_temp_get(pf, 2, 0);
      if (tmp && tmp_sz) {
        pfile_cmd_op_add(pf, operator_assign, &tmp, subscript, 0);
        pfile_cmd_op_add(pf, operator_mul, &tmp, tmp, tmp_sz);
      }
      value_release(subscript);
      value_release(tmp_sz);
      subscript = tmp;
    }
    if (result_ok == rc) {
      value_baseofs_set(val, subscript);
      value_ct_set(val, 1);
    }
    value_release(subscript);
    if (!pf_token_is(pf, pf_token_current, pfile_log_err, ")")) {
        rc = result_syntax;
    } else {
      pf_token_get(pf, pf_token_next);
    }
  }
  return rc;
}

/*
 * NAME
 *   value_get
 *
 * DESCRIPTION
 *   get an expression's value
 *
 * PARAMETERS
 *   pf    : return from pfile_open()
 *   flags : flags passed into expr_parse()
 *   var   : [out] holds result
 *
 * RETURN
 *
 * NOTES
 */
static result_t value_get(pfile_t *pf, flag_t flags, value_t *dst)
{
  result_t    rc;
  value_t     var;
  boolean_t   skipnext;
  const char *ptr;
  operator_t  op;

  /* valid values: 
       constant ('%' or '$' or decimal #)
       variable name (alpha + alpha|digit|'_')
       '(' -- subfunction
       '!' -- logical NOT
       '-' -- negation (two's complement)
       '~' -- complement (one's complement) */

  skipnext = boolean_false;
  ptr = pf_token_get(pf, pf_token_current);
  op = operator_null;
  switch (*ptr) {
    case '*': /* dereference */
      pf_token_get(pf, pf_token_next); /* skip it */
      rc = value_get(pf, flags, &var);
      if (result_ok == rc) {
        if (!value_vflag_test(var, VARIABLE_FLAG_POINTER)
          || value_vflag_test(var, VARIABLE_FLAG_REFERENCE)) {
          /* this isn't a pointer as it should be! */
          pfile_log(pf, pfile_log_err, BSC_MSG_POINTER_EXPECTED);
        } else {
          variable_def_t        def;
          variable_def_member_t mbr;

          def = value_def_get(var);
          mbr = variable_def_member_get(def);
          def = variable_def_member_def_get(mbr);
          value_def_set(var, def, variable_def_sz_get(def),
              variable_def_member_ct_get(mbr));

          value_flag_set(var, VARIABLE_FLAG_REFERENCE);
        }
      }
      break;

    case '!':
      op = ('!' == ptr[1]) ? operator_logical : operator_notl;
      /* fall through */
    case '-':
      if (operator_null == op) {
        op = operator_neg;
      }
      /* fall through */
    case '~':
      if (operator_null == op) {
        op = operator_cmpb;
      }
      pf_token_get(pf, pf_token_next); /* this one's done */
      rc = value_get(pf, flags, &var);   /* var is the next value */
      /* create the expression:
       *   _t# = var op
       *   and return _t#
       */
      if (result_ok == rc) {
        value_t tmp;

        tmp = 0;
        pfile_cmd_op_add(pf, op, &tmp, var, 0);
        value_release(var);
        var = tmp;
      }
      skipnext = boolean_true;
      break;
    case '(':
      pf_token_get(pf, pf_token_next);
      rc = bsc_expr_parse(pf, flags, 0, &var);
      /* prepend the existing result to the new result */
      if (!pf_token_is(pf, pf_token_current, pfile_log_err, ")")
          && (result_ok == rc)) {
        rc = result_syntax;
      }
      break;
    default:
      rc = bsc_token_to_constant(pf, pfile_log_none, &var);
      if (result_invalid == rc) {
        rc = pf_token_to_value(pf, pfile_log_none, &var);
      }
      if (result_ok != rc) {
        pfile_log(pf, pfile_log_err, BSC_MSG_VALUE_EXPECTED);
      } 
      break;
  }
  if (result_ok == rc) {
    *dst = var;
    if (!skipnext) {
      pf_token_get(pf, pf_token_next);
    }
  }
  return rc;
}

/*
 * NAME
 *   operator_get
 *
 * DESCRIPTION
 *   get an operator
 *
 * PARAMETERS
 *   expr : expression
 *   pf   : returned by pfile_open()
 *
 * RETURN
 *   0      : no error
 *   EINVAL : no operator present
 *
 * NOTES
 */
static operator_t operator_get(pfile_t *pf, flag_t flags)
{
  static const struct {
    const char *token;
    operator_t     op;
  } operator_token_map[] = {
    {"+",    operator_add},
    {"-",    operator_sub},
    {"*",    operator_mul},
    {"/",    operator_div},
    {"<",    operator_lt},
    {"<=",   operator_le},
    {"=",    operator_eq},
    {"<>",   operator_ne},
    {"><",   operator_ne},
    {">=",   operator_ge},
    {">",    operator_gt},
    {"&",    operator_andb},
    {"|",    operator_orb},
    {"^",    operator_xorb},
    {"!!",   operator_logical},
    {"mod",  operator_mod},
    {"<<",   operator_shift_left},
    {">>",   operator_shift_right},
    {"&&",   operator_andl},
    {"||",   operator_orl},
    {"andb", operator_andb},
    {"orb",  operator_orb},
    {".",    operator_dot},
    {"(",    operator_subscript}
  };
  size_t ii;
  operator_t op;

  for (ii = 0; 
       (ii < COUNT(operator_token_map))
        && !pf_token_is(pf, pf_token_current, pfile_log_none,
          operator_token_map[ii].token); 
        ii++)
    ;
  if (ii == COUNT(operator_token_map)) {
    op = operator_null;
  } else {
    op = operator_token_map[ii].op;
    if (flags & EXPR_FLAG_IF) {
      if (pf_token_is(pf, pf_token_current, pfile_log_none, "&")) {
        op = operator_andl;
      } else if (pf_token_is(pf, pf_token_current, pfile_log_none, "|")) {
        op = operator_orl;
      }
    }
    if (flags & EXPR_FLAG_ASSIGN) {
      if (operator_eq == op) {
        op = operator_assign;
      }
    }
    pf_token_get(pf, pf_token_next);
  }
  return op;
}

/*
 * NAME
 *   expr_parse
 *
 * DESCRIPTION
 *   parse an expression
 *
 * PARAMETERS
 *   dst   : [out] holds result
 *   pf    : parse file
 *   lhvar : for assignments, this is the base left-hand variable
 *
 * RETURN
 *
 * NOTES
 */
result_t bsc_expr_parse(pfile_t *pf, flag_t flags, value_t lhvar, 
    value_t *dst)
{
  operator_t  op;
  expr_t     *exprstk;
  value_t     var;
  result_t    rc;

  exprstk = 0;
  op      = operator_null;
  do {
    if (lhvar) {
      var = lhvar;
      lhvar = 0;
      rc = result_ok;
      value_lock(var);
    } else {
      rc = value_get(pf, flags, &var);
    }
    if (result_ok == rc) {
      do {
        /* OK, let's figure this out! grab the next member of the structure */
        op = operator_get(pf, flags);

        if (operator_dot == op) {
          variable_def_t        def;
          const char           *ptr;
          variable_def_member_t mbr;
          variable_base_t       ofs;

          def = value_def_get(var);
          if (value_vflag_test(var, VARIABLE_FLAG_POINTER)) {
            value_flag_set(var, VARIABLE_FLAG_REFERENCE);
            mbr = variable_def_member_get(def);
            def = variable_def_member_def_get(mbr);
          }

          ptr = pf_token_get(pf, pf_token_current);
          ofs = 0;
          for (mbr = variable_def_member_get(def);
               mbr && strcmp(variable_def_member_tag_get(mbr), ptr);
               mbr = variable_def_member_link_get(mbr)) {
            ofs += variable_def_member_ct_get(mbr)
              * variable_def_member_sz_get(mbr);
          }
          if (!mbr) {
            pfile_log(pf, pfile_log_err, PFILE_MSG_NOT_FOUND, ptr);
            rc = result_not_found;
          } else {
            value_t vofs;

            pfile_constant_get(pf,
                value_const_get(value_baseofs_get(var)) + ofs, &vofs);
            value_baseofs_set(var, vofs);
            value_release(vofs);
            value_def_set(var, variable_def_member_def_get(mbr),
                variable_def_member_sz_get(mbr),
                variable_def_member_ct_get(mbr));
            pf_token_get(pf, pf_token_next);
          }
        } else if (operator_subscript == op) {
          bsc_subscript_parse(pf, var);
        }
      } while ((result_ok == rc) 
          && ((operator_dot == op) || (operator_subscript == op)));
      if (result_ok == rc) {
        /* if this is in an ``if'' statement, all binary ops become
           logical */
        if (flags & EXPR_FLAG_IF) {
          if (operator_andb == op) {
            op = operator_andl;
          } else if (operator_orb == op) {
            op = operator_orl;
          }
        }
      }
      rc = pfile_expr_push(pf, &exprstk, var, op);
      value_release(var);
      flags &= ~EXPR_FLAG_ASSIGN;
    }
  } while ((operator_null != op) && (result_ok == rc));
  if (exprstk) {
    if (result_ok == rc) {
      var = expr_val_get(exprstk);
      if (dst) {
        value_lock(var);
        *dst = var;
      }
    }
    expr_list_free(&exprstk);
  }
  return rc;
}

