/**********************************************************
 **
 ** jal_expr.c : JAL expression parsing
 **
 ** Copyright (c) 2004-2005, Kyle A. York
 ** All rights reserved
 **
 ***********************************************************/
#include <string.h>
#include "../libutils/mem.h"
#include "../libcore/pf_expr.h"
#include "jal_proc.h"
#include "jal_tokn.h"
#include "jal_vdef.h"
#include "jal_expr.h"

value_t jal_parse_value(pfile_t *pf)
{
  const char    *ptr;
  value_t        val;
  variable_def_t cast_def;

  ptr = pf_token_get(pf, pf_token_current);
  cast_def = pfile_variable_def_find(pf, pfile_log_none, ptr);
  if (cast_def) {
    pf_token_get(pf, pf_token_next);
    if ((variable_def_type_integer == variable_def_type_get(cast_def))
          && !variable_def_flag_test(cast_def, VARIABLE_DEF_FLAG_BIT)
          && (1 == variable_def_sz_get(cast_def))
          && pf_token_is(pf, pf_token_current, pfile_log_none, "*")) {
      value_t cval;

      pf_token_get(pf, pf_token_next);
      cval = jal_parse_expr(pf);
      if (!value_is_const(cval)) {
        pfile_log(pf, pfile_log_err, "constant expression expected");
        value_release(cval);
      } else {
        cast_def = variable_def_alloc(0,
            variable_def_type_get(cast_def),
            variable_def_flags_get_all(cast_def),
            value_const_get(cval));
      }
      value_release(cval);
    }
    if (pf_token_is(pf, pf_token_current, pfile_log_err, "(")) {
      pf_token_get(pf, pf_token_next);
    }
    val = jal_parse_expr(pf);
    if (pf_token_is(pf, pf_token_current, pfile_log_err, ")")) {
      pf_token_get(pf, pf_token_next);
    }
    if (val) {
      if (value_sz_get(val) >= variable_def_sz_get(cast_def)) {
        value_def_set(val, cast_def);
      } else {
        /* if we're casting to a larger value, we need to sock this
         * away into a temporary */
        value_t tmp;

        tmp = pfile_value_temp_get(pf, variable_def_type_get(cast_def),
            variable_def_sz_get(cast_def));
        pfile_cmd_op_add_assign(pf, operator_assign, &tmp, val);
        value_release(val);
        val = tmp;
      }
    }
  } else {
    value_t   jvals[JAL_VAL_TYPE_CT];
    boolean_t rc;
    flag_t    parse_flags;

    parse_flags = JAL_PARSE_CALL_FLAG_NONE;
    rc  = jal_value_find(pf, ptr, jvals);
    val = VALUE_NONE;
    if (result_ok == rc) {
      if (jvals[JAL_VAL_TYPE_GET]) {
        val = jvals[JAL_VAL_TYPE_GET];
        if (variable_def_type_pointer == value_type_get(val)) {
          value_dereference(val);
        }
        parse_flags = JAL_PARSE_CALL_FLAG_GET;
      } else if (jvals[JAL_VAL_TYPE_BASE]) {
        val = jvals[JAL_VAL_TYPE_BASE];
      } else if (jvals[JAL_VAL_TYPE_PUT]) {
        pfile_log(pf, pfile_log_err, 
            "Attempted read from an OUT volatile parameter (%s)", ptr);
      }
    } else {
#if 0
      pfile_log(pf, pfile_log_err, "syntax error before: %s",
          pf_token_get(pf, pf_token_current));
#endif
    }
    if (val) {
      value_lock(val);
      pf_token_get(pf, pf_token_next);
      if (variable_def_type_function == value_type_get(val)) {
        value_t tval;

        tval = jal_parse_call(pf, val, parse_flags);
        value_release(val);
        val = tval;
        if (!tval) {
          pfile_log(pf, pfile_log_err, "no return value");
        }
      } else if (value_is_array(val)) {
        if (pf_token_is(pf, pf_token_current, pfile_log_none, "[")) {
          value_t n;

          pf_token_get(pf, pf_token_next);
          n = jal_parse_expr(pf);
          if (n) {
            /* n *= sz */
            value_t tmp1;

            value_use_ct_bump(val, ctr_bump_incr);
            tmp1 = value_clone(val);
            value_dereference(tmp1);

            if (value_is_const(n) 
                && (value_const_get(n) >= value_ct_get(val))) {
              pfile_log(pf, pfile_log_err, "subscript out of range");
            }
            if ((value_sz_get(tmp1) == 1) || value_is_const(tmp1)) {
              value_baseofs_set(tmp1, n);
            } else if (value_is_const(n)) {
              variable_const_t nc;

              nc = value_sz_get(tmp1) * value_const_get(n);
              value_release(n);
              n = pfile_constant_get(pf, nc, VARIABLE_DEF_NONE);
              value_baseofs_set(tmp1, n);
            } else {
              value_t sz;
              value_t tmp;

              sz = pfile_constant_get(pf, value_sz_get(tmp1), 
                  VARIABLE_DEF_NONE);
              tmp = VALUE_NONE;
              pfile_cmd_op_add(pf, operator_mul, &tmp, n, sz);
              value_release(sz);
              value_baseofs_set(tmp1, tmp);
              value_release(tmp);
            }
            value_release(n);
            value_release(val);
            val = tmp1;
          }
          if (pf_token_is(pf, pf_token_current, pfile_log_err, "]")) {
            pf_token_get(pf, pf_token_next);
          }
        }
      }
    }
    jal_value_release(jvals);
  }
  return val;
}

/*
 * NAME
 *   value_get
 *
 * DESCRIPTION
 *   get an expression's value
 *
 * PARAMETERS
 *   pf    : return from pfile_open()
 *   var   : [out] holds result
 *
 * RETURN
 *
 * NOTES
 */
static value_t value_get(pfile_t *pf)
{
  value_t     val;

  /* valid values: 
       constant ('%' or '$' or decimal #)
       variable name (alpha + alpha|digit|'_')
       '(' -- subfunction
       '!' -- logical NOT
       '-' -- negation (two's complement)
       '~' -- complement (one's complement) */
  val = VALUE_NONE;
  if (pf_token_is(pf, pf_token_current, pfile_log_none, "count")) {
    if (pf_token_is(pf, pf_token_next, pfile_log_err, "(")) {
      value_t tmp;

      pf_token_get(pf, pf_token_next);
      if (result_ok == pf_token_to_value(pf, pfile_log_err, &tmp)) {
        if (!value_is_array(tmp)) {
          pfile_log(pf, pfile_log_err, "\"%s\" is not an array",
              value_name_get(tmp));
        } else {
          val = pfile_constant_get(pf,
              variable_def_member_ct_get(
                variable_def_member_get(
                  value_def_get(
                    tmp))),
              VARIABLE_DEF_NONE);
        }
        value_release(tmp);
        pf_token_get(pf, pf_token_next);
      }
      if (pf_token_is(pf, pf_token_current, pfile_log_err, ")")) {
        pf_token_get(pf, pf_token_next);
      }
    } 
  } else if (pf_token_is(pf, pf_token_current, pfile_log_none, "whereis")) {
    if (pf_token_is(pf, pf_token_next, pfile_log_err, "(")) {
      pf_token_get(pf, pf_token_next);
    }
    if (!jal_token_is_identifier(pf)) {
      pfile_log(pf, pfile_log_err, "identifier expected");
    } else {
      const char *ptr;
      label_t     lbl;

      ptr = pf_token_get(pf, pf_token_current);
      lbl = pfile_label_find(pf, pfile_log_none, ptr);
      if (!lbl) {
        pfile_log(pf, pfile_log_err, "identifier not found");
      } else {
        variable_def_t def;
        variable_t     var;

        def = variable_def_alloc(0, variable_def_type_label,
            VARIABLE_DEF_FLAG_VOLATILE, 2);
        var = variable_alloc(TAG_NONE, def);
        variable_label_set(var, lbl);
        val = value_alloc(var);
        variable_release(var);
        label_release(lbl);
      }
      pf_token_get(pf, pf_token_next);
    }
    if (pf_token_is(pf, pf_token_current, pfile_log_err, ")")) {
      pf_token_get(pf, pf_token_next);
    }
  } else if (pf_token_is(pf, pf_token_current, pfile_log_none, "defined")) {
    if (pf_token_is(pf, pf_token_next, pfile_log_err, "(")) {
      pf_token_get(pf, pf_token_next);
    }
    if (!jal_token_is_identifier(pf)) {
      pfile_log(pf, pfile_log_err, "identifier expected");
    } else {
      const char      *ptr;
      label_t          lbl;
      value_t          tst_val;

      ptr     = pf_token_get(pf, pf_token_current);
      lbl     = pfile_label_find(pf, pfile_log_none, ptr);
      tst_val = pfile_value_find(pf, pfile_log_none, ptr);

      value_release(tst_val);
      label_release(lbl);
      val = pfile_constant_get(pf,
         (lbl != LABEL_NONE) || (tst_val != VALUE_NONE),
          VARIABLE_DEF_NONE);
      pf_token_get(pf, pf_token_next);
    }
    if (pf_token_is(pf, pf_token_current, pfile_log_err, ")")) {
      pf_token_get(pf, pf_token_next);
    }
  } else {
    const char *ptr;
    operator_t  op;

    ptr = pf_token_get(pf, pf_token_current);
    op = operator_null;
    switch (*ptr) {
      case '!':
        op = ('!' == ptr[1]) ? operator_logical : operator_cmpb;
        /* fall through */
      case '-':
      case '+':
        if ('-' == *ptr) {
          op = operator_neg;
        }
        pf_token_get(pf, pf_token_next); /* this one's done */
        val = value_get(pf);   /* var is the next value */
        /* create the expression:
         *   _t# = var op
         *   and return _t#
         */
        if (val && (operator_null != op)) {
          value_t tmp;

          if ((operator_neg == op) && value_is_const(val)) {
            variable_def_t      def;
            variable_sz_t       sz;
            variable_def_type_t type;
            flag_t              flags;
            variable_const_t    n;

            n = -value_const_get(val);

            if (value_name_get(val)) {
              sz    = value_sz_get(val);
              type  = variable_def_type_integer;
              flags = VARIABLE_DEF_FLAG_SIGNED;
            } else {
              variable_calc_sz_min(n, &sz, &type, &flags);
            }
            def = variable_def_alloc(0, type, flags, sz);
            tmp = pfile_constant_get(pf, n, def);
            value_release(val);
            val = tmp;
          } else {
            tmp = VALUE_NONE;
            pfile_cmd_op_add(pf, op, &tmp, val, 0);
            value_release(val);
            val = tmp;
          }
        }
        break;
      case '(':
        pf_token_get(pf, pf_token_next);
        val = jal_parse_expr(pf);
        /* prepend the existing result to the new result */
        if (pf_token_is(pf, pf_token_current, pfile_log_err, ")")) {
          pf_token_get(pf, pf_token_next);
        }
        break;
      default:
        if (jal_token_is_identifier(pf)) {
          val = jal_parse_value(pf);
        } else {
          val = jal_token_to_constant(pf, pfile_log_err);
          if (val) {
            pf_token_get(pf, pf_token_next);
          }
        }
        break;
    }
  }
  return val;
}

/*
 * NAME
 *   operator_get
 *
 * DESCRIPTION
 *   get an operator
 *
 * PARAMETERS
 *   expr : expression
 *   pf   : returned by pfile_open()
 *
 * RETURN
 *   0      : no error
 *   EINVAL : no operator present
 *
 * NOTES
 */
static operator_t operator_get(pfile_t *pf)
{
  static const struct {
    const char *token;
    operator_t     op;
  } operator_token_map[] = {
    {"*",    operator_mul},
    {"/",    operator_div},
    {"%",    operator_mod},
    {"+",    operator_add},
    {"-",    operator_sub},
    {"<<",   operator_shift_left},
    {">>",   operator_shift_right},
    {"<",    operator_lt},
    {"<=",   operator_le},
    {"==",   operator_eq},
    {"!=",   operator_ne},
    {">=",   operator_ge},
    {">",    operator_gt},
    {"&",    operator_andb},
    {"|",    operator_orb},
    {"^",    operator_xorb},
    {"!!",   operator_logical}
  };
  size_t ii;
  operator_t op;

  for (ii = 0; 
       (ii < COUNT(operator_token_map))
        && !pf_token_is(pf, pf_token_current, pfile_log_none,
          operator_token_map[ii].token); 
        ii++)
    ;
  if (ii == COUNT(operator_token_map)) {
    op = operator_null;
  } else {
    op = operator_token_map[ii].op;
    pf_token_get(pf, pf_token_next);
  }
  return op;
}

/*
 * NAME
 *   expr_parse
 *
 * DESCRIPTION
 *   parse an expression
 *
 * PARAMETERS
 *   dst   : [out] holds result
 *   pf    : parse file
 *   lhvar : for assignments, this is the base left-hand variable
 *
 * RETURN
 *
 * NOTES
 */
value_t jal_parse_expr(pfile_t *pf)
{
  operator_t  op;
  expr_t     *exprstk;
  value_t     val;

  exprstk = 0;
  op      = operator_null;
  do {
    val = value_get(pf);
    op = operator_get(pf);
    if (val) {
      pfile_expr_push(pf, &exprstk, val, op);
      value_release(val);
    }
  } while (operator_null != op);
  if (exprstk) {
    val = expr_val_get(exprstk);
    value_lock(val);
    expr_list_free(&exprstk);
  }
  return val;
}

