/************************************************************
 **
 ** jal_incl.c : JAL include and pragma definitions
 **
 ** Copyright (c) 2004-2006, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/
#include <string.h>
#include <errno.h>
#include <assert.h>
#include "../libutils/mem.h"
#include "../libcore/pf_cmd.h"
#include "../libcore/pf_msg.h"
#include "../libcore/pf_proc.h"
#include "../libpic12/pic.h"
#include "jal_expr.h"
#include "jal_tokn.h"
#include "jal_vdef.h"
#include "jal_incl.h"

typedef struct jal_pragma_fuses_entry_
{
  struct jal_pragma_fuses_entry_ *link;
  const char                     *name;
  unsigned                        bits;
} jal_pragma_fuses_entry_t;

typedef struct jal_pragma_fuses_ {
  struct jal_pragma_fuses_ *link;
  const char               *name;
  unsigned                  mask;
  unsigned                  config_word; /* which config word */
  jal_pragma_fuses_entry_t *entry;
} jal_pragma_fuses_t;

/* list of known jal fuses */

static jal_pragma_fuses_t *jal_fuses;


void jal_parse_include(pfile_t *pf, const pfile_pos_t *statement_start)
{
  char include_name[FILENAME_MAX];
  const char *ptr;

  ptr = pf_token_get(pf, pf_token_current);
  strcpy(include_name, ptr);
  strcat(include_name, ".jal");
  pf_token_get(pf, pf_token_next);

  pfile_include_process(pf, strlen(include_name), include_name);
}

/* check that the currently compiled file has the correct name.
   of what possible use is this??? */
static void jal_parse_pragma_name(pfile_t *pf)
{
  pfile_source_t *src;
  const char     *name;
  const char     *ptr;
  size_t          sz;

  src  = pfile_source_get(pf);
  name = pfile_source_name_get(src);
  for (sz = strlen(name);
       sz && ('/' != name[sz-1]) && ('\\' != name[sz-1]);
       sz--)
    ; /* null body */
  ptr = pf_token_get(pf, pf_token_current);
  while (*ptr && name[sz] && (*ptr == name[sz])) {
    sz++;
    ptr++;
  }
  if (*ptr || ('.' != name[sz])) {
    pfile_log(pf, pfile_log_err, "wrong file");
  }
  pf_token_get(pf, pf_token_next);
}

/* force an error to occur */
static void jal_parse_pragma_error(pfile_t *pf)
{
  if (!pfile_codegen_disable_get(pf)) {
    pfile_log(pf, pfile_log_err, "pragma error failed");
  }
}

/* this is analogous to:
   const target_chip = pic_* */
static void jal_parse_pragma_target_chip(pfile_t *pf)
{
  const char *ptr;
  char       *chip;

  ptr = pf_token_get(pf, pf_token_current);
  chip = MALLOC(5 + strlen(ptr));
  if (!chip) {
    pfile_log_syserr(pf, ENOMEM);
  } else {
    value_t src;

    sprintf(chip, "pic_%s", ptr);
    src = pfile_value_find(pf, pfile_log_err, chip);
    if (src) {
      if (!value_is_const(src)) {
        pfile_log(pf, pfile_log_err, "pic_* must all be constant");
      } else {
        jal_variable_info_t inf;
        value_t            *name;
        size_t              ii;

        jal_variable_info_init(&inf);
        inf.name      = "target_chip";
        inf.ct        = 0;
        inf.def_flags = VARIABLE_DEF_FLAG_CONST;

        variable_release(
            jal_variable_alloc(pf, &inf, boolean_false, 1, &src,
              PFILE_VARIABLE_ALLOC_GLOBAL));

        /* also, create 'target_chip_name', an array of bytes that will hold
         * the chip name (for inclusion in the assembly file)
         */
        inf.name      = "target_chip_name";
        inf.ct        = strlen(ptr);
        
        name = MALLOC(sizeof(*name) * inf.ct);
        for (ii = 0; ii < inf.ct; ii++) {
          name[ii] = pfile_constant_get(pf, ptr[ii], VARIABLE_DEF_NONE);
        }
        variable_release(
            jal_variable_alloc(pf, &inf, boolean_false, inf.ct,
              name, PFILE_VARIABLE_ALLOC_GLOBAL));
        for (ii = 0; ii < inf.ct; ii++) {
          value_release(name[ii]);
        }
        FREE(name);
      }
      value_release(src);
    }
    FREE(chip);
  }
  pf_token_get(pf, pf_token_next);
}

/* this is analogous to:
   const target_clock = cexpr */
static void jal_parse_pragma_target_clock(pfile_t *pf)
{
  value_t cexpr;

  cexpr = jal_parse_expr(pf);
  if (cexpr) {
    if (!value_is_const(cexpr)) {
      pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
    } else {
      jal_variable_info_t inf;

      jal_variable_info_init(&inf);
      inf.name      = "target_clock";
      inf.ct        = 0;
      inf.def_flags = VARIABLE_DEF_FLAG_CONST;

      variable_release(
          jal_variable_alloc(pf, &inf, boolean_false, 1, &cexpr,
            PFILE_VARIABLE_ALLOC_GLOBAL));
    }
    value_release(cexpr);
  }
}

static void jal_parse_pragma_target_fuses(pfile_t *pf)
{
  value_t cexpr;

  cexpr = jal_parse_expr(pf);
  if (!cexpr || !value_is_const(cexpr)) {
    pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
  } else {
    value_t n;

    n = pfile_value_find(pf, pfile_log_err, "_fuses");
    if (n) {
      if (value_is_array(n)) {
        if (value_const_get(cexpr) >= value_ct_get(n)) {
          pfile_log(pf, pfile_log_err, "value out of range");
          value_release(n);
          n = VALUE_NONE;
        } else {
          value_t tmp;

          tmp = value_subscript_set(n,
              (variable_ct_t) value_const_get(cexpr));
          value_release(cexpr);
          value_release(n);
          n = tmp;
          cexpr = jal_parse_expr(pf);
          if (!cexpr || !value_is_const(cexpr)) {
            pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
          }
        }
      }
      if (n) {
        value_const_set(n, value_const_get(cexpr));
        value_release(n);
      }
    }
  }
  value_release(cexpr);
}

static void jal_parse_pragma_target(pfile_t *pf)
{
  static const struct {
    const char *tag;
    void      (*action)(pfile_t *);
  } targets[] = {
    {"chip",          jal_parse_pragma_target_chip},
    {"clock",         jal_parse_pragma_target_clock},
    {"fuses",         jal_parse_pragma_target_fuses}
  };
  size_t cmd;

  for (cmd = 0; 
       (cmd < COUNT(targets)) 
       && !pf_token_is(pf, pf_token_current, pfile_log_none, 
         targets[cmd].tag); 
       cmd++)
    ; /* null body */
  if (cmd < COUNT(targets)) {
    pf_token_get(pf, pf_token_next);
    targets[cmd].action(pf);
  } else {
    const jal_pragma_fuses_t *fuses;

    for (fuses = jal_fuses;
         fuses && !pf_token_is(pf, pf_token_current, pfile_log_none,
           fuses->name);
         fuses = fuses->link)
      ;
    if (!fuses) {
      pfile_log(pf, pfile_log_err, "unknown pragma target: %s",
        pf_token_ptr_get(pf));
    } else {
      /* figure out which one it is */
      const jal_pragma_fuses_entry_t *entry;

      pf_token_get(pf, pf_token_next);
      for (entry = fuses->entry;
           entry && !pf_token_is(pf, pf_token_current, pfile_log_none,
             entry->name);
           entry = entry->link)
        ;
      pf_token_get(pf, pf_token_next);
      if (!entry) {
        pfile_log(pf, pfile_log_err, "unknown pragma target: %s %s",
          fuses->name, pf_token_ptr_get(pf));
      } else {
        /* _fuses = (_fuses & fuses->mask) | entry->bits */
        value_t n;

        n = pfile_value_find(pf, pfile_log_err, "_fuses");
        if (n) {
          /* n can be either a simple constant, or an array of constants */
          if (value_is_array(n)) {
            value_t m;

            m = value_subscript_set(n, fuses->config_word);
            value_release(n);
            n = m;
          }
          value_const_set(n, 
              (value_const_get(n) & ~fuses->mask) | entry->bits);
          value_release(n);
        }
      }
    }
  }
}

/* note that the current function is part of the interrupt chain */
static void jal_parse_pragma_interrupt(pfile_t *pf)
{
  pfile_proc_t *proc;

  proc = pfile_proc_active_get(pf);

  if (!pfile_proc_parent_get(proc)) {
    pfile_log(pf, pfile_log_err,
        "pragma interrupt can only be used in a procedure!");
  } else if ((pfile_proc_param_ct_get(proc) > 1)
    || pfile_proc_return_def_get(proc)) {
    pfile_log(pf, pfile_log_err, 
        "An interrupt entry must be a procedure with no parameters!");
  } else {
    pfile_proc_flag_set(proc, PFILE_PROC_FLAG_INTERRUPT);
    label_usage_bump(pfile_proc_label_get(proc), ctr_bump_incr);
  }
}

/* note that the current function contains a jump table.
   nb : i'll likely ignore this and instead look for 
        ``addwf pcl,f; keeping everthing from there to the end
        of either the procedure or block in one bank */
static void jal_parse_pragma_jump_table(pfile_t *pf)
{
  pfile_proc_t *proc;

  proc = pfile_proc_active_get(pf);

  if (!pfile_proc_parent_get(proc)) {
    pfile_log(pf, pfile_log_err,
        "pragma jump_table can only be used in procedures & functions!");
  } else {
    if (pfile_flag_test(pf, PFILE_FLAG_WARN_MISC)) {
      pfile_log(pf, pfile_log_warn,
          "pragma jump_table has no effect. Use lookup tables instead.");
    }
  }
}

/* note that the BANK and PAGE bit must not be optimized away */
static void jal_parse_pragma_keep(pfile_t *pf)
{
  pf_token_get_t which;

  which = pf_token_current;
  do {
    if (pf_token_is(pf, which, pfile_log_none, "page")) {
      pf_token_get(pf, pf_token_next);
      pfile_proc_flag_set(pfile_proc_active_get(pf), 
          PFILE_PROC_FLAG_KEEP_PAGE);
    } else if (pf_token_is(pf, pf_token_current, pfile_log_none, "bank")) {
      pfile_proc_flag_set(pfile_proc_active_get(pf), 
          PFILE_PROC_FLAG_KEEP_BANK);
      pf_token_get(pf, pf_token_next);
    } else {
      pfile_log(pf, pfile_log_err, "PAGE or BANK expected");
    }
    which = pf_token_next;
  } while (pf_token_is(pf, pf_token_current, pfile_log_none, ","));
}

/* eeprom is implemented using two internal variables
     _eeprom      is an array of bytes
     _eeprom_base is where the eeprom goes
     _eeprom_used is a counter */
static void jal_parse_pragma_eeprom(pfile_t *pf)
{
  value_t start;

  start = jal_parse_expr(pf);
  if (start) {
    if (!value_is_const(start)) {
      pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
    } else if (pf_token_is(pf, pf_token_current, pfile_log_err, ",")) {
      value_t sz;

      pf_token_get(pf, pf_token_next);
      sz = jal_parse_expr(pf);
      if (sz) {
        if (!value_is_const(sz)) {
          pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
        } else if (!value_const_get(sz)) {
          pfile_log(pf, pfile_log_err, "size must be > 0");
        } else {
          variable_t eeprom;
          
          eeprom = pfile_variable_find(pf, pfile_log_none, "_eeprom", 0);
          if (eeprom) {
            pfile_log(pf, pfile_log_err, "eeprom already defined");
            variable_release(eeprom);
          } else {
            variable_def_t vdef; /* variable definition */
            variable_def_t mdef; /* array member        */

            mdef = variable_def_alloc(0, variable_def_type_integer,
                VARIABLE_DEF_FLAG_NONE, 1);
            vdef = variable_def_alloc(0, variable_def_type_array,
                VARIABLE_DEF_FLAG_CONST, 0);  
            variable_def_member_add(vdef, 0, mdef, 
				(variable_ct_t) value_const_get(sz));

            pfile_variable_alloc(pf,
                  PFILE_VARIABLE_ALLOC_GLOBAL,
                  "_eeprom", vdef, VARIABLE_NONE, 0);

            vdef = variable_def_alloc(0, variable_def_type_integer,
                VARIABLE_DEF_FLAG_CONST, 2);

            pfile_variable_alloc(pf,
                PFILE_VARIABLE_ALLOC_GLOBAL,
                "_eeprom_used", vdef, VARIABLE_NONE, 0);

            if (result_ok == pfile_variable_alloc(pf,
                PFILE_VARIABLE_ALLOC_GLOBAL,
                "_eeprom_base", vdef, VARIABLE_NONE, &eeprom)) {
              variable_const_set(eeprom, variable_def_get(eeprom),
                  0, value_const_get(start));
              variable_release(eeprom);
            }
          }
        }
        value_release(sz);
      }
    }
    value_release(start);
  }
}

static void jal_parse_pragma_data(pfile_t *pf)
{
  boolean_t is_first;

  is_first = boolean_true;
  do {
    value_t val;

    if (is_first) {
      is_first = boolean_false;
    } else {
      pf_token_get(pf, pf_token_next);
    }

    val = jal_token_to_constant(pf, pfile_log_err);
    if (val) {
      size_t  lo;
      size_t  hi;

      lo = value_const_get(val);
      hi = lo;

      value_release(val);
      val = VALUE_NONE;
      if (pf_token_is(pf, pf_token_next, pfile_log_none, "-")) {
        hi = -1;
        pf_token_get(pf, pf_token_next);
        val = jal_token_to_constant(pf, pfile_log_err);
        if (val) {
          pf_token_get(pf, pf_token_next);
          hi = value_const_get(val);
        }
      }
      if (-1 != hi) {
        pic_blist_add(pf, &pic_blist, lo, hi);
      }
    }
    value_release(val);
  } while (pf_token_is(pf, pf_token_current, pfile_log_none, ","));
}

static void jal_parse_pragma_code_or_stack(pfile_t *pf, char *name)
{
  value_t sz;

  sz = jal_parse_expr(pf);
  if (sz) {
    if (!value_is_const(sz)) {
      pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
    } else if (!value_const_get(sz)) {
      pfile_log(pf, pfile_log_err, "size must be > 0");
    } else {
      jal_variable_info_t inf;

      jal_variable_info_init(&inf);
      inf.name      = name;
      inf.ct        = 0;
      inf.def_flags = VARIABLE_DEF_FLAG_CONST;

      variable_release(
          jal_variable_alloc(pf, &inf, boolean_false, 1, &sz,
            PFILE_VARIABLE_ALLOC_GLOBAL));
    }
    value_release(sz);
  }
}

static void jal_parse_pragma_code(pfile_t *pf)
{
  jal_parse_pragma_code_or_stack(pf, "_code_size");
}

static void jal_parse_pragma_stack(pfile_t *pf)
{
  jal_parse_pragma_code_or_stack(pf, "_stack_size");
}

static void jal_parse_pragma_eedata(pfile_t *pf)
{
  value_t   eeprom;
  value_t   eeprom_used;
  value_t   val;
  boolean_t full;
  size_t    eeprom_ct;

  eeprom = pfile_value_find(pf, pfile_log_err, "_eeprom");
  eeprom_ct   = value_ct_get(eeprom);
  eeprom_used = pfile_value_find(pf, pfile_log_err, "_eeprom_used");
  value_dereference(eeprom);
  value_baseofs_set(eeprom, eeprom_used);
  val = VALUE_NONE;
  /* determine if we're already full so we only report the error once */
  full = value_const_get(eeprom_used) == value_sz_get(eeprom);
  do {
    const char *ptr;

    if (val) {
      pf_token_get(pf, pf_token_next); /* skip the comma */
    }
    ptr = pf_token_ptr_get(pf);
    if ('"' == *ptr) {
      size_t sz;

      sz = pf_token_sz_get(pf);
      sz--;
      ptr++;
      if (sz && ('"' == ptr[sz-1])) {
        sz--;
      }
      while (sz--) {
        if (value_const_get(eeprom_used) == eeprom_ct) {
          if (!full) {
            pfile_log(pf, pfile_log_err, "eeprom is full");
          }
          full = boolean_true;
        } else {
          value_const_set(eeprom, *ptr);
          value_const_set(eeprom_used, value_const_get(eeprom_used) + 1);
        }
        ptr++;
      }
      pf_token_get(pf, pf_token_next);
      val = eeprom; /* set to a non-zero dummy value */
    } else {
      val = jal_parse_expr(pf);
      if (val) {
        if (!value_is_const(val)) {
          pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
        } else if (value_const_get(val) >= 256) {
          pfile_log(pf, pfile_log_err, "value must be < 256");
        } else if (eeprom) {
          if (value_const_get(eeprom_used) == eeprom_ct) {
            if (!full) {
              pfile_log(pf, pfile_log_err, "eeprom is full");
            }
          } else {
            value_const_set(eeprom, value_const_get(val));
            value_const_set(eeprom_used, value_const_get(eeprom_used) + 1);
          }
        }
        value_release(val);
      }
    }
  } while (val && pf_token_is(pf, pf_token_current, pfile_log_none, ","));
  value_release(eeprom_used);
  value_release(eeprom);
}

/*
 * pragma fuse_def tag mask '{'
 *   tag '=' bits
 *   ...
 * '}'
 */ 
static void jal_parse_pragma_fuse_def(pfile_t *pf)
{
  jal_pragma_fuses_t *fuse;
  const char         *ptr;
  size_t              sz;
  value_t             val;

  ptr = pf_token_get(pf, pf_token_current);
  sz  = 1 + strlen(ptr);

  fuse = MALLOC(sizeof(*fuse) + sz);
  if (!fuse) {
    pfile_log_syserr(pf, ENOMEM);
  } else {
    fuse->link        = jal_fuses;
    jal_fuses         = fuse;
    fuse->name        = (void *) (fuse + 1);
    memcpy(fuse + 1, ptr, sz);
    fuse->mask        = 0;
    fuse->entry       = 0;
    fuse->config_word = 0;
  }
  if (pf_token_is(pf, pf_token_next, pfile_log_none, ":")) {
    /* set the config_word entry */
    value_t fuses;

    pf_token_get(pf, pf_token_next);
    val = jal_parse_expr(pf);

    fuses = pfile_value_find(pf, pfile_log_err, "_fuses");
    if (!val || !value_is_const(val)) {
      pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
    } else if (value_const_get(val) >= value_ct_get(fuses)) {
      pfile_log(pf, pfile_log_err, "value out of range");
    } else if (fuse) {
      fuse->config_word = value_const_get(val);
    }
    value_release(val);
    value_release(fuses);
  }

  val = jal_parse_expr(pf);
  if (!val || !value_is_const(val)) {
    pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
  } else if (fuse) {
    fuse->mask = value_const_get(val);
  }
  value_release(val);
  if (pf_token_is(pf, pf_token_current, pfile_log_err, "{")) {
    pf_token_get(pf, pf_token_next);
    while (!pf_token_is_eof(pf)
        && !pf_token_is(pf, pf_token_current, pfile_log_none, "}")) {
      jal_pragma_fuses_entry_t *entry;

      ptr = pf_token_get(pf, pf_token_current);
      if (!fuse) {
        entry = 0;
      } else {
        sz  = 1 + strlen(ptr);

        entry = MALLOC(sizeof(*entry) + sz);
        if (entry) {
          entry->link = fuse->entry;
          fuse->entry = entry;
          entry->name = (void *) (entry + 1);
          memcpy(entry + 1, ptr, sz);
          entry->bits = 0;
        }
      }
      if (pf_token_is(pf, pf_token_next, pfile_log_err, "=")) {
        pf_token_get(pf, pf_token_next);
        val = jal_parse_expr(pf);
        if (!val || !value_is_const(val)) {
          pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
        } else if (entry) {
          entry->bits = value_const_get(val);
        }
        value_release(val);
      }
    }
  }
  pf_token_get(pf, pf_token_next);
}

static void jal_parse_pragma_inline(pfile_t *pf)
{
  pfile_proc_t *proc;

  proc = pfile_proc_active_get(pf);

  if (!pfile_proc_parent_get(proc)) {
    pfile_log(pf, pfile_log_err,
        "pramga inline can only be used in a procedure!");
  } else {
    pfile_proc_flag_set(proc, PFILE_PROC_FLAG_INLINE);
  }
}

static void jal_parse_pragma_task(pfile_t *pf)
{
  value_t task_ct;

  task_ct = jal_parse_expr(pf);
  if (task_ct) {
    if (!value_is_const(task_ct)) {
      pfile_log(pf, pfile_log_err, "constant expression expected");
    } else if (value_const_get(task_ct) < 1) {
      pfile_log(pf, pfile_log_err, "range error");
    } else {
      pfile_task_ct_set(pf, value_const_get(task_ct));
    }
    value_release(task_ct);
  }
}

typedef enum pragma_action_ {
  PRAGMA_ACTION_NONE = 0, /* unused                  */
  PRAGMA_ACTION_FOLLOW,   /* follow the 'child' link */
  PRAGMA_ACTION_FLAG,     /* use mask/set on a flag  */
  PRAGMA_ACTION_FN,       /* execute [fn]            */
  PRAGMA_ACTION_EOL       /* end of list             */
} pragma_action_t;

typedef struct pragma_word_ {
  const char                *key;
  pragma_action_t            action;
  const struct pragma_word_ *child;
  struct {
    flag_t                   mask;
    flag_t                   set;
  } flag;
  void                     (*fn)(pfile_t *pf);
} pragma_word_t;

#define DEFINE_FN(key, fn) \
  { key, PRAGMA_ACTION_FN, 0, {0, 0}, fn }
#define DEFINE_FOLLOW(key, child) \
  { key, PRAGMA_ACTION_FOLLOW, child, {0, 0}, 0 }
#define DEFINE_FLAG(key, mask, set) \
  { key, PRAGMA_ACTION_FLAG, 0, {mask, set}, 0 }
#define DEFINE_END_OF_LIST \
  { 0, PRAGMA_ACTION_EOL, 0, {0, 0}, 0 }

static const pragma_word_t pragma_clear[] = {
  DEFINE_FLAG("yes",     
    ~PFILE_FLAG_MISC_CLEAR_BSS, 
    PFILE_FLAG_MISC_CLEAR_BSS),
  DEFINE_FLAG("no",      
    ~PFILE_FLAG_MISC_CLEAR_BSS, 
    0),
  DEFINE_FLAG(0,         
    ~PFILE_FLAG_MISC_CLEAR_BSS, 
    0),
  DEFINE_END_OF_LIST
};

static const pragma_word_t pragma_bootloader[] = {
  DEFINE_FLAG("rickpic", 
      ~(PFILE_FLAG_BOOT_RICK | PFILE_FLAG_BOOT_LONG_START), 
      PFILE_FLAG_BOOT_RICK),
  DEFINE_FLAG("long_start", 
    ~(PFILE_FLAG_BOOT_RICK | PFILE_FLAG_BOOT_LONG_START), 
    PFILE_FLAG_BOOT_LONG_START),
  DEFINE_FLAG(0,
    ~(PFILE_FLAG_BOOT_RICK | PFILE_FLAG_BOOT_LONG_START), 
    0),
  DEFINE_END_OF_LIST
};

static const pragma_word_t pragma_opt_expr_reduce[] = {
  DEFINE_FLAG("yes",     
    ~PFILE_FLAG_OPT_EXPR_REDUCTION, 
    PFILE_FLAG_OPT_EXPR_REDUCTION ),
  DEFINE_FLAG("no",      
    ~PFILE_FLAG_OPT_EXPR_REDUCTION,
    0 ),
  DEFINE_FLAG(0,         
    ~PFILE_FLAG_OPT_EXPR_REDUCTION, 
    PFILE_FLAG_OPT_EXPR_REDUCTION ),
  DEFINE_END_OF_LIST
};

static const pragma_word_t pragma_opt_cexpr_reduce[] = {
  DEFINE_FLAG("yes",     
    ~PFILE_FLAG_OPT_CEXPR_REDUCTION, 
    PFILE_FLAG_OPT_CEXPR_REDUCTION ),
  DEFINE_FLAG("no",      
    ~PFILE_FLAG_OPT_CEXPR_REDUCTION, 
    0 ),
  DEFINE_FLAG(0,         
    ~PFILE_FLAG_OPT_CEXPR_REDUCTION, 
    PFILE_FLAG_OPT_CEXPR_REDUCTION ),
  DEFINE_END_OF_LIST
};

static const pragma_word_t pragma_opt_temp_reduce[] = {
  DEFINE_FLAG("yes",     
    ~PFILE_FLAG_OPT_TEMP_REDUCE, 
    PFILE_FLAG_OPT_TEMP_REDUCE ),
  DEFINE_FLAG("no",     
    ~PFILE_FLAG_OPT_TEMP_REDUCE, 
    0 ),
  DEFINE_FLAG(0,         
    ~PFILE_FLAG_OPT_TEMP_REDUCE, 
    0 ),
  DEFINE_END_OF_LIST
};

static const pragma_word_t pragma_opt_variable_reduce[] = {
  DEFINE_FLAG("yes",     
    ~PFILE_FLAG_OPT_VARIABLE_REDUCE, 
    PFILE_FLAG_OPT_VARIABLE_REDUCE ),
  DEFINE_FLAG("no",      
    ~PFILE_FLAG_OPT_VARIABLE_REDUCE, 
    0 ),
  DEFINE_FLAG(0,         
    ~PFILE_FLAG_OPT_VARIABLE_REDUCE, 
    PFILE_FLAG_OPT_VARIABLE_REDUCE ),
  DEFINE_END_OF_LIST
};

static const pragma_word_t pragma_opt_load_reduce[] = {
  DEFINE_FLAG("yes",     
    ~PFILE_FLAG_OPT_LOAD_REDUCE, 
    PFILE_FLAG_OPT_LOAD_REDUCE ),
  DEFINE_FLAG("no",      
    ~PFILE_FLAG_OPT_LOAD_REDUCE, 
    0 ),
  DEFINE_FLAG(0,         
    ~PFILE_FLAG_OPT_LOAD_REDUCE, 
    0 ),
  DEFINE_END_OF_LIST
};

static const pragma_word_t pragma_opt_const_detect[] = {
  DEFINE_FLAG("yes",
    ~PFILE_FLAG_OPT_CONST_DETECT,
    PFILE_FLAG_OPT_CONST_DETECT ),
  DEFINE_FLAG("no",
    ~PFILE_FLAG_OPT_CONST_DETECT,
    0 ),
  DEFINE_FLAG(0,
    ~PFILE_FLAG_OPT_CONST_DETECT,
    0 ),
  DEFINE_END_OF_LIST
};

static const pragma_word_t pragma_opt_variable_frame[] = {
  DEFINE_FLAG("yes",
      ~PFILE_FLAG_OPT_VARIABLE_FRAME,
      PFILE_FLAG_OPT_VARIABLE_FRAME),
  DEFINE_FLAG("no",
      ~PFILE_FLAG_OPT_VARIABLE_FRAME,
      0),
  DEFINE_FLAG(0,         
    ~PFILE_FLAG_OPT_VARIABLE_REDUCE, 
    PFILE_FLAG_OPT_VARIABLE_REDUCE ),
  DEFINE_END_OF_LIST
};

static const pragma_word_t pragma_opt[] = {
  DEFINE_FOLLOW("expr_reduce",     pragma_opt_expr_reduce),
  DEFINE_FOLLOW("cexpr_reduce",    pragma_opt_cexpr_reduce),
  DEFINE_FOLLOW("temp_reduce",     pragma_opt_temp_reduce),
  DEFINE_FOLLOW("variable_reduce", pragma_opt_variable_reduce),
  DEFINE_FOLLOW("load_reduce",     pragma_opt_load_reduce),
  DEFINE_FOLLOW("const_detect",    pragma_opt_const_detect),
  DEFINE_FOLLOW("variable_frame",  pragma_opt_variable_frame)
};

static const pragma_word_t pragma_warn_all[] = {
  DEFINE_FLAG("yes",
      ~PFILE_FLAG_WARN_ALL,
      PFILE_FLAG_WARN_ALL ),
  DEFINE_FLAG("no",
      ~PFILE_FLAG_WARN_ALL,
      0 ),
  DEFINE_END_OF_LIST
};

static const pragma_word_t pragma_warn_conversion[] = {
  DEFINE_FLAG("yes",
      ~PFILE_FLAG_WARN_CONVERSION,
      PFILE_FLAG_WARN_CONVERSION ),
  DEFINE_FLAG("no",
      ~PFILE_FLAG_WARN_CONVERSION,
      0 ),
  DEFINE_FLAG(0,
      ~PFILE_FLAG_WARN_CONVERSION,
      PFILE_FLAG_WARN_CONVERSION ),
  DEFINE_END_OF_LIST
};

static const pragma_word_t pragma_warn_directives[] = {
  DEFINE_FLAG("yes",
      ~PFILE_FLAG_WARN_DIRECTIVES,
      PFILE_FLAG_WARN_DIRECTIVES ),
  DEFINE_FLAG("no",
      ~PFILE_FLAG_WARN_DIRECTIVES,
      0 ),
  DEFINE_FLAG(0,
      ~PFILE_FLAG_WARN_DIRECTIVES,
      0 ),
  DEFINE_END_OF_LIST
};

static const pragma_word_t pragma_warn_misc[] = {
  DEFINE_FLAG("yes",
      ~PFILE_FLAG_WARN_MISC,
      PFILE_FLAG_WARN_MISC ),
  DEFINE_FLAG("no",
      ~PFILE_FLAG_WARN_MISC,
      0 ),
  DEFINE_FLAG(0,
      ~PFILE_FLAG_WARN_MISC,
      PFILE_FLAG_WARN_MISC ),
  DEFINE_END_OF_LIST
};

static const pragma_word_t pragma_warn_range[] = {
  DEFINE_FLAG("yes",
      ~PFILE_FLAG_WARN_RANGE,
      PFILE_FLAG_WARN_RANGE ),
  DEFINE_FLAG("no",
      ~PFILE_FLAG_WARN_RANGE,
      0 ),
  DEFINE_FLAG(0,
      ~PFILE_FLAG_WARN_RANGE,
      PFILE_FLAG_WARN_RANGE ),
  DEFINE_END_OF_LIST
};

static const pragma_word_t pragma_warn_backend[] = {
  DEFINE_FLAG("yes",
      ~PFILE_FLAG_WARN_BACKEND,
      PFILE_FLAG_WARN_BACKEND ),
  DEFINE_FLAG("no",
      ~PFILE_FLAG_WARN_BACKEND,
      0 ),
  DEFINE_FLAG(0,
      ~PFILE_FLAG_WARN_BACKEND,
      PFILE_FLAG_WARN_BACKEND ),
  DEFINE_END_OF_LIST
};

static const pragma_word_t pragma_warn_stack_overflow[] = {
  DEFINE_FLAG("yes",
      ~PFILE_FLAG_WARN_STACK_OVERFLOW,
      PFILE_FLAG_WARN_STACK_OVERFLOW ),
  DEFINE_FLAG("no",
      ~PFILE_FLAG_WARN_STACK_OVERFLOW,
      0 ),
  DEFINE_FLAG(0,
      ~PFILE_FLAG_WARN_STACK_OVERFLOW,
      0 )
};

static const pragma_word_t pragma_warn_truncate[] = {
  DEFINE_FLAG("yes",
      ~PFILE_FLAG_WARN_TRUNCATE,
      PFILE_FLAG_WARN_TRUNCATE ),
  DEFINE_FLAG("no",
      ~PFILE_FLAG_WARN_TRUNCATE,
      0 ),
  DEFINE_FLAG(0,
      ~PFILE_FLAG_WARN_TRUNCATE,
      PFILE_FLAG_WARN_TRUNCATE ),
  DEFINE_END_OF_LIST
};

static const pragma_word_t pragma_warn[] = {
  DEFINE_FOLLOW("all",             pragma_warn_all),
  DEFINE_FOLLOW("backend",         pragma_warn_backend),
  DEFINE_FOLLOW("conversion",      pragma_warn_conversion),
  DEFINE_FOLLOW("directives",      pragma_warn_directives),
  DEFINE_FOLLOW("misc",            pragma_warn_misc),
  DEFINE_FOLLOW("range",           pragma_warn_range),
  DEFINE_FOLLOW("stack_overflow",  pragma_warn_stack_overflow),
  DEFINE_FOLLOW("truncate",        pragma_warn_truncate),
  DEFINE_END_OF_LIST
};

static const pragma_word_t pragma_debug_codegen[] = {
  DEFINE_FLAG("yes",
      ~PFILE_FLAG_DEBUG_CODEGEN,
      PFILE_FLAG_DEBUG_CODEGEN),
  DEFINE_FLAG("no",
      ~PFILE_FLAG_DEBUG_CODEGEN,
      0),
  DEFINE_FLAG(0,
      ~PFILE_FLAG_DEBUG_CODEGEN,
      PFILE_FLAG_DEBUG_CODEGEN),
  DEFINE_END_OF_LIST
};

static const pragma_word_t pragma_debug_pcode[] = {
  DEFINE_FLAG("yes",
      ~PFILE_FLAG_DEBUG_PCODE,
      PFILE_FLAG_DEBUG_PCODE),
  DEFINE_FLAG("no",
      ~PFILE_FLAG_DEBUG_PCODE,
      0),
  DEFINE_FLAG(0,
      ~PFILE_FLAG_DEBUG_PCODE,
      0),
  DEFINE_END_OF_LIST
};

static const pragma_word_t pragma_debug[] = {
  DEFINE_FOLLOW("codegen",         pragma_debug_codegen),
  DEFINE_FOLLOW("pcode",           pragma_debug_pcode),
  DEFINE_END_OF_LIST
};

static const pragma_word_t pragma_fuses[] = {
  DEFINE_FLAG("yes",
      ~PFILE_FLAG_BOOT_FUSES,
      PFILE_FLAG_BOOT_FUSES),
  DEFINE_FLAG("no",
      ~PFILE_FLAG_BOOT_FUSES,
      0),
  DEFINE_FLAG(0,
      ~PFILE_FLAG_BOOT_FUSES,
      PFILE_FLAG_BOOT_FUSES),
  DEFINE_END_OF_LIST
};

static const pragma_word_t pragma_interrupt[] = {
  DEFINE_FLAG("normal",
      ~(PFILE_FLAG_MISC_INTERRUPT_RAW | PFILE_FLAG_MISC_INTERRUPT_FAST),
      0),
  DEFINE_FLAG("raw",
      ~(PFILE_FLAG_MISC_INTERRUPT_RAW | PFILE_FLAG_MISC_INTERRUPT_FAST),
      PFILE_FLAG_MISC_INTERRUPT_RAW),
  DEFINE_FLAG("fast",
      ~(PFILE_FLAG_MISC_INTERRUPT_RAW | PFILE_FLAG_MISC_INTERRUPT_FAST),
      PFILE_FLAG_MISC_INTERRUPT_FAST),
  DEFINE_FN(0, jal_parse_pragma_interrupt),
  DEFINE_END_OF_LIST
};

static const pragma_word_t base[] = {
  DEFINE_FN(    "name",       jal_parse_pragma_name),
  DEFINE_FN(    "error",      jal_parse_pragma_error),
  DEFINE_FN(    "target",     jal_parse_pragma_target),
  DEFINE_FOLLOW("interrupt",  pragma_interrupt),
  DEFINE_FN(    "jump_table", jal_parse_pragma_jump_table),
  DEFINE_FN(    "keep",       jal_parse_pragma_keep),
  DEFINE_FN(    "eeprom",     jal_parse_pragma_eeprom),
  DEFINE_FN(    "data",       jal_parse_pragma_data),
  DEFINE_FN(    "code",       jal_parse_pragma_code),
  DEFINE_FN(    "stack",      jal_parse_pragma_stack),
  DEFINE_FN(    "eedata",     jal_parse_pragma_eedata),
  DEFINE_FN(    "fuse_def",   jal_parse_pragma_fuse_def),
  DEFINE_FN(    "inline",     jal_parse_pragma_inline),
  DEFINE_FN(    "task",       jal_parse_pragma_task),
  DEFINE_FOLLOW("bootloader", pragma_bootloader),
  DEFINE_FOLLOW("clear",      pragma_clear),
  DEFINE_FOLLOW("opt",        pragma_opt),
  DEFINE_FOLLOW("warn",       pragma_warn),
  DEFINE_FOLLOW("debug",      pragma_debug),
  DEFINE_FOLLOW("fuses",      pragma_fuses),
  DEFINE_END_OF_LIST
};

void jal_parse_pragma(pfile_t *pf, const pfile_pos_t *statement_start)
{
  static const pragma_word_t *cmd;

  for (cmd = base;
       cmd->key && (PRAGMA_ACTION_EOL != cmd->action);
       /* nothing */
      ) {
    if (pf_token_is(pf, pf_token_current, pfile_log_none, cmd->key)) {
      pf_token_get(pf, pf_token_next);
      if (PRAGMA_ACTION_FOLLOW == cmd->action) {
        cmd = cmd->child;
      } else {
        break; /* found it! */
      }
    } else {
      cmd++;
    }
  }
  if (cmd) {
    switch (cmd->action) {
      case PRAGMA_ACTION_NONE:
      case PRAGMA_ACTION_FOLLOW:
        break; /* shouldn't get here */
      case PRAGMA_ACTION_FLAG:
        pfile_flag_change(pf, cmd->flag.mask, cmd->flag.set);
        break;
      case PRAGMA_ACTION_FN:
        cmd->fn(pf);
        break;
      case PRAGMA_ACTION_EOL:
        pfile_log(pf, pfile_log_err, "unexpected token: '%s'",
            pf_token_get(pf, pf_token_current));
        break;
    }
  }
}


