/**********************************************************
 **
 ** cmd_op.c : the cmd_type_operator functions
 **
 ** Copyright (c) 2004-2005, Kyle A. York
 ** All rights reserved
 **
 ***********************************************************/
#include <assert.h>
#include "cmdd.h"
#include "cmd_op.h"

/* expression, dst = val1 op val2 */
static void cmd_op_dump(cmd_t cmd, FILE *dst)
{
  const char *ostr;

  ostr = "{unknown}";
  switch (cmd_optype_get(cmd)) {
    case operator_null:         ostr = "{null}"; break;
    case operator_add:          ostr = "+"; break;
    case operator_sub:          ostr = "-"; break;
    case operator_mul:          ostr = "*"; break;
    case operator_div:          ostr = "/"; break;
    case operator_mod:          ostr = "%"; break;
    case operator_lt:           ostr = "<"; break;
    case operator_le:           ostr = "<="; break;
    case operator_eq:           ostr = "=="; break;
    case operator_ne:           ostr = "<>"; break;
    case operator_ge:           ostr = ">="; break;
    case operator_gt:           ostr = ">"; break;
    case operator_andl:         ostr = "&&"; break;
    case operator_orl:          ostr = "||"; break;
    case operator_notl:         ostr = "!";  break;
    case operator_andb:         ostr = "&";  break;
    case operator_orb:          ostr = "|"; break;
    case operator_xorb:         ostr = "^"; break;
    case operator_cmpb:         ostr = "~"; break;
    case operator_assign:       ostr = "="; break;
    case operator_neg:          ostr = "neg"; break;
    case operator_shift_left:   ostr = "<<"; break;
    case operator_shift_right:  ostr = ">>"; break;
    case operator_shift_right_arithmetic:  ostr = ">->"; break;
    case operator_logical:      ostr = "!!"; break;
    case operator_incr:         ostr = "++"; break;
    case operator_decr:         ostr = "--"; break;
    case operator_reference:    ostr = "whereis "; break;
  }
  fprintf(dst, "%s ", ostr);
  if (cmd_opdst_get(cmd) || cmd_opval1_get(cmd)) {
    if (cmd_opdst_get(cmd)) {
      value_dump(cmd_opdst_get(cmd), dst);
    }
    if (cmd_opval1_get(cmd)) {
      fprintf(dst, ", ");
      value_dump(cmd_opval1_get(cmd), dst);
      if (cmd_opval2_get(cmd)) {
        fprintf(dst, ", ");
        value_dump(cmd_opval2_get(cmd), dst);
      }
    }
  }
}

static void cmd_op_free(cmd_t cmd)
{
  cmd_opdst_set(cmd, 0);
  cmd_opval1_set(cmd, 0);
  cmd_opval2_set(cmd, 0);
}

static cmd_t cmd_op_dup(const cmd_t cmd)
{
  return cmd_op_alloc(cmd_optype_get(cmd),
      cmd_opdst_get(cmd), cmd_opval1_get(cmd),
      cmd_opval2_get(cmd));
}

static void cmd_op_variable_remap(cmd_t cmd, const variable_map_t *map)
{
  cmd_variable_remap2(cmd, map, cmd_opdst_get, cmd_opdst_set);
  cmd_variable_remap2(cmd, map, cmd_opval1_get, cmd_opval1_set);
  cmd_variable_remap2(cmd, map, cmd_opval2_get, cmd_opval2_set);
}

static void cmd_op_value_remap(cmd_t cmd, const value_map_t *map)
{
  cmd_value_remap2(cmd, map, cmd_opdst_get, cmd_opdst_set);
  cmd_value_remap2(cmd, map, cmd_opval1_get, cmd_opval1_set);
  cmd_value_remap2(cmd, map, cmd_opval2_get, cmd_opval2_set);
}

static flag_t cmd_op_variable_accessed(const cmd_t cmd, variable_t var)
{
  flag_t flags;

  flags = CMD_VARIABLE_ACCESS_FLAG_NONE;
  if (var == value_variable_get(cmd_opdst_get(cmd))) {
    flags |= CMD_VARIABLE_ACCESS_FLAG_WRITTEN;
  }
  if ((var == value_variable_get(cmd_opval1_get(cmd)))
    || (var == value_variable_get(cmd_opval2_get(cmd)))) {
    flags |= CMD_VARIABLE_ACCESS_FLAG_READ;
  }
  return flags;
}

static flag_t cmd_op_value_accessed(const cmd_t cmd, value_t val)
{
  flag_t flags;

  flags = CMD_VARIABLE_ACCESS_FLAG_NONE;
  if (val == cmd_opdst_get(cmd)) {
    flags |= CMD_VARIABLE_ACCESS_FLAG_WRITTEN;
  }
  if ((val == cmd_opval1_get(cmd))
    || (val == cmd_opval2_get(cmd))) {
    flags |= CMD_VARIABLE_ACCESS_FLAG_READ;
  }
  return flags;
}

static const cmd_vtbl_t cmd_op_vtbl = {
  cmd_op_free,
  cmd_op_dump,
  cmd_op_dup,
  0, /* no label remap */
  cmd_op_variable_remap,
  cmd_op_value_remap,
  cmd_op_variable_accessed,
  cmd_op_value_accessed
};

cmd_t cmd_op_alloc(operator_t op,
  value_t dst, value_t val1, value_t val2)
{
  cmd_t cmd;

  cmd = cmd_alloc(cmd_type_operator, &cmd_op_vtbl);
  if (cmd) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      ptr->u.op.op   = op;
      ptr->u.op.dst  = 0;
      ptr->u.op.val1 = 0;
      ptr->u.op.val2 = 0;

      cmd_opdst_set(cmd, dst);
      cmd_opval1_set(cmd, val1);
      cmd_opval2_set(cmd, val2);
    }
  }
  return cmd;
}


/* operator operations */
operator_t cmd_optype_get(const cmd_t cmd)
{
  const struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr && (cmd_type_operator == cmd_type_get(cmd)))
    ? ptr->u.op.op
    : operator_null;
}

void cmd_optype_set(cmd_t cmd, operator_t op)
{
  if (cmd_type_operator == cmd_type_get(cmd)) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      ptr->u.op.op = op;
    }
  }
}

value_t cmd_opdst_get(const cmd_t cmd)
{
  const struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr && (cmd_type_operator == cmd_type_get(cmd)))
    ? ptr->u.op.dst
    : 0;
}

void cmd_opdst_set(cmd_t cmd, value_t dst)
{
  if (cmd_type_operator == cmd_type_get(cmd)) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      if (dst) {
        value_assign_ct_bump(dst, ctr_bump_incr);
        value_lock(dst);
      }
      if (ptr->u.op.dst) {
        value_assign_ct_bump(ptr->u.op.dst, ctr_bump_decr);
        value_release(ptr->u.op.dst);
      }
      ptr->u.op.dst = dst;
    }
  }
}

value_t cmd_opval1_get(const cmd_t cmd)
{
  const struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr && (cmd_type_operator == cmd_type_get(cmd)))
    ? ptr->u.op.val1
    : 0;
}

void cmd_opval1_set(cmd_t cmd, value_t val1)
{
  if (cmd_type_operator == cmd_type_get(cmd)) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      if (val1) {
        pfile_proc_t *proc;

        proc = value_proc_get(val1);
        if (proc) {
          pfile_proc_flag_set(proc, PFILE_PROC_FLAG_INDIRECT);
        }

        value_lock(val1);
        value_use_ct_bump(val1, ctr_bump_incr);
      }
      if (ptr->u.op.val1) {
        value_use_ct_bump(ptr->u.op.val1, ctr_bump_decr);
        value_release(ptr->u.op.val1);
      }
      ptr->u.op.val1 = val1;
    }
  }
}

value_t cmd_opval2_get(const cmd_t cmd)
{
  const struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr && (cmd_type_operator == cmd_type_get(cmd)))
    ? ptr->u.op.val2
    : 0;
}

void cmd_opval2_set(cmd_t cmd, value_t val2)
{
  if (cmd_type_operator == cmd_type_get(cmd)) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      if (val2) {
        value_lock(val2);
        value_use_ct_bump(val2, ctr_bump_incr);
      }
      if (ptr->u.op.val2) {
        value_use_ct_bump(ptr->u.op.val2, ctr_bump_decr);
        value_release(ptr->u.op.val2);
      }
      ptr->u.op.val2 = val2;
    }
  }
}

/*
 * NAME
 *   cmd_op_cleanup
 *
 * DESCRIPTION
 *   cleanup an operator
 *
 * PARAMETERS
 *   cmd        : command
 *   value_zero : constant value 0
 *   value_one  : constant value 1 
 *
 * RETURN
 *   result_ok
 *   result_divide_by_zero
 *   result_range : a shift is out of range for the type
 *
 * NOTES
 *   This does a few different things:
 *     1. the constant is *always* on the right for commutative
 *        operators and relations
 *     2. reduce the following identities:
 *       x = x -- nop
 *       x + 0 -- nop
 *       x - 0 -- nop
 *       0 - x = -x
 *       x - x = 0
 *       x * 1 -- nop
 *       x * 0  = 0
 *       x * -1 = -x
 *       x / 1 -- nop
 *       x / -1 = -x
 *       x / 0 = err
 *       0 / x = 0
 *       x % 0 = 0
 *       x % 1 = 0
 *       x <  0, x unsigned --> 0
 *       x <= 0, x unsigned --> x == 0
 *       x >= 0, x unsigned --> 1
 *       x >  0, x unsigned --> x != 0
 *       x && 0 = 0
 *       x && x = !!x
 *       x || 0 = !!x
 *       x || x = !!x
 *       x & 0  = 0
 *       x & x  = x 
 *       x | 0  = x
 *       x | x  = x 
 *       x ^ 0  = x
 *       x ^ x  = 0
 *       0 << # = 0
 *       x << 0 = x
 *       x << # = 0 if # >= bits in x
 *       0 >> # = 0
 *       x >> 0 = x
 *       x >> # = 0 if # >= bits in x
 *       x == 0 = !x
 *       x != 0 = !!x
 *       x == 1 = x iff x is single bit
 *       x != 1 = !x iff x is single bit
 */
static boolean_t value_const_cmp(const value_t val, variable_const_t n)
{
  return value_is_const(val)
      && (value_const_get(val) == n);
}

static boolean_t value_is_unsigned(const value_t val)
{
  return !value_dflag_test(val, VARIABLE_DEF_FLAG_SIGNED);
}


result_t cmd_op_reduction(cmd_t cmd, value_t value_zero, value_t value_one)

{
  result_t   rc;

  rc   = result_ok;
  if (cmd_type_operator == cmd_type_get(cmd)) {
    operator_t     op;
    value_t        val1;
    value_t        val2;
    variable_sz_t  rsz;

    op   = cmd_optype_get(cmd);
    val1 = cmd_opval1_get(cmd);
    val2 = cmd_opval2_get(cmd);

    rsz = (value_sz_get(val1) < value_sz_get(val2))
      ? value_sz_get(val2)
      : value_sz_get(val1);

    if (value_is_const(val1)
        && (operator_is_commutative(op)
          || operator_is_relation(op))) {
      /* swap val1 & val2 */
      value_t tmp;

      tmp  = val1;
      val1 = val2;
      val2 = tmp;
      /* if it's a relation, swap the relation */
      switch (op) {
        case operator_lt: op = operator_gt; break;
        case operator_le: op = operator_ge; break;
        case operator_ge: op = operator_le; break;
        case operator_gt: op = operator_lt; break;
        default:
          break;
      }
      /* i need to lock/unlock these as setting them could cause
       * the counts to go to zero prematurely */
      value_lock(val1);
      value_lock(val2);
      cmd_opval1_set(cmd, val1);
      cmd_opval2_set(cmd, val2);
      value_release(val1);
      value_release(val2);
    }

    switch (op) {
      case operator_null:
      case operator_reference:
        break;
      case operator_add:
        if (value_const_cmp(val2, 0)) {
          /* x + 0 --> x */
          val2 = VALUE_NONE;
          op   = operator_assign;
        }
        break;
      case operator_sub:
        if (value_const_cmp(val2, 0)) {
          /* x - 0 --> x */
          val2 = VALUE_NONE;
          op   = operator_assign;
        } else if (value_const_cmp(val1, 0)) {
          /* 0 - x --> -x */
          val1 = val2;
          val2 = VALUE_NONE;
          op   = operator_neg;
        } else if (value_is_same(val1, val2)) {
          val1 = value_zero;
          val2 = VALUE_NONE;
        }
        break;
      case operator_mul:
        if (value_const_cmp(val2, 0)) {
          /* x * 0 --> 0 */
          val1 = value_zero;
          val2 = VALUE_NONE;
          op   = operator_assign;
        } else if (value_const_cmp(val2, 1)) {
          /* x * 1 --> x */
          val2 = VALUE_NONE;
          op   = operator_assign;
        } else if (value_const_cmp(val2, -1)) {
          val2 = VALUE_NONE;
          op   = operator_neg;
        }
        break;
      case operator_div:
        if (value_const_cmp(val2, 0)) {
          /* x / 0 --> *err* */
          rc = result_divide_by_zero;
        } else if (value_const_cmp(val2, 1)) {
          /* x / 1 --> x */
          val2 = VALUE_NONE;
          op   = operator_assign;
        } else if (value_const_cmp(val1, 0)) {
          /* 0 / x --> 0 */
          val1 = value_zero;
          val2 = VALUE_NONE;
          op   = operator_assign;
        } else if (value_const_cmp(val2, -1)) {
          val2 = VALUE_NONE;
          op   = operator_neg;
        }
        break;
      case operator_mod:
        if (value_const_cmp(val2, 0) || value_const_cmp(val2, 1)) {
          /* x % 0 --> 0 */
          /* x % 1 --> 0 */
          val1 = value_zero;
          val2 = VALUE_NONE;
          op   = operator_assign;
        }
        break;
      case operator_lt:
        if (value_is_unsigned(val1) && value_const_cmp(val2, 0)) {
          /* x < 0, x unsigned --> 0 */
          val1 = value_zero;
          val2 = VALUE_NONE;
          op   = operator_assign;
        }
        break;
      case operator_le:
        if (value_is_unsigned(val1) && value_const_cmp(val2, 0)) {
          /* x <= 0, x unsigned --> x == 0 */
          val2 = VALUE_NONE;
          op   = operator_notl;
        }
        break;
      case operator_eq:
      case operator_ne:
        if (value_is_const(val2)) {
          if (0 == value_const_get(val2)) {
            /* x == 0, or x != 0 */
            val2 = VALUE_NONE;
            op = (operator_eq == op)
              ? operator_notl
              : operator_logical;
          } else if (value_is_single_bit(val1)
            && (1 == value_const_get(val2))) {
            /* single bit == 1 or single bit != 1 */
            val2 = VALUE_NONE;
            op = (operator_eq == op)
              ? operator_logical
              : operator_notl;
          }
        }
        break;
      case operator_ge:
        if (value_is_unsigned(val1) && value_const_cmp(val2, 0)) {
          /* x >= 0, x unsigned --> x = 1 */
          val1 = value_one;
          val2 = VALUE_NONE;
          op   = operator_assign;
        }
        break;
      case operator_gt:
        if (value_is_unsigned(val1) && value_const_cmp(val2, 0)) {
          /* x > 0, x unsigned --> x != 0 */
          op = operator_ne;
        }
        break;
      case operator_andl:
        if (value_const_cmp(val2, 0)) {
          /* x && 0 --> 0 */
          val1 = value_zero;
          val2 = VALUE_NONE;
          op   = operator_assign;
        } else if (val1 == val2) {
          /* x && x --> !!x */
          val2 = VALUE_NONE;
          op   = operator_logical;
        }
        break;
      case operator_orl:
        if (value_const_cmp(val2, 0) || (val1 == val2)) {
          /* x || 0 --> !!x */
          /* x || x --> !!x */
          val2 = VALUE_NONE;
          op   = operator_logical;
        }
        break;
      case operator_notl: /* unary */
        break;
      case operator_andb:
        if (value_const_cmp(val2, 0)) {
          /* x & 0 --> 0 */
          val1 = value_zero;
          val2 = VALUE_NONE;
          op   = operator_assign;
        } else if (val1 == val2) {
          /* x & x --> x */
          val2 = VALUE_NONE;
          op   = operator_assign;
        }
        break;
      case operator_orb:
        if (value_const_cmp(val2, 0) || (val1 == val2)) {
          /* x | 0 --> x */
          /* x | x --> x */
          val2 = VALUE_NONE;
          op   = operator_assign;
        }
        break;
      case operator_xorb:
        if (value_const_cmp(val2, 0)) {
          /* x ^ 0 --> x */
          val2 = VALUE_NONE;
          op   = operator_assign;
        } else if (val1 == val2) {
          /* x ^ x --> 0 */
          val1 = value_zero;
          val2 = VALUE_NONE;
          op   = operator_assign;
        }
        break;
      case operator_cmpb: /* unary */
        break;
      case operator_neg:     /* unary */
        if ((variable_def_type_boolean == value_type_get(val1))
          || value_is_single_bit(val1)) {
          /* this is a no-op */
          cmd_optype_set(cmd, operator_assign);
        }
        break;
          
      case operator_assign:  /* unary */
      case operator_incr:    /* unary */
      case operator_decr:    /* unary */
        break;
      case operator_shift_left:
      case operator_shift_right:
      case operator_shift_right_arithmetic:
        if (value_const_cmp(val1, 0)) {
          /* 0 << x --> 0 */
          val1 = value_zero;
          val2 = VALUE_NONE;
          op   = operator_assign;
        } else if (value_const_cmp(val2, 0)) {
          /* x << 0 --> x */
          val2 = VALUE_NONE;
          op   = operator_assign;
        } else if (value_is_const(val2)
            && (value_const_get(val2) >= 8UL * rsz)) {
          op = operator_assign;
          rc = result_range;
        }
        break;
      case operator_logical:  /* unary : convert a value to either 0 or 1 */
        break;
    }
    cmd_optype_set(cmd, op);
    cmd_opval1_set(cmd, val1);
    cmd_opval2_set(cmd, val2);
  }
  return rc;
}

boolean_t cmd_op_is_assign(operator_t op, value_t val1, value_t val2)
{
  boolean_t rc;

  if (operator_is_unary(op)) {
    rc = val1 != 0;
  } else if (operator_is_binary(op)) {
    rc = (val1 != 0) && (val2 != 0);
  } else {
    rc = operator_is_assign(op);
  }
  return rc;
}

