/************************************************************
 **
 ** pf_op.c : perform operations on two constant values
 **
 ** Copyright (c) 2007, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/
#include <assert.h>
#include "pf_expr.h"
#include "pf_op.h"

/* return n sign extended as necessary */
static variable_const_t pfile_op_val_sign_extend(
    variable_const_t     n,
    const variable_def_t def)
{
  variable_sz_t sz; /* *bit* size */

  assert((variable_def_type_integer == variable_def_type_get(def))
      || (variable_def_type_boolean == variable_def_type_get(def)));

  sz = variable_def_sz_get(def);
  if (!variable_def_flag_test(def, VARIABLE_DEF_FLAG_BIT)) {
    sz *= 8;
  }
  if (variable_def_type_boolean == variable_def_type_get(def)) {
    n = !!n;
  } else if (sz < 32) {
    if (variable_def_flag_test(def, VARIABLE_DEF_FLAG_SIGNED) 
        && (n & (1UL << (sz - 1)))) {
      n |= ~((1UL << sz) - 1);
    } else {
      n &= (1UL << sz) - 1;
    }
  }
  return n;
}

/* return the result if n is assigned */
static variable_const_t pfile_op_val_get(
    variable_const_t     n,
    const variable_def_t def)
{
  variable_sz_t sz;

  assert((variable_def_type_integer == variable_def_type_get(def))
      || (variable_def_type_boolean == variable_def_type_get(def)));

  sz = variable_def_sz_get(def);
  if (!variable_def_flag_test(def, VARIABLE_DEF_FLAG_BIT)) {
    sz *= 8;
  }
  if (variable_def_type_boolean == variable_def_type_get(def)) {
    n = !!n;
  } else {
    if (sz < 32) {
      n &= ((1 << sz) - 1);
    }
    if (variable_def_flag_test(def, VARIABLE_DEF_FLAG_BIT)
      && (1 == sz) 
      && variable_def_flag_test(def, VARIABLE_DEF_FLAG_SIGNED)) {
      n = -n;
    }
  }
  return pfile_op_val_sign_extend(n, def);
}

variable_const_t pfile_op_const_exec(pfile_t *pf,
    operator_t op,
    value_t    val1,
    value_t    val2)
{
  variable_def_t   def1;
  variable_def_t   def2;
  variable_def_t   rdef;
  variable_const_t n;
  variable_const_t n1;
  variable_const_t n2;
  variable_sz_t    sz;
  variable_const_t mask;
  variable_const_t sign_bit;
  boolean_t        rdef_is_signed;

  assert(value_is_const(val1));
  assert(operator_is_unary(op) 
      || value_is_const(val2) 
      || value_is_pseudo_const(val2));

  def1 = value_def_get(val1);
  def2 = value_def_get(val2);
  rdef = pfile_variable_def_promotion_get(pf, op, val1, val2);

  n1             = pfile_op_val_get(value_const_get(val1), def1);
  n2             = (def2) 
                   ? pfile_op_val_get(value_const_get(val2), def2)
                   : 0;
  sz             = 8 * variable_def_sz_get(rdef);
  mask           = (sz < 32) ? (1UL << sz) - 1 : -1;
  sign_bit       = 1UL << 31;
  rdef_is_signed = variable_def_flag_test(rdef, VARIABLE_DEF_FLAG_SIGNED);

  n = 0xdeadbeef; /* otherwise the compiler flags as maybe unassigned */
  switch (op) {
    case operator_null:
    case operator_reference:
      assert(0);
      break;
    case operator_add:
      n = n1 + n2;
      break;
    case operator_sub:
      n = n1 - n2;
      break;
    case operator_mul:
      n = n1 * n2;
      break;
    case operator_div:
    case operator_mod:
      if (!n2) {
        pfile_log(pf, pfile_log_err, "divide by zero");
        n = (value_is_same(val1, val2)) ? 1 : -1;
      } else {
        boolean_t sign;

        sign = boolean_false;
        if (rdef_is_signed) {
          if (value_is_signed(val1) && (n1 & sign_bit)) {
            sign = boolean_true;
            n1   = -n1;
          }
          if (value_is_signed(val2) && (n2 & sign_bit)) {
            sign ^= boolean_true;
            n2    = -n2;
          }
        }
        n1 = n1 & mask;
        n2 = n2 & mask;
        n = (operator_div == op) ? n1 / n2 : n1 % n2;
        if (sign) {
          n = -n;
        }
      }
      break;
    case operator_lt:
    case operator_le:
    case operator_eq:
    case operator_ne:
    case operator_ge:
    case operator_gt:
      if (boolean_false == rdef_is_signed) {
        size_t sz1;
        size_t sz2;

        sz1 = value_sz_get(val1);
        if (!value_is_bit(val1)) {
          sz1 *= 8;
        }
        sz2 = value_sz_get(val2);
        if (!value_is_bit(val2)) {
          sz2 *= 8;
        }
        sz = (sz1 < sz2) ? sz2 : sz1;
        if (sz < 8) {
          sz = 8;
        }
        mask = (sz < 32) ? (1UL << sz) - 1 : -1;
        n1 &= mask;
        n2 &= mask;
      }
      switch (op) {
        case operator_lt:
          n = (rdef_is_signed)
            ? ((long) n1 < (long) n2)
            : (n1 < n2);
          break;
        case operator_le:
          n = (rdef_is_signed)
            ? ((long) n1 <= (long) n2)
            : (n1 <= n2);
          break;
        case operator_eq:
          n = (n1 == n2);
          break;
        case operator_ne:
          n = (n1 != n2);
          break;
        case operator_ge:
          n = (rdef_is_signed)
            ? ((long) n1 >= (long) n2)
            : (n1 >= n2);
          break;
        case operator_gt:
          n = (rdef_is_signed)
            ? ((long) n1 > (long) n2)
            : (n1 > n2);
          break;
        default:
          break; /* cannot get here */
      }
      break;
    case operator_andl:
      n = n1 && n2;
      break;
    case operator_orl:
      n = n1 || n2;
      break;
    case operator_notl:
      n = !n1;
      break;
    case operator_andb:
      n = n1 & n2;
      break;
    case operator_orb:
      n = n1 | n2;
      break;
    case operator_xorb:
      n = n1 ^ n2;
      break;
    case operator_cmpb:
      if (value_is_boolean(val1)) {
        n = (n1) ? 0 : -1;
      } else {
        n = ~n1;
      }
      break;
    case operator_assign:
      n = n1;
      break;
    case operator_neg:
      n = -n1;
      break;
    case operator_incr:
      n = n1 + 1;
      break;
    case operator_decr:
      n = n1 - 1;
      break;
    case operator_shift_left:
      n = n1 << n2;
      break;
    case operator_shift_right:
    case operator_shift_right_arithmetic:
      sign_bit = variable_def_sz_get(rdef);
      if (!variable_def_flag_test(rdef, VARIABLE_DEF_FLAG_BIT)) {
        sign_bit *= 8;
      }
      if (sign_bit < 8) {
        sign_bit = 8;
      }
      sign_bit = 1UL << (sign_bit - 1);
      if (n2 >= sz) {
        n = ((operator_shift_right_arithmetic == op)
            && (n1 & sign_bit))
          ? -1
          : 0;
      } else {
        n = (n1 & mask) >> n2;
        if ((operator_shift_right_arithmetic == op)
            && (sz - n2 < 32)
            && (n1 & sign_bit)) {
          n |= ~((1UL << (sz - n2)) - 1);
        }
      }
      n &= mask;
      break;
    case operator_logical:
      n = !!n1;
      break;
  }
  return pfile_op_val_get(n, rdef);
}

