/**********************************************************
 **
 ** variable.c : manipulators for variable_t
 **
 ** Copyright (c) 2004-2005, Kyle A. York
 ** All rights reserved
 **
 ***********************************************************/
#include <assert.h>
#include <string.h>
#include "../libutils/mem.h"
#include "pf_proc.h"
#include "variable.h"

struct variable_ {
  variable_t         link;
  refct_t            ref_ct;

  tag_t              tag;
  unsigned           tag_n;

  flag_t             flags;
  variable_def_t     def;       /* the variable's definition  */
  variable_base_t    base[VARIABLE_MIRROR_CT];
                                /* base position, or offset from
                                   master's base position     */
  unsigned short     bit_offset;/* bit #                      */
  ctr_t              ctr_assign;/* # of times assigned        */
  ctr_t              ctr_use;   /* # of times used            */
  unsigned char     *data;      /* misc. data; size = ct * sz */

  /* if this is an alias, master points to the controlling variable */
  variable_t         master;
  /* for functions this is the destination */
  pfile_proc_t      *proc;
  label_t            lbl;

  const char        *name;

  void              *user_data;
};

static cache_t variable_cache;
static boolean_t variable_cache_is_init;

static void variable_cache_cleanup(void)
{
#if 0
  unsigned n;

  for (n = 1; n < variable_cache.next; n++) {
    struct variable_ *ptr;

    ptr = cache_element_seek(&variable_cache, n, boolean_false);
    if (ptr->ref_ct != (refct_t) 0xefbe) {
      printf("var leaked: %u (refct = %u)\n", n, ptr->ref_ct);
    }
  }
#endif
  cache_cleanup(&variable_cache);
}

static variable_t variable_element_alloc(void)
{
  if (!variable_cache_is_init) {
    variable_cache_is_init = boolean_true;
    atexit(variable_cache_cleanup);
    cache_init(&variable_cache, sizeof(struct variable_), "variable");
  }
  return cache_element_alloc(&variable_cache);
}

static struct variable_ *variable_element_seek(variable_t el, boolean_t mod)
{
  struct variable_ *ptr;

  ptr  = cache_element_seek(&variable_cache, el, mod);
  return ptr;
}

/* basic functions */
variable_t variable_alloc(tag_t tag, variable_def_t def)
{
  variable_t var;

  var = variable_element_alloc();
  if (var) {
    struct variable_ *ptr;
    size_t            data_sz;

    ptr = variable_element_seek(var, boolean_true);

    if (tag) {
      tag_lock(tag);
      ptr->tag = tag;
      ptr->tag_n = tag_n_get(tag);
      tag_n_bump(tag);
    } else {
      ptr->tag   = 0;
      ptr->tag_n = 0;
    }

    data_sz = (variable_def_flag_test(def, VARIABLE_DEF_FLAG_CONST))
      ? variable_def_sz_get(def) : 0;
    if (data_sz && var) {
      ptr->data = MALLOC(data_sz);
      if (!ptr->data) {
        var = VARIABLE_NONE;
      } else {
	      memset(ptr->data, 0, data_sz);
	    }
    } else {
      ptr->data = 0;
    }

    if (var) {
      size_t ii;

      ptr->link       = VARIABLE_NONE;
      ptr->ref_ct     = 1;
      ptr->flags      = VARIABLE_FLAG_NONE;

      ptr->def        = def;
      for (ii = 0; ii < VARIABLE_MIRROR_CT; ii++) {
        ptr->base[ii] = VARIABLE_BASE_UNKNOWN;
      }
      ptr->bit_offset = 0;
      ptr->ctr_assign = 0;
      ptr->ctr_use    = 0;
      ptr->master     = VARIABLE_NONE;
      ptr->proc       = PFILE_PROC_NONE;
      ptr->lbl        = LABEL_NONE;
      ptr->name       = tag_name_get(tag);
      ptr->user_data  = 0;
    }
  }
  return var;
}

void variable_release(variable_t var)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_true);
  if (ptr && !--ptr->ref_ct) { 
    variable_master_set(var, VARIABLE_NONE);
    ptr = variable_element_seek(var, boolean_true);
#if 0
    if (ptr->ctr_assign) {
      printf("%s%u: assign=%u\n",
          ptr->name,
          variable_tag_n_get(var),
          ptr->ctr_assign);
    }
    if (ptr->ctr_use) {
      printf("%s%u: use=%u\n",
          ptr->name,
          variable_tag_n_get(var),
          ptr->ctr_use);
    }
#endif
    variable_data_set(var, 0);
    variable_label_set(var, LABEL_NONE);
    tag_release(ptr->tag);
    cache_element_free(&variable_cache, var);
  }
}

void variable_lock(variable_t var)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_true);
  if (ptr) {
    ptr->ref_ct++;
  }
}

/* member retrieval */
boolean_t variable_dflag_test(const variable_t var, flag_t flags)
{
  return variable_def_flag_test(variable_def_get(var), flags);
}

flag_t variable_dflags_get_all(const variable_t var)
{
  return variable_def_flags_get_all(variable_def_get(var));
}

void variable_flag_set(variable_t var, flag_t flags)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_true);
  if (ptr) {
    ptr->flags |= flags;
  }
}

void variable_flag_clr(variable_t var, flag_t flags)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_true);
  if (ptr) {
    ptr->flags &= ~flags;
  }
}

boolean_t variable_flag_test(const variable_t var, flag_t flags)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_false);
  return (ptr) ? ((ptr->flags & flags) == flags) : boolean_false;
}

const char *variable_name_get(const variable_t var)
{
  struct variable_ *ptr;
  const char       *name;

  ptr = variable_element_seek(var, boolean_false);
  name = (ptr) ? tag_name_get(ptr->tag) : 0;
  if (ptr && !name) {
    name = variable_name_get(variable_master_get(var));
  }
  return name;
}

unsigned variable_tag_n_get(const variable_t var)
{
  struct variable_ *ptr;
  const char       *name;
  unsigned          n;

  ptr = variable_element_seek(var, boolean_false);
  name = (ptr) ? tag_name_get(ptr->tag) : 0;

  if (ptr && !name) {
    n = variable_tag_n_get(variable_master_get(var));
  } else if (ptr) {
    n = ptr->tag_n;
  } else {
    n = 0;
  }
  return n;
}

variable_sz_t   variable_sz_get(const variable_t var)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_false);
  return (ptr) ? variable_def_sz_get(ptr->def): 0;
}

variable_base_t variable_base_get(const variable_t var, size_t which)
{
  struct variable_ *ptr;
  variable_base_t   base;

  ptr = variable_element_seek(var, boolean_false);
  if (ptr) {
    base = (which < VARIABLE_MIRROR_CT) 
      ? ptr->base[which] : VARIABLE_BASE_UNKNOWN;
    if (ptr->master) {
      variable_base_t mbase;

      /*assert(0 == which);*/
      mbase = variable_base_get(ptr->master, 0);
      if (VARIABLE_BASE_UNKNOWN != mbase) {
        base += mbase;
      }
    }
  } else {
    base = VARIABLE_BASE_UNKNOWN;
  }
  return base;
}

void variable_base_set(variable_t var, variable_base_t base, size_t which)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_true);
  if (ptr) {
#if 0
    if (ptr->master) {
      printf("trying to set base of %s whose master is %s\n",
          variable_name_get(var), variable_name_get(ptr->master));
    }
#endif
    if (which < VARIABLE_MIRROR_CT) {
      ptr->base[which] = base;
    }
  }
}

variable_const_t variable_const_get(const variable_t var, 
    variable_def_t def, size_t ofs)
{
  struct variable_ *ptr;
  variable_const_t  n;

  ptr = variable_element_seek(var, boolean_false);
  if (ptr && ptr->data) {
    size_t           ii;
    size_t           pos_mx;
    variable_const_t mask;

    if (variable_dflag_test(var, VARIABLE_DEF_FLAG_BIT)) {
      pos_mx = (variable_sz_get(var) + 7) / 8;
      mask = (1 << variable_sz_get(var)) - 1;
    } else {
      pos_mx = variable_sz_get(var);
      mask   = 0;
    }

    for (ii = 0, n = 0; (ii < variable_def_sz_get(def)); ii++) {
      n <<= 8UL;
      if (ofs + ii < pos_mx) {
        n |= ptr->data[ofs + ii];
      }
    }
    if (variable_def_flag_test(def, VARIABLE_DEF_FLAG_SIGNED)) {
      if ((ii < sizeof(n)) && (n & (1UL << ((8 * ii) - 1)))) {
        /* we need to extend the signed bit here! */
        while (ii < sizeof(n)) {
          n |= 255UL << (8 * ii);
          ii++;
        }
      }
    }
#if 0
    if (mask) {
      n &= mask;
    }
#endif
  } else {
    n = 0;
  }
  return n;
}

void variable_const_set(variable_t var, variable_def_t vdef,
    size_t ofs, variable_const_t n)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_true);
  if (ptr && ptr->data) {
    size_t ii;
    size_t ofs_mx;

    if (variable_dflag_test(var, VARIABLE_DEF_FLAG_BIT)) {
      ofs_mx = (variable_sz_get(var) + 7) / 8;
      if (variable_def_type_boolean == variable_type_get(var)) {
        n = !!n;
      } else {
        n &= (1 << variable_sz_get(var)) - 1;
      }
      if (variable_is_signed(var)
        && (n & (1 << (variable_sz_get(var) - 1)))) {
        /* high bit is set, so negate n */
        n = -n;
      }
    } else {
      ofs_mx = variable_sz_get(var);
    }
    for (ii = 0; ii < variable_def_sz_get(vdef); ii++) {
      if (ofs + variable_def_sz_get(vdef) - ii - 1 < ofs_mx) {
        ptr->data[ofs + variable_def_sz_get(vdef) - ii - 1] 
			= (uchar) (n & 0xff);
      }
      n >>= 8;
    }
  }
}

void variable_dump(const variable_t var, FILE *out)
{
  const char *fmt;

#if 0
  fprintf(out, "{%lx:", (unsigned long) var);
#else
  fprintf(out, "{");
#endif
  if (variable_def_type_pointer == variable_type_get(var)) {
    fputc('*', out);
  }
  if (variable_name_get(var)) {
    fmt = (variable_tag_n_get(var)) ? "__%s%u" : "%s";
    fprintf(out, fmt, variable_name_get(var), variable_tag_n_get(var));
  } else {
    variable_t master;

    master = variable_master_get(var);
    if (variable_name_get(master)) {
      fmt = (variable_tag_n_get(var)) ? "__%s%u" : "%s";
      fprintf(out, fmt, variable_name_get(master), 
          variable_tag_n_get(master));
    }
  }
  fprintf(out, "[%c%c%c%c%s%u]", 
      variable_def_type_to_ch(variable_type_get(var)), 
      variable_dflag_test(var, VARIABLE_DEF_FLAG_CONST) ? 'C' : '-',
      variable_dflag_test(var, VARIABLE_DEF_FLAG_VOLATILE) ? 'V' : '-',
      variable_dflag_test(var, VARIABLE_DEF_FLAG_SIGNED) ? 'S' : '-',
    variable_dflag_test(var, VARIABLE_DEF_FLAG_BIT) ? ":" : "",
    variable_sz_get(var));
  if (variable_is_const(var)) {
    fmt = (variable_dflag_test(var, VARIABLE_DEF_FLAG_SIGNED))
      ? "%ld" : "%lu";
    fprintf(out, fmt, variable_const_get(var, variable_def_get(var), 0));
  }
  fputc('}', out);
}

refct_t variable_refct_get(const variable_t var)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_false);
  return (ptr) ? ptr->ref_ct : 0;
}

variable_t variable_link_get(const variable_t var)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_false);
  return (ptr) ? ptr->link : 0;
}

void variable_link_set(variable_t var, variable_t link)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_true);
  if (ptr) {
    ptr->link = link;
  }
}

ctr_t variable_assign_ct_get(const variable_t var)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_false);
  return (ptr) ? ptr->ctr_assign : 0;
}

void variable_assign_ct_set(variable_t var, ctr_t ctr)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_true);
  if (ptr) {
    ptr->ctr_assign = ctr;
  }
}

void variable_assign_ct_bump(variable_t var, ctr_bump_t dir)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_true);
  if (ptr) {
    switch (dir) {
      case ctr_bump_incr: 
        ptr->ctr_assign++; 
        break;
      case ctr_bump_decr: 
        if (ptr->ctr_assign) {
          ptr->ctr_assign--; 
        } else {
          /*fprintf(stderr, "%s: assign is 0\n", variable_name_get(var));*/
        }
        break;
    }
    if (ptr->master) {
      variable_assign_ct_bump(ptr->master, dir);
    }
  }
}

ctr_t variable_use_ct_get(const variable_t var)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_false);
  return (ptr) ? ptr->ctr_use : 0;
}

void variable_use_ct_set(variable_t var, ctr_t ctr)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_true);
  if (ptr) {
    ptr->ctr_use = ctr;
  }
}

void variable_use_ct_bump(variable_t var, ctr_bump_t dir)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_true);
  if (ptr) {
    switch (dir) {
      case ctr_bump_incr: 
        ptr->ctr_use++; 
        break;
      case ctr_bump_decr: 
        if (ptr->ctr_use) {
          ptr->ctr_use--; 
        } else {
          /*fprintf(stderr, "%s:%u use is 0\n", variable_name_get(var),
              variable_tag_n_get(var));*/
        }
        break;
    }
    if (ptr->master) {
      variable_use_ct_bump(ptr->master, dir);
    }
  }
}

ctr_t variable_ref_ct_get(const variable_t var)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_false);
  return (ptr) ? ptr->ref_ct : 0;
}

void variable_list_init(varlist_t *lst)
{
  lst->head = 0;
  lst->tail = 0;
  lst->ct   = 0;
}

void variable_list_append(varlist_t *lst, variable_t var)
{
  variable_lock(var);

  if (lst->tail) {
    variable_link_set(lst->tail, var);
  } else {
    lst->head = var;
  }
  lst->tail = var;
  lst->ct++;
}

variable_t variable_list_find(varlist_t *lst, varfind_cb_t cb, void *cb_arg,
    const void *data)
{
  variable_t var;

  for (var = lst->head;
       var && !cb(cb_arg, var, data);
       var = variable_link_get(var))
    ;
  if (var) {
    variable_lock(var);
  }
  return var;
}

#if 0
/* nb : with the introduction of master variables (aka, aliases)
        it's necessary to run through the list *twice* to find
        if there's a leak */
static void variable_leak_test(variable_t var, pfile_t *pf)
{
  if (variable_refct_get(var) > 1) {
    if (variable_name_get(var)) {
      pfile_log(pf, pfile_log_debug, PFILE_MSG_DBG_VAR_LEAK, 
          var, variable_refct_get(var), variable_name_get(var));
    } else {
      pfile_log(pf, pfile_log_debug, PFILE_MSG_DBG_CONST_LEAK, 
          var, variable_refct_get(var), 
          variable_const_get(var, variable_def_get(var), 0));
    }
  }
}
#endif

void variable_list_reset(varlist_t *lst)
{
  variable_t var;

  /* first, set all of the masters to 0 */
  for (var = lst->head;
       var;
       var = variable_link_get(var)) {
    variable_master_set(var, VARIABLE_NONE);
  }
  while (lst->head) {
    var = lst->head;
    lst->head = variable_link_get(var);
    /*variable_leak_test(var, pf);*/
    variable_release(var);
  }
  lst->tail = 0;
  lst->ct = 0;
}

unsigned variable_list_ct_get(const varlist_t *lst)
{
  return (lst) ? lst->ct : 0;
}

variable_t variable_list_head(varlist_t *lst)
{
  return lst->head;
}

variable_def_t variable_def_get(const variable_t var)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_false);
  return (ptr) ? ptr->def : 0;
}

void variable_def_set(variable_t var, variable_def_t def)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_true);
  if (ptr) {
    ptr->def = def;
  }
}

/* given a constant, n, determine the smallest variable size
   that will hold it */
void variable_calc_sz_min(variable_const_t n, 
  variable_sz_t *psz, variable_def_type_t *ptype,
  flag_t *pflags)
{
  unsigned char       n0, n1, n2, n3;
  variable_sz_t       sz;
  variable_def_type_t type;
  flag_t              flags;


  n0 = (uchar) ((n      ) & 0xff);
  n1 = (uchar) ((n >>  8) & 0xff);
  n2 = (uchar) ((n >> 16) & 0xff);
  n3 = (uchar) ((n >> 24) & 0xff);
  sz = 4;
  type  = variable_def_type_integer;
  flags = VARIABLE_DEF_FLAG_NONE;
  if (0 == n3) {

    sz--;
    if (0 == n2) {
      sz--;
      if (0 == n1) {
        sz--;
      }
    }
  } else if ((0xff == n3) && (n2 & 0x80)) {
    /* the code generator will sign extend as necessary, so we'll
       pretend that this value is signed */
    flags |= VARIABLE_DEF_FLAG_SIGNED;
    sz--;
    if ((0xff == n2) && (n1 & 0x80)) {
      sz--;
      if ((0xff == n1) && (n0 & 0x80)) {
        sz--;
      }
    }
  }
  if (psz) {
    *psz = sz;
  }
  if (ptype) {
    *ptype = type;
  }
  if (pflags) {
    *pflags = flags;
  }
}

void variable_bit_offset_set(variable_t var, unsigned bit)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_true);
  if (ptr) {
    ptr->bit_offset = bit;
  }
}

unsigned variable_bit_offset_get(const variable_t var)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_false);
  return (ptr) ? ptr->bit_offset : 0;
}

variable_t variable_master_get(const variable_t var)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_false);
  return (ptr) ? ptr->master : 0;
}

static void variable_master_counters_adjust(variable_t var,
    ctr_t ctr_assign, ctr_t ctr_use)
{
  while (var) {
    struct variable_ *ptr;

    ptr = variable_element_seek(var, boolean_true);
    ptr->ctr_assign += ctr_assign;
    ptr->ctr_use    += ctr_use;
    var = variable_master_get(var);
  }
}

void variable_master_set(variable_t var, variable_t master)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_true);
  if (ptr) {
    ctr_t ctr_assign;
    ctr_t ctr_use;

    ctr_assign = variable_assign_ct_get(var);
    ctr_use    = variable_use_ct_get(var);

    variable_master_counters_adjust(
        variable_master_get(var), -ctr_assign, -ctr_use);
    /* now, remove the master */
    variable_lock(master);
    ptr = variable_element_seek(var, boolean_true);
    variable_release(ptr->master);
    ptr = variable_element_seek(var, boolean_true);
    ptr->master = master;
    if (master && (VARIABLE_BASE_UNKNOWN == ptr->base[0])) {
      ptr->base[0] = 0;
    }
    variable_master_counters_adjust(master, ctr_assign, ctr_use);
  }
}

variable_def_type_t variable_type_get(const variable_t var)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_false);
  return (ptr) ? variable_def_type_get(ptr->def) : variable_def_type_none;
}

pfile_proc_t *variable_proc_get(const variable_t var)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_false);
  return (ptr) ? ptr->proc : PFILE_PROC_NONE;
}

void variable_proc_set(variable_t var, pfile_proc_t *proc)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_true);
  if (ptr) {
    ptr->proc = proc;
  }
}

label_t variable_label_get(const variable_t var)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_false);
  return (ptr) ? ptr->lbl : LABEL_NONE;
}

void variable_label_set(variable_t var, label_t lbl)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_true);
  if (ptr) {
    label_lock(lbl);
    label_usage_bump(ptr->lbl, ctr_bump_decr);
    label_release(ptr->lbl);
    label_usage_bump(lbl, ctr_bump_incr);
    ptr->lbl = lbl;
  }
}

boolean_t variable_is_bit(const variable_t var)
{
  return variable_dflag_test(var, VARIABLE_DEF_FLAG_BIT);
}

boolean_t variable_is_volatile(const variable_t var)
{
  return variable_dflag_test(var, VARIABLE_DEF_FLAG_VOLATILE);
}

boolean_t variable_is_signed(const variable_t var)
{
  return variable_dflag_test(var, VARIABLE_DEF_FLAG_SIGNED);
}

boolean_t variable_is_alias(const variable_t var)
{
  return variable_flag_test(var, VARIABLE_FLAG_ALIAS);
}

boolean_t variable_is_auto(const variable_t var)
{
  return variable_flag_test(var, VARIABLE_FLAG_AUTO);
}

boolean_t variable_is_sticky(const variable_t var)
{
  return variable_flag_test(var, VARIABLE_FLAG_STICKY);
}

boolean_t variable_is_lookup(const variable_t var)
{
  return variable_flag_test(var, VARIABLE_FLAG_LOOKUP);
}

boolean_t variable_is_const(const variable_t var)
{
  return variable_dflag_test(var, VARIABLE_DEF_FLAG_CONST);
}

boolean_t variable_is_pseudo_const(const variable_t var)
{
  boolean_t rc;

  rc = variable_is_const(var);
  if (!rc) {
    if (variable_master_get(var)) {
      rc = variable_is_pseudo_const(variable_master_get(var));
    } else {
      rc = !(variable_is_volatile(var) || variable_assign_ct_get(var));
    }
  }
  return rc;
}

boolean_t variable_is_array(const variable_t var)
{
  return variable_def_type_array == variable_type_get(var);
}

/* a variable is assigned if it's volatile and used,
 * or if any of it's masters are assigned */
boolean_t variable_is_assigned(const variable_t var)
{
  boolean_t  rc;
  variable_t tvar;

  for (tvar = var, rc = boolean_false;
       tvar && !rc;
       tvar = variable_master_get(tvar)) {
    rc = 0 != variable_assign_ct_get(tvar);
    if (!rc && variable_is_volatile(tvar)) {
      rc = 0 != variable_use_ct_get(tvar);
    }
  }
  return rc;
}

/* a variable is used if it's volatile and assigned,
 * or if any of it's masters are used */
boolean_t variable_is_used(const variable_t var)
{
  boolean_t rc;
  variable_t tvar;

  for (tvar = var, rc = boolean_false;
       tvar && !rc;
       tvar = variable_master_get(tvar)) {
    rc = 0 != variable_use_ct_get(tvar);
    if (!rc && variable_is_volatile(tvar)) {
      rc = 0 != variable_assign_ct_get(tvar);
    }
  }
  return rc;
}

void variable_data_set(variable_t var, void *x)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_true);
  if (ptr) {
    if (ptr->data != x) {
      FREE(ptr->data);
    }
    ptr->data = x;
  }
}

void *variable_data_get(const variable_t var)
{
  const struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_false);
  return (ptr) ? ptr->data : 0;
}

variable_t variable_map_find(const variable_map_t *map, variable_t var)
{
  variable_t ret;

  if (!var) {
    ret = var;
  } else {
    size_t ii;

    for (ii = 0; (ii < map->used) && (map->map[ii].old != var); ii++)
      ;
    ret = (ii < map->used) ? map->map[ii].new : VARIABLE_NONE;
  }
#if 0
  if (ret) {
    printf("variable remap %s:%u --> %s:%u\n",
        variable_name_get(var), variable_tag_n_get(var),
        variable_name_get(ret), variable_tag_n_get(ret));
  }
#endif
  return ret;
}

flag_t variable_flags_get_all(variable_t var)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_false);
  return (ptr) ? ptr->flags : 0;
}

void variable_flags_set_all(variable_t var, flag_t flags)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_true);
  if (ptr) {
    ptr->flags = flags;
  }
}

void  variable_user_data_set(variable_t var, void *data)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_true);
  if (ptr) {
    ptr->user_data = data;
  }
}

void *variable_user_data_get(const variable_t var)
{
  struct variable_ *ptr;

  ptr = variable_element_seek(var, boolean_false);
  return (ptr) ? ptr->user_data : 0;
}

