/************************************************************
 **
 ** pic_inst.c : PIC instruction generation definitions
 **
 ** Copyright (c) 2004-2005, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/
#include <string.h>
#include <assert.h>

#include "piccolst.h"
#include "pic_inst.h"

static flag_t pic_instr_default_flags;

void pic_instr_default_flag_set(pfile_t *pf, flag_t flags)
{
  pic_instr_default_flags |= flags;
}

void pic_instr_default_flag_clr(pfile_t *pf, flag_t flags)
{
  pic_instr_default_flags &= ~flags;
}

boolean_t pic_instr_default_flag_test(pfile_t *pf, flag_t flag)
{
  return (pic_instr_default_flags & flag) == flag;
}

flag_t pic_instr_default_flag_get_all(pfile_t *pf)
{
  return pic_instr_default_flags;
}


pic_code_t pic_instr_alloc(pfile_t *pf, label_t lbl, pic_opcode_t op)
{
  pic_code_t code;

  code = pic_code_alloc(lbl, op, pic_instr_default_flag_get_all(pf));
  pic_code_list_append(pf, code);
  return code;
}


pic_code_t pic_instr_append_label(pfile_t *pf, label_t lbl)
{
  return pic_instr_alloc(pf, lbl, pic_opcode_none);
}

/* 
 * determine if var exists in the region defined by {lo,hi}
 */
static boolean_t pic_variable_region_test(variable_t var, size_t lo, 
  size_t hi)
{
  size_t    ii;
  boolean_t rc;

  while (variable_master_get(var)) {
    var = variable_master_get(var);
  }
  for (rc = boolean_false, ii = 0; !rc && (ii < VARIABLE_MIRROR_CT); ii++) {
    variable_base_t base;

    base = variable_base_get(var, ii);
    rc = (lo <= base) && (base <= hi);
  }
  return rc;
}

/* 
 * returns
 *   pic_opcode_none       : no code needed
 *   pic_opcode_datalo_clr
 *   pic_opcode_datalo_set
 */
pic_opcode_t pic_value_datalo_get(value_t val)
{
  variable_t   var;
  pic_opcode_t op;

  var = value_variable_get(val);

  /* if this exists in both region 0 & 2, nothing need be done with rp0 */
  if (!(pic_variable_region_test(var, 0x0000, 0x007f) 
        && pic_variable_region_test(var, 0x0080, 0x00ff))
      && !(pic_variable_region_test(var, 0x0100, 0x017f) 
        && pic_variable_region_test(var, 0x0180, 0x01ff))) {
    variable_base_t base;

    base = value_base_get(val);
    op = (base & 0x80) 
         ? pic_opcode_datalo_set
         : pic_opcode_datalo_clr;
  } else {
    op = pic_opcode_none;
  }
  return op;
}

/*
 * returns
 *   pic_opcode_none      : no code needed
 *   pic_opcode_datahi_clr
 *   pic_opcode_datahi_set
 */
pic_opcode_t pic_value_datahi_get(value_t val)
{
  variable_t   var;
  pic_opcode_t op;

  var = value_variable_get(val);

  /* if this exists in both region 1 & 3, nothing need be done with rp1 */
  if (!(pic_variable_region_test(var, 0x0000, 0x007f) 
        && pic_variable_region_test(var, 0x0100, 0x017f))
      && !(pic_variable_region_test(var, 0x0080, 0x00ff) 
        && pic_variable_region_test(var, 0x0180, 0x01ff))) {
    variable_base_t base;

    base = value_base_get(val);
    op   = (base & 0x100)
           ? pic_opcode_datahi_set
           : pic_opcode_datahi_clr;
  } else {
    op = pic_opcode_none;
  }
  return op;
}

/*
 * determine if the databit setup required by a and b are different
 */
boolean_t pic_are_variable_databits_different(value_t a, value_t b)
{
  boolean_t rc;

  pic_opcode_t a_lo;
  pic_opcode_t a_hi;
  pic_opcode_t b_lo;
  pic_opcode_t b_hi;

  a_lo = pic_value_datalo_get(a);
  a_hi = pic_value_datahi_get(a);
  b_lo = pic_value_datalo_get(b);
  b_hi = pic_value_datahi_get(b);

  rc = ((pic_opcode_none != a_lo) 
      && (pic_opcode_none != b_lo) 
      && (a_lo != b_lo))
    || ((pic_opcode_none != a_hi)
      && (pic_opcode_none != b_hi)
      && (a_hi != b_hi));
  return rc;
}

void pic_instr_append_daop(pfile_t *pf, value_t f, pic_opcode_t opcode)
{
  if (!pic_instr_default_flag_test(pf, PIC_CODE_FLAG_NO_OPTIMIZE)
      && (pic_opcode_movlw != opcode)
      && (pic_opcode_retlw != opcode)) {
    if (f && !value_is_const(f)) {
      variable_base_t base;
      variable_t      var;
      pic_opcode_t    op;

      base = value_base_get(f);
      var  = value_variable_get(f);
      
      op = pic_value_datalo_get(f);
      if (pic_opcode_none != op) {
        pic_code_t code;

        code = pic_instr_alloc(pf, LABEL_NONE, op);
        pic_code_value_set(code, f);
      }

      op = pic_value_datahi_get(f);
      if (pic_opcode_none != op) {
        pic_code_t code;

        code = pic_instr_alloc(pf, LABEL_NONE, op);
        pic_code_value_set(code, f);
      }
    }
  }
}

pic_code_t pic_instr_append_f_d(pfile_t *pf, pic_opcode_t op, value_t f, 
    size_t ofs, pic_opdst_t dst)
{
  pic_code_t code;
  boolean_t  indirect;

#if 0
  assert(
      (pic_opcode_addwf == op)
      || (pic_opcode_andwf == op)
      || (pic_opcode_xorwf == op)
      || (pic_opcode_iorwf == op)
      || (pic_opcode_subwf == op)
      || (pic_opcode_comf == op)
      || (pic_opcode_decf == op)
      || (pic_opcode_decfsz == op)
      || (pic_opcode_incf == op)
      || (pic_opcode_incfsz == op)
      || (pic_opcode_rlf == op)
      || (pic_opcode_rrf == op)
      || (pic_opcode_movf == op)
      || (pic_opcode_swapf == op)
      );
#endif
  indirect = boolean_false;
  if (value_is_indirect(f)) {
    indirect = boolean_true;
    f = pfile_value_find(pf, pfile_log_err, "_ind");
    ofs = 0;
  }
  pic_instr_append_daop(pf, f, op);
  code = pic_instr_alloc(pf, LABEL_NONE, op);
  if (code) {
    pic_code_value_set(code, f);
    pic_code_ofs_set(code, ofs);
    pic_code_dst_set(code, dst);
  }
  if (indirect) {
    value_release(f);
  }
  return code;
}

pic_code_t pic_instr_append_reg_d(pfile_t *pf, pic_opcode_t op, const char *reg,
  pic_opdst_t dst)
{
  value_t    src;
  pic_code_t code;

  src = pfile_value_find(pf, pfile_log_err, reg);
  if (src) {
    code = pic_instr_append_f_d(pf, op, src, 0, dst);
    value_release(src);
  } else {
    code = PIC_CODE_NONE;
  }
  return code;
}


pic_code_t pic_instr_append_f(pfile_t *pf, pic_opcode_t op, value_t f, 
    size_t ofs)
{
  pic_code_t code;
  boolean_t  indirect;

#if 0
  assert(
    (pic_opcode_clrf == op)
    || (pic_opcode_movwf == op)
    || (pic_opcode_datalo_set == op)
    || (pic_opcode_datalo_clr == op)
    || (pic_opcode_datahi_set == op)
    || (pic_opcode_datahi_clr == op)
    || (pic_opcode_bsf == op)
    || (pic_opcode_bcf == op)
    || (pic_opcode_btfsc == op)
    || (pic_opcode_btfss == op)
    || (pic_opcode_movlw == op)
    );
#endif
  indirect = boolean_false;
  if (value_is_indirect(f)) {
    indirect = boolean_true;
    f = pfile_value_find(pf, pfile_log_err, "_ind");
    ofs = 0;
  }

  pic_instr_append_daop(pf, f, op);
  code = pic_instr_alloc(pf, LABEL_NONE, op);
  value_variable_get(f);
  if (code) {
    pic_code_value_set(code, f);
    pic_code_ofs_set(code, ofs);
  }
  if (indirect) {
    value_release(f);
  }
  return code;
}

pic_code_t pic_instr_append_reg(pfile_t *pf, pic_opcode_t op, const char *reg)
{
  value_t    tmp;
  pic_code_t code;

  tmp = pfile_value_find(pf, pfile_log_err, reg);
  if (tmp) {
    code = pic_instr_append_f(pf, op, tmp, 0);
    value_release(tmp);
  } else {
    code = PIC_CODE_NONE;
  }
  return code;
}

pic_code_t pic_instr_append(pfile_t *pf, pic_opcode_t op)
{
#if 0
  assert(
      (pic_opcode_nop == op)
      || (pic_opcode_clrw == op)
      || (pic_opcode_retfie == op)
      || (pic_opcode_ret == op)
      || (pic_opcode_sleep == op)
      || (pic_opcode_clrwdt == op)
      || (pic_opcode_option == op)
      || (pic_opcode_irp_set == op)
      || (pic_opcode_irp_clr == op));
#endif

  return pic_instr_alloc(pf, LABEL_NONE, op);
}

pic_code_t pic_instr_append_f_b(pfile_t *pf, pic_opcode_t op, value_t f, 
    size_t ofs, value_t bit)
{
  pic_code_t code;
  boolean_t  indirect;

#if 0
  assert(
      (pic_opcode_bsf == op)
      || (pic_opcode_bcf == op)
      || (pic_opcode_btfsc == op)
      || (pic_opcode_btfss == op));
#endif
  indirect = boolean_false;
  if (value_is_indirect(f)) {
    indirect = boolean_true;
    f = pfile_value_find(pf, pfile_log_err, "_ind");
    ofs = 0;
  }
  pic_instr_append_daop(pf, f, op);
  code = pic_instr_alloc(pf, LABEL_NONE, op);
  if (code) {
    pic_code_value_set(code, f);
    pic_code_ofs_set(code, ofs);
    if (bit) {
      value_use_ct_bump(bit, ctr_bump_incr);
    }
    pic_code_literal_set(code, bit);
  }
  if (indirect) {
    value_release(f);
  }
  return code;
}

pic_code_t pic_instr_append_reg_bn(pfile_t *pf, pic_opcode_t op, 
    const char *reg, variable_const_t bit)
{
  value_t    vreg;
  pic_code_t code;

  code = PIC_CODE_NONE;
  vreg = pfile_value_find(pf, pfile_log_err, reg);
  if (vreg) {
    value_t vflag;

    vflag = pfile_constant_get(pf, bit, VARIABLE_DEF_NONE);
    if (vflag) {
      code = pic_instr_append_f_b(pf, op, vreg, 0, vflag);
      value_release(vflag);
    }
    value_release(vreg);
  }
  return code;
}

pic_code_t pic_instr_append_reg_flag(pfile_t *pf, pic_opcode_t op, 
    const char *reg, const char *flag)
{
  value_t    vreg;
  pic_code_t code;

  code = PIC_CODE_NONE;
  vreg = pfile_value_find(pf, pfile_log_err, reg);
  if (vreg) {
    value_t vflag;

    vflag = pfile_value_find(pf, pfile_log_err, flag);
    if (vflag) {
      code = pic_instr_append_f_b(pf, op, vreg, 0, vflag);
      value_release(vflag);
    }
    value_release(vreg);
  }
  return code;
}

pic_code_t pic_instr_append_f_bn(pfile_t *pf, pic_opcode_t op,
    value_t f, size_t ofs, variable_const_t n)
{
  value_t    vn;
  pic_code_t code;

  vn = pfile_constant_get(pf, n, VARIABLE_DEF_NONE);
  if (vn) {
    code = pic_instr_append_f_b(pf, op, f, ofs
        /*+ value_const_get(value_baseofs_get(f))*/, vn);
    value_release(vn);
  } else {
    code = PIC_CODE_NONE;
  }
  return code;
}

pic_code_t pic_instr_append_w_k(pfile_t *pf, pic_opcode_t op, 
    value_t k)
{
  pic_code_t code;

#if 0
  assert(
      (pic_opcode_addlw == op)
      || (pic_opcode_andlw == op)
      || (pic_opcode_iorlw == op)
      || (pic_opcode_movlw == op)
      || (pic_opcode_sublw == op)
      || (pic_opcode_xorlw == op)
      || (pic_opcode_retlw == op)
      || (pic_opcode_tris == op)
      || (pic_opcode_org == op));
#endif
  pic_instr_append_daop(pf, k, op);
  code = pic_instr_alloc(pf, LABEL_NONE, op);
  if (code) {
    pic_code_literal_set(code, k);
  }
  return code;
}

pic_code_t pic_instr_append_w_kn(pfile_t *pf, pic_opcode_t op,
    variable_const_t n)
{
  value_t    vn;
  pic_code_t code;

  vn = pfile_constant_get(pf, n, VARIABLE_DEF_NONE);
  if (vn) {
    code = pic_instr_append_w_k(pf, op, vn);
    value_release(vn);
  } else {
    code = PIC_CODE_NONE;
  }
  return code;
}

pic_code_t pic_instr_append_n(pfile_t *pf, pic_opcode_t op, label_t dst)
{
  pic_code_t code;

#if 0
  assert(
      (pic_opcode_call == op)
      || (pic_opcode_goto == op)
      || (pic_opcode_branchlo_set == op)
      || (pic_opcode_branchlo_clr == op)
      || (pic_opcode_branchlo_nop == op)
      || (pic_opcode_branchhi_set == op)
      || (pic_opcode_branchhi_clr == op)
      || (pic_opcode_branchhi_nop == op)
      );
#endif
  if (dst && !(pic_instr_default_flag_test(pf, PIC_CODE_FLAG_NO_OPTIMIZE))) {
    /* if the previous code is a skip conditional insert the 
     * PCLATH instructions *before* code_pv, otherwise insert
     * them *after* code_pv */
    pic_code_t brcode;
    pic_code_t brlist;
    value_t    code_sz;

    code_sz = pfile_value_find(pf, pfile_log_err, "_code_size");
    brlist = PIC_CODE_NONE;
    brcode = pic_code_list_tail_get(pf);
    while ((pic_opcode_btfsc == pic_code_op_get(brcode))
        || (pic_opcode_btfss == pic_code_op_get(brcode))
        || (pic_opcode_decfsz == pic_code_op_get(brcode))
        || (pic_opcode_incfsz == pic_code_op_get(brcode))) {
      pic_code_list_remove(pf, brcode);
      pic_code_next_set(brcode, brlist);
      brlist = brcode;
      brcode = pic_code_list_tail_get(pf);
    }

    /* note: we've not enough information to set the branchbits yet,
     *       just make sure they appear *before* any conditionals */
    if (value_const_get(code_sz) > 2048) {
      code = pic_instr_alloc(pf, LABEL_NONE, pic_opcode_branchlo_clr);
      pic_code_brdst_set(code, dst);
    }
    if (value_const_get(code_sz) > 4096) {
      code = pic_instr_alloc(pf, LABEL_NONE, pic_opcode_branchhi_clr);
      pic_code_brdst_set(code, dst);
    }

    while (brlist) {
      brcode = brlist;
      brlist = pic_code_next_get(brlist);
      pic_code_next_set(brcode, PIC_CODE_NONE);
      pic_code_list_append(pf, brcode);
    }
    value_release(code_sz);
  }
  code = pic_instr_alloc(pf, LABEL_NONE, op);
  pic_code_brdst_set(code, dst);
  return code;
}

unsigned pic_code_to_pcode(pfile_t *pf, const pic_code_t code)
{
  unsigned pcode;
  unsigned literal;
  unsigned val;
  unsigned dst;
  unsigned bit;
  unsigned lit8;
  unsigned lit10; /* branch literal */
  boolean_t lit_assigned;

  lit_assigned = boolean_false;

  if (pic_code_brdst_get(code)) {
    literal = label_pc_get(pic_code_brdst_get(code));
    lit_assigned = boolean_true;
    if (pic_code_ofs_get(code)) {
      /* this is kind of a hack, but I if there's a branch
         destination, I use offset 0 for the LSB, and offset
         1 for the MSB */
      literal >>= 8;
    }
  } else if (pic_code_literal_get(code)) {
    lit_assigned = boolean_true;
    if (value_is_const(pic_code_literal_get(code))) {
      literal = value_const_get(pic_code_literal_get(code));
    } else {
      literal = value_base_get(pic_code_literal_get(code))
        + value_const_get(value_baseofs_get(pic_code_literal_get(code)));
    }
  } else {
    literal = 0xffff;
  }

  if (pic_code_value_get(code)) {
    value_t cv;

    cv = pic_code_value_get(code);
    if (value_is_const(cv)) {
      val = value_const_get(cv);
    } else if (value_is_indirect(cv)) {
      value_t ind;

      ind = pfile_value_find(pf, pfile_log_err, "_ind");
      val = value_base_get(ind);
      value_release(ind);
    } else {
      val = (value_base_get(cv) 
        + value_const_get(value_baseofs_get(cv))
        + value_bit_offset_get(cv) / 8
        + pic_code_ofs_get(code));
      if (value_base_get(cv) == VARIABLE_BASE_UNKNOWN) {
        fprintf(stderr, "%x (%x + %lu): ", 
            pic_code_pc_get(code), 
            value_base_get(cv),
            (ulong) pic_code_ofs_get(code));
        variable_dump(value_variable_get(cv), stderr);
        fprintf(stderr, "\n");
        /*assert(0);*/
      } else {
        if (!lit_assigned) {
          literal = val;
        }
      }
      val &= 0x7f;
    }
    if (value_is_bit(cv)) {
      literal  = value_bit_offset_get(cv);
    }
  } else {
    val = 0xffff;
  }
  bit     = literal & 0x07;
  lit8    = literal & 0x00ff; /* 8 bit literal */
  lit10   = literal & 0x07ff; /* 10 bit literal */
  dst     = (pic_opdst_f == pic_code_dst_get(code)) << 7;
  pcode   = 0xffff; /* an impossible code (it's greater than 14 bits) */

  switch (pic_code_op_get(code)) {
    /* directives (no code) */
    case pic_opcode_org:    /* ORG xxxx */
    case pic_opcode_end:    /* END      */
    case pic_opcode_none:   /* no code here */
    case pic_opcode_branchlo_nop:
    case pic_opcode_branchhi_nop:
      break;
    case pic_opcode_nop:    pcode = 0x0000; break;
    /* file operator (code, dst, val) */
    case pic_opcode_movwf:  pcode = 0x0000 | 0x80 | val; break;
    case pic_opcode_clrf:   pcode = 0x0100 | 0x80 | val; break;
    case pic_opcode_subwf:  pcode = 0x0200 |  dst | val; break;
    case pic_opcode_decf:   pcode = 0x0300 |  dst | val; break;
    case pic_opcode_iorwf:  pcode = 0x0400 |  dst | val; break;
    case pic_opcode_andwf:  pcode = 0x0500 |  dst | val; break;
    case pic_opcode_xorwf:  pcode = 0x0600 |  dst | val; break;
    case pic_opcode_addwf:  pcode = 0x0700 |  dst | val; break;
    case pic_opcode_movf:   pcode = 0x0800 |  dst | val; break;
    case pic_opcode_comf:   pcode = 0x0900 |  dst | val; break;
    case pic_opcode_incf:   pcode = 0x0a00 |  dst | val; break;
    case pic_opcode_decfsz: pcode = 0x0b00 |  dst | val; break;
    case pic_opcode_rrf:    pcode = 0x0c00 |  dst | val; break;
    case pic_opcode_rlf:    pcode = 0x0d00 |  dst | val; break;
    case pic_opcode_swapf:  pcode = 0x0e00 |  dst | val; break;
    case pic_opcode_incfsz: pcode = 0x0f00 |  dst | val; break;
    /* simple opcodes (no operands) */
    case pic_opcode_retfie: pcode = 0x0009; break;
    case pic_opcode_ret:    pcode = 0x0008; break;
    case pic_opcode_sleep:  pcode = 0x0063; break;
    case pic_opcode_clrwdt: pcode = 0x0064; break;
    /* the following doesn't look right to me based on the microchip doc's, but
     * it is what MPLAB puts out */
    case pic_opcode_clrw:   pcode = 0x0103; break;
    /* bit oriented (3 bit literal) */
    case pic_opcode_bcf:    pcode = 0x1000 | (bit << 7) | val; break;
    case pic_opcode_bsf:    pcode = 0x1400 | (bit << 7) | val; break;
    case pic_opcode_btfsc:  pcode = 0x1800 | (bit << 7) | val; break;
    case pic_opcode_btfss:  pcode = 0x1c00 | (bit << 7) | val; break;
    /* literal & w  (8 bit literal) */ 
    case pic_opcode_movlw:  pcode = 0x3000 | lit8; break;
    case pic_opcode_retlw:  pcode = 0x3400 | lit8; break;
    case pic_opcode_iorlw:  pcode = 0x3800 | lit8; break;
    case pic_opcode_andlw:  pcode = 0x3900 | lit8; break;
    case pic_opcode_xorlw:  pcode = 0x3a00 | lit8; break;
    case pic_opcode_sublw:  pcode = 0x3c00 | lit8; break;
    case pic_opcode_addlw:  pcode = 0x3e00 | lit8; break;
    /* branching (10 bit literal) */
    case pic_opcode_call:   pcode = 0x2000 | lit10; break;
    case pic_opcode_goto:   pcode = 0x2800 | lit10; break;
    /* special (pseudo) opcodes */
    case pic_opcode_datalo_set: /* bsf _status, _rp0 */
      pcode = 0x1400 | (5 << 7) | 0x0003;
      break;
    case pic_opcode_datalo_clr: /* bcf _status, _rp0 */
      pcode = 0x1000 | (5 << 7) | 0x0003;
      break;
    case pic_opcode_datahi_set: /* bsf _status, _rp1 */
      pcode = 0x1400 | (6 << 7) | 0x0003;
      break;
    case pic_opcode_datahi_clr: /* bcf _status, _rp1 */
      pcode = 0x1000 | (6 << 7) | 0x0003;
      break;
    case pic_opcode_irp_set:    /* bsf _status, _irp */
      pcode = 0x1400 | (7 << 7) | 0x0003;
      break;
    case pic_opcode_irp_clr:    /* bcf _status, _irp */
      pcode = 0x1000 | (7 << 7) | 0x0003;
      break;
    case pic_opcode_branchlo_set: /* bsf _pclath: 3 */
      pcode = 0x1400 | (3 << 7) | 0x000a;
      break;
    case pic_opcode_branchlo_clr: /* bcf _pclath: 3 */
      pcode = 0x1000 | (3 << 7) | 0x000a;
      break;
    case pic_opcode_branchhi_set: /* bsf _pclath: 4 */
      pcode = 0x1400 | (4 << 7) | 0x000a;
      break;
    case pic_opcode_branchhi_clr: /* bcf _pclath: 4 */
      pcode = 0x1000 | (4 << 7) | 0x000a;
      break;
    case pic_opcode_option:       /* option */
      pcode = 0x0002;
      break;
    case pic_opcode_tris:         /* tris 6 */
      pcode = lit8 & 0x07;
      break;
    case pic_opcode_db: /* this is handled elsewhere */
      break;
  }
  return pcode;
}

boolean_t pic_opcode_is_branch_bit(pic_opcode_t op)
{
  return (pic_opcode_branchlo_clr == op)
    || (pic_opcode_branchlo_set == op)
    || (pic_opcode_branchlo_nop == op)
    || (pic_opcode_branchhi_clr == op)
    || (pic_opcode_branchhi_set == op)
    || (pic_opcode_branchhi_nop == op);
}

boolean_t pic_opcode_is_data_bit(pic_opcode_t op)
{
  return (pic_opcode_datalo_clr == op)
    || (pic_opcode_datalo_set == op)
    || (pic_opcode_datahi_clr == op)
    || (pic_opcode_datahi_set == op);
}

