/************************************************************
 **
 ** pic_var.c : PIC variable allocation definitions
 **
 ** Copyright (c) 2004-2005, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/
#include <assert.h>

#include "../libutils/mem.h"
#include "../libcore/pf_proc.h"
#include "pic_msg.h"
#include "pic_stvar.h"
#include "pic_var.h"

#undef BLIST_DEBUG

static unsigned  auto_mem_reg = 1;

unsigned         pic_blist_max; /* max amount used     */
unsigned         pic_blist_total;
pic_bank_info_t *pic_blist;

size_t           pic_stk_sz;
variable_base_t  pic_stk_base;

#ifdef BLIST_DEBUG
static void pic_blist_dump(pfile_t *pf, pic_bank_info_t *blist)
{
  const pic_bank_info_t *bptr;

  pfile_log(pf, pfile_log_debug, "--- blist start ---");
  for (bptr = blist; bptr; bptr = bptr->link) {
    pfile_log(pf, pfile_log_debug, 
        PIC_MSG_DBG_BANK, bptr->lo, bptr->hi);
  }
  pfile_log(pf, pfile_log_debug, "--- blist end ---");
}

static void pic_blist_check(const pic_bank_info_t *blist)
{
  const pic_bank_info_t *b1;
  const pic_bank_info_t *b2;

  b1 = blist;
  b2 = (b1) ? b1->link : 0;
  while (b1 && b2 && (b1 != b2)) {
    b1 = b1->link;
    b2 = b2->link;
    if (b2) {
      b2 = b2->link;
    }
  }
  assert(b1 != b2);
  for (b1 = blist; b1; b1 = b1->link) {
    assert(b1->lo <= b1->hi);
    if (b1->link) {
      assert(b1->hi < b1->link->lo);
    }
  }
}
#endif

/*
 * NAME
 *   pic_variable_alloc
 *
 * DESCRIPTION
 *   allocate space for all variables
 *
 * PARAMETERS
 *   pf
 *
 * RETURN
 *
 * NOTES
 *   this is dummied up for now. eventually i'll need to sort
 *   the variables by size for a more optimimal filling
 */
/* this is the union of  blist with {lo,hi} */
void pic_blist_add(pfile_t *pf, pic_bank_info_t **pblist, 
    size_t lo, size_t hi)
{
  pic_bank_info_t *ptr;
  pic_bank_info_t *pv;
  pic_bank_info_t *blist;

  assert(lo <= hi);
  blist = *pblist;
  for (ptr = blist, pv = 0; 
       ptr && (lo > ptr->hi);
       pv = ptr, ptr = ptr->link)
    ;
  /* this could use *a lot* of improvement, but will work for now
   * 1st I'll simply allocate a new entry & insert it, then I'll
   * do some cleanup */
  if (ptr && (ptr->lo <= lo) && (ptr->hi >= hi)) {
  } else {
    ptr = MALLOC(sizeof(*ptr));
    ptr->lo = lo;
    ptr->hi = hi;
    ptr->link = (pv) ? pv->link : blist;
    if (pv) {
      ptr->link = pv->link;
      pv->link  = ptr;
    } else {
      ptr->link = blist;
      blist     = ptr;
    }

    ptr = blist;
    while (ptr->link) {
      pic_bank_info_t *link;

      link = ptr->link;
      if (ptr->hi + 1 >= link->lo) {
        /* remove ptr->link */
        if (ptr->hi < link->hi) {
          ptr->hi = link->hi;
        }
        ptr->link = link->link;
        FREE(link);
      } else {
        ptr = link;
      }
    }
  }
  *pblist = blist;
#ifdef BLIST_DEBUG
  pic_blist_check(blist);
  pfile_log(pf, pfile_log_debug, "BLIST ADD(%u,%u)", lo, hi);
  pic_blist_dump(pf, pic_blist);
#endif
}

static void pic_blist_remove(pfile_t *pf, pic_bank_info_t **pblist,
    size_t base, size_t sz)
{
  pic_bank_info_t *ptr;
  pic_bank_info_t *pv;
  pic_bank_info_t *blist;
  size_t           lo;
  size_t           hi;

  blist = *pblist;
  lo = base;
  hi = base + sz - 1;

  for (ptr = blist, pv = 0;
       ptr && (lo > ptr->hi);
       pv = ptr, ptr = ptr->link)
    ;
  if (ptr && (lo <= ptr->hi) && (hi >= ptr->lo)) {
    /* lo <= ptr->hi
       ptr->lo <= hi <= ptr->hi */
    if (lo <= ptr->lo) {
      ptr->lo = hi + 1;
    }
    if (hi >= ptr->hi) {
      ptr->hi = lo - 1;
    }
    /* if (ptr->hi < ptr->lo), this bit is gone */
    if (ptr->hi < ptr->lo) {
      if (pv) {
        pv->link = ptr->link;
      } else {
        blist = ptr->link;
      }
      FREE(ptr);
    } else if ((ptr->lo < lo) && (hi < ptr->hi)) {
      /* split this into:
           {ptr->lo, lo - 1}
           {hi + 1, ptr->hi} */
      pv = ptr;
      ptr = MALLOC(sizeof(*ptr));
      if (!ptr) {
        pfile_log_syserr(pf, result_memory);
      } else {
        ptr->link = pv->link;
        pv->link  = ptr;
        ptr->hi   = pv->hi;
        ptr->lo   = hi + 1;
        pv->hi    = lo - 1;
      }
    }
  }
  *pblist = blist;
#ifdef BLIST_DEBUG
  pic_blist_check(blist);
  pfile_log(pf, pfile_log_debug, "BLIST REMOVE(%u,%u)", lo, lo + sz - 1);
  pic_blist_dump(pf, pic_blist);
#endif
}

/* allocate a block out of the data area */
static variable_base_t pic_data_alloc(pfile_t *pf, pic_bank_info_t **blist,
  variable_sz_t need)
{
  pic_bank_info_t *bank;
  variable_base_t  base;

  base = VARIABLE_BASE_UNKNOWN;

  for (bank = *blist; 
       bank && (bank->lo + need - 1 > bank->hi); 
       bank = bank->link)
    ;
  if (bank) {
    base = bank->lo;
    pic_blist_remove(pf, blist, bank->lo, need);
#ifdef BLIST_DEBUG
    fprintf(stderr, "Allocating %u\n", base);
#endif
    pic_blist_max += need;
  } 
  return base;
}

static variable_sz_t pic_data_biggest(const pic_bank_info_t *blist)
{
  variable_sz_t biggest;

  biggest = 0;
  while (blist) {
    variable_sz_t sz;

    sz = blist->hi - blist->lo + 1;
    if (sz > biggest) {
      biggest = sz;
    }
    blist = blist->link;
  }
  return biggest;
}

/* returns boolean_true if a new base was set, else boolean_false */
static boolean_t pic_variable_base_set(pfile_t *pf, pfile_proc_t *proc, pic_bank_info_t **blist,
    variable_t vptr)
{
  variable_sz_t   need;
  variable_base_t base;
  boolean_t       rc;

  need = variable_sz_get(vptr);
  assert(need);
  /* first we'll see if the entire thing can fit in a bank.
     If so, good! */

  base = pic_data_alloc(pf, blist, need);
  rc   = boolean_false;
  if (VARIABLE_BASE_UNKNOWN == base) {
    pfile_log(pf, pfile_log_err, "no space for %s:%s (need=%u available=%u)",
        (proc) ? pfile_proc_tag_get(proc) : "",
        variable_name_get(vptr),
        need, pic_data_biggest(*blist));
    variable_flag_set(vptr, VARIABLE_FLAG_ALLOC_FAIL);
  } else {
    assert(VARIABLE_BASE_UNKNOWN == variable_base_get(vptr, 0));
    rc = boolean_true;
    variable_base_set(vptr, base, 0);
  }
  return rc;
}

boolean_t pic_variable_alloc_one(pfile_t *pf, pfile_proc_t *proc,
    variable_t var)
{
  return pic_variable_base_set(pf, proc, &pic_blist, var);
}

/* find all variables that the user has placed & make sure these
   positions do not get reused */
static void blist_total(void)
{
  const pic_bank_info_t *binfo;

  for (pic_blist_total = 0, binfo = pic_blist; 
       binfo; 
       pic_blist_total += binfo->hi - binfo->lo + 1, binfo = binfo->link)
    ;
}

static void pic_variable_alloc_proc(pfile_t *pf, pfile_proc_t *proc);
static void pic_variable_alloc_block(pfile_t *pf, pfile_proc_t *proc, 
    pfile_block_t *blk)
{
  variable_t     vptr;
  pfile_block_t *cblk;
  size_t         ct;
  pfile_proc_t  *cproc;
  /*size_t         entry_sz;*/
  unsigned       auto_mem_reg_init;

  auto_mem_reg_init = auto_mem_reg;

#if 0
  printf("allocating space for: %s\n",
      label_name_get(pfile_proc_label_get(proc)));
#endif

  if (!blk) {
    blk = pfile_proc_block_root_get(proc);
  }
  /* first, allocate all variables for this procedure */
  for (vptr = pfile_block_variable_list_head(blk); 
       vptr; 
       vptr = variable_link_get(vptr)) {
    if (variable_is_auto(vptr) 
        && !variable_is_sticky(vptr)
        && !variable_flag_test(vptr, VARIABLE_FLAG_ALLOC_FAIL)) {
      /* find the first bank that will hold this variable */
      variable_sz_t   sz;
      variable_base_t old_base;

      sz = variable_sz_get(vptr);
      assert(sz);
      old_base = variable_base_get(vptr, 0);
      if ((old_base == VARIABLE_BASE_UNKNOWN)
        || (old_base < auto_mem_reg)) {
        variable_base_set(vptr, auto_mem_reg, 0);
        auto_mem_reg += sz;
        if (old_base != VARIABLE_BASE_UNKNOWN) {
          pfile_log(pf, pfile_log_debug, 
              "Resetting...%s%u (was %u, now %u)",
              variable_name_get(vptr), variable_tag_n_get(vptr),
              old_base, variable_base_get(vptr, 0)
              );
        }
      }
    }
  }
  /* next, recurse into all procedures this one calls and
   * allocate *those* variables */
  for (cblk = pfile_block_child_get(blk);
       cblk;
       cblk = pfile_block_sibbling_get(cblk)) {
    pic_variable_alloc_block(pf, proc, cblk);
  }
  for (ct = 0; (cproc = pfile_proc_calls_get(proc, ct)) != 0; ct++) {
    if (!pfile_proc_flag_test(cproc, PFILE_PROC_FLAG_VISITED)) {
      pfile_proc_flag_set(cproc, PFILE_PROC_FLAG_VISITED);
      pic_variable_alloc_proc(pf, cproc);
      pfile_proc_flag_clr(cproc, PFILE_PROC_FLAG_VISITED);
    }
  }
  if (pfile_flag_test(pf, PFILE_FLAG_OPT_VARIABLE_REUSE)) {
    auto_mem_reg = auto_mem_reg_init;
  }
}

static void pic_variable_alloc_proc(pfile_t *pf, pfile_proc_t *proc)
{
  /*
   * if the procedure is ever called indirectly its variables were
   * allocated earlier so this can be ignored
   */
  pic_variable_alloc_block(pf, proc, 0);
}

/* allocate all variables out of a single block. base holds the block */
static void pic_block_data_alloc(pfile_block_t *blk, variable_base_t base)
{
  variable_t vptr;

  for (vptr = pfile_block_variable_list_head(blk);
       vptr;
       vptr = variable_link_get(vptr)) {
    if (variable_is_auto(vptr)) {
      variable_base_set(vptr, base, 0);
      variable_flag_set(vptr, VARIABLE_FLAG_STICKY);
      base += variable_sz_get(vptr);
    }
  }
  for (blk = pfile_block_child_get(blk);
       blk;
       blk = pfile_block_sibbling_get(blk)) {
    pic_block_data_alloc(blk, base);
  }
}

/*
 * allocate variables for all of the indirect functions. It's probably
 * to analyze these a bit better & do a smarter allocation so we can
 * reuse data space, but it's far easier to just say dataspace for all 
 * functions called indirectly cannot be reused
 */
static void pic_variable_alloc_indirect_proc(pfile_t *pf, pfile_proc_t *proc)
{
  variable_base_t base;
  size_t          ct;
  pfile_proc_t   *cproc;

#if 0
  printf("allocating non-reusable space for: %s\n",
      label_name_get(pfile_proc_label_get(proc)));
#endif
  base = pic_data_alloc(pf, &pic_blist, pfile_proc_frame_sz_get(proc));
  pic_block_data_alloc(pfile_proc_block_root_get(proc), base);
  /* now...anything *called* by this procedure must also be allocated
   * in permanent, non-reusable store because I don't track indirect
   * references
   */ 
  for (ct = 0; (cproc = pfile_proc_calls_get(proc, ct)) != 0; ct++) {
    if (!pfile_proc_flag_test(cproc, PFILE_PROC_FLAG_VISITED)) {
      pfile_proc_flag_set(cproc, PFILE_PROC_FLAG_VISITED);
      pic_variable_alloc_indirect_proc(pf, cproc);
      pfile_proc_flag_clr(cproc, PFILE_PROC_FLAG_VISITED);
    }
  }
}

static void pic_variable_alloc_indirect(pfile_t *pf)
{
  pfile_proc_t *proc;

  for (proc = pfile_proc_root_get(pf);
       proc;
       proc = pfile_proc_next(proc)) {
    if (pfile_proc_frame_sz_get(proc)
        && (pfile_proc_flag_test(proc, PFILE_PROC_FLAG_INDIRECT)
          || pfile_proc_flag_test(proc, PFILE_PROC_FLAG_REENTRANT)
          || pfile_proc_flag_test(proc, PFILE_PROC_FLAG_TASK))) {
      pic_variable_alloc_indirect_proc(pf, proc);
    }
  }
}

/* all interrupt entry points share the same variable space, so...
     1. calculate the maximum size needed
     2. allocate a single block that size
     3. divy it up appropriately
 */
static void pic_variable_alloc_isr(pfile_t *pf)
{
  variable_base_t isr_block;
  pfile_proc_t   *proc;
  size_t          sz;

  for (proc = pfile_proc_root_get(pf), sz = 0; 
       proc; 
       proc = pfile_proc_next(proc)) {
    if (pfile_proc_flag_test(proc, PFILE_PROC_FLAG_INTERRUPT) 
      && (sz < pfile_proc_frame_sz_get(proc))) {
      sz = pfile_proc_frame_sz_get(proc);
    }
  }

  isr_block = (sz)
    ? pic_data_alloc(pf, &pic_blist, sz)
    : VARIABLE_BASE_UNKNOWN;

  for (proc = pfile_proc_root_get(pf);
       proc;
       proc = pfile_proc_next(proc)) {
    if (pfile_proc_flag_test(proc, PFILE_PROC_FLAG_INTERRUPT)) {
      pfile_block_t *blk;

      /* allocate all of the procedure's variables out of isr_block */
      pic_block_data_alloc(pfile_proc_block_root_get(proc), isr_block);
      /* remove the AUTO flag from all of the procedure's variables */
      for (blk = pfile_proc_block_root_get(proc);
           blk;
           blk = pfile_block_next(blk)) {
        variable_t vptr;

        for (vptr = pfile_block_variable_list_head(blk);
             vptr;
             vptr = variable_link_get(vptr)) {
          variable_flag_clr(vptr, VARIABLE_FLAG_AUTO);
        }
      }
      /* allocate anything called by this proc */
      pic_variable_alloc_indirect_proc(pf, proc);
    }
  }
}

static void pic_variable_alloc_volatile(pfile_t *pf)
{
  pfile_proc_t *proc;

  for (proc = pfile_proc_root_get(pf);
       proc;
       proc = pfile_proc_next(proc)) {
    pfile_block_t *blk;

    for (blk = pfile_proc_block_root_get(proc);
         blk;
         blk = pfile_block_next(blk)) {
      variable_t var;

      for (var = pfile_block_variable_list_head(blk);
           var;
           var = variable_link_get(var)) {
        if (variable_is_volatile(var) && variable_is_auto(var)) {
          if (variable_is_volatile(var)
              && (VARIABLE_BASE_UNKNOWN == variable_base_get(var, 0))
              && (variable_is_used(var) || variable_is_assigned(var))) {
            pic_variable_alloc_one(pf, proc, var);
            variable_flag_clr(var, VARIABLE_FLAG_AUTO);
          }
        }
      }
    }
  }
}

/* variables are allocated in an infinite memory, virtual space,
 * starting at location one. this routine maps the virtual space
 * into necessary PIC banks
 */ 
static int vinfo_cmp(const void *A, const void *B)
{
  variable_t       a;
  variable_t       b;
  variable_base_t  a_base;
  variable_base_t  b_base;

  a = *(struct variable_ * const *) A;
  b = *(struct variable_ * const *) B;
  a_base = variable_base_get(a, 0);
  b_base = variable_base_get(b, 0);

  /* this isn't as obscure as it looks. remember relational operators
   * are guarenteed to return either 0 or 1, then work out the three
   * possibilities
   */ 
  return (a_base > b_base) - (a_base < a_base);
}

static void pic_variable_alloc_fixup(pfile_t *pf)
{
  struct {
    size_t      used;
    size_t      alloc;
    struct variable_ **data;
  } vinfo;
  pfile_proc_t *proc;
  size_t        ii;

  vinfo.used  = 0;
  vinfo.alloc = 0;
  vinfo.data  = 0;
  /* first, find each allocated variable */
  for (proc = pfile_proc_root_get(pf);
       proc;
       proc = pfile_proc_next(proc)) {
    pfile_block_t *blk;

    for (blk = pfile_proc_block_root_get(proc);
         blk;
         blk = pfile_block_next(blk)) {
      variable_t vptr;

      for (vptr = pfile_block_variable_list_head(blk);
           vptr;
           vptr = variable_link_get(vptr)) {
        if (variable_is_auto(vptr)
            && !variable_is_sticky(vptr)
            && (VARIABLE_BASE_UNKNOWN != variable_base_get(vptr, 0))) {
          if (vinfo.used == vinfo.alloc) {
            size_t resize;
            void  *tmp;

            resize = (vinfo.alloc) ? 2 * vinfo.alloc : 1024;
            tmp    = REALLOC(vinfo.data, sizeof(*vinfo.data) * resize);
            if (tmp) {
              vinfo.alloc = resize;
              vinfo.data  = tmp;
            }
          }
          if (vinfo.used < vinfo.alloc) {
            vinfo.data[vinfo.used] = vptr;
            vinfo.used++;
          }
        }
      }
    }
  }
  /* sort the variables by position */
  qsort(vinfo.data, vinfo.used, sizeof(*vinfo.data), vinfo_cmp);
  /* finally, put the variables in their correct positions. find the
   * largest variable at each position, allocate that much space, then
   * set the base of all affected variables
   */
  ii = 0;
  while (ii < vinfo.used) {
    size_t          jj;
    size_t          sz_reqd; /* largest size required at position */
    variable_base_t base;

    sz_reqd = variable_sz_get(vinfo.data[ii]);
    for (jj = ii + 1;
         (jj < vinfo.used) 
         && (variable_base_get(vinfo.data[ii], 0) 
           == variable_base_get(vinfo.data[jj], 0));
         jj++) {
      size_t vsz;

      vsz = variable_sz_get(vinfo.data[jj]);
      if (vsz > sz_reqd) {
        sz_reqd = vsz;
      }
    }
    base = pic_data_alloc(pf, &pic_blist, sz_reqd);
    if (VARIABLE_BASE_UNKNOWN == base) {
      pfile_log(pf, pfile_log_err, "Out of data space!");
      break; /* no use to continue! */
    } else {
      pfile_log(pf, pfile_log_debug, "register %u mapped to %u",
          variable_base_get(vinfo.data[ii], 0),
          base);
      while (ii < jj) {
        variable_base_set(vinfo.data[ii], base, 0);
        ii++;
      }
    }
  }
  printf("Total variables: %u\n", vinfo.used);
  FREE(vinfo.data);
}

/* allocate variables but don't allow data space re-use */
static void pic_variable_alloc_no_reuse(pfile_t *pf)
{
  pfile_proc_t *proc;
  boolean_t     err;

  err = boolean_false;
  for (proc = pfile_proc_root_get(pf);
       proc && !err;
       proc = pfile_proc_next(proc)) {
    pfile_block_t *blk;

    for (blk = pfile_proc_block_root_get(proc);
         blk && !err; 
         blk = pfile_block_next(blk)) {
      variable_t vptr;

      for (vptr = pfile_block_variable_list_head(blk);
           vptr && !err;
           vptr = variable_link_get(vptr)) {
        if (variable_is_auto(vptr)
            && !variable_is_sticky(vptr)) {
          variable_base_t base;

          base = pic_data_alloc(pf, &pic_blist, variable_sz_get(vptr));
          if (VARIABLE_BASE_UNKNOWN == base) {
            pfile_log(pf, pfile_log_err, "Out of data space!");
            err = boolean_true;
          } else {
            variable_base_set(vptr, base, 0);
          }
        }
      }
    }
  }
}

void pic_variable_alloc(pfile_t *pf)
{
  pfile_proc_t *proc;

  blist_total();
  pfile_log(pf, pfile_log_debug, PIC_MSG_ALLOCATING_VARS);
  pic_stvar_fixup(pf);

  /* make sure the frame sizes are correct. currently only re-entrant
   * functions create an extra variable but that may change */

  for (proc = pfile_proc_root_get(pf);
       proc;
       proc = pfile_proc_next(proc)) {
    pfile_proc_frame_sz_calc(proc);
  }
  /* remove any preset variables */
  for (proc = pfile_proc_root_get(pf);
       proc;
       proc = pfile_proc_next(proc)) {
    pfile_block_t *blk;

    for (blk = pfile_proc_block_root_get(proc);
         blk;
         blk = pfile_block_next(blk)) {
      variable_t    vptr;

      for (vptr = pfile_block_variable_list_head(blk);
           vptr; 
           vptr = variable_link_get(vptr)) {
        if (!variable_master_get(vptr)
            && (VARIABLE_BASE_UNKNOWN != variable_base_get(vptr, 0))) {
          variable_base_t base;

          if (variable_sz_get(vptr)) {
            size_t ii;

            for (ii = 0; ii < VARIABLE_MIRROR_CT; ii++) {
              base = variable_base_get(vptr, 0);
              if (VARIABLE_BASE_UNKNOWN != base) {
                pic_blist_remove(pf, &pic_blist, variable_base_get(vptr, 0),
                      variable_sz_get(vptr));
              }
            }
          }
        }
      }
    }
  }
  pic_variable_alloc_volatile(pf);
  pic_variable_alloc_isr(pf);
  pic_variable_alloc_indirect(pf);
  /* finally, allocate some variables! */
  if (pfile_flag_test(pf, PFILE_FLAG_OPT_VARIABLE_REUSE)) {
    pic_variable_alloc_proc(pf, pfile_proc_root_get(pf));
    pic_variable_alloc_fixup(pf);
  } else {
    pic_variable_alloc_no_reuse(pf);
  }
  pfile_log(pf, pfile_log_debug, PIC_MSG_DBG_DATA_USED, pic_blist_max);
}

void pic_blist_info_log(pfile_t *pf)
{
#if 0
  pic_bank_info_t *bank;
  unsigned         total_avail;
  unsigned         total_used;
  unsigned         ct;

  for (ct = 0, total_used = 0, total_avail = 0, bank = pic_blist;
       bank;
       ct++, total_used += bank->next - bank->lo, 
         total_avail += bank->hi + 1 - bank->lo, bank = bank->link)
    ;
  pfile_log(pf, pfile_log_info, PIC_MSG_DATA_USED,
      total_used, total_avail);
#endif
}

void pic_blist_free(pfile_t *pf)
{
  while (pic_blist) {
    pic_bank_info_t *bank;

    bank = pic_blist->link;
    FREE(pic_blist);
    pic_blist = bank;
  }
}

pic_bank_info_t *pic_bank_info_get(pfile_t *pf)
{
  return pic_blist;
}

