/************************************************************
 **
 ** bsc_file.c : BSC file structure definitions
 **
 ** Copyright (c) 2004, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/
#include <string.h>
#include "pfile.h"
#include "pftoken.h"
#include "bsc_tokn.h"
#include "bsc_file.h"

typedef struct bsc_file_ {
  bsc_pass_t  pass;     /* pass #         */
  label_t     isrs[1 + bsc_isr_type_ct];
  size_t      str_ct;   /* # of strings   */
  size_t      str_nxt;  /* next string ID */
  flag_t      flags;
} bsc_file_t;

result_t bsc_file_open(pfile_t *pf)
{
  result_t    rc;
  bsc_file_t *bsc;

  bsc = MALLOC(sizeof(*bsc));
  if (!bsc) {
    rc = result_memory;
  } else {
    size_t ii;

    bsc->pass     = 0;
    bsc->str_ct   = 0;
    bsc->str_nxt  = 0;

    for (ii = 0; ii < COUNT(bsc->isrs); ii++) {
      bsc->isrs[ii] = 0;
    }
    bsc->isrs[bsc_isr_type_ct] = pfile_label_find(pf, pfile_log_err, 
      "_isr_cleanup");
    if (!bsc->isrs[bsc_isr_type_ct]) {
      FREE(bsc);
      rc = result_internal;
    } else {
      rc = result_ok;
      pfile_vector_arg_set(pf, bsc);
    }
  }
  return rc;
}

void bsc_file_close(pfile_t *pf)
{
  bsc_file_t *bsc;

  bsc = pfile_vector_arg_get(pf);
  if (bsc) {
    size_t ii;

    pfile_vector_arg_set(pf, 0);
    for (ii = 0; ii < COUNT(bsc->isrs); ii++) {
      if (bsc->isrs[ii]) {
        label_release(bsc->isrs[ii]);
      }
    }
    FREE(bsc);
  }
}

result_t bsc_file_isr_get(pfile_t *pf, bsc_isr_type_t isr, 
    boolean_t alloc, label_t *dst)
{
  label_t     lbl;
  result_t    rc;
  bsc_file_t *bsc;

  bsc = pfile_vector_arg_get(pf);
  lbl = bsc->isrs[isr];
  if (lbl) {
    rc = result_ok;
  } else if (!alloc) {
    rc = result_not_found;
  } else {
    const char *str;

    str = "{unknown}";
    switch (isr) {
      case bsc_isr_type_adc:   str = "_ir_adc";   break;
      case bsc_isr_type_pwm:   str = "_ir_pwm";   break;
      case bsc_isr_type_rx:    str = "_ir_rx";    break;
      case bsc_isr_type_tx:    str = "_ir_tx";    break;
      case bsc_isr_type_timer: str = "_ir_timer"; break;
      case bsc_isr_type_rb:    str = "_ir_rb";    break;
      case bsc_isr_type_user:  str = "_ir_user";  break;
      case bsc_isr_type_ct: break;
    }
    if (str) {
      rc = pfile_label_alloc(pf, str, &lbl);
      if (result_ok == rc) {
        bsc->isrs[isr] = lbl;
      }
    } else {
      rc = result_internal;
    }
  }
  if (result_ok == rc) {
    if (dst) {
      label_lock(lbl);
      *dst = lbl;
    }
  }
  return rc;
}

label_t bsc_file_isr_next_get(pfile_t *pf, bsc_isr_type_t type)
{
  bsc_file_t *bsc;

  bsc = pfile_vector_arg_get(pf);
  while ((++type < COUNT(bsc->isrs)) && !bsc->isrs[type]) 
    ;
  return (type < COUNT(bsc->isrs)) ? bsc->isrs[type] : 0;
}

void bsc_file_pass_reset(pfile_t *pf)
{
  bsc_file_t *bsc;

  bsc = pfile_vector_arg_get(pf);
  pfile_reset(pf);
  bsc->pass++;
  pf_token_get(pf, pf_token_first);
  bsc->str_ct = 0;
}

bsc_pass_t bsc_file_pass_get(pfile_t *pf)
{
  bsc_file_t *bsc;

  bsc = pfile_vector_arg_get(pf);
  return bsc->pass;
}

size_t bsc_file_strid_next(pfile_t *pf)
{
  bsc_file_t *bsc;

  bsc = pfile_vector_arg_get(pf);
  return bsc->str_ct++;
}

void bsc_file_log_unavail(pfile_t *pf, const char *cmd)
{
  pfile_log(pf, pfile_log_err, BSC_MSG_CMD_NOT_AVAIL, cmd);
}

boolean_t bsc_file_flag_test(const pfile_t *pf, bsc_file_flag_t flag)
{
  const bsc_file_t *bsc;

  bsc = pfile_vector_arg_get(pf);
  return (bsc->flags & (1 << flag)) != 0;
}

void bsc_file_flag_set(pfile_t *pf, bsc_file_flag_t flag)
{
  bsc_file_t *bsc;

  bsc = pfile_vector_arg_get(pf);
  bsc->flags |= (1 << flag);
}

void bsc_file_flag_clr(pfile_t *pf, bsc_file_flag_t flag)
{
  bsc_file_t *bsc;

  bsc = pfile_vector_arg_get(pf);
  bsc->flags &= ~(1 << flag);
}

