/**********************************************************
 **
 ** jal_ctrl.c : parser for JAL control functions
 **
 ** Copyright (c) 2004-2005, Kyle A. York
 ** All rights reserved
 **
 ***********************************************************/
#include <errno.h>
#include "../libutils/mem.h"
#include "../libcore/pf_cmd.h"
#include "../libcore/pf_expr.h"
#include "../libcore/pf_proc.h"
#include "jal_expr.h"
#include "jal_blck.h"
#include "jal_file.h"
#include "jal_ctrl.h"
#include "jal_tokn.h"

/*
 * NAME
 *   jal_parse_for
 *
 * DESCRIPTION
 *   implement FOR
 *
 * PARAMETERS
 *   pf       : 
 *
 * RETURN
 *   none
 *
 * NOTES
 *   FOR expr LOOP ... END LOOP
 * this becomes:
 *   tmp = expr
 *   label_top:
 *     if (!tmp) goto label_end
 *     block
 *     tmp--
 *     goto label_top
 *   label_end:
 */
void jal_parse_for(pfile_t *pf, const pfile_pos_t *statement_start)
{
  value_t     expr;
  boolean_t   codegen_disable;
  label_t     lbl_top;
  label_t     lbl_end;
  value_t     val;
  value_t     zero;
  cmd_t       cmd_start;
  boolean_t   is_modified;

  expr = jal_parse_expr(pf);
  if (value_is_temp(expr)) {
    /* The lifetime of a temporary is assumed to be one statement
     * but that's not true for a FOR loop. */
    value_t tmp;

    tmp = jal_file_loop_var_get(pf, expr);
    pfile_cmd_op_add(pf, operator_assign, &tmp, expr, VALUE_NONE);
    value_release(expr);
    expr = tmp;
  }

  if (pf_token_is(pf, pf_token_current, pfile_log_none, "using")) {
    pf_token_get(pf, pf_token_next);
    val = VALUE_NONE;
    pf_token_to_value(pf, pfile_log_err, &val);
    if (VALUE_NONE != val) {
      pf_token_get(pf, pf_token_next);
    }
    if (value_is_const(expr)) {
      /* check that expr will fit in the resulting variable */
      if (value_is_bit(val)) {
        if (((1UL << value_sz_get(val)) - 1) < value_const_get(expr)) {
          pfile_log(pf, pfile_log_err, "expr doesn't fit into variable");
        } else if (value_is_boolean(val)) {
          if (pfile_flag_test(pf, PFILE_FLAG_WARN_MISC)) {
            pfile_log(pf, pfile_log_warn, 
                "BIT values shouldn't be used in loops");
          }
        }
      } else {
        if (!value_is_universal(expr) 
            && (value_sz_get(expr) > value_sz_get(val))) {
          pfile_log(pf, pfile_log_err, "expr doesn't fit into variable");
        }
      }
    }
  } else {
    val = jal_file_loop_var_get(pf, expr);
  }

  codegen_disable = (value_is_const(expr) && !value_const_get(expr));

  if (codegen_disable) {
    pfile_codegen_disable(pf, boolean_true);
  }

  zero = pfile_constant_get(pf, 0, VARIABLE_DEF_NONE);
  pfile_cmd_op_add(pf, operator_assign, &val, zero, VALUE_NONE);
  value_release(zero);

  if (pf_token_is(pf, pf_token_current, pfile_log_err, "loop")) {
    pf_token_get(pf, pf_token_next);
  }

  lbl_top = pfile_label_alloc(pf, 0);
  lbl_end = pfile_label_alloc(pf, 0);

  if (lbl_end) {
    pfile_cmd_branch_add(pf, cmd_branchtype_goto,
      cmd_branchcond_none, lbl_end, VALUE_NONE, 0, 0);
  }

  pfile_cmd_label_add(pf, lbl_top);
  cmd_start = pfile_cmdlist_tail_get(pf);
  jal_block_process(pf, JAL_BLOCK_PROCESS_FLAG_NONE);
  /* make sure the control variable isn't modified in the block! */
  is_modified = boolean_false;
  while (cmd_start && !is_modified) {
    is_modified = cmd_variable_accessed_get(cmd_start,
        value_variable_get(val)) & CMD_VARIABLE_ACCESS_FLAG_WRITTEN;
    cmd_start = cmd_link_get(cmd_start);
  }
  if (is_modified) {
    if (pfile_flag_test(pf, PFILE_FLAG_WARN_MISC)) {
      pfile_log(pf, pfile_log_warn,
          "%s should not be modified in the LOOP block",
          value_name_get(val));
    }
  }

  pfile_cmd_op_add(pf, operator_incr, &val, VALUE_NONE, 
      VALUE_NONE);
  if (lbl_end) {
    value_t tmp;

    pfile_cmd_label_add(pf, lbl_end);
    tmp = VALUE_NONE;
    if (VALUE_NONE != expr) {
      pfile_cmd_op_add(pf, operator_eq, &tmp, val, expr);
    }
    pfile_cmd_branch_add(pf, cmd_branchtype_goto, 
        cmd_branchcond_false, lbl_top, tmp, 0, 0);
    value_release(tmp);
  }
  if (pf_token_is(pf, pf_token_current, pfile_log_err, "end")
    && pf_token_is(pf, pf_token_next, pfile_log_err, "loop")) {
    pf_token_get(pf, pf_token_next);
  } else {
    jal_block_start_show(pf, "FOR", statement_start);
  }

  label_release(lbl_end);
  label_release(lbl_top);
  if (codegen_disable) {
    pfile_codegen_disable(pf, boolean_false);
  }
  value_release(val);
  value_release(expr);
}

/*
 * NAME
 *   jal_parse_forever
 *
 * DESCRIPTION
 *   implement FOREVER
 *
 * PARAMETERS
 *   pf: 
 *
 * RETURN
 *   none
 *
 * NOTES
 *   FOREVER LOOP ... END LOOP
 * this becomes:
 *   label_top:
 *      block
 *      goto label_top
 */
void jal_parse_forever(pfile_t *pf, const pfile_pos_t *statement_start)
{
  label_t     lbl_top;

  if (pf_token_is(pf, pf_token_current, pfile_log_err, "loop")) {
    pf_token_get(pf, pf_token_next);
  }

  lbl_top = pfile_label_alloc(pf, 0);

  pfile_cmd_label_add(pf, lbl_top);
  jal_block_process(pf, JAL_BLOCK_PROCESS_FLAG_NONE);
  if (pf_token_is(pf, pf_token_current, pfile_log_err, "end")
    && pf_token_is(pf, pf_token_next, pfile_log_err, "loop")) {
    pf_token_get(pf, pf_token_next);
  } else {
    jal_block_start_show(pf, "FOREVER", statement_start);
  }
  if (lbl_top) {
    pfile_cmd_branch_add(pf, cmd_branchtype_goto,
        cmd_branchcond_none, lbl_top, VALUE_NONE, 0, 0);
  }
  label_release(lbl_top);
}

/*
 * NAME
 *   jal_parse_while
 *
 * DESCRIPTION
 *   implement WHILE
 *
 * PARAMETERS
 *   pf: 
 *
 * RETURN
 *   none
 *
 * NOTES
 *   WHILE bexpr LOOP ... END LOOP
 * this becomes:
 *   tmp = bexpr
 *   label_top:
 *     if (!tmp) goto label_end
 *     block
 *     goto label_top
 *   label_end: 
 */
void jal_parse_while(pfile_t *pf, const pfile_pos_t *statement_start)
{
  label_t     lbl_top;
  label_t     lbl_end;
  value_t     expr;
  boolean_t   codegen_disable;

  lbl_top = pfile_label_alloc(pf, 0);
  lbl_end = pfile_label_alloc(pf, 0);
  expr    = VALUE_NONE;

  pfile_cmd_label_add(pf, lbl_top);
  expr = jal_parse_expr(pf);

  if (!value_is_boolean(expr)) {
    pfile_log(pf, pfile_log_warn, "boolean expression expected");
  }
  codegen_disable = (value_is_const(expr) && !value_const_get(expr));
  if (codegen_disable) {
    pfile_codegen_disable(pf, boolean_true);
  }
  if (lbl_end && expr) {
    pfile_cmd_branch_add(pf, cmd_branchtype_goto,
        cmd_branchcond_false, lbl_end, expr, 0, 0);
  }
  value_release(expr);
  if (pf_token_is(pf, pf_token_current, pfile_log_err, "loop")) {
    pf_token_get(pf, pf_token_next);
  }
  jal_block_process(pf, JAL_BLOCK_PROCESS_FLAG_NONE);
  if (pf_token_is(pf, pf_token_current, pfile_log_err, "end")
    && pf_token_is(pf, pf_token_next, pfile_log_err, "loop")) {
    pf_token_get(pf, pf_token_next);
  } else {
    jal_block_start_show(pf, "WHILE", statement_start);
  }
  pfile_cmd_branch_add(pf, cmd_branchtype_goto,
      cmd_branchcond_none, lbl_top, VALUE_NONE, 0, 0);
  pfile_cmd_label_add(pf, lbl_end);
  if (codegen_disable) {
    pfile_codegen_disable(pf, boolean_false);
  }
  label_release(lbl_end);
  label_release(lbl_top);
}

/*
 * NAME
 *   jal_skip_if
 *
 * DESCRIPTION
 *   skip an if block
 *
 * PARAMETERS
 *   pf       : pfile
 *   is_else  : TRUE if this is an else clause
 *
 * RETURN
 *   TRUE  : this resulted in an `end if' (absorbed)
 *   FALSE : the return is either 'elsif' or 'else'
 *
 * NOTES
 *   this is a rather wacky result of the way the IF statement works in
 *   JAL. JAL, having no preprocessor, uses `IF cexpr THEN...' like the
 *   C preprocessor's #if ...
 */
static boolean_t jal_skip_if(pfile_t *pf, boolean_t is_else)
{
  boolean_t   is_end;

  is_end = boolean_false;
  do {
    /* embedded IF */
    if (pf_token_is(pf, pf_token_current, pfile_log_none, "end")
      && pf_token_is(pf, pf_token_next, pfile_log_none, "if")) {
      pf_token_get(pf, pf_token_next);
      is_end = boolean_true;
    } else if (pf_token_is(pf, pf_token_current, pfile_log_none, "if")) {
      while (pf_token_is(pf, pf_token_current, pfile_log_none, "if")) {
        pf_token_get(pf, pf_token_next);
        while (!pf_token_is_eof(pf) && !jal_skip_if(pf, boolean_false)) {
          /* empty */
        }
      }
    } else {
      pf_token_get(pf, pf_token_next);
    }
    if (!is_else
      && (pf_token_is(pf, pf_token_current, pfile_log_none, "elsif")
        || pf_token_is(pf, pf_token_current, pfile_log_none, "else"))) {
      break;
    }
  } while (!pf_token_is_eof(pf) && !is_end);
  return is_end;
}

/*
 * NAME
 *   jal_parse_if
 *
 * DESCRIPTION
 *   implement IF
 *
 * PARAMETERS
 *   pf: 
 *
 * RETURN
 *   none
 *
 * NOTES
 *   IF expr THEN .. [ELSIF expr THEN ...] [ELSE ...] END IF
 * becomes:
 *   release for each ELSIF:
 *     if !expr goto label_next
 *       block
 *       goto label_end
 *     label_next
 *     ..
 *     label_end
 */
void jal_parse_if(pfile_t *pf, const pfile_pos_t *statement_start)
{
  label_t     lbl_end;
  boolean_t   else_skip;
  boolean_t   is_first;
  boolean_t   is_end;

  lbl_end = pfile_label_alloc(pf, 0);

  /* else_skip is set to TRUE if any of the preceding IF clauses
     has evaluated to a constant TRUE which means the ELSE can
     never be hit */
  else_skip = boolean_false;
  is_first  = boolean_true;
  is_end    = boolean_false;
  do {
    value_t   expr;
    boolean_t skip_clause;
    label_t   lbl_next;

    if (is_first) {
      is_first = boolean_false;
    } else {
      /* the current token must be "elsif" */
      pf_token_get(pf, pf_token_next);
    }
    expr = jal_parse_expr(pf);
#if 1
    if (!value_is_boolean(expr)) {
      pfile_log(pf, pfile_log_warn, "boolean expression expected");
    }
#endif
    skip_clause = boolean_false;
    if (value_is_const(expr)) {
      if (pfile_flag_test(pf, PFILE_FLAG_WARN_DIRECTIVES)) {
        pfile_log(pf, pfile_log_warn, "compiler directive");
      }
      if (value_const_get(expr)) {
        else_skip = boolean_true;
      } else {
        skip_clause = boolean_true;
      }
    }
    if (skip_clause) {
      pfile_codegen_disable(pf, boolean_true);
      lbl_next = LABEL_NONE;
    } else {
      lbl_next = pfile_label_alloc(pf, 0);
      if (lbl_next) {
        pfile_cmd_branch_add(pf, cmd_branchtype_goto,
            cmd_branchcond_false, lbl_next, expr, 0, 0);
      }
    }
    value_release(expr);
    jal_token_is(pf, "then");
    if (skip_clause) {
      is_end = jal_skip_if(pf, boolean_false);
    } else {
      jal_block_process(pf, JAL_BLOCK_PROCESS_FLAG_NONE);
    }
    if (lbl_end) {
      pfile_cmd_branch_add(pf, cmd_branchtype_goto,
          cmd_branchcond_none, lbl_end, VALUE_NONE, 0, 0);
    }
    if (skip_clause) {
      pfile_codegen_disable(pf, boolean_false);
    } else {
      pfile_cmd_label_add(pf, lbl_next);
      label_release(lbl_next);
    }
  } while (!is_end && pf_token_is(pf, pf_token_current, pfile_log_none, 
        "elsif"));
  if (!is_end && pf_token_is(pf, pf_token_current, pfile_log_none, "else")) {
    pf_token_get(pf, pf_token_next);
    if (else_skip) {
      is_end = jal_skip_if(pf, boolean_true);
    } else {
      jal_block_process(pf, JAL_BLOCK_PROCESS_FLAG_NONE);
    }
  }
  if (!is_end) {
    if (pf_token_is(pf, pf_token_current, pfile_log_err, "end")
      && pf_token_is(pf, pf_token_next, pfile_log_err, "if")) {
      pf_token_get(pf, pf_token_next);
    } else {
      jal_block_start_show(pf, "IF", statement_start);
    }
  }
  pfile_cmd_label_add(pf, lbl_end);
  label_release(lbl_end);
}

/*
 * NAME
 *   jal_parse_usec_delay
 *
 * DESCRIPTION
 *   parse the usec_delay command
 *
 * PARAMETERS
 *
 * RETURN
 *
 * NOTES
 */

void jal_parse_usec_delay(pfile_t *pf, const pfile_pos_t *statement_start)
{
  if (pf_token_is(pf, pf_token_current, pfile_log_err, "(")) {
    value_t cval;

    pf_token_get(pf, pf_token_next);
    cval = jal_parse_expr(pf);
    if (cval) {
      if (!value_is_const(cval)) {
        pfile_log(pf, pfile_log_err, "constant expression expected");
      } else {
        pfile_cmd_usec_delay_add(pf, value_const_get(cval));
      }
      value_release(cval);
    }
    if (pf_token_is(pf, pf_token_current, pfile_log_err, ")")) {
      pf_token_get(pf, pf_token_next);
    }
  }
}

/*
 * NAME
 *   jal_parse_case
 *
 * DESCRIPTION
 *   parse the case command
 *
 * PARAMETERS
 *   pf : pfile handle
 *
 * RETURN
 *   none
 *
 * NOTES
 *   CASE expr OF
 *      cexpr[, cexpr2...] : statement
 *      ...
 *      [OTHERWISE] statement
 *   END CASE
 */

/* #define this to use a single branch with a complex expression;
 * #undef this to use multiple branches */
#undef JAL_CASE_SINGLE_BRANCH

void jal_parse_case(pfile_t *pf, const pfile_pos_t *statement_start)
{
  value_t  expr;   /* controlling expression */
  value_t  brexpr; /* branch expression      */
  label_t  label_done;
  label_t  label_skip;
  unsigned cexpr_ct;
  struct {
    size_t            alloc;
    size_t            used;
    variable_const_t *data;
  } consts_used;
  cmd_t    dummy;

  consts_used.alloc = 0;
  consts_used.used  = 0;
  consts_used.data  = 0;
  expr = jal_parse_expr(pf);
  dummy = cmd_alloc(cmd_type_comment, 0);
  pfile_cmd_add(pf, dummy);

  if (pf_token_is(pf, pf_token_current, pfile_log_err, "of")) {
    pf_token_get(pf, pf_token_next);
  }
  label_done = LABEL_NONE;
  label_skip = LABEL_NONE;
  brexpr     = VALUE_NONE;
  do {
    value_t cexpr;
    label_t label_go;
    value_t val;

    label_go   = LABEL_NONE;

    if (label_done != LABEL_NONE) {
      pfile_cmd_branch_add(pf, cmd_branchtype_goto,
        cmd_branchcond_none, label_done, VALUE_NONE, VALUE_NONE, 0);
      pfile_cmd_label_add(pf, label_skip);
    }
    label_skip = pfile_label_alloc(pf, 0);
    cexpr_ct = 0;
    cexpr    = VALUE_NONE;
    do {
      pfile_pos_t pos;

      pf_token_start_get(pf, &pos);
      pfile_statement_start_set(pf, &pos);
      if (VALUE_NONE != cexpr) {
        /* the current token must be ',' */
        pf_token_get(pf, pf_token_next);
        val = VALUE_NONE;
        pfile_cmd_op_add(pf, operator_eq, &val, expr, cexpr);
#ifndef JAL_CASE_SINGLE_BRANCH
        pfile_cmd_branch_add(pf, cmd_branchtype_goto,
          cmd_branchcond_true, label_go, val, VALUE_NONE, 0);
        value_release(val);
#else
        if (VALUE_NONE == brexpr) {
          brexpr = val;
        } else {
          pfile_cmd_op_add(pf, operator_andl, &brexpr, val, VALUE_NONE);
          value_release(val);
        }
#endif
        value_release(cexpr);
      }
      cexpr = jal_parse_expr(pf);
      if ((cexpr && !value_is_const(cexpr))
        || (!cexpr && (LABEL_NONE != label_go))) {
        pfile_log(pf, pfile_log_err, "constant expression expected");
      } 
      if (cexpr) {
        variable_const_t n;
        size_t           ii;

        n = value_const_get(cexpr);
        for (ii = 0; 
             (ii < consts_used.used) && (consts_used.data[ii] != n);
             ii++)
          ; /* empty body */
        if (ii < consts_used.used) {
          pfile_log(pf, pfile_log_err, "CASE expression already used: %lu",
            n);
        }
        cexpr_ct++;
        if (consts_used.used == consts_used.alloc) {
          size_t newalloc;
          void  *tmp;

          newalloc = (consts_used.alloc) ? 2 * consts_used.alloc : 16;
          tmp = REALLOC(consts_used.data, 
            sizeof(*consts_used.data) * newalloc);
          if (!tmp) {
            pfile_log_syserr(pf, ENOMEM);
          } else {
            consts_used.alloc = newalloc;
            consts_used.data  = tmp;
          }
        }
        if (consts_used.used < consts_used.alloc) {
          consts_used.data[consts_used.used++] = n;
        }
      }
      if (LABEL_NONE == label_go) {
        label_go = pfile_label_alloc(pf, 0);
      }
    } while (pf_token_is(pf, pf_token_current, pfile_log_none, ","));
    if (cexpr_ct) {
      if (cexpr) {
        val = VALUE_NONE;
        pfile_cmd_op_add(pf, operator_eq, &val, expr, cexpr);
#ifndef JAL_CASE_SINGLE_BRANCH
        pfile_cmd_branch_add(pf, cmd_branchtype_goto,
          cmd_branchcond_false, label_skip, val, VALUE_NONE, 0);
#else
        if (brexpr) {
          pfile_cmd_op_add(pf, operator_andl, &brexpr, val, VALUE_NONE);
          pfile_cmd_branch_add(pf, cmd_branchtype_goto,
            cmd_branchcond_false, label_skip, brexpr, VALUE_NONE, 0);
        } else {
          pfile_cmd_branch_add(pf, cmd_branchtype_goto,
            cmd_branchcond_false, label_skip, val, VALUE_NONE, 0);
        }
#endif
        value_release(val);
      }
      if (pf_token_is(pf, pf_token_current, pfile_log_err, ":")) {
        pf_token_get(pf, pf_token_next);
      }
      if (label_go != LABEL_NONE) {
        pfile_cmd_label_add(pf, label_go);
      }
      jal_statement_process(pf);
      if (LABEL_NONE == label_done) {
        label_done = pfile_label_alloc(pf, 0);
      }
    }
    value_release(cexpr);
    label_release(label_go);
    label_release(label_skip);
  } while (cexpr_ct && !pf_token_is_eof(pf));
  if (!consts_used.used) {
    pfile_log(pf, pfile_log_err, "No CASE entries");
  }
  if (value_is_const(expr)) {
    variable_const_t n;
    size_t           ii;

    n = value_const_get(expr);
    for (ii = 0; 
         (ii < consts_used.used) && (consts_used.data[ii] != n);
         ii++)
      ; /* empty body */
    if (ii == consts_used.used) {
      if (pfile_flag_test(pf, PFILE_FLAG_WARN_MISC)) {
        pfile_log(pf, pfile_log_warn, "No matching CASE entries");
      }
    }
  }
  value_release(expr);
  if (pf_token_is(pf, pf_token_current, pfile_log_none, "otherwise")) {
    pf_token_get(pf, pf_token_next);
    jal_statement_process(pf);
  }
  pfile_cmd_label_add(pf, label_done);
  label_release(label_done);
  if (pf_token_is(pf, pf_token_current, pfile_log_err, "end")
    && pf_token_is(pf, pf_token_next, pfile_log_err, "case")) {
    pf_token_get(pf, pf_token_next);
  } else {
    jal_block_start_show(pf, "CASE", statement_start);
  }
  FREE(consts_used.data);
}

