/**********************************************************
 **
 ** jal_vdef.c : JAL variable defintion parsing
 **
 ** Copyright (c) 2004-2005, Kyle A. York
 ** All rights reserved
 **
 ***********************************************************/
#include <errno.h>
#include <string.h>
#include <assert.h>

#include "../libutils/mem.h"
#include "../libcore/pf_msg.h"
#include "../libcore/pf_expr.h"
#include "../libcore/pf_proc.h"
#include "../libcore/pf_block.h"
#include "jal_expr.h"
#include "jal_tokn.h"
#include "jal_vdef.h"

/* 
 * parse:
 *   VAR [[volatile] type] name 
 *   [(cexpr)]
 *   [AT cexpr | variable : bit pos] 
 *   [IS variable] 
 *   [= expr]
 * on entry, the next token is either VOLATILE or type
 *
 * if var exists, this changes a bit
 *    1. require 'in' and/or 'out'
 *    2. don't allow assignment
 */   
static void jal_parse_var_init(pfile_t *pf, variable_t var, size_t el, 
    value_t expr)
{
  size_t         sz;
  variable_def_t vdef;

  vdef = variable_def_get(var);
  if (variable_is_array(var)) {
    vdef = variable_def_member_def_get(
          variable_def_member_get(vdef)
          );
  }
  sz = variable_def_sz_get(vdef);
  if (variable_is_const(var)) {
    if (!value_is_const(expr)) {
      pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
    } else {
      variable_const_set(var, vdef, el * sz, value_const_get(expr));
    }
  } else {
    if (var) {
      value_t val;

      val = value_alloc(var);
      if (val) {
        if (el) {
          value_t ofs;

          ofs = pfile_constant_get(pf, el * sz, VARIABLE_DEF_NONE);
          value_baseofs_set(val, ofs);
          value_release(ofs);
        }
        if (value_is_array(val)) {
          value_dereference(val);
        }
        pfile_cmd_op_add(pf, operator_assign, &val, expr, VALUE_NONE);
        value_release(val);
      }
    }
  }
}

/*
 * determine the smallest type that can hold all initializers
 */
static variable_def_t jal_variable_def_calc(size_t init_ct,
    const value_t *init)
{
  flag_t              flags;
  variable_def_type_t vtype;
  size_t              sz;
#if 0
  variable_const_t    mx;
  size_t              ii;

  if ((1 == init_ct) && value_is_boolean(init[0])) {
    vtype = variable_def_type_boolean;
    flags = VARIABLE_DEF_FLAG_BIT;
    sz    = 1;
  } else {
    for (ii = 0, mx = 0, flags = VARIABLE_DEF_FLAG_NONE; ii < init_ct; ii++) {
      variable_const_t n;

      n = value_const_get(init[ii]);
      if (((long) n) < 0) {
        n = -n;
        flags = VARIABLE_DEF_FLAG_SIGNED;
      }
      if (n > mx) {
        mx = n;
      }
    }
    for (sz = 0; mx; sz++, mx /= 256)
      ;
    if (!sz) {
      sz++;
    }
    vtype = variable_def_type_integer;
  }
#else
  vtype = variable_def_type_integer;
  flags = VARIABLE_DEF_FLAG_NONE;
  sz    = 4;
#endif
  return variable_def_alloc(0, vtype, flags | VARIABLE_DEF_FLAG_UNIVERSAL, 
      sz);
}

variable_t jal_variable_alloc(pfile_t *pf, 
  const jal_variable_info_t *inf, boolean_t is_param,
  size_t init_ct, value_t *init, pfile_variable_alloc_t where)
{
  variable_t     var;
  variable_def_t vdef;

  var = VARIABLE_NONE;

  vdef = (inf->vdef) ? inf->vdef : jal_variable_def_calc(init_ct, init);

  if (inf->name) {
    size_t         ii;
    variable_def_t vvdef;
    flag_t         def_flags;

    def_flags = inf->def_flags | variable_def_flags_get_all(vdef);

    if (inf->master && variable_is_volatile(inf->master)) {
      def_flags |= VARIABLE_DEF_FLAG_VOLATILE;
    }
    vvdef = variable_def_flags_change(vdef, def_flags);
    if (inf->ct) {
      variable_def_t adef;

      adef = variable_def_alloc(0, variable_def_type_array,
          def_flags, 0);
      variable_def_member_add(adef, 0, vvdef, inf->ct);
      vvdef = adef;
    }
    if (is_param) {
      var = variable_alloc(pfile_tag_alloc(pf, inf->name), vvdef);
      if (!var) {
        pfile_log_syserr(pf, ENOMEM);
      }
      variable_master_set(var, inf->master);
    } else {
      if (result_ok != pfile_variable_alloc(pf, 
        where, inf->name, vvdef, inf->master, &var)) {
        var = VARIABLE_NONE;
      }
    }
    variable_flag_set(var, inf->var_flags);

    for (ii = 0; ii < inf->base_ct; ii++) {
      variable_base_set(var, inf->base[ii], ii);
    }
    variable_bit_offset_set(var, inf->bit);

    for (ii = 0; ii < init_ct; ii++) {
      jal_parse_var_init(pf, var, ii, init[ii]);
    }
  }
  return var;
}

void jal_variable_info_init(jal_variable_info_t *inf)
{
  inf->name      = 0;
  inf->master    = VARIABLE_NONE;
  inf->base_ct   = 0;
  inf->bit       = 0;
  inf->ct        = 0;
  inf->var_flags = VARIABLE_FLAG_NONE;
  inf->def_flags = VARIABLE_DEF_FLAG_NONE;
  inf->vdef      = VARIABLE_DEF_NONE;
}

void jal_variable_info_cleanup(jal_variable_info_t *inf)
{
  FREE(inf->name);
  variable_release(inf->master);
}

/* this looks for byte[*n] or other base definition */
variable_def_t jal_vdef_get(pfile_t *pf, flag_t flags)
{
  variable_def_t vdef;
  const char    *ptr;

  ptr = pf_token_get(pf, pf_token_current);

  vdef = pfile_variable_def_find(pf, 
    (flags & VARIABLE_DEF_FLAG_CONST) ? pfile_log_none : pfile_log_err,
    ptr);
  if (!vdef) {
    if (!(flags & VARIABLE_DEF_FLAG_CONST) && jal_token_is_identifier(pf)) {
      /* assume a mis-spelled type and continue */
      pf_token_get(pf, pf_token_next);
    }
  } else {
    pf_token_get(pf, pf_token_next);
    if ((variable_def_sz_get(vdef) == 1)
      && ((variable_def_type_integer == variable_def_type_get(vdef))
        || (variable_def_type_boolean == variable_def_type_get(vdef)))
      && pf_token_is(pf, pf_token_current, pfile_log_none, "*")) {
      value_t val;

      /* this is a user-selected size */
      pf_token_get(pf, pf_token_next);
      val = jal_parse_expr(pf);
      if (val) {
        if (!value_is_const(val)) {
          pfile_log(pf, pfile_log_err, "constant expected");
        } else {
          variable_def_type_t type;

          type = variable_def_type_get(vdef);
          if (variable_def_type_boolean == type) {
            type = variable_def_type_integer;
          }

          vdef = variable_def_alloc(0,
            type,
            variable_def_flags_get_all(vdef),
            value_const_get(val));
        }
        value_release(val);
      }
    }
  }
  return vdef;
}

/*
 * NAME
 *   jal_value_find
 *
 * DESCRIPTION
 *   find a named value
 *
 * PARAMETERS
 *   pf      : pfile handle
 *   name    : value name
 *   val[]   : [out] holds all values found
 *
 * RETURN
 *   block in which the values were found, or NULL if none
 *
 * NOTES
 *   1. unlike most routines, all entries in val[] are set to VALUE_NONE
 *      on entry.
 *   2. val, 'get and 'put, 'vget, or 'vput if defined, must all be defined 
 *      at the same place (same block).
 */
const char *jal_value_type_to_name(char *dst, const char *name, 
    jal_val_type_t type)
{
  const char *fmt_str;

  fmt_str = "%s";
  switch (type) {
    case JAL_VAL_TYPE_CT:   assert(0);
    case JAL_VAL_TYPE_BASE: break;
    case JAL_VAL_TYPE_GET:  fmt_str = "_%s_get"; break;
    case JAL_VAL_TYPE_IGET: fmt_str = "_%s_vget"; break;
    case JAL_VAL_TYPE_PUT:  fmt_str = "_%s_put"; break;
    case JAL_VAL_TYPE_IPUT: fmt_str = "_%s_vput"; break;
  }
  sprintf(dst, fmt_str, name);
  return dst;
}

pfile_block_t *jal_value_find(pfile_t *pf, const char *name,
    value_t val[JAL_VAL_TYPE_CT])
{
  char          *name_buf;
  size_t         name_sz;
  pfile_block_t *blk;

  blk = PFILE_BLOCK_NONE;
  if (!jal_token_is_identifier(pf)) {
    jal_val_type_t ii;

    for (ii = JAL_VAL_TYPE_MIN; ii < JAL_VAL_TYPE_CT; ii++) {
      val[ii] = VALUE_NONE;
    }
  } else {
    /* space for the passed in name is not needed */
    name_sz  = 7 + strlen(name);
    name_buf = MALLOC(name_sz);
    if (!name_buf) {
      pfile_log_syserr(pf, ENOMEM);
    } else {
      /* fill the name buffers */
      pfile_proc_t  *proc;
      variable_t     var[JAL_VAL_TYPE_CT];
      jal_val_type_t ii;
      boolean_t      found;

      for (ii = JAL_VAL_TYPE_MIN; ii < JAL_VAL_TYPE_CT; ii++) {
        var[ii] = VARIABLE_NONE;
      }

      for (proc = pfile_proc_active_get(pf), found = boolean_false;
           proc && !found;
           proc = pfile_proc_parent_get(proc)) {

        for (blk = pfile_proc_block_active_get(proc);
             blk;
             blk = pfile_block_parent_get(blk)) {
          for (ii = JAL_VAL_TYPE_MIN; ii < JAL_VAL_TYPE_CT; ii++) {
            var[ii] = pfile_block_variable_find(blk, jal_value_type_to_name(
                  name_buf, name, ii));
            if (VARIABLE_NONE != var[ii]) {
              found = boolean_true;
              if (JAL_VAL_TYPE_BASE == ii) {
                /* if this was an alias, the name will have changed */
                name = variable_name_get(var[ii]);
                FREE(name_buf);
                name_sz  = 7 + strlen(name);
                name_buf = MALLOC(name_sz);
              }
            }
          }
          if (found) {
            break; /* don't want to change blk! */
          }
        }
      }
      for (ii = JAL_VAL_TYPE_MIN; ii < JAL_VAL_TYPE_CT; ii++) {
        if (var[ii]) {
          val[ii] = value_alloc(var[ii]);
          variable_release(var[ii]);
        } else {
          val[ii] = VALUE_NONE;
        }
      }
      if (!found && !jal_token_is_keyword(pf)) {
        /* assume a mis-spelling & simply create the current token
         * as a BYTE value */
        pfile_log(pf, pfile_log_err, "\"%s\" not defined",
          pf_token_ptr_get(pf));
        pfile_value_alloc(pf, PFILE_VARIABLE_ALLOC_LOCAL,
            pf_token_ptr_get(pf),
            pfile_variable_def_find(pf, pfile_log_none, "byte"),
            &val[JAL_VAL_TYPE_BASE]);
        found = boolean_true;
      }
      FREE(name_buf);
    }
  }
  return blk;
}

void jal_value_release(value_t val[JAL_VAL_TYPE_CT])
{
  size_t ii;

  for (ii = 0; ii < JAL_VAL_TYPE_CT; ii++) {
    value_release(val[ii]);
  }
}


void jal_parse_var_common(pfile_t *pf, variable_t *dst, flag_t flags)
{
  variable_def_t vdef;
  const char    *ptr;
  pf_token_get_t which;

  which = pf_token_current;
  if (pf_token_is(pf, pf_token_current, pfile_log_none, "volatile")) {
    flags |= VARIABLE_DEF_FLAG_VOLATILE;
    which = pf_token_next;
  }
  ptr = pf_token_get(pf, which);
  vdef = jal_vdef_get(pf, flags);
  which = pf_token_current;
  flags |= variable_def_flags_get_all(vdef);
  do {
    /* parse a list of variable names */
    jal_variable_info_t inf;

    jal_variable_info_init(&inf);

    inf.vdef      = vdef;
    inf.def_flags = flags;

    if (pf_token_is(pf, which, pfile_log_none, "volatile")) {
      inf.def_flags |= VARIABLE_DEF_FLAG_VOLATILE;
      which = pf_token_next;
    } else {
      which = pf_token_current;
    }
    if (dst) {
      if (pf_token_is(pf, which, pfile_log_none, "in")) {
        inf.def_flags |= VARIABLE_DEF_FLAG_IN;
        which = pf_token_next;
      } else {
        which = pf_token_current;
      }
      if (pf_token_is(pf, which, pfile_log_none, "out")) {
        inf.def_flags |= VARIABLE_DEF_FLAG_OUT;
        which = pf_token_next;
      } else {
        which = pf_token_current;
      }
      if (!(inf.def_flags & (VARIABLE_DEF_FLAG_IN | VARIABLE_DEF_FLAG_OUT))) {
        pfile_log(pf, pfile_log_err, "in and/or out required");
      }
    }

    ptr = pf_token_get(pf, which);
    if (!jal_token_is_identifier(pf)) {
      pfile_log(pf, pfile_log_err, PFILE_MSG_VARIABLE_EXPECTED);
    } else {
      /* encapsulate everything needed to create this variable */
      variable_t   var;

      var = VARIABLE_NONE;

      inf.name = STRDUP(pf_token_get(pf, pf_token_current));
      if (!inf.name) {
        pfile_log_syserr(pf, ENOMEM);
      }
      if (pf_token_is(pf, pf_token_next, pfile_log_none, "[")) {
        /* an array */
        if (pf_token_is(pf, pf_token_next, pfile_log_none, "]")) {
          inf.ct = -1;
        } else {
          value_t n;

          n = jal_parse_expr(pf);
          if (n) {
            if (!value_is_const(n)) {
              pfile_log(pf, pfile_log_err, "constant expected");
            } else if (value_const_get(n) <= 0) {
              pfile_log(pf, pfile_log_err, "dimension must be >= 1");
            } else {
              inf.ct = value_const_get(n);
            }
            value_release(n);
          }
        }
        if (pf_token_is(pf, pf_token_current, pfile_log_err, "]")) {
          pf_token_get(pf, pf_token_next);
        }
      }
      if (pf_token_is(pf, pf_token_current, pfile_log_none, "shared")) {
        inf.var_flags = VARIABLE_FLAG_SHARED;
        pf_token_get(pf, pf_token_next);
      }
      if (pf_token_is(pf, pf_token_current, pfile_log_none, "at")) {
        /* a placed variable */
        if (pf_token_is(pf, pf_token_next, pfile_log_none, "{")) {
          /* 
           * this is a variable mirrored in multiple places
           */
          size_t ii;

          for (ii = 0; 
               (ii < VARIABLE_MIRROR_CT)
               && (!ii 
                 || pf_token_is(pf, pf_token_current, pfile_log_none, ",")); 
                 ii++) {
            /* skip the "{" or "," */
            value_t n;

            pf_token_get(pf, pf_token_next);
            n = jal_parse_expr(pf);
            if (n) {
              if (!value_is_const(n)) {
                pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
              } else {
                inf.base[inf.base_ct] = value_const_get(n);
                inf.base_ct++;
              }
              value_release(n);
            }
          }
          if (pf_token_is(pf, pf_token_current, pfile_log_err, "}")) {
            pf_token_get(pf, pf_token_next);
          }
        } else {
          ptr    = pf_token_get(pf, pf_token_current);
          inf.master = pfile_variable_find(pf, pfile_log_none, ptr, 0);
          /* if this is a bit variable, the master will be set to
             _bit_bucket. We need to remove that */
          if (inf.master && !variable_is_const(inf.master)) {
            /* this is AT variable [: bitpos] */
            pf_token_get(pf, pf_token_next);
            if (variable_is_array(inf.master)
                && pf_token_is(pf, pf_token_current, pfile_log_none, "[")) {
              /* an array offset! */
              value_t               n;

              pf_token_get(pf, pf_token_next);
              n = jal_parse_expr(pf);
              if (n) {
                variable_def_t        def;
                variable_def_member_t mbr;

                mbr = variable_def_member_get(
                        variable_def_get(inf.master));
                def = variable_def_member_def_get(mbr);

                if (!value_is_const(n)) {
                  pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
                } else if (value_const_get(n) 
                    >= variable_def_member_ct_get(mbr)) {
                  pfile_log(pf, pfile_log_err, "subscript out of range");
                } else {
                  inf.base[0] = value_const_get(n) * variable_def_sz_get(def);
                  inf.base_ct = 1;
                }
                value_release(n);
              }
              if (pf_token_is(pf, pf_token_current, pfile_log_err, "]")) {
                pf_token_get(pf, pf_token_next);
              }
            }
          } else {
            /* this is AT cexpr [ : bitpos] */
            value_t n;

            n = jal_parse_expr(pf);
            if (n) {
              if (!value_is_const(n)) {
                pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
              } else {
                inf.base[0] = value_const_get(n);
                inf.base_ct = 1;
              }
              value_release(n);
            }
          }
          if (pf_token_is(pf, pf_token_current, pfile_log_none, ":")) {
            value_t n;

            pf_token_get(pf, pf_token_next);
            n = jal_parse_expr(pf);
            if (n) {
              if (!value_is_const(n)) {
                pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
              } else {
                inf.bit = value_const_get(n);
              }
              value_release(n);
            }
          }
        }
      } else if (pf_token_is(pf, pf_token_current, pfile_log_none, "is")) {
        /* an alias */
        value_t        jvals[JAL_VAL_TYPE_CT];
        pfile_block_t *jval_blk;

        ptr = pf_token_get(pf, pf_token_next);
#if 0
        inf.master = pfile_variable_find(pf, pfile_log_err, ptr, 0);
        if (!inf.master) {
          pfile_log(pf, pfile_log_err, PFILE_MSG_VARIABLE_EXPECTED);
        } else {
          if (!variable_def_is_same(inf.vdef, variable_def_get(inf.master))) {
            pfile_log(pf, pfile_log_err, "type mismatch");
          }
          pf_token_get(pf, pf_token_next);
          inf.var_flags = VARIABLE_FLAG_ALIAS;
        }
#else
        jval_blk = jal_value_find(pf, ptr, jvals);
        if (jval_blk) {
          /* If JAL_VAL_TYPE_BASE is set, no problem. Otherwise
           * we need to create one based on whether the appropriate
           * 'get and/or 'put routines are in place
           */
          if (VALUE_NONE == jvals[JAL_VAL_TYPE_BASE]) {
            /* create a new variable */
            variable_t     jvar;
            variable_def_t jdef;
            flag_t         jvar_flags;

            /* let's find the definition! */
            jvar_flags = VARIABLE_FLAG_NONE;
            jdef       = VARIABLE_DEF_NONE;
            if (jvals[JAL_VAL_TYPE_GET]) {
              /* the definition is the return value, or first parameter */
              jdef = value_def_get(jvals[JAL_VAL_TYPE_GET]);
              jdef = variable_def_member_def_get(
                       variable_def_member_get(
                         jdef));
              jvar_flags |= VARIABLE_FLAG_READ;
            } 
            if (jvals[JAL_VAL_TYPE_PUT]) {
              /* the definition is the second parameter */
              if (!jdef) {
                jdef = value_def_get(jvals[JAL_VAL_TYPE_PUT]);
                jdef = variable_def_member_def_get(
                         variable_def_member_link_get(
                           variable_def_member_get(
                             jdef)));
              }
              jvar_flags |= VARIABLE_FLAG_WRITE;
            } 
            if (!jdef) {
              abort();
            }
            jvar = VARIABLE_NONE;
            pfile_block_variable_alloc(jval_blk,
              pfile_tag_alloc(pf, ptr), jdef, VARIABLE_NONE, pf, &jvar);
            if (jvar) {
              variable_flags_set_all(jvar, jvar_flags);
              jvals[JAL_VAL_TYPE_BASE] = value_alloc(jvar);
              variable_release(jvar);
            }
          }
          pf_token_get(pf, pf_token_next);
          inf.master = value_variable_get(jvals[JAL_VAL_TYPE_BASE]);
          variable_lock(inf.master);
          if (!inf.master) {
            pfile_log(pf, pfile_log_err, PFILE_MSG_VARIABLE_EXPECTED);
          } else {
            if (!variable_def_is_same(inf.vdef, 
                variable_def_get(inf.master))) {
              pfile_log(pf, pfile_log_err, "type mismatch");
            }
            inf.var_flags = VARIABLE_FLAG_ALIAS;
          }
          jal_value_release(jvals);
        }
#endif
      }
      if (!vdef 
          || pf_token_is(pf, pf_token_current, 
               (flags & VARIABLE_DEF_FLAG_CONST) 
                ? pfile_log_err : pfile_log_none, "=")) {
        /* pre-assigned */
        size_t      ii;
        boolean_t   need_close;
        value_t     init_1;  /* no need to allocate only 1! */
        value_t    *init;
        unsigned    init_ct;
        unsigned    init_alloc;
        const char *str;

        pf_token_get(pf, pf_token_next);
        str = 0;
        if (inf.ct) {
          need_close = boolean_true;
          str = pf_token_get(pf, pf_token_current);
          if ('"' == *str) {
            need_close = boolean_false;
          } else {
            if (pf_token_is(pf, pf_token_current, pfile_log_err, "{")) {
              pf_token_get(pf, pf_token_next);
            }
            str = 0;
          }
          init = 0;
          init_alloc = 0;
        } else {
          need_close = boolean_false;
          init = &init_1;
          init_alloc = 1;
        }
        init_ct = 0;

        if (str) {
          size_t sz;

          while (str) {
            str++;
            sz = pf_token_sz_get(pf) - 1;
            if (sz && ('"' == str[sz-1])) {
              sz--;
            }
            if (sz) {
              void *tmp;

              tmp = REALLOC(init, sizeof(*init) * (init_ct + sz));
              if (!tmp) {
                pfile_log_syserr(pf, ENOMEM);
              } else {
                init = tmp;
                for (ii = 0; ii < sz; ii++) {
                  init[init_ct + ii] = pfile_constant_get(pf, str[ii], VARIABLE_DEF_NONE);
                }
                init_ct += sz;
              }
              str = pf_token_get(pf, pf_token_next);
              if ('"' != *str) {
                str = 0;
              }
            }
          }
        } else {
          for (ii = 0, init_ct = 0; 
               (!ii || pf_token_is(pf, pf_token_current, pfile_log_none, ","))
               && !(need_close 
                 && pf_token_is(pf, pf_token_current, pfile_log_none, "}"))
               && (need_close || (0 == ii));
               ii++) {
            value_t expr;

            if (ii) {
              pf_token_get(pf, pf_token_next);
            }
            expr = jal_parse_expr(pf);
            if (expr) {
              if ((!vdef 
                || variable_def_flag_test(vdef, VARIABLE_DEF_FLAG_CONST))
                && !value_is_const(expr)) {
                pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
                value_release(expr);
                expr = VALUE_NONE;
              } 
            }
            if (init || !init_ct) {
              if (init_ct == init_alloc) {
                value_t *tmp;

                init_alloc = (init_alloc)
                  ? init_alloc * 2
                  : 256;
                tmp = REALLOC(init, sizeof(*tmp) * init_alloc);
                if (!tmp) {
                  pfile_log_syserr(pf, ENOMEM);
                  while (init_ct) {
                    value_release(init[--init_ct]);
                  }
                  FREE(init);
                  init = 0;
                } else {
                  init = tmp;
                }
              }
              if (init && (init_ct < init_alloc)) {
                init[init_ct++] = expr;
              }
            }
          }
        }
        if (-1 == inf.ct) {
          inf.ct = init_ct;
        }
        if (need_close && pf_token_is(pf, pf_token_current, pfile_log_err, 
          "}")) {
          pf_token_get(pf, pf_token_next);
        }
        if (init_ct < inf.ct) {
          if (pfile_flag_test(pf, PFILE_FLAG_WARN_MISC)) {
            pfile_log(pf, pfile_log_warn, "fewer initializers than expected");
          }
        } else if (inf.ct && (init_ct > inf.ct)) {
          pfile_log(pf, pfile_log_err, "more initializers than allowed");
        }
        if (init) {
          if ((VARIABLE_DEF_NONE == inf.vdef) && (1 == init_ct)) {
            inf.vdef = value_def_get(init[0]);
          }
          var = jal_variable_alloc(pf, &inf, dst != 0, init_ct,
              init, PFILE_VARIABLE_ALLOC_LOCAL);
          while (init_ct) {
            value_release(init[--init_ct]);
          }
          if (init != &init_1) {
            FREE(init);
          }
        } else {
          var = VARIABLE_NONE;
        }

        jal_variable_info_cleanup(&inf);
      } else {
        var = jal_variable_alloc(pf, &inf, dst != 0, 0, 0, 
            PFILE_VARIABLE_ALLOC_LOCAL);
        jal_variable_info_cleanup(&inf);
      }
      if (dst) {
        *dst = var;
      } else {
        variable_release(var);
      }
    }
    which = pf_token_next;
  } while (!dst && pf_token_is(pf, pf_token_current, pfile_log_none, ","));
}

void jal_parse_var(pfile_t *pf, const pfile_pos_t *statement_start)
{
  jal_parse_var_common(pf, 0, VARIABLE_DEF_FLAG_NONE);
}

void jal_parse_const(pfile_t *pf, const pfile_pos_t *statement_start)
{
  jal_parse_var_common(pf, 0, VARIABLE_DEF_FLAG_CONST);
}

