/************************************************************
 **
 ** pic16.c : 16-bit pic code generation definitions
 **
 ** Copyright (c) 2007, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/
#include "pic16.h"

boolean_t pic16_code_to_pcode(pfile_t *pf, pic_code_t code,
    unsigned val, unsigned literal, pic_code_to_pcode_t *pdst)
{
  unsigned pcode_lo;
  unsigned pcode_hi;
  unsigned dst;
  unsigned access;
  unsigned bit;
  unsigned lit1;
  unsigned lit8;
  unsigned lit9;
  unsigned lit11;
  unsigned lit12;
  ulong    lit20;
  unsigned shadow;
  unsigned ct;

  bit     = (literal & 0x07) << 9;
  lit1    = literal & 0x00000001; /*  1 bit literal */
  lit8    = literal & 0x000000ff; /*  8 bit literal */
  lit9    = literal & 0x000001ff; /*  9 bit literal */
  lit11   = literal & 0x000007ff; /* 11 bit literal */
  lit12   = literal & 0x00000fff; /* 12 bit literal */
  lit20   = literal & 0x000fffff; /* 20 bit literal */
  dst     = (pic_opdst_f == pic_code_dst_get(code)) << 9;
  access  = (variable_flag_test(
              value_variable_get(
                pic_code_value_get(code)
                ),
              VARIABLE_FLAG_SHARED)) ? 0 : 0x0100;
  shadow  = 0;
  pcode_lo= 0xffff; /* an impossible code (it's greater than 14 bits) */
  pcode_hi= 0x0000;
  ct      = 1;
  switch (pic_code_op_get(code)) {
    /* directives (no code) */
    case pic_opcode_org:    /* ORG xxxx */
    case pic_opcode_end:    /* END      */
    case pic_opcode_none:   /* no code here */
      ct = 0;
      break;
    case pic_opcode_nop:    
      pcode_lo = 0x0000; 
      pcode_hi = 0xf000; 
      ct       = 2;
      break;
    /* file operator (code, dst, val) */
    case pic_opcode_clrf:   pcode_lo = 0x6a00 |       access | val; break;
    case pic_opcode_cpfseq: pcode_lo = 0x6200 |       access | val; break;
    case pic_opcode_cpfsgt: pcode_lo = 0x6400 |       access | val; break;
    case pic_opcode_cpfslt: pcode_lo = 0x6000 |       access | val; break;
    case pic_opcode_movwf:  pcode_lo = 0x6e00 |       access | val; break;
    case pic_opcode_mulwf:  pcode_lo = 0x0200 |       access | val; break;
    case pic_opcode_negf:   pcode_lo = 0x6c00 |       access | val; break;
    case pic_opcode_setf:   pcode_lo = 0x6800 |       access | val; break;
    case pic_opcode_tstfsz: pcode_lo = 0x6600 |       access | val; break;
    case pic_opcode_addwf:  pcode_lo = 0x2400 | dst | access | val; break;
    case pic_opcode_addwfc: pcode_lo = 0x2000 | dst | access | val; break;
    case pic_opcode_andwf:  pcode_lo = 0x1400 | dst | access | val; break;
    case pic_opcode_comf:   pcode_lo = 0x1c00 | dst | access | val; break;
    case pic_opcode_decf:   pcode_lo = 0x0400 | dst | access | val; break;
    case pic_opcode_decfsz: pcode_lo = 0x2c00 | dst | access | val; break;
    case pic_opcode_dcfsnz: pcode_lo = 0x4c00 | dst | access | val; break;
    case pic_opcode_incf:   pcode_lo = 0x2800 | dst | access | val; break;
    case pic_opcode_incfsz: pcode_lo = 0x3c00 | dst | access | val; break;
    case pic_opcode_infsnz: pcode_lo = 0x4800 | dst | access | val; break;
    case pic_opcode_iorwf:  pcode_lo = 0x1000 | dst | access | val; break;
    case pic_opcode_movf:   pcode_lo = 0x5000 | dst | access | val; break;
    case pic_opcode_rlcf:
    case pic_opcode_rlf:    pcode_lo = 0x3400 | dst | access | val; break;
    case pic_opcode_rlncf:  pcode_lo = 0x4400 | dst | access | val; break;
    case pic_opcode_rrf:    pcode_lo = 0x3000 | dst | access | val; break;
    case pic_opcode_rrncf:  pcode_lo = 0x4000 | dst | access | val; break;
    case pic_opcode_subfwb: pcode_lo = 0x5400 | dst | access | val; break;
    case pic_opcode_subwf:  pcode_lo = 0x5c00 | dst | access | val; break;
    case pic_opcode_subwfb: pcode_lo = 0x5800 | dst | access | val; break;
    case pic_opcode_swapf:  pcode_lo = 0x3800 | dst | access | val; break;
    case pic_opcode_xorwf:  pcode_lo = 0x1800 | dst | access | val; break;
    /* simple opcodes (no operands) */
    case pic_opcode_clrwdt: pcode_lo = 0x0004; break;
    case pic_opcode_daw:    pcode_lo = 0x0007; break;
    case pic_opcode_retfie: pcode_lo = 0x0010 | lit1; break;
    case pic_opcode_return: pcode_lo = 0x0012 | lit1; break;
    case pic_opcode_push:   pcode_lo = 0x0005; break;
    case pic_opcode_pop:    pcode_lo = 0x0006; break;
    case pic_opcode_reset:  pcode_lo = 0x00ff; break;
    case pic_opcode_sleep:  pcode_lo = 0x0003; break;
    /* bit oriented (3 bit literal) */
    case pic_opcode_bcf:    pcode_lo = 0x9000 | bit | access | val; break;
    case pic_opcode_bsf:    pcode_lo = 0x8000 | bit | access | val; break;
    case pic_opcode_btfsc:  pcode_lo = 0xb000 | bit | access | val; break;
    case pic_opcode_btfss:  pcode_lo = 0xa000 | bit | access | val; break;
    case pic_opcode_btg:    pcode_lo = 0x7000 | bit | access | val; break;
    /* literal & w  (8 bit literal) */
    case pic_opcode_movlw:  pcode_lo = 0x0e00 | lit8; break;
    case pic_opcode_retlw:  pcode_lo = 0x0c00 | lit8; break;
    case pic_opcode_iorlw:  pcode_lo = 0x0900 | lit8; break;
    case pic_opcode_andlw:  pcode_lo = 0x0b00 | lit8; break;
    case pic_opcode_xorlw:  pcode_lo = 0x0a00 | lit8; break;
    case pic_opcode_sublw:  pcode_lo = 0x0800 | lit8; break;
    case pic_opcode_addlw:  pcode_lo = 0x0f00 | lit8; break;
    case pic_opcode_mullw:  pcode_lo = 0x0d00 | lit8; break;
    case pic_opcode_movlb:  pcode_lo = 0x0100 | lit8; break;
    /* branching (8 or 9 bit literal) */
    case pic_opcode_call:   pcode_lo = 0xec00 | (lit20 & 0x000000ff) | shadow;
                            pcode_hi = 0xf000 | (lit20 >> 8);
                            ct       = 2;
                            break;
    case pic_opcode_goto:   pcode_lo = 0xef00 | (lit20 & 0x000000ff);
                            pcode_hi = 0xf000 | (lit20 >> 8);
                            ct       = 2;
                            break;
    case pic_opcode_bc:     pcode_lo = 0xe200 | lit8; break;
    case pic_opcode_bn:     pcode_lo = 0xe600 | lit8; break;
    case pic_opcode_bnc:    pcode_lo = 0xe300 | lit8; break;
    case pic_opcode_bnn:    pcode_lo = 0xe700 | lit8; break;
    case pic_opcode_bnov:   pcode_lo = 0xe500 | lit8; break;
    case pic_opcode_bnz:    pcode_lo = 0xe100 | lit8; break;
    case pic_opcode_bov:    pcode_lo = 0xe400 | lit8; break;
    case pic_opcode_bz:     pcode_lo = 0xe000 | lit8; break;
    case pic_opcode_bra:    pcode_lo = 0xd000 | lit11; break;
    case pic_opcode_rcall:  pcode_lo = 0xd800 | lit11; break;
    /* special (pseudo) opcodes */
    case pic_opcode_db: /* this is handled elsewhere */
      break;
    case pic_opcode_lfsr:
      pcode_lo = 0xee00 | (pic_code_ofs_get(code) << 4) | (lit12 >> 8);
      pcode_hi = 0xf000 | (lit12 & 0x00ff);
      ct = 2;
      break;
    case pic_opcode_tblrd:
      pcode_lo = 0x0008 | (lit12 & 0x03);
      break;
    case pic_opcode_tblwt:
      pcode_lo = 0x000c | (lit12 & 0x03);
      break;
    case pic_opcode_movff:
    case pic_opcode_branchlo_clr:
    case pic_opcode_branchlo_set:
    case pic_opcode_branchlo_nop:
    case pic_opcode_branchhi_clr:
    case pic_opcode_branchhi_set:
    case pic_opcode_branchhi_nop:
    case pic_opcode_datalo_clr:
    case pic_opcode_datalo_set:
    case pic_opcode_datahi_clr:
    case pic_opcode_datahi_set:
    case pic_opcode_irp_set:
    case pic_opcode_irp_clr:
    case pic_opcode_clrw:
    case pic_opcode_tris:         /* tris 6 or 7 */
    case pic_opcode_option:       /* option */
      pfile_log(pf, pfile_log_err, "illegal instruction");
      ct = 0;
      break;
  }
  pdst->ct      = ct;
  pdst->code[0] = pcode_lo;
  pdst->code[1] = pcode_hi;
  return pdst->ct != 0;
}

