/**********************************************************
 **
 ** pic_code.c : manipulators for pic_code_t
 **
 ** Copyright (c) 2004-2005, Kyle A. York
 ** All rights reserved
 **
 ***********************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#include "../libcore/cmd_asm.h"
#include "pic.h"
#include "pic_code.h" 

struct pic_code_ {
  pic_code_t        next;
  pic_code_t        prev;

  flag_t            flags;

  cmd_t             cmd;
  pic_pc_t          pc;       /* program counter */

  /* databits are modified by variable reference */
  pic_databits_state_t databits;
  uchar                bsr;     /* track what's in BSR */
  /* branchbits are modified by branching instructions */
  pic_branchbits_state_t branchbits;

  label_t            label;
  pic_opcode_t       op;
  /* unsigned char  op; */
  /* pic_opcode_t       op; */
  /* pic_instr_append_f_d   */
  /* pic_instr_append_reg_d */
  /* pic_instr_append_f     */
  /* pic_instr_append_reg   */
  value_t   value; /* this *MUST BE* a variable (not a constant) */
  size_t    ofs;
  pic_opdst_t dst;          /* f, w or none   */
  /* pic_instr_append_f_b      */
  /* pic_instr_append_reg_bn   */
  /* pic_instr_append_reg_flag */
  /* pic_instr_append_w_k      */
  /* pic_instr_append_w_kn     */
  value_t   literal; /* bit or literal */
  /* pic_instr_append_n        */
  label_t  brdst;

  unsigned depth;

  value_t w_value; /* tracks what W is holding */

  size_t   data_sz; /* data size for pic_opcode_db */
  void    *data;    /* the actual data             */
};

static cache_t   pic_code_cache;
static boolean_t pic_code_cache_is_init;

static void pic_code_cache_cleanup(void)
{
  cache_cleanup(&pic_code_cache);
}

static pic_code_t pic_code_element_alloc(void)
{
  if (!pic_code_cache_is_init) {
    pic_code_cache_is_init = boolean_true;
    atexit(pic_code_cache_cleanup);
    cache_init(&pic_code_cache, sizeof(struct pic_code_), "pic_code");
  }
  return cache_element_alloc(&pic_code_cache);
}

static struct pic_code_ *pic_code_element_seek(pic_code_t el, boolean_t mod)
{
  return cache_element_seek(&pic_code_cache, el, mod);
}

pic_code_t pic_code_alloc(label_t lbl, pic_opcode_t op, flag_t flags)
{
  pic_code_t code;

  code = pic_code_element_alloc();
  if (code) {
    struct pic_code_ *ptr;
    
    ptr = pic_code_element_seek(code, boolean_true);
    
    if (ptr) {
      static const pic_databits_state_t   db_state_none = {{0}};
      static const pic_branchbits_state_t bb_state_none = {{0}};

      ptr->next       = PIC_CODE_NONE;
      ptr->prev       = PIC_CODE_NONE;
      ptr->flags      = PIC_CODE_FLAG_NONE;
      ptr->cmd        = CMD_NONE;
      ptr->pc         = 0;
      ptr->databits   = db_state_none;
      ptr->branchbits = bb_state_none;
      ptr->label      = LABEL_NONE;
      ptr->op         = pic_opcode_none;
      ptr->value      = VALUE_NONE;
      ptr->ofs        = 0;
      ptr->dst        = pic_opdst_none;
      ptr->literal    = VALUE_NONE;
      ptr->brdst      = LABEL_NONE;
      ptr->depth      = -1;
      pic_code_label_set(code, lbl);
      pic_code_op_set(code, op);
      pic_code_flag_set_all(code, flags);
      ptr->w_value    = VALUE_NONE;
      ptr->bsr        = PIC16_BSR_UNKNOWN;
      ptr->data_sz    = 0;
      ptr->data       = 0;
    }
  }
  return code;
}

void pic_code_free(pic_code_t code)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_true);
  if (ptr) {
    pic_code_label_set(code, LABEL_NONE);
    value_release(ptr->value);
    value_release(ptr->literal);
    label_release(ptr->brdst);
    value_release(ptr->w_value);
    free(ptr->data);
    cache_element_free(&pic_code_cache, code);
  }
}


void pic_code_prev_set(pic_code_t code, pic_code_t link)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_true);
  if (ptr) {
    ptr->prev = link;
  }
}

pic_code_t pic_code_prev_get(pic_code_t code)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_false);
  return (ptr) ? ptr->prev : 0;
}

void pic_code_next_set(pic_code_t code, pic_code_t link)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_true);
  if (ptr) {
    ptr->next = link;
  }
}

pic_code_t pic_code_next_get(pic_code_t code)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_false);
  return (ptr) ? ptr->next : 0;
}

void pic_code_cmd_set(pic_code_t code, cmd_t cmd)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_true);
  if (ptr) {
    ptr->cmd = cmd;
  }
}

cmd_t pic_code_cmd_get(pic_code_t code)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_false);
  return (ptr) ? ptr->cmd : 0;
}

void pic_code_pc_set(pic_code_t code, pic_pc_t pc)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_true);
  if (ptr) {
    ptr->pc = pc;
  }
}

pic_pc_t pic_code_pc_get(pic_code_t code)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_false);
  return (ptr) ? ptr->pc : 0;
}

void pic_code_label_set(pic_code_t code, label_t label)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_true);
  if (ptr) {
    label_lock(label);
    label_release(ptr->label);
    ptr->label = label;
    label_code_set(label, (void *) code);
  }
}

label_t pic_code_label_get(pic_code_t code)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_false);
  return (ptr) ? ptr->label : 0;
}

void pic_code_op_set(pic_code_t code, pic_opcode_t op)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_true);
  if (ptr) {
    ptr->op = op;
  }
}

pic_opcode_t pic_code_op_get(pic_code_t code)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_false);
  return (ptr) ? ptr->op : 0;
}

void pic_code_value_set(pic_code_t code, value_t val)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_true);
  if (ptr) {
    value_lock(val);
    value_release(ptr->value);
    ptr->value = val;
  }
}

value_t pic_code_value_get(pic_code_t code)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_false);
  return (ptr) ? ptr->value : 0;
}

void pic_code_ofs_set(pic_code_t code, size_t ofs)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_true);
  if (ptr) {
    ptr->ofs = ofs;
  }
}

size_t pic_code_ofs_get(pic_code_t code)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_false);
  return (ptr) ? ptr->ofs : 0;
}

void pic_code_dst_set(pic_code_t code, pic_opdst_t dst)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_true);
  if (ptr) {
    ptr->dst = dst;
  }
}

pic_opdst_t pic_code_dst_get(pic_code_t code)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_false);
  return (ptr) ? ptr->dst : 0;
}

void pic_code_literal_set(pic_code_t code, value_t lit)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_true);
  if (ptr) {
    value_lock(lit);
    value_release(ptr->literal);
    ptr->literal = lit;
  }
}

value_t pic_code_literal_get(pic_code_t code)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_false);
  return (ptr) ? ptr->literal : 0;
}

void pic_code_brdst_set(pic_code_t code, label_t brdst)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_true);
  if (ptr) {
    label_lock(brdst);
    label_usage_bump(ptr->brdst, ctr_bump_decr);
    label_release(ptr->brdst);
    ptr->brdst = brdst;
    label_usage_bump(brdst, ctr_bump_incr);
  }
}

label_t pic_code_brdst_get(pic_code_t code)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_false);
  return (ptr) ? ptr->brdst : 0;
}

void pic_code_databits_get(pic_code_t code, pic_databits_state_t *dst)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_false);
  if (ptr) {
    *dst = ptr->databits;
  }
}

void pic_code_databits_set(pic_code_t code, const pic_databits_state_t *src)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_true);
  if (ptr) {
    ptr->databits = *src;
  }
}

void pic_code_branchbits_get(pic_code_t code, pic_branchbits_state_t *dst)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_false);
  if (ptr) {
    *dst = ptr->branchbits;
  }
}

void pic_code_branchbits_set(pic_code_t code, 
    const pic_branchbits_state_t *src)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_true);
  if (ptr) {
    ptr->branchbits = *src;
  }
}

boolean_t  pic_code_flag_test(pic_code_t code, flag_t flag)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_false);
  return ptr && ((ptr->flags & flag) == flag);
}

void pic_code_flag_set(pic_code_t code, flag_t flag)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_true);
  if (ptr) {
    ptr->flags |= flag;
  }
}

void pic_code_flag_clr(pic_code_t code, flag_t flag)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_true);
  if (ptr) {
    ptr->flags &= ~flag;
  }
}

flag_t pic_code_flag_get_all(pic_code_t code)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_false);
  return ptr ? ptr->flags : 0;
}

void pic_code_flag_set_all(pic_code_t code, flag_t flags)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_true);
  if (ptr) {
    ptr->flags = flags;
  }
}

void pic_code_depth_set(pic_code_t code, unsigned depth)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_true);
  if (ptr) {
    ptr->depth = depth;
  }
}

unsigned pic_code_depth_get(pic_code_t code)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_false);
  return (ptr) ? ptr->depth : 0;
}


value_t pic_code_w_value_get(const pic_code_t code)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_false);
  return (ptr) ? ptr->w_value : VALUE_NONE;
}

void pic_code_w_value_set(pic_code_t code, value_t val)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_true);
  if (ptr) {
    value_lock(val);
    value_release(ptr->w_value);
    ptr->w_value = val;
  }
}

uchar pic_code_bsr_get(const pic_code_t code)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_false);
  return (ptr) ? ptr->bsr : PIC16_BSR_UNKNOWN;
}

void pic_code_bsr_set(pic_code_t code, uchar bsr)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_true);
  if (ptr) {
    ptr->bsr = bsr;
  }
}

/* return the # of words used by this code */
size_t pic_code_sz_get(pic_code_t code, pic_target_cpu_t target)
{
  size_t sz;

  sz = 0;
  switch (pic_code_op_get(code)) {
    case pic_opcode_org:
    case pic_opcode_end:
    case pic_opcode_none:
    case pic_opcode_branchlo_nop:
    case pic_opcode_branchhi_nop:
      sz = 0;
      break;
    case pic_opcode_addwf:
    case pic_opcode_andwf:
    case pic_opcode_xorwf:
    case pic_opcode_iorwf:
    case pic_opcode_subwf:
    case pic_opcode_comf:
    case pic_opcode_decf:
    case pic_opcode_decfsz:
    case pic_opcode_incf:
    case pic_opcode_incfsz:
    case pic_opcode_rlf:
    case pic_opcode_rlcf:
    case pic_opcode_rrf:
    case pic_opcode_movf:
    case pic_opcode_swapf:
    case pic_opcode_clrf:
    case pic_opcode_movwf:
    case pic_opcode_nop:
    case pic_opcode_clrw:
    case pic_opcode_retfie:
    case pic_opcode_return:
    case pic_opcode_sleep:
    case pic_opcode_clrwdt:
    case pic_opcode_option:
    case pic_opcode_bcf:
    case pic_opcode_bsf:
    case pic_opcode_btfsc:
    case pic_opcode_btfss:
    case pic_opcode_addlw:
    case pic_opcode_andlw:
    case pic_opcode_iorlw:
    case pic_opcode_movlw:
    case pic_opcode_sublw:
    case pic_opcode_xorlw:
    case pic_opcode_retlw:
    case pic_opcode_tris:
    case pic_opcode_datalo_set:
    case pic_opcode_datalo_clr:
    case pic_opcode_datahi_set:
    case pic_opcode_datahi_clr:
    case pic_opcode_irp_set:
    case pic_opcode_irp_clr:
    case pic_opcode_branchlo_set:
    case pic_opcode_branchlo_clr:
    case pic_opcode_branchhi_set:
    case pic_opcode_branchhi_clr:
    case pic_opcode_mullw:
    case pic_opcode_addwfc:
    case pic_opcode_dcfsnz:
    case pic_opcode_infsnz:
    case pic_opcode_rlncf:
    case pic_opcode_rrncf:
    case pic_opcode_subfwb:
    case pic_opcode_subwfb:
    case pic_opcode_cpfseq:
    case pic_opcode_cpfsgt:
    case pic_opcode_cpfslt:
    case pic_opcode_mulwf:
    case pic_opcode_negf:
    case pic_opcode_setf:
    case pic_opcode_tstfsz:
    case pic_opcode_btg:
    case pic_opcode_bc:
    case pic_opcode_bn:
    case pic_opcode_bnc:
    case pic_opcode_bnn:
    case pic_opcode_bnov:
    case pic_opcode_bnz:
    case pic_opcode_bov:
    case pic_opcode_bz:
    case pic_opcode_bra:
    case pic_opcode_rcall:
    case pic_opcode_daw:
    case pic_opcode_pop:
    case pic_opcode_push:
    case pic_opcode_reset:
    case pic_opcode_tblrd:
    case pic_opcode_tblwt:
    case pic_opcode_movlb:
      sz = 1;
      break;
    case pic_opcode_lfsr:
    case pic_opcode_movff:
      sz = 2;
      break;
    case pic_opcode_call:
    case pic_opcode_goto:
      sz = (PIC_TARGET_CPU_16BIT == target) ? 2 : 1;
      break;
    case pic_opcode_db:
      {
        cmd_t cmd;

        sz  = pic_code_data_sz_get(code);
        if (!sz) {
          cmd = pic_code_cmd_get(code);
          sz  = cmd_asm_data_sz_get(cmd);
        }
        sz = (sz + 1);
        sz /= 2;
      }
      break;
  }
  return sz;
}

size_t pic_code_data_sz_get(const pic_code_t code)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_false);
  return (ptr) ? ptr->data_sz : 0;
}

const void *pic_code_data_get(const pic_code_t code)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_false);
  return (ptr) ? ptr->data : 0;
}

/* data is assumed to have been malloc()'d and will be free()'d */
void pic_code_data_set(pic_code_t code, size_t sz, void *data)
{
  struct pic_code_ *ptr;

  ptr = pic_code_element_seek(code, boolean_true);
  if (ptr) {
    free(ptr->data);
    ptr->data_sz = sz;
    ptr->data    = data;
  }
}

