/************************************************************
 **
 ** pic_daop.c : PIC data optimization definitions
 **
 ** Copyright (c) 2004-2005, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/
#include "pic_msg.h"
#include "piccolst.h"
#include "pic_daop.h"

/*
 * NAME
 *   pic_code_databits_analyze
 *
 * DESCRIPTION
 *   analyze the databits in the pic_codes[] array
 *
 * PARAMETERS
 *   pf       : 
 *   pcl      : pcl on entry
 *   databits : current databits
 *   depth    : depth (for statistics only)
 *
 * RETURN
 *   none
 *
 * NOTES
 *   This follows all codepaths to determine when
 *   status<irp> and status<rp1:rp0> need to be changed, thus minimizing
 *   these changes.
 *   Note about calls: the statement after a call is assumed to
 *     have lost all information about the databits. This is unfortunate,
 *     but it is very difficult to accurately detemine without building
 *     a rather large and complex tree. This is something I'll work on
 *     when I've the chance. The main problem I've run into is how to
 *     detect loops.
 */
static void pic_code_databits_analyze(pfile_t *pf, pic_code_t code,
  pic_databits_t *pdatabits, unsigned depth)
{
  pic_databits_t       dbits;      /* dereferenced pdatabits */
  boolean_t            pv_code_is_cond;

  dbits = *pdatabits;
  /* now, traverse the code until either
     (1) all bits get known values
     (2) we hit an unconditional return or a leave */
  pv_code_is_cond = boolean_false;
  do {
    pic_databits_state_t cbits;
    boolean_t            code_is_cond;
    pic_code_t           next;

    code_is_cond = boolean_false;
    next = pic_code_next_get(code);

    pic_code_databits_get(code, &cbits);
    
    if (((cbits.before.rp0 == dbits.rp0)
        || (pic_bitstate_indeterminate == cbits.before.rp0))
      && ((cbits.before.rp1 == dbits.rp1)
        || (pic_bitstate_indeterminate == cbits.before.rp1))) {
      next = 0;
    } else if ((pic_bitstate_indeterminate == cbits.before.rp0)
        && (pic_bitstate_indeterminate == cbits.before.rp1)) {
      next = 0;
    } else {
      if (pic_bitstate_unknown == cbits.before.rp0) {
        cbits.before = dbits;
      } else {
        if (cbits.before.rp0 != dbits.rp0) {
          cbits.before.rp0 = pic_bitstate_indeterminate;
        }
        if (cbits.before.rp1 != dbits.rp1) {
          cbits.before.rp1 = pic_bitstate_indeterminate;
        }
      }
      pic_code_databits_set(code, &cbits);
      switch (pic_code_op_get(code)) {
        case pic_opcode_datalo_set:
          dbits.rp0 = pic_bitstate_set;
          break;
        case pic_opcode_datalo_clr:
          dbits.rp0 = pic_bitstate_clr;
          break;
        case pic_opcode_datahi_set:
          dbits.rp1 = pic_bitstate_set;
          break;
        case pic_opcode_datahi_clr:
          dbits.rp1 = pic_bitstate_clr;
          break;
        case pic_opcode_return:
        case pic_opcode_retfie:
        case pic_opcode_retlw:
          /* if this is unconditional we're done otherwise simply continue */
          if (!pv_code_is_cond) {
            next = 0;
          }
          break;
        case pic_opcode_goto:
        case pic_opcode_call:
          {
            boolean_t  is_suspend;
            pic_code_t brcode;

            brcode = pic_code_label_find(pf, pic_code_brdst_get(code));
            is_suspend = pic_code_is_suspend(pf, code);

            if ((pic_opcode_goto == pic_code_op_get(code))
                && !pv_code_is_cond
                && !is_suspend) {
              /* an absolute goto so next = brcode */
              next = brcode;
            } else {
              /* this is either a conditional goto or a call. either
                 way we first process then branch, then continue
                 processing from where we were. for the time being
                 a call is going to invalidate the known bits */
              pic_code_databits_analyze(pf, brcode, &dbits, depth + 1);
              if (is_suspend || (pic_opcode_call == pic_code_op_get(code))) {
                dbits.rp0 = pic_bitstate_indeterminate;
                dbits.rp1 = pic_bitstate_indeterminate;
              }
            }
          }
          break;
        case pic_opcode_incfsz:
        case pic_opcode_decfsz:
        case pic_opcode_btfsc:
        case pic_opcode_btfss:
          code_is_cond = boolean_true;
          break;
        default:
          break;
      }
      if (pic_code_modifies_pcl(pf, code)) {
        next = 0;
      }
      cbits.action = dbits;
      pic_code_databits_set(code, &cbits);
    }
    pv_code_is_cond = code_is_cond;
    code = next;
  } while (code);
}

/* run through the pic_codes and remove any redundant dataset commands */
void pic_code_databits_post_analyze(pfile_t *pf)
{
  pic_code_t pv;
  pic_code_t code;
  unsigned   ins_ct; /* instruction ct */
  unsigned   rem_ct;

  for (ins_ct = 0, rem_ct = 0, pv = 0, code = pic_code_list_head_get(pf);
       code;
       ins_ct++) {
    pic_code_t next;

    next = pic_code_next_get(code);
    if (!pic_code_flag_test(code, PIC_CODE_FLAG_NO_OPTIMIZE)) {
      boolean_t            cremove;
      pic_databits_state_t cbitstate;

      pic_code_databits_get(code, &cbitstate);

      switch (pic_code_op_get(code)) {
        case pic_opcode_datalo_set:
        case pic_opcode_datalo_clr:
          cremove = cbitstate.before.rp0 == cbitstate.action.rp0;
          break;
        case pic_opcode_datahi_set:
        case pic_opcode_datahi_clr:
          cremove = cbitstate.before.rp1 == cbitstate.action.rp1;
          break;
        default:
          cremove = boolean_false;
          break;
      }
      if (cremove) {
        pic_code_list_remove(pf, code);
        pic_code_free(code);
        rem_ct++;
      } else {
        pv = code;
      }
    }
    code = next;
  }
  pfile_log(pf, pfile_log_debug, "...removed %u of %u instructions", 
      rem_ct, ins_ct);
}

void pic_code_databits_optimize(pfile_t *pf)
{
  label_t        lbl;
  pic_databits_t databits;

  pfile_log(pf, pfile_log_debug, PIC_MSG_FIXING_DATA_BITS);
  /* first -- analyze the ISR bits */
  lbl = pfile_label_find(pf, pfile_log_none, PIC_LABEL_ISR);
  if (lbl) {
    pic_code_t code;

    pfile_log(pf, pfile_log_debug, "...analyzing ISR");
    code = pic_code_label_find(pf, lbl);
    if (code) {
      databits.rp1 = pic_bitstate_clr;
      databits.rp0 = pic_bitstate_clr;

      pic_code_databits_analyze(pf, code, &databits, 1);
    }
    label_release(lbl);
  }

  /* second -- analyze the USER bits */
  lbl = pfile_label_find(pf, pfile_log_err, PIC_LABEL_RESET);
  if (lbl) {
    pic_code_t code;

    pfile_log(pf, pfile_log_debug, "...analyzing USER");
    code = pic_code_label_find(pf, lbl);
    if (code) {
      databits.rp1 = pic_bitstate_clr;
      databits.rp0 = pic_bitstate_clr;

      pic_code_databits_analyze(pf, code, &databits, 1);
    }
    label_release(lbl);
  }
  /* once the analysis is done, we need to run through & make
     sure no further label fixups are needed, or fix them up
     as necessary */
  pfile_log(pf, pfile_log_debug, "...post analyze");
  pic_code_databits_post_analyze(pf);
}

/* look to see if datalo_set or datahi_set are used.
 * if not, datalo_clr and datahi_clr respectively can
 * be removed
 */ 
void pic_code_databits_remove(pfile_t *pf)
{
  pic_code_t code;
  boolean_t  datalo_set;
  boolean_t  datahi_set;

  for (datalo_set = boolean_false,
         datahi_set = boolean_false,
         code = pic_code_list_head_get(pf);
       code && !(datalo_set && datahi_set);
       code = pic_code_next_get(code)) {
    if (pic_opcode_datalo_set== pic_code_op_get(code)) {
      datalo_set = boolean_true;
    } else if (pic_opcode_datahi_set== pic_code_op_get(code)) {
      datahi_set = boolean_true;
    }
  }
  if (!(datalo_set && datahi_set)) {
    unsigned   rem_ct;
    pic_code_t next;

    for (rem_ct = 0,
           code = pic_code_list_head_get(pf);
         code;
         code = next) {
      next = pic_code_next_get(code);
      if (!pic_code_flag_test(code, PIC_CODE_FLAG_NO_OPTIMIZE)
        && ((!datalo_set 
          && (pic_opcode_datalo_clr == pic_code_op_get(code)))
          || (!datahi_set
            && (pic_opcode_datahi_clr == pic_code_op_get(code))))) {
        pic_code_list_remove(pf, code);
        pic_code_free(code);
        rem_ct++;
      }
    }
    pfile_log(pf, pfile_log_debug, "...removed %u data bit instructions",
        rem_ct);
  }
}

