/************************************************************
 **
 ** pic_op.c : PIC operator generation definitions
 **
 ** Copyright (c) 2004-2005, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/
#include <assert.h>
#include <errno.h>
#include <string.h>

#include "../libcore/pf_proc.h"
#include "../libcore/pf_expr.h"
#include "../libcore/pf_op.h"
#include "pic_opfn.h"
#include "pic_stvar.h"
#include "pic_var.h"
#include "piccolst.h"
#include "pic_inst.h"
#include "pic_op.h"

/*
 * general note:
 *   all operations must take into account:
 *     constants
 *     single bit
 *     multi-bit
 *     simple
 *     indirect
 *     in W
 *     different banks
 * nb: operations on bitfields will occur *as if*
 *     the bitfield was promoted to the smallest
 *     integer type that can contain it
 *
 * naming convention
 *   pic_op_* -- only called via pic_op(...)
 */
#define SWAP(type, a, b) \
   { type tmp__; tmp__ = a; a = b; b = tmp__; }

/*
 * these are used to prevent duplicate calls to _multiply or _divide
 * for example:
 *    ch = n % 10
 *    n  = n / 10
 * they are reset when a label or branch is encountered
 * also if any of the values are used as a destination
 */
typedef enum pic_last_value_ {
  PIC_LAST_VALUE_FIRST,
  PIC_LAST_VALUE_DIVISOR = PIC_LAST_VALUE_FIRST,
  PIC_LAST_VALUE_DIVIDEND,
  PIC_LAST_VALUE_MULTIPLIER,
  PIC_LAST_VALUE_MULTIPLICAND,
  PIC_LAST_VALUE_CT
} pic_last_value_t;

static value_t pic_last_values[PIC_LAST_VALUE_CT];

void pic_last_value_set(pic_last_value_t which, value_t n)
{
  assert(which < PIC_LAST_VALUE_CT);

  value_lock(n);
  value_release(pic_last_values[which]);
  pic_last_values[which] = n;
}

void pic_last_values_reset(void)
{
  pic_last_value_t which;
  
  for (which = PIC_LAST_VALUE_FIRST; 
       which < PIC_LAST_VALUE_CT;
       which++) {
    pic_last_value_set(which, VALUE_NONE);
  }
}

static boolean_t pic_value_is_w(value_t val)
{
  return (VALUE_NONE == val)
    || (VARIABLE_NONE == value_variable_get(val));
}

/* 
 * for a binary op, the result size is the greater of the two
 * operand sizes
 * If the result size is greater than the destination size, the
 * result size is lowered to the destination size
 */
static variable_sz_t pic_result_sz_get(value_t val1, value_t val2, 
    value_t dst)
{
  variable_sz_t sz;

  /*assert(val1);*/

  if (value_is_universal(val1)) {
    sz = value_byte_sz_get(val2);
  } else if (value_is_universal(val2)) {
    sz = value_byte_sz_get(val1);
  } else {
    sz = pic_value_is_w(val1) ? 1 : value_byte_sz_get(val1);

    if (val2 && !pic_value_is_w(val2)) {
      variable_sz_t val2_sz;

      val2_sz = value_byte_sz_get(val2);
      if (val2_sz > sz) {
        sz = val2_sz;
      }
    }
  }
  /* if dst is boolean, the result must be done in the full
   * size of val1 & val2 because dst will be 1 if the result
   * is non-0, and 0 if the result is 0
   */
  if (dst && !value_is_boolean(dst)) {
    variable_sz_t dst_sz;

    dst_sz = (pic_value_is_w(dst)) ? 1 : value_byte_sz_get(dst);
    if (dst_sz < sz) {
      sz = dst_sz;
    }
  }
  return sz;
}

/*
 * determine the result sign flag based on the promotion rules
 */
static flag_t pic_result_flag_get(pfile_t *pf, value_t val1, value_t val2)
{
  variable_def_t rdef;

  rdef = pfile_variable_def_promotion_get(pf, operator_null, val1, val2);
  return variable_def_flag_test(rdef, VARIABLE_DEF_FLAG_SIGNED)
    ? VARIABLE_DEF_FLAG_SIGNED
    : VARIABLE_DEF_FLAG_NONE;
}

static boolean_t pic_result_is_signed(pfile_t *pf, value_t val1, 
    value_t val2)
{
  return VARIABLE_DEF_FLAG_SIGNED == pic_result_flag_get(pf, val1, val2);
}

/* 
 * allocate a pic temp variable & assign val to it. The size of the result
 * is less{dst, greater{val1, val2}}
 */
static value_t pic_var_temp_get_assign(pfile_t *pf, value_t val, value_t val2,
    value_t dst)
{
  value_t        tmp;
  size_t         sz;

  sz = pic_result_sz_get(val, val2, VALUE_NONE/*dst*/);
#if 0
  tmp = pic_var_temp_get(pf, value_is_signed(val) 
      ? VARIABLE_DEF_FLAG_SIGNED : VARIABLE_DEF_FLAG_NONE, sz);
#else
  tmp = pic_var_temp_get(pf, pic_result_flag_get(pf, val, val2), sz);
#endif
  pic_op(pf, operator_assign, tmp, val, VALUE_NONE);
  return tmp;
}

/* get the constant byte at offset n. This differs from simply shifting
 * because it preserves the sign bit at higher offsets */
static uchar pic_value_const_byte_get(const value_t val, unsigned ofs)
{
  variable_const_t cn;

  cn = value_const_get(val);
  if (ofs >= value_sz_get(val)) {
    cn = (value_is_signed(val) && (cn >= 0x8000000))
      ? 0xff
      : 0x00;
  } else {
    cn >>= (8 * ofs);
  }
  return (uchar) cn;
}

value_t pic_fsr_get(pfile_t *pf)
{
  return pfile_value_find(pf, pfile_log_err, 
    (PIC_TARGET_CPU_16BIT == pic_target_cpu_get(pf))
    ? "_fsr0l"
    : "_fsr");
}

void pic_fsr_setup(pfile_t *pf, value_t val)
{
  pic_instr_append_f_d(pf, pic_opcode_movf, val, 0, pic_opdst_w);
  if (PIC_TARGET_CPU_16BIT == pic_target_cpu_get(pf)) {
    pic_instr_append_reg(pf, pic_opcode_movwf, "_fsr0l");
    pic_instr_append_f_d(pf, pic_opcode_movf, val, 1, pic_opdst_w);
    pic_instr_append_reg(pf, pic_opcode_movwf, "_fsr0h");
  } else {
    pic_instr_append_reg(pf, pic_opcode_movwf, "_fsr");
    pic_instr_append_reg_flag(pf, pic_opcode_bcf, "_status", "_irp");
    pic_instr_append_f_d(pf, pic_opcode_movf, val, 1, pic_opdst_f);
    pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_z");
    pic_instr_append_reg_flag(pf, pic_opcode_bsf, "_status", "_irp");
  }
}

/*
 * NAME
 *   pic_indirect_setup
 *
 * DESCRIPTION
 *   setup irp:fsr
 *
 * PARAMETERS
 *   pf    : file
 *   val   : value
 *   offset: offset from start of value
 *
 * RETURN
 *
 * NOTES
 *   1. A value is assumed to not span banks
 *   2. Normally this is called if val.baseofs is non-const
 */
void pic_indirect_setup(pfile_t *pf, value_t val, size_t offset)
{
  value_t baseofs;

  baseofs = value_baseofs_get(val);
  if (value_is_bit(val)) {
    offset += (value_bit_offset_get(val) + 7) / 8;
  }

  if (value_is_indirect(val)) {
    /* STATUS<IRP>:FSR = *value */
    value_t tmp;

    tmp = value_clone(val);
    value_baseofs_set(tmp, VALUE_NONE);
    if (value_flag_test(val, VALUE_FLAG_ARRAY)) {
      pic_opcode_t op;

      offset += value_const_get(baseofs);
      value_baseofs_set(tmp, VALUE_NONE);
      if (PIC_TARGET_CPU_16BIT == pic_target_cpu_get(pf)) {
        pic_instr_append_lfsr(pf, 0, value_base_get(tmp));
        if (!value_is_const(baseofs) || offset) {
          if (value_is_const(baseofs)) {
            pic_instr_append_w_kn(pf, pic_opcode_movlw, offset);
          } else {
            pic_instr_append_f_d(pf, pic_opcode_movf, baseofs, 0, pic_opdst_w);
            if (offset) {
              pic_instr_append_w_kn(pf, pic_opcode_addlw, offset);
            }
          }
          pic_instr_append_reg_d(pf, pic_opcode_addwf, "_fsr0l", pic_opdst_f);
        }
      } else {
        pic_instr_append_f(pf, pic_opcode_movlw, tmp, offset);
        pic_instr_append_reg(pf, pic_opcode_movwf, "_fsr");
        op = (value_base_get(tmp) >= 256)
          ? pic_opcode_irp_set
          : pic_opcode_irp_clr;
        pic_instr_append(pf, op);
        /* finally, add any additional offset to _fsr */
        if (!value_is_const(baseofs)) {
          pic_instr_append_f_d(pf, pic_opcode_movf, baseofs, 0, pic_opdst_w);
          pic_instr_append_reg_d(pf, pic_opcode_addwf, "_fsr", pic_opdst_f);
        }
      }
    } else {
      /* assert(value_is_pointer(val)); */
      value_flag_clr(tmp, VALUE_FLAG_INDIRECT);

      pic_instr_append_f_d(pf, pic_opcode_movf, tmp, 0, pic_opdst_w);
      if (baseofs) {
        if (value_is_const(baseofs)) {
          if (value_const_get(baseofs)) {
            pic_instr_append_w_kn(pf, pic_opcode_addlw, 
                value_const_get(baseofs));
          }
        } else {
          pic_instr_append_f_d(pf, pic_opcode_addwf, baseofs, 0, pic_opdst_w);
        }
      }
      if (PIC_TARGET_CPU_16BIT == pic_target_cpu_get(pf)) {
        pic_instr_append_reg(pf, pic_opcode_movwf, "_fsr0l");
        pic_instr_append_f_d(pf, pic_opcode_movf, tmp, 1, pic_opdst_w);
        pic_instr_append_reg(pf, pic_opcode_movwf, "_fsr0h");
      } else {
        pic_instr_append_reg(pf, pic_opcode_movwf, "_fsr");
        pic_instr_append_reg_flag(pf, pic_opcode_bcf, "_status", "_irp");
        pic_instr_append_f_bn(pf, pic_opcode_btfsc, tmp, 1, 0);
        pic_instr_append_reg_flag(pf, pic_opcode_bsf, "_status", "_irp");
      }
    }
    value_release(tmp);
  } else {
    variable_base_t base;

    if (PIC_TARGET_CPU_16BIT != pic_target_cpu_get(pf)) {
      pic_instr_append(pf, 
         (variable_base_get(value_variable_get(val), 0) >= 256) 
         ? pic_opcode_irp_set : pic_opcode_irp_clr);
    }
    base = value_base_get(val) + offset;
    if (value_is_const(baseofs)) {
      /* fsr = value_base_get(value) + value_const_get(baseofs) + sz */
      base += (variable_base_t) value_const_get(baseofs);
      if (PIC_TARGET_CPU_16BIT == pic_target_cpu_get(pf)) {
        pic_instr_append_lfsr(pf, 0, base);
      } else {
        pic_instr_append_w_kn(pf, pic_opcode_movlw, base);
        pic_instr_append_reg(pf, pic_opcode_movwf, "_fsr");
      }
    } else {
      /* fsr = base (+ sz - 1) */
      if (PIC_TARGET_CPU_16BIT == pic_target_cpu_get(pf)) {
        pic_instr_append_lfsr(pf, 0, base);
      } else {
        pic_instr_append_w_kn(pf, pic_opcode_movlw, base);
        pic_instr_append_reg(pf, pic_opcode_movwf, "_fsr");
      }
      if (VALUE_NONE != baseofs) {
        value_t fsr;

        fsr = pic_fsr_get(pf);
        /* w = baseofs */
        pic_instr_append_f_d(pf, pic_opcode_movf, baseofs, 0, pic_opdst_w);
        /* _fsr += w */
        pic_instr_append_f_d(pf, pic_opcode_addwf, fsr, 0, pic_opdst_f);
        value_release(fsr);
      }
    }
  }
}

/*
 * given three values, determine which one is indirect. This assumes
 * (obviously) that only one is indirect!
 */
static value_t pic_indirect_value_get(value_t val1, value_t val2, value_t val3)
{
  value_t val;

  if (value_is_indirect(val1)) {
    assert((value_is_same(val1, val2) || !value_is_indirect(val2))
           && (value_is_same(val1, val3) || !value_is_indirect(val3)));
    val = val1;
  } else if (value_is_indirect(val2)) {
    assert(value_is_same(val2, val3) || !value_is_indirect(val3));
    val = val2;
  } else if (value_is_indirect(val3)) {
    val = val3;
  } else {
    val = VALUE_NONE;
  }
  return val;
}

/*
 * given three values (only one of which is indirect) and an offset,
 * setup the irp:ind to the requested offset. This is a no-op
 * if none of the values is indirect
 *   ipos is used to track the relative indirect offset & should
 *        *never* be modified except by pic_indirect_bump3
 */
static void pic_indirect_setup3(pfile_t *pf, value_t val1, value_t val2,
  value_t val3, variable_sz_t ofs, variable_sz_t *ipos)
{
  value_t val;

  val = pic_indirect_value_get(val1, val2, val3);
  if (VALUE_NONE != val) {
    variable_sz_t n;

    n = (ofs >= value_sz_get(val))
      ? value_sz_get(val) - 1
      : ofs;
    pic_indirect_setup(pf, val, n);
  }
  *ipos = ofs;
}

/*
 * if any of the three values is indirect, and the position is in range
 * bump the position in the desired direction
 */
static void pic_indirect_bump3(pfile_t *pf, value_t val1, value_t val2,
  value_t val3, variable_sz_t ix, variable_sz_t *ipos)
{
  value_t val;

  val = pic_indirect_value_get(val1, val2, val3);
  if (VALUE_NONE != val) {
    /* we can only move one in either direction */
    value_t fsr;
    
    assert((ix == *ipos)
        || (ix + 1 == *ipos) 
        || (ix - 1 == *ipos));

    fsr = pic_fsr_get(pf);
    if ((ix > *ipos) && (ix < value_sz_get(val))) {
      pic_instr_append_f_d(pf, pic_opcode_incf, fsr, 0, pic_opdst_f);
    } else if ((ix < *ipos) && (*ipos < value_sz_get(val))) {
      pic_instr_append_f_d(pf, pic_opcode_decf, fsr, 0, pic_opdst_f);
    }
    value_release(fsr);
    *ipos = ix;
  }
}

/*
 * return the pic sign value appropriately set if val is signed
 * indirect values are expected to be at the MSB entry.
 * If accum is VALUE_NONE, the returned value will be 
 *   pic_var_accum so pic_var_accum_release() must be used!
 */
static void pic_value_sign_get_in_w(pfile_t *pf, value_t val)
{
  pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
  if (value_is_signed(val)) {
    pic_instr_append_f_bn(pf, pic_opcode_btfsc, val, value_sz_get(val) - 1, 7);
    pic_instr_append_w_kn(pf, pic_opcode_movlw, 255);
  }
}

static value_t pic_value_sign_get(pfile_t *pf, value_t val, value_t accum)
{
  if (value_is_signed(val)) {
    if (VALUE_NONE == accum) {
      accum = pic_var_accum_get(pf);
    }
    pic_value_sign_get_in_w(pf, val);
    pic_instr_append_f(pf, pic_opcode_movwf, accum, 0);
  } else {
    accum = VALUE_NONE;
  }
  return accum;
}

/*
 * determine if two values reside in the same bank
 */
static boolean_t pic_val_bank_same(pfile_t *pf, value_t val1, value_t val2)
{
  boolean_t  rc;

  if (value_is_indirect(val1) 
      || value_is_indirect(val2)
      || pic_value_is_w(val1)
      || pic_value_is_w(val2)
      || value_is_shared(val1)
      || value_is_shared(val2)
      || value_is_const(val1)
      || value_is_const(val2)) {
    rc = boolean_true;
  } else {
    variable_t var1;
    variable_t var2;

    rc   = boolean_false;
    var1 = value_variable_get(val1);
    var2 = value_variable_get(val2);

    if (PIC_TARGET_CPU_16BIT == pic_target_cpu_get(pf)) {
      rc = (value_base_get(val1) / 256) == (value_base_get(val2) / 256);
    } else {
      size_t ii;

      for (ii = 0, rc = boolean_false; 
           (boolean_false == rc) && (ii < VARIABLE_MIRROR_CT);
           ii++) {
        variable_base_t base1;
        size_t          jj;

        base1 = variable_base_get(var1, ii);
        if (VARIABLE_BASE_UNKNOWN == base1) {
          /* nothing more to do! */
          break; /* <--- */
        }
        for (jj = 0;
             (boolean_false == rc) && (jj < VARIABLE_MIRROR_CT);
             jj++) {
          variable_base_t base2;
          unsigned        bank_sz;

          base2 = variable_base_get(var2, jj);
          if (VARIABLE_BASE_UNKNOWN == base2) {
            /* nothing more to do! */
            break; /* <--- */
          }
          bank_sz = pic_target_bank_size_get(pf);
          rc = (((base1 ^ base2) & (bank_sz * 2 | bank_sz)) == 0);
        }
      }
    }
  }
  return rc;
}

/*
 * sign extend a value. If the value is indirect, irp:ind are expected
 * to point to the highest byte in the value.
 */
static void pic_value_sign_extend(pfile_t *pf, value_t val, 
    variable_sz_t pos, boolean_t is_signed)
{
  if (pos + 1 < value_sz_get(val)) {
    if (is_signed) {
      pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
      pic_instr_append_f_bn(pf, pic_opcode_btfsc, val, pos, 7);
      pic_instr_append_w_kn(pf, pic_opcode_movlw, 255);
    }
    while (pos + 1 < value_sz_get(val)) {
      pos++;
      if (value_is_indirect(val)) {
        value_t fsr;

        fsr = pic_fsr_get(pf);
        pic_instr_append_f_d(pf, pic_opcode_incf, fsr, 0, pic_opdst_f);
        value_release(fsr);
      }
      pic_instr_append_f(pf, is_signed ? pic_opcode_movwf : pic_opcode_clrf, 
          val, pos);
    }
  }
}

static void pic_assign_const(pfile_t *pf, value_t dst,
  value_t val1, value_t val2, variable_const_t n)
{
  value_t        tmp;
  variable_def_t def;

  def = variable_def_alloc(0, variable_def_type_integer,
      pic_result_flag_get(pf, val1, val2),
      pic_result_sz_get(val1, val2, 
        (value_is_boolean(dst) ? VALUE_NONE : dst)));
  tmp = pfile_constant_get(pf, n, def);
  pic_op(pf, operator_assign, dst, tmp, VALUE_NONE);
  value_release(tmp);
}

#define PIC_WORK_IN_TEMP_FLAG_NONE          0x0000
#define PIC_WORK_IN_TEMP_FLAG_VAL1          0x0001 /* check val1 */
#define PIC_WORK_IN_TEMP_FLAG_VAL2          0x0002 /* check val2 */
#define PIC_WORK_IN_TEMP_FLAG_DST           0x0004 /* check dst  */
/* if DST is volatile assume we cannot work in it */
#define PIC_WORK_IN_TEMP_FLAG_DST_VOLATILE  0x0008
/* if {DST, VALx} are indirect *and* equal, move VALx to a temp */
#define PIC_WORK_IN_TEMP_FLAG_DST_VAL_EQUAL 0x0010
/* check both values (most commmon) */
#define PIC_WORK_IN_TEMP_FLAG_VALS \
  PIC_WORK_IN_TEMP_FLAG_VAL1       \
  | PIC_WORK_IN_TEMP_FLAG_VAL2
/* check all values */
#define PIC_WORK_IN_TEMP_FLAG_VALS_DST \
  PIC_WORK_IN_TEMP_FLAG_VALS      \
  | PIC_WORK_IN_TEMP_FLAG_DST

/* determine if the work must be done in a temporary
 * returns TRUE if work was done in a temporary, FALSE if no
 * nb: in the 16 bit cores, increment and decrement muck with
 *     STATUS<C>. To simplify things, I'll simply disallow
 *     indirect operations for the 16 bit cores
 */
static boolean_t pic_work_in_temp(pfile_t *pf, operator_t op,
  value_t dst, value_t val1, value_t val2, unsigned flags)
{
  boolean_t rc;
#define PWIT_FLAG_IND \
  (VARIABLE_FLAG_PTR_LOOKUP \
   | VARIABLE_FLAG_PTR_EEPROM \
   | VARIABLE_FLAG_PTR_FLASH)

  rc = boolean_true;
  if (PIC_TARGET_CPU_16BIT == pic_target_cpu_get(pf)) {
    flags |= PIC_WORK_IN_TEMP_FLAG_VALS_DST;
  }
  if ((flags & PIC_WORK_IN_TEMP_FLAG_VAL1)
      && !value_is_const(val1)
      && ((value_is_indirect(val1) 
          && value_is_indirect(dst)
          && ((flags & PIC_WORK_IN_TEMP_FLAG_DST_VAL_EQUAL)
            || !value_is_same(val1, dst)))
        || (value_is_indirect(val1)
          && value_is_indirect(val2)
          && !value_is_same(val1, val2))
        || value_is_bit(val1)
        || value_is_lookup(val1)
        || (value_is_indirect(val1)
          && (variable_flags_get_all(value_variable_get(val1)) 
            & PWIT_FLAG_IND))
        || (value_is_indirect(val1)
          && (PIC_TARGET_CPU_16BIT == pic_target_cpu_get(pf))))) {
    /* val1 cannot be used as is, get a temp for it instead */
    value_t tmp;

#if 1
    tmp = pic_var_temp_get_assign(pf, val1, val2, dst);
#else
    tmp = pic_var_temp_get_assign(pf, val1, VALUE_NONE, dst);
#endif
    pic_op(pf, op, dst, tmp, val2);
    pic_var_temp_release(pf, tmp);
  } else if ((flags & PIC_WORK_IN_TEMP_FLAG_VAL2)
      && !value_is_const(val2)
      && ((value_is_indirect(val2)
          && value_is_indirect(dst)
          && ((flags & PIC_WORK_IN_TEMP_FLAG_DST_VAL_EQUAL)
           || !value_is_same(dst, val2)))
        || value_is_bit(val2)
        || value_is_lookup(val2)
        || (value_is_indirect(val2)
          && (variable_flags_get_all(value_variable_get(val2))
            & PWIT_FLAG_IND))
        || (value_is_indirect(val2)
          && (PIC_TARGET_CPU_16BIT == pic_target_cpu_get(pf))))) {
    /* val2 cannot be used as it, get a temp for it instead */
    value_t tmp;

#if 1
    tmp = pic_var_temp_get_assign(pf, val2, val1, dst);
#else
    tmp = pic_var_temp_get_assign(pf, val2, VALUE_NONE, dst);
#endif
    pic_op(pf, op, dst, val1, tmp);
    pic_var_temp_release(pf, tmp);
  } else if ((flags & PIC_WORK_IN_TEMP_FLAG_DST)
      && ((value_is_indirect(dst)
          && (variable_flags_get_all(value_variable_get(dst)) & PWIT_FLAG_IND))
        || (value_is_bit(dst)
          || ((flags & PIC_WORK_IN_TEMP_FLAG_DST_VOLATILE)
            && value_is_volatile(dst)))
        || (value_is_indirect(dst)
          && (PIC_TARGET_CPU_16BIT == pic_target_cpu_get(pf))))) {
    value_t tmp;

    tmp = pic_var_temp_get(pf, pic_result_flag_get(pf, val1, val2),
        pic_result_sz_get(val1, val2, dst));
    pic_op(pf, op, tmp, val1, val2);
    pic_op(pf, operator_assign, dst, tmp, VALUE_NONE);
    pic_var_temp_release(pf, tmp);
  } else {
    rc = boolean_false;
  }
  return rc;
#undef PWIT_FLAG_IND
}

/* on exit, Z is set if src is 0, clear otherwise */
static void pic_value_is_zero(pfile_t *pf, value_t src)
{
  assert(!value_is_const(src));
  assert(!value_is_bit(src));
  if (value_is_const(src)) {
    pic_instr_append_w_kn(pf, pic_opcode_movlw, value_const_get(src) ? 1 : 0);
  } else if (pic_value_is_w(src)) {
    pic_instr_append_w_kn(pf, pic_opcode_andlw, 0xff);
  } else if (value_is_single_bit(src)) {
    pic_instr_append_reg_flag(pf, pic_opcode_bcf, "_status", "_z");
    pic_instr_append_f(pf, pic_opcode_btfss, src, 0);
    pic_instr_append_reg_flag(pf, pic_opcode_bsf, "_status", "_z");
  } else {
    size_t        ii;
    variable_sz_t ipos;

    pic_indirect_setup3(pf, src, VALUE_NONE, VALUE_NONE, 0, &ipos);
    for (ii = 0; ii < value_sz_get(src); ii++) {
      pic_indirect_bump3(pf, src, VALUE_NONE, VALUE_NONE, 
          ii, &ipos);
      pic_instr_append_f_d(pf, 
          (ii) ? pic_opcode_iorwf : pic_opcode_movf, src, ii, pic_opdst_w);
    }
  }
}

/* create an integer variable that overlays a bit variable */
static value_t pic_bit_overlay_create(value_t val)
{
  variable_t     var;
  variable_def_t def;
  unsigned       bit_ofs;
  unsigned       sz;
  value_t        dst;

  bit_ofs = value_bit_offset_get(val);
  sz      = value_sz_get(val);

  /* create a byte array covering dst */
  def   = variable_def_alloc(0, variable_def_type_integer,
      VARIABLE_DEF_FLAG_NONE, (bit_ofs + sz + 7) / 8);
  var   = variable_alloc(TAG_NONE, def);
  variable_master_set(var, variable_master_get(value_variable_get(val)));

  dst   = value_alloc(var);
  variable_release(var);
  return dst;
}

/* dst is a bit, src is not lookup */
static void pic_assign_to_bit(pfile_t *pf, value_t dst, value_t src)
{
  unsigned dst_bit_ofs;

  dst_bit_ofs  = value_bit_offset_get(dst);
  if (value_sz_get(dst) == 1) {
    if (value_is_const(src)) {
      variable_const_t cn;

      cn = value_const_get(src);
      pic_instr_append_f_bn(pf,
          ((value_is_boolean(dst) && cn)
          || (cn & 1)) ? pic_opcode_bsf : pic_opcode_bcf,
          dst, 0, 0);
    } else {
      if (value_is_bit(src) && (value_sz_get(src) == 1)) {
        /* single bit --> single bit */
        if (value_is_volatile(dst) || !pic_val_bank_same(pf, dst, src)) {
          label_t lbl_set;
          label_t lbl_done;

          lbl_set  = pfile_label_alloc(pf, 0);
          lbl_done = pfile_label_alloc(pf, 0);
          pic_instr_append_f(pf, pic_opcode_btfsc, src, 0);
          pic_instr_append_n(pf, pic_opcode_goto, lbl_set);
          pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
          pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
          pic_instr_append_label(pf, lbl_set);
          pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
          pic_instr_append_label(pf, lbl_done);
          label_release(lbl_done);
          label_release(lbl_set);
        } else {
          pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
          pic_instr_append_f(pf, pic_opcode_btfsc, src, 0);
          pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
        }
      } else {
        /* determine if src is 0, set src to _status:_z, and invert test */
        if (pic_value_is_w(src)) {
          pic_instr_append_w_kn(pf, pic_opcode_andlw, 
            (value_is_boolean(dst) ? 0xff : 0x01));
        } else {
          if (value_is_boolean(dst)) {
            pic_value_is_zero(pf, src);
          } else {
            if (value_is_indirect(src)) {
              variable_sz_t ipos;

              pic_indirect_setup3(pf, src, VALUE_NONE, VALUE_NONE, 0, &ipos);
            }
            pic_instr_append_f_d(pf, pic_opcode_movf, src, 0, pic_opdst_w);
            pic_instr_append_w_kn(pf, pic_opcode_andlw, 1);
          }
        }
        /* set or clear dst as appropriate */
        if (value_is_volatile(dst) || !pic_val_bank_same(pf, dst, src)) {
          /* cannot change dst twice or cannot skip a single inst. :( */
          label_t done;
          label_t skip;

          skip = pfile_label_alloc(pf, 0);
          done = pfile_label_alloc(pf, 0);

          pic_instr_append_reg_flag(pf, pic_opcode_btfsc, "_status", "_z");
          pic_instr_append_n(pf, pic_opcode_goto, skip);
          pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
          pic_instr_append_n(pf, pic_opcode_goto, done);
          pic_instr_append_label(pf, skip);
          pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
          pic_instr_append_label(pf, done);
          label_release(done);
          label_release(skip);
        } else {
          /* simple single-bit transform */
          pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
          pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_z");
          pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
        }
      }
    }
  } else if (!value_is_const(src) 
      && !value_is_bit(src)
      && (value_sz_get(dst) <= 4)) {
    variable_sz_t dst_sz;
    variable_sz_t dst_ofs;
    variable_sz_t ii;
    value_t       tmp;

    tmp = src;
    if (pic_value_is_w(tmp) || !pic_val_bank_same(pf, src, dst)) {
      tmp = pic_var_accum_get(pf);
      pic_instr_append_f(pf, pic_opcode_movwf, tmp, 0);
    }

    dst_sz  = value_sz_get(dst);
    dst_ofs = value_bit_offset_get(dst);
    dst = pic_bit_overlay_create(dst);
    for (ii = 0; ii < dst_sz; ii++) {
      pic_instr_append_f_bn(pf, pic_opcode_bcf, dst,
          (dst_ofs + ii) / 8,
          (dst_ofs + ii) & 7);
      pic_instr_append_f_bn(pf, pic_opcode_btfsc, tmp, ii / 8, ii & 7);
      pic_instr_append_f_bn(pf, pic_opcode_bsf, dst,
          (dst_ofs + ii) / 8,
          (dst_ofs + ii) & 7);
    }
    if (tmp != src) {
      pic_var_accum_release(pf, tmp);
    }
    value_release(dst);
  } else {
    /* mask = (1 << dst_sz) - 1)  */
    /* tmp  = src & mask          */
    /* dst  = (dst & ~mask) | src */
    value_t        tdst;   /* needed if dst is volatile          */
    variable_sz_t  tmp_sz;
    variable_sz_t  dst_sz;
    boolean_t      dst_is_volatile;

    dst_sz          = value_sz_get(dst);
    dst_is_volatile = value_is_volatile(dst);
    dst             = pic_bit_overlay_create(dst);

    tdst = (dst_is_volatile)
      ? pic_var_temp_get(pf, VARIABLE_DEF_FLAG_NONE, value_sz_get(dst))
      : dst;
    tmp_sz = (dst_bit_ofs + dst_sz + 7) / 8;
    /* tmp = src with bits masked off */
    if (value_is_const(src)) {
      value_t mask;

      src = pfile_constant_get(pf, value_const_get(src) & ((1 << dst_sz) - 1),
          VARIABLE_DEF_NONE);
      mask   = pfile_constant_get(pf, ~(((1 << dst_sz) - 1) << dst_bit_ofs),
          VARIABLE_DEF_NONE);
      /* dst &= mask */
      pic_op(pf, operator_andb, tdst, dst, mask);
      /* dst |= src */
      pic_op(pf, operator_orb, dst, tdst, src);
      value_release(src);
    } else {
      value_t mask;
      value_t tmp;

      tmp    = pic_var_temp_get(pf, VARIABLE_DEF_FLAG_NONE, tmp_sz);
      mask   = pfile_constant_get(pf, (1 << dst_sz) - 1, VARIABLE_DEF_NONE);
      pic_op(pf, operator_andb, tmp, src, mask);
      value_release(mask);
      /* move tmp into position for binary OR with dst */
      mask   = pfile_constant_get(pf, dst_bit_ofs, VARIABLE_DEF_NONE);
      pic_op(pf, operator_shift_left, tmp, tmp, mask);
      value_release(mask);
      /* mask off bits in dst */
      mask   = pfile_constant_get(pf, ~(((1 << dst_sz) - 1) << dst_bit_ofs),
          VARIABLE_DEF_NONE);
      pic_op(pf, operator_andb, tdst, dst, mask);
      /* OR in src */
      pic_op(pf, operator_orb, dst, tdst, tmp);
      pic_var_temp_release(pf, tmp);
    }
    if (tdst != dst) {
      pic_var_temp_release(pf, tdst);
    }
    value_release(dst);
  }
}

/* dst is not bit, src is non-const bit */
static void pic_assign_from_bit(pfile_t *pf, value_t dst, value_t src)
{
  if (value_sz_get(src) == 1) {
    variable_sz_t ipos;

    /* the result is either all 0 or all 0xff */
    pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE, 0, &ipos);
    pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
    pic_instr_append_f(pf, pic_opcode_btfsc, src, 0);
    pic_instr_append_w_kn(pf, pic_opcode_movlw, 
        value_is_signed(src) ? 0xff : 1);
    if (!pic_value_is_w(dst)) {
      pic_instr_append_f(pf, pic_opcode_movwf, dst, 0);
      pic_value_sign_extend(pf, dst, 0, value_is_signed(src));
    }
  } else if (!value_is_volatile(src)
      && (value_sz_get(src) < 5)) {
    variable_sz_t ipos;
    variable_sz_t ii;
    variable_sz_t src_sz;
    variable_sz_t src_bit_ofs;

    src_sz      = value_sz_get(src);
    src_bit_ofs = value_bit_offset_get(src);
    src = pic_bit_overlay_create(src);
    pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE, 0, &ipos);
    if (value_is_volatile(dst)
        || pic_value_is_w(dst)
        || !pic_val_bank_same(pf, dst, src)) {
      /* pic_instr_append(pf, pic_opcode_clrw); */
      pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
      for (ii = 0; ii < src_sz; ii++) {
        pic_instr_append_f_bn(pf, pic_opcode_btfsc, src, 
            (src_bit_ofs + ii) / 8, 
            (src_bit_ofs + ii) & 7);
        pic_instr_append_w_kn(pf, pic_opcode_iorlw, 1 << ii);
      }
      if (!pic_value_is_w(dst)) {
        pic_instr_append_f(pf, pic_opcode_movwf, dst, 0);
      }
    } else {
      pic_instr_append_f(pf, pic_opcode_clrf, dst, 0);
      for (ii = 0; ii < src_sz; ii++) {
        pic_instr_append_f_bn(pf, pic_opcode_btfsc, src, 
            (src_bit_ofs + ii) / 8, 
            (src_bit_ofs + ii) & 7);
        pic_instr_append_f_bn(pf, pic_opcode_bsf, dst, 0, ii);
      }
    }
    value_release(src);
    if (!pic_value_is_w(dst)) {
      pic_value_sign_extend(pf, dst, 0, value_is_signed(src));
    }
  } else {
    /* dst = (src >> src_bit_ofs) & mask */
    value_t       tdst;
    value_t       mask;
    variable_sz_t src_sz;
    variable_sz_t src_bit_ofs;

    tdst = (pic_value_is_w(dst) || value_is_volatile(dst)) 
      ? pic_var_temp_get(pf, VARIABLE_DEF_FLAG_NONE,
          pic_value_is_w(dst) ? 1 : pic_result_sz_get(src, VALUE_NONE, dst))
      : dst;
    src_bit_ofs = value_bit_offset_get(src);
    src_sz      = value_sz_get(src);
    src         = pic_bit_overlay_create(src);
    mask        = pfile_constant_get(pf, src_bit_ofs, VARIABLE_DEF_NONE);
    pic_op(pf, operator_shift_right, tdst, src, mask);
    value_release(mask);
    if (pic_value_is_w(dst)) {
      uchar n;

      n = ((1 << src_sz) - 1) & 0xff;
      pic_instr_append_f_d(pf, pic_opcode_movf, tdst, 0, pic_opdst_w);
      pic_instr_append_w_kn(pf, pic_opcode_andlw, n);
    } else {
      mask = pfile_constant_get(pf, (1 << src_sz) - 1,
          VARIABLE_DEF_FLAG_NONE);
      pic_op(pf, operator_andb, dst, tdst, mask);
      value_release(mask);
    }

    if (dst != tdst) {
      pic_var_temp_release(pf, tdst);
    }
    value_release(src);
  }
}

/* dst is not bit, src is const */
static void pic_assign_from_const(pfile_t *pf, value_t dst, value_t src)
{
  variable_sz_t sz;
  variable_sz_t ii;
  variable_sz_t ipos;
  uchar         last;

  sz   = (pic_value_is_w(dst) ? 1 : value_sz_get(dst));
  last = 0;
  pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE, 0, &ipos);
  for (ii = 0; ii < sz; ii++) {
    uchar ch;

    pic_indirect_bump3(pf, dst, VALUE_NONE, VALUE_NONE, ii, &ipos);
    ch = pic_value_const_byte_get(src, ii);
    if ((ii == 0) || (ch != last)) {
      if (ch || pic_value_is_w(dst)) {
        pic_instr_append_w_kn(pf, pic_opcode_movlw, ch);
      }
      last = ch;
    }
    if (!pic_value_is_w(dst)) {
      pic_instr_append_f(pf, (ch) ? pic_opcode_movwf : pic_opcode_clrf, dst, 
          ii);
    }
  }
}

/* dst is anything, src is lookup */
static void pic_assign_from_lookup(pfile_t *pf, value_t dst,
    value_t src)
{
  if (value_is_bit(dst)) {
    /* assign src to a temporary, then the temporary to dst */
    value_t       tmp;
    variable_sz_t sz;

    sz = value_sz_get(src);
    if (sz > value_byte_sz_get(dst)) {
      sz = value_byte_sz_get(dst);
    }
    tmp = pic_var_temp_get(pf, pic_result_flag_get(pf, src, VALUE_NONE), sz);
    pic_op(pf, operator_assign, tmp, src, VALUE_NONE);
    pic_op(pf, operator_assign, dst, tmp, VALUE_NONE);
    pic_var_temp_release(pf, tmp);
  } else {
    variable_sz_t  ii;
    variable_sz_t  ipos;
    variable_sz_t  sz;
    value_t        tmp;
    variable_def_t def;
    value_t        baseofs;
    label_t        lbl;
    value_t         tablat;
    variable_def_member_t mbr;

    lbl = pic_lookup_label_find(pf, value_variable_get(src), 
        PIC_LOOKUP_LABEL_FIND_FLAG_NONE);
    def = variable_def_get(value_variable_get(src));
    mbr = variable_def_member_get(def);
    baseofs = value_baseofs_get(src);

    sz = pic_result_sz_get(src, VALUE_NONE, dst);
    pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE, 0, &ipos);
    if (PIC_TARGET_CPU_16BIT == pic_target_cpu_get(pf)) {
      pic_code_t code;
      value_t    tblptr;

      tblptr = pfile_value_find(pf, pfile_log_err, "_tblptr");
      for (ii = 0; ii < 3; ii++) {
        code   = pic_instr_append(pf, pic_opcode_movlw);
        pic_code_brdst_set(code, lbl);
        pic_code_ofs_set(code, ii);
        pic_instr_append_f(pf, pic_opcode_movwf, tblptr, ii);
      }
      if (baseofs) {
        pic_op(pf, operator_add, tblptr, baseofs, VALUE_NONE);
      }
      value_release(tblptr);
      tablat = pfile_value_find(pf, pfile_log_err, "_tablat");
    } else {
      tablat = VALUE_NONE;
      if (variable_def_member_sz_get(mbr) * variable_def_member_ct_get(mbr)
          > 255) {
        /* need to pass LSB in _pic_loop, MSB in W */
        tmp = pic_var_loop_get(pf);
        pic_instr_append_f_d(pf, pic_opcode_movf, baseofs, 0, pic_opdst_w);
        pic_instr_append_f(pf, pic_opcode_movwf, tmp, 0);
        if (value_byte_sz_get(baseofs) > 1) {
          pic_instr_append_f_d(pf, pic_opcode_movf, baseofs, 1, pic_opdst_w);
        } else {
          /* pic_instr_append(pf, pic_opcode_clrw); */
          pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
        }
      } else {
        /* need to find a way to set PCLATH to the high byte of the dst */
        pic_code_t code;

        code = pic_instr_append(pf, pic_opcode_movlw);
        pic_code_brdst_set(code, lbl);
        pic_code_ofs_set(code, 1);
        pic_instr_append_reg(pf, pic_opcode_movwf, "_pclath");
        pic_instr_append_f_d(pf, pic_opcode_movf, baseofs, 0, pic_opdst_w);
        tmp = baseofs;
      }
    }
    for (ii = 0; ii < sz; ii++) {
      pic_indirect_bump3(pf, dst, VALUE_NONE, VALUE_NONE, ii, &ipos);
      if (PIC_TARGET_CPU_16BIT == pic_target_cpu_get(pf)) {
        pic_instr_append_w_kn(pf, pic_opcode_tblrd, pic_tblptr_change_post_inc);
        pic_instr_append_f_d(pf, pic_opcode_movf, tablat, 0, pic_opdst_w);
      } else {
        if (ii) {
          pic_instr_append_w_kn(pf, pic_opcode_movlw, ii);
          pic_instr_append_f_d(pf, pic_opcode_addwf, tmp, 0, pic_opdst_w);
        }
        pic_instr_append_n(pf, pic_opcode_call, lbl);
      }
      if (!pic_value_is_w(dst)) {
        pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
      }
    }
    if (tmp != baseofs) {
      pic_var_loop_release(pf, tmp);
    }
    value_release(tablat);
    label_release(lbl);
    pic_value_sign_extend(pf, dst, ii - 1, value_is_signed(src));
  }
}

static void pic_assign_from_label(pfile_t *pf, value_t dst, value_t src)
{
  label_t       lbl;
  pic_code_t    code;
  variable_sz_t ipos;

  lbl = value_label_get(src);
  pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE, 0, &ipos);
  code = pic_instr_append(pf, pic_opcode_movlw);
  pic_code_brdst_set(code, lbl);
  pic_instr_append_f(pf, pic_opcode_movwf, dst, 0);
  if (value_sz_get(dst) > 1) {
    size_t ii;

    pic_indirect_bump3(pf, dst, VALUE_NONE, VALUE_NONE, 1, &ipos);
    code = pic_instr_append(pf, pic_opcode_movlw);
    pic_code_brdst_set(code, lbl);
    pic_code_ofs_set(code, 1);
    pic_instr_append_f(pf, pic_opcode_movwf, dst, 1);
    for (ii = 2; ii < value_sz_get(dst); ii++) {
      pic_indirect_bump3(pf, dst, VALUE_NONE, VALUE_NONE, ii, &ipos);
      pic_instr_append_f(pf, pic_opcode_clrf, dst, ii);
    }
  }
}

static void pic_assign_from_function(pfile_t *pf, value_t dst,
  value_t src)
{
  variable_def_t src_def;
  variable_def_t dst_def;
  pic_code_t     code;
  pfile_proc_t  *proc;
  label_t        lbl;
  variable_sz_t  ipos;
  size_t         ii;

  assert(value_is_pointer(dst));
  src_def = value_def_get(src);
  dst_def = value_def_get(dst);
  /* dereference dst def */
  dst_def = variable_def_member_def_get(variable_def_member_get(dst_def));
  assert(variable_def_is_same(src_def, dst_def));

  proc = value_proc_get(src);
  lbl  = pfile_proc_label_get(proc);

  pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE, 0, &ipos);
  for (ii = 0; ii < pic_pointer_size_get(pf); ii++) {
    code = pic_instr_append(pf, pic_opcode_movlw);
    pic_code_brdst_set(code, lbl);
    pic_code_ofs_set(code, ii);
    pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
  }
#if 0
  pic_indirect_bump3(pf, dst, VALUE_NONE, VALUE_NONE, 1, &ipos);
  code = pic_instr_append(pf, pic_opcode_movlw);
  pic_code_brdst_set(code, lbl);
  pic_code_ofs_set(code, 1);
  pic_instr_append_f(pf, pic_opcode_movwf, dst, 1);
#endif
}

static void pic_assign_array_to_pointer(pfile_t *pf, value_t dst, value_t src)
{
  if (value_is_const(src)) {
    /* this is assign from lookup */
    pic_code_t code;
    label_t        lbl;

    abort();
    lbl = pic_lookup_label_find(pf, value_variable_get(src), 
        PIC_LOOKUP_LABEL_FIND_FLAG_NONE);
    code = pic_instr_append(pf, pic_opcode_movlw);
    pic_code_brdst_set(code, lbl);
    pic_instr_append_f(pf, pic_opcode_movwf, dst, 0);
    code = pic_instr_append(pf, pic_opcode_movlw);
    pic_code_brdst_set(code, lbl);
    pic_code_ofs_set(code, 1);
    pic_instr_append_f(pf, pic_opcode_movwf, dst, 1);
    label_release(lbl);
  } else {
    variable_base_t base;
    variable_sz_t   ii;

    base = value_base_get(src);
    for (ii = 0, base = value_base_get(src);
         ii < value_sz_get(dst);
         ii++, base >>= 8) {
      if (base) {
        pic_instr_append_w_kn(pf, pic_opcode_movlw, base & 0xff);
        pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
      } else {
        pic_instr_append_f(pf, pic_opcode_clrf, dst, ii);
      }
    }
  }
}

static void pic_assign_ptr_to_ptr(pfile_t *pf, value_t dst, value_t src)
{
  variable_t    svar;
  variable_t    dvar;
  variable_sz_t ii;

  assert(!value_baseofs_get(src));
  src = value_clone(src);
  value_baseofs_set(src, VALUE_NONE);
  value_indirect_clear(src);

  assert(!value_baseofs_get(dst));
  dst = value_clone(dst);
  value_baseofs_set(dst, VALUE_NONE);
  value_indirect_clear(dst);

  for (ii = 0; ii < pic_pointer_size_get(pf); ii++) {
    pic_instr_append_f_d(pf, pic_opcode_movf, src, ii, pic_opdst_w);
    pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
  }
  svar = value_variable_get(src);
  dvar = value_variable_get(dst);
  value_release(dst);
  value_release(src);
}

/* dst has one or more of the LOOKUP/EEPROM/FLASH bits set */
static void pic_assign_to_ptr(pfile_t *pf, value_t dst, value_t src)
{
  if (value_is_pointer(src)) {
    pic_assign_ptr_to_ptr(pf, dst, src);
  } else if (variable_is_lookup(value_variable_get(src))) {
    pic_code_t    code;
    label_t       lbl;
    variable_t    dvar;
    variable_sz_t ii;

    lbl = pic_lookup_label_find(pf, value_variable_get(src), 
        PIC_LOOKUP_LABEL_FIND_FLAG_ALLOC
        | PIC_LOOKUP_LABEL_FIND_FLAG_DATA);
    dvar = value_variable_get(dst);
    for (ii = 0; ii < pic_pointer_size_get(pf); ii++) {
      code = pic_instr_append(pf, pic_opcode_movlw);
      pic_code_brdst_set(code, lbl);
      pic_code_ofs_set(code, ii);
      if ((ii + 1 == pic_pointer_size_get(pf))
        && ((variable_flag_test(dvar, VARIABLE_FLAG_PTR_PTR)
            && variable_flag_test(dvar, VARIABLE_FLAG_PTR_LOOKUP))
            || variable_flag_test(dvar, VARIABLE_FLAG_PTR_EEPROM)
            || variable_flag_test(dvar, VARIABLE_FLAG_PTR_FLASH))) {
        pic_instr_append_w_kn(pf, pic_opcode_iorlw, 0x40);
      }
      pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
    }
    label_release(lbl);
  } else if (value_is_array(src)) {
    variable_sz_t   ii;
    variable_base_t base;

    base = value_base_get(src);
    for (ii = 0, base = value_base_get(src); 
         ii < pic_pointer_size_get(pf);
         ii++, base >>= 8) {
      if (base) {
        pic_instr_append_w_kn(pf, pic_opcode_movlw, base & 0xff);
        pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
      } else {
        pic_instr_append_f(pf, pic_opcode_clrf, dst, ii);
      }
    }
  } else {
    if (variable_flag_test(value_variable_get(dst),
      VARIABLE_FLAG_PTR_LOOKUP)) {
      pfile_log(pf, pfile_log_err, 
        "Attempt to assign to a lookup table");
    } else {
      assert(0);
    }
  }
}

/* src has one or more of the LOOKUP/EEPROM/FLASH bits set */
static void pic_assign_from_ptr(pfile_t *pf, value_t dst, value_t src)
{
  if (value_is_pointer(dst)) {
    pic_assign_ptr_to_ptr(pf, dst, src);
  } else {
    variable_t    svar;
    value_t       tmp;
    value_t       baseofs;
    variable_sz_t ii;

    src     = value_clone(src);
    baseofs = value_baseofs_get(src);
    value_baseofs_set(src, VALUE_NONE);
    value_indirect_clear(src);

    tmp = pic_var_pointer_get(pf);

#if 1
    for (ii = 0; ii < pic_pointer_size_get(pf); ii++) {
      pic_instr_append_f_d(pf, pic_opcode_movf, src, ii, pic_opdst_w);
      pic_instr_append_f(pf, pic_opcode_movwf, tmp, ii);
    }
    if (VALUE_NONE != baseofs) {
      pic_op(pf, operator_add, tmp, baseofs, VALUE_NONE);
    }
#else
    if (VALUE_NONE != baseofs) {
      pic_op(pf, operator_add, tmp, src, baseofs);
    } else {
      pic_op(pf, operator_assign, tmp, src, VALUE_NONE);
    }
#endif

    svar = value_variable_get(src);
    if ((variable_flag_test(svar, VARIABLE_FLAG_PTR_PTR)
          && variable_flag_test(svar, VARIABLE_FLAG_PTR_LOOKUP))
        || variable_flag_test(svar, VARIABLE_FLAG_PTR_EEPROM)
        || variable_flag_test(svar, VARIABLE_FLAG_PTR_FLASH)) {
      /* this will necessitate a function call */
      variable_sz_t sz;
      variable_sz_t ipos;
      label_t       lbl;

      lbl = pic_label_find(pf, PIC_LABEL_PTR_READ, boolean_true);
      sz  = pic_result_sz_get(src, VALUE_NONE, dst);
      pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE, 0, &ipos);
      for (ii = 0; ii < sz; ii++) {
        if (ii) {
          pic_instr_append_f_d(pf, pic_opcode_incf, tmp, 0, pic_opdst_f);
          pic_instr_append_reg_flag(pf, pic_opcode_btfsc, "_status", "_z");
          pic_instr_append_f_d(pf, pic_opcode_incf, tmp, 1, pic_opdst_f);
        }
        pic_indirect_bump3(pf, dst, VALUE_NONE, VALUE_NONE, ii, &ipos);
        pic_instr_append_n(pf, pic_opcode_call, lbl);
        if (!pic_value_is_w(dst)) {
          pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
        }
      }
      label_release(lbl);
    } else if (variable_flag_test(svar, VARIABLE_FLAG_PTR_PTR)) {
      /* ptr --> dst */
      assert(0);
    } else if (variable_flag_test(svar, VARIABLE_FLAG_PTR_LOOKUP)) {
      /* lookup --> dst */
      /* pic_sign holds LSB */
      /* W holds MSB */
      variable_sz_t sz;
      variable_sz_t ipos;

      pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE, 0, &ipos);
      sz = pic_result_sz_get(src, VALUE_NONE, dst);
      if (PIC_TARGET_CPU_16BIT == pic_target_cpu_get(pf)) {
        value_t tblptr;
        value_t tablat;

        tblptr = pfile_value_find(pf, pfile_log_err, "_tblptr");
        tablat = pfile_value_find(pf, pfile_log_err, "_tablat");
        pic_op(pf, operator_assign, tblptr, tmp, VALUE_NONE);
        for (ii = 0; ii < sz; ii++) {
          pic_indirect_bump3(pf, dst, VALUE_NONE, VALUE_NONE, ii, &ipos);
          pic_instr_append_w_kn(pf, pic_opcode_tblrd, pic_tblptr_change_post_inc);
          pic_instr_append_f_d(pf, pic_opcode_movf, tablat, 0, pic_opdst_w);
          pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
        }
        value_release(tablat);
        value_release(tblptr);
      } else {
        label_t       lbl;

        lbl = pic_label_find(pf, PIC_LABEL_INDIRECT, boolean_true);
        for (ii = 0; ii < sz; ii++) {
          if (ii) {
            pic_op(pf, operator_incr, tmp, VALUE_NONE, VALUE_NONE);
          }
          pic_indirect_bump3(pf, dst, VALUE_NONE, VALUE_NONE, ii, &ipos);
          pic_instr_append_f_d(pf, pic_opcode_movf, tmp, 
            pic_pointer_size_get(pf) - 1, pic_opdst_w);
          pic_instr_append_n(pf, pic_opcode_call, lbl);
          if (!pic_value_is_w(dst)) {
            pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
          }
        }
        label_release(lbl);
      }
    }
    value_release(src);
    pic_var_pointer_release(pf, tmp);
  }
}

/* note: dst or src can be W */
static void pic_op_assign(pfile_t *pf, operator_t op, value_t dst,
    value_t src, value_t val2)
{
  if (value_is_same(dst, src)) {
    /* this might happen during some internal work, ignore it */
  } else if (value_is_function(src)) {
    pic_assign_from_function(pf, dst, src);
  } else if (value_is_label(src)) {
    pic_assign_from_label(pf, dst, src);
  } else if (variable_flags_get_all(value_variable_get(dst))
      & (VARIABLE_FLAG_PTR_LOOKUP
        | VARIABLE_FLAG_PTR_EEPROM
        | VARIABLE_FLAG_PTR_FLASH)) {
    pic_assign_to_ptr(pf, dst, src);
  } else if (variable_flags_get_all(value_variable_get(src))
      & (VARIABLE_FLAG_PTR_LOOKUP
        | VARIABLE_FLAG_PTR_EEPROM
        | VARIABLE_FLAG_PTR_FLASH)) {
    pic_assign_from_ptr(pf, dst, src);
  } else if (value_is_indirect(dst) && value_is_indirect(src)) {
    /* src must move to a temporary */
    value_t tmp;

    tmp = pic_var_temp_get_assign(pf, src, VALUE_NONE, dst);
    pic_op(pf, operator_assign, dst, tmp, VALUE_NONE);
    pic_var_temp_release(pf, tmp);
  } else if (value_is_lookup(src)) {
    pic_assign_from_lookup(pf, dst, src);
  } else if (value_is_bit(dst)) {
    pic_assign_to_bit(pf, dst, src);
  } else if (value_is_pointer(dst) && value_is_array(src)) {
    pic_assign_array_to_pointer(pf, dst, src);
  } else if (value_is_const(src)) {
    pic_assign_from_const(pf, dst, src);
  } else if (value_is_bit(src)) {
    pic_assign_from_bit(pf, dst, src);
  } else {
    /* simple assignment from src to dst */
    variable_sz_t ipos;
    variable_sz_t ii;
    variable_sz_t sz;
    value_t       accum;

    accum = VALUE_NONE;
    sz = pic_result_sz_get(src, VALUE_NONE, dst);
    if (pic_value_is_w(dst)) {
      sz = 1;
    }
    if (pic_value_is_w(src) && value_is_indirect(dst)) {
      /* store src in the accumulator */
      accum = pic_var_accum_get(pf);
      src   = accum;
      pic_instr_append_f(pf, pic_opcode_movwf, accum, 0);
    }
    pic_indirect_setup3(pf, src, VALUE_NONE, dst, 0, &ipos);
    for (ii = 0; ii < sz; ii++) {
      pic_indirect_bump3(pf, dst, src, VALUE_NONE, ii, &ipos);
      if (!pic_value_is_w(src)) {
        pic_instr_append_f_d(pf, pic_opcode_movf, src, ii, pic_opdst_w);
      }
      if (!pic_value_is_w(dst)) {
        pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
      }
    }
    pic_value_sign_extend(pf, dst, ii - 1, value_is_signed(src));
    if (accum) {
      pic_var_accum_release(pf, accum);
    }
  }
}

/* operator on val1 & dst (assign, neg, etc.). if val1 is signed,
 * and its size is smaller than dst, and the result *is not*
 * signed, shrink dst to val1 to eliminate sign extension,
 * perform the operation, then zero out the upper bytes in dst
 */
static void pic_unary_fixup(pfile_t *pf, operator_t op,
    value_t dst, value_t val1, value_t val2)
{
  if (value_is_signed(val1)
      && !pic_result_is_signed(pf, val1, val2)
      && (pic_result_sz_get(val1, val2, dst) < value_byte_sz_get(dst))) {
    variable_def_t def;
    variable_def_t def_fixed;

    def = value_def_get(dst);
    def_fixed = variable_def_alloc(0, 
        variable_def_type_get(def),
        variable_def_flags_get_all(def),
        pic_result_sz_get(val1, val2, dst));
    value_def_set(dst, def_fixed);
    pic_op(pf, op, dst, val1, val2);
    value_def_set(dst, def);
    pic_value_sign_extend(pf, dst, pic_result_sz_get(val1, val2, dst) - 1,
        boolean_false);
  } else {
    pic_op(pf, op, dst, val1, val2);
  }
}

/*
 * ADD: a = a + b
 *   movf   b_n, w
 *   if (n >= 1)
 *     btfsc _c
 *     incfsz b_n, w
 *   fi  
 *   addwf  a_n, f
 * 
 * SUB: a = a - b
 *   movf   b_n, w
 *   if (n >= 1)
 *     btfss  _c
 *     incfsz b_n
 *   fi  
 *   subwf  a_n
 *
 * nb : in the case where size <= 2, a and b can be in different
 *      banks. in the case where size > 2, a and b *must* be in the
 *      same bank. For that case a special register is defined,
 *      pic_accum. This variable *must* exist in all banks either in 
 *      a shared space (preferable), or at the same offset in each bank.
 */
/* val1 must be const, 1 or 2 bytes */
static void pic_sub_from_const_small(pfile_t *pf, value_t dst,
    value_t val1, value_t val2)
{
  variable_sz_t sz;
  variable_sz_t ipos;

  assert(value_is_const(val1));
  sz = pic_result_sz_get(val1, val2, dst);
  assert(sz <= 2);
  pic_indirect_setup3(pf, dst, val1, val2, 0, &ipos);
  if ((1 == sz) && !pic_is_12bit(pf)) {
    /* yet another odd special case. negate val2 & add the appropriate
     * constant. 2s complement negate is complement + increment which
     * explains why we're adding (1 + val1)
     */
    pic_indirect_setup3(pf, val1, val2, dst, 0, &ipos);
    pic_instr_append_f_d(pf, pic_opcode_comf, val2, 0, pic_opdst_w);
    pic_instr_append_w_kn(pf, pic_opcode_addlw,
        (1 + value_const_get(val1)) & 0xff);
    pic_instr_append_f(pf, pic_opcode_movwf, dst, 0);
    pic_value_sign_extend(pf, dst, 0, pic_result_is_signed(pf, val1, val2));
  } else {
    value_t       accum;
    value_t       val2_sign;
    variable_sz_t ii;

    accum = pic_var_accum_get(pf);
    val2_sign = VALUE_NONE;
    for (ii = 0; ii < sz; ii++) {
      pic_indirect_bump3(pf, dst, val1, val2, ii, &ipos);
      if (ii == value_sz_get(val2)) {
        value_t tmp;

        tmp = pic_var_temp_get(pf, VARIABLE_DEF_FLAG_NONE, 1);
        val2_sign = pic_value_sign_get(pf, val2, tmp);
        if (!val2_sign) {
          val2_sign = tmp;
          pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
          pic_instr_append_f(pf, pic_opcode_movwf, val2_sign, 0);
        }
        val2 = val2_sign;
      }
      pic_instr_append_w_kn(pf, pic_opcode_movlw, 
          pic_value_const_byte_get(val1, ii));
      pic_instr_append_f(pf, pic_opcode_movwf, accum, 0);
      pic_instr_append_f_d(pf, pic_opcode_movf, val2, 
          (val2 == val2_sign) ? 0 : ii, pic_opdst_w);
      if (ii) {
        pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_c");
        pic_instr_append_f_d(pf, pic_opcode_incf, val2, 
            (val2 == val2_sign) ? 0 : ii, pic_opdst_w);
      }
      pic_instr_append_f_d(pf, pic_opcode_subwf, accum, 0, pic_opdst_w);
      pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
    }
    if (VALUE_NONE != val2_sign) {
      pic_var_temp_release(pf, val2_sign);
    }
    pic_var_accum_release(pf, accum);
    pic_value_sign_extend(pf, dst, ii - 1, 
      pic_result_is_signed(pf, val1, val2));
  }
}

/* val2 must be const (any size) */
static void pic_add_sub_const(pfile_t *pf, operator_t op,
    value_t dst, value_t val1, value_t val2)
{
  value_t       accum;
  value_t       val1_sign;
  variable_sz_t sz;
  variable_sz_t ipos;
  variable_sz_t ii;
  pic_opcode_t  op_action;
  pic_opcode_t  op_condition;

  assert(value_is_const(val2));
  sz = pic_result_sz_get(val1, val2, dst);
  accum = (sz > 2)
    ? pic_var_accum_get(pf)
    : VALUE_NONE;
  val1_sign = VALUE_NONE;

  if (operator_add == op) {
    op_action    = pic_opcode_addwf;
    op_condition = pic_opcode_btfsc;
  } else {
    op_action    = pic_opcode_subwf;
    op_condition = pic_opcode_btfss;
  }
  pic_indirect_setup3(pf, dst, val1, val2, 0, &ipos);
  for (ii = 0; ii < sz; ii++) {
    pic_indirect_bump3(pf, dst, val1, val2, ii, &ipos);
    if (ii == value_sz_get(val1)) {
      value_t tmp;

      tmp = (VALUE_NONE == accum) 
        ? pic_var_accum_get(pf)
        : pic_var_temp_get(pf, VARIABLE_DEF_FLAG_NONE, 1);

      val1_sign = pic_value_sign_get(pf, val1, tmp);
      if (VALUE_NONE == val1_sign) {
        pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
        pic_instr_append_f(pf, pic_opcode_movwf, tmp, 0);
        val1_sign = tmp;
      }
      val1 = val1_sign;
    }
    pic_instr_append_w_kn(pf, pic_opcode_movlw,
        pic_value_const_byte_get(val2, ii));
    if (ii) {
      if (sz > 2) {
        pic_instr_append_f(pf, pic_opcode_movwf, accum, 0);
        pic_instr_append_daop(pf, val1, op_action);
      }
      pic_instr_append_reg_flag(pf, op_condition, "_status", "_c");
      if (sz > 2) {
        pic_instr_append_f_d(pf, pic_opcode_incfsz, accum, 0, pic_opdst_w);
      } else {
        if (pic_is_12bit(pf)) {
          pic_instr_append_w_kn(pf, pic_opcode_movlw,
            pic_value_const_byte_get(val2, ii) + 1);
          assert(ii + 1 == sz);
        } else {
          pic_instr_append_w_kn(pf, pic_opcode_addlw, 1);
        }
      }
    }
    pic_instr_append_f_d(pf, op_action, val1,
        (val1 == val1_sign) ? 0 : ii,
        (value_is_same(val1, dst)) ? pic_opdst_f : pic_opdst_w);
    if (!value_is_same(val1, dst)) {
      pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
    }
  }
  if (VALUE_NONE != val1_sign) {
    if (VALUE_NONE == accum) {
      pic_var_accum_release(pf, val1_sign);
    } else {
      pic_var_temp_release(pf, val1_sign);
    }
  }
  if (VALUE_NONE != accum) {
    pic_var_accum_release(pf, accum);
  }
  pic_value_sign_extend(pf, dst, ii - 1, pic_result_is_signed(pf, val1, val2));
}

static void pic_add_sub_with_accum(pfile_t *pf, operator_t op, value_t dst,
  value_t val1, value_t val2)
{
  variable_sz_t sz;
  variable_sz_t ii;
  variable_sz_t ipos;
  value_t       accum;
  pic_opcode_t  op_action_f;
  pic_opcode_t  op_condition;
  value_t       val1_sign;
  value_t       val2_sign;

  val1_sign = VALUE_NONE;
  val2_sign = VALUE_NONE;
  if (operator_add == op) {
    op_action_f  = pic_opcode_addwf;
    op_condition = pic_opcode_btfsc;
  } else {
    op_action_f  = pic_opcode_subwf;
    op_condition = pic_opcode_btfss;
  }
  accum = pic_var_accum_get(pf);
  sz    = pic_result_sz_get(val1, val2, dst);
  pic_indirect_setup3(pf, dst, val1, val2, 0, &ipos);
  for (ii = 0; ii < sz; ii++) {
    pic_indirect_bump3(pf, dst, val1, val2, ii, &ipos);
    if (ii == value_sz_get(val1)) {
      value_t tmp;

      tmp = pic_var_temp_get(pf, VARIABLE_DEF_FLAG_NONE, 1);
      val1_sign = pic_value_sign_get(pf, val1, tmp);
      if (VALUE_NONE == val1_sign) {
        val1_sign = tmp;
        pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
        pic_instr_append_f(pf, pic_opcode_movwf, val1_sign, 0);
      }
      val1 = val1_sign;
    } else if ((ii == value_sz_get(val2)) && !value_is_const(val2)) {
      val2_sign = pic_value_sign_get(pf, val2, accum);
      val2      = val2_sign;
    }
    if (value_is_const(val2) || (VALUE_NONE == val2)) {
      pic_instr_append_w_kn(pf, pic_opcode_movlw, 
          (VALUE_NONE == val2) ? 0 : pic_value_const_byte_get(val2, ii));
    } else {
      pic_instr_append_f_d(pf, pic_opcode_movf, val2, 
        (val2 == val2_sign) ? 0 : ii, pic_opdst_w);
    }

    if (ii) {
      if (val2 != accum) {
        pic_instr_append_f(pf, pic_opcode_movwf, accum, 0);
      }
      pic_instr_append_daop(pf, dst, op_action_f);
      pic_instr_append_reg_flag(pf, op_condition, "_status", "_c");
      pic_instr_append_f_d(pf, 
          pic_opcode_incfsz, /* (sz > 2) ? pic_opcode_incfsz : pic_opcode_incf, */
          accum, 0, pic_opdst_w);
    }
    pic_instr_append_f_d(pf, op_action_f, dst, ii, pic_opdst_f);
  }
  pic_var_accum_release(pf, accum);
  if (val1_sign == val1) {
    pic_var_temp_release(pf, val1_sign);
  }
  pic_value_sign_extend(pf, dst, ii - 1,
      pic_result_is_signed(pf, val1, val2));
}

/* 
 * the 16 bit cores change the status<c> on incf/decf operations
 * so the 12 & 14 bit add/sub routines cannot be used. Use the new
 *    addwfc -- w + f + c --> dst
 *    subfwb -- w - f - !c --> dst
 *    subwfb -- f - w - !c --> dst
 * these routines are much simpler:
 *    movf val1,w
 *    addwf val2,w
 *    movwf dst
 *    repeat
 *      movf   val1+n, w
 *      addwfc val2+n, w
 *      movwf  dst
 */
static void pic_add_sub_16bit(pfile_t *pf, operator_t op, value_t dst,
  value_t val1, value_t val2)
{
  variable_sz_t sz;
  pic_opdst_t   pop_dst;
  pic_opcode_t  pop;      /* opcode for first byte      */
  pic_opcode_t  pop_cont; /* opcode for remaining bytes */
  variable_sz_t ii;
  value_t       val1_sign;
  value_t       val2_sign;
  boolean_t     release_val;
  boolean_t     result_is_signed;

  assert(!value_is_indirect(dst));
  assert(!value_is_indirect(val1));
  assert(!value_is_indirect(val2));

  sz = pic_result_sz_get(val1, val2, dst);
  result_is_signed = pic_result_is_signed(pf, val1, val2);
  /* nb: we always want the const to be in val1! */
  release_val = boolean_false;
  if ((operator_sub == op) && value_is_const(val2)) {
    /* var + const = (-const) + var */
    val2 = pfile_constant_get(pf, -value_const_get(val2), VARIABLE_DEF_NONE);
    release_val = boolean_true;
    op = operator_add;
  }
  if (value_is_const(val2)
    || ((operator_add == op) && value_is_same(val1, dst))) {
    value_t tmp;

    tmp  = val1;
    val1 = val2;
    val2 = tmp;
  }
  assert(!value_is_const(val2));
  pop_dst = (value_is_same(val2, dst)) ? pic_opdst_f : pic_opdst_w;
  if (operator_add == op) {
    pop      = pic_opcode_addwf;
    pop_cont = pic_opcode_addwfc;
  } else {
    pop      = pic_opcode_subfwb;
    pop_cont = pic_opcode_subfwb;
  }
  val1_sign = val1;
  val2_sign = val2;
  if (operator_sub == op) {
    pic_instr_append_reg_flag(pf, pic_opcode_bsf, "_status", "_c");
  }
  for (ii = 0; ii < sz; ii++) {
    if (ii == value_sz_get(val1) && !value_is_const(val1)) {
      val1_sign = pic_value_sign_get(pf, val1, VALUE_NONE);
      if (VALUE_NONE == val1_sign) {
        val1_sign = pic_var_accum_get(pf);
        pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
        pic_instr_append_f(pf, pic_opcode_movwf, val1_sign, 0);
      }
    } else if (ii == value_sz_get(val2)) {
      val2_sign = pic_value_sign_get(pf, val2, VALUE_NONE);
      if (VALUE_NONE == val2_sign) {
        val2_sign = pic_var_accum_get(pf);
        pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
        pic_instr_append_f(pf, pic_opcode_movwf, val2_sign, 0);
      }
    }
    if (value_is_const(val1)) {
      pic_instr_append_w_kn(pf, pic_opcode_movlw, 
        pic_value_const_byte_get(val1, ii));
    } else {
      pic_instr_append_f_d(pf, pic_opcode_movf, val1_sign, 
        (val1_sign == val1) ? ii : 0, pic_opdst_w);
    }
    pic_instr_append_f_d(pf, pop, val2_sign,
      (val2_sign == val2) ? ii : 0, pop_dst);
    if (pic_opdst_w == pop_dst) {
      pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
    }
    pop = pop_cont;
  }
  if (val1_sign != val1) {
    pic_var_accum_release(pf, val1_sign);
  }
  if (val2_sign != val2) {
    pic_var_accum_release(pf, val2_sign);
  }
  if (release_val) {
    value_release(val1);
  }
  pic_value_sign_extend(pf, dst, ii - 1, result_is_signed);
}

static void pic_op_add_sub(pfile_t *pf, operator_t op, value_t dst,
  value_t val1, value_t val2)
{
  variable_sz_t sz;

  assert(!pic_value_is_w(dst));
  assert(!pic_value_is_w(val1));
  assert(!pic_value_is_w(val2));

  sz = pic_result_sz_get(val1, val2, dst);
  /* one constant for addition : put into val2
   * if (dst,val2) are the same swap
   */
  if ((operator_add == op) && 
      (value_is_const(val1) || value_is_same(dst, val2))) {
    SWAP(value_t, val1, val2);
  }
  if (value_is_const(val2) && (0 == value_const_get(val2))) {
    /* if val1 is signed, but the result is unsigned, change val1 */
    pic_unary_fixup(pf, operator_assign, dst, val1, val2);
  } else if (value_is_const(val2) && (1 == value_const_get(val2))) {
    pic_unary_fixup(pf, (operator_add == op) ? operator_incr : operator_decr, 
        dst, val1, val2);
  } else if (value_is_const(val1) && (0 == value_const_get(val1))) {
    pic_unary_fixup(pf, (operator_add == op) ? operator_assign : operator_neg,
        dst, val2, val1);
  } else if (pic_work_in_temp(pf, op, dst, val1, val2,
        PIC_WORK_IN_TEMP_FLAG_VALS
        | PIC_WORK_IN_TEMP_FLAG_DST
        | ((sz > 1) ? PIC_WORK_IN_TEMP_FLAG_DST_VOLATILE : 0))) {
  } else if (PIC_TARGET_CPU_16BIT == pic_target_cpu_get(pf)) {
    pic_add_sub_16bit(pf, op, dst, val1, val2);
  } else if (!value_is_same(dst, val1) 
    /* && !value_is_const(val2) */ /* CHECK THIS! */ 
    /* && (sz > 2)*/ ) {
    /* PIC addition & subtraction is funky if the size is greater than
     * two. In this case, create a temporary! */
    value_t tmp;

    tmp = pic_var_temp_get(pf, pic_result_flag_get(pf, val1, val2),
        pic_result_sz_get(val1, val2, dst));
    pic_op(pf, operator_assign, tmp, val1, VALUE_NONE);
    pic_op(pf, op, tmp, tmp, val2);
    pic_op(pf, operator_assign, dst, tmp, VALUE_NONE);
    pic_var_temp_release(pf, tmp);
  } else {
    if ((operator_sub == op) && value_is_const(val1)) {
      pic_sub_from_const_small(pf, dst, val1, val2);
    } else if (value_is_const(val2)) {
      pic_add_sub_const(pf, op, dst, val1, val2);
    } else if (((sz > 1) && (value_is_const(val2) 
            || value_is_volatile(val2)
            || !pic_val_bank_same(pf, val1, val2)))
        /* || ((sz > 1) && value_is_volatile(val2))*/) {
      /* we need to use the accumulator */
      pic_add_sub_with_accum(pf, op, dst, val1, val2);
    } else {
      value_t       val1_sign;
      value_t       val2_sign;
      variable_sz_t ii;
      variable_sz_t ipos;
      pic_opcode_t  op_action_f;
      pic_opcode_t  op_condition;

      if (operator_add == op) {
        op_action_f  = pic_opcode_addwf;
        op_condition = pic_opcode_btfsc;
      } else {
        op_action_f  = pic_opcode_subwf;
        op_condition = pic_opcode_btfss;
      }
      val1_sign = VALUE_NONE;
      val2_sign = VALUE_NONE;
      sz = pic_result_sz_get(val1, val2, dst);
      pic_indirect_setup3(pf, dst, val1, val2, 0, &ipos);
      for (ii = 0; ii < sz; ii++) {
        pic_indirect_bump3(pf, dst, val1, val2, ii, &ipos);
        if (ii == value_sz_get(val1)) {
          val1_sign = pic_value_sign_get(pf, val1, VALUE_NONE);
          if (VALUE_NONE == val1_sign) {
            val1_sign = pic_var_accum_get(pf);
            pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
            pic_instr_append_f(pf, pic_opcode_movwf, val1_sign, 0);
          }
          val1 = val1_sign;
        } else if ((ii == value_sz_get(val2)) && !value_is_const(val2)) {
          val2_sign = pic_value_sign_get(pf, val2, VALUE_NONE);
          if (VALUE_NONE == val2_sign) {
            val2_sign = pic_var_accum_get(pf);
            pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
            pic_instr_append_f(pf, pic_opcode_movwf, val2_sign, 0);
          }
          val2 = val2_sign;
        }
        pic_instr_append_f_d(pf, pic_opcode_movf, val2, 
            (val2 == val2_sign) ? 0 : ii, pic_opdst_w);
        if (ii >= 1) {
          if (!value_is_indirect(val1)) {
            pic_instr_append_daop(pf, val1, op_action_f);
          }
          pic_instr_append_reg_flag(pf, op_condition, "_status", "_c");
          /*
           * do not use incfsz if the banks are different or
           * dst isn't the same as val1. In the former case,
           * the bank setup bits will be skipped, in the later
           * the move from result --> dst will be skipped
           */
          pic_instr_append_f_d(pf, 
              (value_is_same(dst, val1)
              && pic_val_bank_same(pf, val1, val2))
                ? pic_opcode_incfsz : pic_opcode_incf, 
              val2, (val2_sign == val2) ? 0 : ii, pic_opdst_w);
        }
        if (value_is_same(dst, val1)) {
          pic_instr_append_f_d(pf, op_action_f, dst, ii, pic_opdst_f);
        } else {
          pic_instr_append_f_d(pf, op_action_f, val1, ii, pic_opdst_w);
          pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
        }
      }
      if (VALUE_NONE != val1_sign) {
        pic_var_accum_release(pf, val1_sign);
      }
      if (VALUE_NONE != val2_sign) {
        pic_var_accum_release(pf, val2_sign);
      }
      pic_value_sign_extend(pf, dst, ii - 1,
          pic_result_is_signed(pf, val1, val2));
    }
  }
}

/* 
 * return the shift value for a multiply or divide,
 * or 0 if the value is not const or not shiftable
 * (assumes shift of 0 is optimized out earlier)
 */
static value_t pic_shift_get(pfile_t *pf, value_t val, value_t val2)
{
  value_t shift_val;

  shift_val = VALUE_NONE;
  if (value_is_const(val)) {
    variable_const_t n;

    n = value_const_get(val);
    if (!(n & (n - 1))) {
      /* make this a shift */
      variable_const_t shift;

      for (shift = -1; n; shift++, n >>= 1)
        ;
      shift_val = pfile_constant_get(pf, shift, 
          pfile_variable_def_promotion_get(pf, operator_null, val, val2));
    }
  }
  return shift_val;
}

static void pic_op_mul(pfile_t *pf, operator_t op, value_t dst,
  value_t val1, value_t val2)
{
  /* make sure the constant is on the right */
  assert(!pic_value_is_w(dst));
  assert(!pic_value_is_w(val1));
  assert(!pic_value_is_w(val2));

  if (value_is_const(val1)) {
    SWAP(value_t, val1, val2);
  }
  if (value_is_const(val2) && (0 == value_const_get(val2))) {
    pic_assign_const(pf, dst, val1, val2, 0);
  } else if (value_is_const(val2) && (1 == value_const_get(val2))) {
    pic_unary_fixup(pf, operator_assign, dst, val1, val2);
  } else if (value_is_const(val2) && (-1 == value_const_get(val2))) {
    pic_unary_fixup(pf, operator_neg, dst, val1, val2);
  } else {
    value_t shift_val;

    shift_val = pic_shift_get(pf, val2, val1);
    if (shift_val) {
      pic_op(pf, operator_shift_left, dst, val1, shift_val);
      value_release(shift_val);
    } else {
      pic_var_mul_t mvars;
      variable_sz_t sz;

      sz = pic_result_sz_get(val1, val2, dst);

      pic_var_mul_get(pf, sz, &mvars);
      if ((value_is_same(val1, pic_last_values[PIC_LAST_VALUE_MULTIPLIER])
          && value_is_same(val2, pic_last_values[PIC_LAST_VALUE_MULTIPLICAND]))
          || (value_is_same(val1, pic_last_values[PIC_LAST_VALUE_MULTIPLICAND])
            && value_is_same(val2, pic_last_values[PIC_LAST_VALUE_MULTIPLIER]))
          ) {
        /* we needn't do anything here */
      } else {
        label_t fn;

        pic_op(pf, operator_assign, mvars.multiplier,   val1, VALUE_NONE);
        pic_op(pf, operator_assign, mvars.multiplicand, val2, VALUE_NONE);
        fn = pic_label_find(pf, PIC_LABEL_MULTIPLY, boolean_true);
        pic_instr_append_n(pf, pic_opcode_call, fn);
        label_release(fn);
        pic_last_values_reset();
        pic_last_value_set(PIC_LAST_VALUE_MULTIPLIER,   val1);
        pic_last_value_set(PIC_LAST_VALUE_MULTIPLICAND, val2);
      }
      /* make sure mvars.mresult has the correct sign! */
      if (pic_result_is_signed(pf, val1, val2) 
          ^ value_is_signed(mvars.mresult)) {
        variable_def_t def;
        value_t        tmp;

        tmp = value_clone(mvars.mresult);
        def = pfile_variable_def_promotion_get(pf, operator_null,
            val1, val2);
        value_def_set(tmp, def);
        pic_op(pf, operator_assign, dst, tmp, VALUE_NONE);
        value_release(tmp);
      } else {
        pic_op(pf, operator_assign, dst, mvars.mresult, VALUE_NONE);
      }
      pic_var_mul_release(pf, &mvars);
    }
  }
}

static void pic_op_div_mod(pfile_t *pf, operator_t op, value_t dst, 
    value_t val1, value_t val2)
{
  assert(!pic_value_is_w(dst));
  assert(!pic_value_is_w(val1));
  assert(!pic_value_is_w(val2));

  if (value_is_const(val2) && (0 == value_const_get(val2))) {
    pfile_log(pf, pfile_log_err, "division by zero");
  } else if (value_is_same(val1, val2)) {
    pic_assign_const(pf, dst, val1, val2, (operator_div == op) ? 1 : 0);
  } else if (value_is_const(val2) && (1 == value_const_get(val2))) {
    if (operator_div == op) {
      pic_unary_fixup(pf, operator_assign, dst, val1, val2);
    } else {
      pic_assign_const(pf, dst, val1, val2, 0);
    }
  } else if (value_is_const(val2) 
      && pic_result_is_signed(pf, val1, val2)
      && (-1 == value_const_get(val2))) {
    if (operator_div == op) {
      pic_unary_fixup(pf, operator_neg, dst, val1, val2);
    } else {
      pic_assign_const(pf, dst, val1, val2,  0);
    }
  } else {
    value_t shift_val;

    shift_val = pic_shift_get(pf, val2, val1);
    /* cannot shift right if the result is signed, for example
     * -1 / 8 should be zero, but would shift into -1 */
    if (shift_val && !pic_result_is_signed(pf, val1, val2)) {
      if (operator_div == op) {
        /* right shift */
        pic_op(pf, operator_shift_right, dst, val1, shift_val);
      } else if (operator_mod == op) {
        /* binary and */
        value_t tmp;

        tmp = pfile_constant_get(pf, value_const_get(val2) - 1, 
            pfile_variable_def_promotion_get(pf, op, val1, val2));
        pic_op(pf, operator_andb, dst, val1, tmp);
        value_release(tmp);
      }
      value_release(shift_val);
    } else {
      variable_sz_t sz;
      pic_var_div_t dvars;
      value_t       result;

      sz = pic_result_sz_get(val1, val2, VALUE_NONE);
      pic_var_div_get(pf, sz, &dvars);
      if (value_is_same(pic_last_values[PIC_LAST_VALUE_DIVIDEND], val1)
          && value_is_same(pic_last_values[PIC_LAST_VALUE_DIVISOR], val2)) {
      } else {
        label_t fn;

        pic_op(pf, operator_assign, dvars.divisor,  val2, VALUE_NONE);
        pic_op(pf, operator_assign, dvars.dividend, val1, VALUE_NONE);
        fn = pic_label_find(pf,
            (VARIABLE_DEF_FLAG_SIGNED == pic_result_flag_get(pf, val1, val2))
            ? PIC_LABEL_SDIVIDE
            : PIC_LABEL_DIVIDE,
            boolean_true);
        pic_instr_append_n(pf, pic_opcode_call, fn);
        label_release(fn);
        pic_last_values_reset();
        pic_last_value_set(PIC_LAST_VALUE_DIVIDEND, val1);
        pic_last_value_set(PIC_LAST_VALUE_DIVISOR, val2);
      }
      result = (operator_div == op)
        ? dvars.quotient : dvars.remainder;
      if (pic_result_is_signed(pf, val1, val2)
          ^ value_is_signed(result)) {
        variable_def_t def;
        value_t        tmp;

        tmp = value_clone(result);
        def = pfile_variable_def_promotion_get(pf, operator_null,
            val1, val2);
        value_def_set(tmp, def);
        pic_op(pf, operator_assign, dst, tmp, VALUE_NONE);
        value_release(tmp);
      } else {
        pic_op(pf, operator_assign, dst, result, VALUE_NONE);
      }
      pic_var_div_release(pf, &dvars);
    }
  }
}

/* if dst is a single, non-volatile bit, clear it for 0, set for 1
 */
#define PIC_ASSIGN_LOGICAL_TO_DST_NONE 0x0000
#define PIC_ASSIGN_LOGICAL_TO_DST_OP   0x0001 /* finish with pic_op(..) */
static void pic_assign_logical_to_dst(pfile_t *pf, value_t dst, uchar n,
    unsigned flags)
{
  assert((0 == n) || (1 == n));
  if (value_is_single_bit(dst)) {
    pic_instr_append_f(pf,
        (n) ? pic_opcode_bsf : pic_opcode_bcf,
        dst, 0);
  } else {
    pic_instr_append_w_kn(pf, pic_opcode_movlw, n);
    if (flags & PIC_ASSIGN_LOGICAL_TO_DST_OP) {
      /* assign the result to dst */
      pic_op(pf, operator_assign, dst, VALUE_NONE, VALUE_NONE);
    }
  }
}

typedef enum {
  PIC_VALUE_TO_W_NONE, /* unused */
  PIC_VALUE_TO_W_MOV,
  PIC_VALUE_TO_W_SUB,
  PIC_VALUE_TO_W_ADD
} pic_value_to_w_t;

static void pic_value_to_w(pfile_t *pf, pic_value_to_w_t op,
    value_t val, value_t val_sign, variable_sz_t pos)
{
  if (value_is_const(val) 
      || (VALUE_NONE == val)
      || ((VALUE_NONE == val_sign) && (pos >= value_sz_get(val)))) {
    pic_opcode_t pop;

    pop = pic_opcode_none;
    switch (op) {
      case PIC_VALUE_TO_W_NONE: break;
      case PIC_VALUE_TO_W_MOV:  pop = pic_opcode_movlw; break;
      case PIC_VALUE_TO_W_SUB:  pop = pic_opcode_sublw; break;
      case PIC_VALUE_TO_W_ADD:  pop = pic_opcode_addlw; break;
    }
    if ((VALUE_NONE == val)/* || (pos >= value_sz_get(val))*/) {
      pic_instr_append_w_kn(pf, pop, 0);
    } else {
      pic_instr_append_w_kn(pf, pop, pic_value_const_byte_get(val, pos));
    }
  } else {
    pic_opcode_t pop;

    pop = pic_opcode_none;
    switch (op) {
      case PIC_VALUE_TO_W_NONE: break;
      case PIC_VALUE_TO_W_MOV:  pop = pic_opcode_movf;  break;
      case PIC_VALUE_TO_W_SUB:  pop = pic_opcode_subwf; break;
      case PIC_VALUE_TO_W_ADD:  pop = pic_opcode_addwf; break;
    }
    if (pos >= value_sz_get(val)) {
      pic_instr_append_f_d(pf, pop, val_sign, 0, pic_opdst_w);
    } else {
      pic_instr_append_f_d(pf, pop, val, pos, pic_opdst_w);
    }
  }
}

/* when this is called, W is either 0, 1, or 255
 * in the later case, if w is 255 we'll want to replicate it
 */
static void pic_assign_from_w(pfile_t *pf, value_t dst, boolean_t sign_extend)
{
  if (!pic_value_is_w(dst)) {
    variable_sz_t ipos;
    variable_sz_t ii;

    ipos = 0;
    if (value_is_indirect(dst)) {
      value_t accum;

      accum = pic_var_accum_get(pf);
      pic_instr_append_f(pf, pic_opcode_movwf, accum, 0);
      pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE, 0, &ipos);
      pic_instr_append_f_d(pf, pic_opcode_movf, accum, 0, pic_opdst_w);
      pic_var_accum_release(pf, accum);
    }
    for (ii = 0; ii < value_sz_get(dst); ii++) {
      pic_indirect_bump3(pf, dst, VALUE_NONE, VALUE_NONE, ii, &ipos);
      if ((0 == ii) || sign_extend) {
        pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
      } else {
        pic_instr_append_f(pf, pic_opcode_clrf, dst, ii);
      }
    }
  }
}



/* much simpler than relational -- need only compare bytes;
 * nb : this is called for both operator_eq & operator_ne
 * if dst is VALUE_NONE, the caller must check _Z and act
 * appropriately.
 */
static void pic_equality(pfile_t *pf, operator_t op, value_t dst,
  value_t val1, value_t val2)
{
  assert(!pic_value_is_w(val1));
  assert(!pic_value_is_w(val2));

  if (value_is_same(val1, val2)) {
    value_t          c;

    c = pfile_constant_get(pf, 
        (operator_eq == op) ? 1 : 0, VARIABLE_DEF_NONE);
    pic_op(pf, operator_assign, dst, c, VALUE_NONE);
    value_release(c);
  } else if (!pic_work_in_temp(pf, op, dst, val1, val2,
        PIC_WORK_IN_TEMP_FLAG_VALS
        | (value_is_single_bit(dst)
          ? 0
          : PIC_WORK_IN_TEMP_FLAG_DST))) {
    value_t       tmp; /* for multi-byte operations */
    value_t       val1_sign;
    value_t       val2_sign;
    variable_sz_t sz;
    variable_sz_t ii;
    variable_sz_t ipos;

    sz = pic_result_sz_get(val1, val2, VALUE_NONE);
    val1_sign = VALUE_NONE;
    val2_sign = VALUE_NONE;
    tmp       = VALUE_NONE;
    if (sz > 1) {
      tmp = pic_var_temp_get(pf, VARIABLE_DEF_FLAG_NONE, 1);
    }
    pic_indirect_setup3(pf, val1, val2, VALUE_NONE, 0, &ipos);
    for (ii = 0; ii < sz; ii++) {
      if (!value_is_const(val1) && (ii == value_sz_get(val1))) {
        val1_sign = pic_value_sign_get(pf, val1, VALUE_NONE);
        val1      = val1_sign;
      } else if (!value_is_const(val2) && (ii == value_sz_get(val2))) {
        val2_sign = pic_value_sign_get(pf, val2, VALUE_NONE);
        val2      = val2_sign;
      }
      pic_indirect_bump3(pf, val1, val2, VALUE_NONE, ii, &ipos);

      if (value_is_const(val2)) {
        assert(!value_is_const(val1));
        SWAP(value_t, val1, val2);
      }
      if (value_is_const(val1)) {
        pic_instr_append_w_kn(pf, pic_opcode_movlw, 
            pic_value_const_byte_get(val1, ii));
      } else if (VALUE_NONE != val1) {
        pic_instr_append_f_d(pf, pic_opcode_movf, val1, 
            (val1 == val1_sign) ? 0 : ii, pic_opdst_w);
      }
      if (value_is_const(val2)) {
        pic_instr_append_w_kn(pf, pic_opcode_sublw,
            pic_value_const_byte_get(val2, ii));
      } else if (VALUE_NONE != val2) {
        pic_instr_append_f_d(pf, 
            (VALUE_NONE == val1) ? pic_opcode_movf : pic_opcode_subwf, 
            val2, (val2 == val2_sign) ? 0 : ii, pic_opdst_w);
      }
      if (VALUE_NONE != tmp) {
        if (ii) {
          pic_instr_append_f_d(pf,
              pic_opcode_iorwf, tmp, 0, 
              (ii + 1 == sz) ? pic_opdst_w : pic_opdst_f);
        } else {
          pic_instr_append_f(pf, pic_opcode_movwf, tmp, 0);
        }
      }
    }
    /* _status:_z is set for equal (w = 0), clear for not (w != 0) */
    if ((VALUE_NONE != dst) 
        && !value_is_single_bit(dst)) {
      pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_z");
      pic_instr_append_w_kn(pf, pic_opcode_movlw, 1);
      if (operator_eq == op) {
        pic_instr_append_w_kn(pf, pic_opcode_xorlw, 1);
      }
    }
    if (VALUE_NONE != val2_sign) {
      pic_var_accum_release(pf, val2_sign);
    }
    if (VALUE_NONE != val1_sign) {
      pic_var_accum_release(pf, val1_sign);
    }
    if (VALUE_NONE != tmp) {
      pic_var_temp_release(pf, tmp);
    }
    if (value_is_indirect(dst)) {
      /* must sock W away */
      tmp = pic_var_temp_get(pf, VARIABLE_DEF_FLAG_NONE, 1);
      pic_instr_append_f(pf, pic_opcode_movwf, tmp, 0);
      pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE, 0, &ipos);
      pic_instr_append_f_d(pf, pic_opcode_movf, tmp, 0, pic_opdst_w);
      pic_var_temp_release(pf, tmp);
    }
    if (value_is_single_bit(dst)) {
      pic_opcode_t skip_op;

      skip_op = (operator_eq == op) 
        ? pic_opcode_btfsc 
        : pic_opcode_btfss;
      if (value_is_volatile(dst)) {
        label_t lbl_set;
        label_t lbl_done;

        lbl_set = pfile_label_alloc(pf, 0);
        lbl_done = pfile_label_alloc(pf, 0);
        pic_instr_append_reg_flag(pf, skip_op, "_status", "_z");
        pic_instr_append_n(pf, pic_opcode_goto, lbl_set);
        pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
        pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
        pic_instr_append_label(pf, lbl_set);
        pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
        pic_instr_append_label(pf, lbl_done);
        label_release(lbl_done);
        label_release(lbl_set);
      } else {
        pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
        pic_instr_append_reg_flag(pf, skip_op, "_status", "_z");
        pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
      }
    } else if (!pic_value_is_w(dst)) {
      pic_assign_from_w(pf, dst, boolean_false);
    }
  }
}

/* this only handles <, <=, >=, >; the rest are handled by pic_equality
 * on exit, if dst is VALUE_NONE, _C and/or _Z must be tested by the caller 
 * also note the operator might change.
 */
operator_t pic_relational(pfile_t *pf, operator_t op, value_t dst,
  value_t val1, value_t val2)
{
  assert(!pic_value_is_w(val1));
  assert(!pic_value_is_w(val2));

  if ((operator_eq == op) || (operator_ne == op)) {
    pic_equality(pf, op, dst, val1, val2);
  } else if (value_is_same(val1, val2)) {
    value_t          c;

    c = pfile_constant_get(pf,
          ((operator_le == op) || (operator_ge == op)) ? 1 : 0,
          VARIABLE_DEF_NONE);
    pic_op(pf, operator_assign, dst, c, VALUE_NONE);
    value_release(c);
  } else if (!pic_work_in_temp(pf, op, dst, val1, val2,
        PIC_WORK_IN_TEMP_FLAG_VALS
        | (value_is_single_bit(dst)
          ? 0
          : PIC_WORK_IN_TEMP_FLAG_DST))) {
    /* unlike equality, relationals must work from the highest
     * part of the # to the lowest */
    variable_sz_t sz;
    variable_sz_t ipos;
    variable_sz_t ii;
    value_t       val1_sign;
    value_t       val2_sign;
    label_t       lbl_done;
    boolean_t     is_first;

    if (value_is_const(val2)) {
      /* val1 *must* be the const, so swap these & fixup the op */
      SWAP(value_t, val1, val2);
      switch (op) {
        case operator_lt: op = operator_gt; break;
        case operator_le: op = operator_ge; break;
        case operator_ge: op = operator_le; break;
        case operator_gt: op = operator_lt; break;
        default: abort();
      }
    }
    val1_sign = VALUE_NONE;
    val2_sign = VALUE_NONE;
    is_first  = boolean_true;

    lbl_done = pfile_label_alloc(pf, 0);
    sz = pic_result_sz_get(val1, val2, VALUE_NONE);
    pic_indirect_setup3(pf, val1, val2, VALUE_NONE, sz - 1, &ipos);
    ii = sz;
    if (!value_is_const(val1) && (ii > value_sz_get(val1))) {
      val1_sign = pic_value_sign_get(pf, val1, VALUE_NONE);
      if (VALUE_NONE == val1_sign) {
        val1_sign = pic_var_accum_get(pf);
        pic_instr_append_f(pf, pic_opcode_clrf, val1_sign, 0);
      }
    }
    if (!value_is_const(val2) && (ii > value_sz_get(val2))) {
      val2_sign = pic_value_sign_get(pf, val2, VALUE_NONE);
      if (VALUE_NONE == val2_sign) {
        val2_sign = pic_var_accum_get(pf);
        pic_instr_append_f(pf, pic_opcode_clrf, val2_sign, 0);
      }
    }
    if (pic_result_is_signed(pf, val1, val2)) {
      value_t tmp;

      tmp = pic_var_temp_get(pf, VARIABLE_DEF_FLAG_NONE, 1);
      pic_value_to_w(pf, PIC_VALUE_TO_W_MOV, val2, val2_sign, ii - 1);
      pic_instr_append_w_kn(pf, pic_opcode_xorlw, 0x80);
      pic_instr_append_f(pf, pic_opcode_movwf, tmp, 0);
      pic_value_to_w(pf, PIC_VALUE_TO_W_MOV, val1, val1_sign, ii - 1);
      pic_instr_append_w_kn(pf, pic_opcode_xorlw, 0x80);
      pic_instr_append_f_d(pf, pic_opcode_subwf, tmp, 0, pic_opdst_w);
      pic_var_temp_release(pf, tmp);
      is_first = boolean_false;
      ii--;
    }
    while (ii--) {
      if (!is_first) {
        pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_z");
        pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
      }
      is_first = boolean_false;
      pic_indirect_bump3(pf, val1, val2, VALUE_NONE, ii, &ipos);
      pic_value_to_w(pf, PIC_VALUE_TO_W_MOV, val1, val1_sign, ii);
      pic_value_to_w(pf, 
          (value_is_const(val2) ? PIC_VALUE_TO_W_ADD : PIC_VALUE_TO_W_SUB), 
           val2, val2_sign, ii);
    }
    pic_instr_append_label(pf, lbl_done);

    /* _c & !_z : val1 < val2  */
    /* _c & _z  : val1 <= val2 */
    /* !_c & _z : val1 >= val2 */
    /* !_c & !_z: val1 >  val2 */
    if (VALUE_NONE != dst) {
      label_t lbl_skip;

      lbl_skip = LABEL_NONE;
      if ((operator_lt == op) || (operator_gt == op)) {
        lbl_skip = pfile_label_alloc(pf, 0);
      }

      if (value_is_single_bit(dst)) {
        pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
      } else {
        pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
      }
      switch (op) {
        case operator_lt:
          pic_instr_append_reg_flag(pf, pic_opcode_btfsc, "_status", "_z");
          pic_instr_append_n(pf, pic_opcode_goto, lbl_skip);
          pic_instr_append_reg_flag(pf, pic_opcode_btfsc, "_status", "_c");
          break;
        case operator_le:
          pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_z");
          pic_instr_append_reg_flag(pf, pic_opcode_btfsc, "_status", "_c");
          break;
        case operator_ge:
          pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_z");
          pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_c");
          break;
        case operator_gt:
          pic_instr_append_reg_flag(pf, pic_opcode_btfsc, "_status", "_z");
          pic_instr_append_n(pf, pic_opcode_goto, lbl_skip);
          pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_c");
          break;
        default:
          abort();
      }
      if (value_is_single_bit(dst)) {
        pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
      } else {
        pic_instr_append_w_kn(pf, pic_opcode_movlw, 1);
      }
      if (LABEL_NONE != lbl_skip) {
        pic_instr_append_label(pf, lbl_skip);
      }
      label_release(lbl_skip);
    }
    if (VALUE_NONE != val2_sign) {
      pic_var_accum_release(pf, val2_sign);
    }
    if (VALUE_NONE != val1_sign) {
      pic_var_accum_release(pf, val1_sign);
    }
    label_release(lbl_done);
    if ((VALUE_NONE != dst) && !value_is_single_bit(dst)) {
      pic_assign_from_w(pf, dst, boolean_false);
    }
  }
  return op;
}

static void pic_op_relational(pfile_t *pf, operator_t op, value_t dst,
  value_t val1, value_t val2)
{
  pic_relational(pf, op, dst, val1, val2);
}

/* if dst is VALUE_NONE, the caller is responsible for checking _Z
 * _Z is set if val1 && val2 == 0, clear if not
 */
static void pic_op_logical_and(pfile_t *pf, operator_t op, value_t dst,
  value_t val1, value_t val2)
{
  assert(!pic_value_is_w(val1));
  assert(!pic_value_is_w(val2));

  /* make sure dst/val1 are the same */
  if (value_is_same(dst, val2)) {
    SWAP(value_t, val1, val2);
  }

  if (value_is_same(val1, val2)) {
    pic_op(pf, operator_logical, dst, val1, VALUE_NONE);
  } else if (value_is_const(val1) || value_is_const(val2)) {
    if (value_is_const(val2)) {
      SWAP(value_t, val1, val2);
    }
    /* val1 is const */
    if (value_const_get(val1)) {
      pic_op(pf, operator_logical, dst, val2, VALUE_NONE);
    } else {
      pic_assign_logical_to_dst(pf, dst, 0, PIC_ASSIGN_LOGICAL_TO_DST_OP);
    }
  } else if (value_is_single_bit(val1) && value_is_single_bit(val2)) {
    if (pic_val_bank_same(pf, val1, val2) && !value_is_same(dst, val1)) {
      if (!value_is_single_bit(dst)
          || (value_is_single_bit(dst)
            && !value_is_volatile(dst)
            && pic_val_bank_same(pf, dst, val1))) {
        pic_assign_logical_to_dst(pf, dst, 1, PIC_ASSIGN_LOGICAL_TO_DST_NONE);
        pic_instr_append_f(pf, pic_opcode_btfsc, val1, 0);
        pic_instr_append_f(pf, pic_opcode_btfss, val2, 0);
        pic_assign_logical_to_dst(pf, dst, 0, PIC_ASSIGN_LOGICAL_TO_DST_OP);
      } else {
        label_t lbl_done;
        label_t lbl_clr;

        lbl_done = pfile_label_alloc(pf, 0);
        lbl_clr  = pfile_label_alloc(pf, 0);

        pic_instr_append_f(pf, pic_opcode_btfss, val1, 0);
        pic_instr_append_n(pf, pic_opcode_goto, lbl_clr);
        pic_instr_append_f(pf, pic_opcode_btfss, val2, 0);
        pic_instr_append_n(pf, pic_opcode_goto, lbl_clr);
        pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
        pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
        pic_instr_append_label(pf, lbl_clr);
        pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
        pic_instr_append_label(pf, lbl_done);

        label_release(lbl_clr);
        label_release(lbl_done);
      }
    } else {
      label_t lbl_done;
      label_t lbl_clr;

      lbl_done = pfile_label_alloc(pf, 0);
      lbl_clr  = pfile_label_alloc(pf, 0);

      pic_instr_append_f(pf, pic_opcode_btfss, val1, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_clr);
      pic_instr_append_f(pf, pic_opcode_btfss, val2, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_clr);
      pic_assign_logical_to_dst(pf, dst, 1, PIC_ASSIGN_LOGICAL_TO_DST_NONE);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
      pic_instr_append_label(pf, lbl_clr);
      pic_assign_logical_to_dst(pf, dst, 0, PIC_ASSIGN_LOGICAL_TO_DST_NONE);
      pic_instr_append_label(pf, lbl_done);
      if (!value_is_single_bit(dst)) {
        pic_assign_from_w(pf, dst, boolean_false);
      }
      label_release(lbl_clr);
      label_release(lbl_done);
    }
  } else if (value_is_single_bit(val1) || value_is_single_bit(val2)) {
    /* do some work */
    if (value_is_single_bit(val2)) {
      SWAP(value_t, val1, val2);
    }
    pic_value_is_zero(pf, val2);
    /* W is 0 or 1 */
    if (value_is_single_bit(dst)
        && (value_is_volatile(dst) 
          || !pic_val_bank_same(pf, dst, val1)
          || value_is_same(dst, val1)
          || value_is_same(dst, val2))) {
      label_t lbl_clr;
      label_t lbl_done;

      lbl_done = pfile_label_alloc(pf, 0);
      lbl_clr  = pfile_label_alloc(pf, 0);
      pic_instr_append_reg_flag(pf, pic_opcode_btfsc, "_status", "_z");
      pic_instr_append_n(pf, pic_opcode_goto, lbl_clr);
      pic_instr_append_f(pf, pic_opcode_btfss, val1, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_clr);
      pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
      pic_instr_append_label(pf, lbl_clr);
      pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
      pic_instr_append_label(pf, lbl_done);
      label_release(lbl_clr);
      label_release(lbl_done);
    } else {
      pic_assign_logical_to_dst(pf, dst, 1, PIC_ASSIGN_LOGICAL_TO_DST_NONE);
      pic_instr_append_daop(pf, val1, pic_opcode_btfss);
      pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_z");
      pic_instr_append_f(pf, pic_opcode_btfss, val1, 0);
      pic_assign_logical_to_dst(pf, dst, 0, PIC_ASSIGN_LOGICAL_TO_DST_NONE);
      if (!value_is_single_bit(dst)) {
        pic_assign_from_w(pf, dst, boolean_false);
      }
    }
  } else {
    label_t lbl_done;
    label_t lbl_clr;

    lbl_done = pfile_label_alloc(pf, 0);
    lbl_clr  = pfile_label_alloc(pf, 0);

    pic_value_is_zero(pf, val1);
    pic_instr_append_reg_flag(pf, pic_opcode_btfsc, "_status", "_z");
    pic_instr_append_n(pf, pic_opcode_goto, lbl_clr);
    pic_value_is_zero(pf, val2);
    pic_instr_append_reg_flag(pf, pic_opcode_btfsc, "_status", "_z");
    pic_instr_append_n(pf, pic_opcode_goto, lbl_clr);
    pic_assign_logical_to_dst(pf, dst, 1, PIC_ASSIGN_LOGICAL_TO_DST_NONE);
    pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
    pic_instr_append_label(pf, lbl_clr);
    pic_assign_logical_to_dst(pf, dst, 0, PIC_ASSIGN_LOGICAL_TO_DST_NONE);
    pic_instr_append_label(pf, lbl_done);
    if (!value_is_single_bit(dst)) {
      pic_assign_from_w(pf, dst, boolean_false);
    }
    label_release(lbl_clr);
    label_release(lbl_done);
  }
}

/* if dst is VALUE_NONE, _Z is set if A || B is true, clear if false */
static void pic_op_logical_or(pfile_t *pf, operator_t op, value_t dst,
  value_t val1, value_t val2)
{
  assert(!pic_value_is_w(val1));
  assert(!pic_value_is_w(val2));

  if (value_is_same(val1, val2)) {
    pic_op(pf, operator_logical, dst, val1, VALUE_NONE);
  } else if (value_is_const(val1) || value_is_const(val2)) {
    if (value_is_const(val2)) {
      SWAP(value_t, val1, val2);
    }
    /* val1 is const */
    if (value_const_get(val1)) {
      pic_assign_logical_to_dst(pf, dst, 1, PIC_ASSIGN_LOGICAL_TO_DST_OP);
    } else {
      pic_op(pf, operator_logical, dst, val2, VALUE_NONE);
    }
  } else if (value_is_single_bit(val1) && value_is_single_bit(val2)) {
    if (pic_val_bank_same(pf, val1, val2)) {
      if (!value_is_single_bit(dst)
          || (value_is_single_bit(dst)
            && !value_is_volatile(dst)
            && !value_is_same(dst, val1)
            && !value_is_same(dst, val2)
            && pic_val_bank_same(pf, dst, val1))) {
        pic_assign_logical_to_dst(pf, dst, 0, PIC_ASSIGN_LOGICAL_TO_DST_NONE);
        pic_instr_append_f(pf, pic_opcode_btfss, val1, 0);
        pic_instr_append_f(pf, pic_opcode_btfsc, val2, 0);
        pic_assign_logical_to_dst(pf, dst, 1, PIC_ASSIGN_LOGICAL_TO_DST_OP);
      } else {
        label_t lbl_done;
        label_t lbl_set;

        lbl_done = pfile_label_alloc(pf, 0);
        lbl_set  = pfile_label_alloc(pf, 0);

        pic_instr_append_f(pf, pic_opcode_btfsc, val1, 0);
        pic_instr_append_n(pf, pic_opcode_goto, lbl_set);
        pic_instr_append_f(pf, pic_opcode_btfsc, val2, 0);
        pic_instr_append_n(pf, pic_opcode_goto, lbl_set);
        pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
        pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
        pic_instr_append_label(pf, lbl_set);
        pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
        pic_instr_append_label(pf, lbl_done);

        label_release(lbl_set);
        label_release(lbl_done);
      }
    } else {
      label_t lbl_done;
      label_t lbl_set;

      lbl_done = pfile_label_alloc(pf, 0);
      lbl_set  = pfile_label_alloc(pf, 0);

      pic_instr_append_f(pf, pic_opcode_btfsc, val1, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_set);
      pic_instr_append_f(pf, pic_opcode_btfsc, val2, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_set);
      pic_assign_logical_to_dst(pf, dst, 0, PIC_ASSIGN_LOGICAL_TO_DST_NONE);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
      pic_instr_append_label(pf, lbl_set);
      pic_assign_logical_to_dst(pf, dst, 1, PIC_ASSIGN_LOGICAL_TO_DST_NONE);
      pic_instr_append_label(pf, lbl_done);
      if (!value_is_single_bit(dst)) {
        pic_assign_from_w(pf, dst, boolean_false);
      }
      label_release(lbl_set);
      label_release(lbl_done);
    }
  } else if (value_is_single_bit(val1) || value_is_single_bit(val2)) {
    /* do some work */
    if (value_is_single_bit(val2)) {
      SWAP(value_t, val1, val2);
    }
    pic_value_is_zero(pf, val2);
    /* W is 0 or 1 */
    if (value_is_single_bit(dst)
        && (value_is_volatile(dst) 
          || !pic_val_bank_same(pf, dst, val1)
          || value_is_same(dst, val1)
          || value_is_same(dst, val2))) {
      label_t lbl_set;
      label_t lbl_done;

      lbl_done = pfile_label_alloc(pf, 0);
      lbl_set  = pfile_label_alloc(pf, 0);
      pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_z");
      pic_instr_append_n(pf, pic_opcode_goto, lbl_set);
      pic_instr_append_f(pf, pic_opcode_btfsc, val1, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_set);
      pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
      pic_instr_append_label(pf, lbl_set);
      pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
      pic_instr_append_label(pf, lbl_done);
      label_release(lbl_set);
      label_release(lbl_done);
    } else {
      pic_assign_logical_to_dst(pf, dst, 0, PIC_ASSIGN_LOGICAL_TO_DST_NONE);
      pic_instr_append_daop(pf, val1, pic_opcode_btfss);
      pic_instr_append_reg_flag(pf, pic_opcode_btfsc, "_status", "_z");
      pic_instr_append_f(pf, pic_opcode_btfsc, val1, 0);
      pic_assign_logical_to_dst(pf, dst, 1, PIC_ASSIGN_LOGICAL_TO_DST_NONE);
      if (!value_is_single_bit(dst)) {
        pic_assign_from_w(pf, dst, boolean_false);
      }
    }
  } else {
    label_t lbl_done;
    label_t lbl_set;

    lbl_done = pfile_label_alloc(pf, 0);
    lbl_set  = pfile_label_alloc(pf, 0);

    pic_value_is_zero(pf, val1);
    pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_z");
    pic_instr_append_n(pf, pic_opcode_goto, lbl_set);
    pic_value_is_zero(pf, val2);
    pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_z");
    pic_instr_append_n(pf, pic_opcode_goto, lbl_set);
    pic_assign_logical_to_dst(pf, dst, 0, PIC_ASSIGN_LOGICAL_TO_DST_NONE);
    pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
    pic_instr_append_label(pf, lbl_set);
    pic_assign_logical_to_dst(pf, dst, 1, PIC_ASSIGN_LOGICAL_TO_DST_NONE);
    pic_instr_append_label(pf, lbl_done);
    if (!value_is_single_bit(dst)) {
      pic_assign_from_w(pf, dst, boolean_false);
    }
    label_release(lbl_set);
    label_release(lbl_done);
  }
}

/* perform an AND, OR, or XOR */
static void pic_binary_and_or_xor(pfile_t *pf, operator_t op,
    value_t dst, value_t val1, value_t val2, pic_opcode_t pop)
{
  if (value_is_const(val1) || value_is_const(val2)) {
    variable_const_t cval;

    if (value_is_const(val1)) {
      SWAP(value_t, val1, val2);
    }
    cval = value_const_get(val2);
    if (!cval) {
      if (operator_andb == op) {
        /* anything AND 0 is 0, so set dst to 0 */
        /* pic_instr_append(pf, pic_opcode_clrw); */
        pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
        pic_op(pf, operator_assign, dst, VALUE_NONE, VALUE_NONE);
      } else {
        /* anything XOR or OR 0 is unchanged */
        pic_unary_fixup(pf, operator_assign, dst, val1, val2);
      }
    } else if ((cval == -1) && (operator_andb == op)) {
      /* anything AND all bits is unchanged */
      pic_unary_fixup(pf, operator_assign, dst, val1, val2);
    } else if (value_is_same(dst, val1)
      && (((operator_andb == op) && ((~cval & (~cval - 1)) == 0))
        || ((operator_andb != op) && ((cval & (cval - 1)) == 0)))) {
      /* single bit set, clear, or flip */
      unsigned bit;

      if (operator_andb == op) {
        cval = ~cval;
      }

      for (bit = -1; cval; cval >>= 1, bit++)
        ;
      if (value_is_bit(dst)) {
        if (bit < value_sz_get(dst)) {
          bit += value_bit_offset_get(dst);
        } else {
          dst = VALUE_NONE;
        }
      } else if (bit >= 8 * value_sz_get(dst)) {
        dst = VALUE_NONE;
      }
      if (VALUE_NONE != dst) {
        variable_sz_t ipos;

        pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE,
            bit / 8, &ipos);
        if (operator_andb == op) {
          pic_instr_append_f_bn(pf, pic_opcode_bcf, dst, bit / 8, bit & 7);
        } else if (operator_orb == op) {
          pic_instr_append_f_bn(pf, pic_opcode_bsf, dst, bit / 8, bit & 7);
        } else if (operator_xorb == op) {
          if (value_is_volatile(dst) || value_is_same(dst, val1)) {
            label_t lbl_done;
            label_t lbl_set;

            lbl_done = pfile_label_alloc(pf, 0);
            lbl_set  = pfile_label_alloc(pf, 0);
            pic_instr_append_f_bn(pf, pic_opcode_btfss, val1, bit / 8, bit & 7);
            pic_instr_append_n(pf, pic_opcode_goto, lbl_set);
            pic_instr_append_f_bn(pf, pic_opcode_bcf, dst, bit / 8, bit & 7);
            pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
            pic_instr_append_label(pf, lbl_set);
            pic_instr_append_f_bn(pf, pic_opcode_bsf, dst, bit / 8, bit & 7);
            pic_instr_append_label(pf, lbl_done);
            label_release(lbl_set);
            label_release(lbl_done);
          } else {
            pic_instr_append_f_bn(pf, pic_opcode_btfsc, dst, bit / 8, bit & 7);
            pic_instr_append_f_bn(pf, pic_opcode_bcf,   dst, bit / 8, bit & 7);
            pic_instr_append_f_bn(pf, pic_opcode_btfss, dst, bit / 8, bit & 7);
            pic_instr_append_f_bn(pf, pic_opcode_bsf,   dst, bit / 8, bit & 7);
          }
        }
      }
    } else if (!pic_work_in_temp(pf, op, dst, val1, val2,
          PIC_WORK_IN_TEMP_FLAG_VALS_DST)) {
      /* dst = val1 op c */
      variable_sz_t sz;
      variable_sz_t ii;
      variable_sz_t ipos;
      value_t       sign;     /* for sign extension */
      variable_sz_t val1_ofs; /* index into val1    */
      uchar         ch_last;

      sign    = VALUE_NONE;
      ch_last = 0; /* prevent a compiler warning */

      sz = pic_result_sz_get(val1, val2, dst);
      pic_indirect_setup3(pf, dst, val1, VALUE_NONE, 0, &ipos);
      for (ii = 0, val1_ofs = 0; ii < sz; ii++) {
        uchar ch;

        if (ii) {
          pic_indirect_bump3(pf, dst, val1, val2, ii, &ipos);
          if (!sign && (++val1_ofs == value_sz_get(val1))) {
            sign     = pic_value_sign_get(pf, val1, VALUE_NONE);
            val1     = sign;
            val1_ofs = 0;
          }
          cval >>= 8;
        }
        ch = cval & 0xff;
        /* 0 && x --> 0; just clear dst */
        if (((0 == ch) || (VALUE_NONE == val1)) && (operator_andb == op)) {
          pic_instr_append_f(pf, pic_opcode_clrf, dst, ii);
        } else {
          pic_opcode_t opcode;

          if (((0xff == ch) && (operator_andb == op))
            || (((0x00 == ch) || (VALUE_NONE == val1))
              && ((operator_orb == op) || (operator_xorb == op)))) {
            /* no action: x & 0xff == x */
            /*            x | 0x00 == x */
            /*            x ^ 0x00 == x */
            opcode = pic_opcode_movf;
            if ((VALUE_NONE == val1) && ch) {
              pic_instr_append_w_kn(pf, pic_opcode_movlw, ch);
            }
          } else {
            /* if val1 & dst are different, W was destroyed */
            if (!value_is_same(dst, val1) || (ch != ch_last) || (0 == ii)) {
              pic_instr_append_w_kn(pf, pic_opcode_movlw, ch);
              ch_last = ch;
            }
            opcode = pop;
          }
          if (value_is_same(dst, val1)) {
            if (opcode != pic_opcode_movf) {
              pic_instr_append_f_d(pf, opcode, val1, val1_ofs, pic_opdst_f);
            }
          } else {
            if (val1 != VALUE_NONE) {
              pic_instr_append_f_d(pf, opcode, val1, val1_ofs, pic_opdst_w);
            }
            if ((VALUE_NONE == val1) && !ch) {
              pic_instr_append_f(pf, pic_opcode_clrf, dst, ii);
            } else {
              pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
            }
          }
        }
      }
      pic_value_sign_extend(pf, dst, ii - 1,
          pic_result_is_signed(pf, val1, val2));
      if (sign) {
        pic_var_accum_release(pf, sign);
      }
    }
  } else if (!pic_work_in_temp(pf, op, dst, val1, val2,
        PIC_WORK_IN_TEMP_FLAG_VALS_DST)) {
    value_t       val1_sign;
    value_t       val2_sign;
    variable_sz_t sz;
    variable_sz_t ii;
    variable_sz_t ipos;
    variable_sz_t val1_ofs;
    variable_sz_t val2_ofs;
    pic_opdst_t   opdst;

    /* if dst is the same as an argument, make sure that argument is first */
    if (value_is_same(dst, val2)) {
      SWAP(value_t, val1, val2);
    }

    opdst = (value_is_same(dst, val1))
      ? pic_opdst_f
      : pic_opdst_w;

    val1_sign = VALUE_NONE;
    val2_sign = VALUE_NONE;
    sz = pic_result_sz_get(val1, val2, dst);
    pic_indirect_setup3(pf, val1, val2, dst, 0, &ipos);
    for (ii = 0, val1_ofs = 0, val2_ofs = 0; ii < sz; ii++) {
      if (ii) {
        if ((VALUE_NONE == val1_sign) && (++val1_ofs == value_sz_get(val1))) {
          val1_sign = pic_value_sign_get(pf, val1, VALUE_NONE);
          val1      = val1_sign;
          val1_ofs  = 0;
        }
        if ((VALUE_NONE == val2_sign) && (++val2_ofs == value_sz_get(val2))) {
          val2_sign = pic_value_sign_get(pf, val2, VALUE_NONE);
          val2      = val2_sign;
          val2_ofs  = 0;
        }
        pic_indirect_bump3(pf, dst, val1, val2, ii, &ipos);
      }
      /* if we've run out of val1 or val2 and it's not sign extended
       * it will always be `and 0' which is effectively clear dst
       */
      if ((VALUE_NONE == val1) || (VALUE_NONE == val2)) {
        if (operator_andb == op) {
          pic_instr_append_f(pf, pic_opcode_clrf, dst, ii);
        } else if (pic_opdst_w == opdst) {
          if (VALUE_NONE == val1) {
            pic_instr_append_f_d(pf, pic_opcode_movf, val2, val2_ofs, opdst);
          } else {
            pic_instr_append_f_d(pf, pic_opcode_movf, val1, val1_ofs, opdst);
          }
          pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
        }
      } else {
        pic_instr_append_f_d(pf, pic_opcode_movf, val2, val2_ofs, pic_opdst_w);
        pic_instr_append_f_d(pf, pop, val1, val1_ofs, opdst);
        if (pic_opdst_w == opdst) {
          pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
        }
      }
    }
    if (VALUE_NONE != val1_sign) {
      pic_var_accum_release(pf, val1_sign);
    }
    if (VALUE_NONE != val2_sign) {
      pic_var_accum_release(pf, val2_sign);
    }
    pic_value_sign_extend(pf, dst, ii - 1, 
        pic_result_is_signed(pf, val1, val2));
  }
}

static void pic_op_binary_and(pfile_t *pf, operator_t op, value_t dst,
  value_t val1, value_t val2)
{
  assert(!pic_value_is_w(dst));
  assert(!pic_value_is_w(val1));
  assert(!pic_value_is_w(val2));

  if (value_is_same(val1, val2)) {
    pic_op(pf, operator_assign, dst, val1, VALUE_NONE);
  } else if (!value_is_const(val1)
      && !value_is_const(val2)
      && value_is_single_bit(val1) 
      && value_is_single_bit(val2)) {
    /* the quick case only works if:
     *   val1 & val2 are in the same bank
     *   dst is not bit
     *     or dst is not volatile & dst is in the same bank as val1
     */
    label_t lbl_clr;
    label_t lbl_done;
    uchar   true_value;

    true_value = 1;
    if (pic_result_is_signed(pf, val1, val2)) {
      true_value = 255;
    }

    if (value_is_same(dst, val2)) {
      SWAP(value_t, val1, val2);
    }

    lbl_done = pfile_label_alloc(pf, 0);
    lbl_clr  = pfile_label_alloc(pf, 0);
    if (pic_val_bank_same(pf, val1, val2)) {
      if (!value_is_same(dst, val1)
          && (!value_is_single_bit(dst)
            || (!value_is_volatile(dst) 
              && pic_val_bank_same(pf, dst, val1)))) {
        /* 4 instructions (+ assignment if dst is not a bit) */
        if (value_is_single_bit(dst)) {
          pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
        } else {
          pic_instr_append_w_kn(pf, pic_opcode_movlw, true_value);
        }
        pic_instr_append_f(pf, pic_opcode_btfsc, val1, 0);
        pic_instr_append_f(pf, pic_opcode_btfss, val2, 0);
        if (value_is_single_bit(dst)) {
          pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
        } else {
          /* pic_instr_append(pf, pic_opcode_clrw); */
          pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
        }
      } else if (!value_is_same(dst, val1)
          && !value_is_volatile(dst) 
          && !pic_val_bank_same(pf, dst, val1)) {
        /* 5 instructions */
        pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
        pic_instr_append_f(pf, pic_opcode_btfsc, val1, 0);
        pic_instr_append_f(pf, pic_opcode_btfss, val2, 0);
        pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
        pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
      } else {
        /* 6 instructions */
        pic_instr_append_f(pf, pic_opcode_btfsc, val1, 0);
        pic_instr_append_f(pf, pic_opcode_btfss, val2, 0);
        pic_instr_append_n(pf, pic_opcode_goto, lbl_clr);
        pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
        pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
        pic_instr_append_label(pf, lbl_clr);
        pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
      }
    } else if (!value_is_same(dst, val1)
        && (!value_is_single_bit(dst) || !value_is_volatile(dst))) {
      /* 6 instructions */
      if (value_is_single_bit(dst)) {
        pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
      } else {
        /* pic_instr_append(pf, pic_opcode_clrw); */
        pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
      }
      pic_instr_append_f(pf, pic_opcode_btfss, val1, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
      pic_instr_append_f(pf, pic_opcode_btfss, val2, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
      if (value_is_single_bit(dst)) {
        pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
      } else {
        pic_instr_append_w_kn(pf, pic_opcode_movlw, true_value);
      }
    } else {
      /* 7 instructions */
      pic_instr_append_f(pf, pic_opcode_btfss, val1, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_clr);
      pic_instr_append_f(pf, pic_opcode_btfss, val2, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_clr);
      pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
      pic_instr_append_label(pf, lbl_clr);
      pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
    }
    pic_instr_append_label(pf, lbl_done);
    if (!value_is_single_bit(dst)) {
      pic_assign_from_w(pf, dst, 255 == true_value);
    }
    label_release(lbl_clr);
    label_release(lbl_done);
  } else {
    pic_binary_and_or_xor(pf, op, dst, val1, val2, pic_opcode_andwf);
  } 
}

static void pic_op_binary_or(pfile_t *pf, operator_t op, value_t dst,
  value_t val1, value_t val2)
{
  assert(!pic_value_is_w(dst));
  assert(!pic_value_is_w(val1));
  assert(!pic_value_is_w(val2));

  if (value_is_same(val1, val2)) {
    pic_op(pf, operator_assign, dst, val1, VALUE_NONE);
  } else if (!value_is_const(val1)
    && !value_is_const(val2)
    && value_is_single_bit(val1) 
    && value_is_single_bit(val2)
    && !(value_is_signed(val1) ^ value_is_signed(val2))) {
    label_t lbl_done;
    label_t lbl_set;
    uchar   true_value;

    true_value = 1;

    if (value_is_signed(val1) || value_is_signed(val2)) {
      true_value = 255;
    }

    lbl_done = pfile_label_alloc(pf, 0);
    lbl_set  = pfile_label_alloc(pf, 0);

    /* make sure (dst, val1) are same if any */
    if (value_is_same(dst, val2)) {
      SWAP(value_t, val1, val2);
    }
    if (pic_val_bank_same(pf, val1, val2)) {
      if (!value_is_same(dst, val1)
        && (!value_is_single_bit(dst)
            || (!value_is_volatile(dst) 
              && pic_val_bank_same(pf, dst, val1)))) {
        /* 4 instructions (+ assignment if dst is not a bit) */
        if (value_is_single_bit(dst)) {
          pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
        } else {
          /* pic_instr_append(pf, pic_opcode_clrw); */
          pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
        }
        pic_instr_append_f(pf, pic_opcode_btfss, val1, 0);
        pic_instr_append_f(pf, pic_opcode_btfsc, val2, 0);
        if (value_is_single_bit(dst)) {
          pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
        } else {
          pic_instr_append_w_kn(pf, pic_opcode_movlw, true_value);
        }
      } else if (!value_is_same(dst, val1)
        && !value_is_volatile(dst) 
        && !pic_val_bank_same(pf, dst, val1)) {
        /* 5 instructions */
        pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
        pic_instr_append_f(pf, pic_opcode_btfss, val1, 0);
        pic_instr_append_f(pf, pic_opcode_btfsc, val2, 0);
        pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
        pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
      } else {
        /* 6 instructions */
        pic_instr_append_f(pf, pic_opcode_btfss, val1, 0);
        pic_instr_append_f(pf, pic_opcode_btfsc, val2, 0);
        pic_instr_append_n(pf, pic_opcode_goto, lbl_set);
        pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
        pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
        pic_instr_append_label(pf, lbl_set);
        pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
      }
    } else if (!value_is_same(dst, val1)
      && (!value_is_single_bit(dst) || !value_is_volatile(dst))) {
      /* 6 instructions */
      if (value_is_single_bit(dst)) {
        pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
      } else {
        pic_instr_append_w_kn(pf, pic_opcode_movlw, true_value);
      }
      pic_instr_append_f(pf, pic_opcode_btfsc, val1, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
      pic_instr_append_f(pf, pic_opcode_btfsc, val2, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
      if (value_is_single_bit(dst)) {
        pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
      } else {
        /* pic_instr_append(pf, pic_opcode_clrw); */
        pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
      }
    } else {
      /* 7 instructions */
      pic_instr_append_f(pf, pic_opcode_btfsc, val1, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_set);
      pic_instr_append_f(pf, pic_opcode_btfsc, val2, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_set);
      pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
      pic_instr_append_label(pf, lbl_set);
      pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
    }
    pic_instr_append_label(pf, lbl_done);
    if (!value_is_single_bit(dst)) {
      pic_assign_from_w(pf, dst, 255 == true_value);
    }
    label_release(lbl_set);
    label_release(lbl_done);
  } else {
    pic_binary_and_or_xor(pf, op, dst, val1, val2, pic_opcode_iorwf);
  }
}

static void pic_op_binary_xor(pfile_t *pf, operator_t op, value_t dst,
  value_t val1, value_t val2)
{
  assert(!pic_value_is_w(dst));
  assert(!pic_value_is_w(val1));
  assert(!pic_value_is_w(val2));

  if (value_is_same(val1, val2)) {
    /* a ^ a = 0 */
    variable_sz_t ii;
    variable_sz_t ipos;

    pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE, 0, &ipos);
    for (ii = 0; ii < value_sz_get(dst); ii++) {
      pic_indirect_bump3(pf, dst, VALUE_NONE, VALUE_NONE, ii, &ipos);
      pic_instr_append_f(pf, pic_opcode_clrf, dst, ii);
    }
  } else if (!value_is_const(val1)
      && !value_is_const(val2)
      && value_is_single_bit(val1) 
      && value_is_single_bit(val2)
      && !(value_is_signed(val1) ^ value_is_signed(val2))) {
    label_t lbl_done;
    label_t lbl_skip;
    label_t lbl_set;
    label_t lbl_clr;
    uchar   true_value; /* value for TRUE */

    true_value = 1;
    if (value_is_signed(val1) || value_is_signed(val2)) {
      true_value = 255;
    }

    lbl_done = pfile_label_alloc(pf, 0);
    lbl_skip = pfile_label_alloc(pf, 0);
    lbl_set  = pfile_label_alloc(pf, 0);
    lbl_clr  = pfile_label_alloc(pf, 0);

    if (value_is_same(dst, val2)) {
      SWAP(value_t, val1, val2);
    }

    if (!value_is_single_bit(dst)
        || (!value_is_volatile(dst)
          && !value_is_same(dst, val1)
          && pic_val_bank_same(pf, dst, val2))) {
      if (value_is_single_bit(dst)) {
        pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
      } else {
        /* pic_instr_append(pf, pic_opcode_clrw); */
        pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
      }
      pic_instr_append_f(pf, pic_opcode_btfsc, val1, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_skip);
      pic_instr_append_f(pf, pic_opcode_btfsc, val2, 0);
      if (value_is_single_bit(dst)) {
        pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
      } else {
        pic_instr_append_w_kn(pf, pic_opcode_movlw, true_value);
      }
      pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
      pic_instr_append_label(pf, lbl_skip);
      pic_instr_append_f(pf, pic_opcode_btfss, val2, 0);
      if (value_is_single_bit(dst)) {
        pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
      } else {
        pic_instr_append_w_kn(pf, pic_opcode_movlw, true_value);
      }
    } else {
      pic_instr_append_f(pf, pic_opcode_btfsc, val1, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_skip);
      pic_instr_append_f(pf, pic_opcode_btfsc, val2, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_set);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_clr);
      pic_instr_append_label(pf, lbl_skip);
      pic_instr_append_f(pf, pic_opcode_btfss, val2, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_set);
      pic_instr_append_label(pf, lbl_clr);
      pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
      pic_instr_append_label(pf, lbl_set);
      pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
    }
    pic_instr_append_label(pf, lbl_done);
    if (!value_is_single_bit(dst)) {
      pic_assign_from_w(pf, dst, 255 == true_value);
    }
    label_release(lbl_clr);
    label_release(lbl_set);
    label_release(lbl_skip);
    label_release(lbl_done);
  } else {
    pic_binary_and_or_xor(pf, op, dst, val1, val2, pic_opcode_xorwf);
  }
}

static void pic_op_cmp(pfile_t *pf, operator_t op, value_t dst,
  value_t src, value_t val2)
{
  assert(!pic_value_is_w(dst));
  assert(!pic_value_is_w(src));

  if (value_is_single_bit(dst) && value_is_single_bit(src)) {
    if (value_is_boolean(dst) 
        && !value_is_signed(src)
        && !value_is_boolean(src)) {
      /* if src is 0, dst is clearly 1; if src is 1,
       * the inverse is -2 */
      pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
    } else if (value_is_volatile(src)
        || value_is_same(dst, src)
        || !pic_val_bank_same(pf, dst, src)) {
      label_t lbl_done;
      label_t lbl_set;

      lbl_done = pfile_label_alloc(pf, 0);
      lbl_set  = pfile_label_alloc(pf, 0);
      pic_instr_append_f(pf, pic_opcode_btfss, src, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_set);
      pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
      pic_instr_append_label(pf, lbl_set);
      pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
      pic_instr_append_label(pf, lbl_done);
      label_release(lbl_set);
      label_release(lbl_done);
    } else {
      pic_instr_append_f(pf, pic_opcode_btfsc, src, 0);
      pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
      pic_instr_append_f(pf, pic_opcode_btfss, src, 0);
      pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
    }
  } else if (!pic_work_in_temp(pf, op, dst, src, val2,
        PIC_WORK_IN_TEMP_FLAG_VAL1
        | PIC_WORK_IN_TEMP_FLAG_DST)) {
    variable_sz_t sz;
    variable_sz_t sz_hi;
    variable_sz_t ii;
    variable_sz_t ipos;
    pic_opdst_t   opdst;

    sz    = pic_result_sz_get(src, val2, dst);
    sz_hi = value_sz_get(src);
    if (sz < sz_hi) {
      sz_hi = sz;
    }
    pic_indirect_setup3(pf, dst, src, val2, 0, &ipos);
    opdst = (value_is_same(dst, src))
      ? pic_opdst_f
      : pic_opdst_w;
    for (ii = 0; ii < sz_hi; ii++) {
      pic_indirect_bump3(pf, dst, src, val2, ii, &ipos);
      pic_instr_append_f_d(pf, pic_opcode_comf, src, ii, opdst);
      if (pic_opdst_w == opdst) {
        pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
      }
    }
    if (ii < sz) {
      src = pic_value_sign_get(pf, src, VALUE_NONE);
      if (VALUE_NONE == src) {
        pic_instr_append_w_kn(pf, pic_opcode_movlw, 0xff);
      } else {
        pic_instr_append_f_d(pf, pic_opcode_comf, src, 0, pic_opdst_w);
      }
      while (ii < sz) {
        pic_indirect_bump3(pf, dst, src, val2, ii, &ipos);
        pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
        ii++;
      }
      if (VALUE_NONE != src) {
        pic_var_accum_release(pf, src);
      }
    }
    pic_value_sign_extend(pf, dst, ii - 1, 
        pic_result_is_signed(pf, src, val2));
  }
}

static void pic_op_neg(pfile_t *pf, operator_t op, value_t dst,
  value_t src, value_t val2)
{
  assert(!pic_value_is_w(dst));
  assert(!pic_value_is_w(src));

  if (!pic_work_in_temp(pf, op, dst, src, val2,
        PIC_WORK_IN_TEMP_FLAG_VAL1
        | PIC_WORK_IN_TEMP_FLAG_DST)) {
    variable_sz_t sz;

    sz = pic_result_sz_get(src, val2, dst);
    if ((1 == sz) 
      && !pic_is_12bit(pf)  
      && (value_is_volatile(dst) || !value_is_same(dst, src))) {
      variable_sz_t ipos;

      pic_indirect_setup3(pf, dst, src, val2, 0, &ipos);
      pic_instr_append_f_d(pf, pic_opcode_comf, src, 0, pic_opdst_w);
      pic_instr_append_w_kn(pf, pic_opcode_addlw, 1);
      pic_instr_append_f(pf, pic_opcode_movwf, dst, 0);
      
      pic_value_sign_extend(pf, dst, 0, 
          pic_result_is_signed(pf, src, val2));
    } else if (value_is_volatile(dst)) {
        value_t tmp;

        tmp = pic_var_temp_get(pf, pic_result_flag_get(pf, src, val2),
            pic_result_sz_get(src, val2, dst));
        pic_op(pf, operator_cmpb, tmp, src, val2);
        pic_op(pf, operator_incr, dst, tmp, VALUE_NONE);
        pic_var_temp_release(pf, tmp);
    } else if (value_sz_get(dst) > sz) {
      value_t        tmp;
      variable_def_t def;

      tmp = value_clone(dst);
      def = variable_def_alloc(0, variable_def_type_integer,
          value_dflag_test(src, VARIABLE_DEF_FLAG_SIGNED)
            ? VARIABLE_DEF_FLAG_SIGNED
            : VARIABLE_DEF_FLAG_NONE,
          sz);
      value_def_set(tmp, def);
      pic_op(pf, operator_cmpb, tmp, src, val2);
      pic_op(pf, operator_incr, tmp, tmp, VALUE_NONE);
      pic_value_sign_extend(pf, dst, sz - 1, 
          pic_result_is_signed(pf, src, val2));
      value_release(tmp);
    } else {
      pic_op(pf, operator_cmpb, dst, src, val2);
      pic_op(pf, operator_incr, dst, dst, VALUE_NONE);
    }
  }
}

static void pic_op_incr(pfile_t *pf, operator_t op, value_t dst,
  value_t src, value_t val2)
{
  assert(!pic_value_is_w(dst));
  assert(!pic_value_is_w(src));

  if (!pic_work_in_temp(pf, op, dst, src, val2,
    PIC_WORK_IN_TEMP_FLAG_VAL1
    | PIC_WORK_IN_TEMP_FLAG_DST)) {
    variable_sz_t sz;
    variable_sz_t ii;

    ii = 0;
    sz = pic_result_sz_get(src, val2, dst);
    if (1 == sz) {
      /* trivial case */
      pic_opdst_t   opdst;
      variable_sz_t ipos;

      opdst = (value_is_same(src, dst)) ? pic_opdst_f : pic_opdst_w;

      ii = 1; /* for sign extend */
      pic_indirect_setup3(pf, src, val2, dst, 0, &ipos);
      pic_instr_append_f_d(pf, pic_opcode_incf, src, 0, opdst);
      if (pic_opdst_w == opdst) {
        pic_instr_append_f(pf, pic_opcode_movwf, dst, 0);
      }
    } else if (value_is_same(src, dst) && !value_is_indirect(src)) {
      for (ii = 0; ii < sz; ii++) {
        if (ii) {
          pic_instr_append_reg_flag(pf, pic_opcode_btfsc, "_status", "_z");
        }
        pic_instr_append_f_d(pf, pic_opcode_incf, dst, ii, pic_opdst_f);
      }
    } else {
      variable_sz_t ipos;
      pic_opdst_t   opdst;
      value_t       src_sign;

      opdst = (value_is_same(dst, src)) ? pic_opdst_f : pic_opdst_w;
      pic_indirect_setup3(pf, dst, src, VALUE_NONE, 0, &ipos);
      pic_instr_append_w_kn(pf, pic_opcode_movlw, 1);
      src_sign = VALUE_NONE;
      for (ii = 0; ii < sz; ii++) {
        pic_indirect_bump3(pf, dst, src, val2, ii, &ipos);
        if (ii) {
          if (ii == value_sz_get(src)) {
            src_sign = pic_value_sign_get(pf, src, VALUE_NONE);
            src      = src_sign;
            if (VALUE_NONE == src) {
              opdst = pic_opdst_w;
            }
          }
          pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
          pic_instr_append_reg_flag(pf, pic_opcode_btfsc, "_status", "_c");
          pic_instr_append_w_kn(pf, pic_opcode_movlw, 1);
        }
        if (VALUE_NONE == src) {
          if (ii + 1 < sz) {
            pic_instr_append_reg_flag(pf, pic_opcode_bcf, "_status", "_c");
          }
        } else {
          pic_instr_append_f_d(pf, pic_opcode_addwf, src, 
            (src == src_sign) ? 0 : ii, opdst);
        }
        if (pic_opdst_w == opdst) {
          pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
        }
      }
      if (VALUE_NONE != src_sign) {
        pic_var_accum_release(pf, src_sign);
      }
    }
    pic_value_sign_extend(pf, dst, ii - 1,
      pic_result_is_signed(pf, src, val2));
  }
}

static void pic_op_decr(pfile_t *pf, operator_t op, value_t dst,
  value_t src, value_t val2)
{
  assert(!pic_value_is_w(dst));
  assert(!pic_value_is_w(src));

  if (!pic_work_in_temp(pf, op, dst, src, val2,
    PIC_WORK_IN_TEMP_FLAG_VAL1
    | PIC_WORK_IN_TEMP_FLAG_DST)) {
    variable_sz_t sz;
    variable_sz_t ii;

    sz = pic_result_sz_get(src, val2, dst);
    if (1 == sz) {
      /* trivial case */
      pic_opdst_t   opdst;
      variable_sz_t ipos;

      opdst = (value_is_same(src, dst)) ? pic_opdst_f : pic_opdst_w;

      ii = 1;
      pic_indirect_setup3(pf, src, VALUE_NONE, dst, 0, &ipos);
      pic_instr_append_f_d(pf, pic_opcode_decf, src, 0, opdst);
      if (pic_opdst_w == opdst) {
        pic_instr_append_f(pf, pic_opcode_movwf, dst, 0);
      }
    } else {
      variable_sz_t ipos;
      pic_opdst_t   opdst;
      variable_sz_t sz1;

      opdst = (value_is_same(src, dst)) ? pic_opdst_f : pic_opdst_w;
      pic_indirect_setup3(pf, dst, src, VALUE_NONE, 0, &ipos);
      pic_instr_append_w_kn(pf, pic_opcode_movlw, 1);

      sz1 = (sz > value_sz_get(src))
        ? value_sz_get(src)
        : sz;

      for (ii = 0; ii < sz1; ii++) {
        pic_indirect_bump3(pf, dst, src, VALUE_NONE, ii, &ipos);
        if (ii) {
          if (pic_opdst_w == opdst) {
            pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
          }
          pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_c");
          if (pic_opdst_w == opdst) {
            pic_instr_append_w_kn(pf, pic_opcode_movlw, 1);
          }
        }
        pic_instr_append_f_d(pf, pic_opcode_subwf, src, ii, opdst);
        if (pic_opdst_w == opdst) {
          pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
        }
      }
      if (ii < sz) {
        pic_instr_append_w_kn(pf, pic_opcode_movlw, 0xff);
        pic_instr_append_reg_flag(pf, pic_opcode_btfsc, "_status", "_c");
        pic_instr_append_w_kn(pf, pic_opcode_movlw, 0x00);
        while (ii < sz) {
          pic_indirect_bump3(pf, dst, src, VALUE_NONE, ii, &ipos);
          pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
          ii++;
        }
      }
    }
    pic_value_sign_extend(pf, dst, ii - 1,
      pic_result_is_signed(pf, src, val2));
  }
}

/* assign src to dst, shifting left one bit in the process */
static void pic_assign_shift_left_1_bit(pfile_t *pf, value_t dst, value_t src,
    variable_sz_t sz)
{
  variable_sz_t ii;
  variable_sz_t ipos;
  pic_opdst_t   opdst;
  value_t       sign;

  assert(VALUE_NONE != dst);
  assert(VALUE_NONE != src);
  assert(!value_is_lookup(src)
    && !value_is_bit(src));
  assert(!value_is_bit(dst));
  assert(!(value_is_indirect(dst) && value_is_indirect(src))
    || value_is_same(dst, src));

  sign  = VALUE_NONE;
  opdst = (value_is_same(dst, src)) ? pic_opdst_f : pic_opdst_w;
  pic_indirect_setup3(pf, dst, src, VALUE_NONE, 0, &ipos);
  pic_instr_append_reg_flag(pf, pic_opcode_bcf, "_status", "_c");
  for (ii = 0; ii < sz; ii++) {
    if (ii == value_sz_get(src)) {
      sign = pic_value_sign_get(pf, src, VALUE_NONE);
      if (VALUE_NONE == sign) {
        /* cannot use clrf as that clears the "C" flag */
        sign = pic_var_accum_get(pf);
        pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
        pic_instr_append_f(pf, pic_opcode_movwf, sign, 0);
      }
      src  = sign;
    }
    pic_indirect_bump3(pf, dst, src, VALUE_NONE, ii, &ipos);
    pic_instr_append_f_d(pf, pic_opcode_rlf, src, 
        (VALUE_NONE == sign) ? ii : 0, opdst);
    if (pic_opdst_w == opdst) {
      pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
    }
  }
  if (VALUE_NONE != sign) {
    pic_var_accum_release(pf, sign);
  }
}

/* assign src to dst, shifting left (n) bytes in the process */
static void pic_assign_shift_left_n_bytes(pfile_t *pf, value_t dst, 
  value_t src, variable_sz_t sz, variable_sz_t byte_shift)
{
  /* if dst & src are equal and indirect, we need to have one of them
     moved to a temporary */
  if (!pic_work_in_temp(pf, operator_shift_left, dst, src, VALUE_NONE,
      PIC_WORK_IN_TEMP_FLAG_DST_VAL_EQUAL)) {
    /* we need to work *backward* in case src & dst are the same! */
    /* start by positioning src to byte_ofs */
    variable_sz_t ipos;
    variable_sz_t ii;
    value_t       accum;

    assert(sz > byte_shift);

    pic_indirect_setup3(pf, src, VALUE_NONE, VALUE_NONE,
      sz - byte_shift - 1, &ipos);
    if (sz - byte_shift > value_sz_get(src)) {
      accum = pic_value_sign_get(pf, src, VALUE_NONE);
    } else {
      accum = src;
    }
    if (value_is_indirect(dst)) {
      pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE,
        sz - 1, &ipos);
    }
    ii = sz;
    while (ii--) {
      if (value_is_indirect(dst)) {
        pic_indirect_bump3(pf, dst, VALUE_NONE, VALUE_NONE, ii, &ipos);
      } 
      if (ii >= byte_shift) {
        if ((accum != src) && (ii - byte_shift < value_sz_get(src))) {
          if (VALUE_NONE != accum) {
            pic_var_accum_release(pf, accum);
          }
          accum = src;
        }
        if (VALUE_NONE == accum) {
          pic_instr_append_f(pf, pic_opcode_clrf, dst, ii);
        } else {
          pic_instr_append_f_d(pf, pic_opcode_movf, accum, 
            (accum == src) ? ii - byte_shift : 0,
            pic_opdst_w);
          pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
        }
      } else {
        pic_instr_append_f(pf, pic_opcode_clrf, dst, ii);
      }
      if (ii && value_is_indirect(accum) && ipos) {
        value_t fsr;

        fsr = pic_fsr_get(pf);
        pic_instr_append_f_d(pf, pic_opcode_decf, fsr, 0, pic_opdst_f);
        value_release(fsr);
        ipos--;
      }
    }
    assert(accum == src);
    if (value_is_indirect(dst) && (sz < value_sz_get(dst))) {
      value_t fsr;

      fsr = pic_fsr_get(pf);
      if (2 == sz) {
        pic_instr_append_f_d(pf, pic_opcode_incf, fsr, 0, pic_opdst_f);
      } else {
        pic_instr_append_w_kn(pf, pic_opcode_movlw, sz - 1);
        pic_instr_append_f_d(pf, pic_opcode_addwf, fsr, 0, pic_opdst_f);
      }
      value_release(fsr);
    }
  }
}

static void pic_op_shift_left(pfile_t *pf, operator_t op, value_t dst,
  value_t val1, value_t val2)
{
  variable_sz_t sz;

  assert(!pic_value_is_w(dst));
  assert(!pic_value_is_w(val1));
  assert(!pic_value_is_w(val2));

  sz = pic_result_sz_get(val1, val2, dst);

  if (value_is_const(val2)) {
    variable_const_t cn;

    cn = value_const_get(val2);
    if (0 == cn) {
      /* this is a simple assignment */
      pic_unary_fixup(pf, operator_assign, dst, val1, val2);
    } else if (cn >= 8 * sz) {
      pic_assign_const(pf, dst, val1, val2, 0);
    } else if (!pic_work_in_temp(pf, op, dst, val1, val2, 
      PIC_WORK_IN_TEMP_FLAG_VALS
      | PIC_WORK_IN_TEMP_FLAG_DST)) {
      variable_const_t byte_shift;
      variable_const_t bit_shift;

      byte_shift = cn / 8;
      bit_shift  = cn & 7;

      if (byte_shift && !bit_shift) {
        pic_assign_shift_left_n_bytes(pf, dst, val1, sz, byte_shift);
        pic_value_sign_extend(pf, dst, sz - 1,
            pic_result_is_signed(pf, val1, val2));
      } else if (!byte_shift && (1 == bit_shift)) {
        /* this can be done in one pass */
        pic_assign_shift_left_1_bit(pf, dst, val1, sz);
        pic_value_sign_extend(pf, dst, sz - 1,
            pic_result_is_signed(pf, val1, val2));
      } else if (!byte_shift && (4 == bit_shift) && (1 == sz)) {
        /* this can be done in one pass */
        variable_sz_t ipos;

        pic_indirect_setup3(pf, dst, val1, val2, 0, &ipos);
        pic_instr_append_f_d(pf, pic_opcode_swapf, val1, 0, pic_opdst_w);
        pic_instr_append_w_kn(pf, pic_opcode_andlw, 0xf0);
        pic_instr_append_f(pf, pic_opcode_movwf, dst, 0);
        pic_value_sign_extend(pf, dst, sz - 1,
            pic_result_is_signed(pf, val1, val2));
      } else if (!pic_work_in_temp(pf, op, dst, val1, val2,
        PIC_WORK_IN_TEMP_FLAG_DST_VOLATILE)) {
        /* shift bytes and/or bits */
        if (byte_shift) {
          pic_assign_shift_left_n_bytes(pf, dst, val1, sz, byte_shift);
        }

        if (bit_shift) {
          if (!byte_shift) {
            pic_assign_shift_left_1_bit(pf, dst, val1, sz);
            bit_shift--;
          }
        }
        while (bit_shift--) {
          variable_sz_t ii;
          variable_sz_t ipos;

          pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE, 0, &ipos);
          pic_instr_append_reg_flag(pf, pic_opcode_bcf, "_status", "_c");
          for (ii = 0; ii < sz; ii++) {
            pic_indirect_bump3(pf, dst, VALUE_NONE, VALUE_NONE, ii, &ipos);
            pic_instr_append_f_d(pf, pic_opcode_rlf, dst, ii, pic_opdst_f);
          }
        }
        pic_value_sign_extend(pf, dst, sz - 1,
            pic_result_is_signed(pf, val1, val2));
      }
    }
  } else if (!pic_work_in_temp(pf, op, dst, val1, val2,
    PIC_WORK_IN_TEMP_FLAG_DST
    | PIC_WORK_IN_TEMP_FLAG_DST_VOLATILE)) {
      /* nothing to do */
    if (!pic_is_12bit(pf)
        && value_is_same(dst, val1) 
        && !value_is_indirect(dst)) {
      /* minor optimization, val2 goes into W so we don't need a temp */
      variable_sz_t ipos;
      variable_sz_t ii;
      label_t       lbl_top;
      label_t       lbl_test;

      lbl_top  = pfile_label_alloc(pf, 0);
      lbl_test = pfile_label_alloc(pf, 0);

      pic_indirect_setup3(pf, VALUE_NONE, VALUE_NONE, val2, 0, &ipos);
      pic_instr_append_f_d(pf, pic_opcode_movf, val2, 0, pic_opdst_w);
      pic_instr_append_n(pf, pic_opcode_goto, lbl_test);
      pic_instr_append_label(pf, lbl_top);
      pic_instr_append_reg_flag(pf, pic_opcode_bcf, "_status", "_c");
      for (ii = 0; ii < sz; ii++) {
        pic_instr_append_f_d(pf, pic_opcode_rlf, dst, ii, pic_opdst_f);
      }
      pic_instr_append_w_kn(pf, pic_opcode_sublw, 1);
      pic_instr_append_label(pf, lbl_test);
      pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_z");
      pic_instr_append_n(pf, pic_opcode_goto, lbl_top);

      label_release(lbl_test);
      label_release(lbl_top);
    } else if (pic_result_sz_get(val1, val2, dst) < value_sz_get(dst)) {
      variable_def_t def;
      variable_def_t def_fixed;

      def = value_def_get(dst);
      def_fixed = variable_def_alloc(0, 
          variable_def_type_get(def),
          variable_def_flags_get_all(def),
          pic_result_sz_get(val1, val2, dst));
      value_def_set(dst, def_fixed);
      pic_op(pf, op, dst, val1, val2);
      value_def_set(dst, def);
      pic_value_sign_extend(pf, dst, pic_result_sz_get(val1, val2, dst) - 1,
          pic_result_is_signed(pf, val1, val2));
    } else {
      /* do the shift, one bit at a time */
      value_t       accum;
      label_t       lbl_top;
      label_t       lbl_test;
      variable_sz_t ipos;
      variable_sz_t ii;
      value_t       fsr;

      fsr      = pic_fsr_get(pf);
      lbl_top  = pfile_label_alloc(pf, 0);
      lbl_test = pfile_label_alloc(pf, 0);

      accum = pic_var_accum_get(pf);
      pic_op(pf, operator_assign, accum, val2, VALUE_NONE);

      if (!value_is_same(dst, val1)) {
        pic_op(pf, operator_assign, dst, val1, VALUE_NONE);
      }
      /* start dst at the end */
      pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE, sz - 1, &ipos);
      if (!value_is_same(dst, val1) || value_is_indirect(dst)) {
        pic_instr_append_f_d(pf, pic_opcode_movf, accum, 0, pic_opdst_w);
      }
      pic_instr_append_n(pf, pic_opcode_goto, lbl_test);
      pic_instr_append_label(pf, lbl_top);
      if (value_is_indirect(dst)) {
        /* adjust dst to point to the beginning */
        if ((sz > 1) && value_is_indirect(dst)) {
          if (1 == sz - 1) {
            pic_instr_append_f_d(pf, pic_opcode_decf, fsr, 0, pic_opdst_f);
          } else {
            pic_instr_append_w_kn(pf, pic_opcode_movlw, sz - 1);
            pic_instr_append_f_d(pf, pic_opcode_subwf, fsr, 0, pic_opdst_f);
          }
        }
      }
      ipos = 0;
      pic_instr_append_reg_flag(pf, pic_opcode_bcf, "_status", "_c");
      for (ii = 0; ii < sz; ii++) {
        pic_indirect_bump3(pf, dst, VALUE_NONE, VALUE_NONE, ii, &ipos);
        pic_instr_append_f_d(pf, pic_opcode_rlf, dst, ii, pic_opdst_f);
      }
      pic_instr_append_f_d(pf, pic_opcode_decf, accum, 0, pic_opdst_f);
      pic_instr_append_label(pf, lbl_test);
      pic_instr_append_reg_flag(pf, pic_opcode_btfss, "_status", "_z");
      pic_instr_append_n(pf, pic_opcode_goto, lbl_top);
      pic_value_sign_extend(pf, dst, ii - 1, 
          pic_result_is_signed(pf, val1, val2));

      label_release(lbl_test);
      label_release(lbl_top);
      value_release(fsr);
      pic_var_accum_release(pf, accum);
    }
  }
}

static void pic_assign_shift_right_n_bytes(pfile_t *pf, operator_t op,
    value_t dst, value_t src, variable_const_t byte_shift)
{
  variable_sz_t ii;
  variable_sz_t ipos;

  if (value_is_indirect(dst)) {
    pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE,
        0, &ipos);
  } else if (value_is_indirect(src)) {
    pic_indirect_setup3(pf, src, VALUE_NONE, VALUE_NONE,
        byte_shift, &ipos);
  }
  for (ii = 0; ii + byte_shift < value_sz_get(src); ii++) {
    if (value_is_indirect(dst)) {
      pic_indirect_bump3(pf, dst, VALUE_NONE, VALUE_NONE, ii, &ipos);
    } else if (value_is_indirect(src)) {
      pic_indirect_bump3(pf, src, VALUE_NONE, VALUE_NONE, ii + byte_shift, 
          &ipos);
    }
    pic_instr_append_f_d(pf, pic_opcode_movf, src, ii + byte_shift, 
        pic_opdst_w);
    pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
  }
  if (ii < value_sz_get(dst)) {
    if (operator_shift_right_arithmetic == op) {
      pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
      pic_instr_append_f_bn(pf, pic_opcode_btfsc, src, ii + byte_shift - 1,
          7);
      pic_instr_append_w_kn(pf, pic_opcode_movlw, 0xff);
      while (ii < value_sz_get(dst)) {
        pic_indirect_bump3(pf, dst, VALUE_NONE, VALUE_NONE, ii, &ipos);
        pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
        ii++;
      }
    } else {
      while (ii < value_sz_get(dst)) {
        pic_indirect_bump3(pf, dst, VALUE_NONE, VALUE_NONE, ii, &ipos);
        pic_instr_append_f(pf, pic_opcode_clrf, dst, ii);
        ii++;
      }
    }
  }
}

static void pic_assign_shift_right_1_bit(pfile_t *pf, operator_t op,
    value_t dst, value_t src)
{
  variable_sz_t ipos;
  variable_sz_t ii;
  pic_opdst_t   opdst;
  value_t       accum;

  opdst = (value_is_same(dst, src)) ? pic_opdst_f : pic_opdst_w;
  pic_indirect_setup3(pf, dst, src, VALUE_NONE, value_sz_get(src) - 1, &ipos);
  ii = value_sz_get(src);
  pic_instr_append_reg_flag(pf, pic_opcode_bcf, "_status", "_c");
  accum = VALUE_NONE;
  if (operator_shift_right_arithmetic == op) {
    if (value_is_volatile(src)) {
      /* normally src will be accessed twice. clearly that's not allowed
       * for a volatile, so here we go */
      accum = pic_var_accum_get(pf);
      pic_instr_append_f_d(pf, pic_opcode_movf, src, ii - 1, pic_opdst_w);
      pic_instr_append_f(pf, pic_opcode_movwf, accum, 0);
      pic_instr_append_f_bn(pf, pic_opcode_btfsc, accum, 0, 7);
    } else {
      pic_instr_append_f_bn(pf, pic_opcode_btfsc, src, ii - 1, 7);
    }
    pic_instr_append_reg_flag(pf, pic_opcode_bsf, "_status", "_c");
  }
  while (ii--) {
    pic_indirect_bump3(pf, dst, src, VALUE_NONE, ii, &ipos);
    if (accum != VALUE_NONE) {
      pic_instr_append_f_d(pf, pic_opcode_rrf, accum, 0, pic_opdst_w);
      pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
      pic_var_accum_release(pf, accum);
      accum = VALUE_NONE;
    } else {
      pic_instr_append_f_d(pf, pic_opcode_rrf, src, ii, opdst);
      if (pic_opdst_w == opdst) {
        pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
      }
    }
  }
}

static boolean_t pic_shift_right_resize(pfile_t *pf, operator_t op,
    value_t dst, value_t val1, value_t val2)
{
  boolean_t rc;

  rc = boolean_false;
  if ((value_sz_get(val1) < value_sz_get(val2)) && !value_is_universal(val2)) {
    value_t tmp;

    tmp = pic_var_temp_get(pf,
        value_is_signed(val1) 
        ? VARIABLE_DEF_FLAG_SIGNED : VARIABLE_DEF_FLAG_NONE,
        value_sz_get(val2));
    pic_op(pf, operator_assign, tmp, val1, VALUE_NONE);
    pic_op(pf, op, dst, tmp, val2);
    pic_var_temp_release(pf, tmp);
    rc = boolean_true;
  } else if (value_is_bit(dst)
      || (value_sz_get(dst) < pic_result_sz_get(val1, val2, VALUE_NONE))) {
    value_t tmp;

    tmp = pic_var_temp_get(pf, pic_result_flag_get(pf, val1, val2),
      pic_result_sz_get(val1, val2, VALUE_NONE));
    pic_op(pf, op, tmp, val1, val2);
    pic_op(pf, operator_assign, dst, tmp, VALUE_NONE);
    pic_var_temp_release(pf, tmp);
    rc = boolean_true;
  } else {
    variable_sz_t  sz;

    sz = pic_result_sz_get(val1, val2, VALUE_NONE);
    if (value_sz_get(dst) > sz) {
      /* only work in the bottom part of dst, then sign extend */
      value_t        tmp;
      variable_def_t def;
      variable_sz_t  ipos;

      tmp = value_clone(dst);
      def = variable_def_alloc(0, variable_def_type_integer,
          pic_result_flag_get(pf, val1, val2), sz);
      value_def_set(tmp, def);
      pic_op(pf, op, tmp, val1, val2);
      pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE, sz - 1, &ipos);
      pic_value_sign_extend(pf, dst, sz - 1,
          pic_result_is_signed(pf, val1, val2));
      value_release(tmp);
      rc = boolean_true;
    }
  }
  return rc;
}

/* note: op can be operator_shift_right_arithmetic which preserves the
 *       high bit during shifting
 */
static void pic_op_shift_right(pfile_t *pf, operator_t op, value_t dst,
  value_t val1, value_t val2)
{
  variable_sz_t sz;

  assert(!pic_value_is_w(dst));
  assert(!pic_value_is_w(val1));
  assert(!pic_value_is_w(val2));

  /* the only size that matters is val1
   * if dst is smaller, right shifting could end up shifting into dst
   */

  sz = pic_result_sz_get(val1, val2, VALUE_NONE);
  if (value_is_const(val2) && !value_const_get(val2)) {
    /* shifting 0 = 0 */
    pic_unary_fixup(pf, operator_assign, dst, val1, val2);
  } else if ((operator_shift_right_arithmetic == op)
      && ((value_is_single_bit(val1) && (value_sz_get(val1) >= sz * 8))
        || (value_is_const(val1)
          && (value_const_get(val1) == -1)))) {
    /* arithmetic shifting of -1 == -1 */
    /* arithmetic shifting of a single bit = assignment */
    pic_unary_fixup(pf, operator_assign, dst, val1, val2);
  } else if (value_is_const(val2)) {
    variable_const_t cn;

    cn = value_const_get(val2);
    if (0 == cn) {
      /* shouldn't get here */
      pic_unary_fixup(pf, operator_assign, dst, val1, val2);
    } else if (cn >= 8 * sz) {
      if (!pic_work_in_temp(pf, op, dst, val1, val2,
            PIC_WORK_IN_TEMP_FLAG_DST)) {
        /* dst is either all 0x00s or all 0xffs */
        if ((operator_shift_right_arithmetic == op)) {
          variable_sz_t ii;
          variable_sz_t ipos;

          if (value_is_signed(val1) || (value_sz_get(val1) == sz)) {
            pic_indirect_setup3(pf, val1, VALUE_NONE, VALUE_NONE, 
                value_sz_get(val1) - 1, &ipos);
            pic_instr_append_w_kn(pf, pic_opcode_movlw, 0x00);
            if (value_is_bit(val1)) {
              pic_instr_append_f_bn(pf, pic_opcode_btfsc,
                  val1, 0, value_sz_get(val1) - 1);
            } else {
              pic_instr_append_f_bn(pf, pic_opcode_btfsc, val1,
                  value_sz_get(val1) - 1, 7);
            }
            pic_instr_append_w_kn(pf, pic_opcode_movlw, 0xff);
            if (value_is_indirect(dst)) {
              value_t accum;

              accum = pic_var_accum_get(pf);
              pic_instr_append_f(pf, pic_opcode_movwf, accum, 0);
              pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE, 0, &ipos);
              pic_instr_append_f_d(pf, pic_opcode_movf, accum, 0, pic_opdst_w);
              pic_var_accum_release(pf, accum);
            }
            if (pic_result_is_signed(pf, val1, val2)) {
              sz = value_sz_get(dst);
            }
            for (ii = 0; ii < sz; ii++) {
              pic_indirect_bump3(pf, dst, VALUE_NONE, VALUE_NONE, ii, &ipos);
              pic_instr_append_f(pf, pic_opcode_movwf, dst, ii);
            }
          } else {
            pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE, 0, &ipos);
            for (ii = 0; ii < sz; ii++) {
              pic_indirect_bump3(pf, dst, VALUE_NONE, VALUE_NONE, ii, &ipos);
              pic_instr_append_f(pf, pic_opcode_clrf, dst, ii);
            }
          }
          sz = value_sz_get(dst);
          while (ii < sz) {
            pic_indirect_bump3(pf, dst, VALUE_NONE, VALUE_NONE, ii, &ipos);
            pic_instr_append_f(pf, pic_opcode_clrf, dst, ii);
            ii++;
          }
        } else {
          variable_sz_t ii;
          variable_sz_t ipos;

          pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE, 0, &ipos);
          sz = value_sz_get(dst);
          for (ii = 0; ii < sz; ii++) {
            pic_indirect_bump3(pf, dst, VALUE_NONE, VALUE_NONE, ii, &ipos);
            pic_instr_append_f(pf, pic_opcode_clrf, dst, ii);
          }
        }
      }
    } else if (!pic_shift_right_resize(pf, op, dst, val1, val2)
      && !pic_work_in_temp(pf, op, dst, val1, val2,
          PIC_WORK_IN_TEMP_FLAG_VAL1)) {
      variable_sz_t bit_shift;
      variable_sz_t byte_shift;

      byte_shift = cn / 8;
      bit_shift  = cn & 0x07;

      if (byte_shift && !bit_shift) {
        /* this can be done in one pass */
        variable_sz_t ipos;

        pic_assign_shift_right_n_bytes(pf, op, dst, val1, byte_shift);
        pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE, 
            value_sz_get(val1) - 1, &ipos);
        pic_value_sign_extend(pf, dst, value_sz_get(val1) - 1,
            pic_result_is_signed(pf, val1, val2));
      } else if (!byte_shift && (1 == bit_shift)) {
        /* this can be done in one pass */
        variable_sz_t ipos;

        pic_assign_shift_right_1_bit(pf, op, dst, val1);
        pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE, 
            value_sz_get(val1) - 1, &ipos);
        pic_value_sign_extend(pf, dst, value_sz_get(val1) - 1,
            pic_result_is_signed(pf, val1, val2));
      } else if (!byte_shift
          && (4 == bit_shift) 
          && (1 == sz)
          /*&& (operator_shift_right == op) */) {
        /* this can be done in one pass */
        variable_sz_t ipos;
        value_t       accum;

        pic_indirect_setup3(pf, dst, val1, VALUE_NONE, 0, &ipos);
        pic_instr_append_f_d(pf, pic_opcode_swapf, val1, 0, pic_opdst_w);
        pic_instr_append_w_kn(pf, pic_opcode_andlw, 0x0f);
        accum = val1;
        if (operator_shift_right_arithmetic == op) {
          if (value_is_volatile(val1)) {
            /* val1 will be read twice, so move the value into accum */
            accum = pic_var_accum_get(pf);
            pic_instr_append_f(pf, pic_opcode_movwf, accum, 0);
          }
          pic_instr_append_f_bn(pf, pic_opcode_btfsc, accum, 0, 
            (accum == val1) ? 7 : 3);
          pic_instr_append_w_kn(pf, pic_opcode_iorlw, 0xf0);
        }
        pic_instr_append_f(pf, pic_opcode_movwf, dst, 0);
        if (accum != val1) {
          pic_var_accum_release(pf, accum);
        }
      } else if (!pic_work_in_temp(pf, op, dst, val1, val2,
            PIC_WORK_IN_TEMP_FLAG_DST_VOLATILE)) {
        if (byte_shift) {
          pic_assign_shift_right_n_bytes(pf, op, dst, val1, byte_shift);
        }
        if (bit_shift && !byte_shift 
            && (!value_is_same(dst, val1) || (bit_shift * sz < 4))) {
          pic_assign_shift_right_1_bit(pf, op, dst, val1);
          bit_shift--;
        }
        pic_value_sign_extend(pf, dst, value_sz_get(val1) - 1,
            pic_result_is_signed(pf, val1, val2)
            || (value_is_signed(val1) 
              && (value_sz_get(val2) > value_sz_get(val1))));
        if (bit_shift * sz < 4) {
          value_t fsr;

          fsr = pic_fsr_get(pf);
          while (bit_shift--) {
            variable_sz_t ii;

            /* we're at 0 and need to move back to the end */
            if (value_is_indirect(dst) && (sz != 1)) {
              if (byte_shift) {
                /* if we've byte-shifted and are indirect, we're positioned
                 * at the end of the indirect already, so set byte_shift
                 * to zero */
                byte_shift = 0;
              } else {
                pic_instr_append_w_kn(pf, pic_opcode_movlw, sz - 1);
                pic_instr_append_f_d(pf, pic_opcode_addwf, fsr, 0,
                    pic_opdst_f);
              }
            }
            pic_instr_append_reg_flag(pf, pic_opcode_bcf, "_status", "_c");
            if (operator_shift_right_arithmetic == op) {
              pic_instr_append_f_bn(pf, pic_opcode_btfsc, dst, sz - 1, 7);
              pic_instr_append_reg_flag(pf, pic_opcode_bsf, "_status", "_c");
            }
            ii = sz;
            while (ii) {
              if ((ii != sz) && value_is_indirect(dst)) {
                pic_instr_append_f_d(pf, pic_opcode_decf, fsr, 0,
                    pic_opdst_f);
              }
              ii--;
              pic_instr_append_f_d(pf, pic_opcode_rrf, dst, ii, pic_opdst_f);
            }
          }
          value_release(fsr);
        } else {
          value_t       loop;
          label_t       top;
          variable_sz_t ipos;
          variable_sz_t ii;

          loop = pic_var_temp_get(pf, VARIABLE_FLAG_NONE, 1);
          top = pfile_label_alloc(pf, 0);
          pic_instr_append_w_kn(pf, pic_opcode_movlw, bit_shift);
          pic_instr_append_f(pf, pic_opcode_movwf, loop, 0);
          pic_instr_append_label(pf, top);
          pic_indirect_setup3(pf, dst, VALUE_NONE, VALUE_NONE, sz - 1, &ipos);
          pic_instr_append_reg_flag(pf, pic_opcode_bcf, "_status", "_c");
          if (operator_shift_right_arithmetic == op) {
            pic_instr_append_f_bn(pf, pic_opcode_btfsc, dst, sz - 1, 7);
            pic_instr_append_reg_flag(pf, pic_opcode_bsf,"_status", "_c");
          }
          for (ii = sz; ii; ii--) {
            pic_indirect_bump3(pf, dst, VALUE_NONE, VALUE_NONE, ii - 1, &ipos);
            pic_instr_append_f_d(pf, pic_opcode_rrf, dst, ii - 1, pic_opdst_f);
          }
          pic_instr_append_f_d(pf, pic_opcode_decfsz, loop, 0, pic_opdst_f);
          pic_instr_append_n(pf, pic_opcode_goto, top);
          label_release(top);
          pic_var_temp_release(pf, loop);
        }
      }
    }
  } else if (!pic_shift_right_resize(pf, op, dst, val1, val2)
      && !pic_work_in_temp(pf, op, dst, val1, val2,
        PIC_WORK_IN_TEMP_FLAG_VAL1
        | PIC_WORK_IN_TEMP_FLAG_DST
        | PIC_WORK_IN_TEMP_FLAG_DST_VOLATILE)) {
    value_t       accum;
    label_t       lbl_top;
    label_t       lbl_end;
    value_t       c1;
    variable_sz_t ipos;
    variable_sz_t ii;

    lbl_top = pfile_label_alloc(pf, 0);
    lbl_end = pfile_label_alloc(pf, 0);

    c1 = pfile_constant_get(pf, 1, VARIABLE_DEF_NONE);
    accum = pic_var_accum_get(pf);
    pic_op(pf, operator_assign, accum, val2, VALUE_NONE);
    pic_op(pf, operator_assign, dst, val1, VALUE_NONE);
    if (!value_is_same(dst, val1)) {
      pic_instr_append_f_d(pf, pic_opcode_movf, accum, 0, pic_opdst_f);
    }
    pic_instr_append_reg_flag(pf, pic_opcode_btfsc, "_status", "_z");
    pic_instr_append_n(pf, pic_opcode_goto, lbl_end);
    pic_instr_append_label(pf, lbl_top);
    pic_indirect_setup3(pf, val1, VALUE_NONE, dst, sz - 1, &ipos);
    pic_instr_append_reg_flag(pf, pic_opcode_bcf, "_status", "_c");
    for (ii = sz; ii; ii--) {
      pic_indirect_bump3(pf, val1, VALUE_NONE, dst, ii - 1, &ipos);
      pic_instr_append_f_d(pf, pic_opcode_rrf, dst, ii - 1, pic_opdst_f);
      if ((ii == sz) && (operator_shift_right_arithmetic == op)) {
        pic_instr_append_f_bn(pf, pic_opcode_btfsc, dst, ii - 1, 6);
        pic_instr_append_f_bn(pf, pic_opcode_bsf, dst, ii - 1, 7);
      }
    }
    pic_instr_append_f_d(pf, pic_opcode_decfsz, accum, 0, pic_opdst_f);
    pic_instr_append_n(pf, pic_opcode_goto, lbl_top);
    pic_instr_append_label(pf, lbl_end);

    pic_var_accum_release(pf, accum);
    value_release(c1);
    label_release(lbl_end);
    label_release(lbl_top);
  }
}

/* set dst to 0 if val is 0, 1 if val != 0 */
static void pic_op_logical_or_not(pfile_t *pf, operator_t op, value_t dst,
  value_t src, value_t val2)
{
  assert(!pic_value_is_w(src));

  if (value_is_single_bit(src)) {
    if (value_is_single_bit(dst)) {
      pic_opcode_t opcode;

      opcode = (operator_logical == op) ? pic_opcode_btfsc : pic_opcode_btfss;

      if ((operator_logical == op) && value_is_same(dst, src)) {
        /* this is a no-op */
      } else if (value_is_same(dst, src) 
          || value_is_volatile(dst)
          || !pic_val_bank_same(pf, dst, src)) {
        label_t lbl_set;
        label_t lbl_done;

        lbl_set  = pfile_label_alloc(pf, 0);
        lbl_done = pfile_label_alloc(pf, 0);
        pic_instr_append_f(pf, opcode, src, 0);
        pic_instr_append_n(pf, pic_opcode_goto, lbl_set);
        pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
        pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
        pic_instr_append_label(pf, lbl_set);
        pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
        pic_instr_append_label(pf, lbl_done);

        label_release(lbl_done);
        label_release(lbl_set);
      } else {
        pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
        pic_instr_append_f(pf, opcode, src, 0);
        pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
      }
    } else {
      pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
      pic_instr_append_f(pf, (operator_logical == op)
          ? pic_opcode_btfsc
          : pic_opcode_btfss,
          src, 0);
      pic_instr_append_w_kn(pf, pic_opcode_movlw, 1);
      pic_assign_from_w(pf, dst, boolean_false);
    }
  } else {
    pic_value_is_zero(pf, src); /* Z is set if src is zero */
    if (value_is_single_bit(dst)) {
      if (value_is_volatile(dst)) {
        label_t lbl_clr;
        label_t lbl_done;

        lbl_clr = pfile_label_alloc(pf, 0);
        lbl_done = pfile_label_alloc(pf, 0);
        pic_instr_append_reg_flag(pf, (operator_logical == op)
            ? pic_opcode_btfsc : pic_opcode_btfss, "_status", "_z");
        pic_instr_append_n(pf, pic_opcode_goto, lbl_clr);
        pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
        pic_instr_append_n(pf, pic_opcode_goto, lbl_done);
        pic_instr_append_label(pf, lbl_clr);
        pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
        pic_instr_append_label(pf, lbl_done);
        label_release(lbl_done);
        label_release(lbl_clr);
      } else {
        pic_instr_append_f(pf, pic_opcode_bsf, dst, 0);
        pic_instr_append_reg_flag(pf, (operator_logical == op)
            ? pic_opcode_btfsc : pic_opcode_btfss, "_status", "_z");
        pic_instr_append_f(pf, pic_opcode_bcf, dst, 0);
      }
    } else if (VALUE_NONE != dst) {
      pic_instr_append_w_kn(pf, pic_opcode_movlw, 1);
      pic_instr_append_reg_flag(pf, (operator_logical == op)
          ? pic_opcode_btfsc : pic_opcode_btfss, "_status", "_z");
      pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
      pic_assign_from_w(pf, dst, boolean_false);
    }
  }
}

/*
 * NAME
 *   pic_op
 *
 * DESCRIPTION
 *   create code for an operator
 *
 * PARAMETERS
 *   pf  : file
 *   op  :
 *   dst :
 *   val1:
 *   val2:
 *
 * RETURN
 *   none
 *
 * NOTES
 */
extern variable_sz_t pic_temp_sz_in_use;

typedef void (*operator_fn_t)(pfile_t *pf, operator_t op, value_t dst, 
    value_t val1, value_t val2);

static const struct operator_fn_tbl_ {
  operator_t    op;
  operator_fn_t fn;
  unsigned      flags;
} operator_fn_tbl[] = {
  { operator_null,        0,                     0},
  { operator_add,         pic_op_add_sub,        0},
  { operator_sub,         pic_op_add_sub,        0},
  { operator_mul,         pic_op_mul,            0},
  { operator_div,         pic_op_div_mod,        0},
  { operator_mod,         pic_op_div_mod,        0},
  { operator_lt,          pic_op_relational,     0},
  { operator_le,          pic_op_relational,     0},
  { operator_eq,          pic_op_relational,     0},
  { operator_ne,          pic_op_relational,     0},
  { operator_ge,          pic_op_relational,     0},
  { operator_gt,          pic_op_relational,     0},
  { operator_andl,        pic_op_logical_and,    0},
  { operator_orl,         pic_op_logical_or,     0},
  { operator_notl,        pic_op_logical_or_not, 0},
  { operator_andb,        pic_op_binary_and,     0},
  { operator_orb,         pic_op_binary_or,      0},
  { operator_xorb,        pic_op_binary_xor,     0},
  { operator_cmpb,        pic_op_cmp,            0},
  { operator_assign,      pic_op_assign,         0},
  { operator_neg,         pic_op_neg,            0},
  { operator_incr,        pic_op_incr,           0},
  { operator_decr,        pic_op_decr,           0},
  { operator_shift_left,  pic_op_shift_left,     0},
  { operator_shift_right, pic_op_shift_right,    0},
  { operator_shift_right_arithmetic, pic_op_shift_right,    0},
  { operator_logical,     pic_op_logical_or_not, 0}
};

void pic_op(pfile_t *pf, operator_t op, value_t dst, value_t val1,
    value_t val2)
{
  variable_sz_t    init;

  init = pic_temp_sz_in_use;
  if ((operator_assign != op) 
      && (value_is_label(val1) || value_is_label(val2))) {
    if (value_is_label(val1)) {
      value_t tmp;

      tmp = pic_var_temp_get_assign(pf, val1, VALUE_NONE, dst);
      pic_op(pf, op, dst, tmp, val2);
      pic_var_temp_release(pf, tmp);
    } else if (value_is_label(val2)) {
      value_t tmp;

      tmp = pic_var_temp_get_assign(pf, val2, VALUE_NONE, dst);
      pic_op(pf, op, dst, val1, tmp);
      pic_var_temp_release(pf, tmp);
    }
  } else {
    if (operator_is_binary(op)) {
      if (!val2) {
        val2 = val1;
        val1 = dst;
      }
    } else if (operator_is_unary(op)) {
      if (!val1) {
        val1 = dst;
      }
    }
    if ((operator_assign != op)
        && value_is_const(val1)
        && (operator_is_unary(op) || value_is_const(val2))) {
      value_t tmp;

      tmp = pfile_constant_get(pf,
              pfile_op_const_exec(pf, op, val1, val2),
              VARIABLE_DEF_NONE);
      pic_op(pf, operator_assign, dst, tmp, VALUE_NONE);
      value_release(tmp);
    } else {
      size_t           ii;
      pic_last_value_t which;

      for (ii = 0;
           (ii < COUNT(operator_fn_tbl))
           && (operator_fn_tbl[ii].op != op);
           ii++)
        ;
      if (ii == COUNT(operator_fn_tbl)) {
        assert(0);
      } else if (!operator_fn_tbl[ii].fn) {
        assert(0);
      } else {
        operator_fn_tbl[ii].fn(pf, op, dst, val1, val2);
      }
      /* if one of the last values has changed, invalid these */
      for (which = PIC_LAST_VALUE_FIRST;
           (which < PIC_LAST_VALUE_CT)
           && !value_is_same(pic_last_values[which], dst);
           which++)
        ;
      if (which < PIC_LAST_VALUE_CT) {
        pic_last_values_reset();
      }
    }
  }
  assert(init == pic_temp_sz_in_use);
}

