/************************************************************
 **
 ** pic_wopt.c : W value optimization definitions
 **
 ** Copyright (c) 2006, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/
#include "../libcore/pf_proc.h"
#include "piccolst.h"
#include "pic_wopt.h"

/*
 * NAME
 *   pic_code_z_changed
 *
 * DESCRIPTION
 *   determine if an instruction changes the 'z' flag
 *
 * PARAMETERS
 *   op : opcode
 *
 * RETURN
 *   return TRUE if an instruction changes 'z', FALSE if not
 *
 * NOTES
 */
static boolean_t pic_code_z_changed(pic_opcode_t op)
{
  boolean_t rc;

  rc = boolean_false;
  switch (op) {
    case pic_opcode_org:
    case pic_opcode_end:
    case pic_opcode_none:
    case pic_opcode_nop:
    case pic_opcode_sleep:
    case pic_opcode_clrwdt:
    case pic_opcode_option:
    case pic_opcode_bcf:
    case pic_opcode_bsf:
    case pic_opcode_btfsc:
    case pic_opcode_btfss:
    case pic_opcode_tris:
    case pic_opcode_datalo_set:
    case pic_opcode_datalo_clr:
    case pic_opcode_datahi_set:
    case pic_opcode_datahi_clr:
    case pic_opcode_irp_set:
    case pic_opcode_irp_clr:
    case pic_opcode_branchlo_set:
    case pic_opcode_branchlo_clr:
    case pic_opcode_branchlo_nop:
    case pic_opcode_branchhi_set:
    case pic_opcode_branchhi_clr:
    case pic_opcode_branchhi_nop:
    case pic_opcode_rlcf:
    case pic_opcode_rlf:
    case pic_opcode_rrf:
    case pic_opcode_swapf:
    case pic_opcode_movwf:
    case pic_opcode_movlw:
    case pic_opcode_retlw:
    case pic_opcode_retfie:
    case pic_opcode_return:
    case pic_opcode_goto:
    case pic_opcode_call:
    case pic_opcode_decfsz:
    case pic_opcode_incfsz:
    case pic_opcode_mullw:
    case pic_opcode_dcfsnz:
    case pic_opcode_infsnz:
    case pic_opcode_cpfseq:
    case pic_opcode_cpfsgt:
    case pic_opcode_cpfslt:
    case pic_opcode_mulwf:
    case pic_opcode_setf:
    case pic_opcode_tstfsz:
    case pic_opcode_btg:
    case pic_opcode_bc:
    case pic_opcode_bnc:
    case pic_opcode_bn:
    case pic_opcode_bnov:
    case pic_opcode_bnn:
    case pic_opcode_bnz:
    case pic_opcode_bov:
    case pic_opcode_bz:
    case pic_opcode_bra:
    case pic_opcode_rcall:
    case pic_opcode_daw:
    case pic_opcode_pop:
    case pic_opcode_push:
    case pic_opcode_lfsr:
    case pic_opcode_movff:
    case pic_opcode_movlb:
    case pic_opcode_tblrd:
    case pic_opcode_tblwt:
    case pic_opcode_db:
      rc = boolean_false;
      break;
    case pic_opcode_reset:
    case pic_opcode_negf:
    case pic_opcode_rlncf:
    case pic_opcode_rrncf:
    case pic_opcode_addwfc:
    case pic_opcode_subfwb:
    case pic_opcode_subwfb:
    case pic_opcode_clrf:
    case pic_opcode_addwf:
    case pic_opcode_andwf:
    case pic_opcode_xorwf:
    case pic_opcode_iorwf:
    case pic_opcode_subwf:
    case pic_opcode_comf:
    case pic_opcode_decf:
    case pic_opcode_incf:
    case pic_opcode_movf:
    case pic_opcode_clrw:
    case pic_opcode_addlw:
    case pic_opcode_andlw:
    case pic_opcode_iorlw:
    case pic_opcode_sublw:
    case pic_opcode_xorlw:
      rc = boolean_true;
      break;
  }
  return rc;
}

/*
 * NAME
 *   pic_code_z_used
 *
 * DESCRIPTION
 *   determine if status:z is used before an intervening
 *   command changes it
 *
 * PARAMETERS
 *   code : place to start
 *
 * RETURN
 *   TRUE -- status:z is used, FALSE, it's not
 *
 * NOTES
 *   an instruction set like:
 *     movwf x
 *     movf  x,w
 *     btfss _status, _z
 *   will be broken if the movwf/movf set is removed. To avoid
 *   this, determine if _status:_z is used before it is changed
 *   by another code. If so, change the movwf/movf to iorlw 0
 *   which will correctly set the flag
 */
static boolean_t pic_code_z_used(pfile_t *pf, pic_code_t code)
{
  boolean_t rc;
  value_t   status;
  value_t   status_z;

  rc       = boolean_false;
  status   = pfile_value_find(pf, pfile_log_err, "_status");
  status_z = pfile_value_find(pf, pfile_log_err, "_z");

  while ((boolean_false == rc)
      && code 
      && !pic_code_z_changed(pic_code_op_get(code))) {
    pic_code_t next;

    next = pic_code_next_get(code);
    if ((pic_opcode_btfss == pic_code_op_get(code))
      || (pic_opcode_btfsc == pic_code_op_get(code))) {
      if (value_is_same(pic_code_value_get(code), status)
          && (value_const_get(status_z) 
            == value_const_get(pic_code_literal_get(code)))) {
        rc = boolean_true;
      }
    } else if (pic_opcode_goto == pic_code_op_get(code)) {
      /* follow the goto path */
      /*next = pic_code_label_find(pf, pic_code_brdst_get(code));*/
    }
    code = next;
  }
  value_release(status_z);
  value_release(status);
  return rc;
}

/*
 * NAME
 *   pic_code_w_value_detect
 *
 * DESCRIPTION
 *   determine the value in W at each code location
 *
 * PARAMETERS
 *   pf      : pfile handle
 *   code    : current code
 *   w_value : current value in w
 *
 * RETURN
 *   last known value in W
 *
 * NOTES
 */
static value_t pic_code_w_value_detect(pfile_t *pf, pic_code_t code, 
    value_t w_value)
{
  boolean_t pv_code_is_cond;
  boolean_t changed;

  pv_code_is_cond = boolean_false;
  changed         = boolean_true;
  while (code 
      && (changed
        || !pic_code_flag_test(code, PIC_CODE_FLAG_W_VISITED))) {
    pic_opcode_t op;
    boolean_t    code_is_cond;
    pic_code_t   next;

    next = pic_code_next_get(code);

    pic_code_flag_set(code, PIC_CODE_FLAG_W_VISITED);
    /* we know changed is boolean_true, so need only set it to false */
    if (pic_code_flag_test(code, PIC_CODE_FLAG_W_UNKNOWN)) {
      w_value = VALUE_NONE;
      changed = boolean_false;
    } else if (VALUE_NONE == pic_code_w_value_get(code)) {
      if (w_value) {
        pic_code_w_value_set(code, w_value);
      } else {
        changed = boolean_false;
      }
    } else if (!value_is_same(w_value, pic_code_w_value_get(code))) {
      pic_code_w_value_set(code, VALUE_NONE);
      pic_code_flag_set(code, PIC_CODE_FLAG_W_UNKNOWN);
    } else {
      changed = boolean_false;
    }
    op = pic_code_op_get(code);
    code_is_cond = (pic_opcode_btfss == op)
       || (pic_opcode_btfsc == op)
       || (pic_opcode_incfsz == op)
       || (pic_opcode_decfsz == op);
    switch (op) {
      case pic_opcode_org:
      case pic_opcode_end:
      case pic_opcode_none:
      case pic_opcode_clrf:
      case pic_opcode_nop:
      case pic_opcode_sleep:
      case pic_opcode_clrwdt:
      case pic_opcode_option:
      case pic_opcode_bcf:
      case pic_opcode_bsf:
      case pic_opcode_btfsc:
      case pic_opcode_btfss:
      case pic_opcode_tris:
      case pic_opcode_datalo_set:
      case pic_opcode_datalo_clr:
      case pic_opcode_datahi_set:
      case pic_opcode_datahi_clr:
      case pic_opcode_irp_set:
      case pic_opcode_irp_clr:
      case pic_opcode_branchlo_set:
      case pic_opcode_branchlo_clr:
      case pic_opcode_branchlo_nop:
      case pic_opcode_branchhi_set:
      case pic_opcode_branchhi_clr:
      case pic_opcode_branchhi_nop:
      case pic_opcode_mullw:
      case pic_opcode_dcfsnz:
      case pic_opcode_infsnz:
      case pic_opcode_cpfseq:
      case pic_opcode_cpfsgt:
      case pic_opcode_cpfslt:
      case pic_opcode_mulwf:
      case pic_opcode_negf:
      case pic_opcode_setf:
      case pic_opcode_tstfsz:
      case pic_opcode_btg:
      case pic_opcode_daw:
      case pic_opcode_pop:
      case pic_opcode_push:
      case pic_opcode_reset:
      case pic_opcode_lfsr:
      case pic_opcode_movff:
      case pic_opcode_movlb:
      case pic_opcode_tblrd:
      case pic_opcode_tblwt:
      case pic_opcode_db:
        break; /* these don't change W */
      case pic_opcode_addwfc:
      case pic_opcode_addwf:
      case pic_opcode_andwf:
      case pic_opcode_xorwf:
      case pic_opcode_iorwf:
      case pic_opcode_subwf:
      case pic_opcode_comf:
      case pic_opcode_decf:
      case pic_opcode_decfsz:
      case pic_opcode_incf:
      case pic_opcode_incfsz:
      case pic_opcode_rlcf:
      case pic_opcode_rlf:
      case pic_opcode_rrf:
      case pic_opcode_rlncf:
      case pic_opcode_rrncf:
      case pic_opcode_subfwb:
      case pic_opcode_subwfb:
      case pic_opcode_swapf:
        /* if the destination of these is W, the result is unknown */
        if (pic_opdst_w == pic_code_dst_get(code)) {
          w_value = VALUE_NONE;
        } else if ((pic_opdst_f == pic_code_dst_get(code))
          && value_is_same(pic_code_value_get(code), 
            pic_code_w_value_get(code))) {
          w_value = VALUE_NONE;
        }
        break;
      case pic_opcode_movwf:
        /* if W goes into this we'll assume W has a new value (we only
         * track one value for W).
         * nb: cannot assume anything about volatile variables! */
        w_value = (!value_is_volatile(pic_code_value_get(code)))
          ? pic_code_value_get(code)
          : VALUE_NONE;
        break;
      case pic_opcode_movf:
        /* if this goes into W we know what W contains; */
        w_value = (!value_is_volatile(pic_code_value_get(code))
            && (pic_opdst_w == pic_code_dst_get(code)))
          ? pic_code_value_get(code)
          : VALUE_NONE;
        break;
      case pic_opcode_clrw:
      case pic_opcode_addlw:
      case pic_opcode_andlw:
      case pic_opcode_iorlw:
      case pic_opcode_movlw:
      case pic_opcode_sublw:
      case pic_opcode_xorlw:
        w_value = VALUE_NONE;
        break;
      case pic_opcode_retlw:
        w_value = VALUE_NONE;
        /* fall through */
      case pic_opcode_retfie:
      case pic_opcode_return:
        /* we're done here */
        next = PIC_CODE_NONE;
        break;
      case pic_opcode_goto:   /* goto n */
      case pic_opcode_call:   /* call n */
      case pic_opcode_bc:
      case pic_opcode_bn:
      case pic_opcode_bnc:
      case pic_opcode_bnn:
      case pic_opcode_bnov:
      case pic_opcode_bnz:
      case pic_opcode_bov:
      case pic_opcode_bz:
      case pic_opcode_bra:
      case pic_opcode_rcall:
        {
          pic_code_t dst;

          dst = pic_code_label_find(pf, pic_code_brdst_get(code));
          if ((pic_opcode_call == op) || pv_code_is_cond) {
            w_value = pic_code_w_value_detect(pf, dst, w_value);
            if (pic_opcode_call == op) {
              /* after a call, assume we don't know the value of W */
              w_value = VALUE_NONE;
            }
          } else {
            next = dst;
          }
        }
        break;
    }
    pv_code_is_cond = code_is_cond;
    code = next;
  }
  return w_value;
}

/*
 * NAME
 *   pic_w_cleanup
 *
 * DESCRIPTION
 *   after determining what W holds, look for assignments from the same
 *   value & remove these
 *
 * PARAMETERS
 *   pf : pfile handle
 *
 * RETURN
 *   number of instructions removed
 *
 * NOTES
 *   this also looks for values that are no longer used and removes
 *   assignments to them.
 */
static unsigned pic_w_cleanup(pfile_t *pf)
{
  pic_code_t code;
  pic_code_t next;
  unsigned   rem_ct;
  value_t    zero;

  rem_ct = 0;
  zero = pfile_constant_get(pf, 0, VARIABLE_DEF_NONE);
  for (code = pic_code_list_head_get(pf);
       code;
       code = next) {
    value_t val;
    value_t w_val;

    next  = pic_code_next_get(code);
    val   = pic_code_value_get(code);
    w_val = pic_code_w_value_get(code);

    if (val
        && !value_is_volatile(val)
        && !pic_code_flag_test(code, PIC_CODE_FLAG_NO_OPTIMIZE)) {
      pic_opcode_t op;

      op = pic_code_op_get(code);
      if ((pic_opcode_movf == op)
        && (pic_opdst_w == pic_code_dst_get(code))  
        && value_is_same(val, w_val)) {
        value_use_ct_bump(w_val, ctr_bump_decr);
        if (pic_code_z_used(pf, pic_code_next_get(code))) {
          /* change this to iorlw to make sure _status:_z is set correctly */
          pic_code_op_set(code, pic_opcode_iorlw);
          pic_code_value_set(code, VALUE_NONE);
          pic_code_literal_set(code, zero);
        } else {
          pic_code_list_remove(pf, code);
          pic_code_free(code);
          rem_ct++;
        }
#if 1
      } else if (0 == value_use_ct_get(val)) {
        variable_t master;

        for (master = variable_master_get(value_variable_get(val));
             master && !variable_use_ct_get(master);
             master = variable_master_get(master))
          ; /* empty loop */

        if (!master) {
          if ((pic_opcode_datalo_set != op)
            && (pic_opcode_datalo_clr != op)
            && (pic_opcode_datahi_set != op)
            && (pic_opcode_datahi_clr != op)) {
            value_assign_ct_bump(val, ctr_bump_decr);
          }
          pic_code_list_remove(pf, code);
          pic_code_free(code);
          rem_ct++;
        }
#endif
      }
    }
  }
  value_release(zero);
  return rem_ct;
}

void pic_w_value_optimize(pfile_t *pf)
{
  pfile_proc_t *proc;
  pic_code_t    code;
  label_t       lbl;

  for (code = pic_code_list_head_get(pf);
       code;
       code = pic_code_next_get(code)) {
    pic_code_flag_clr(code, PIC_CODE_FLAG_W_VISITED);
  }
  /* first mark the use code */
  pic_code_w_value_detect(pf, pic_code_list_head_get(pf), VALUE_NONE);
  /* next the ISRs */
  lbl = pfile_label_find(pf, pfile_log_none, PIC_LABEL_ISR);
  if (lbl) {
    pic_code_w_value_detect(pf, pic_code_label_find(pf, lbl), VALUE_NONE);
    label_release(lbl);
  }
  /* finally, any indirect or tasks */
  for (proc = pfile_proc_root_get(pf);
       proc;
       proc = pfile_proc_next(proc)) {
    if (pfile_proc_flag_test(proc, PFILE_PROC_FLAG_INDIRECT)
        || pfile_proc_flag_test(proc, PFILE_PROC_FLAG_TASK)) {
      pic_code_w_value_detect(pf, 
          pic_code_label_find(pf, pfile_proc_label_get(proc)), VALUE_NONE);
    }
  }

  if (pfile_flag_test(pf, PFILE_FLAG_OPT_LOAD_REDUCE)) {
    unsigned total;
    unsigned removed;
    unsigned pass;

    total = 0;
    pass  = 1;
    while ((removed = pic_w_cleanup(pf)) != 0) {
      total += removed;
      pass++;
    }
    printf("CODE (%u) removed in (%u) passes\n", total, pass);
  }
}

