/**********************************************************
 **
 ** cmd_asm.c : manipulators for cmd_asm_t
 **
 ** Copyright (c) 2005, Kyle A. York
 ** All rights reserved
 **
 ***********************************************************/
#include <stdio.h>
#include <assert.h>
#include "../libutils/mem.h"
#include "../libpic12/pic_inst.h"
#include "cmdd.h"
#include "cmd_asm.h"

static void cmd_asm_dump_value(value_t val, const char *str, FILE *dst)
{
  if (val) {
    if (value_name_get(val)) {
      fprintf(dst, "%s%s", str, value_name_get(val));
    } else {
      fprintf(dst, "%s%u", str, (unsigned) value_const_get(val));
    }
  }
}

static void cmd_asm_dump(cmd_t cmd, FILE *dst)
{
  pic_optype_t type;

  fprintf(dst, "%s", pic_opcode_str(cmd_asm_op_get(cmd)));
  type = pic_optype_get(cmd_asm_op_get(cmd));
  switch (type) {
    case pic_optype_none:
      break;
    case pic_optype_f:
    case pic_optype_f_d:
    case pic_optype_f_b:
      cmd_asm_dump_value(cmd_asm_val_get(cmd), " ", dst);
      if (pic_optype_f_d == type) {
        fprintf(dst, ", %s", pic_opdst_str(cmd_asm_opdst_get(cmd)));
      } else if (pic_optype_f_b == type) {
        cmd_asm_dump_value(cmd_asm_n_get(cmd), ", ", dst);
      }
      break;
    case pic_optype_n:
    case pic_optype_tris:
      cmd_asm_dump_value(cmd_asm_n_get(cmd), " ", dst);
      break;
    case pic_optype_k:
      if (cmd_asm_lbl_get(cmd)) {
        fprintf(dst, " %s", label_name_get(cmd_asm_lbl_get(cmd)));
      } else {
        cmd_asm_dump_value(cmd_asm_n_get(cmd), " ", dst);
      }
      break;
    case pic_optype_db:
      {
        uchar *data;
        size_t data_sz;
        size_t ii;

        data_sz = cmd_asm_data_sz_get(cmd);
        data    = cmd_asm_data_get(cmd);
        for (ii = 0; ii < data_sz; ii++) {
          fprintf(dst, "%u%s", data[ii], (ii + 1 < data_sz) ? "," : "");
        }
      }
      break;
  }
}

static void cmd_asm_free(cmd_t cmd)
{
  cmd_asm_val_set(cmd, VALUE_NONE);
  cmd_asm_n_set(cmd, VALUE_NONE);
  cmd_asm_lbl_set(cmd, LABEL_NONE);
  cmd_asm_data_set(cmd, 0, 0);
}

static cmd_t cmd_asm_dup(const cmd_t cmd)
{
  return cmd_asm_alloc(cmd_asm_op_get(cmd), cmd_asm_val_get(cmd),
    cmd_asm_valofs_get(cmd),
    cmd_asm_opdst_get(cmd), cmd_asm_n_get(cmd), cmd_asm_lbl_get(cmd),
    cmd_asm_flag_get_all(cmd),
    cmd_asm_data_sz_get(cmd),
    cmd_asm_data_get(cmd));
}

static void cmd_asm_label_remap(cmd_t cmd, const label_map_t *map)
{
  cmd_label_remap2(cmd, map, cmd_asm_lbl_get, cmd_asm_lbl_set);
}

static void cmd_asm_variable_remap(cmd_t cmd, const variable_map_t *map)
{
  cmd_variable_remap2(cmd, map, cmd_asm_val_get, cmd_asm_val_set);
  cmd_variable_remap2(cmd, map, cmd_asm_n_get, cmd_asm_n_set);
}

static void cmd_asm_value_remap(cmd_t cmd, const value_map_t *map)
{
  cmd_value_remap2(cmd, map, cmd_asm_val_get, cmd_asm_val_set);
  cmd_value_remap2(cmd, map, cmd_asm_n_get, cmd_asm_n_set);
}

static flag_t cmd_asm_variable_accessed(const cmd_t cmd, variable_t var)
{
  return ((value_variable_get(cmd_asm_val_get(cmd)) == var)
        || (value_variable_get(cmd_asm_n_get(cmd)) == var))
    ? CMD_VARIABLE_ACCESS_FLAG_READ | CMD_VARIABLE_ACCESS_FLAG_WRITTEN
    : CMD_VARIABLE_ACCESS_FLAG_NONE;
}

static flag_t cmd_asm_value_accessed(const cmd_t cmd, value_t val)
{
  return (cmd_asm_val_get(cmd) == val)
        || (cmd_asm_n_get(cmd) == val)
    ? CMD_VARIABLE_ACCESS_FLAG_READ | CMD_VARIABLE_ACCESS_FLAG_WRITTEN
    : CMD_VARIABLE_ACCESS_FLAG_NONE;
}

static void cmd_asm_assigned_use_set(cmd_t cmd)
{
  variable_t var;

  var = value_variable_get(cmd_asm_n_get(cmd));
  if ((VARIABLE_NONE != var) && !variable_is_const(var)) {
    switch (cmd_asm_op_get(cmd)) {
      case pic_opcode_org:
      case pic_opcode_end:
      case pic_opcode_none:
      case pic_opcode_tris:
      case pic_opcode_bc:
      case pic_opcode_bn:
      case pic_opcode_bnc:
      case pic_opcode_bnn:
      case pic_opcode_bnov:
      case pic_opcode_bnz:
      case pic_opcode_bov:
      case pic_opcode_bz:
      case pic_opcode_bra:
      case pic_opcode_rcall:
      case pic_opcode_call:
      case pic_opcode_goto:
      case pic_opcode_clrw:
      case pic_opcode_clrwdt:
      case pic_opcode_daw:
      case pic_opcode_nop:
      case pic_opcode_option:
      case pic_opcode_pop:
      case pic_opcode_push:
      case pic_opcode_reset:
      case pic_opcode_return:
      case pic_opcode_retfie:
      case pic_opcode_sleep:
      case pic_opcode_tblrd:
      case pic_opcode_tblwt:
      case pic_opcode_db:
      case pic_opcode_addlw:
      case pic_opcode_andlw:
      case pic_opcode_iorlw:
      case pic_opcode_movlw:
      case pic_opcode_mullw:
      case pic_opcode_retlw:
      case pic_opcode_sublw:
      case pic_opcode_xorlw:
      case pic_opcode_branchlo_set:
      case pic_opcode_branchlo_clr:
      case pic_opcode_branchlo_nop:
      case pic_opcode_branchhi_set:
      case pic_opcode_branchhi_clr:
      case pic_opcode_branchhi_nop:
        break; /* should never get here */
      case pic_opcode_addwf:
      case pic_opcode_addwfc:
      case pic_opcode_andwf:
      case pic_opcode_comf:
      case pic_opcode_decf:
      case pic_opcode_dcfsnz:
      case pic_opcode_decfsz:
      case pic_opcode_incf:
      case pic_opcode_incfsz:
      case pic_opcode_infsnz:
      case pic_opcode_iorwf:
      case pic_opcode_movf:
      case pic_opcode_rlf:
      case pic_opcode_rlcf:
      case pic_opcode_rlncf:
      case pic_opcode_rrf:
      case pic_opcode_rrncf:
      case pic_opcode_subfwb:
      case pic_opcode_subwf:
      case pic_opcode_subwfb:
      case pic_opcode_swapf:
      case pic_opcode_xorwf:
        cmd_gen_add(cmd, var);
        if (pic_opdst_f == cmd_asm_opdst_get(cmd)) {
          cmd_kill_add(cmd, var);
        }
        break;
      case pic_opcode_cpfseq:
      case pic_opcode_cpfsgt:
      case pic_opcode_cpfslt:
      case pic_opcode_mulwf:
      case pic_opcode_tstfsz:
      case pic_opcode_btfsc:
      case pic_opcode_btfss:
      case pic_opcode_lfsr:
      case pic_opcode_movlb:
      case pic_opcode_datalo_set:
      case pic_opcode_datalo_clr:
      case pic_opcode_datahi_set:
      case pic_opcode_datahi_clr:
      case pic_opcode_irp_set:
      case pic_opcode_irp_clr:
        cmd_gen_add(cmd, var);
        break;
      case pic_opcode_negf:
      case pic_opcode_btg:
        cmd_gen_add(cmd, var);
        cmd_kill_add(cmd, var);
        break;
      case pic_opcode_setf:
      case pic_opcode_bcf:
      case pic_opcode_bsf:
      case pic_opcode_movwf:
      case pic_opcode_clrf:
        cmd_kill_add(cmd, var);
        break;
      case pic_opcode_movff:
        cmd_gen_add(cmd, var);
        /* cmd_kill_add(cmd, var2); */
        break;
    }
  }
}

static cmd_successor_rc_t cmd_asm_successor_get(cmd_t cmd, size_t ix, 
  cmd_t *dst)
{
  cmd_successor_rc_t rc;

  rc = CMD_SUCCESSOR_RC_IX_BAD;
  switch (cmd_asm_op_get(cmd)) {
    case pic_opcode_org:
    case pic_opcode_end:
    case pic_opcode_none:
    case pic_opcode_tris:
    case pic_opcode_clrw:
    case pic_opcode_clrwdt:
    case pic_opcode_daw:
    case pic_opcode_nop:
    case pic_opcode_option:
    case pic_opcode_pop:
    case pic_opcode_push:
    case pic_opcode_reset:
    case pic_opcode_return:
    case pic_opcode_retfie:
    case pic_opcode_sleep:
    case pic_opcode_tblrd:
    case pic_opcode_tblwt:
    case pic_opcode_db:
    case pic_opcode_addlw:
    case pic_opcode_andlw:
    case pic_opcode_iorlw:
    case pic_opcode_movlw:
    case pic_opcode_mullw:
    case pic_opcode_retlw:
    case pic_opcode_sublw:
    case pic_opcode_xorlw:
    case pic_opcode_branchlo_set:
    case pic_opcode_branchlo_clr:
    case pic_opcode_branchlo_nop:
    case pic_opcode_branchhi_set:
    case pic_opcode_branchhi_clr:
    case pic_opcode_branchhi_nop:
    case pic_opcode_addwf:
    case pic_opcode_addwfc:
    case pic_opcode_andwf:
    case pic_opcode_comf:
    case pic_opcode_decf:
    case pic_opcode_incf:
    case pic_opcode_iorwf:
    case pic_opcode_movf:
    case pic_opcode_rlf:
    case pic_opcode_rlcf:
    case pic_opcode_rlncf:
    case pic_opcode_rrf:
    case pic_opcode_rrncf:
    case pic_opcode_subfwb:
    case pic_opcode_subwf:
    case pic_opcode_subwfb:
    case pic_opcode_swapf:
    case pic_opcode_xorwf:
    case pic_opcode_mulwf:
    case pic_opcode_lfsr:
    case pic_opcode_movlb:
    case pic_opcode_datalo_set:
    case pic_opcode_datalo_clr:
    case pic_opcode_datahi_set:
    case pic_opcode_datahi_clr:
    case pic_opcode_irp_set:
    case pic_opcode_irp_clr:
    case pic_opcode_negf:
    case pic_opcode_btg:
    case pic_opcode_setf:
    case pic_opcode_bcf:
    case pic_opcode_bsf:
    case pic_opcode_movwf:
    case pic_opcode_clrf:
    case pic_opcode_movff:
      if (0 == ix) {
        *dst = cmd_link_get(cmd);
        rc = CMD_SUCCESSOR_RC_DONE;
      } 
      break;
    case pic_opcode_dcfsnz:
    case pic_opcode_decfsz:
    case pic_opcode_incfsz:
    case pic_opcode_infsnz:
    case pic_opcode_cpfseq:
    case pic_opcode_cpfsgt:
    case pic_opcode_cpfslt:
    case pic_opcode_tstfsz:
    case pic_opcode_btfsc:
    case pic_opcode_btfss:
      /* successor is both the next command or the command that follows */
      if (0 == ix) {
        *dst = cmd_link_get(cmd);
        rc = CMD_SUCCESSOR_RC_MORE;
      } else if (1 == ix) {
        *dst = cmd_link_get(cmd_link_get(cmd));
        rc = CMD_SUCCESSOR_RC_DONE;
      }
      break;
    case pic_opcode_bc:
    case pic_opcode_bn:
    case pic_opcode_bnc:
    case pic_opcode_bnn:
    case pic_opcode_bnov:
    case pic_opcode_bnz:
    case pic_opcode_bov:
    case pic_opcode_bz:
    case pic_opcode_bra:
    case pic_opcode_rcall:
    case pic_opcode_call:
      /* successor is both the next command or the label */
      if (0 == ix) {
        *dst = cmd_link_get(cmd);
        rc = CMD_SUCCESSOR_RC_MORE;
      } else if (1 == ix) {
        *dst = label_cmd_get(cmd_asm_lbl_get(cmd));
        rc = CMD_SUCCESSOR_RC_DONE;
      }
      break;
    case pic_opcode_goto:
      /* successor is only the label */
      if (0 == ix) {
        *dst = label_cmd_get(cmd_asm_lbl_get(cmd));
        rc = CMD_SUCCESSOR_RC_DONE;
      }
      break;
  }
  return rc;
}

static const cmd_vtbl_t cmd_asm_vtbl = {
  cmd_asm_free,
  cmd_asm_dump,
  cmd_asm_dup,
  cmd_asm_label_remap,
  cmd_asm_variable_remap,
  cmd_asm_value_remap,
  cmd_asm_variable_accessed,
  cmd_asm_value_accessed,
  cmd_asm_assigned_use_set,
  cmd_asm_successor_get
};

cmd_t cmd_asm_alloc(pic_opcode_t op, value_t val, size_t valofs,
    pic_opdst_t opdst, value_t n, label_t lbl, flag_t flags,
    size_t data_sz, uchar *data)
{
  cmd_t cmd;

  cmd = cmd_alloc(cmd_type_asm, &cmd_asm_vtbl);
  if (cmd) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      ptr->u.op_asm.op      = op;
      ptr->u.op_asm.val     = VALUE_NONE;
      ptr->u.op_asm.valofs  = valofs;
      ptr->u.op_asm.opdst   = opdst;
      ptr->u.op_asm.n       = VALUE_NONE;
      ptr->u.op_asm.lbl     = LABEL_NONE;
      ptr->u.op_asm.flags   = flags;
      ptr->u.op_asm.data_sz = 0;
      ptr->u.op_asm.data    = 0;
      
      cmd_asm_val_set(cmd, val);
      cmd_asm_n_set(cmd, n);
      cmd_asm_lbl_set(cmd, lbl);
      cmd_asm_data_set(cmd, data_sz, data);
    }
  }
  return cmd;
}

pic_opcode_t cmd_asm_op_get(const cmd_t cmd)
{
  struct cmd_ *ptr;

  ptr = 0;
  if (cmd_type_asm == cmd_type_get(cmd)) {
    ptr = cmd_element_seek(cmd, boolean_false);
  }
  return (ptr) ? ptr->u.op_asm.op : pic_opcode_none;
}

void cmd_asm_op_set(cmd_t cmd, pic_opcode_t op)
{
  assert(cmd_type_asm == cmd_type_get(cmd));
  if (cmd_type_asm == cmd_type_get(cmd)) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      ptr->u.op_asm.op = op;
    }
  }
}

value_t cmd_asm_val_get(const cmd_t cmd)
{
  const struct cmd_ *ptr;

  ptr = 0;
  if (cmd_type_asm == cmd_type_get(cmd)) {
    ptr = cmd_element_seek(cmd, boolean_false);
  }
  return (ptr) ? ptr->u.op_asm.val : VALUE_NONE;
}

void cmd_asm_val_set(cmd_t cmd, value_t val)
{
  struct cmd_ *ptr;

  assert(cmd_type_asm == cmd_type_get(cmd));
  if (cmd_type_asm == cmd_type_get(cmd)) {
    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      /* if a varaible is used in an assembly statement, mark it both
       * assigned *and* used */
      value_lock(val);
      value_assign_ct_bump(ptr->u.op_asm.val, ctr_bump_decr);
      value_use_ct_bump(ptr->u.op_asm.val, ctr_bump_decr);
      value_release(ptr->u.op_asm.val);
      ptr->u.op_asm.val = val;
      value_assign_ct_bump(val, ctr_bump_incr);
      value_use_ct_bump(val, ctr_bump_incr);
    }
  }
}

size_t cmd_asm_valofs_get(const cmd_t cmd)
{
  const struct cmd_ *ptr;

  ptr = 0;
  if (cmd_type_asm == cmd_type_get(cmd)) {
    ptr = cmd_element_seek(cmd, boolean_false);
  }
  return (ptr) ? ptr->u.op_asm.valofs : 0;
}

void cmd_asm_valofs_set(cmd_t cmd, size_t valofs)
{

  assert(cmd_type_asm == cmd_type_get(cmd));
  if (cmd_type_asm == cmd_type_get(cmd)) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      /* if a varaible is used in an assembly statement, mark it both
       * assigned *and* used */
      ptr->u.op_asm.valofs = valofs;
    }
  }
}

pic_opdst_t  cmd_asm_opdst_get(const cmd_t cmd)
{
  const struct cmd_ *ptr;

  ptr = 0;
  if (cmd_type_asm == cmd_type_get(cmd)) {
    ptr = cmd_element_seek(cmd, boolean_false);
  }
  return (ptr) ? ptr->u.op_asm.opdst : pic_opdst_none;
}

void cmd_asm_opdst_set(cmd_t cmd, pic_opdst_t dst)
{
  assert(cmd_type_asm == cmd_type_get(cmd));
  if (cmd_type_asm == cmd_type_get(cmd)) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      ptr->u.op_asm.opdst = dst;
    }
  }
}

value_t cmd_asm_n_get(const cmd_t cmd)
{
  const struct cmd_ *ptr;

  ptr = 0;
  if (cmd_type_asm == cmd_type_get(cmd)) {
    ptr = cmd_element_seek(cmd, boolean_false);
  }
  return (ptr) ? ptr->u.op_asm.n : VALUE_NONE;
}

void         cmd_asm_n_set(cmd_t cmd, value_t n)
{
  assert(cmd_type_asm == cmd_type_get(cmd));
  if (cmd_type_asm == cmd_type_get(cmd)) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      value_lock(n);
      value_use_ct_bump(ptr->u.op_asm.n, ctr_bump_decr);
      value_release(ptr->u.op_asm.n);
      ptr->u.op_asm.n = n;
      value_use_ct_bump(n, ctr_bump_incr);
    }
  }
}

label_t cmd_asm_lbl_get(const cmd_t cmd)
{
  struct cmd_ *ptr;

  ptr = 0;
  if (cmd_type_asm == cmd_type_get(cmd)) {
    ptr = cmd_element_seek(cmd, boolean_false);
  }
  return (ptr) ? ptr->u.op_asm.lbl : LABEL_NONE;
}

void cmd_asm_lbl_set(cmd_t cmd, label_t lbl)
{
  assert(cmd_type_asm == cmd_type_get(cmd));
  if (cmd_type_asm == cmd_type_get(cmd)) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      label_lock(lbl);
      label_usage_bump(ptr->u.op_asm.lbl, ctr_bump_decr);
      label_release(ptr->u.op_asm.lbl);
      ptr->u.op_asm.lbl = lbl;
      label_usage_bump(lbl, ctr_bump_incr);
    }
  }
}

boolean_t cmd_asm_flag_test(const cmd_t cmd, flag_t flag)
{
  const struct cmd_ *ptr;

  ptr = 0;
  if (cmd_type_asm == cmd_type_get(cmd)) {
    ptr = cmd_element_seek(cmd, boolean_false);
  }
  return (ptr) ? (ptr->u.op_asm.flags & flag) == flag : boolean_false;
}

flag_t cmd_asm_flag_get_all(const cmd_t cmd)
{
  const struct cmd_ *ptr;

  ptr = 0;
  if (cmd_type_asm == cmd_type_get(cmd)) {
    ptr = cmd_element_seek(cmd, boolean_false);
  }
  return (ptr) ? ptr->u.op_asm.flags : 0;
}

size_t cmd_asm_data_sz_get(const cmd_t cmd)
{
  const struct cmd_ *ptr;

  ptr = 0;
  if (cmd_type_asm == cmd_type_get(cmd)) {
    ptr = cmd_element_seek(cmd, boolean_false);
  }
  return (ptr) ? ptr->u.op_asm.data_sz : 0;
}

uchar *cmd_asm_data_get(const cmd_t cmd)
{
  const struct cmd_ *ptr;

  ptr = 0;
  if (cmd_type_asm == cmd_type_get(cmd)) {
    ptr = cmd_element_seek(cmd, boolean_false);
  }
  return (ptr) ? ptr->u.op_asm.data : 0;
}

void   cmd_asm_data_set(cmd_t cmd, size_t sz, uchar *data)
{
  assert(cmd_type_asm == cmd_type_get(cmd));
  if (cmd_type_asm == cmd_type_get(cmd)) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      FREE(ptr->u.op_asm.data);
      ptr->u.op_asm.data_sz = sz;
      ptr->u.op_asm.data    = data;
    }
  }
}

