/**********************************************************
 **
 ** cmdoptmz.c : command optimizier functions
 **
 ** Copyright (c) 2004-2005, Kyle A. York
 ** All rights reserved
 **
 ***********************************************************/
#include <assert.h>
#include <string.h>

#include "pfile.h"
#include "pf_proc.h"
#include "pf_op.h"
#include "cmd_brch.h"
#include "cmd_op.h"
#include "cmd_optm.h"

/*
 * NAME
 *   cmd_opt_branch
 *
 * DESCRIPTION
 *   basic branch optimization
 *
 * PARAMETERS
 *   cmd : head of the cmd list
 *
 * RETURN
 *   none
 *
 * NOTES
 *   first, the true branch destination is determined by following
 *     any chains of absolute GOTOs
 *   next:
 *     1. call to a return --> removed
 *     2. call immediately followed by return --> call changed to goto
 *     3. goto a return --> goto changed to return
 *     4. if (cond) goto l1
 *        absolute branch
 *        l1:
 *        ...
 *        --> if (!cond) goto l2
 *     5. goto l1 (or if cond goto l1)
 *        l1:
 *        remove!
 */
/* on entry, cmd points to a GOTO or a CALL
 * the returned cmd is the *true* destination (aka, this follows
 * a string of absolute GOTOs until it reaches the end) */
static cmd_t cmd_branch_true_dest_get(cmd_t cmd, cmd_t *cmd_head, label_t *dlbl)
{
  label_t lbl;
  unsigned opt;

  opt = cmd_opt_get(cmd) + 1;

  do {
    cmd_opt_set(cmd, opt);
    lbl = cmd_brdst_get(cmd);
    cmd = cmd_next_exec_get(cmd_label_find(*cmd_head, lbl));
  } while ((cmd_branchtype_goto == cmd_brtype_get(cmd))
    && (cmd_branchcond_none == cmd_brcond_get(cmd))
    && (opt != cmd_opt_get(cmd)));
  *dlbl = lbl;
  return cmd;
}

boolean_t cmd_opt_branch(cmd_t *cmd_head)
{
  cmd_t cmd;
  unsigned rem_ct;
  cmd_t    cmd_prev;

  /* first, move cmd to the first *executable* instruction */
  rem_ct = 0;
  cmd = cmd_next_exec_get(*cmd_head);
  cmd_prev = CMD_NONE;

  while (cmd) {
    cmd_t cmd_next;

    cmd_next = cmd_link_get(cmd);
    if (value_is_const(cmd_brval_get(cmd))
        && ((cmd_branchcond_true == cmd_brcond_get(cmd))
          || (cmd_branchcond_false == cmd_brcond_get(cmd)))) {
      /* we know the condition, can we remove the instruction? */
      boolean_t rem;

      if (value_const_get(cmd_brval_get(cmd))) {
        rem = cmd_branchcond_false == cmd_brcond_get(cmd);
      } else {
        rem = cmd_branchcond_true == cmd_brcond_get(cmd);
      }
      if (rem) {
        cmd_remove(cmd_head, cmd);
        rem_ct++;
        cmd = CMD_NONE;
      } else {
        cmd_brcond_set(cmd, cmd_branchcond_none);
      }
    }
    if ((cmd_branchtype_goto == cmd_brtype_get(cmd))
      || (cmd_branchtype_call == cmd_brtype_get(cmd))
      || (cmd_branchtype_task_start == cmd_brtype_get(cmd))) {
      /* find the TRUE destination (aka, follow the call chain) */
      cmd_t   cmd_lbl_exec;
      label_t lbl;

      cmd_lbl_exec = cmd_branch_true_dest_get(cmd, cmd_head, &lbl);
      /* lbl == TRUE destination
       * cmd_lbl_exec = first executable statement after lbl */
      cmd_brdst_set(cmd, lbl);
      if ((cmd_branchtype_return == cmd_brtype_get(cmd_lbl_exec))
        && (cmd_branchcond_none == cmd_brcond_get(cmd_lbl_exec))) {
        if (cmd_branchtype_goto == cmd_brtype_get(cmd)) {
          /* GOTO destination is a RETURN; replace GOTO with RETURN */
          cmd_brtype_set(cmd, cmd_branchtype_return);
        } else {
          /* CALL destination is a RETURN; delete the CALL */
          cmd_remove(cmd_head, cmd);
          rem_ct++;
        }
      } else if (cmd_branchtype_call == cmd_brtype_get(cmd)) {
        /* look at the next exectuteable statement, if it's a return,
         * change the CALL to a GOTO */
        cmd_t cmd_ptr;

        cmd_ptr = cmd_next_exec_get(cmd_link_get(cmd));
        if ((cmd_branchtype_return == cmd_brtype_get(cmd_ptr))
            && (cmd_branchcond_none == cmd_brcond_get(cmd_ptr))) {
          cmd_brtype_set(cmd, cmd_branchtype_goto);
        }
      }
      if (cmd_branchtype_goto == cmd_brtype_get(cmd)) {
        /* is this goto 1 forward? if so, get rid of it */
        cmd_t cmd_ptr;

        for (cmd_ptr = cmd_link_get(cmd);
             cmd_ptr
             && !cmd_is_executable(cmd_ptr)
             && (cmd_label_get(cmd_ptr) != cmd_brdst_get(cmd));
             cmd_ptr = cmd_link_get(cmd_ptr))
          ; /* null body */
        if (cmd_label_get(cmd_ptr) == cmd_brdst_get(cmd)) {
          /* goto following label --> remove cmd */
          cmd_remove(cmd_head, cmd);
          rem_ct++;
        } else if ((cmd_branchcond_none != cmd_brcond_get(cmd))
          && (cmd_branchtype_goto == cmd_brtype_get(cmd_ptr))
          && (cmd_branchcond_none == cmd_brcond_get(cmd_ptr))) {
          /* so far we've 
           *   if (cond) goto l1:
           *   goto l2:
           *   ???
           */
          cmd_t cmd_ptr2;

          for (cmd_ptr2 = cmd_link_get(cmd_ptr);
               cmd_ptr2
               && !cmd_is_executable(cmd_ptr2)
               && (cmd_label_get(cmd_ptr2) != cmd_brdst_get(cmd));
               cmd_ptr2 = cmd_link_get(cmd_ptr2))
            ; /* null body */
          if (cmd_label_get(cmd_ptr2) == cmd_brdst_get(cmd)) {
#if 0
            /* 
             * change the condition on cmd
             * remove cmd_ptr
             */
            cmd_brcond_set(cmd,
                (cmd_branchcond_true == cmd_brcond_get(cmd))
                ? cmd_branchcond_false
                : cmd_branchcond_true);
            cmd_brdst_set(cmd, cmd_brdst_get(cmd_ptr));
            if (cmd_next == cmd_ptr) {
              cmd_next = cmd_link_get(cmd_ptr);
            }
            cmd_remove(cmd_head, cmd_ptr);
            rem_ct++;
            /* we need to re-evaluate cmd! */
            cmd_next = cmd;
#endif
          }
        }
      }
    }
    if (cmd != cmd_next) {
      cmd_prev = cmd;
    }
    cmd = cmd_next;
  }
  return rem_ct != 0;
}

/*
 * look for:
 *   _temp = !x
 *   branch [true|false] _temp, ...
 *   --> branch [false|true] x
 *
 *   _temp = !!x
 *   branch [true|false] _temp, ...
 *   --> branch [true|false] x
 */
boolean_t cmd_opt_branch2(cmd_t *cmd_head)
{
  cmd_t    cmd;
  cmd_t    cmd_pv;
  unsigned rem_ct;

  rem_ct = 0;

  cmd_pv = CMD_NONE;
  cmd    = cmd_next_exec_get(*cmd_head);
  while (cmd) {
    cmd_t cmd_next;

    cmd_next = cmd_next_exec_get(cmd_link_get(cmd));
    if (((operator_notl == cmd_optype_get(cmd)
        || operator_logical == cmd_optype_get(cmd)))
      && value_is_temp(cmd_opdst_get(cmd))
      && value_is_same(cmd_opdst_get(cmd), cmd_brval_get(cmd_next))
      && ((cmd_branchcond_true == cmd_brcond_get(cmd_next))
        || (cmd_branchcond_false == cmd_brcond_get(cmd_next)))) {
      /* remove cmd */
      cmd_branchcond_t brcond;

      brcond = cmd_brcond_get(cmd_next);
      if (operator_notl == cmd_optype_get(cmd)) {
        /* invert the condition */
        brcond = (cmd_branchcond_true == cmd_brcond_get(cmd_next))
            ? cmd_branchcond_false
            : cmd_branchcond_true;
      }
      if (cmd_pv) {
        /*
         * cmd_pv points to the previous *executable* statement,
         * it need to move to the actual previous statement
         */  
        while (cmd_link_get(cmd_pv) != cmd) {
          cmd_pv = cmd_link_get(cmd_pv);
        }
        cmd_link_set(cmd_pv, cmd_link_get(cmd));
      } else {
        *cmd_head = cmd_link_get(cmd);
      }
      cmd_brcond_set(cmd_next, brcond);
      cmd_brval_set(cmd_next, cmd_opval1_get(cmd));
      cmd_free(cmd);
      rem_ct++;
    } else {
      cmd_pv = cmd;
    }
    cmd = cmd_next;
  }
  return rem_ct != 0;
}

/*
 * NAME
 *   cmd_opt_branch4
 *
 * DESCRIPTION
 *   do branch4 optimizations (see below)
 *
 * PARAMETERS
 *   cmd_head : start of cmd list
 *
 * RETURN
 *   boolean_true : something was changed
 *
 * NOTES
 *   This looks for operations of the form:
 *     val = cexpr
 *     goto [true|false] val, label
 *   and either changes the goto to an unconditional, or removes it  
 *   then, if val doesn't exist or is temporary & not further used,
 *   it will also be removed
 */
boolean_t cmd_opt_branch4(pfile_t *pf, cmd_t *cmd_head)
{
  cmd_t    cmd;
  cmd_t    cmd_pv;
  unsigned rem_ct;

  rem_ct = 0;

  cmd_pv = CMD_NONE;
  cmd    = *cmd_head; /* cmd_next_exec_get(*cmd_head); */
  while (cmd) {
    cmd_t cmd_next;

    cmd_next = cmd_link_get(cmd);
    if ((cmd_branchcond_none != cmd_brcond_get(cmd_next))
        && ((cmd_brval_get(cmd_next) 
            && value_is_pseudo_const(cmd_brval_get(cmd_next)))
          || (cmd_type_operator == cmd_type_get(cmd)
            && (value_is_same(cmd_opdst_get(cmd), cmd_brval_get(cmd_next)))
            && value_is_pseudo_const(cmd_opval1_get(cmd))
            && value_is_pseudo_const(cmd_opval2_get(cmd))))) {
      variable_const_t n;

      if (cmd_brval_get(cmd_next)
          && value_is_pseudo_const(cmd_brval_get(cmd_next))) {
        n = value_is_const(cmd_brval_get(cmd_next))
          ? value_const_get(cmd_brval_get(cmd_next))
          : 0;
      } else {
        n = pfile_op_const_exec(pf, cmd_optype_get(cmd), 
            cmd_opval1_get(cmd), cmd_opval2_get(cmd));
      }
      /* determine whether to take the branch or not */
      if (((cmd_branchcond_false == cmd_brcond_get(cmd_next)) && !n)
        || ((cmd_branchcond_true == cmd_brcond_get(cmd_next)) && n)) {
        /* this becomes an absolute branch */
        cmd_brcond_set(cmd_next, cmd_branchcond_none);
        cmd_brval_set(cmd_next, VALUE_NONE);
      } else {
        /* this is removed */
        cmd_link_set(cmd, cmd_link_get(cmd_next));
        cmd_free(cmd_next);
        rem_ct++;
      }
      if (cmd_type_operator == cmd_type_get(cmd)) {
        flag_t accessed;

        accessed = CMD_VARIABLE_ACCESS_FLAG_NONE;
        if (cmd_opdst_get(cmd)) {
          for (cmd_next = cmd_link_get(cmd);
               cmd_next 
               && (cmd_type_block_end != cmd_type_get(cmd_next));
               cmd_next = cmd_link_get(cmd_next)) {
            accessed = cmd_value_accessed_get(cmd_next, cmd_opdst_get(cmd));
            if (accessed) {
              break;
            }
          }
        }
        if (!(accessed & CMD_VARIABLE_ACCESS_FLAG_READ)) {
          /* if the result of cmd is not used again, remove it */
          if (cmd_pv) {
            while (cmd_link_get(cmd_pv) != cmd) {
              cmd_pv = cmd_link_get(cmd_pv);
            }
            cmd_link_set(cmd_pv, cmd_link_get(cmd));
          } else {
            *cmd_head = cmd_link_get(cmd);
          }
          cmd_free(cmd);
          cmd = (cmd_pv) ? cmd_pv : *cmd_head;
          rem_ct++;
        }
      }
    }
    cmd_pv = cmd;
    cmd = cmd_link_get(cmd);
  }
  return rem_ct != 0;
}



/*
 * NAME
 *   cmd_analyze
 *
 * DESCRIPTION
 *   analyze the command list for dead code
 *
 * PARAMETERS
 *   cmd_head : head of the cmd list
 *   cmd      : current entry
 *   flag     : CMD_FLAG_USER      : analyzing user area
 *              CMD_FLAG_INTERRUPT : analyzing an ISR
 *
 * RETURN
 *   none
 *
 * NOTES
 *   this doesn't solve the NP problem, it simply takes each
 *   path to determine if it's possible to get there from here
 */
static size_t depth_max;
void cmd_analyze(cmd_t cmd_top, cmd_t cmd, flag_t flag, size_t depth,
  pfile_t *pf)
{
  if (depth > depth_max) {
    depth_max = depth;
  }
  while (cmd && !cmd_flag_test(cmd, flag)) {
    cmd_flag_set(cmd, flag);
    if (cmd_type_operator == cmd_type_get(cmd)) {
      if (operator_assign == cmd_optype_get(cmd)) {
        if (variable_def_type_function 
            == value_type_get(cmd_opval1_get(cmd))) {
          /* assigning a function to a variable. eventually this
           * variable will be called (it's probably worth checking
           * use first), so go ahead & make the call here */
          cmd_t ptr;

          ptr = cmd_label_find(cmd_top,
              pfile_proc_label_get(
                variable_proc_get(
                  value_variable_get(
                    cmd_opval1_get(cmd)))));
          cmd_analyze(cmd_top, ptr, flag, depth + 1, pf);
        }
      }
      cmd = cmd_link_get(cmd);
    } else if (cmd_type_branch == cmd_type_get(cmd)) {
      switch (cmd_brtype_get(cmd)) {
        case cmd_branchtype_goto:
        case cmd_branchtype_call:
        case cmd_branchtype_task_start:
          {
            value_t  proc;
            value_t *params;
            size_t   param_ct;
            size_t   ii;

            proc     = cmd_brproc_get(cmd);
            param_ct = cmd_brproc_param_ct_get(cmd);
            params   = cmd_brproc_params_get(cmd);

            if (cmd_brdst_get(cmd)) {
              /* look for the label */
              cmd_t ptr;

              ptr = cmd_label_find(cmd_top, cmd_brdst_get(cmd));
              if ((cmd_branchtype_goto == cmd_brtype_get(cmd))
                && (cmd_brdst_get(cmd) == cmd_label_get(cmd_link_get(ptr)))) {
                /* GOTO (cmd+1) -- this is unnecessary! */
                cmd_flag_clr(ptr, flag);
                cmd = cmd_link_get(cmd);
              } else if ((cmd_branchtype_goto == cmd_brtype_get(cmd))
                  && (cmd_branchcond_none == cmd_brcond_get(cmd))) {
                /* unconditional goto, simply change cmd */
                cmd = ptr;
              } else {
                /* recurse using ptr, then continue */
                cmd_analyze(cmd_top, ptr, flag, depth + 1, pf);
                cmd = cmd_link_get(cmd);
              }
            } else if (proc) {
              /* this is an indirect call */
              value_t val;

              val = proc;
              if (variable_def_type_function == value_type_get(val)) {
                cmd_t ptr;

                ptr = cmd_label_find(cmd_top,
                    pfile_proc_label_get(
                      variable_proc_get(
                        value_variable_get(
                          val))));
                cmd_analyze(cmd_top, ptr, flag, depth + 1, pf);
                cmd = cmd_link_get(cmd);
              }
            }
            /* all the parameter must also be checked. any function
               references or pointers will have to be assumed to have
               been called. */
            /* parameter 0 is the return value & can be ignored */
            for (ii = 1; ii < param_ct; ii++) {
              if (variable_def_type_function == value_type_get(params[ii])) {
                cmd_t ptr;
                label_t lbl;

                lbl = pfile_proc_label_get(
                  variable_proc_get(
                    value_variable_get(
                      params[ii])
                  )
                );
                assert(lbl);
                if (lbl) {
                  ptr = cmd_label_find(cmd_top, lbl);
                  cmd_analyze(cmd_top, ptr, flag, depth + 1, pf);
                }
              }
            }
          }
          break;
        case cmd_branchtype_return:
          if (cmd_branchcond_none == cmd_brcond_get(cmd)) {
            /* at the end of the chain, so exit */
            cmd = 0;
            break;
          }
          /* fall through */
        case cmd_branchtype_none: 
        case cmd_branchtype_task_end:
        case cmd_branchtype_task_suspend:
          cmd = cmd_link_get(cmd);
          break;
      }
    } else if ((cmd_type_end == cmd_type_get(cmd))
      || (cmd_type_proc_leave == cmd_type_get(cmd))
      || (cmd_type_isr_cleanup == cmd_type_get(cmd))) {
      cmd = 0; /* we're done */
    } else {
      cmd = cmd_link_get(cmd);
    }
  }
}

