/************************************************************
 **
 ** bsc_main.c : BSC main file
 **
 ** Copyright (c) 2004, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "pfile.h"
#include "bsc_file.h"
#include "bsc_tokn.h"
#include "bsc_blck.h"

#ifdef MSDOS
extern unsigned _stklen = 16384;
#endif

/* format: 
 *   picbsc file 
 *     -hex name : set the name of the hex  file, default is file.hex
 *     -asm name : set the name of the assembly file, default is file.asm
 */     

/* this enum & the following opt table must be kept in sync! */
typedef enum {
  opt_hex,
  opt_asm,
  opt_norickpic,
  opt_debug,
  opt_quiet,
  opt_include,
  opt_nopcode,
  /* below this there should be no corresponding entry in the opt table */
  opt_none
} opt_t;

static const struct {
  const char *key;
  size_t      param_ct; /* 0 or 1 */
  const char *descr;
} opt_defs[] = {
  {"-hex",       1, "set the name of the hex file"},
  {"-asm",       1, "set the name of the asm file"},
  {"-norickpic", 0, "not using Rick Farmer's PIC loader"},
  {"-debug",     0, "show debug information"},
  {"-quiet",     0, "no status updates"},
  {"-I",         1, "set the include path, elements seperated with ';'"},
  {"-nopcode",   0, "do not show pcode in the asm file"}
};

static int file_open(FILE **dst, const char *name, const char *mode,
    const char *ext, size_t base_sz, char *namebuf)
{
  FILE *f;
  int   rc;

  if (!name) {
    strcpy(namebuf + base_sz, ext);
    name = namebuf;
  }
  f = fopen(name, mode);
  if (0 == f) {
    fprintf(stderr, "Cannot open: %s\n", name);
    rc = 1;
  } else {
    *dst = f;
    rc = 0;
  }
  return rc;
}

static void file_remove(const char *name, const char *ext,
    size_t base_sz, char *namebuf)
{
  if (!name) {
    strcpy(namebuf + base_sz, ext);
    name = namebuf;
  }
  remove(name);
}

static pfile_vectors_t vectors = {
  bsc_file_open,
  bsc_file_close,
  bsc_token_get,
  bsc_include_process
};

int main(int argc, char **argv)
{
  FILE       *f_asm;
  FILE       *f_hex;
  const char *hex_name;
  const char *asm_name;
  const char *bsc_name;
  const char *prog_name;
  flag_t      pfile_flags;
  char       *namebuf; /* for creating default names */
  pfile_t    *pf;
  opt_t       opt;
  boolean_t   err;
  const char *inc_path;

  err      = boolean_false;
  inc_path = 0;

#ifdef MSDOS
  cos(1.0); /* silly, but it links in the math bits */
#endif

  setvbuf(stdout, 0, _IONBF, 0);
  if (argc) {
    prog_name = *argv;
    argv++;
  } else {
    prog_name = "picbsc";
  }
  f_asm    = 0;
  f_hex    = 0;

  hex_name = 0;
  asm_name = 0;
  bsc_name = 0;
  namebuf  = 0;
  pfile_flags = 0;

  if (!*argv) {
    size_t ii;

    printf("Format: %s {options}\n", prog_name);
    for (ii = 0; ii < COUNT(opt_defs); ii++) {
      int sz;

      sz = printf("  %s", opt_defs[ii].key);
      if (opt_defs[ii].param_ct) {
        sz += printf(" arg");
      }
      if (sz < 16) {
        sz = 16 - sz;
      } else {
        sz = 1;
      }
      printf("  %*s : %s\n", sz, "", opt_defs[ii].descr);
    }
    exit(0);
  }

  while (*argv) {
    size_t       arg_len;
    size_t       ii;
    const char **need;

    arg_len = strlen(*argv);

    for (ii = 0, opt = opt_none; ii < COUNT(opt_defs); ii++) {
      if (0 == memcmp(opt_defs[ii].key, *argv, arg_len)) {
        if (opt_none == opt) {
          opt = ii;
        } else {
          fprintf(stderr, "ambiguous argument: %s\n", *argv);
          err = boolean_true;
          break; /* <--- */
        }
      }
    }
    need = 0;
    switch (opt) {
      case opt_hex:       need = &hex_name; break;
      case opt_asm:       need = &asm_name; break;
      case opt_norickpic: pfile_flags |= PFILE_FLAG_NORICK; break;
      case opt_debug:     pfile_flags |= PFILE_FLAG_DEBUG;  break;
      case opt_quiet:     pfile_flags |= PFILE_FLAG_QUIET;  break;
      case opt_none:      need = &bsc_name; argv--; break;
      case opt_include:   need = &inc_path; break;
      case opt_nopcode:   pfile_flags |= PFILE_FLAG_NOPCODE; break;
    }
    if (need) {
      if (*need) {
        fprintf(stderr, "%s previously defined\n", *argv);
        err = boolean_true;
      } else if (!argv[1]) {
        fprintf(stderr, "%s needs a parameter\n", *argv);
        err = boolean_true;
      } else {
        *need = argv[1];
      }
      argv++;
    }
    if (*argv) {
      argv++;
    }
  }
  if (!err) {
    if (!bsc_name) {
      fprintf(stderr, "no source file\n");
    } else {
      size_t sz;

      sz = 0;
      if (!hex_name || !asm_name) {
        /* need to create a name buffer */
        for (sz = strlen(bsc_name);
             sz 
             && ('.' != bsc_name[sz-1]) 
             && ('/' != bsc_name[sz-1])
             && ('\\' != bsc_name[sz-1]);
             sz--)
          ; /* null body */
        if (!sz || ('.' != bsc_name[sz-1])) {
          sz = strlen(bsc_name);
        } else {
          sz--;
        }
        namebuf = MALLOC(sz + 5);
        if (!namebuf) {
          fprintf(stderr, "out of memory");
          err = 1;
        } else {
          memcpy(namebuf, bsc_name, sz);
        }
      }
      if (!err
          && (0 == file_open(&f_asm, asm_name, "wt", ".asm", sz, namebuf))
          && (0 == file_open(&f_hex, hex_name, "wt", ".hex", sz, namebuf))) {
        result_t rc;

        rc = pfile_open(&pf, f_asm, f_asm, f_hex, stdout, pfile_flags,
          &vectors, 0);
        if (result_ok == rc) {
          rc = pfile_source_set(pf, bsc_name);
          if (result_ok == rc) {
            pfile_include_path_set(pf, inc_path);
            pfile_log(pf, pfile_log_info, BSC_MSG_COMP_DATE, __DATE__);

            bsc_file_process(pf);
          } else {
            pfile_log(pf, pfile_log_err, BSC_MSG_CANNOT_OPEN, bsc_name);
          }
          err = pfile_errct_get(pf);
          pfile_close(pf);
        }
      }
      if (f_hex) {
        fclose(f_hex);
      }
      if (f_asm) {
        fclose(f_asm);
      }
      if (err) {
        if (!(pfile_flags & PFILE_FLAG_DEBUG)) {
          file_remove(asm_name, ".asm", sz, namebuf);
        }
        file_remove(hex_name, ".hex", sz, namebuf);
      }

      if (namebuf) {
        FREE(namebuf);
      }
    }
  }
  return 0;
}

