/**********************************************************
 **
 ** bsc_vdef.c : BSC variable defintion parsing
 **
 ** Copyright (c) 2004, Kyle A. York
 ** All rights reserved
 **
 ***********************************************************/
#include <stdlib.h>
#include <string.h>

#include "vardef.h"
#include "variable.h"
#include "ptype.h"
#include "pfile.h"
#include "pf_expr.h"
#include "pftoken.h"
#include "pfproc.h"

#include "bsc_file.h"
#include "bsc_parm.h"
#include "bsc_blck.h"
#include "bsc_tokn.h"
#include "bsc_expr.h"
#include "bsc_vdef.h"

/* parse a variable name + dimension. on success,
 *   vname will hold the 1st vname_sz - 1 bytes of the variable name
 *   ct will hold the dimension
 */   
static boolean_t bsc_parse_var(pfile_t *pf, flag_t flags,
    size_t vname_sz, char *vname, variable_ct_t *ct)
{
  boolean_t   err;
  const char *ptr;

  err = boolean_true;
  ptr = pf_token_get(pf, pf_token_current);

  if (!bsc_token_is_identifier(pf)) {
    pfile_log(pf, pfile_log_err, BSC_MSG_VARIABLE_EXPECTED);
  } else {
    size_t  sz;

    sz = 1 + strlen(ptr);
    if (sz >= vname_sz) {
      sz = vname_sz - 1;
    }
    memcpy(vname, ptr, sz);
    vname[sz] = 0;

    if (!(flags & VARIABLE_FLAG_BIT)
        && pf_token_is(pf, pf_token_next, pfile_log_none, "(")) {
      value_t dim;

      pf_token_get(pf, pf_token_next);
      if (result_ok == bsc_expr_parse(pf, 0, 0, &dim)) {
        if (!value_is_const(dim)) {
          pfile_log(pf, pfile_log_err, BSC_MSG_CONSTANT_EXPECTED);
        } else if (value_const_get(dim) == 0) {
          pfile_log(pf, pfile_log_err, BSC_MSG_ARRAY_SZ_ZERO);
        } else if ((value_const_get(dim) > 255) 
            && (flags & VARIABLE_FLAG_CONST)) {
          pfile_log(pf, pfile_log_err, BSC_MSG_LOOKUP_TOO_BIG);
        }
        /* get a constant expression */
        if (pf_token_is(pf, pf_token_current, pfile_log_err, ")")) {
          pf_token_get(pf, pf_token_next);
          err = boolean_false;
          *ct = value_const_get(dim);
        }
        value_release(dim);
      }
    } else {
      if (flags & VARIABLE_FLAG_BIT) {
        pf_token_get(pf, pf_token_next);
      }
      err = boolean_false;
      *ct = 1;
    }
  }
  return err;
}

/*
 * NAME
 *   bsc_parse_def
 *
 * DESCRIPTION
 *   variable definition
 *
 * PARAMETERS
 *   pf       : parser file
 *   param_ct : 0
 *   params   : 0
 *   sz       : variable size (1, 2, 4)
 *   flags    : variable flags
 *
 * RETURN
 *   none
 *
 * NOTES
 *   create variables, format : {type} id[(positive constant expr > 0)][,...]
 */
void bsc_parse_def(pfile_t *pf, flag_t oflags,
  variable_def_t def)
{
  size_t            ct;

  /* only execute this on the first pass. after that just skip it */
  for (ct = 0;
       ((0 == ct)
         || (pf_token_is(pf, pf_token_current, pfile_log_none, ",")));
       ct++) {
    const char *ptr;
    flag_t      flags;

    flags = oflags;

    ptr = pf_token_get(pf, (0 == ct) ? pf_token_current : pf_token_next);
    if (pf_token_is(pf, pf_token_current, pfile_log_none, "volatile")) {
      flags |= VARIABLE_FLAG_VOLATILE;
      ptr = pf_token_get(pf, pf_token_next);
    }
    if (pf_token_is(pf, pf_token_current, pfile_log_none, "*")) {
      variable_def_t def2;

      ptr = pf_token_get(pf, pf_token_next);
      def2 = variable_def_alloc(0, 2, VARIABLE_FLAG_POINTER);
      variable_def_member_add(def2, 0 /*tag */, def, 1);
      def = def2;
    }

    if (!bsc_token_is_identifier(pf)) {
      pfile_log(pf, pfile_log_err, BSC_MSG_VARIABLE_EXPECTED);
    } else {
      variable_ct_t dim;
      char          varname[64];

      if (!bsc_parse_var(pf, oflags, sizeof(varname), varname, &dim)) {
        variable_t  var;

        if (1 == bsc_file_pass_get(pf)) {
          if (result_ok != pfile_variable_alloc(pf,
            varname, flags, 0, dim, def, &var)) {
            var = 0;
          }
        } else {
          var = pfile_variable_find(pf, pfile_log_err, strlen(varname), 
            varname);
          flags = variable_flags_get_all(var);
        }

        if (oflags & VARIABLE_FLAG_BIT) {
#if 1
          variable_flag_set(var, VARIABLE_FLAG_ASSIGNED);
          variable_flag_set(var, VARIABLE_FLAG_USED);
#endif
          if (pf_token_is(pf, pf_token_current, pfile_log_none, ":")) {
            /* define the size */
            value_t sz;

            pf_token_get(pf, pf_token_next);
            bsc_expr_parse(pf, 0, 0, &sz);
            if (sz) {
              if (!value_is_const(sz)) {
                pfile_log(pf, pfile_log_err, BSC_MSG_CONSTANT_EXPECTED);
              } else {
                variable_bit_size_set(var, value_const_get(sz));
                value_release(sz);
              }
            }
          } else {
            variable_bit_size_set(var, 1);
          }
        }

        if (pf_token_is(pf, pf_token_current, pfile_log_none, "@")) {
          variable_t master;

          pf_token_get(pf, pf_token_next);
          if (result_ok == pf_token_to_variable(pf, pfile_log_none,
                &master)) {
            pf_token_get(pf, pf_token_next);
            variable_master_set(var, master);
            variable_release(master);
          } else {
            value_t pos;

            pos = 0;
            bsc_expr_parse(pf, 0, 0, &pos);
            if (pos) {
              if (!value_is_const(pos)) {
                pfile_log(pf, pfile_log_err, BSC_MSG_CONSTANT_EXPECTED);
              } else if (var) {
                variable_base_set(var, value_const_get(pos));
              }
              value_release(pos);
            }
            if (pf_token_is(pf, pf_token_current, pfile_log_none, "[")) {
              if (pf_token_is(pf, pf_token_next, pfile_log_none, "all")) {
                if (var) {
                  variable_region_set_all(var, (variable_region_t) -1);
                }
                pf_token_get(pf, pf_token_next);
              } else {
                /* [region1[,region2...]] */
                pf_token_get_t which;

                which = pf_token_current;

                do {
                  value_t region;

                  pf_token_get(pf, which); /* skip '[' or ',' */
                  region = 0;
                  bsc_expr_parse(pf, 0, 0, &region);
                  if (region) {
                    if (!value_is_const(region)) {
                      pfile_log(pf, pfile_log_err, BSC_MSG_CONSTANT_EXPECTED);
                    } else if (var) {
                      variable_region_set(var, value_const_get(region));
                    }
                    value_release(region);
                  }
                  which = pf_token_next;
                } while (pf_token_is(pf, pf_token_current, pfile_log_none, ","));
              }
              if (pf_token_is(pf, pf_token_current, pfile_log_err, "]")) {
                pf_token_get(pf, pf_token_next);
              }
            }
          }
          if ((oflags & VARIABLE_FLAG_BIT)
            && pf_token_is(pf, pf_token_current, pfile_log_none, ":")) {
            value_t pos;

            pf_token_get(pf, pf_token_next);
            bsc_expr_parse(pf, 0, 0, &pos);
            if (pos) {
              if (!value_is_const(pos)) {
                pfile_log(pf, pfile_log_err, BSC_MSG_CONSTANT_EXPECTED);
              } else {
                variable_bit_size_set(var, value_const_get(pos));
                value_release(pos);
              }
            }
          }
        }
        if ((oflags & VARIABLE_FLAG_BIT)
          && (pf_token_is(pf, pf_token_current, pfile_log_none, ":"))) {
          value_t          bit;

          pf_token_get(pf, pf_token_next);
          bit = 0;
          bsc_expr_parse(pf, 0, 0, &bit);
          if (bit) {
            if ((!value_is_const(bit)) 
              || (value_const_get(bit) < 0)
              || (value_const_get(bit) > 7)) {
              variable_bit_pos_set(var, value_const_get(bit));
            }
            value_release(bit);
          }
        }
        if (pf_token_is(pf, pf_token_current, pfile_log_none, "=")) {
          boolean_t     is_multiple;
          variable_ct_t ii;
          variable_ct_t need;

          is_multiple = pf_token_is(pf, pf_token_next, pfile_log_none, "{");
          need = (is_multiple) ? variable_ct_get(var) : 1;
          for (ii = 0;
               (ii < need)
               && (!ii 
                 || pf_token_is(pf, pf_token_current, pfile_log_none, ","));
               ii++) {
            value_t n;

            if (ii || is_multiple) {
              /* skip either '{' or ',' */
              pf_token_get(pf, pf_token_next);
            }
            if (result_ok == bsc_expr_parse(pf, 0, 0, &n)) {
              if (flags & VARIABLE_FLAG_CONST) {
                if (!value_is_const(n)) {
                  pfile_log(pf, pfile_log_err, BSC_MSG_CONSTANT_EXPECTED);
                } else {
                  if (var) {
                    variable_const_set(var, ii * variable_sz_get(var), 
                      value_const_get(n));
                  }
                }
              } else if (var) {
                value_t vval;
                value_t baseofs;
                int     rc;

                rc = value_alloc(&vval, var); /* get a value */
                pfile_constant_get(pf, ii, &baseofs);
#if 0
                if (value_sz_get(vval) > 1) {
                  /* adjust the base ofs */
                  abort();
                }
#endif
                value_baseofs_set(vval, baseofs);
                value_release(baseofs);
                if (result_ok != rc) {
                  pfile_log_syserr(pf, rc);
                } else {
                  pfile_cmd_op_add(pf, operator_assign, &vval, n, 0);
                  value_release(vval);
                }
              }
              value_release(n);
            }
          } 
          if (is_multiple
            && pf_token_is(pf, pf_token_current, pfile_log_err, "}")) {
            pf_token_get(pf, pf_token_next);
          }
          if (var && (ii < variable_ct_get(var))) {
            pfile_log(pf, 
              (flags & VARIABLE_FLAG_CONST) ? pfile_log_err : pfile_log_warn, 
              BSC_MSG_ARRAY_INIT,
              ii, variable_ct_get(var), variable_name_get(var));
          }
        }
        if ((oflags & VARIABLE_FLAG_BIT)
           && !variable_master_get(var)) {
          /* add this to _bitbucket */
          variable_t bitbucket;

          bitbucket = pfile_proc_variable_find(pfile_proc_active_get(pf), 
              PFILE_LENGTH_DEF, PFILE_BITBUCKET_VARNAME);
          if (!bitbucket) {
            pfile_variable_alloc(pf, PFILE_BITBUCKET_VARNAME, 0, 0, 1, 0, 
              &bitbucket);
          }
          if (bitbucket) {
            variable_master_set(var, bitbucket);
            variable_bit_pos_set(var, variable_bit_size_get(bitbucket));
            variable_bit_size_set(bitbucket, variable_bit_size_get(var)
                + variable_bit_size_get(bitbucket));
            variable_release(bitbucket);
          }
        }
        variable_release(var);
      }
    }
  }
}

void bsc_parse_constdef(pfile_t *pf, unsigned sub, size_t param_ct,
  const param_t *params)
{
  variable_def_t def;

  def = pfile_variable_def_find(pf, pfile_log_err, pf_token_ptr_get(pf));
  if (def) {
    pf_token_get(pf, pf_token_next);
    bsc_parse_def(pf, VARIABLE_FLAG_CONST, def);
  }
}

/*
 * define a new structure (aka record):
 *
 * record {tag}
 *   type {var}[(cexpr)]
 *   ...
 *   record {tag}
 *     type {var}[(cexpr)]
 *     ...
 *   end record
 * end record  
 */

void bsc_parse_record(pfile_t *pf, unsigned sub, size_t param_ct,
    const param_t *params)
{
  const char *ptr;

  ptr = pf_token_ptr_get(pf);
  if (!bsc_token_is_identifier(pf)) {
    pfile_log(pf, pfile_log_err, BSC_MSG_VARIABLE_EXPECTED);
  } else {
    variable_def_t def;

    def = variable_def_alloc(ptr, 0, 0);
    if (!def) {
      pfile_log_syserr(pf, result_memory);
    } else {
      boolean_t err;

      err = boolean_false;

      pf_token_get(pf, pf_token_next); /* skip the tag */
      bsc_statement_next(pf);
      while (!err
        && (!pf_token_is(pf, pf_token_current, pfile_log_none, "end"))
        && (!pf_token_is(pf, pf_token_current, pfile_log_none, 
            "endrecord"))) {
        /* expect {type} var[(cexpr)][,...] */
        variable_def_t vdef;

        vdef = pfile_variable_def_find(pf, pfile_log_err, 
            pf_token_ptr_get(pf));
        if (!vdef) {
          err = boolean_true;
        } else {
          do {
            /* skip either the type or comment */
            variable_ct_t dim;
            char          varname[64];

            pf_token_get(pf, pf_token_next);
            err = bsc_parse_var(pf, 0, sizeof(varname), varname, &dim);
            if (!err) {
              result_t rc;

              rc = variable_def_member_add(def, varname, vdef, dim);
              if (result_ok != rc) {
                err = boolean_true;
                if (result_exists == rc) {
                  pfile_log(pf, pfile_log_err, BSC_MSG_MULTI_DEF, varname);
                } else {
                  pfile_log_syserr(pf, rc);
                }
              }
            }
          } while (!err && pf_token_is(pf, pf_token_current, pfile_log_none, 
               ","));
          bsc_statement_next(pf);
        }
      }
      if (pf_token_is(pf, pf_token_current, pfile_log_none, "end")) {
        if (pf_token_is(pf, pf_token_next, pfile_log_err, "record")) {
          pf_token_get(pf, pf_token_next);
        }
      }
      if (err) {
        variable_def_free(def);
      } else {
        result_t rc;

        if (bsc_file_pass_get(pf) == 1) {
          rc = pfile_variable_def_add(pf, def);
          if (result_ok != rc) {
            if (result_exists == rc) {
              pfile_log(pf, pfile_log_err, BSC_MSG_MULTI_DEF, 
                  variable_def_tag_get(def));
            } else {
              pfile_log_syserr(pf, rc);
            }
            variable_def_free(def);
          }
        } else {
          variable_def_free(def);
        }
      }
    }
  }
}

