#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "libutils/array.h"
#include "libutils/cod_file.h"

static void COD_code_map_table_dump(const COD_code_map_table_t *tbl)
{
  if (tbl) {
    size_t ii;

    printf("COD code_map table (%u entries)\n"
           "   start     end\n"
           "----------------\n", COD_code_map_table_entry_ct(tbl));
    for (ii = 0; ii < COD_code_map_table_entry_ct(tbl); ii++) {
      const COD_code_map_entry_t *ety;

      ety = COD_code_map_table_entry_get(tbl, ii);
      if (ety->end) {
        printf("%8x%8x\n", (unsigned) ety->start, (unsigned) ety->end);
      }
    }
  }
}

static void COD_symbol_table_dump(const COD_symbol_table_t *tbl)
{
  if (tbl) {
    size_t ii;
    
    printf("symbol table (%u entries)\n"
           "label type value\n"
           "----------------\n", COD_symbol_table_entry_ct(tbl));
    for (ii = 0; ii < COD_symbol_table_entry_ct(tbl); ii++) {
      const COD_symbol_entry_t *ety;

      ety = COD_symbol_table_entry_get(tbl, ii);
      if (ety->label[0]) {
        printf("%-12.*s %04x %02x\n",
          ety->label[0], ety->label + 1, ety->value, ety->type);
      }
    }
  }
}

static void COD_local_vars_table_dump(const COD_local_vars_table_t *tbl)
{
  if (tbl) {
    size_t ii;

    printf("local variables table (%u entries)\n"
           "label start  end\n"
           "----------------\n", COD_local_vars_table_entry_ct(tbl));
    for (ii = 0; ii < COD_local_vars_table_entry_ct(tbl); ii++) {
      const COD_local_vars_entry_t *local_ety;

      local_ety = COD_local_vars_table_entry_get(tbl, ii);
      /*if (local_ety->header.label[0])*/ {
        size_t jj;

        printf("%8lx %8lx\n", 
          local_ety->header.start, local_ety->header.end);
        for (jj = 0; 
             jj < COD_symbol_table_entry_ct(local_ety->symbols); 
             jj++) {
          const  COD_symbol_entry_t *symbol_ety;

          symbol_ety = COD_symbol_table_entry_get(local_ety->symbols, jj);
          printf("...%-12.*s %04x %02x\n",
            symbol_ety->label[0], symbol_ety->label + 1, 
            symbol_ety->value, 
            symbol_ety->type);
        }
      }
    }
  }
}

static void COD_name_table_dump(const COD_name_table_t *tbl)
{
  if (tbl) {
    size_t ii;

    printf("name table (%u)\n", COD_name_table_entry_ct(tbl));
    for (ii = 0; ii < COD_name_table_entry_ct(tbl); ii++) {
      const COD_name_entry_t *ety;

      ety = COD_name_table_entry_get(tbl, ii);
      if (ety->label[0]) {
        printf("%-64.*s\n", ety->label[0], ety->label + 1);
      }
    }
  }
}

static void COD_line_symbol_table_dump(const COD_line_symbol_table_t *tbl,
  const COD_directory_t *dir)
{
  if (tbl) {
    size_t ii;

    printf("line symbols (%u)\n"
           "list flag sloc file\n"
           "-------------------\n",
           COD_line_symbol_table_entry_ct(tbl));
    for (ii = 0; ii < COD_line_symbol_table_entry_ct(tbl); ii++) {
      const COD_line_symbol_entry_t *ety;
      const COD_name_entry_t        *fname;

      ety = COD_line_symbol_table_entry_get(tbl, ii);
      fname = COD_name_table_entry_get(dir->name_table, ety->file);
      printf("%4u %4x %4x %.*s:%u\n",
        ii,
        ety->smod, ety->sloc, 
        fname->label[0], fname->label + 1, ety->line);
    }
  }
}

static void COD_long_symbol_table_dump(const COD_long_symbol_table_t *tbl)
{
  if (tbl) {
    size_t ii;

    printf("long symbol table (%u)\n"
           "stype0 stype1 value  name\n"
           "-------------------------\n", COD_long_symbol_table_entry_ct(tbl));
    for (ii = 0; ii < COD_long_symbol_table_entry_ct(tbl); ii++) {
      const COD_long_symbol_entry_t *ety;

      ety = COD_long_symbol_table_entry_get(tbl, ii);
      printf("%4x %4x %8lu %.*s\n",
        ety->stype0,
        ety->stype1,
        ety->svalue,
        ety->name[0], ety->name + 1);
    }
  }
}

static void COD_directory_dump(const COD_directory_t *dir)
{
  printf("Source file : %.*s\n"
         "Date        : %.*s\n"
         "Time        : %2u:%02u\n"
         "Version     : %.*s\n"
         "Compiler    : %.*s\n"
         "Copyright   : %.*s\n"
         "COD version : %u\n"
         "Processor   : %.*s\n"
         "Address size: %u\n\n",
         dir->source[0], dir->source + 1,
         dir->date[0],   dir->date + 1,
         dir->time / 100, dir->time % 100,
         dir->version[0], dir->version + 1,
         dir->compiler[0], dir->compiler + 1,
         dir->notice[0], dir->notice + 1,
         dir->COD_type,
         dir->processor[0], dir->processor + 1,
         dir->addr_size);
  printf("Short symbol table      : %4u to %4u\n"
         "Long symbol table       : %4u to %4u\n"
         "File names              : %4u to %4u\n"
         "Source debug information: %4u to %4u\n"
         "Next directory          : %4u\n"
         "Code memory map         : %4u to %4u\n"
         "Local scoping           : %4u to %4u\n"
         "Debug message area      : %4u to %4u\n\n",
         dir->symbol_table_start, dir->symbol_table_end,
         dir->long_symbol_start, dir->long_symbol_end,
         dir->name_table_start, dir->name_table_end,
         dir->list_table_start, dir->list_table_end,
         dir->next_dir,
         dir->code_map_start, dir->code_map_end,
         dir->local_vars_start, dir->local_vars_end,
         dir->message_start, dir->message_end);
}

static void COD_index_dump(const COD_directory_t *dir)
{
  size_t                      ii;
  unsigned                    pos;
  unsigned                    map_idx;
  const COD_code_map_entry_t *map_entry;

  printf("Formatted Code Dump\n"
         "-------------------\n");
  map_idx   = 0;
  map_entry = COD_code_map_table_entry_get(dir->code_map_table, map_idx);
  for (ii = 0, pos = 0; ii < 128; ii++) {
    if (dir->index[ii]) {
      const COD_code_entry_t *ety;
      const uchar            *dptr;
      unsigned                row;

      ety  = dir->code[ii];
      dptr = ety->data;

      for (row = 0; row < COD_BLOCK_SIZE; row += 16) {
        if (map_entry
          && (pos + 15 >= map_entry->start)
          && (pos <= map_entry->end)) {
          uchar    xbuf[16];
          unsigned col;

          printf("   %04x  ", row + ii * COD_BLOCK_SIZE);
          for (col = 0; col < 16; col++) {
            uchar n;
            
            n = *dptr;
            dptr++;
            if (map_entry 
              && (map_entry->start <= pos) 
              && (pos <= map_entry->end)) {
              printf("%02x ", n);
              xbuf[col] = (isprint(n) ? n : '.');
            } else {
              printf(".. ");
              xbuf[col] = '.';
            }
            pos++;
            if (map_entry && (pos > map_entry->end)) {
              map_idx++;
              map_entry = COD_code_map_table_entry_get(dir->code_map_table, 
                map_idx);
            }
            if (7 == col) {
              printf(" ");
            }
          }
          printf("  %.16s\n", xbuf);
        } else {
          pos += 16;
        }
      }
    } else {
      pos += COD_BLOCK_SIZE;
    }
  }
  printf("\n");
}

int main(int argc, char **argv)
{
  COD_directory_t *dir;

  dir = COD_directory_read(argv[1]);
  COD_directory_dump(dir);
  COD_index_dump(dir);
  COD_code_map_table_dump(dir->code_map_table);
  COD_symbol_table_dump(dir->symbol_table);
  COD_local_vars_table_dump(dir->local_vars_table);
  COD_name_table_dump(dir->name_table);
  COD_line_symbol_table_dump(dir->xref_table, dir);
  COD_long_symbol_table_dump(dir->long_symbol_table);
  COD_directory_free(dir);
  return 0;
}

