/************************************************************
 **
 ** jal_asm.c : JAL assembler definitions
 **
 ** Copyright (c) 2005, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/
#include <errno.h>
#include "../libutils/mem.h"
#include "../libpic12/pic_inst.h"
#include "../libcore/cmd_asm.h"
#include "../libcore/pf_msg.h"
#include "../libcore/pf_proc.h"
#include "../libcore/pf_cmd.h"
#include "jal_expr.h"
#include "jal_tokn.h"
#include "jal_incl.h"
#include "jal_blck.h"
#include "jal_asm.h"

/* needed MACROS:
   ADDCF  f,d : btfsc _status, _c;  incf f,d
   ADDDCF f,d : btfsc _status, _dc; incf f,d
   B      n   : goto n
   BC     n   : btfsc _status, _c;  goto n
   BDC    n   : btfsc _status, _dc; goto n
   BNC    n   : btfss _status, _c;  goto n
   BNDC   n   : btfss _status, _dc; goto n
   BNZ    n   : btfss _status, _z;  goto n
   BZ     n   : btfss _status, _z;  goto n
   CLRC       : bcf _status, _c
   CLRDC      : bcf _status, _dc
   CLRZ       : bcf _status, _z
   LCALL  n   : page call n
   LGOTO  n   : page goto n
   MOVFW  f   : movf f,w
   NEGF   f,d : comf f,f; inc f,f
   SETC       : bsf _status, _c
   SETDC      : bsf _status, _dc
   SETZ       : bsf _status, _z
   SKPC       : btfss _status, _c
   SKPDC      : btfss _status, _dc
   SKPNC      : btfsc _status, _c
   SKPNDC     : btfsc _status, _dc
   SKPNZ      : btfsc _status, _z
   SKPZ       : btfss _status, _z
   SUBCF  f,d : btfsc _status, _c ; decf f,d
   SUBDCF f,d : btfsc _status, _dc; decf f,d
*/
/* there are only six types of instructions:
     op     --
     op f   --
     op f,d -- d = 0 or 1
     op f,b -- 0 <= b <= 7
     op n   -- 0 <= n <= 255
     op k   -- 0 <= k <= 2047
   plus two special commands:
     bank -- set the bank bits for the next instruction
     page -- set the page bits for the next instruction
*/
/* parse a single assembly command */
#define ASM_BIT_NONE 0x0000
#define ASM_BIT_BANK 0x0001 /* BANK found */
#define ASM_BIT_PAGE 0x0002 /* PAGE found */

/* changes to this enum must be reflected in the following arrays! */
typedef enum {
  asm_cmd_special_none,
  asm_cmd_special_addcf, /* btfsc _status, _c; incf f, d */
  asm_cmd_special_adddcf,/* btfsc _status, _dc; incf f, d */
  asm_cmd_special_b,     /* goto n */
  asm_cmd_special_bc,    /* btfsc _status, _c; goto n */
  asm_cmd_special_bdc,   /* btfsc _status, _dc; goto n */
  asm_cmd_special_bnc,   /* btfss _status, _c; goto n */
  asm_cmd_special_bndc,  /* btfsc _status, _c; goto n */
  asm_cmd_special_bnz,   /* btfss _status, _z; goto n */
  asm_cmd_special_bz,    /* btfsc _status, _z; goto n */
  asm_cmd_special_clrc,  /* bcf   _status, _c */
  asm_cmd_special_clrdc, /* bcf   _status, _dc */
  asm_cmd_special_clrz,  /* bcf   _status, _z  */
  asm_cmd_special_lcall, /* page call n */
  asm_cmd_special_lgoto, /* page goto n */
  asm_cmd_special_movfw, /* movf f,0 */
  asm_cmd_special_negf,  /* comf  f, 1; incf f, d */
  asm_cmd_special_setc,  /* bsf   _status, _c */
  asm_cmd_special_setdc, /* bsf   _status, _dc */
  asm_cmd_special_setz,  /* bsf   _status, _z  */
  asm_cmd_special_skpc,  /* btfss _status, _c  */
  asm_cmd_special_skpdc, /* btfss _status, _dc */
  asm_cmd_special_skpnc, /* btfsc _status, _c  */
  asm_cmd_special_skpndc,/* btfsc _status, _dc */
  asm_cmd_special_skpnz, /* btfsc _status, _z  */
  asm_cmd_special_skpz,  /* btfss _status, _z  */
  asm_cmd_special_subcf, /* btfss _status, _c; decf f, d */
  asm_cmd_special_subdcf,/* btfsc _status, _dc; decf f, d */
  asm_cmd_special_tstf,  /* movf f, f */
  asm_cmd_special_ct
} asm_cmd_special_t;

static const struct {
  unsigned     asm_bit; /* BANK and/or PAGE flags to add   */
  pic_opcode_t flag_op; /* operator used by _status, _flag */
  const char  *flag;    /* flag to test                    */
  pic_opcode_t op;      /* translated op                   */
} macro_xlate[asm_cmd_special_ct] = {
  {ASM_BIT_NONE, pic_opcode_none,  0,     pic_opcode_none}, /* */
  {ASM_BIT_NONE, pic_opcode_btfsc, "_c",  pic_opcode_incf}, /* addcf  */
  {ASM_BIT_NONE, pic_opcode_btfsc, "_dc", pic_opcode_incf}, /* adddcf */
  {ASM_BIT_NONE, pic_opcode_none,  0,     pic_opcode_goto}, /* b      */
  {ASM_BIT_NONE, pic_opcode_btfsc, "_c",  pic_opcode_goto}, /* bc     */
  {ASM_BIT_NONE, pic_opcode_btfsc, "_dc", pic_opcode_goto}, /* bdc    */
  {ASM_BIT_NONE, pic_opcode_btfss, "_c",  pic_opcode_goto}, /* bnc    */
  {ASM_BIT_NONE, pic_opcode_btfss, "_dc", pic_opcode_goto}, /* bndc   */
  {ASM_BIT_NONE, pic_opcode_btfss, "_z",  pic_opcode_goto}, /* bnz    */
  {ASM_BIT_NONE, pic_opcode_btfsc, "_z",  pic_opcode_goto}, /* bz     */
  {ASM_BIT_NONE, pic_opcode_bcf,   "_c",  pic_opcode_none}, /* clrc   */
  {ASM_BIT_NONE, pic_opcode_bcf,   "_dc", pic_opcode_none}, /* clrdc  */
  {ASM_BIT_NONE, pic_opcode_bcf,   "_z",  pic_opcode_none}, /* clrz   */
  {ASM_BIT_PAGE, pic_opcode_none,  0,     pic_opcode_call}, /* lcall  */
  {ASM_BIT_PAGE, pic_opcode_none,  0,     pic_opcode_goto}, /* lgoto  */
  {ASM_BIT_NONE, pic_opcode_none,  0,     pic_opcode_movf}, /* movfw  */
  {ASM_BIT_NONE, pic_opcode_none,  0,     pic_opcode_incf}, /* negf   */
  {ASM_BIT_NONE, pic_opcode_bsf,   "_c",  pic_opcode_none}, /* setc   */
  {ASM_BIT_NONE, pic_opcode_bsf,   "_dc", pic_opcode_none}, /* setdc  */
  {ASM_BIT_NONE, pic_opcode_bsf,   "_z",  pic_opcode_none}, /* setz   */
  {ASM_BIT_NONE, pic_opcode_btfss, "_c",  pic_opcode_none}, /* skpc   */
  {ASM_BIT_NONE, pic_opcode_btfss, "_dc", pic_opcode_none}, /* skpdc  */
  {ASM_BIT_NONE, pic_opcode_btfsc, "_c",  pic_opcode_none}, /* skpnc  */
  {ASM_BIT_NONE, pic_opcode_btfsc, "_dc", pic_opcode_none}, /* skpndc */
  {ASM_BIT_NONE, pic_opcode_btfsc, "_z",  pic_opcode_none}, /* skpnz  */
  {ASM_BIT_NONE, pic_opcode_btfss, "_z",  pic_opcode_none}, /* skpz   */
  {ASM_BIT_NONE, pic_opcode_btfsc, "_c",  pic_opcode_none}, /* subcf  */
  {ASM_BIT_NONE, pic_opcode_btfsc, "_dc", pic_opcode_none}, /* subdcf */
  {ASM_BIT_NONE, pic_opcode_none,  0,     pic_opcode_movf}, /* tstf   */
};


static const struct {
  const char       *tag;
  pic_optype_t      type;
  pic_opcode_t      op;
  asm_cmd_special_t special; /* a hack for the special operations */
} asm_cmds[] = {
  {"addwf",  pic_optype_f_d,  pic_opcode_addwf},
  {"andwf",  pic_optype_f_d,  pic_opcode_andwf},
  {"clrf",   pic_optype_f,    pic_opcode_clrf},
  {"clrw",   pic_optype_none, pic_opcode_clrw},
  {"comf",   pic_optype_f_d,  pic_opcode_comf},
  {"decf",   pic_optype_f_d,  pic_opcode_decf},
  {"decfsz", pic_optype_f_d,  pic_opcode_decfsz},
  {"incf",   pic_optype_f_d,  pic_opcode_incf},
  {"incfsz", pic_optype_f_d,  pic_opcode_incfsz},
  {"iorwf",  pic_optype_f_d,  pic_opcode_iorwf},
  {"movf",   pic_optype_f_d,  pic_opcode_movf},
  {"movwf",  pic_optype_f,    pic_opcode_movwf},
  {"nop",    pic_optype_none, pic_opcode_nop},
  {"rlf",    pic_optype_f_d,  pic_opcode_rlf},
  {"rrf",    pic_optype_f_d,  pic_opcode_rrf},
  {"subwf",  pic_optype_f_d,  pic_opcode_subwf},
  {"swapf",  pic_optype_f_d,  pic_opcode_swapf},
  {"xorwf",  pic_optype_f_d,  pic_opcode_xorwf},
  {"bcf",    pic_optype_f_b,  pic_opcode_bcf},
  {"bsf",    pic_optype_f_b,  pic_opcode_bsf},
  {"btfsc",  pic_optype_f_b,  pic_opcode_btfsc},
  {"btfss",  pic_optype_f_b,  pic_opcode_btfss},
  {"addlw",  pic_optype_n,    pic_opcode_addlw},
  {"andlw",  pic_optype_n,    pic_opcode_andlw},
  {"call",   pic_optype_k,    pic_opcode_call},
  {"clrwdt", pic_optype_none, pic_opcode_clrwdt},
  {"goto",   pic_optype_k,    pic_opcode_goto},
  {"iorlw",  pic_optype_n,    pic_opcode_iorlw},
  {"movlw",  pic_optype_n,    pic_opcode_movlw},
  {"retfie", pic_optype_none, pic_opcode_retfie},
  {"retlw",  pic_optype_n,    pic_opcode_retlw},
  {"return", pic_optype_none, pic_opcode_return},
  {"sleep",  pic_optype_none, pic_opcode_sleep},
  {"sublw",  pic_optype_n,    pic_opcode_sublw},
  {"xorlw",  pic_optype_n,    pic_opcode_xorlw},
  {"option", pic_optype_none, pic_opcode_option}, /* don't use! */
  {"tris",   pic_optype_tris, pic_opcode_tris},   /* don't use! */
  /* let's add some common macros */
  {"addcf",  pic_optype_f_d,  pic_opcode_none, asm_cmd_special_addcf},
  {"adddcf", pic_optype_f_d,  pic_opcode_none, asm_cmd_special_adddcf},
  {"b",      pic_optype_k,    pic_opcode_none, asm_cmd_special_b},
  {"bc",     pic_optype_k,    pic_opcode_none, asm_cmd_special_bc},
  {"bdc",    pic_optype_k,    pic_opcode_none, asm_cmd_special_bdc},
  {"bnc",    pic_optype_k,    pic_opcode_none, asm_cmd_special_bnc},
  {"bndc",   pic_optype_k,    pic_opcode_none, asm_cmd_special_bndc},
  {"bnz",    pic_optype_k,    pic_opcode_none, asm_cmd_special_bnz},
  {"bz",     pic_optype_k,    pic_opcode_none, asm_cmd_special_bz},
  {"clrc",   pic_optype_none, pic_opcode_none, asm_cmd_special_clrc},
  {"clrdc",  pic_optype_none, pic_opcode_none, asm_cmd_special_clrdc},
  {"clrz",   pic_optype_none, pic_opcode_none, asm_cmd_special_clrz},
  {"lcall",  pic_optype_k,    pic_opcode_none, asm_cmd_special_lcall},
  {"lgoto",  pic_optype_k,    pic_opcode_none, asm_cmd_special_lgoto},
  {"movfw",  pic_optype_f,    pic_opcode_none, asm_cmd_special_movfw},
  {"negf",   pic_optype_f,    pic_opcode_none, asm_cmd_special_negf},
  {"setc",   pic_optype_none, pic_opcode_none, asm_cmd_special_setc},
  {"setdc",  pic_optype_none, pic_opcode_none, asm_cmd_special_setdc},
  {"setz",   pic_optype_none, pic_opcode_none, asm_cmd_special_setz},
  {"skpc",   pic_optype_none, pic_opcode_none, asm_cmd_special_skpc},
  {"skpdc",  pic_optype_none, pic_opcode_none, asm_cmd_special_skpdc},
  {"skpnc",  pic_optype_none, pic_opcode_none, asm_cmd_special_skpnc},
  {"skpndc", pic_optype_none, pic_opcode_none, asm_cmd_special_skpndc},
  {"skpnz",  pic_optype_none, pic_opcode_none, asm_cmd_special_skpnz},
  {"skpz",   pic_optype_none, pic_opcode_none, asm_cmd_special_skpz},
  {"subcf",  pic_optype_f_d,  pic_opcode_none, asm_cmd_special_subcf},
  {"subdcf", pic_optype_f_d,  pic_opcode_none, asm_cmd_special_subdcf},
  {"tstf",   pic_optype_f,    pic_opcode_none, asm_cmd_special_tstf}
};


static void asm_cmd_add(pfile_t *pf,
    pic_opcode_t op, value_t val, size_t valofs, pic_opdst_t opdst, 
    value_t n, label_t lbl, flag_t flags, size_t data_sz, uchar *data)
{
  cmd_t cmd;

  cmd = cmd_asm_alloc(op, val, valofs, opdst, n, lbl, flags, data_sz, data);
  if (cmd) {
    pfile_cmd_add(pf, cmd);
  }
}

typedef struct asm_db_data_ {
  size_t alloc;
  size_t used;
  uchar *data;
} asm_db_data_t;

/* parse: 
 *   db c1,c2[,c3,c4...] 
 *   dw c1[,c2...]
 *   ds "blah"|c1[,...]
 */
typedef enum {
  jal_parse_asm_db,
  jal_parse_asm_dw,
  jal_parse_asm_ds
} jal_parse_asm_t;

static void asm_data_append(pfile_t *pf, asm_db_data_t *data, 
  jal_parse_asm_t type, variable_const_t n)
{
  variable_const_t n_max;

  if (data->used == data->alloc) {
    size_t tmp_alloc;
    void  *tmp;

    tmp_alloc = (data->alloc) ? 2 * data->alloc : 256;
    tmp       = REALLOC(data->data, sizeof(*data->data) * tmp_alloc);
    if (!tmp) {
      pfile_log_syserr(pf, ENOMEM);
    } else {
      data->data  = tmp;
      data->alloc = tmp_alloc;
    }
  }
  /* let's validate n */
  if (jal_parse_asm_ds == type) {
    n_max = 127;
  } else {
    /* on a 14 bit core, even bytes are limited to 6 bits */
    n_max = (data->used & 0x0001) ? 255 : 63;
  }
  if ((n > n_max) && pfile_flag_test(pf, PFILE_FLAG_WARN_RANGE)) {
    pfile_log(pf, pfile_log_warn, "%u out of range", (unsigned) n & 0xff);
  }
  if (data->used < data->alloc) {
    data->data[data->used] = n & n_max;
    data->used++;
  }
}

void jal_parse_asm_data(pfile_t *pf, jal_parse_asm_t type)
{
  asm_db_data_t data;

  data.alloc = 0;
  data.used  = 0;
  data.data  = 0;
  do {
    /* skip the ',' or dx */
    pf_token_get(pf, pf_token_next);
    if ('"' == *pf_token_get(pf, pf_token_current)) {
      /* parse a string */
      const char *str;
      size_t      str_sz;

      str    = pf_token_get(pf, pf_token_current) + 1;
      str_sz = pf_token_sz_get(pf) - 1;
      if (str_sz && ('"' == str[str_sz-1])) {
        str_sz--;
      }
      while (str_sz) {
        if ((jal_parse_asm_db == type)
          || (jal_parse_asm_dw == type)) {
          asm_data_append(pf, &data, type, 0);
        }
        asm_data_append(pf, &data, type, *str);
        str++;
        str_sz--;
      }
      pf_token_get(pf, pf_token_next);
    } else {
      value_t          cexpr;
      variable_const_t n;

      cexpr = jal_parse_expr(pf);
      if (!value_is_const(cexpr)) {
        pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
        n = 0;
      } else {
        n = value_const_get(cexpr);
      }
      value_release(cexpr);
      if (jal_parse_asm_dw == type) {
        asm_data_append(pf, &data, type, n / 256);
      }
      asm_data_append(pf, &data, type, n & 0xff);
    }
  } while (pf_token_is(pf, pf_token_current, pfile_log_none, ","));
  if (data.used) {
    if (data.used & 0x01) {
      /* due to the way the data member is allocated, we know that
         an even number of bytes always exists (this padding will not
         cause a buffer overflow */
      if (pfile_flag_test(pf, PFILE_FLAG_WARN_MISC)) {
        pfile_log(pf, pfile_log_warn, "size is odd, padding used");
      }
      data.data[data.used] = 0;
      data.used++;
    }
    if (jal_parse_asm_ds == type) {
      /* pack 2 chars into 1 14-bit word */
      size_t ii;
      size_t jj;

      for (ii = 0, jj = 0; ii < data.used; ii += 2) {
        unsigned n;

        n = data.data[ii] * 128U + data.data[ii+1];
        data.data[ii]   = n >> 8;
        data.data[ii+1] = n & 0xff;
      }
    }
    asm_cmd_add(pf, pic_opcode_db, VALUE_NONE, 0, pic_opdst_none,
      VALUE_NONE, LABEL_NONE, 0, data.used, data.data);
  }
}

/* this returns TRUE if at least one token has been absorbed */
void jal_parse_asm(pfile_t *pf, const pfile_pos_t *statement_start)
{
  if (pf_token_is(pf, pf_token_current, pfile_log_none, "db")) {
    jal_parse_asm_data(pf, jal_parse_asm_db);
  } else if (pf_token_is(pf, pf_token_current, pfile_log_none, "dw")) {
    jal_parse_asm_data(pf, jal_parse_asm_dw);
  } else if (pf_token_is(pf, pf_token_current, pfile_log_none, "ds")) {
    jal_parse_asm_data(pf, jal_parse_asm_ds);
  } else if (pf_token_is(pf, pf_token_current, pfile_log_none, "local")) {
    /* declare some local labels */
    do {
      const char *ptr;

      ptr = pf_token_get(pf, pf_token_next);
      if (jal_token_is_identifier(pf)) {
        label_release(pfile_label_alloc(pf, ptr));
        pf_token_get(pf, pf_token_next);
      }
    } while (pf_token_is(pf, pf_token_current, pfile_log_none, ","));
  } else {
    unsigned     flags;
    size_t       ii;
    value_t      val;
    size_t       valofs;
    value_t      n;
    pic_opcode_t op;
    pic_opdst_t  opdst;
    label_t      lbl;
    boolean_t    err;
    unsigned     n_lo;
    unsigned     n_hi;
   
    lbl = LABEL_NONE;
    if (jal_token_is_identifier(pf)) {
      /* protect against an ASM statement also being a label
       * I need to find a better solution to this! */
      for (ii = 0; 
           (ii < COUNT(asm_cmds)) 
           && !pf_token_is(pf, pf_token_current, pfile_log_none, 
             asm_cmds[ii].tag);
           ii++)
        ; /* null body */
      if (ii == COUNT(asm_cmds)) {
        const char *ptr;

        ptr = pf_token_get(pf, pf_token_current);
        lbl = pfile_label_find(pf, pfile_log_none, ptr);
        if (lbl) {
          pfile_cmd_label_add(pf, lbl);
          label_release(lbl);
          if (pf_token_is(pf, pf_token_next, pfile_log_err, ":")) {
            pf_token_get(pf, pf_token_next);
          }
        }
      }
    }
    if (!lbl) {
      err    = boolean_false;
      flags  = 0;
      op     = pic_opcode_none;
      opdst  = pic_opdst_none;
      val    = VALUE_NONE;
      valofs = 0;
      n      = VALUE_NONE;
      lbl    = LABEL_NONE;
      n_lo   = 0;
      n_hi   = -1;

      /* first, look for any pre BANK or PAGE elements */
      /* it looks like the assembler should handle the following:
         BANK value
         PAGE label
         BANK | PAGE statement
         Logically, only one of BANK or PAGE can exist for a statement and
         the flag effects only the *following* statement */
      if (pf_token_is(pf, pf_token_current, pfile_log_none, "bank")) {
        flags |= ASM_BIT_BANK;
        pf_token_get(pf, pf_token_next);
      } else if (pf_token_is(pf, pf_token_current, pfile_log_none, "page")) {
        flags |= ASM_BIT_PAGE;
        pf_token_get(pf, pf_token_next);
        if (pic_is_16bit(pf) && pfile_flag_test(pf, PFILE_FLAG_WARN_MISC)) {
          pfile_log(pf, pfile_log_warn, "PAGE has no meaning on 16 bit cores");
        }
      }
      for (ii = 0; 
           (ii < COUNT(asm_cmds)) 
           && !pf_token_is(pf, pf_token_current, pfile_log_none, 
             asm_cmds[ii].tag);
           ii++)
        ; /* null body */


      if (COUNT(asm_cmds) == ii) {
        /* let's allow:
           PAGE label
           BANK [# | value] */
        if (flags & ASM_BIT_BANK) {
          /* apparently there's some assembly command ``bank n'' */
          /* so allow it here */
          val = pfile_value_find(pf, pfile_log_none,
              pf_token_get(pf, pf_token_current));
          if (val && !value_is_const(val)) {
            pf_token_get(pf, pf_token_next);
          } else {
            if (val) {
              value_release(val);
            }
            val = jal_parse_expr(pf);
            if (val) {
              if (!value_is_const(val)) {
                pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
                err = boolean_true;
              }
              value_release(val);
              val = VALUE_NONE;
              flags &= ~ASM_BIT_BANK;
            }
          }
        } else if (flags & ASM_BIT_PAGE) {
          lbl = pfile_label_find(pf, pfile_log_err, 
                  pf_token_get(pf, pf_token_current));
          if (lbl) {
            pf_token_get(pf, pf_token_next);
          } else {
            err = boolean_true;
          }
        } else {
          err = boolean_true;
          pfile_log(pf, pfile_log_err, PFILE_MSG_SYNTAX_ERROR);
        }
      } else {
        /* process the cmd */
        op = asm_cmds[ii].op;
        pf_token_get(pf, pf_token_next);

        switch (asm_cmds[ii].type) {
          case pic_optype_none:
            /* nothing more to do */
            break;
          case pic_optype_f:
          case pic_optype_f_d:
          case pic_optype_f_b:
            /* allow ''BANK'', a variable name, or a constant expression */
            if (pf_token_is(pf, pf_token_current, pfile_log_none, "bank")) {
              flags |= ASM_BIT_BANK;
              pf_token_get(pf, pf_token_next);
            }
            val = pfile_value_find(pf, pfile_log_none,
                pf_token_get(pf, pf_token_current));
            if (val) {
              value_t cexpr;

              pf_token_get(pf, pf_token_next);
              if (value_is_array(val)
                  && (pf_token_is(pf, pf_token_current, pfile_log_none, "["))) {
                value_t               subs;

                pf_token_get(pf, pf_token_next);
                subs = jal_parse_expr(pf);
                if (subs) {
                  variable_def_t        def;
                  variable_def_member_t mbr;

                  mbr = variable_def_member_get(
                          value_def_get(val));
                  def = variable_def_member_def_get(mbr);

                  if (!value_is_const(subs)) {
                    pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
                  } else if (value_const_get(subs) 
                      >= variable_def_member_ct_get(mbr)) {
                    pfile_log(pf, pfile_log_err, "subscript out of range");
                  } else {
                    if (variable_def_sz_get(def) > 1) {
                      value_t tmp;

                      tmp = pfile_constant_get(pf, variable_def_sz_get(def)
                          * value_const_get(subs), VARIABLE_DEF_NONE);
                      value_release(subs);
                      subs = tmp;
                    }
                    value_baseofs_set(val, subs);
                  }
                  value_release(subs);
                }
                if (pf_token_is(pf, pf_token_current, pfile_log_err, "]")) {
                  pf_token_get(pf, pf_token_next);
                }
              }
              if (pf_token_is(pf, pf_token_current, pfile_log_none, "+")
                || pf_token_is(pf, pf_token_current, pfile_log_none, "-")) {
                cexpr = jal_parse_expr(pf);
                if (cexpr) {
                  if (!value_is_const(cexpr)) {
                    pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
                  } else {
                    valofs = value_const_get(cexpr);
                  }
                  value_release(cexpr);
                }
              }
            } else {
              val = jal_parse_expr(pf);
              if (val && !value_is_const(val)) {
                pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
                value_release(val);
                val = VALUE_NONE;
                err = boolean_true;
              }
            }
            if (pic_optype_f == asm_cmds[ii].type) {
              if (asm_cmd_special_movfw == asm_cmds[ii].special) {
                n = pfile_value_find(pf, pfile_log_none, "w");
                n_lo = 0;
                n_hi = 1;
              }
            } else if (pic_optype_f_d == asm_cmds[ii].type) {
              if (pf_token_is(pf, pf_token_current, pfile_log_err, ",")) {
                pf_token_get(pf, pf_token_next);
              }
              n = jal_parse_expr(pf);
              n_lo = 0;
              n_hi = 1;
            } else {
              /* if value is a bit variable, position is assumed */
              if (!value_dflag_test(val, VARIABLE_DEF_FLAG_BIT)) {
                if (pf_token_is(pf, pf_token_current, pfile_log_err, ",")) {
                  pf_token_get(pf, pf_token_next);
                }
                n = jal_parse_expr(pf);
                n_lo = 0;
                n_hi = 7;
              }
            }
            break;
          case pic_optype_n:
            n = jal_parse_expr(pf);
            n_lo = 0;
            n_hi = 255;
            break;
          case pic_optype_tris:
            n = jal_parse_expr(pf);
            n_lo = 5;
            n_hi = 7;
            break;
          case pic_optype_k:
            if (pf_token_is(pf, pf_token_current, pfile_log_none, "page")) {
              flags |= ASM_BIT_PAGE;
              pf_token_get(pf, pf_token_next);
            }
            lbl = pfile_label_find(pf, pfile_log_none, pf_token_ptr_get(pf));
            if (lbl) {
              pf_token_get(pf, pf_token_next);
            } else {
              n = jal_parse_expr(pf);
              if ((VALUE_NONE == n) || !value_is_const(n)) {
                pfile_log(pf, pfile_log_err, "constant expression expected");
                value_release(n);
                n = VALUE_NONE;
                err = boolean_true;
              } else {
                lbl = pfile_label_alloc(pf, 0);
                label_pc_set(lbl, value_const_get(n));
                label_flag_set(lbl, LABEL_FLAG_DEFINED);
                label_flag_set(lbl, LABEL_FLAG_USER);
                value_release(n);
                n = VALUE_NONE;
              }
            }
            break;
          case pic_optype_db:
            /* this is special cased above */
            break;
        }
      }
      if (n) {
        if (!value_is_const(n)) {
          pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_EXPECTED);
          err = boolean_true;
        } else if ((value_const_get(n) < n_lo) 
          || (value_const_get(n) > n_hi)) {
          pfile_log(pf, pfile_log_err, PFILE_MSG_CONSTANT_RANGE);
          err = boolean_true;
        }
        if (err) {
          value_release(n);
          n = VALUE_NONE;
        }
      }

      if (!err) {
        asm_cmd_special_t special;

        special = (ii < COUNT(asm_cmds))
          ? asm_cmds[ii].special
          : asm_cmd_special_none;

        if (asm_cmd_special_none != special) {
          /* do the flags before anything else */
          flags |= macro_xlate[special].asm_bit;
        }
        if ((flags & ASM_BIT_PAGE) && !pic_is_16bit(pf))  {
          if (!lbl) {
            if (pfile_flag_test(pf, PFILE_FLAG_WARN_MISC)) {
              pfile_log(pf, pfile_log_warn, 
                  "PAGE used without an associated label");
            }
          } else {
            /* generate page bits */
            value_t code_sz;
            flag_t  keep;

            keep = ((asm_cmd_special_lcall == special)
              || (asm_cmd_special_lgoto == special)
              || pfile_proc_flag_test(pfile_proc_active_get(pf), 
                   PFILE_PROC_FLAG_KEEP_PAGE))
              ? PIC_CODE_FLAG_NO_OPTIMIZE
              : 0;

            code_sz = pfile_value_find(pf, pfile_log_err, "_code_size");
            if (value_const_get(code_sz) > 2048) {
              asm_cmd_add(pf, pic_opcode_branchlo_set, VALUE_NONE, 0,
                  pic_opdst_none,
                  VALUE_NONE, lbl, keep, 0, 0);
            }
            if (value_const_get(code_sz) > 4096) {
              asm_cmd_add(pf, pic_opcode_branchhi_set, VALUE_NONE, 0,
                  pic_opdst_none,
                  VALUE_NONE, lbl, keep, 0, 0);
            }
            value_release(code_sz);
          }
        }
        if (flags & ASM_BIT_BANK) {
          if (!val) {
            if (pfile_flag_test(pf, PFILE_FLAG_WARN_MISC)) {
              pfile_log(pf, pfile_log_warn,
                  "BANK used without an associated value");
            }
          } else {
            if (pic_is_16bit(pf)) {
              value_t lb;

              lb = pfile_constant_get(pf, 0, VARIABLE_DEF_NONE);
              asm_cmd_add(pf, pic_opcode_movlb, lb, 0, pic_opdst_none,
                VALUE_NONE, LABEL_NONE, 0, 0, 0);
              value_release(lb);
            } else {
              asm_cmd_add(pf, pic_opcode_datalo_set, val, 0, pic_opdst_none,
                  VALUE_NONE, LABEL_NONE, 
                  pfile_proc_flag_test(pfile_proc_active_get(pf),
                    PFILE_PROC_FLAG_KEEP_BANK) ? PIC_CODE_FLAG_NO_OPTIMIZE : 0,
                    0, 0);
              asm_cmd_add(pf, pic_opcode_datahi_set, val, 0, pic_opdst_none,
                  VALUE_NONE, LABEL_NONE,
                  pfile_proc_flag_test(pfile_proc_active_get(pf),
                    PFILE_PROC_FLAG_KEEP_BANK) ? PIC_CODE_FLAG_NO_OPTIMIZE : 0,
                    0, 0);
            }
          }
        }
        if (asm_cmd_special_none != special) {
          /* do the translation; first any status bits */
          if (pic_opcode_none != macro_xlate[special].flag_op) {
            value_t status;
            value_t status_flag;

            status = pfile_value_find(pf, pfile_log_err, "_status");
            status_flag = pfile_value_find(pf, pfile_log_err, 
              macro_xlate[special].flag);
            asm_cmd_add(pf, macro_xlate[special].flag_op, status, 0,
              pic_opdst_none, status_flag, LABEL_NONE, PIC_CODE_FLAG_NO_OPTIMIZE,
              0, 0);
            value_release(status_flag);
            value_release(status);
          }
          op = macro_xlate[special].op;
          /* for lack of a better way to do this, I'm adding some special
             processing here */
          if (asm_cmd_special_movfw == special) {
            opdst = pic_opdst_w;
            value_release(n);
            n = VALUE_NONE;
          } else if (asm_cmd_special_tstf == special) {
            opdst = pic_opdst_f;
          } else if (asm_cmd_special_negf == special) {
            asm_cmd_add(pf, pic_opcode_comf, val, valofs, pic_opdst_f, VALUE_NONE,
              LABEL_NONE, PIC_CODE_FLAG_NO_OPTIMIZE, 0, 0);
          }
        }
        if (pic_optype_f_d == asm_cmds[ii].type) {
          opdst = value_const_get(n) ? pic_opdst_f : pic_opdst_w;
          value_release(n);
          n = VALUE_NONE;
        }

        asm_cmd_add(pf, op, val, valofs, opdst, n, lbl, PIC_CODE_FLAG_NO_OPTIMIZE,
          0, 0);
        label_release(lbl);
        value_release(n);
        value_release(val);
      }
    }
  }
}

void jal_parse_assembler(pfile_t *pf, const pfile_pos_t *statement_start)
{
  pfile_pos_t pos_start;

  pf_token_start_get(pf, &pos_start);
  pfile_block_enter(pf);
  while (!pf_token_is_eof(pf)
    && !pf_token_is(pf, pf_token_current, pfile_log_none, "end")) {
    pfile_token_ct_t token_ct;
    const char      *ptr;

    token_ct = pfile_token_ct_get(pf);
    pf_token_start_get(pf, &pos_start);
    pfile_statement_start_set(pf, &pos_start);
    ptr = pf_token_get(pf, pf_token_current);
    if (pf_token_is(pf, pf_token_current, pfile_log_none, "pragma")) {
      /* allow pragmas in assembler blocks. who knew? */
      pf_token_get(pf, pf_token_next);
      jal_parse_pragma(pf, statement_start);
    } else {
      jal_parse_asm(pf, statement_start);
    }
    /* if nothing has been parsed we need to skip the current token */
    if (token_ct == pfile_token_ct_get(pf)) {
      pf_token_get(pf, pf_token_next);
    }
  }
  if (pf_token_is(pf, pf_token_next, pfile_log_err, "assembler")) {
    pf_token_get(pf, pf_token_next);
  } else {
    jal_block_start_show(pf, "ASSEMBLER", statement_start);
  }

  pfile_block_leave(pf);
  pf_token_start_get(pf, &pos_start);
  pfile_statement_start_set(pf, &pos_start);
}

