/**********************************************************
 **
 ** cmd.c : the cmd_type base functions
 **
 ** Copyright (c) 2004-2005, Kyle A. York
 ** All rights reserved
 **
 ***********************************************************/
#include <assert.h>
#include "cmdd.h"

static cache_t   cmd_cache;
static boolean_t cmd_cache_is_init;

static void cmd_cache_cleanup(void)
{
  cache_cleanup(&cmd_cache);
}

static cmd_t cmd_element_alloc(void)
{
  if (!cmd_cache_is_init) {
    cmd_cache_is_init = boolean_true;
    atexit(cmd_cache_cleanup);
    cache_init(&cmd_cache, sizeof(struct cmd_), "cmd");
  }
  return cache_element_alloc(&cmd_cache);
}

struct cmd_ *cmd_element_seek(cmd_t el, boolean_t mod)
{
  return cache_element_seek(&cmd_cache, el, mod);
}

cmd_t cmd_alloc(cmd_type_t type, 
  const cmd_vtbl_t *vtbl)
{
  cmd_t cmd;

  cmd = cmd_element_alloc();
  if (cmd) {
    struct cmd_ *ptr;
    
    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      ptr->link  = 0;
      ptr->flags = 0;
      ptr->pos.src   = 0;
      ptr->pos.line  = 0;
      ptr->type  = type;
      ptr->opt   = 0;
      ptr->vtbl  = vtbl;
      ptr->u.proc = PFILE_PROC_NONE;
    }
  }
  return cmd;
}

void cmd_free(cmd_t cmd)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr) {
    if (ptr->vtbl && ptr->vtbl->free_fn) {
      ptr->vtbl->free_fn(cmd);
    }
    pfile_source_release(ptr->pos.src);
    cache_element_free(&cmd_cache, cmd);
  }
}

cmd_t cmd_dup(const cmd_t cmd)
{
  const struct cmd_ *ptr;
  cmd_t        dup;

  ptr = cmd_element_seek(cmd, boolean_false);
  if (!ptr) {
    dup = CMD_NONE;
  } else {
    if (ptr->vtbl && ptr->vtbl->dup_fn) {
      dup = ptr->vtbl->dup_fn(cmd);
    } else {
      dup = cmd_alloc(cmd_type_get(cmd), ptr->vtbl);
    }
    if (dup) {
      cmd_flag_set_all(dup, cmd_flag_get_all(cmd));
      cmd_line_set(dup, cmd_line_get(cmd));
      cmd_source_set(dup, cmd_source_get(cmd));
    }
  }
  return dup;
}

void cmd_label_remap(cmd_t cmd, const label_map_t *map)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr && ptr->vtbl && ptr->vtbl->label_remap_fn) {
    ptr->vtbl->label_remap_fn(cmd, map);
  }
}

void cmd_variable_remap(cmd_t cmd, const variable_map_t *map)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr && ptr->vtbl && ptr->vtbl->variable_remap_fn) {
    ptr->vtbl->variable_remap_fn(cmd, map);
  }
}

void cmd_value_remap(cmd_t cmd, const value_map_t *map)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr && ptr->vtbl && ptr->vtbl->value_remap_fn) {
    ptr->vtbl->value_remap_fn(cmd, map);
  }
}

void cmd_list_free(cmd_t cmd)
{
  while (cmd) {
    cmd_t link;

    link = cmd_link_get(cmd);
    cmd_free(cmd);
    cmd = link;
  }
}

cmd_t cmd_link_get(const cmd_t cmd)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr) ? ptr->link : 0;
}

void         cmd_link_set(cmd_t cmd, cmd_t link)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr) {
    ptr->link = link;
  }
}

cmd_type_t cmd_type_get(const cmd_t cmd)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr) ? ptr->type : cmd_type_statement_end;
}

void       cmd_type_set(cmd_t cmd, cmd_type_t type)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr) {
    ptr->type = type;
  }
}

unsigned cmd_opt_get(const cmd_t cmd)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr) ? ptr->opt : 0;
}

void cmd_opt_set(cmd_t cmd, unsigned opt)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr) {
    ptr->opt = opt;
  }
}

unsigned   cmd_line_get(const cmd_t cmd)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr) ? ptr->pos.line : -1;
}

void       cmd_line_set(cmd_t cmd, unsigned line)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr) {
    ptr->pos.line = line;
  }
}

pfile_source_t *cmd_source_get(const cmd_t cmd)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr) ? ptr->pos.src : 0;
}

void cmd_source_set(cmd_t cmd, pfile_source_t *src)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr) {
    pfile_source_lock(src);
    if (ptr->pos.src) {
      pfile_source_release(ptr->pos.src);
    }
    ptr->pos.src = src;
  }
}

void cmd_dump(cmd_t cmd, FILE *dst)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
#if 0
  if (ptr->label) { /*  && label_usage_get(cmd->label)) { */
    fprintf(dst, "%s: ", label_name_get(ptr->label));
  }
#endif
  if (ptr->vtbl && ptr->vtbl->dump_fn) {
    ptr->vtbl->dump_fn(cmd, dst);
  } else {
    const char *str;

    str = "{unknown}";
    switch (ptr->type) {
      case cmd_type_isr_cleanup:   str = "{isr_cleanup}";  break;
      case cmd_type_end:           str = "{end}";          break;
      case cmd_type_sleep:         str = "{sleep}";        break;
      case cmd_type_nop:           str = "{nop}";          break;
      case cmd_type_branch:        str = "{branch}";       break;
      case cmd_type_operator:      str = "{operator}";     break;
      case cmd_type_proc_enter:    str = "{enter: %s}";    break;
      case cmd_type_proc_leave:    str = "{leave: %s}";    break;
      case cmd_type_block_start:   str = "{block}";        break;
      case cmd_type_block_end:     str = "{end-of-block}"; break;
      case cmd_type_statement_end: str = "{eos}";          break;
      case cmd_type_asm:           str = "{asm}";          break;
      case cmd_type_label:         str = "{label}";        break;
      case cmd_type_usec_delay:    str = "{usec-delay}";   break;
      case cmd_type_assert:        str = "{assert}";       break;
      case cmd_type_comment:       str = "{comment}";      break;
    }
    if ((cmd_type_proc_enter == ptr->type)
      || (cmd_type_proc_leave == ptr->type)) {
      fprintf(dst, str, pfile_proc_tag_get(cmd_proc_get(cmd)));
    } else {
      fprintf(dst, str);
    }
  }
  fputc('\n', dst);
}

void cmd_flag_set_all(cmd_t cmd, flag_t flags)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr) {
    ptr->flags = flags;
  }
}

flag_t cmd_flag_get_all(cmd_t cmd)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr) ? ptr->flags : 0;
}

void cmd_flag_set(cmd_t cmd, flag_t flag)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr) {
    ptr->flags |= flag;
  }
}

void cmd_flag_clr(cmd_t cmd, flag_t flag)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr) {
    ptr->flags = (ptr->flags & ~flag);
  }
}

boolean_t cmd_flag_test(const cmd_t cmd, flag_t flag)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr) 
    ? ((ptr->flags & flag) == flag)
    : 0;
}

pfile_proc_t *cmd_proc_get(const cmd_t cmd)
{
  pfile_proc_t *proc;

  proc = 0;
  if ((cmd_type_proc_enter == cmd_type_get(cmd))
    || (cmd_type_proc_leave == cmd_type_get(cmd))) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_false);
    if (ptr) {
      proc = ptr->u.proc;
    }
  }
  return proc;
}

static void cmd_proc_param_ctrs_bump(pfile_proc_t *proc, ctr_bump_t dir)
{
  size_t ii;

  /* bump the counters on the new proc parameters */
  for (ii = 0; ii < pfile_proc_param_ct_get(proc); ii++) {
    value_t val;

    val  = pfile_proc_param_get(proc, ii);
    if (value_dflag_test(val, VARIABLE_DEF_FLAG_IN)) {
      value_assign_ct_bump(val, dir);
    }
    if (value_dflag_test(val, VARIABLE_DEF_FLAG_OUT)) {
      value_use_ct_bump(val, dir);
    }
  }
}

void cmd_proc_set(cmd_t cmd, pfile_proc_t *proc)
{
  if ((cmd_type_proc_enter == cmd_type_get(cmd))
    || (cmd_type_proc_leave == cmd_type_get(cmd))) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      if ((cmd_type_proc_enter == cmd_type_get(cmd))
        || (cmd_type_proc_leave == cmd_type_get(cmd))) {
        label_cmd_set(pfile_proc_label_get(ptr->u.proc), CMD_NONE);
        cmd_proc_param_ctrs_bump(proc, ctr_bump_incr);
        cmd_proc_param_ctrs_bump(ptr->u.proc, ctr_bump_decr);
        ptr->u.proc = proc;
        if (cmd_type_proc_enter == cmd_type_get(cmd)) {
          label_cmd_set(pfile_proc_label_get(proc), cmd);
        }
      }
    }
  }
}

static void cmd_label_free(cmd_t cmd)
{
  cmd_label_set(cmd, LABEL_NONE);
}

static void cmd_label_dump(cmd_t cmd, FILE *dst)
{
  fprintf(dst, "%s:", label_name_get(cmd_label_get(cmd)));
}

static cmd_t cmd_label_dup(const cmd_t cmd)
{
  /* we cannot simply pass the label to this, 'cause the label
   * points to a command & we don't want to change that. fortunately,
   * this should only be used when the labels are going to be remapped
   */ 
  return cmd_label_alloc(LABEL_NONE);
}

static void cmd_label_label_remap(cmd_t cmd, const label_map_t *map)
{
  label_t lbl;

  lbl = label_map_find(map, cmd_label_get(cmd));
  if (lbl) {
    cmd_label_set(cmd, lbl);
  }
}

static const cmd_vtbl_t cmd_label_vtbl = {
  cmd_label_free,
  cmd_label_dump,
  cmd_label_dup,
  cmd_label_label_remap,
  0, /* ignore variable map          */
  0, /* ignore value map             */
  0, /* ignore cmd variable accessed */
  0  /* ignore cmd value accessed    */
};

cmd_t cmd_label_alloc(label_t lbl)
{
  cmd_t cmd;

  cmd = cmd_alloc(cmd_type_label, &cmd_label_vtbl);
  if (cmd) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      ptr->u.label = LABEL_NONE;
      cmd_label_set(cmd, lbl);
    }
  }
  return cmd;
}

label_t cmd_label_get(const cmd_t cmd)
{
  struct cmd_ *ptr;
  label_t lbl;

  lbl = LABEL_NONE;
  ptr = cmd_element_seek(cmd, boolean_false);
  if (ptr) {
    if (cmd_type_label == cmd_type_get(cmd)) {
      lbl = ptr->u.label;
    } else if (cmd_type_proc_enter == cmd_type_get(cmd)) {
      lbl = pfile_proc_label_get(ptr->u.proc);
    }
  }
  return lbl;
}

void cmd_label_set(cmd_t cmd, label_t lbl)
{
  if (cmd_type_label == cmd_type_get(cmd)) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      label_lock(lbl);
      if (ptr->u.label) {
        label_release(ptr->u.label);
      }
      ptr->u.label = lbl;
      label_cmd_set(lbl, cmd);
    }
  }
}

/*
 * NAME
 *   cmd_next_exec_get
 *
 * DESCRIPTION
 *   get the next *executable* cmd
 *
 * PARAMETERS
 *   cmd : starting point
 *
 * RETURN
 *   next executable command
 *
 * NOTES
 *   this skips non-exectuable commands. it's helpful when looking
 *   past all labels
 */
boolean_t cmd_is_executable(cmd_t cmd)
{
  return (cmd_type_nop != cmd_type_get(cmd))
      && (cmd_type_label != cmd_type_get(cmd))
      && (cmd_type_block_start != cmd_type_get(cmd))
      && (cmd_type_block_end != cmd_type_get(cmd))
      && (cmd_type_statement_end != cmd_type_get(cmd));
}

cmd_t cmd_next_exec_get(cmd_t cmd)
{
  while (cmd && !cmd_is_executable(cmd)) {
    cmd = cmd_link_get(cmd);
  }
  return cmd;
}

/*
 * NAME
 *   cmd_label_find
 *
 * DESCRIPTION
 *   find a label in the command list
 *
 * PARAMETERS
 *   cmd : head of list
 *
 * RETURN
 *   label, or 0
 *
 * NOTES
 */
cmd_t cmd_label_find(cmd_t cmd, const label_t lbl)
{
  if (!lbl) {
    cmd = 0;
  } else {
#if 0
    while (cmd && (cmd_label_get(cmd) != lbl)) {
      cmd = cmd_link_get(cmd);
    }
#else
    cmd = label_cmd_get(lbl);
    if (cmd) {
      if (cmd_label_get(cmd) != lbl) {
        fprintf(stderr, "looking for %s got %s\n",
            label_name_get(lbl),
            label_name_get(cmd_label_get(cmd)));
      }
    }
#endif
    if (!cmd) {
      fprintf(stderr, "!!!LABEL %s NOT FOUND!!!\n", label_name_get(lbl));
    }
  }
  return cmd;
}

void cmd_remove(cmd_t *cmd_head, cmd_t cmd)
{
  cmd_t cmd_pv;

  if (cmd == *cmd_head) {
    cmd_pv = CMD_NONE;
  } else {
    for (cmd_pv = *cmd_head;
         cmd_link_get(cmd_pv) != cmd;
         cmd_pv = cmd_link_get(cmd_pv))
      ;
    if (!cmd_pv) {
      assert(0);
      cmd = CMD_NONE;
    }
  }
  if (CMD_NONE != cmd) {
    if (cmd_pv) {
      cmd_link_set(cmd_pv, cmd_link_get(cmd));
    } else {
      *cmd_head = cmd_link_get(cmd);
    }
    cmd_free(cmd);
  }
}

static void cmd_proc_free(cmd_t cmd)
{
  cmd_proc_set(cmd, PFILE_PROC_NONE);
}

static void cmd_proc_dump(cmd_t cmd, FILE *dst)
{
  pfile_proc_t *proc;

  proc = cmd_proc_get(cmd);
  fprintf(dst, "{%s %s",
      (cmd_type_proc_enter == cmd_type_get(cmd))
      ? "enter" : "leave",
      pfile_proc_tag_get(proc));
  if (cmd_type_proc_enter == cmd_type_get(cmd)) {
    size_t                ii;
    variable_def_member_t mbr;

    fprintf(dst, "(");
    mbr = variable_def_member_get(pfile_proc_def_get(proc));
    for (ii = 1; ii < pfile_proc_param_ct_get(proc); ii++) {
      mbr = variable_def_member_link_get(mbr);
      fprintf(dst, 
        "%s%s", (ii > 1) ? ", " : "",
        variable_def_member_tag_get(mbr));
    }
    fprintf(dst, ")");
  } 
  fprintf(dst, "}");
}

static cmd_t cmd_proc_dup(const cmd_t cmd)
{
  return cmd_proc_alloc(cmd_type_get(cmd), cmd_proc_get(cmd));
}

static const cmd_vtbl_t cmd_proc_vtbl = {
  cmd_proc_free,
  cmd_proc_dump,
  cmd_proc_dup,
  0,  /* ignore label remap           */
  0,  /* ignore variable remap        */
  0,  /* ignore value remap           */
  0,  /* ignore cmd variable accessed */
  0   /* ignore cmd value accessed    */
};

cmd_t cmd_proc_alloc(cmd_type_t type, pfile_proc_t *proc)
{
  cmd_t        cmd;
  struct cmd_ *ptr;

  cmd = cmd_alloc(type, &cmd_proc_vtbl);
  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr) {
    ptr->u.proc = PFILE_PROC_NONE;
  }
  cmd_proc_set(cmd, proc);
  return cmd;
}

boolean_t cmd_is_reachable(cmd_t cmd)
{
  return cmd_flag_test(cmd, CMD_FLAG_USER)
    || cmd_flag_test(cmd, CMD_FLAG_INTERRUPT);
}

void cmd_label_remap2(cmd_t cmd, const label_map_t *map,
    label_t (*cmd_lbl_get)(const cmd_t cmd),
    void (*cmd_lbl_set)(cmd_t cmd, label_t lbl))
{
  label_t lbl;

  lbl = label_map_find(map, cmd_lbl_get(cmd));
  if (lbl) {
    cmd_lbl_set(cmd, lbl);
  }
}

void cmd_variable_remap2(cmd_t cmd, const variable_map_t *map,
    value_t (*cmd_val_get)(const cmd_t cmd), 
    void (*cmd_val_set)(cmd_t cmd, value_t n))
{
  value_t val;

  val = value_variable_remap(cmd_val_get(cmd), map);
  if (val) {
    cmd_val_set(cmd, val);
    value_release(val);
  }
}

void cmd_value_remap2(cmd_t cmd, const value_map_t *map,
    value_t (*cmd_val_get)(const cmd_t cmd),
    void (*cmd_val_set)(cmd_t cmd, value_t n))
{
  value_t val;

  val = value_map_find(map, cmd_val_get(cmd));
  if (val) {
    cmd_val_set(cmd, val);
  }
}

flag_t cmd_variable_accessed_get(const cmd_t cmd, variable_t var)
{
  struct cmd_ *ptr;
  flag_t       flags;

  flags = CMD_VARIABLE_ACCESS_FLAG_NONE;
  ptr = cmd_element_seek(cmd, boolean_false);
  if (ptr && ptr->vtbl && ptr->vtbl->variable_accessed_fn) {
    flags = ptr->vtbl->variable_accessed_fn(cmd, var);
  }
  return flags;
}

flag_t cmd_value_accessed_get(const cmd_t cmd, value_t val)
{
  struct cmd_ *ptr;
  flag_t       flags;

  flags = CMD_VARIABLE_ACCESS_FLAG_NONE;
  ptr = cmd_element_seek(cmd, boolean_false);
  if (ptr && ptr->vtbl && ptr->vtbl->value_accessed_fn) {
    flags = ptr->vtbl->value_accessed_fn(cmd, val);
  }
  return flags;
}

value_t cmd_assert_value_get(const cmd_t cmd)
{
  value_t val;

  val = VALUE_NONE;
  if (cmd_type_assert == cmd_type_get(cmd)) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_false);
    if (ptr) {
      val = ptr->u.val;
    }
  }
  return val;
}

void cmd_assert_value_set(cmd_t cmd, value_t val)
{
  if (cmd_type_assert == cmd_type_get(cmd)) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      value_lock(val);
      value_assign_ct_bump(val, ctr_bump_incr);
      value_use_ct_bump(val, ctr_bump_incr);
      value_assign_ct_bump(ptr->u.val, ctr_bump_decr);
      value_use_ct_bump(ptr->u.val, ctr_bump_decr);
      value_release(ptr->u.val);
      ptr->u.val = val;
    }
  }
}

static void cmd_assert_free(cmd_t cmd)
{
  cmd_assert_value_set(cmd, VALUE_NONE);
}

static void cmd_assert_dump(cmd_t cmd, FILE *dst)
{
  value_dump(cmd_assert_value_get(cmd), dst);
}

static cmd_t cmd_assert_dup(const cmd_t cmd)
{
  return cmd_assert_alloc(cmd_assert_value_get(cmd));
}

static const cmd_vtbl_t cmd_assert_vtbl = {
  cmd_assert_free,
  cmd_assert_dump,
  cmd_assert_dup,
  0, /* ignore label remap           */
  0, /* ignore variable map          */
  0, /* ignore value map             */
  0, /* ignore cmd variable accessed */
  0  /* ignore cmd value accessed    */
};

cmd_t cmd_assert_alloc(value_t val)
{
  cmd_t cmd;

  cmd = cmd_alloc(cmd_type_assert, &cmd_assert_vtbl);
  if (cmd) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      ptr->u.val = VALUE_NONE;
      cmd_assert_value_set(cmd, val);
    }
  }
  return cmd;
}


