/**********************************************************
 **
 ** cmd_br.c : the cmd_type_branch functions
 **
 ** Copyright (c) 2004-2005, Kyle A. York
 ** All rights reserved
 **
 ***********************************************************/
#include <assert.h>
#include "../libutils/mem.h"
#include "cmdd.h"
#include "cmd_brch.h"

static void cmd_branch_dump(cmd_t cmd, FILE *dst)
{
  const char *tstr;
  const char *cstr;

  tstr = "{unknown}";
  switch (cmd_brtype_get(cmd)) {
    case cmd_branchtype_none:         tstr = "{none}";       break;
    case cmd_branchtype_goto:         tstr = "goto";         break;
    case cmd_branchtype_call:         tstr = "call";         break;
    case cmd_branchtype_return:       tstr = "return";       break;
    case cmd_branchtype_task_start:   tstr = "task_start";   break;
    case cmd_branchtype_task_suspend: tstr = "task_suspend"; break;
    case cmd_branchtype_task_end:     tstr = "task_end";     break;
  }
  cstr = "{unknown}";
  switch (cmd_brcond_get(cmd)) {
    case cmd_branchcond_none:   cstr = 0 ;      break;
    case cmd_branchcond_true:   cstr = "true";  break;
    case cmd_branchcond_false:  cstr = "false"; break;
  }
  fprintf(dst, "%s ", tstr);
  if (cstr) {
    fprintf(dst, "%s ? ", cstr);
    value_dump(cmd_brval_get(cmd), dst);
  }
  if (cmd_brdst_get(cmd)) {
    if (cstr) {
      fprintf(dst, "-->");
    }
    fprintf(dst, "%s", label_name_get(cmd_brdst_get(cmd)));
  } else if (cmd_brproc_get(cmd)) {
    variable_def_t        vdef;
    variable_def_member_t mbr;
    size_t                ii;
    value_t              *params;
    value_t               proc;

    proc   = cmd_brproc_get(cmd);
    params = cmd_brproc_params_get(cmd);

    if (cstr) {
      fprintf(dst, "-->");
    }
    fprintf(dst, "%s%s(", value_is_indirect(proc)
      ? "&" : "", value_name_get(proc));
    vdef = value_def_get(proc);
    for (mbr = variable_def_member_get(vdef), ii = 0;
         mbr;
         mbr = variable_def_member_link_get(mbr), ii++) {
      if (ii) {
        fputc(',', dst);
      }
      value_dump(params[ii], dst);
    }
    fprintf(dst, ")");
  }
}

static void cmd_branch_free(cmd_t cmd)
{
  struct cmd_ *ptr;

  cmd_brval_set(cmd, 0);
  cmd_brdst_set(cmd, 0);

  ptr = cmd_element_seek(cmd, boolean_true);
  if (ptr) {
    if (ptr->u.br.proc) {
      value_t              *params;
      variable_def_t        def;
      variable_def_member_t mbr;
      size_t                ii;
      pfile_proc_t         *proc_ptr;

      value_use_ct_bump(ptr->u.br.proc, ctr_bump_decr);
      def    = value_def_get(ptr->u.br.proc);
      params = ptr->u.br.proc_params;
      for (ii = 0, mbr = variable_def_member_get(def);
           mbr;
           ii++, mbr = variable_def_member_link_get(mbr)) {
        variable_def_t mdef;

        mdef = variable_def_member_def_get(mbr);
        /* if ii == 0 this is the return value. we'll assume it's used */
        if (!ii || variable_def_flag_test(mdef, VARIABLE_DEF_FLAG_IN)) {
          value_use_ct_bump(params[ii], ctr_bump_decr);
        }
        if (variable_def_flag_test(mdef, VARIABLE_DEF_FLAG_OUT)) {
          value_assign_ct_bump(params[ii], ctr_bump_decr);
        }
        value_release(params[ii]);
      }
      FREE(ptr->u.br.proc_params);
      proc_ptr = value_proc_get(ptr->u.br.proc);
      if (proc_ptr) {
        label_usage_bump(pfile_proc_label_get(proc_ptr), ctr_bump_decr);
      }
      value_release(ptr->u.br.proc);
    }
  }
}

static cmd_t cmd_branch_dup(const cmd_t cmd)
{
  value_t *dup_params;
  value_t *params;

  params = cmd_brproc_params_get(cmd);
  if (params) {
    dup_params = CALLOC(sizeof(*dup_params), cmd_brproc_param_ct_get(cmd));
    if (dup_params) {
      size_t ii;

      for (ii = 0; ii < cmd_brproc_param_ct_get(cmd); ii++) {
        dup_params[ii] = params[ii];
        value_lock(dup_params[ii]);
      }
    }
  } else {
    dup_params = 0;
  }

  return cmd_branch_alloc(cmd_brtype_get(cmd),
      cmd_brcond_get(cmd), cmd_brdst_get(cmd),
      cmd_brval_get(cmd), cmd_brproc_get(cmd),
      dup_params);
}

static void cmd_branch_label_remap(cmd_t cmd, const label_map_t *map)
{
  cmd_label_remap2(cmd, map, cmd_brdst_get, cmd_brdst_set);
}

static void cmd_branch_variable_remap(cmd_t cmd, const variable_map_t *map)
{
  size_t        ii;
  value_t      *params;
  pfile_proc_t *proc;

  cmd_variable_remap2(cmd, map, cmd_brval_get, cmd_brval_set);
  cmd_variable_remap2(cmd, map, cmd_brproc_get, cmd_brproc_set);
  params = cmd_brproc_params_get(cmd);
  proc = value_proc_get(cmd_brproc_get(cmd));
  for (ii = 0; ii < cmd_brproc_param_ct_get(cmd); ii++) {
    value_t tmp;

    tmp = value_variable_remap(params[ii], map);
    if (tmp) {
      cmd_brproc_param_set(cmd, ii, tmp);
      value_release(tmp);
    }
  }
}

static void cmd_branch_value_remap(cmd_t cmd, const value_map_t *map)
{
  size_t   ii;
  value_t *params;

  cmd_value_remap2(cmd, map, cmd_brval_get, cmd_brval_set);
  cmd_value_remap2(cmd, map, cmd_brproc_get, cmd_brproc_set);
  params = cmd_brproc_params_get(cmd);
  for (ii = 0; ii < cmd_brproc_param_ct_get(cmd); ii++) {
    value_t tmp;

    tmp = value_map_find(map, params[ii]);
    if (tmp) {
      cmd_brproc_param_set(cmd, ii, tmp);
    }
  }
}

static flag_t cmd_branch_variable_accessed(const cmd_t cmd, variable_t var)
{
  flag_t                flags;
  size_t                ii;
  variable_def_t        def;
  variable_def_member_t mbr;
  value_t              *params;

  flags = CMD_VARIABLE_ACCESS_FLAG_NONE;
  if (value_variable_get(cmd_brval_get(cmd)) == var) {
    flags |= CMD_VARIABLE_ACCESS_FLAG_READ;
  }
  def = value_def_get(cmd_brproc_get(cmd));
  params = cmd_brproc_params_get(cmd);
  for (ii = 0, mbr = variable_def_member_get(def);
       mbr;
       ii++, mbr = variable_def_member_link_get(mbr)) {
    if (var == value_variable_get(params[ii])) {
      if (variable_def_flag_test(variable_def_member_def_get(mbr),
        VARIABLE_DEF_FLAG_IN)) {
        flags |= CMD_VARIABLE_ACCESS_FLAG_READ;
      }
      if (variable_def_flag_test(variable_def_member_def_get(mbr),
        VARIABLE_DEF_FLAG_OUT)) {
        flags |= CMD_VARIABLE_ACCESS_FLAG_WRITTEN;
      }
    }
  }
  return flags;
}

static flag_t cmd_branch_value_accessed(const cmd_t cmd, value_t val)
{
  flag_t                flags;
  size_t                ii;
  variable_def_t        def;
  variable_def_member_t mbr;
  value_t              *params;

  flags = CMD_VARIABLE_ACCESS_FLAG_NONE;
  if (cmd_brval_get(cmd) == val) {
    flags |= CMD_VARIABLE_ACCESS_FLAG_READ;
  }
  def = value_def_get(cmd_brproc_get(cmd));
  params = cmd_brproc_params_get(cmd);
  for (ii = 0, mbr = variable_def_member_get(def);
       mbr;
       ii++, mbr = variable_def_member_link_get(mbr)) {
    if (val == params[ii]) {
      if (variable_def_flag_test(variable_def_member_def_get(mbr),
        VARIABLE_DEF_FLAG_IN)) {
        flags |= CMD_VARIABLE_ACCESS_FLAG_READ;
      }
      if (variable_def_flag_test(variable_def_member_def_get(mbr),
        VARIABLE_DEF_FLAG_OUT)) {
        flags |= CMD_VARIABLE_ACCESS_FLAG_WRITTEN;
      }
    }
  }
  return flags;
}

static const cmd_vtbl_t cmd_br_vtbl = {
  cmd_branch_free,
  cmd_branch_dump,
  cmd_branch_dup,
  cmd_branch_label_remap,
  cmd_branch_variable_remap,
  cmd_branch_value_remap,
  cmd_branch_variable_accessed,
  cmd_branch_value_accessed
};

/* proc_params[0] = where to put the return value */
cmd_t cmd_branch_alloc(
  cmd_branchtype_t type,
  cmd_branchcond_t cond, label_t dst, value_t val,
  value_t proc, value_t *proc_params)
{
  cmd_t cmd;

  if (cmd_branchtype_call == type) {
    assert(!dst);
    assert(proc);
  }
  cmd = cmd_alloc(cmd_type_branch, &cmd_br_vtbl);
  if (cmd) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      ptr->u.br.type        = type;
      ptr->u.br.cond        = cond;
      ptr->u.br.dst         = 0;
      ptr->u.br.var         = 0;
      ptr->u.br.proc        = proc;
      ptr->u.br.proc_params = proc_params;
      
      cmd_brdst_set(cmd, dst);
      cmd_brval_set(cmd, val);
      if (proc) {
        pfile_proc_t         *proc_ptr;
        variable_def_t        def;
        variable_def_member_t mbr;
        size_t                ii;

        value_lock(proc);

        def = value_def_get(proc);
        assert(variable_def_type_function == variable_def_type_get(def));
        value_use_ct_bump(proc, ctr_bump_incr);
        for (mbr = variable_def_member_get(def), ii = 0;
             mbr;
             mbr = variable_def_member_link_get(mbr), ii++) {
          variable_def_t mdef;

          mdef = variable_def_member_def_get(mbr);
          if (mdef) {
            /* if ii == 0 this is the return value. we'll assume it's used */
            if (!ii || variable_def_flag_test(mdef, VARIABLE_DEF_FLAG_IN)) {
              value_use_ct_bump(proc_params[ii], ctr_bump_incr);
            }
            if (variable_def_flag_test(mdef, VARIABLE_DEF_FLAG_OUT)) {
              value_assign_ct_bump(proc_params[ii], ctr_bump_incr);
            }
            proc_ptr = value_proc_get(proc_params[ii]);
            if (proc_ptr) {
              pfile_proc_flag_set(proc_ptr, PFILE_PROC_FLAG_INDIRECT);
              label_usage_bump(pfile_proc_label_get(proc_ptr), ctr_bump_incr);
            }
          }
        }
        proc_ptr = value_proc_get(proc);
        if (proc_ptr) {
          label_usage_bump(pfile_proc_label_get(proc_ptr), ctr_bump_incr);
        }
      }
    }
  }
  return cmd;
}

cmd_branchtype_t cmd_brtype_get(const cmd_t cmd)
{
  struct cmd_ *ptr;
  
  ptr = cmd_element_seek(cmd, boolean_false);

  return (ptr && (cmd_type_branch == cmd_type_get(cmd)))
    ? ptr->u.br.type
    : cmd_branchtype_none;
}

void cmd_brtype_set(cmd_t cmd, cmd_branchtype_t brtype)
{
  if (cmd_type_branch == cmd_type_get(cmd)) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      ptr->u.br.type = brtype;
    }
  }
}

cmd_branchcond_t cmd_brcond_get(const cmd_t cmd)
{
  struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (ptr && (cmd_type_branch == cmd_type_get(cmd)))
    ? ptr->u.br.cond
    : cmd_branchcond_none;
}

void cmd_brcond_set(cmd_t cmd, cmd_branchcond_t brcond)
{
  if (cmd_type_branch == cmd_type_get(cmd)) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      ptr->u.br.cond = brcond;
    }
  }
}

label_t cmd_brdst_get(const cmd_t cmd)
{
  const struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (cmd_type_branch == cmd_type_get(cmd))
    ? ptr->u.br.dst
    : 0;
}

void cmd_brdst_set(cmd_t cmd, label_t dst)
{
  if (cmd_type_branch == cmd_type_get(cmd)) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);
    if (ptr) {
      if (dst) {
        label_lock(dst);
        label_usage_bump(dst, ctr_bump_incr);
      }
      if (ptr->u.br.dst) {
        label_usage_bump(ptr->u.br.dst, ctr_bump_decr);
        label_release(ptr->u.br.dst);
      }
      ptr->u.br.dst = dst;
    }
  }
}

value_t cmd_brval_get(const cmd_t cmd)
{
  const struct cmd_ *ptr;

  ptr = cmd_element_seek(cmd, boolean_false);
  return (cmd && (cmd_type_branch == cmd_type_get(cmd)))
    ? ptr->u.br.var
    : 0;
}

void cmd_brval_set(cmd_t cmd, value_t val)
{
  if (cmd_type_branch == cmd_type_get(cmd)) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);

    if (ptr) {
      if (val) {
        value_lock(val);
        value_use_ct_bump(val, ctr_bump_incr);
      }
      if (ptr->u.br.var) {
        value_use_ct_bump(ptr->u.br.var, ctr_bump_decr);
        value_release(ptr->u.br.var);
      }
      ptr->u.br.var = val;
    }
  }
}

value_t cmd_brproc_get(const cmd_t cmd)
{
  value_t proc;

  proc = VALUE_NONE;
  if (cmd_type_branch == cmd_type_get(cmd)) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_false);

    if (ptr) {
      proc = ptr->u.br.proc;
    }
  }
  return proc;
}

void cmd_brproc_set(cmd_t cmd, value_t proc)
{
  if (cmd_type_branch == cmd_type_get(cmd)) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_true);

    if (ptr) {
      value_lock(proc);
      value_use_ct_bump(proc, ctr_bump_incr);
      label_usage_bump(pfile_proc_label_get(value_proc_get(proc)),
        ctr_bump_incr);

      value_use_ct_bump(ptr->u.br.proc, ctr_bump_decr);
      label_usage_bump(pfile_proc_label_get(value_proc_get(ptr->u.br.proc)),
        ctr_bump_decr);
      value_release(ptr->u.br.proc);
      ptr->u.br.proc = proc;
    }
  }
}

value_t *cmd_brproc_params_get(const cmd_t cmd)
{
  value_t *params;

  params = 0;
  if (cmd_type_branch == cmd_type_get(cmd)) {
    struct cmd_ *ptr;

    ptr = cmd_element_seek(cmd, boolean_false);

    if (ptr) {
      params = ptr->u.br.proc_params;
    }
  }
  return params;
}

size_t cmd_brproc_param_ct_get(const cmd_t cmd)
{
  variable_def_t        def;
  variable_def_member_t mbr;
  size_t                ct;

  def = value_def_get(cmd_brproc_get(cmd));
  for (ct = 0, mbr = variable_def_member_get(def);
       mbr;
       ct++, mbr = variable_def_member_link_get(mbr))
    ; /* null body */
  return ct;
}

void cmd_brproc_param_set(cmd_t cmd, size_t ix, value_t val)
{
  value_t *params;

  params = cmd_brproc_params_get(cmd);
  if (params && (ix <= cmd_brproc_param_ct_get(cmd))) {
    variable_def_t        def;
    variable_def_member_t mbr;
    value_t               old_val;
    
    old_val = params[ix];
    params[ix] = val;
    def = value_def_get(cmd_brproc_get(cmd));
    for (mbr = variable_def_member_get(def);
         mbr && ix;
         mbr = variable_def_member_link_get(mbr), ix--)
      ;
    if (variable_def_flag_test(variable_def_member_def_get(mbr),
          VARIABLE_DEF_FLAG_IN)) {
      value_use_ct_bump(val, ctr_bump_incr);
      value_use_ct_bump(old_val, ctr_bump_decr);
    }
    if (variable_def_flag_test(variable_def_member_def_get(mbr),
          VARIABLE_DEF_FLAG_OUT)) {
      value_assign_ct_bump(val, ctr_bump_incr);
      value_assign_ct_bump(old_val, ctr_bump_decr);
    }
    value_lock(val);
    value_release(old_val);
  }
}

value_t cmd_brproc_param_get(const cmd_t cmd, size_t ix)
{
  value_t *params;

  params = cmd_brproc_params_get(cmd);
  return (params && (ix < cmd_brproc_param_ct_get(cmd))) 
    ? params[ix]
    : VALUE_NONE;
}

