/************************************************************
 **
 ** pf_expr.c : p-code expression & operator functions
 **
 ** Copyright (c) 2005, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/
#include <assert.h>
#include "pfiled.h"
#include "pf_cmd.h"
#include "pf_op.h"
#include "pf_expr.h"

variable_def_t pfile_variable_def_promotion_get_default(pfile_t *pf,
    operator_t op, value_t val1, value_t val2)
{
  variable_def_t      rdef; /* result */
  variable_sz_t       sz1;
  variable_sz_t       sz2;
  variable_def_type_t type1;
  variable_def_type_t type2;
  variable_def_t      def1;
  variable_def_t      def2;

  def1 = value_def_get(val1);
  def2 = value_def_get(val2);

  sz1 = variable_def_sz_get(def1);
  if (!value_is_bit(val1)) {
    sz1 *= 8;
  }
  sz2 = variable_def_sz_get(def2);
  if (!value_is_bit(val2)) {
    sz2 *= 8;
  }
  type1 = variable_def_type_get(def1);
  type2 = variable_def_type_get(def2);

  if (!def2 || variable_def_is_same(def1, def2)) {
    /* (1) definition is the same */
    rdef = def1;
  } else if (variable_def_flag_test(def1, VARIABLE_DEF_FLAG_SIGNED)
    && variable_def_flag_test(def2, VARIABLE_DEF_FLAG_SIGNED)) {
    /* (2) both operands are signed, the result is the one with the
     * greatest size */
    rdef = (sz1 > sz2) ? def1 : def2;
  } else if (!variable_def_flag_test(def1, VARIABLE_DEF_FLAG_SIGNED)
      && (sz1 >= sz2)) {
    /* (3a) unsigned type sz >= signed type size; result is unsigned */
    rdef = def1;
  } else if (!variable_def_flag_test(def2, VARIABLE_DEF_FLAG_SIGNED)
      && (sz2 >= sz1)) {
    /* (3b) unsigned type sz >= signed type size; result is unsigned */
    rdef = def2;
  } else if (variable_def_flag_test(def1, VARIABLE_DEF_FLAG_SIGNED)
      && (sz1 > sz2)) {
    /* (4a) */
    rdef = def1;
  } else if (variable_def_flag_test(def2, VARIABLE_DEF_FLAG_SIGNED)
      && (sz2 > sz1)) {
    /* (4b) */
    rdef = def2;
  } else {
    /* (5) */
    rdef = variable_def_alloc(0, variable_def_type_integer,
        VARIABLE_DEF_FLAG_NONE, (sz1 > sz2) ? sz1 : sz2);
  }
  if (variable_def_flag_test(rdef, VARIABLE_DEF_FLAG_CONST)
      || variable_def_flag_test(rdef, VARIABLE_DEF_FLAG_BIT)) {
    /* flip this bit */
    rdef = variable_def_flags_change(rdef,
        variable_def_flags_get_all(rdef)
        & ~(VARIABLE_DEF_FLAG_CONST | VARIABLE_DEF_FLAG_BIT));
  }
  if (variable_def_type_boolean == variable_def_type_get(rdef)) {
    rdef = variable_def_alloc(0,
        variable_def_type_integer,
        variable_def_flags_get_all(rdef),
        variable_def_sz_get(rdef));
  }
  return rdef;
}

variable_def_t pfile_variable_def_promotion_get(pfile_t *pf,
    operator_t op, value_t val1, value_t val2)
{
  variable_def_t rdef;

  rdef = (pf->vectors->pf_expr_result_def_get_fn)
    ? pf->vectors->pf_expr_result_def_get_fn(pf, op, val1, val2)
    : pfile_variable_def_promotion_get_default(pf, op, val1, val2);
  if (!rdef) {
    pfile_log(pf, pfile_log_err, "invalid operation");
  }
  return rdef;
}

/*
 * NAME
 *   pfile_expr_push
 *
 * DESCRIPTION
 *   push a subexpression onto the expression stack
 *
 * PARAMETERS
 *   pf   : returned by pfile_open()
 *   stk  : top of the expression stack
 *   expr : expression to append
 *
 * RETURN
 *   none
 *
 * NOTES
 *   this creates commands as necessary
 */
result_t pfile_expr_push(pfile_t *pf, expr_t **stk, value_t val,
    operator_t op)
{
  result_t rc;
  expr_t  *stkptr;
  expr_t  *expr;

  rc     =  expr_alloc(&expr, op, val);
  stkptr = *stk;
  if (result_ok == rc) {
    while (stkptr 
      && (operator_priority_get(expr_operator_get(stkptr))
        >= operator_priority_get(expr_operator_get(expr)))) {
      expr_t  *stkpv;
      value_t  dst;

      stkpv = expr_link_get(stkptr);
      dst   = 0;

      if (operator_is_assign(expr_operator_get(stkptr))) {
        /* stkptr->val stkptr->op expr->val */
        dst = expr_val_get(stkptr);
        pfile_cmd_op_add(pf, expr_operator_get(stkptr),
          &dst, expr_val_get(expr), 0);
        value_lock(dst);
      } else {
        /* dst = stkptr->val stkptr->op expr->val */
        operator_t s_op;

        s_op = expr_operator_get(stkptr);
        if ((operator_shift_right == s_op)
          && value_is_signed(expr_val_get(stkptr))) {
          s_op = operator_shift_right_arithmetic;
        }
        pfile_cmd_op_add(pf, s_op, &dst, expr_val_get(stkptr), 
            expr_val_get(expr));
      }
      expr_val_set(expr, dst);
      expr_free(stkptr);
      value_release(dst);
      stkptr = stkpv;
    }
    expr_link_set(expr, stkptr);
    *stk = expr;
  }
  if (result_ok != rc) {
    pfile_log_syserr(pf, rc);
  }
  return rc;
}

/* do two things
 *   1. if val2 exists, make sure val1 & val2 are compatible
 *   2. make sure the result of (val1/val2) is compatible with dst
 */
static void pfile_value_type_check(pfile_t *pf, operator_t op, value_t dst,
    value_t val1, value_t val2)
{
  if (value_is_number(dst) 
      && value_is_number(val1)
      && ((VALUE_NONE == val2) || value_is_number(val2))
      && (pfile_flag_test(pf, PFILE_FLAG_WARN_TRUNCATE)
          || pfile_flag_test(pf, PFILE_FLAG_WARN_CONVERSION))) {
    variable_def_t rdef;
    flag_t         warnings;

    warnings = 0; /* assume no warnings */
    rdef = pfile_variable_def_promotion_get(pf, op, val1, val2);
    if (value_is_universal(val2)) {
      value_t tmp;

      tmp  = val1;
      val1 = val2;
      val2 = tmp;
    }
    if (!value_is_universal(val1)) {
      if (((VALUE_NONE != val2) 
              && (value_is_signed(val1) ^ value_is_signed(val2))) 
            || ((VALUE_NONE == val2)
              && (value_is_signed(dst) 
                ^ variable_def_flag_test(rdef, VARIABLE_DEF_FLAG_SIGNED)))) {
        warnings |= PFILE_FLAG_WARN_CONVERSION;
      }
      if (value_sz_get(dst) < variable_def_sz_get(rdef)) {
        warnings |= PFILE_FLAG_WARN_TRUNCATE;
      }
    } else {
      /* determine which warnings are appropriate; these are done
       * using a flag to make sure warnings are only reported once
       */
      variable_sz_t val1_bitsz;
      flag_t        val1_flags;
      variable_sz_t dst_bitsz;
      variable_const_t n;

      variable_calc_sz_min(value_const_get(val1), &val1_bitsz, 0, 
          &val1_flags);
      n = value_const_get(val1);
      for (val1_bitsz = 0; n; n >>= 1, val1_bitsz++)
        ;
      if (VALUE_NONE != val2) {
        if (value_is_signed(val2) ^ (VARIABLE_DEF_FLAG_SIGNED == val1_flags)) {
          warnings |= PFILE_FLAG_WARN_CONVERSION;
        }
      }

      dst_bitsz = value_sz_get(dst);
      if (!value_is_bit(dst)) {
        dst_bitsz *= 8;
      }

      if (dst_bitsz < val1_bitsz) {
        warnings |= PFILE_FLAG_WARN_TRUNCATE;
      }
    }
    if ((warnings & PFILE_FLAG_WARN_TRUNCATE)
      && pfile_flag_test(pf, PFILE_FLAG_WARN_TRUNCATE)) {
      pfile_log(pf, pfile_log_warn, 
          "assignment to smaller type; truncation possible");
    }
    if ((warnings & PFILE_FLAG_WARN_CONVERSION)
      && pfile_flag_test(pf, PFILE_FLAG_WARN_CONVERSION)) {
      pfile_log(pf, pfile_log_warn, "signed/unsigned mismatch");
    }
  }
}


/*
 * NAME
 *   pfile_cmd_op_add
 *
 * DESCRIPTION
 *   add an operation
 *
 * PARAMETERS
 *   pf:
 *   label : line label
 *   op    : operation
 *   dst   : destination, can be 0 to allocate a temporary
 *   val1  : value 1 (can be 0 for unary operations, cannot be 0 for binary)
 *   val2  : value 2 (must be 0 for unary, can be 0 for binary)
 *
 * RETURN
 *
 * NOTES
 *   if dst is NULL on entry, it will hold a value on successful exit
 *   if dst is a function
 *     *error*
 *   if dst is a reference
 *     if *dst is a function
 *       src must be a function with a matching signature
 */
void pfile_cmd_op_add_assign(pfile_t *pf, operator_t op,
  value_t *pdst, value_t src)
{
  if (!*pdst 
    && (operator_assign == op) 
    && value_is_const(src)
    && value_is_number(src)) {
    *pdst = pfile_constant_get(pf, value_const_get(src),
      value_def_get(src));
  } else {
    value_t dst;

    dst = *pdst;
    if (!dst) {
      assert(variable_def_type_function != value_type_get(src));
      dst = pfile_value_temp_get_from_def(pf, value_def_get(src));
      *pdst = dst;
    }
    /* assignment is allowed between variables that are logically
       the same (the definitions match), and between any two
       numbers */
    if ((variable_def_is_same(value_def_get(dst), value_def_get(src)))
      || (value_is_number(dst) && value_is_number(src))) {
      cmd_t cmd;

      cmd = cmd_op_alloc(op, dst, src, VALUE_NONE);
      pfile_cmd_add(pf, cmd);
    } else if (value_is_pointer(dst) 
      && value_is_array(src)
      && variable_def_is_same(variable_def_member_def_get(
        variable_def_member_get(value_def_get(dst))),
          variable_def_member_def_get(variable_def_member_get(
          value_def_get(src))))) {
      /* assign array to pointer */
      cmd_t cmd;

      cmd = cmd_op_alloc(op, dst, src, VALUE_NONE);
      pfile_cmd_add(pf, cmd);
    } else if (value_is_pointer(dst) 
      && value_is_function(src)
      && variable_def_is_same(variable_def_member_def_get(
        variable_def_member_get(value_def_get(dst))),
          value_def_get(src))) {
      /* assign function to pointer */
      cmd_t cmd;

      cmd = cmd_op_alloc(op, dst, src, VALUE_NONE);
      pfile_cmd_add(pf, cmd);
    } else {
      pfile_log(pf, pfile_log_err, "invalid operation");
    }
  }
}

static void pfile_cmd_op_add_unary(pfile_t *pf, operator_t op,
    value_t *pdst, value_t src)
{
  value_t dst;

  if (!value_is_number(src)) {
    pfile_log(pf, pfile_log_err, "invalid operation");
  } else if (!*pdst && value_is_const(src)) {
    variable_const_t n;

    n = pfile_op_const_exec(pf, op, src, VALUE_NONE);
    if (value_is_bit(src)) {
      n &= (1 << value_sz_get(src)) - 1;
    }
    *pdst = pfile_constant_get(pf, n, value_def_get(src));
  } else {
    cmd_t cmd;

    dst = *pdst;
    if (!dst) {
      dst = pfile_value_temp_get_from_def(pf, 
          pfile_variable_def_promotion_get(pf, op, src, VALUE_NONE));
      *pdst = dst;
    }
    /* see nb about volatile in add_binary */
    cmd = cmd_op_alloc(op, dst, src, VALUE_NONE);
    pfile_cmd_add(pf, cmd);
  }
}
/*
  (1) if both operands are the same type
        no conversion is done
  (2) if both operands are signed
        the operand of lesser width is promoted to the operand of greater width
  (3) if the operand that is unsigned has a greater or equal width than the 
        signed, the signed operand is converted to the unsigned type
  (4) if the operand that is signed can represent all values of the operand 
        that is unsigned, the unsigned operand is converted to the signed
  (5) failing all else, both operands are converted to the unsigned integer 
        type with the same width as the signed operand
 */
static void pfile_cmd_op_add_binary(pfile_t *pf, operator_t op,
    value_t *pdst, value_t val1, value_t val2)
{
  if (!*pdst 
    && (value_is_const(val1) && !value_is_lookup(val1))
    && value_is_number(val1) 
    && (value_is_const(val2) && !value_is_lookup(val2))
    && value_is_number(val2)) {
    variable_def_t   def;
    variable_const_t n;

    n = 0;
    def = pfile_variable_def_promotion_get(pf, op, val1, val2);
    n = pfile_op_const_exec(pf, op, val1, val2);
    *pdst = pfile_constant_get(pf, n, def); /* VARIABLE_DEF_NONE); */
  } else {
    value_t dst;

    dst = *pdst;
    if ((!value_is_number(val1) || !value_is_number(val2))
      && ((operator_eq != op) || (operator_ne != op))) {
      pfile_log(pf, pfile_log_err, "invalid operation");
    } else {
      variable_def_t rdef; /* result */
      cmd_t          cmd;

      if (!value_is_number(val1)) {
        /* the result is a byte */

        rdef = variable_def_alloc(0, variable_def_type_integer, 
            VARIABLE_DEF_FLAG_NONE, 1);
      } else {
        rdef = pfile_variable_def_promotion_get(pf, op, val1, val2);
      }
      if (!dst) {
        dst = pfile_value_temp_get_from_def(pf, rdef);
        *pdst = dst;
      }
      /* nb: if dst is volatile, the work must be done in a temporary
             *but* this needs to be left to the backend because some
             operations can be optimized at a lower level */
      cmd = cmd_op_alloc(op, dst, val1, val2);
      pfile_cmd_add(pf, cmd);
    }
  }
}

void pfile_cmd_op_add(pfile_t *pf,
  operator_t op, value_t *pdst, value_t val1, value_t val2)
{
  value_t  dst;
  result_t rc;

  rc = result_ok;

  dst = (pdst) ? *pdst : VALUE_NONE;

  /* 1st, make sure val1 and/or val2 exit */
  if (operator_is_binary(op)) {
    assert(val1);
    assert(val2 || dst);
    if (!val2) {
      val2 = val1;
      val1 = dst;
    }
  } else if (operator_is_unary(op)) {
    assert(val1 || dst);
    if (!val1) {
      val1 = dst;
    }
  }
  value_lock(val1);

  value_lock(val2);

  /* non-numbers only allow:
   *   assign to 0
   *   assign to like type
   *   equal (of like type)
   *   not equal (of like type)
   */   
  pfile_value_type_check(pf, op, dst, val1, val2);
  if (dst && value_is_const(dst) && !value_name_get(dst)) {
    dst = VALUE_NONE;
  }
  /* check for an invalid read or write.
   * VARIABLE_FLAG_[READ|WRITE] are not fully implemented yet
   * so this is kind of a hack. For dst, if READ is set and
   * not WRITE issue an error. For val[1|2], if WRITE is set
   * and not READ issue an error. */
  if (value_vflag_test(dst, VARIABLE_FLAG_READ)
    && !value_vflag_test(dst, VARIABLE_FLAG_WRITE)) {
    pfile_log(pf, pfile_log_err, "cannot write to %s",
      value_name_get(dst));
  } else if (value_vflag_test(val1, VARIABLE_FLAG_WRITE)
    && !value_vflag_test(val1, VARIABLE_FLAG_READ)) {
    pfile_log(pf, pfile_log_err, "cannot read from %s",
      value_name_get(val1));
  } else if (value_vflag_test(val2, VARIABLE_FLAG_WRITE)
    && !value_vflag_test(val2, VARIABLE_FLAG_READ)) {
    pfile_log(pf, pfile_log_err, "cannot read from %s",
      value_name_get(val2));
  } else if (dst && (value_is_const(dst) || value_is_lookup(dst))) {
    pfile_log(pf, pfile_log_err, "assign to constant");
  } else if (operator_null == op) {
  } else if (operator_is_assign(op)) {
    if (!(val1 || val2)) {
      pfile_log(pf, pfile_log_err, "value missing");
    } else {
      pfile_cmd_op_add_assign(pf, op, &dst, val1);
    }
  } else if (operator_is_unary(op)) {
    if (val2 || !(val1 || dst)) {
      pfile_log(pf, pfile_log_err, "value missing");
    } else {
      assert(val2 == 0);
      assert(val1 || dst);
      if (!val1) {
        val1 = dst;
      }
      pfile_cmd_op_add_unary(pf, op, &dst, val1);
    }
  } else {
    if (!val1 || !(val2 || dst)) {
    } else {
      assert(val1 != 0);
      assert(val2 || dst);
      if (!val2) {
        val2 = val1;
        val1 = dst;
      }
      pfile_cmd_op_add_binary(pf, op, &dst, val1, val2);
    }
  }

  value_release(val1);
  value_release(val2);
  if (pdst) {
    if (dst != *pdst) {
      if (*pdst) {
        value_release(*pdst);
      }
      *pdst = dst;
    }
  } else if (dst) {
    value_release(dst);
  }
}

