/************************************************************
 **
 ** pic14.c : 14-bit pic code generation definitions
 **
 ** Copyright (c) 2007, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/
#include "pic14.h"

boolean_t pic14_code_to_pcode(pfile_t *pf, pic_code_t code,
    unsigned val, unsigned literal, pic_code_to_pcode_t *pdst)
{
  unsigned pcode;
  unsigned dst;
  unsigned bit;
  unsigned lit8;
  unsigned lit10; /* branch literal */

  bit     = literal & 0x07;
  lit8    = literal & 0x00ff; /* 8 bit literal */
  lit10   = literal & 0x07ff; /* 10 bit literal */
  dst     = (pic_opdst_f == pic_code_dst_get(code)) << 7;
  pcode   = 0xffff; /* an impossible code (it's greater than 14 bits) */
  switch (pic_code_op_get(code)) {
    /* directives (no code) */
    case pic_opcode_org:    /* ORG xxxx */
    case pic_opcode_end:    /* END      */
    case pic_opcode_none:   /* no code here */
    case pic_opcode_branchlo_nop:
    case pic_opcode_branchhi_nop:
      break;
    case pic_opcode_nop:    pcode = 0x0000; break;
    /* file operator (code, dst, val) */
    case pic_opcode_movwf:  pcode = 0x0000 | 0x80 | val; break;
    case pic_opcode_clrf:   pcode = 0x0100 | 0x80 | val; break;
    case pic_opcode_subwf:  pcode = 0x0200 |  dst | val; break;
    case pic_opcode_decf:   pcode = 0x0300 |  dst | val; break;
    case pic_opcode_iorwf:  pcode = 0x0400 |  dst | val; break;
    case pic_opcode_andwf:  pcode = 0x0500 |  dst | val; break;
    case pic_opcode_xorwf:  pcode = 0x0600 |  dst | val; break;
    case pic_opcode_addwf:  pcode = 0x0700 |  dst | val; break;
    case pic_opcode_movf:   pcode = 0x0800 |  dst | val; break;
    case pic_opcode_comf:   pcode = 0x0900 |  dst | val; break;
    case pic_opcode_incf:   pcode = 0x0a00 |  dst | val; break;
    case pic_opcode_decfsz: pcode = 0x0b00 |  dst | val; break;
    case pic_opcode_rrf:    pcode = 0x0c00 |  dst | val; break;
    case pic_opcode_rlcf:
    case pic_opcode_rlf:    pcode = 0x0d00 |  dst | val; break;
    case pic_opcode_swapf:  pcode = 0x0e00 |  dst | val; break;
    case pic_opcode_incfsz: pcode = 0x0f00 |  dst | val; break;
    /* simple opcodes (no operands) */
    case pic_opcode_retfie: pcode = 0x0009; break;
    case pic_opcode_return: pcode = 0x0008; break;
    case pic_opcode_sleep:  pcode = 0x0063; break;
    case pic_opcode_clrwdt: pcode = 0x0064; break;
    /* the following doesn't look right to me based on the microchip doc's, but
     * it is what MPLAB puts out */
    case pic_opcode_clrw:   pcode = 0x0103; break;
    /* bit oriented (3 bit literal) */
    case pic_opcode_bcf:    pcode = 0x1000 | (bit << 7) | val; break;
    case pic_opcode_bsf:    pcode = 0x1400 | (bit << 7) | val; break;
    case pic_opcode_btfsc:  pcode = 0x1800 | (bit << 7) | val; break;
    case pic_opcode_btfss:  pcode = 0x1c00 | (bit << 7) | val; break;
    /* literal & w  (8 bit literal) */ 
    case pic_opcode_movlw:  pcode = 0x3000 | lit8; break;
    case pic_opcode_retlw:  pcode = 0x3400 | lit8; break;
    case pic_opcode_iorlw:  pcode = 0x3800 | lit8; break;
    case pic_opcode_andlw:  pcode = 0x3900 | lit8; break;
    case pic_opcode_xorlw:  pcode = 0x3a00 | lit8; break;
    case pic_opcode_sublw:  pcode = 0x3c00 | lit8; break;
    case pic_opcode_addlw:  pcode = 0x3e00 | lit8; break;
    /* branching (10 bit literal) */
    case pic_opcode_call:   pcode = 0x2000 | lit10; break;
    case pic_opcode_goto:   pcode = 0x2800 | lit10; break;
    /* special (pseudo) opcodes */
    case pic_opcode_datalo_set: /* bsf _status, _rp0 */
      pcode = 0x1400 | (5 << 7) | 0x0003;
      break;
    case pic_opcode_datalo_clr: /* bcf _status, _rp0 */
      pcode = 0x1000 | (5 << 7) | 0x0003;
      break;
    case pic_opcode_datahi_set: /* bsf _status, _rp1 */
      pcode = 0x1400 | (6 << 7) | 0x0003;
      break;
    case pic_opcode_datahi_clr: /* bcf _status, _rp1 */
      pcode = 0x1000 | (6 << 7) | 0x0003;
      break;
    case pic_opcode_irp_set:    /* bsf _status, _irp */
      pcode = 0x1400 | (7 << 7) | 0x0003;
      break;
    case pic_opcode_irp_clr:    /* bcf _status, _irp */
      pcode = 0x1000 | (7 << 7) | 0x0003;
      break;
    case pic_opcode_branchlo_set: /* bsf _pclath: 3 */
      pcode = 0x1400 | (3 << 7) | 0x000a;
      break;
    case pic_opcode_branchlo_clr: /* bcf _pclath: 3 */
      pcode = 0x1000 | (3 << 7) | 0x000a;
      break;
    case pic_opcode_branchhi_set: /* bsf _pclath: 4 */
      pcode = 0x1400 | (4 << 7) | 0x000a;
      break;
    case pic_opcode_branchhi_clr: /* bcf _pclath: 4 */
      pcode = 0x1000 | (4 << 7) | 0x000a;
      break;
    case pic_opcode_option:       /* option */
      pcode = 0x0002;
      break;
    case pic_opcode_tris:         /* tris 6 */
      pcode = lit8 & 0x07;
      break;
    case pic_opcode_db: /* this is handled elsewhere */
      break;
    case pic_opcode_mullw:
    case pic_opcode_addwfc:
    case pic_opcode_dcfsnz:
    case pic_opcode_infsnz:
    case pic_opcode_rlncf:
    case pic_opcode_rrncf:
    case pic_opcode_subfwb:
    case pic_opcode_subwfb:
    case pic_opcode_cpfseq:
    case pic_opcode_cpfsgt:
    case pic_opcode_cpfslt:
    case pic_opcode_mulwf:
    case pic_opcode_negf:
    case pic_opcode_setf:
    case pic_opcode_tstfsz:
    case pic_opcode_btg:
    case pic_opcode_bc:
    case pic_opcode_bn:
    case pic_opcode_bnc:
    case pic_opcode_bnn:
    case pic_opcode_bnov:
    case pic_opcode_bnz:
    case pic_opcode_bov:
    case pic_opcode_bz:
    case pic_opcode_bra:
    case pic_opcode_rcall:
    case pic_opcode_daw:
    case pic_opcode_pop:
    case pic_opcode_push:
    case pic_opcode_reset:
    case pic_opcode_lfsr:
    case pic_opcode_movff:
    case pic_opcode_movlb:
    case pic_opcode_tblrd:
    case pic_opcode_tblwt:
      pfile_log(pf, pfile_log_err, "illegal instruction");
      break;
  }
  pdst->ct = 0;
  if (0xffff != pcode) {
    pdst->ct      = 1;
    pdst->code[0] = pcode;
  }
  return pdst->ct != 0;
}

void pic14_asm_header_write(pfile_t *pf, variable_const_t n_code_sz)
{
  pfile_write(pf, pfile_write_asm, 
      "datahi_set macro val\n"
      "  bsf 3, 6 ; STATUS<rp1>\n"
      "  endm\n");
  pfile_write(pf, pfile_write_asm,
      "datahi_clr macro val\n"
      "  bcf 3, 6 ; STATUS<rp1>\n"
      "  endm\n");
  pfile_write(pf, pfile_write_asm,
      "datalo_set macro val\n"
      "  bsf 3, 5 ; STATUS<rp0>\n"
      "  endm\n");
  pfile_write(pf, pfile_write_asm,
      "datalo_clr macro val\n"
      "  bcf 3, 5 ; STATUS<rp0>\n"
      "  endm\n");
  pfile_write(pf, pfile_write_asm,
      "irp_clr macro\n"
      "  bcf 3, 7 ; STATUS<irp>\n"
      "  endm\n");
  pfile_write(pf, pfile_write_asm,
      "irp_set macro\n"
      "  bsf 3, 7 ; STATUS<irp>\n"
      "  endm\n");
  pfile_write(pf, pfile_write_asm,
      "branchhi_set macro lbl\n"
      "  %s\n"
      "  endm\n", 
      (n_code_sz < 4096) ? "nop" : "  bsf 10, 4 ; PCLATH<4>");
  pfile_write(pf, pfile_write_asm,
      "branchhi_clr macro lbl\n"
      "  %s\n"
      "  endm\n",
      (n_code_sz < 4096) ? "nop" : "  bcf 10, 4 ; PCLATH<4>");
  pfile_write(pf, pfile_write_asm,
      "branchlo_set macro lbl\n"
      "  %s\n"
      "  endm\n",
      (n_code_sz  < 2048) ? "nop" : "  bsf 10, 3 ; PCLATH<3>");
  pfile_write(pf, pfile_write_asm,
      "branchlo_clr macro lbl\n"
      "  %s\n"
      "  endm\n",
      (n_code_sz  < 2048) ? "nop" : "  bcf 10, 3 ; PCLATH<3>");
  if (pfile_flag_test(pf, PFILE_FLAG_DEBUG_COMPILER)) {
    pfile_write(pf, pfile_write_asm,
        "branchhi_nop macro lbl\n"
        "  endm\n");
    pfile_write(pf, pfile_write_asm,
        "branchlo_nop macro lbl\n"
        "  endm\n");
  }
}

