/************************************************************
 **
 ** pic_test.c : PIC operator test harness
 **
 ** Copyright (c) 2007, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/
#include <assert.h>
#include <string.h>
#include "../libcore/pf_op.h"
#include "pic.h"
#include "pic_cmdo.h"
#include "pic_brop.h"
#include "pic_daop.h"
#include "picbsrop.h"
#include "pic_stk.h"
#include "pic_op.h"
#include "pic_inst.h"
#include "piccolst.h"
#include "pic_stvar.h"
#include "pic_emu.h"
#include "pic_opfn.h"

/* define below to test the pfile_op_const_exec functions */
#undef  TEST_PFILE_OP_CONST_EXEC

static unsigned interest_first;
static unsigned interest_last;

static result_t dummy_open(pfile_t *pf) { return result_ok; }
static void     dummy_close(pfile_t *pf) { }

static pfile_vectors_t vectors = {
  dummy_open,
  dummy_close,
  0, /* token get */
  0, /* include */
  0, /* expr result def */
  pic_cmd_optimize,
  pic_cmd_generate,
  pic_cmd_dump,
  pic_code_cleanup
};

operator_priority_t operator_priority_get(operator_t op)
{
  return 0;
}

/* test the assignment operator
   multi-bit is deferred until the shift, binary or, and binary and
             operators are fully tested
   nb: vals need never be bool because bool only affects assignment          

   dst types:
     W
     boolean
     [v][s]bit
     [v][s]multi-bit (defered)
     [v][i][s]byte
     [v][i][s]word
     [v][i][s]dword

   val1 types & values
     W
     [v][s]bit   {0, 1}
     [v][s]multi-bit (deferred)
     [v][i|c|l][s]byte  {s=-5, 5; u = 5}
     [v][i|c|l][s]word  {s=-5, 5; u = 5}
     [v][i|c|l][s]dword {s=-5, 5; u = 5}
*/
typedef enum {
  PIC_VAL_QUALIFIER_CONST,
  /* PIC_VAL_QUALIFIER_LOOKUP, */
  PIC_VAL_QUALIFIER_W,
  PIC_VAL_QUALIFIER_NONE,
  PIC_VAL_QUALIFIER_VOLATILE,
  PIC_VAL_QUALIFIER_CT
} pic_val_qualifier_t;

typedef enum {
  PIC_VAL_INDIRECT_INDIRECT,
  PIC_VAL_INDIRECT_NONE,
  PIC_VAL_INDIRECT_CT
} pic_val_indirect_t;

typedef enum {
  PIC_VAL_SIGN_SIGNED,
  PIC_VAL_SIGN_NONE,
  PIC_VAL_SIGN_CT
} pic_val_sign_t;

typedef enum {
  PIC_VAL_TYPE_BOOLEAN,
  PIC_VAL_TYPE_SBIT,    /* single bit */
  /*PIC_VAL_TYPE_MBIT, */   /* multi-bit  */
  PIC_VAL_TYPE_BYTE,
  PIC_VAL_TYPE_WORD,
  PIC_VAL_TYPE_DWORD,
  PIC_VAL_TYPE_CT
} pic_val_type_t;

typedef enum {
  PIC_VAL_BANK_0,
  PIC_VAL_BANK_1,
  PIC_VAL_BANK_2,
  PIC_VAL_BANK_3
} pic_val_bank_t;

#define PIC_VAL_BANK_MAX PIC_VAL_BANK_1

typedef struct {
  pic_val_qualifier_t qualifier;
  pic_val_indirect_t  indirect;
  pic_val_sign_t      sign;
  pic_val_type_t      type;
  pic_val_bank_t      bank;
} pic_val_attribute_t;

#define PIC_VAL_ATTRIBUTE_NONE ((pic_val_attribute_t *) 0)

static unsigned test_ct;

typedef enum {
  PIC_VAL_WHICH_DST,
  PIC_VAL_WHICH_VAL1,
  PIC_VAL_WHICH_VAL2
} pic_val_which_t;

static value_t pic_test_value_create(pfile_t *pf,
  const pic_val_attribute_t *attr, pic_val_which_t which,
  variable_const_t init_value)
{
  value_t val;

  val = VALUE_NONE;
  if (attr) {
    flag_t              def_flags;
    variable_def_type_t def_type;
    variable_def_t      def;
    char                var_name[32];
    const char         *str;
    unsigned            bit_ofs;
    unsigned            sz;
    variable_t          var;

    bit_ofs  = 0;
    sz       = 0;

    def_flags = VARIABLE_DEF_FLAG_NONE;
    if (PIC_VAL_QUALIFIER_CONST == attr->qualifier) {
      def_flags |= VARIABLE_DEF_FLAG_CONST;
    } 
    if (PIC_VAL_QUALIFIER_VOLATILE == attr->qualifier) {
      def_flags |= VARIABLE_DEF_FLAG_VOLATILE;
    }
    /* W, LOOKUP, NONE not handled yet */
    if (PIC_VAL_SIGN_SIGNED == attr->sign) {
      def_flags |= VARIABLE_DEF_FLAG_SIGNED;
    }
    str = "fixme"; /* fix compiler warning */
    switch (which) {
      case PIC_VAL_WHICH_DST:  str = "dst"; break;
      case PIC_VAL_WHICH_VAL1: str = "val1"; break;
      case PIC_VAL_WHICH_VAL2: str = "val2"; break;
    }
    sprintf(var_name, "%s_%u", str, attr->bank);
    def_type = variable_def_type_integer;
    switch (attr->type) {
      case PIC_VAL_TYPE_BOOLEAN:
        def_type = variable_def_type_boolean;
        /* fall through */
      case PIC_VAL_TYPE_SBIT:
        sz        = 1;
        bit_ofs   = 5;
        def_flags |= VARIABLE_DEF_FLAG_BIT;
        break;
#if 0
      case PIC_VAL_TYPE_MBIT:
        sz        = 5;
        bit_ofs   = 5;
        def_flags |= VARIABLE_DEF_FLAG_BIT;
        break;
#endif
      case PIC_VAL_TYPE_BYTE:
        sz        = 1;
        break;
      case PIC_VAL_TYPE_WORD:
        sz        = 2;
        break;
      case PIC_VAL_TYPE_DWORD:
        sz        = 4;
        break;
      case PIC_VAL_TYPE_CT: 
        assert(0); 
        break;
    }
    def = variable_def_alloc(0, def_type, def_flags, sz);
    var = VARIABLE_NONE;
    if (def_flags & VARIABLE_DEF_FLAG_CONST) {
      val = pfile_constant_get(pf, init_value, def);
    } else {
      if (PIC_VAL_QUALIFIER_W == attr->qualifier) {
        val = value_alloc(VARIABLE_NONE);
        value_def_set(val, def);
      } else {
        var = pfile_variable_find(pf, pfile_log_err, var_name, 0);
        variable_def_set(var, def);
        variable_bit_offset_set(var, bit_ofs);
        val = value_alloc(var);
      }
      if (PIC_VAL_INDIRECT_INDIRECT == attr->indirect) {
        value_t baseofs;

        baseofs = pfile_value_find(pf, pfile_log_err, "_baseofs");
        value_baseofs_set(val, baseofs);
        value_release(baseofs);
        value_flag_set(val, VALUE_FLAG_ARRAY);
      }
      variable_release(var);
    }
  }
  return val;
}

static variable_const_t pic_test_val_sign_extend(
  variable_const_t n,
  pic_val_type_t   type,
  pic_val_sign_t   sign,
  size_t           sz
  )
{
  if (sz < 32) {
    if ((PIC_VAL_SIGN_SIGNED == sign) && (n & (1 << (sz - 1)))) {
      n |= ~((1 << sz) - 1);
    } else if (PIC_VAL_TYPE_BOOLEAN == type) {
      n = !!n;
    } else {
      /* mask off any high bits */
      n &= (1 << sz) - 1;
    }
  }
  return n;
}

/* return the real value if (n) is assigned to (src) */
static variable_const_t pic_test_val_result_get(
    variable_const_t n,
    pic_val_type_t   type,
    pic_val_sign_t   sign

    )
{
  size_t sz;

  sz = 0;
  switch (type) {
    case PIC_VAL_TYPE_BOOLEAN: 
      n = !!n; 
      break;
    case PIC_VAL_TYPE_SBIT:    
      n &= 1; 
      if (PIC_VAL_SIGN_SIGNED == sign) {
        n = -n;
      }
      break;
    case PIC_VAL_TYPE_BYTE:
      n &= 0xff;
      sz = 8;
      break;
    case PIC_VAL_TYPE_WORD:
      n &= 0xffff;
      sz = 16;
      break;
    case PIC_VAL_TYPE_DWORD:
      break;
    case PIC_VAL_TYPE_CT:
      break;
  }
  if (sz) {
    n = pic_test_val_sign_extend(n, type, sign, sz);
  }
  return n;
}

static size_t bit_size_from_type(pic_val_type_t type)
{
  size_t sz;

  sz = 0; /* fix compiler warning */
  switch (type) {
    case PIC_VAL_TYPE_BOOLEAN:
    case PIC_VAL_TYPE_SBIT:
      sz   = 1;
      break;
    case PIC_VAL_TYPE_BYTE:
      sz   = 8;
      break;
    case PIC_VAL_TYPE_WORD:
      sz   = 16;
      break;
    case PIC_VAL_TYPE_DWORD:
    case PIC_VAL_TYPE_CT:
      sz   = 32;
      break;
  }
  return sz;
}

static size_t result_sz_get(const pic_val_attribute_t *val1,
    const pic_val_attribute_t *val2)
{
  size_t sz1;
  size_t sz2;

  sz1 = bit_size_from_type(val1->type);
  sz2 = bit_size_from_type(val2->type);
  return (sz1 > sz2) ? sz1 : sz2;
}

static boolean_t result_is_signed(const pic_val_attribute_t *val1,
    const pic_val_attribute_t *val2)
{
  if (bit_size_from_type(val1->type) > bit_size_from_type(val2->type)) {
    return PIC_VAL_SIGN_SIGNED == val1->sign;
  } else if (bit_size_from_type(val1->type) 
      == bit_size_from_type(val2->type)) {
    return (PIC_VAL_SIGN_SIGNED == val1->sign)
      && (PIC_VAL_SIGN_SIGNED == val2->sign);
  } else {
    return PIC_VAL_SIGN_SIGNED == val2->sign;
  }
}

static variable_const_t pic_test_op_expect_get(
  operator_t                 op,
  const pic_val_attribute_t *dst_attr,
  const pic_val_attribute_t *val1_attr,
  const pic_val_attribute_t *val2_attr,
  variable_const_t           cv1,
  variable_const_t           cv2)
{
  variable_const_t n1;
  variable_const_t n2;
  variable_const_t n;
  variable_const_t sz;
  pic_val_sign_t   sign;
  pic_val_type_t   type;

  type = val1_attr->type;
  sign = val1_attr->sign;
  n  = 0;
  n1 = pic_test_val_result_get(cv1, val1_attr->type, val1_attr->sign);
  n2 = 0;
  if (val2_attr) {
    n2 = pic_test_val_result_get(cv2, val2_attr->type, val2_attr->sign);
    if (val1_attr->type < val2_attr->type) {
      type = val2_attr->type;
      sign = val2_attr->sign;
    } else if (val1_attr->type == val2_attr->type) {
      sign = ((PIC_VAL_SIGN_SIGNED == val1_attr->sign)
        && (PIC_VAL_SIGN_SIGNED == val2_attr->sign)) 
          ? PIC_VAL_SIGN_SIGNED
          : PIC_VAL_SIGN_NONE;
    } else {
      type = val1_attr->type;
      sign = val1_attr->sign;
    }
  }
  if (type < PIC_VAL_TYPE_BYTE) {
    type = PIC_VAL_TYPE_BYTE;
  }
  sz = bit_size_from_type(dst_attr->type);

  switch (op) {
    case operator_null:         /* no operation   */
    case operator_reference:/* unary : return the address of a variable or label */
      assert(0);
      break;
    case operator_assign:  /* unary */
      n = n1;
      break;
    case operator_andb:
      n = n1 & n2;
      break;
    case operator_orb:
      n = n1 | n2;
      break;
    case operator_xorb:
      n = n1 ^ n2; 
      break;
    case operator_andl:
      n = n1 && n2;
      break;
    case operator_orl:
      n = n1 || n2;
      break;
    case operator_logical:  /* unary : convert a value to either 0 or 1 */
      n = !!n1;
      break;
    case operator_notl: /* unary */
      n = !n1;
      break;
    case operator_cmpb: /* unary */
      if (PIC_VAL_TYPE_BOOLEAN == val1_attr->type) {
        n = (n1) ? 0 : -1;
      } else {
        n = ~n1;
      }
      break;
    case operator_incr:    /* unary */
      n = n1 + 1;
      break;
    case operator_decr:    /* unary */
      n = n1 - 1;
      break;
    case operator_neg:
      n = -n1;
      break;
    case operator_shift_left:
      n = n1 << n2;
      break;
    case operator_shift_right:
    case operator_shift_right_arithmetic: /* preserve the sign bit */
      /* this is a special case because higher bits can shift down,
       * so the real size depends upon val1 & val2 *not* dst */
      {
        size_t sz0;
        size_t sz1;
        size_t sz2;

        sz1 = bit_size_from_type(val1_attr->type);
        sz2 = bit_size_from_type(val2_attr->type);

        sz0 = (sz1 > sz2) ? sz1 : sz2;
        if (sz0 < 8) {
          sz0 = 8;
        }
        if (sz0 < 32) {
          n1 &= ((1 << sz0) - 1);
        }
        if (n2 >= sz0) {
          if ((sz1 < sz0) && (PIC_VAL_SIGN_SIGNED == val1_attr->sign)) {
            n = ((operator_shift_right_arithmetic == op) 
              && (n1 & (1 << (sz1 - 1))))
              ? -1
              : 0;
          } else {
            n = ((operator_shift_right_arithmetic == op) 
              && (n1 & (1 << (sz0 - 1))))
              ? -1
              : 0;
          }
        } else {
          n = n1 >> n2;
          if ((sz0 - n2 < 32) 
              && (operator_shift_right_arithmetic == op)
              && (n1 & (1 << (sz0 - 1)))) {
            /* set the high bits */
            n |= ~((1UL << (sz0 - n2)) - 1);
          }
        }
        if (sz0 < 32) {
          n &= ((1 << sz0) - 1);
        }
      }
      break;
    case operator_eq:
    case operator_ne:
      {
        variable_sz_t sz0;

        sz0 = result_sz_get(val1_attr, val2_attr);
        if (sz0 < 8) {
          sz0 = 8;
        }
        if (sz0 < 32) {
          n1 &= ((1 << sz0) - 1);
          n2 &= ((1 << sz0) - 1);
        }
        if (operator_eq == op) {
          n = n1 == n2;
        } else {
          n = n1 != n2;
        }
      }
      break;
    case operator_lt:
    case operator_le:
    case operator_ge:
    case operator_gt:
      {
        if (!result_is_signed(val1_attr, val2_attr)) {
          variable_sz_t sz0;

          sz0 = result_sz_get(val1_attr, val2_attr);
          if (sz0 < 8) {
            sz0 = 8;
          }
          if (sz0 < 32) {
            n1 &= ((1 << sz0) - 1);
            n2 &= ((1 << sz0) - 1);
          }
        }
        switch (op) {
          case operator_lt:
            n = (result_is_signed(val1_attr, val2_attr))
              ? ((long) n1 < (long) n2)
              : (n1 < n2);
            break;
          case operator_le:
            n = (result_is_signed(val1_attr, val2_attr))
              ? ((long) n1 <= (long) n2)
              : (n1 <= n2);
            break;
          case operator_ge:
            n = (result_is_signed(val1_attr, val2_attr))
              ? ((long) n1 >= (long) n2)
              : (n1 >= n2);
            break;
          case operator_gt:
            n = (result_is_signed(val1_attr, val2_attr))
              ? ((long) n1 > (long) n2)
              : (n1 > n2);
            break;
          default:
            abort();
        }
      }
      break;
    case operator_add:
        n = n1 + n2;
        break;
    case operator_sub:
        n = n1 - n2;
        break;
    case operator_mul:
        n = n1 * n2;
        break;
    case operator_div:
    case operator_mod:
      if (!n2) {
        n = (val1_attr == val2_attr) ? 1 : -1;
      } else if (PIC_VAL_SIGN_SIGNED == sign) {
        unsigned s;

        s = 0;
        if (PIC_VAL_SIGN_SIGNED == val1_attr->sign) {
          if (n1 & (1 << 31)) {
            s = 1;
            n1 = -n1;
          }
        }
        if (PIC_VAL_SIGN_SIGNED == val2_attr->sign) {
          if (n2 & (1 << 31)) {
            s ^= 1;
            n2 = -n2;
          }
        }
        n = (operator_div == op) ? n1 / n2 : n1 % n2;
        if (s) {
          n = - n;
        }
      } else {
        variable_sz_t sz0;

        sz0 = result_sz_get(val1_attr, val2_attr);
        if (sz0 < 8) {
          sz0 = 8;
        }
        if (sz0 < 32) {
          n1 &= ((1 << sz0) - 1);
          n2 &= ((1 << sz0) - 1);
        }
        n = (operator_div == op) ? n1 / n2 : n1 % n2;
      }
      break;
  }
  return pic_test_val_sign_extend(
      pic_test_val_result_get(n, type, sign), 
      dst_attr->type, dst_attr->sign, sz);
}

static const char *op_to_str(operator_t op)
{
  const char *str;

  str = "";
  switch(op) {
    case operator_null: break;
    case operator_reference:
    case operator_add:  str = "+"; break;
    case operator_sub:  str = "-"; break;
    case operator_mul:  str = "x"; break;
    case operator_div:  str = "/"; break;
    case operator_mod:  str = "%"; break;
    case operator_lt:   str = "<"; break;
    case operator_le:   str = "<="; break;
    case operator_eq:   str = "=="; break;
    case operator_ne:   str = "!="; break;
    case operator_ge:   str = ">="; break;
    case operator_gt:   str = ">";  break;
    case operator_andl: str = "&&"; break;
    case operator_orl:  str = "||"; break;
    case operator_notl: str = "!";  break;
    case operator_andb: str = "&";  break;
    case operator_orb:  str = "|";  break;
    case operator_xorb: str = "^";  break;
    case operator_cmpb: str = "~";  break;
    case operator_neg:  str = "neg";  break;
    case operator_incr: str = "++"; break;
    case operator_decr: str = "--"; break;
    case operator_shift_left: str = "<<"; break;
    case operator_shift_right: str = ">>"; break;
    case operator_shift_right_arithmetic: str = ">->"; break;
    case operator_logical:  str = "!!"; break;
    case operator_assign:  str = "="; break;
  }
  return str;
}

#ifndef TEST_PFILE_OP_CONST_EXEC

static void pic_test_state_init(pfile_t *pf, pic_emu_state_t state)
{
  static const pic_emu_var_def_t vars16[] = {
    {"_ind",    VARIABLE_DEF_FLAG_VOLATILE, 1,
                {0x0fef}},
    {"_pcl",    VARIABLE_DEF_FLAG_VOLATILE, 1,
                {0x0ff9}},
    {"_status", VARIABLE_DEF_FLAG_VOLATILE, 1,
                {0x0fd8}},
    {"_fsr0h",  VARIABLE_DEF_FLAG_VOLATILE, 1,
                {0x0fea}},
    {"_fsr0l",  VARIABLE_DEF_FLAG_VOLATILE, 1,
                {0x0fe9}},
    {"_pic_accum", VARIABLE_DEF_FLAG_NONE, 1,
                {0x0000}}
  };
  static const pic_emu_var_def_t vars14[] = {
    {"_ind",    VARIABLE_DEF_FLAG_VOLATILE, 1,
                {0x0000, 0x0080, 0x0100, 0x0180}},
    {"_pcl",    VARIABLE_DEF_FLAG_VOLATILE, 1,
                {0x0002, 0x0082, 0x0102, 0x0182}},
    {"_status", VARIABLE_DEF_FLAG_VOLATILE, 1,
                {0x0003, 0x0083, 0x0103, 0x0183}},
    {"_fsr",    VARIABLE_DEF_FLAG_VOLATILE, 1,
                {0x0004, 0x0084, 0x0104, 0x0184}},
    {"_pclath", VARIABLE_DEF_FLAG_VOLATILE, 1,
                {0x000a, 0x008a, 0x010a, 0x018a}},
    {"_pic_accum", VARIABLE_DEF_FLAG_NONE, 1,
                {0x007e, 0x00fe, 0x017e, 0x01fe}},
    {"_irp",    VARIABLE_DEF_FLAG_CONST, 1,
                {7}},
    {"_rp1",    VARIABLE_DEF_FLAG_CONST, 1,
                {6}},
    {"_rp0",    VARIABLE_DEF_FLAG_CONST, 1,
                {5}}
  };
  static const pic_emu_var_def_t vars12[] = {
    {"_ind",    VARIABLE_DEF_FLAG_VOLATILE, 1,
                {0x0000, 0x0020, 0x0040, 0x0060}},
    {"_pcl",    VARIABLE_DEF_FLAG_VOLATILE, 1,
                {0x0002, 0x0022, 0x0042, 0x0062}},
    {"_status", VARIABLE_DEF_FLAG_VOLATILE, 1,
                {0x0003, 0x0023, 0x0043, 0x0063}},
    {"_fsr",    VARIABLE_DEF_FLAG_VOLATILE, 1,
                {0x0004, 0x0024, 0x0044, 0x0064}},
    {"_pic_accum", VARIABLE_DEF_FLAG_NONE, 1,
                {0x0009, 0x0029, 0x0049, 0x0069}}
  };
  static const pic_emu_var_def_t vars_common[] = {
    {"_z",      VARIABLE_DEF_FLAG_CONST, 1,
                {2}},
    {"_c",      VARIABLE_DEF_FLAG_CONST, 1,
                {0}},
    {"_code_size", VARIABLE_DEF_FLAG_CONST, 4, {8192}}
  };
  const pic_emu_var_def_t *vars;
  variable_t               var;
  size_t                   ii;
  size_t                   ct;

  vars = 0; /* fix compiler warning */
  ct   = 0; /* fix compiler warning */
  switch (pic_emu_state_target_get(state)) {
    case PIC_TARGET_CPU_NONE:
      abort();
      break;
    case PIC_TARGET_CPU_12BIT:
      vars = vars12;
      ct   = COUNT(vars12);
      pic_data_bank_list_add(pf, 0x70, 0x7f);
      break;
    case PIC_TARGET_CPU_14BIT:
      vars = vars14;
      ct   = COUNT(vars14);
      pic_data_bank_list_add(pf, 0x21, 0x3f);
      break;
    case PIC_TARGET_CPU_16BIT:
      vars = vars16;
      ct   = COUNT(vars16);
      pic_data_bank_list_add(pf, 0x100, 0x1ff);
      break;
  }
  for (ii = 0; ii < ct; ii++) {
    var = pic_emu_var_add(pf, vars + ii);
    if (!variable_is_const(var)) {
      variable_flag_set(var, VARIABLE_FLAG_SHARED);
    }
  }
  for (ii = 0; ii < COUNT(vars_common); ii++) {
    pic_emu_var_add(pf, vars_common + ii);
  }
}

variable_const_t pic_test_op_doit_emu(pfile_t *pf, pic_target_cpu_t target,
    operator_t op, value_t dst, value_t val1, value_t val2, 
    variable_const_t cv1, variable_const_t cv2)
{
  pic_emu_state_t  state;
  label_t          lbl_tmp;
  unsigned         pass;
  variable_const_t result;

  state = pic_emu_state_alloc(pf, target);

  pic_test_state_init(pf, state);

  pic_emu_data_mem_init(state);

  lbl_tmp = pfile_lblptr_get(pf);

  for (pass = 0; pass < 2; pass++) {
    label_t entry;

    pic_code_list_reset(pf);
    pic_last_values_reset();
    pfile_lblptr_set(pf, lbl_tmp);
    if (pass) {
      /* alloc the necessary temp system vars */
      pic_stvar_fixup(pf);
    }
    pic_code_preamble(pf);
    entry = pfile_user_entry_get(pf);
    pic_instr_append_label(pf, entry);
    label_release(entry);
    /* this is probably a hack, but when testing multiply,
     * pic_mresult is accessed without being set. That's because
     * it is always shifted the number of bits it holds. For this
     * case, I'm going to simply set it to zero here & make the
     * testing happy!
     */
    if (1 == pass) {
      value_t tmp;

      /* clear some internal variables. this isn't necessary for
       * proper operation, but the emulator will complain about
       * uninitialzed access */
      tmp = pfile_value_find(pf, pfile_log_none, "_pic_mresult");
      if (VALUE_NONE != tmp) {
        /* pic_instr_append(pf, pic_opcode_clrw); */
        pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
        pic_op(pf, operator_assign, tmp, VALUE_NONE, VALUE_NONE);
        value_release(tmp);
      }
      tmp = pfile_value_find(pf, pfile_log_none, "_pic_quotient");
      if (VALUE_NONE != tmp) {
        /* pic_instr_append(pf, pic_opcode_clrw); */
        pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
        pic_op(pf, operator_assign, tmp, VALUE_NONE, VALUE_NONE);
        value_release(tmp);
      }
      tmp = pfile_value_find(pf, pfile_log_none, "_pic_divaccum");
      if (VALUE_NONE != tmp) {
        /* pic_instr_append(pf, pic_opcode_clrw); */
        pic_instr_append_w_kn(pf, pic_opcode_movlw, 0);
        pic_op(pf, operator_assign, tmp, VALUE_NONE, VALUE_NONE);
        value_release(tmp);
      }
    }
    /* setup the values */
    if (value_is_indirect(val1)
        || value_is_indirect(val2)
        || value_is_indirect(dst)) {
      value_t baseofs;
      value_t c;

      baseofs = pfile_value_find(pf, pfile_log_err, "_baseofs");
      c = pfile_constant_get(pf, 0, VARIABLE_DEF_NONE);
      pic_op(pf, operator_assign, baseofs, c, VALUE_NONE);
      value_release(c);
      value_release(baseofs);
    }

    if (!value_is_const(val1)) {
      value_t c;
      value_t tmp;

      c = pfile_constant_get(pf, cv1, VARIABLE_DEF_NONE);
      tmp = value_clone(val1);
      value_baseofs_set(tmp, VALUE_NONE);
      pic_op(pf, operator_assign, tmp, c, VALUE_NONE);
      value_release(tmp);
      value_release(c);
    }
    if (val2 && !value_is_const(val2)) {
      value_t c;
      value_t tmp;

      c = pfile_constant_get(pf, cv2, VARIABLE_DEF_NONE);
      tmp = value_clone(val2);
      value_baseofs_set(tmp, VALUE_NONE);
      pic_op(pf, operator_assign, tmp, c, VALUE_NONE);
      value_release(tmp);
      value_release(c);
    }
    pic_op(pf, op, dst, val1, val2);
    if (0 == pass) {
      pic_intrinsics_create(pf);
    }
  }
  /* get rid of any unneeded data bits. required because
   * instruction skips (think add/sub) won't work correctly
   * if there are stray data bit instructions lying around
   */
  pic_code_databits_optimize(pf);
  pic_code_bsr_optimize(pf);
  pic_code_branchbits_remove(pf);
  if (interest_first == test_ct) {
    pic_variable_counters_set(pf);
    switch (pic_target_cpu_get(pf)) {
      case PIC_TARGET_CPU_12BIT:
        pic_asm_header_write(pf, "12c509");
        break;
      case PIC_TARGET_CPU_14BIT:
        pic_asm_header_write(pf, "16f877");
        break;
      case PIC_TARGET_CPU_16BIT:
        pic_asm_header_write(pf, "18f2420");
        break;
      case PIC_TARGET_CPU_NONE:
        break;
    }
    pic_cmd_dump(pf, CMD_NONE, boolean_true);
  }
  pic_emu(pf, state);
  result = pic_emu_value_read(state, dst, 0);
  pic_code_list_reset(pf);
  pic_last_values_reset();
  /* remove pic_state if it was allocated earlier */
  pic_emu_state_free(state);

  return result;
}

#else

variable_const_t pic_test_op_doit_pfile_op(pfile_t *pf, operator_t op,
    value_t dst, value_t val1, value_t val2, variable_const_t cv1,
    variable_const_t cv2)
{
  return pic_test_val_sign_extend(pfile_op_const_exec(pf, op, val1, val2),
      result, dst_attr->type, dst_attr->sign,
        bit_size_from_type(dst_attr->type));
}

#endif

static void pic_test_vars_create(pfile_t *pf, pic_target_cpu_t target)
{
  /* 14 bit definitions */
  static const pic_emu_var_def_t vars14[] = {
    /* baseofs is used to force indirect access; it's value
       is will always be zero */
    {"_baseofs", VARIABLE_DEF_FLAG_NONE, 1, 
      {0x20, VARIABLE_BASE_UNKNOWN}},
    /* these are dummy variables used when allocating from each bank */
    {"dst_0",   VARIABLE_DEF_FLAG_NONE, 4, 
      {0x040, VARIABLE_BASE_UNKNOWN}},
    {"dst_1",   VARIABLE_DEF_FLAG_NONE, 4, 
      {0x0c0, VARIABLE_BASE_UNKNOWN}},
    {"dst_2",   VARIABLE_DEF_FLAG_NONE, 4, 
      {0x140, VARIABLE_BASE_UNKNOWN}},
    {"dst_3",   VARIABLE_DEF_FLAG_NONE, 4, 
      {0x1c0, VARIABLE_BASE_UNKNOWN}},
    {"val1_0",  VARIABLE_DEF_FLAG_NONE, 4, 
      {0x144, VARIABLE_BASE_UNKNOWN}},
    {"val1_1",  VARIABLE_DEF_FLAG_NONE, 4, 
      {0x0c4, VARIABLE_BASE_UNKNOWN}},
    {"val1_2",  VARIABLE_DEF_FLAG_NONE, 4, 
      {0x144, VARIABLE_BASE_UNKNOWN}},
    {"val1_3",  VARIABLE_DEF_FLAG_NONE, 4, 
      {0x1c4, VARIABLE_BASE_UNKNOWN}},
    {"val2_0",  VARIABLE_DEF_FLAG_NONE, 4, 
      {0x048, VARIABLE_BASE_UNKNOWN}},
    {"val2_1",  VARIABLE_DEF_FLAG_NONE, 4,
      {0x0c8, VARIABLE_BASE_UNKNOWN}},
    {"val2_2",  VARIABLE_DEF_FLAG_NONE, 4, 
      {0x148, VARIABLE_BASE_UNKNOWN}},
    {"val2_3",  VARIABLE_DEF_FLAG_NONE, 4, 
      {0x1c8, VARIABLE_BASE_UNKNOWN}}
  };
  static const pic_emu_var_def_t vars12[] = {
    /* baseofs is used to force indirect access; it's value
       is will always be zero */
    {"_baseofs", VARIABLE_DEF_FLAG_NONE, 1, 
      {0x08, VARIABLE_BASE_UNKNOWN}},
    /* these are dummy variables used when allocating from each bank */
    {"dst_0",   VARIABLE_DEF_FLAG_NONE, 4, 
      {0x010, VARIABLE_BASE_UNKNOWN}},
    {"dst_1",   VARIABLE_DEF_FLAG_NONE, 4, 
      {0x030, VARIABLE_BASE_UNKNOWN}},
    {"dst_2",   VARIABLE_DEF_FLAG_NONE, 4, 
      {0x050, VARIABLE_BASE_UNKNOWN}},
    {"dst_3",   VARIABLE_DEF_FLAG_NONE, 4, 
      {0x070, VARIABLE_BASE_UNKNOWN}},
    {"val1_0",  VARIABLE_DEF_FLAG_NONE, 4, 
      {0x014, VARIABLE_BASE_UNKNOWN}},
    {"val1_1",  VARIABLE_DEF_FLAG_NONE, 4, 
      {0x034, VARIABLE_BASE_UNKNOWN}},
    {"val1_2",  VARIABLE_DEF_FLAG_NONE, 4, 
      {0x054, VARIABLE_BASE_UNKNOWN}},
    {"val1_3",  VARIABLE_DEF_FLAG_NONE, 4, 
      {0x074, VARIABLE_BASE_UNKNOWN}},
    {"val2_0",  VARIABLE_DEF_FLAG_NONE, 4, 
      {0x018, VARIABLE_BASE_UNKNOWN}},
    {"val2_1",  VARIABLE_DEF_FLAG_NONE, 4,
      {0x038, VARIABLE_BASE_UNKNOWN}},
    {"val2_2",  VARIABLE_DEF_FLAG_NONE, 4, 
      {0x058, VARIABLE_BASE_UNKNOWN}},
    {"val2_3",  VARIABLE_DEF_FLAG_NONE, 4, 
      {0x078, VARIABLE_BASE_UNKNOWN}}
  };
  static const pic_emu_var_def_t *vars;
  size_t ii;
  size_t ct;

  if (PIC_TARGET_CPU_12BIT == target) {
    vars = vars12;
    ct   = COUNT(vars12);
  } else {
    vars = vars14;
    ct   = COUNT(vars14);
  }
  for (ii = 0; ii < ct; ii++) {
    pic_emu_var_add(pf, vars + ii);
  }
}

static void pic_test_op_doit(
  pic_target_cpu_t           target,
  unsigned                   flags,
  operator_t                 op,
  const pic_val_attribute_t *dst_attr,
  const pic_val_attribute_t *val1_attr,
  const pic_val_attribute_t *val2_attr,
  variable_const_t           cv1,
  variable_const_t           cv2)
{
  value_t          val1;
  value_t          val2;
  value_t          dst;
  variable_const_t expect;
  variable_const_t result;
  refct_t          val1_ref;
  refct_t          val2_ref;
  refct_t          dst_ref;

  if (((operator_div == op) || (operator_mod == op))
    && (0 
        == pic_test_val_result_get(cv2, val2_attr->type, val2_attr->sign))) {
    /* don't allow divide by zero errors to creep in */
  } else {
    pfile_t *pf;
    label_t  lbl_user;

#ifdef TEST_PFILE_OP_CONST_EXEC
    if (!(val1_attr
        && (PIC_VAL_QUALIFIER_CONST == val1_attr->qualifier)
        && (operator_is_unary(op)
          || (val2_attr
            && (PIC_VAL_QUALIFIER_CONST == val2_attr->qualifier))))) {
              return;
    }
#endif

    test_ct++;
    if ((test_ct < interest_first) || (test_ct > interest_last)) {
      return;
    }
    fprintf(stderr, "\rtest(%u) ", test_ct);

    pfile_open(&pf, stdout, 0, 0, 0, 0, 
        flags,
        &vectors, 0, 0, 0);

    pic_test_vars_create(pf, target);
    lbl_user = pfile_label_alloc(pf, "_entry");
    pfile_user_entry_set(pf, lbl_user);
    label_release(lbl_user);

    if (val1_attr == val2_attr) {
      assert(cv1 == cv2);
    }
    dst  = VALUE_NONE;
    val1 = pic_test_value_create(pf, val1_attr, PIC_VAL_WHICH_VAL1, cv1);
    val2 = VALUE_NONE;

    dst_ref  = 1;
    val1_ref = 1;
    val2_ref = 1;
    if (val2_attr == val1_attr) {
      value_lock(val1);
      val1_ref++;
      val2 = val1;
    } else {
      val2 = pic_test_value_create(pf, val2_attr, 
        PIC_VAL_WHICH_VAL2, cv2);
    }
    if (dst_attr == val1_attr) {
      value_lock(val1);
      val1_ref++;
      dst  = val1;
    } else if (dst_attr == val2_attr) {
      value_lock(val2);
      val2_ref++;
      dst = val2;
    }
    if (VALUE_NONE == dst) {
      dst = pic_test_value_create(pf, dst_attr,
        PIC_VAL_WHICH_DST, 0);
    }
    if (dst == val1) {
      dst_ref = val1_ref;
    } else if (dst == val2) {
      dst_ref = val2_ref;
    }
    if (val2 == val1) {
      val2_ref = val1_ref;
    }
    /* this will likely go away... */
    if ((interest_first == test_ct) /*|| (0 == (test_ct % 1000))*/) {
      printf("; ");
      variable_dump(value_variable_get(dst), stdout);
      printf("%s = ", (PIC_VAL_INDIRECT_INDIRECT == dst_attr->indirect)
          ? "::ind" : "");
      if (operator_is_unary(op) && (operator_assign != op)) {
        printf(" %s ", op_to_str(op));
      }
      variable_dump(value_variable_get(val1), stdout);
      printf("%s %08x", 
          (PIC_VAL_INDIRECT_INDIRECT == val1_attr->indirect)
          ? "::ind" : "",
          (unsigned) pic_test_val_result_get(cv1, val1_attr->type, 
            val1_attr->sign));
      if (operator_is_binary(op)) {
        printf(" %s ", op_to_str(op));
        variable_dump(value_variable_get(val2), stdout);
        printf("%s %08x", 
            (PIC_VAL_INDIRECT_INDIRECT == val2_attr->indirect)
            ? "::ind" : "",
            (unsigned) pic_test_val_result_get(cv2, val2_attr->type,
              val2_attr->sign));
      }
      printf("\n");
    }

    /* cleanup anything we've setup here */
    expect = pic_test_op_expect_get(op, dst_attr, val1_attr, val2_attr,
        cv1, cv2);
#ifdef TEST_PFILE_OP_CONST_EXEC
    result = pic_test_op_doit_pfile_op(pf, op, dst, val1, val2, 
        cv1, cv2);
#else
    result = pic_test_op_doit_emu(pf, target, op, dst, val1, val2, 
        cv1, cv2);
#endif
    if (expect != result) {
      printf("error! expected (%u 0x%08x) got (%u 0x%08x)\n",
        (unsigned) expect, 
        (unsigned) expect,
        (unsigned) result,
        (unsigned) result);
      abort();
    }
    assert((VALUE_NONE == dst)  || (dst_ref  == value_ref_ct_get(dst)));
    assert((VALUE_NONE == val1) || (val1_ref == value_ref_ct_get(val1)));
    assert((VALUE_NONE == val2) || (val2_ref == value_ref_ct_get(val2)));
    value_release(val2);
    value_release(val1);
    value_release(dst);
    pfile_close(pf);
  }
}

static void pic_test_op(
  pic_target_cpu_t           target,
  unsigned                   flags,
  operator_t                 op,
  const pic_val_attribute_t *dst_attr,
  const pic_val_attribute_t *val1_attr,
  const pic_val_attribute_t *val2_attr)
{
  switch (op) {
    case operator_null:
    case operator_reference:
      assert(0);
      break;
    case operator_andl:
    case operator_orl:
      if ((dst_attr->qualifier != PIC_VAL_QUALIFIER_W)
          && (PIC_VAL_QUALIFIER_W != val1_attr->qualifier)
          && (PIC_VAL_QUALIFIER_W != val2_attr->qualifier)) {
        if (val1_attr == val2_attr) {
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
              0, 0);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
              1, 1);
        } else {
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
              0, 0);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
              0, -1);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
              -1, 0);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
              -1, -1);
        }
      }
      break;
    case operator_andb:
    case operator_orb:
    case operator_xorb:
      if ((dst_attr->qualifier != PIC_VAL_QUALIFIER_W)
        && (val1_attr->qualifier != PIC_VAL_QUALIFIER_W)
        && (val2_attr->qualifier != PIC_VAL_QUALIFIER_W)) {
        if (val1_attr == val2_attr) {
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            0, 0);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            0xff, 0xff);
        } else {
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            0, 0);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            0, 1);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            1, 0);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            1, 1);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            8, 128);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            8, 255);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            1234, 0xffff);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            1234, 0x8000);
        }
      }
      break;
    case operator_assign:
      switch (dst_attr->type) {
        case PIC_VAL_TYPE_BOOLEAN:
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            0, 0);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
            1, 0);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
            2, 0);
          break;
        case PIC_VAL_TYPE_SBIT:
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            0, 0);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
            1, 0);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
            2, 0);
          break;
        case PIC_VAL_TYPE_BYTE:
        case PIC_VAL_TYPE_WORD:
        case PIC_VAL_TYPE_DWORD:
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
            5, 0);
          if (PIC_VAL_SIGN_SIGNED == val1_attr->sign) {
            pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
              -5, 0);
          }
          break;
        case PIC_VAL_TYPE_CT:
          break;
      }
      break;
    case operator_logical:
    case operator_notl:
      if (val1_attr->qualifier != PIC_VAL_QUALIFIER_W) {
        pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
          0, -1);
        pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
          0, 1);
        pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
          0, 0);
      }
      break;
    case operator_cmpb:
    case operator_neg:
      if ((dst_attr->qualifier != PIC_VAL_QUALIFIER_W)
        && (val1_attr->qualifier != PIC_VAL_QUALIFIER_W)) {
        pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
          0, 0);
        pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
          1, 0);
        pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
          -1, 0);
        pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
          1234, 0);
      }
      break;
    case operator_incr:
    case operator_decr:
      if ((dst_attr->qualifier != PIC_VAL_QUALIFIER_W)
        && (val1_attr->qualifier != PIC_VAL_QUALIFIER_W)) {
        pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
          0, 0);
        pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
          1, 0);
        pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
          -1, 0);
      }
      break;
    case operator_shift_left:
    case operator_shift_right:
    case operator_shift_right_arithmetic:
      if ((dst_attr->qualifier != PIC_VAL_QUALIFIER_W)
        && (val1_attr->qualifier != PIC_VAL_QUALIFIER_W)
        && (val2_attr->qualifier != PIC_VAL_QUALIFIER_W)) {
        pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
          0, 0);
        if (val1_attr == val2_attr) {
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            4, 4);
        } else {
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            1234, 1);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            1234, 2);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            1234, 4);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            1234, 8);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            1234, 9);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
            1234, 17);
          /* some special ones */
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
            1234, 5);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
            1234, 6);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
            1234, 7);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
            1234, 12);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
            1234, 13);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
            1234, 14);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
            1234, 15);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
            1234, 28);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
            1234, 29);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
            1234, 30);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
            1234, 31);
          if (operator_shift_right_arithmetic == op) {
            pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
              -1234, 1);
            pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
              -1234, 2);
            pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
              -1234, 4);
            pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
              -1234, 8);
            pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
              -1234, 9);
          }
        }
      }
      break;
    case operator_eq:
    case operator_ne:
    case operator_lt:
    case operator_le:
    case operator_ge:
    case operator_gt:
      if ((val1_attr->qualifier != PIC_VAL_QUALIFIER_W)
        && (val2_attr->qualifier != PIC_VAL_QUALIFIER_W)) {
        if (val1_attr == val2_attr) {
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            0, 0);
        } else {
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            0, 1);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            1, 0);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            -1, 1);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            1, -1);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            1234, 1234);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            0, 1234);
        }
      }
      break;
    case operator_add:
    case operator_sub:
      if ((dst_attr->qualifier != PIC_VAL_QUALIFIER_W)
        && (val1_attr->qualifier != PIC_VAL_QUALIFIER_W)
        && (val2_attr->qualifier != PIC_VAL_QUALIFIER_W)) {
        if (val1_attr == val2_attr) {
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            64, 64);
        } else {
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
              0, 1);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
              0, -1);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
              255, 255);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
              0, 512);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
              65535, 65535);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
              65535, 0);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
              0, 65535);
        }
      }
      break;
    case operator_mul:
      if ((dst_attr->qualifier != PIC_VAL_QUALIFIER_W)
        && (val1_attr->qualifier != PIC_VAL_QUALIFIER_W)
        && (val2_attr->qualifier != PIC_VAL_QUALIFIER_W)) {
        if (val1_attr == val2_attr) {
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            64, 64);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            30, 30);
        } else {
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
              0, 10);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
              1, -10);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
              5, 8);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
              65, 97);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
              65, 0);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
              65, 1);
        }
      }
      break;
    case operator_div:
    case operator_mod:
      if ((dst_attr->qualifier != PIC_VAL_QUALIFIER_W)
        && (val1_attr->qualifier != PIC_VAL_QUALIFIER_W)
        && (val2_attr->qualifier != PIC_VAL_QUALIFIER_W)) {
        if (val1_attr == val2_attr) {
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            64, 64);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr, 
            30, 30);
        } else {
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
              0, 10);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
              12345, 8);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
              12345, 7);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
              12345, 16);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
              12345, 23);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
              23, 12345);
          pic_test_op_doit(target, flags, op, dst_attr, val1_attr, val2_attr,
              16, 12345);
        }
      }
      break;
  }
}

/* on initial entry, dst_attr, val1_attr, and val2_attr are all NULL
 * this recusive function defines all (dst, val1) combinations for
 * unary ops, and all (dst, val2) combinations for binary ops
 */
static void pic_test_op_setup(
  pic_target_cpu_t           target,
  unsigned                   flags,
  operator_t                 op, 
  const pic_val_attribute_t *dst_attr,
  const pic_val_attribute_t *val1_attr
  )
{
  pic_val_attribute_t attr;
  pic_val_qualifier_t q_first;

  /* dst cannot be CONST or LOOKUP */
  if (PIC_VAL_ATTRIBUTE_NONE == dst_attr) {
    q_first = PIC_VAL_QUALIFIER_W;
  } else if ((PIC_VAL_QUALIFIER_W == dst_attr->qualifier)
    || (val1_attr && (PIC_VAL_QUALIFIER_W == val1_attr->qualifier))) {
    /* W can only be used once */
    q_first = PIC_VAL_QUALIFIER_NONE;
  } else {
    q_first = PIC_VAL_QUALIFIER_CONST;
  }

  /* dst doesn't get CONST or LOOKUP */
  for (attr.qualifier = q_first;
       attr.qualifier < PIC_VAL_QUALIFIER_CT; 
       attr.qualifier++) {
    /* clearly CONST, LOOKUP and W cannot be indirect */
    pic_val_indirect_t ind_start;

    if (PIC_TARGET_CPU_12BIT == target) {
      ind_start = PIC_VAL_INDIRECT_NONE;
    } else {
      ind_start = (attr.qualifier < PIC_VAL_QUALIFIER_NONE)
           ? PIC_VAL_INDIRECT_NONE : PIC_VAL_INDIRECT_INDIRECT; 
    }
    for (attr.indirect = ind_start;
         attr.indirect < PIC_VAL_INDIRECT_CT;
         attr.indirect++) {
      /* W is always unsigned */
      for (attr.sign = (PIC_VAL_QUALIFIER_W == attr.qualifier)
             ? PIC_VAL_SIGN_NONE : PIC_VAL_SIGN_SIGNED;
           attr.sign < PIC_VAL_SIGN_CT;
           attr.sign++) {
        pic_val_bank_t bank_max;

        /* if the value is CONST, LOOKUP, or W the bank field is ignored
           so don't worry about it */
#if 1
        bank_max = ((attr.qualifier >= PIC_VAL_QUALIFIER_NONE)
            && (PIC_VAL_INDIRECT_NONE == attr.indirect))
          ? PIC_VAL_BANK_MAX
          : PIC_VAL_BANK_0;
#else
        bank_max = PIC_VAL_BANK_0;
#endif
        for (attr.bank = PIC_VAL_BANK_0;
             attr.bank <= bank_max;
             attr.bank++) {
          /* only dst_attr gets the BOOLEAN attribute */
          pic_val_type_t t_first;
          pic_val_type_t t_last;

          if ((PIC_VAL_QUALIFIER_W == attr.qualifier)
            || (PIC_VAL_INDIRECT_INDIRECT == attr.indirect)) {
            t_first = PIC_VAL_TYPE_BYTE;
            t_last  = (PIC_VAL_QUALIFIER_W == attr.qualifier)
              ? t_first + 1
              : PIC_VAL_TYPE_CT;
          } else {
            t_first = ((PIC_VAL_ATTRIBUTE_NONE == dst_attr)
              && (PIC_VAL_SIGN_NONE == attr.sign))
              ? PIC_VAL_TYPE_BOOLEAN
              : PIC_VAL_TYPE_SBIT;
            t_last = PIC_VAL_TYPE_CT;
          }
          for (attr.type = t_first;
               attr.type < t_last; 
               attr.type++) {
            if (PIC_VAL_ATTRIBUTE_NONE == dst_attr) {
              /* check either ( x = op x ) or ( x = x op x ) */
              if (PIC_VAL_QUALIFIER_W != attr.qualifier) {
                pic_test_op(target, flags, op, &attr, &attr,
                  (operator_is_binary(op)) ? &attr : PIC_VAL_ATTRIBUTE_NONE);
              }
              /* recurse to create val1 */
              pic_test_op_setup(target, flags, op, &attr, PIC_VAL_ATTRIBUTE_NONE);
            } else if (PIC_VAL_ATTRIBUTE_NONE == val1_attr) {
              pic_test_op(target, flags, op, dst_attr, &attr, 
                (operator_is_binary(op)) ? &attr : PIC_VAL_ATTRIBUTE_NONE);
              if (operator_is_binary(op)) {
                pic_test_op(target, flags, op, dst_attr, &attr, dst_attr);
                pic_test_op_setup(target, flags, op, dst_attr, &attr);
              }
            } else {
              pic_test_op(target, flags, op, dst_attr, val1_attr, &attr);
            }
          }
        }
      }
    }
  }
}

/*
 * test all combinations and all operators
 *   dst        : W, bool, [v][s]bit, [v][s]multi-bit, [v][i][s]byte, 
 *                [v][i][s]word, [v][i][s]dword
 *   val1, val2 : W, [v][s]bit, [v][s]multi-bit, [v][i|c|l][s]byte, 
 *                [v][i|c|l][s]word, [v][i|c|l][s]dword
 *
 *   also: dst=val1, dst=val2, val1=val2
 *   and...{dst,val1}{dst,val2}{val1,val2}
 *     where {a,b} = a,b in the same bank
 *
 * [s] -- sign or unsigned
 * [i] -- indirect
 * [c] -- constant
 * [l] -- lookup
 * [v] -- volatile
 *
 * so, there are 18 possible dst and 29 possible val[1|2]
 *      522 possible unary combinations
 *   15,138 possible binary combinations
 * not including multi-bank accesses
 */
static void help_show(const char *program, const char *err)
{
  operator_t op;
  int        pos;

  fprintf(stderr, "Error: %s\n"
                  "Format: %s [-12 | -14 | -16] op [first [last]]\n"
                  "  Execute tests starting at first and ending at last\n"
                  "  If first is missing, all tests are performed.\n"
                  "  If last is missing all tests from first through the end\n"
                  "  are performed.\n"
                  "    -12 : emulate the 12 bit core\n"
                  "    -14 : emulate the 14 bit core (default)\n"
                  "    -16 : emulate the 16 bit core\n",
                  err,
                  program);
  for (pos = 0, op = operator_null; op < operator_ct; op++) {
    const char *str;

    str = op_to_str(op);
    if (*str) {
      pos += printf("%s, ", str);
      if (pos > 70) {
        printf("\n");
        pos = 0;
      }
    }
  }
  if (pos) {
    printf("\n");
  }
  exit(1);
}

int main(int argc, char **argv)
{
  operator_t       op;
  pic_target_cpu_t target;
  const char      *prog;
  unsigned         flags;

  prog  = argv[0];
  flags = PFILE_FLAG_MISC_QUIET;

  interest_first = 0;
  interest_last  = -1;
  target = PIC_TARGET_CPU_14BIT;
  op     = operator_null;
  if (argc < 2) {
    help_show(prog, "no operator");
  } else {

    while (argc > 1) {
      if (0 == strcmp(argv[1], "-12")) {
        target = PIC_TARGET_CPU_12BIT;
        argv++;
        argc--;
      } else if (0 == strcmp(argv[1], "-14")) {
        target = PIC_TARGET_CPU_14BIT;
        argv++;
        argc--;
      } else if (0 == strcmp(argv[1], "-16")) {
        target = PIC_TARGET_CPU_16BIT;
        argv++;
        argc--;
      } else if (0 == strcmp(argv[1], "-d")) {
        flags |= PFILE_FLAG_DEBUG_COMPILER | PFILE_FLAG_DEBUG_EMULATOR;
        argv++;
        argc--;
      } else {
        break;
      }
    }
    for (op = operator_null; 
         (op < operator_ct) && strcmp(op_to_str(op), argv[1]); 
         op++)
      ;
    if (operator_ct == op) {
      help_show(prog, "invalid operator");
    } else if (argc > 2) {
      char *eptr;

      interest_first = strtoul(argv[2], &eptr, 10);
      if (*eptr) {
        help_show(prog, "first is not a number\n");
      } else if (argc > 3) {
        interest_last = strtoul(argv[3], &eptr, 10);
        if (*eptr) {
          help_show(prog, "last is not a number\n");
        } else if (interest_first > interest_last) {
          help_show(prog, "first is greater than last\n");
        } else if (argc > 4) {
          help_show(prog, "too many arguments");
        }
      }
    }
  }
  /* setvbuf(stdout, 0, _IONBF, 0); */
#ifdef TEST_PFILE_OP_CONST_EXEC
  for (op = operator_null + 1;
       op < operator_reference;
       op++) {
    pic_test_op_setup(op,
      PIC_VAL_ATTRIBUTE_NONE,
      PIC_VAL_ATTRIBUTE_NONE);
  }
#else
  pic_test_op_setup(target, flags, op,
    PIC_VAL_ATTRIBUTE_NONE,
    PIC_VAL_ATTRIBUTE_NONE);
#endif

  printf("test_ct = %u\n", test_ct);
  printf("instructions executed %u\n", pic_emu_instr_ct);
  printf("instructions/test %u\n", pic_emu_instr_ct / test_ct);
  return 0;
}

