/************************************************************
 **
 ** picbsrop.c : PIC data optimization definitions
 **
 ** Copyright (c) 2007, Kyle A. York
 ** All rights reserved
 **
 ************************************************************/
#include "pic_msg.h"
#include "piccolst.h"
#include "picbsrop.h"

/*
 * NAME
 *   pic_code_bsr_analyze
 *
 * DESCRIPTION
 *   analyze the bsr in the pic_codes[] array
 *
 * PARAMETERS
 *   pf    : 
 *   pcl   : pcl on entry
 *   bsr   : current contents of BSR
 *   depth : depth (for statistics only)
 *
 * RETURN
 *   none
 *
 * NOTES
 */
static void pic_code_bsr_analyze(pfile_t *pf, pic_code_t code,
  uchar bsr, unsigned depth)
{
  boolean_t pv_code_is_cond;

  /* now, traverse the code until either
     (1) all bits get known values
     (2) we hit an unconditional return or a leave */
  pv_code_is_cond = boolean_false;

  do {
    boolean_t  code_is_cond;
    pic_code_t next;

    code_is_cond = boolean_false;
    next = pic_code_next_get(code);

    if ((pic_code_bsr_get(code) == bsr) 
      || (PIC16_BSR_INDETERMINITE == pic_code_bsr_get(code))) {
      next = 0; /* we've a match, we're done! */
    } else {
      if (PIC16_BSR_UNKNOWN == pic_code_bsr_get(code)) {
        pic_code_bsr_set(code, bsr);
      } else if (bsr != pic_code_bsr_get(code)) {
        pic_code_bsr_set(code, PIC16_BSR_INDETERMINITE);
      }
      switch (pic_code_op_get(code)) {
        case pic_opcode_movlb:
          bsr = value_const_get(pic_code_literal_get(code));
          break;
        case pic_opcode_return:
        case pic_opcode_retfie:
        case pic_opcode_retlw:
          /* if this is unconditional we're done otherwise simply continue */
          if (!pv_code_is_cond) {
            next = 0;
          }
          break;
        case pic_opcode_goto:
        case pic_opcode_call:
          {
            boolean_t  is_suspend;
            pic_code_t brcode;

            brcode = pic_code_label_find(pf, pic_code_brdst_get(code));
            is_suspend = pic_code_is_suspend(pf, code);

            if ((pic_opcode_goto == pic_code_op_get(code))
                && !pv_code_is_cond
                && !is_suspend) {
              /* an absolute goto so next = brcode */
              next = brcode;
            } else {
              /* this is either a conditional goto or a call. either
                 way we first process then branch, then continue
                 processing from where we were. for the time being
                 a call is going to invalidate the known bits */
              pic_code_bsr_analyze(pf, brcode, bsr, depth + 1);
              if (is_suspend || (pic_opcode_call == pic_code_op_get(code))) {
                bsr = PIC16_BSR_INDETERMINITE;
              }
            }
          }
          break;
        case pic_opcode_incfsz:
        case pic_opcode_decfsz:
        case pic_opcode_btfsc:
        case pic_opcode_btfss:
          code_is_cond = boolean_true;
          break;
        default:
          break;
      }
      if (pic_code_modifies_pcl(pf, code)) {
        next = 0;
      }
      /* pic_code_bsr_set(code, bsr); */
    }
    pv_code_is_cond = code_is_cond;
    code = next;
  } while (code);
}

/* run through the pic_codes and remove any redundant dataset commands */
static void pic_code_bsr_post_analyze(pfile_t *pf)
{
  pic_code_t pv;
  pic_code_t code;
  unsigned   ins_ct; /* instruction ct */
  unsigned   rem_ct;

  for (ins_ct = 0, rem_ct = 0, pv = 0, code = pic_code_list_head_get(pf);
       code;
       ins_ct++) {
    pic_code_t next;

    next = pic_code_next_get(code);
    if (!pic_code_flag_test(code, PIC_CODE_FLAG_NO_OPTIMIZE)) {
      boolean_t            cremove;

      switch (pic_code_op_get(code)) {
        case pic_opcode_movlb:
          cremove = value_const_get(pic_code_literal_get(code))
              == pic_code_bsr_get(code);
          break;
        default:
          cremove = boolean_false;
          break;
      }
      if (cremove) {
        pic_code_list_remove(pf, code);
        pic_code_free(code);
        rem_ct++;
      } else {
        pv = code;
      }
    }
    code = next;
  }
  pfile_log(pf, pfile_log_debug, "...removed %u of %u instructions", 
      rem_ct, ins_ct);
}

void pic_code_bsr_optimize(pfile_t *pf)
{
  label_t lbl;

  pfile_log(pf, pfile_log_debug, PIC_MSG_FIXING_DATA_BITS);
  /* first -- analyze the ISR bits */
  lbl = pfile_label_find(pf, pfile_log_none, PIC_LABEL_ISR);
  if (lbl) {
    pic_code_t code;

    pfile_log(pf, pfile_log_debug, "...analyzing ISR");
    code = pic_code_label_find(pf, lbl);
    if (code) {
      pic_code_bsr_analyze(pf, code, PIC16_BSR_INDETERMINITE, 1);
    }
    label_release(lbl);
  }

  /* second -- analyze the USER bits */
  lbl = pfile_label_find(pf, pfile_log_err, PIC_LABEL_RESET);
  if (lbl) {
    pic_code_t code;

    pfile_log(pf, pfile_log_debug, "...analyzing USER");
    code = pic_code_label_find(pf, lbl);
    if (code) {
      pic_code_bsr_analyze(pf, code, 0, 1);
    }
    label_release(lbl);
  }
  /* once the analysis is done, we need to run through & make
     sure no further label fixups are needed, or fix them up
     as necessary */
  pfile_log(pf, pfile_log_debug, "...post analyze");
  pic_code_bsr_post_analyze(pf);
}

