#include <stdio.h>
#include <stdlib.h>
#include <limits.h>

typedef enum {
  VALUE_TYPE_NONE,
  VALUE_TYPE_FIRST,
  VALUE_TYPE_BIT = VALUE_TYPE_FIRST,
  VALUE_TYPE_BYTE,
  VALUE_TYPE_WORD,
  VALUE_TYPE_DWORD,
  VALUE_TYPE_SBYTE,
  VALUE_TYPE_SWORD,
  VALUE_TYPE_SDWORD,
  VALUE_TYPE_BYTE_ARRAY,
  VALUE_TYPE_WORD_ARRAY,
  VALUE_TYPE_DWORD_ARRAY,
  VALUE_TYPE_SBYTE_ARRAY,
  VALUE_TYPE_SWORD_ARRAY,
  VALUE_TYPE_SDWORD_ARRAY,
  VALUE_TYPE_WORD_LOOKUP,
  VALUE_TYPE_CONST,
  VALUE_TYPE_CT
} value_type_t;

typedef struct {
  value_type_t  type; /* value type         */
  unsigned long n;    /* the value          */
} value_t;


static int value_type_is_array(value_type_t type)
{
  int rc;

  rc = 0;
  switch (type) {
    case VALUE_TYPE_NONE:
    case VALUE_TYPE_BIT:
    case VALUE_TYPE_BYTE:
    case VALUE_TYPE_WORD:
    case VALUE_TYPE_DWORD:
    case VALUE_TYPE_SBYTE:
    case VALUE_TYPE_SWORD:
    case VALUE_TYPE_SDWORD:
    case VALUE_TYPE_CONST:
    case VALUE_TYPE_CT:
      rc = 0;
      break;
    case VALUE_TYPE_BYTE_ARRAY:
    case VALUE_TYPE_WORD_ARRAY:
    case VALUE_TYPE_DWORD_ARRAY:
    case VALUE_TYPE_SBYTE_ARRAY:
    case VALUE_TYPE_SWORD_ARRAY:
    case VALUE_TYPE_SDWORD_ARRAY:
    case VALUE_TYPE_WORD_LOOKUP:
      rc = 1;
      break;
  }
  return rc;
}

static int value_type_is_signed(value_type_t type)
{
  int rc;

  rc = 0;
  switch (type) {
    case VALUE_TYPE_NONE:
    case VALUE_TYPE_BIT:
    case VALUE_TYPE_BYTE:
    case VALUE_TYPE_WORD:
    case VALUE_TYPE_DWORD:
    case VALUE_TYPE_BYTE_ARRAY:
    case VALUE_TYPE_WORD_ARRAY:
    case VALUE_TYPE_DWORD_ARRAY:
    case VALUE_TYPE_WORD_LOOKUP:
    case VALUE_TYPE_CONST:
    case VALUE_TYPE_CT:
      rc = 0;
      break;
    case VALUE_TYPE_SBYTE:
    case VALUE_TYPE_SWORD:
    case VALUE_TYPE_SDWORD:
    case VALUE_TYPE_SBYTE_ARRAY:
    case VALUE_TYPE_SWORD_ARRAY:
    case VALUE_TYPE_SDWORD_ARRAY:
      rc = 1;
      break;
  }
  return rc;
}

#if 0
static int value_type_is_const(value_type_t type)
{
  int rc;

  rc = 0;
  switch (type) {
    case VALUE_TYPE_NONE:
    case VALUE_TYPE_BIT:
    case VALUE_TYPE_BYTE:
    case VALUE_TYPE_WORD:
    case VALUE_TYPE_DWORD:
    case VALUE_TYPE_BYTE_ARRAY:
    case VALUE_TYPE_WORD_ARRAY:
    case VALUE_TYPE_DWORD_ARRAY:
    case VALUE_TYPE_CT:
    case VALUE_TYPE_SBYTE:
    case VALUE_TYPE_SWORD:
    case VALUE_TYPE_SDWORD:
    case VALUE_TYPE_SBYTE_ARRAY:
    case VALUE_TYPE_SWORD_ARRAY:
    case VALUE_TYPE_SDWORD_ARRAY:
      rc = 0;
      break;
    case VALUE_TYPE_WORD_LOOKUP:
    case VALUE_TYPE_CONST:
      rc = 1;
      break;
  }
  return rc;
}
#endif

#define FLAG_NONE         0x0000
#define FLAG_EXPAND_ARRAY 0x0001
static void value_type_name_write(FILE *dst, value_type_t type, 
    unsigned n, unsigned flags)
{
  const char *str;

  str = "";
  switch (type) {
    case VALUE_TYPE_NONE:                           break;
    case VALUE_TYPE_BIT:           str = "bit";     break;
    case VALUE_TYPE_BYTE:          str = "byte";    break;
    case VALUE_TYPE_WORD_LOOKUP:
    case VALUE_TYPE_WORD:          str = "word";    break;
    case VALUE_TYPE_DWORD:         str = "dword";   break;
    case VALUE_TYPE_SBYTE:         str = "sbyte";   break;
    case VALUE_TYPE_SWORD:         str = "sword";   break;
    case VALUE_TYPE_SDWORD:        str = "sdword";  break;
    case VALUE_TYPE_BYTE_ARRAY:    str = "abyte";   break;
    case VALUE_TYPE_WORD_ARRAY:    str = "aword";   break;
    case VALUE_TYPE_DWORD_ARRAY:   str = "adword";  break;
    case VALUE_TYPE_SBYTE_ARRAY:   str = "asbyte";  break;
    case VALUE_TYPE_SWORD_ARRAY:   str = "asword";  break;
    case VALUE_TYPE_SDWORD_ARRAY:  str = "asdword"; break;
    case VALUE_TYPE_CONST:         str = "const";   break;
    case VALUE_TYPE_CT:                             break;
  }
  fprintf(dst, "u_%s%s_%u", (VALUE_TYPE_WORD_LOOKUP == type) ? "l" : "", str, n);
  if ((flags & FLAG_EXPAND_ARRAY) && value_type_is_array(type)) {
    fprintf(dst, "[ix]");
  }
}

static void value_type_type_write(FILE *dst, value_type_t type)
{
  const char *str;

  str = "";
  switch (type) {
    case VALUE_TYPE_NONE:                          break;
    case VALUE_TYPE_BIT:           str = "bit";    break;
    case VALUE_TYPE_BYTE:          str = "byte";   break;
    case VALUE_TYPE_WORD_LOOKUP:
    case VALUE_TYPE_WORD:          str = "word";   break;
    case VALUE_TYPE_DWORD:         str = "dword";  break;
    case VALUE_TYPE_SBYTE:         str = "sbyte";  break;
    case VALUE_TYPE_SWORD:         str = "sword";  break;
    case VALUE_TYPE_SDWORD:        str = "sdword"; break;
    case VALUE_TYPE_BYTE_ARRAY:    str = "byte";   break;
    case VALUE_TYPE_WORD_ARRAY:    str = "word";   break;
    case VALUE_TYPE_DWORD_ARRAY:   str = "dword";  break;
    case VALUE_TYPE_SBYTE_ARRAY:   str = "sbyte";  break;
    case VALUE_TYPE_SWORD_ARRAY:   str = "sword";  break;
    case VALUE_TYPE_SDWORD_ARRAY:  str = "sdword"; break;
    case VALUE_TYPE_CONST:         str = "const";  break;
    case VALUE_TYPE_CT:                            break;
  }
  fprintf(dst, "%s", str);
}

static unsigned value_type_to_bits(value_type_t type)
{
  unsigned bits;

  bits = 0;
  switch (type) {
    case VALUE_TYPE_NONE:                      break;
    case VALUE_TYPE_BIT:            bits =  1; break;
    case VALUE_TYPE_BYTE:           bits =  8; break;
    case VALUE_TYPE_WORD_LOOKUP:
    case VALUE_TYPE_WORD:           bits = 16; break;
    case VALUE_TYPE_DWORD:          bits = 32; break;
    case VALUE_TYPE_SBYTE:          bits =  8; break;
    case VALUE_TYPE_SWORD:          bits = 16; break;
    case VALUE_TYPE_SDWORD:         bits = 32; break;
    case VALUE_TYPE_BYTE_ARRAY:     bits =  8; break;
    case VALUE_TYPE_WORD_ARRAY:     bits = 16; break;
    case VALUE_TYPE_DWORD_ARRAY:    bits = 32; break;
    case VALUE_TYPE_SBYTE_ARRAY:    bits =  8; break;
    case VALUE_TYPE_SWORD_ARRAY:    bits = 16; break;
    case VALUE_TYPE_SDWORD_ARRAY:   bits = 32; break;
    case VALUE_TYPE_CONST:          bits = 32; break;
    case VALUE_TYPE_CT:                        break;
  }
  return bits;
}


/*
 * procedure result_add below probably looks a bit odd, but the idea
 * is to be able to run each of these tests in the MPLAB simulator,
 * and simply use `run to cursor'. If the program ends and cursor is
 * never hit -- no errors. If the cursor is hit, simply step back to
 * the caller & determine what caused the problem
 */
static void preamble(FILE *dst)
{
  value_type_t type;

  fprintf(dst, 
    "include c16f877\n"
    "var volatile byte result\n"
    "procedure result_set(byte in x) is\n"
    "  if (x == 0) then\n"
    "    result = x\n"
    "  end if\n"
    "end procedure\n"
    "\n"
    "var byte ix=2\n"
  );
  for (type = VALUE_TYPE_FIRST; type < VALUE_TYPE_CT; type++) {
    unsigned    n;

    if (VALUE_TYPE_CONST != type) {
      if (VALUE_TYPE_WORD_LOOKUP == type) {
        fputs("const ", dst);
      } else {
        fputs("var ", dst);
      }
      value_type_type_write(dst, type);
      fputc(' ', dst);

      for (n = 0; n < 3; n++) {
        if (n) {
          fputs(", ", dst);
        }
        value_type_name_write(dst, type, n, FLAG_NONE);
        if (value_type_is_array(type)) {
          fputs("[5]", dst);
          if (VALUE_TYPE_WORD_LOOKUP == type) {
            fputs(" = { 5, 6, 7, 8 }", dst);
          }
        }
      }
      fputc('\n', dst);
    }
  }
}

static unsigned testfile_number;

static const char *testfile_name_get(char *dst)
{
  sprintf(dst, "test%03u.jal", testfile_number);
  return dst;
}

static FILE *testfile_open(void)
{
  FILE *out;
  char  name[32];

  testfile_number++;
  testfile_name_get(name);
  out = fopen(name, "w");
  preamble(out);
  return out;
}

static void value_type_set(value_t *val, value_type_t type)
{
  val->type = type;
}

static value_type_t value_type_get(const value_t *n)
{
  return n->type;
}

static void value_n_set(value_t *val, unsigned long n)
{
  unsigned bits;

  bits = value_type_to_bits(val->type);
  if (1 == bits) {
    val->n = (n) ? 1 : 0;
  } else if (32 == bits) {
    val->n = n;
  } else {
    val->n = n & ((1UL << bits) - 1);
  }
}

static unsigned long value_n_get(const value_t *val)
{
  unsigned      bits;
  unsigned long n;

  bits = value_type_to_bits(val->type);

  if ((32 == bits) || !value_type_is_signed(val->type)) {
    n = val->n;
  } else {
    unsigned long sign_bit;
    unsigned long mask;

    sign_bit = 1UL << (bits - 1);
    mask     = (1UL << bits) - 1;
    n = (val->n & sign_bit)
      ? val->n | ~mask
      : val->n;
  }
  return n;
}

static void value_write(FILE *dst, const value_t *val)
{
  if (value_type_is_signed(value_type_get(val))) {
    fprintf(dst, "%ld", (long) value_n_get(val));
  } else {
    fprintf(dst, "%lu", value_n_get(val));
  }
}

static value_type_t promotion_get(value_type_t type1, value_type_t type2)
{
  value_type_t type;

  if ((value_type_to_bits(type1) == 1)
    && (value_type_to_bits(type2) == 1)) {
    type = (value_type_is_signed(type1) && value_type_is_signed(type2))
      ? VALUE_TYPE_SBYTE
      : VALUE_TYPE_BYTE;
  } else if (value_type_to_bits(type1) < value_type_to_bits(type2)) {
    type = type2;
  } else if (value_type_to_bits(type1) > value_type_to_bits(type2)) {
    type = type1;
  } else if (value_type_is_signed(type1) && !value_type_is_signed(type2)) {
    type = type2;
  } else if (!value_type_is_signed(type1) && value_type_is_signed(type2)) {
    type = type1;
  } else {
    type = type1;
  }
  return type;
}

static unsigned long op_add(unsigned long a, unsigned long b)
{
  return a + b;
}

static unsigned long op_sub(unsigned long a, unsigned long b)
{
  return a - b;
}

static unsigned long op_mul(unsigned long a, unsigned long b)
{
  return a * b;
}

static unsigned long op_div(unsigned long a, unsigned long b)
{
  return a / b;
}

static unsigned long op_mod(unsigned long a, unsigned long b)
{
  return a % b;
}

static unsigned long op_and(unsigned long a, unsigned long b)
{
  return a & b;
}

static unsigned long op_or(unsigned long a, unsigned long b)
{
  return a | b;
}

typedef struct op_table_entry_ {
  const char     *op;
  unsigned long (*fn)(unsigned long a, unsigned long b);
} op_table_entry_t;

op_table_entry_t op_table[] = {
  { "+", op_add },
  { "-", op_sub },
  { "*", op_mul },
  { "/", op_div },
  { "%", op_mod },
  { "&", op_and },
  { "|", op_or  }
};

int main(int argc, char **argv)
{
  value_type_t            type_dst;
  FILE                   *out;
  unsigned                test_ct;
  unsigned                test_total;
  unsigned                test_start;
  unsigned                tests_per_file;
  unsigned                test_first;
  unsigned                test_last;
  int                     err;
  const op_table_entry_t *op;

  tests_per_file = 100;
  test_first     =   0;
  test_last      =  -1;

  op = op_table + 1; /* subtract */
  err = 0;
  if (argc >= 2) {
    char *eptr;

    tests_per_file = strtoul(argv[1], &eptr, 10);
    err = (*eptr != 0);
    if (!err && (argc >= 3)) {
      test_first = strtoul(argv[2], &eptr, 10);
      err = (*eptr != 0);
      if (!err && (argc >= 4)) {
        test_last = strtoul(argv[3], &eptr, 10);
        err = (*eptr != 0) || (argc > 4);
      }
    }
  }
  if (err) {
    fprintf(stderr, "Format: %s [tests_per_file [test_first [test_last]]]\n",
        argv[0]);
    exit(1);
  }

  test_ct = 0;
  test_total = 0;
  test_start = 1;
  out = 0;
  for (type_dst = VALUE_TYPE_FIRST; 
       (type_dst < VALUE_TYPE_CT) && (test_total < test_last); 
       type_dst++) {
    /* clearly dst cannot be const! */
    if (VALUE_TYPE_CONST != type_dst) {
      value_t      dst;
      value_type_t type1;

      value_type_set(&dst, type_dst);
      for (type1 = VALUE_TYPE_FIRST; 
           (type1 < VALUE_TYPE_CT) && (test_total < test_last); 
           type1++) {
        value_t      val1;
        value_type_t type2;

        value_type_set(&val1, type1);
        for (type2 = VALUE_TYPE_FIRST; 
             (type2 < VALUE_TYPE_CT) && (test_total < test_last); 
             type2++) {
          /* also pointless if both type1 & type2 are const */
          if (!(VALUE_TYPE_CONST == type1)
            && !(VALUE_TYPE_CONST == type2)) {
            test_total++;

            if (test_total >= test_first) {
              value_t val2;
              value_t tmp;

              value_type_set(&val2, type2);
              value_type_set(&tmp, promotion_get(type1, type2));

              value_n_set(&val1, rand());
              value_n_set(&val2, rand());
              value_n_set(&tmp, 
                  op->fn(value_n_get(&val1), value_n_get(&val2)));
              value_n_set(&dst, value_n_get(&tmp));

              if (!out) {
                out = testfile_open();
              }
              /* first assign to val1 */
              value_type_name_write(out, type1, 1, FLAG_EXPAND_ARRAY);
              fputs(" = ", out);
              value_write(out, &val1);
              fputc('\n', out);

              /* next assign to val2 */
              value_type_name_write(out, type2, 2, FLAG_EXPAND_ARRAY);
              fputs(" = ", out);
              value_write(out, &val2);
              fputc('\n', out);

              /* next write the operation */
              value_type_name_write(out, type_dst, 0, FLAG_EXPAND_ARRAY);
              fputs(" = ", out);
              value_type_name_write(out, type1, 1, FLAG_EXPAND_ARRAY);
              fprintf(out, " %s ", op->op);
              value_type_name_write(out, type2, 2, FLAG_EXPAND_ARRAY);

              /* finally, create the conditional */
              fputs("\nif ", out);
              value_type_name_write(out, type_dst, 0, FLAG_EXPAND_ARRAY);
              fputs(" == ", out);
              value_write(out, &dst);
              fputs(" then result_set(true) else result_set(false) end if\n",
                  out);
              test_ct++;
              if (test_ct == tests_per_file) {
                char name[32];

                fprintf(stderr, "File %s contains %u...%u\n",
                    testfile_name_get(name), test_first, test_total);
                test_start = test_total + 1;
                fclose(out);
                out = 0;
                test_ct = 0;
              }
            }
          }
        }
      }
    }
  }
  if (out) {
    char name[32];

    fprintf(stderr, "File %s contains %u...%u\n",
        testfile_name_get(name), test_first, test_total);
    fclose(out);
  }

  return 0;
}

